/*
 
 *  nfs.c --
 
 *      NFS server implementation for PC.  Runs on top of the
 
 *      net daemon (netd.c).
 
 *
 
 *  Author:
 
 *      See-Mong Tan, 6/12/88.
 
 */
 

 
#include "common.h"
 
#include "msc-dos.h"		/* dos and ... */
 
#include <direct.h>		/* directory ops */
 

 
/*
 
 *  bool_t nfs_init() --
 
 *      Initializes NFS server.  Creates and registers transport
 
 *      handle for NFS server.  Returns TRUE if successful, or FALSE
 
 *      if an error occurred.
 
 */
 
bool_t nfs_init()
 
{
 
	SVCXPRT *transp;
 
	int nfs_sock;
 
	struct sockaddr_in addr;
 

 
	addr.sin_family = AF_INET;
 
	addr.sin_port = htons(NFS_PORT);
 
	addr.sin_addr.s_addr = INADDR_ANY;
 
	if ((nfs_sock = sock_create(SOCK_DGRAM, IPPROTO_UDP, &addr)) < 0) {
 
		(void) fprintf(stderr, "nfs_init: cannot create nfs socket\n");
 
		return FALSE;
 
	}
 
	if ((transp = svcudp_create(nfs_sock, 1)) == (SVCXPRT *) NULL) {
 
		(void) fprintf(stderr, "nfs_init: cannot create udp handle\n");
 
		sock_close(nfs_sock);
 
		return FALSE;
 
	}
 
	if (! svc_register(transp, NFS_PROGRAM, NFS_VERSION, nfs_dispatch, 
 
			   IPPROTO_UDP)) {
 
		(void) fprintf(stderr, "nfs_init: cannot register handle\n");
 
		sock_close(nfs_sock);
 
		return FALSE;
 
	}
 

 
	return TRUE;
 
}
 

 
/*
 
 *  bool_t nfs_dispatch(struct svc_req *req, SVCXPRT *transp) --
 
 *      NFS server dispatch routine.
 
 */
 
bool_t nfs_dispatch(req, transp)
 
	struct svc_req *req;
 
	SVCXPRT *transp;
 
{
 
	/* find which procedure to call */
 
	switch((int) req->rq_proc) {	
 

 
	case NFS_GETATTR:			/* get file attribute */
 
		nfs_getattr(transp, req);
 
		break;
 

 
	case NFS_NULL:
 
		nfs_null();			/* does nothing */
 
		break;
 

 
	case NFS_READ:
 
		nfs_read(transp, req);		/* read a file */
 
		break;
 

 
	case NFS_WRITE:
 
		nfs_write(transp, req);		/* write a file */
 
		break;
 

 
	case NFS_STATFS:
 
		nfs_statfs(transp, req);	/* filesystem status */
 
		break;
 

 
	case NFS_LOOKUP:			/* directory lookup */
 
		nfs_lookup(transp, req);
 
		break;
 

 
	case NFS_READDIR:
 
		nfs_readdir(transp, req);	/* read directory */
 
		break;
 

 
	case NFS_CREATE:
 
		nfs_create(transp, req);	/* create a file */
 
		break;
 

 
	case NFS_MKDIR:
 
		nfs_mkdir(transp, req);		/* make a directory */
 
		break;
 

 
	case NFS_REMOVE:
 
		nfs_remove(transp, req);	/* remove a file */
 
		break;
 

 
	case NFS_RMDIR:
 
		nfs_rmdir(transp, req);		/* remove a directory */
 
		break;
 

 
	case NFS_RENAME:
 
		nfs_rename(transp, req);	/* rename a file */
 
		break;
 

 
	case NFS_SETATTR:
 
		nfs_setattr(transp, req);	/* set file attributes */
 
		break;
 

 
	default:
 
		nfs_error(transp, req);		/* stuff not supported */
 
		break;
 
	}
 
}
 

 
/*
 
 *  void nfs_null(SVCXPRT *xprt, struct svc_req *req) --
 
 *      Sends an empty reply.  Used for "ping-ing" the nfs server.
 
 */
 
void nfs_null(xprt, req)
 
	SVCXPRT *xprt;
 
	struct svc_req *req;
 
{
 
	if (! svc_sendreply(xprt, xdr_void, (char *) NULL))
 
		(void) fprintf(stderr, "nfs_null: cannot send reply\n");
 
}
 

 
/*
 
 *  void nfs_read(SVCXPRT *xprt, struct svc_req *req) --
 
 *      Reads requested file and sends it over the wire.
 
 */
 
void nfs_read(xprt, req)
 
	SVCXPRT *xprt;
 
	struct svc_req *req;
 
{
 
	struct nfsreadargs *r_args;		/* the argument */
 
	struct nfsrdresult *r_rslt;		/* the read result */
 
	char *fullpath;				/* DOS file name path */
 
	int bytes;
 

 
	r_args = (struct nfsreadargs *) malloc(sizeof(struct nfsreadargs));
 
	if (r_args == NULL) {
 
		(void) fprintf(stderr, "nfs_read: error, out of memory\n");
 
		abort();
 
	}
 
	(void) bzero(r_args, sizeof(struct nfsreadargs));
 
	if (! svc_getargs(xprt, xdr_readargs, r_args)) {
 
		(void) fprintf(stderr, "nfs_read: cannot read args\n");
 
		svcerr_decode(xprt);
 
		(void) free(r_args);
 
		return;
 
	}
 
	r_rslt = (struct nfsrdresult *) malloc(sizeof(struct nfsrdresult));
 
	if (r_rslt == NULL) {
 
		(void) fprintf(stderr, "nfs_read: error, out of memory\n");
 
		abort();
 
	}
 
	(void) bzero(r_rslt, sizeof(struct nfsrdresult));
 
	/* init and allocate space for data in results struct */
 
	r_rslt->rr_data = (char *) malloc((unsigned) r_args->ra_count);
 
	if (r_rslt->rr_data == NULL) {
 
		(void) fprintf(stderr, "nfs: out of memory\n");
 
		abort();
 
	}
 
	/* get path name from inode number */
 
	fullpath = intopn(r_args->ra_fhandle.f.fh_fno);
 

 
	/* include file attributes */
 
	if (! file_getattr(fullpath, &(r_rslt->rr_attr)))
 
		r_rslt->rr_status = NFSERR_NOENT;
 
	else  {
 
		if ((bytes = file_read(fullpath, r_args->ra_offset, 
 
			       r_args->ra_count,r_rslt->rr_data)) == -1) {
 
			(void) fprintf(stderr, "nfs: read err\n");
 
			r_rslt->rr_status = NFSERR_IO;
 
		}
 
		else {
 
#ifdef DEBUG
 
			(void) printf("nfs_read: %d bytes read from %s\n",
 
					bytes, fullpath);
 
#endif
 
			r_rslt->rr_bufsize = r_args->ra_count;
 
			r_rslt->rr_count = bytes;
 
			r_rslt->rr_status = NFS_OK;
 
			r_rslt->rr_bp = NULL;
 
			r_rslt->rr_vp = NULL;
 
		}
 
	}
 
	if (! svc_sendreply(xprt, xdr_rdresult, r_rslt))
 
		(void) fprintf(stderr, ">>> NFS_READ: cannot reply\n");
 
	else if (NFS_VERBOSE)
 
		(void) printf(">>> NFS_READ: %s\n", fullpath);
 

 
	/* free arguments */	
 
	svc_freeargs(xprt, xdr_readargs, r_args);
 
	(void) free(r_rslt->rr_data);
 
	(void) free(r_rslt);
 
	if (fullpath != NULL)
 
		(void) free(fullpath);
 
}
 

 
/*
 
 *  void nfs_error(SVCXPRT *xprt, struct svc_req *req) --
 
 *      Returns nfs error message.
 
 */
 
void nfs_error(xprt, req)
 
	SVCXPRT *xprt;
 
	struct svc_req *req;
 
{
 
	(void) fprintf(stderr, 
 
		">>> NFS_ERROR: procedure %d not supported\n", req->rq_proc);
 
	svcerr_noproc(xprt);		/* send server error reply msg */
 
}
 

 
/*
 
 *  void nfs_getattr(SVCXPRT *xprt, struct svc_req *req) --
 
 *      Gets file attributes.
 
 */
 
void nfs_getattr(xprt, req)
 
	SVCXPRT *xprt;
 
	struct svc_req *req;
 
{
 
	fhandle_t *fhp;			/* file handle is argument */
 
	struct nfsattrstat *attr;	/* return attributes */	
 
	char *fullpath;			/* full DOS path name */
 

 
	fhp = (fhandle_t *) malloc(sizeof(fhandle_t));
 
	if (fhp == NULL) {
 
		(void) fprintf(stderr, "nfs: out of memory\n");
 
		abort();
 
	}
 
	(void) bzero(fhp, sizeof(fhandle_t));
 
	if (! svc_getargs(xprt, xdr_fhandle, fhp)) {
 
		(void) fprintf(stderr, "nfs_getattr: cannot read args\n");
 
		svcerr_decode(xprt);
 
		(void) free(fhp);
 
		return;
 
	}
 
	/* make space for return attributes */
 
	attr = (struct nfsattrstat *) malloc(sizeof(struct nfsattrstat));
 
	if (attr == NULL) {
 
		(void) fprintf(stderr, "nfs: out of memory\n");
 
		abort();
 
 	}
 
	(void) bzero(attr, sizeof(struct nfsattrstat));
 
	fullpath = intopn(fhp->f.fh_fno);	/* path from inode */
 
	if (! file_getattr(fullpath, &(attr->ns_attr))) {
 
		attr->ns_status = NFSERR_NOENT;
 
	}
 
	else {
 
		attr->ns_status = NFS_OK;
 
	}
 
	if (! svc_sendreply(xprt, xdr_attrstat, attr))
 
		(void) fprintf(stderr, ">>> NFS_GETATTR send error\n");
 
	else if (NFS_VERBOSE)
 
		(void) printf(">>> NFS_GETATTR: %s\n", fullpath);
 

 
	svc_freeargs(xprt, xdr_fhandle, fhp);
 
	(void) free(attr);
 
	if (fullpath != NULL)
 
		(void) free(fullpath);
 
}
 

 
/*
 
 *  void nfs_statfs(SVCXPRT *xprt, struct svc_req *req) --
 
 *      Returns file system status
 
 */
 
void nfs_statfs(xprt, req)
 
	SVCXPRT *xprt;
 
	struct svc_req *req;
 
{
 
	struct nfsstatfs *fs;
 
	fhandle_t *fhp;
 

 
	fhp = (fhandle_t *) malloc(sizeof(fhandle_t));
 
	if (fhp == NULL) {
 
		(void) fprintf(stderr, "nfs: out of memory\n");
 
		abort();
 
	}
 
	(void) bzero(fhp, sizeof(fhandle_t));
 
	if (! svc_getargs(xprt, xdr_fhandle, fhp)) {
 
		(void) fprintf(stderr, "nfs_statfs: cannot decode args\n");
 
		svcerr_decode(xprt);
 
		(void) free(fhp);
 
		return;
 
	}
 
	/* alloc space for results struct */
 
	fs = (struct nfsstatfs *) malloc(sizeof(struct nfsstatfs));
 
	if (fs == NULL) {
 
		(void) fprintf(stderr, "nfs:  out of memory\n");
 
		abort();
 
	}
 
	(void) bzero(fs, sizeof(struct nfsstatfs));
 
	/* set up struct */
 
	if (! file_freeblocks(fhp->f.fh_fsid, &(fs->fs_bfree), 
 
			      &(fs->fs_blocks))) {
 
		(void) fprintf(stderr, "statfs: file system read err\n");
 
		fs->fs_status = NFSERR_IO;
 
	}
 
	else {
 
		fs->fs_tsize = 512;		/* because of Toolkit limits */
 
		fs->fs_bsize = FS_BLOCKSIZE;
 
		fs->fs_status = NFS_OK;
 
		fs->fs_bavail = fs->fs_bfree;
 
	}
 
	if (! svc_sendreply(xprt, xdr_statfs, fs))
 
		(void) fprintf(stderr, "nfs_statfs: cannot send reply\n");
 
	else if (NFS_VERBOSE)
 
		(void) printf(">>> NFS_STATFS: drive %d\n", fhp->f.fh_fsid);
 
	svc_freeargs(xprt, xdr_fhandle, fhp);
 
	(void) free(fs);
 
}
 

 
/*
 
 *  void nfs_readdir(SVCXPRT *xprt, struct svc_req *req) --
 
 *      Read a directory.
 
 */
 
void nfs_readdir(xprt, req)
 
	SVCXPRT *xprt;
 
	struct svc_req *req;
 
{
 
	struct nfsrddirargs *args;		/* args */
 
	struct nfsrddirres *res;		/* results */
 
	char *fullpath;				/* full directory path name */
 
	int offs;				/* offset */
 
	int bytes = 0;				/* # of dir bytes read */
 
	int maxbytes;
 
	struct udirect *udp;			/* directory cookie array */
 
#define SUD	sizeof(struct udirect)		/* size of directory cookie */
 

 
	args = (struct nfsrddirargs *) malloc(sizeof(struct nfsrddirargs));
 
	if (args == NULL) {
 
		(void) fprintf(stderr, "nfs: out of memory\n");
 
		abort();
 
	}
 
	maxbytes = (args->rda_count > NFS_MAXDATA ? NFS_MAXDATA : args->rda_count);
 
	(void) bzero(args, sizeof(struct nfsrddirargs));
 
	if (! svc_getargs(xprt, xdr_rddirargs, args)) {
 
		(void) fprintf(stderr, "nfs_readdir: cannot decode request\n");
 
		svcerr_decode(xprt);
 
		(void) free(args);
 
		return;
 
	}
 
	offs = args->rda_offset;
 
	fullpath = intopn(args->rda_fh.f.fh_fno);	   /* get path name */
 

 
	/* alloc space for results */
 
	res = (struct nfsrddirres *) malloc(sizeof(struct nfsrddirres));
 
	if (res == NULL) {
 
		(void) fprintf(stderr, "nfs: out of memory\n");
 
		abort();
 
	}
 
	(void) bzero(res, sizeof(struct nfsrddirres));	/* zero results */
 
	res->rd_eof = TRUE;		         	/* end of entries */
 
	res->rd_bufsize = maxbytes;		/* size of clnt req */
 
	/* alloc space for the directory entries */
 
	res->rd_entries = (struct udirect *) malloc(maxbytes + SUD);
 
	if ((udp = res->rd_entries) == NULL) {
 
		(void) fprintf(stderr, "nfs: out of memory\n");
 
		abort();
 
	}
 
	(void) bzero(udp, maxbytes);		/* zero everything */
 
	/* read until filled */
 
	res->rd_status = NFS_OK;
 
	for(;;) {
 
		int rstat;		/* read status */
 
#define NEXTUDP(udp) ((struct udirect *)  \
 
			((u_long) udp + UDIRSIZ(udp)))
 

 
		rstat = file_rddir(fullpath, offs, udp);  /* read one entry */
 
		if (rstat == 0)	{	   		  /* no more entries */
 
			bytes += udp->d_reclen;		  /* + size of entry */
 
			break;
 
		}
 
		if (rstat == -1) {			/* error reading */
 
			res->rd_status = NFSERR_NOENT;
 
			break;
 
		}
 
		/* else was a successful read */
 
		if (bytes > maxbytes) {	/* filled up */
 
			res->rd_eof = FALSE;
 
			break;
 
		}
 
		bytes += udp->d_reclen;			/* + size of entry */
 
		udp = NEXTUDP(udp);			/* next udp entry */
 
		offs += SUD;				/* next offset */
 

 
#undef NEXTUDP
 
	}
 
	/* broke out of the loop */
 
	if (res->rd_status == NFS_OK) {			/* good read */
 
		res->rd_offset = offs + SUD;		/* next offset */
 
		res->rd_size = bytes;			/* # of bytes */
 
	}
 
	if (! svc_sendreply(xprt, xdr_putrddirres, res))
 
		(void) fprintf(stderr, "nfs_readdir: cannot reply\n");
 
	else if (NFS_VERBOSE)
 
		(void) printf(">>> NFS_READDIR: %s\n", fullpath);
 

 
	/* free space */
 
	svc_freeargs(xprt, xdr_rddirargs, args);
 
	(void) free(res->rd_entries);
 
	(void) free(res);
 
	if (fullpath != NULL)
 
		(void) free(fullpath);
 
#undef SUD
 
}
 

 
/*
 
 *  void nfs_lookup(SVCXPRT *xprt, struct svc_req *req) --
 
 *      Directory lookup.
 
 */
 
void nfs_lookup(xprt, req)
 
	SVCXPRT *xprt;
 
	struct svc_req *req;
 
{
 
	struct nfsdiropargs *args;	/* arguments to call */
 
	struct nfsdiropres *res;	/* and results */
 
	struct nfsfattr attr;		/* return attributes */
 
	char *fullpath;			/* full path of file being looked up */
 
	
 
	args = (struct nfsdiropargs *) malloc(sizeof(struct nfsdiropargs));
 
	if (args == NULL) {
 
		(void) fprintf(stderr, "nfs: out of memory\n");
 
		abort();
 
	}
 
	(void) bzero(args, sizeof(struct nfsdiropargs));
 
	if (! svc_getargs(xprt, xdr_diropargs, args)) {
 
		(void) fprintf(stderr, "nfs_lookup: cannot decode\n");
 
		svcerr_decode(xprt);
 
		(void) free(args);
 
		return;
 
	}
 
	/* alloc space for return struct */
 
	res = (struct nfsdiropres *) malloc(sizeof(struct nfsdiropres));
 
	if (res == NULL) {
 
		(void) fprintf(stderr, "nfs: out of memory\n");
 
		abort();			
 
	}
 
	(void) bzero(res, sizeof(struct nfsdiropres));
 
	fullpath = intopn(args->da_fhandle.f.fh_fno);	/* path from inode */
 
							/* of parent */
 
	if (fullpath != NULL) {
 
		/* 
 
		 * extra code here to handle XCOPY's brain-damaged
 
		 * way of passing wildcards when it really shouldn't.
 
		 */
 
		register int i;
 
		register int namelen;
 

 
		namelen = strlen(args->da_name);
 
		for(i = 0; i < namelen; i++) {
 
			if ((args->da_name)[i] == '*' || 
 
			    (args->da_name)[i] == '?')
 
				res->dr_status = NFSERR_NOENT;
 
				goto end;
 
		}
 
		/* end of extra code */
 
		
 
		if (strcmp(args->da_name, "..") == 0) {	/* asking for parent */
 
			(void) free(fullpath);
 
			fullpath = intopn(args->da_fhandle.p.fh_fno);
 
		}
 
		else if (strcmp(args->da_name , ".") != 0) {
 
			(void) strcat(fullpath, "\\");	/* append seperator */
 
			(void) strcat(fullpath, args->da_name);	
 
			(void) strtolower(fullpath);	/* all lower case */
 
		}
 
	}
 
#ifdef DEBUG
 
	(void) printf("fullpath = %s\n", fullpath);
 
#endif
 
	if (! file_getattr(fullpath, &attr)) {
 
#ifdef DEBUG
 
		(void) fprintf(stderr, "nfs_lookup: no such file\n");
 
#endif
 
		res->dr_status = NFSERR_NOENT;
 
	}
 
	else {		/* copy the attributes over */
 
		res->dr_attr = attr;
 
		res->dr_status = NFS_OK;		/* set proper status */
 
		/* get a fhandle for it */
 
		if (strcmp(args->da_name, ".") != 0)
 
			res->dr_fhandle = pntofh(fullpath);
 
		else
 
			res->dr_fhandle = args->da_fhandle;
 
	}
 
	/* reply to caller */
 
end:	if (! svc_sendreply(xprt, xdr_diropres, res))
 
		(void) fprintf(stderr, "nfs_lookup: cannot reply\n");
 
	else if (NFS_VERBOSE)
 
		(void) printf(">>> NFS_LOOKUP: %s\n", fullpath);
 

 
	/* free used space */
 
	svc_freeargs(xprt, xdr_diropargs, args);
 
	(void) free(res);
 
	if (fullpath != NULL)
 
		(void) free(fullpath);
 
}
 

 
/*
 
 *   void nfs_write(SVCXPRT *xprt, struct svc_req *req) --
 
 *      Do an atomic write operation.
 
 */
 
void nfs_write(xprt, req)
 
	SVCXPRT *xprt;
 
	struct svc_req *req;
 
{
 
	struct nfswriteargs *args;		/* args */
 
	struct nfsattrstat *res;		/* return stat */
 
	char *fullname;				/* full path name of file */
 

 
	/* alloc space for args */
 
	args = (struct nfswriteargs *) malloc(sizeof(struct nfswriteargs));
 
	if (args == NULL) {
 
		(void) fprintf(stderr, "nfs: out of memory\n");
 
		abort();
 
	}
 
	(void) bzero(args, sizeof(struct nfswriteargs));
 
	if (! svc_getargs(xprt, xdr_writeargs, args)) {
 
		(void) fprintf(stderr, "nfs_write: cannot decode\n");
 
		svcerr_decode(xprt);
 
		(void) free(args);
 
		return;
 
	}
 
	if (NFS_READONLYFS) {
 
		nfserr_readonlyfs(xprt, xdr_writeargs, args);
 
		return;
 
	}
 
	/* alloc space for return struct */
 
	res = (struct nfsattrstat *) malloc(sizeof(struct nfsattrstat));
 
	if (res == NULL) {
 
		(void) fprintf(stderr, "nfs: out of memory\n");
 
		abort();
 
	}
 
	(void) bzero(res, sizeof(struct nfsattrstat));
 
	fullname = intopn(args->wa_fhandle.f.fh_fno);
 
	res->ns_status =  file_write(fullname, args->wa_offset, args->wa_count, 
 
			 	     args->wa_data);
 
	if (res->ns_status == NFS_OK) {
 
		/* get file attributes */
 
		if (! file_getattr(fullname, &(res->ns_attr))) {
 
			(void) printf("nfs_write: cannot get attributes\n");
 
			res->ns_status = NFSERR_IO;
 
		}
 
	}
 
	if (! svc_sendreply(xprt, xdr_attrstat, res))
 
		(void) fprintf(stderr, "nfs_write: cannot reply\n");
 
	else if (NFS_VERBOSE)
 
		(void) printf(">>> NFS_WRITE: %s\n", fullname);
 

 
	svc_freeargs(xprt, xdr_writeargs, args);	/* free all data */
 
	(void) free(res);
 
	if (fullname != NULL)
 
		(void) free(fullname);
 
}
 

 
/*
 
 *  void nfs_create(SVCXPRT *xprt, struct svc_req *req) --
 
 *      Create a file.
 
 */
 
void nfs_create(xprt, req)
 
	SVCXPRT *xprt;
 
	struct svc_req *req;
 
{
 
	struct nfscreatargs *args;			/* create args */
 
	struct nfsdiropres *res;			/* return result */
 
	char *fullpath;					/* full name */
 
	int stat;					/* create return stat */
 

 
	args = (struct nfscreatargs *) malloc(sizeof(struct nfscreatargs));
 
	if (args == NULL) {
 
		(void) fprintf(stderr, "nfs: out of memory\n");
 
		abort();
 
	}
 
	(void) bzero(args, sizeof(struct nfscreatargs));
 
	if (! svc_getargs(xprt, xdr_creatargs, args)) {
 
		(void) fprintf(stderr, "nfs_creatargs: cannot decode\n");
 
		svcerr_decode(xprt);
 
		(void) free(args);
 
		return;
 
	}
 
	if (NFS_READONLYFS) {		/* read only file system */
 
		nfserr_readonlyfs(xprt, xdr_creatargs, args);
 
		return;
 
	}
 
	/* alloc space for return struct */
 
	res = (struct nfsdiropres *) malloc(sizeof(struct nfsdiropres));
 
	if (res == NULL) {
 
		(void) fprintf(stderr, "nfs: out of memory\n");
 
		abort();
 
	}
 
	(void) bzero(res, sizeof(struct nfsdiropres));
 
	fullpath = intopn(args->ca_da.da_fhandle.f.fh_fno);
 
	if (fullpath != NULL) {
 
		(void) strcat(fullpath, "\\");			/* make rest of name */
 
		(void) strcat(fullpath, args->ca_da.da_name);
 
	}
 
	if (file_getattr(fullpath, &(res->dr_attr))) 	/* file exists */
 
		res->dr_status = NFSERR_EXIST;
 
	/* create a file */
 
	res->dr_status = file_create(fullpath, &(args->ca_sa), &(res->dr_attr));
 
	if (res->dr_status == NFS_OK) {			/* no errors */
 
		/* make file handle */
 
		res->dr_fhandle = pntofh(fullpath);
 
	}
 
	if (! svc_sendreply(xprt, xdr_diropres, res))
 
		(void) fprintf(stderr, "nfs_create: cannot reply\n");
 
	else if (NFS_VERBOSE)
 
		(void) printf(">>> NFS_CREATE: %s\n", fullpath);
 

 
	/* free all data */
 
	svc_freeargs(xprt, xdr_creatargs, args);
 
	(void) free(res);
 
	if (fullpath != NULL)
 
		(void) free(fullpath);
 
}
 
	
 
/*
 
 *  void nfs_mkdir(SVCXPRT *xprt, struct svc_req *req) --
 
 *      Make a directory request.  All DOS directories are readable and
 
 *	writeable, hence the set attribute field is ignored.
 
 */
 
void nfs_mkdir(xprt, req)
 
	SVCXPRT *xprt;
 
	struct svc_req *req;
 
{
 
	struct nfscreatargs *args;
 
	struct nfsdiropres *res;
 
	char *fullpath;
 

 
	args = (struct nfscreatargs *) malloc(sizeof(struct nfscreatargs));
 
	if (args == NULL) {
 
		(void) fprintf(stderr, "nfs: out of memory\n");
 
		abort();
 
	}
 
	(void) bzero(args, sizeof(struct nfscreatargs));
 
	if (! svc_getargs(xprt, xdr_diropargs, args)) {
 
		(void) fprintf(stderr, "nfs_mkdir: cannot decode\n");
 
		svcerr_decode(xprt);
 
		(void) free(args);
 
		return;
 
	}
 
	if (NFS_READONLYFS) {
 
		nfserr_readonlyfs(xprt, xdr_diropargs, args);
 
		return;
 
	}
 
	/* construct path name of new directory */
 
	fullpath = intopn(args->ca_da.da_fhandle.f.fh_fno);
 
	if (fullpath != NULL) {
 
		(void) strcat(fullpath, "\\");
 
		(void) strcat(fullpath, args->ca_da.da_name);
 
	}
 
	res = (struct nfsdiropres *) malloc(sizeof(struct nfsdiropres));
 
	if (res == NULL) {
 
		(void) fprintf(stderr, "nfs: out of memory\n");
 
		abort();
 
	}
 
	(void) bzero(res, sizeof(struct nfsdiropres));
 

 
	res->dr_status = mkdir(fullpath);
 
	if (res->dr_status == NFS_OK)
 
		res->dr_status = !file_getattr(fullpath, &(res->dr_attr));
 

 
	if (res->dr_status == NFS_OK) 		/* make new file handle */
 
		res->dr_fhandle = pntofh(fullpath);
 
	else
 
		res->dr_status = errno;
 
	if (! svc_sendreply(xprt, xdr_diropres, res)) 
 
		(void) fprintf(stderr, "nfs_mkdir: cannot reply\n");
 
	else if (NFS_VERBOSE)
 
		(void) printf(">>> NFS_MKDIR: %s\n", fullpath);
 

 
	/* free all data */
 
	svc_freeargs(xprt, xdr_creatargs, args);
 
	(void) free(res);
 
	if (fullpath != NULL)
 
		(void) free(fullpath);
 
}
 

 
/*
 
 *  void nfs_remove(SVCXPRT *xprt, struct svc_req *req) --
 
 *      Remove a file specified by the handle.
 
 */
 
void nfs_remove(xprt, req)
 
	SVCXPRT *xprt;
 
	struct svc_req *req;
 
{
 
	struct nfsdiropargs *args;		/* dir op arguments */
 
	enum nfsstat stat;			/* status of the remove */
 
	char *fullpath;				/* full path of file */
 

 
	args = (struct nfsdiropargs *) malloc(sizeof(struct nfsdiropargs));
 
	if (args == NULL) {
 
		(void) fprintf(stderr, "nfs: out of memory\n");
 
		abort();
 
	}
 
	(void) bzero(args, sizeof(struct nfsdiropargs));	/* zero it */
 
	if (! svc_getargs(xprt, xdr_diropargs, args)) {
 
		(void) fprintf(stderr, "nfs_remove: cannot decode args\n");
 
		svcerr_decode(xprt);
 
		(void) free(args);		/* couldn't decode it */
 
		return;
 
	}
 
	if (NFS_READONLYFS) {
 
		nfserr_readonlyfs(xprt, xdr_diropargs, args);
 
		return;
 
	}
 
	/* get full directory name from inode number */
 
	fullpath = intopn(args->da_fhandle.f.fh_fno);
 
	if (fullpath == NULL)
 
		stat = NFSERR_NOENT;
 
	else {
 
		(void) strcat(fullpath, "\\");	/* append the name */
 
		(void) strcat(fullpath, args->da_name);
 
		stat = file_unlink(fullpath);
 
	}
 
	/* now reply to request */
 
	if (! svc_sendreply(xprt, xdr_enum, &stat))
 
		(void) fprintf(stderr, "nfs_remove: cannot reply\n");
 
	else if (NFS_VERBOSE)
 
		(void) printf(">>> NFS_REMOVE: %s\n", fullpath);
 

 
	svc_freeargs(xprt, xdr_diropargs, args);
 
	if (fullpath != NULL)
 
		(void) free(fullpath);
 
}
 

 
/*
 
 *  void nfs_rmdir(SVCXPRT *xprt, struct svc_req *req) --
 
 *      Remove a directory specified by the handle.
 
 */
 
void nfs_rmdir(xprt, req)
 
	SVCXPRT *xprt;
 
	struct svc_req *req;
 
{
 
	struct nfsdiropargs *args;		/* dir op arguments */
 
	enum nfsstat stat;			/* status of the remove */
 
	char *fullpath;				/* full path of file */
 

 
	args = (struct nfsdiropargs *) malloc(sizeof(struct nfsdiropargs));
 
	if (args == NULL) {
 
		(void) fprintf(stderr, "nfs: out of memory\n");
 
		abort();
 
	}
 
	(void) bzero(args, sizeof(struct nfsdiropargs));	/* zero it */
 
	if (! svc_getargs(xprt, xdr_diropargs, args)) {
 
		(void) fprintf(stderr, "nfs_remove: cannot decode args\n");
 
		svcerr_decode(xprt);
 
		(void) free(args);		/* couldn't decode it */
 
		return;
 
	}
 
	/* get full path name from inode number */
 
	fullpath = intopn(args->da_fhandle.f.fh_fno);
 
	if (fullpath == NULL)
 
		stat = NFSERR_NOENT;			/* doesn't exist */
 
	else {
 
		(void) strcat(fullpath, "\\");		/* append the name */
 
		(void) strcat(fullpath, args->da_name);
 
		stat = (enum nfsstat) rmdir(fullpath);	/* remove it */
 
	}
 
	/* now reply to request */
 
	if (! svc_sendreply(xprt, xdr_enum, &stat))
 
		(void) fprintf(stderr, "nfs_remove: cannot reply\n");
 
	else if (NFS_VERBOSE)
 
		(void) printf(">>> NFS_RMDIR: %s\n", fullpath);
 

 
	svc_freeargs(xprt, xdr_diropargs, args);	/* free data */
 
	if (fullpath != NULL)
 
		(void) free(fullpath);
 
}
 

 
/*
 
 *  void nfs_rename(SVCXPRT *xprt, struct svc_req *req) --
 
 *      Renames given file to new name specified in the args.
 
 */
 
void nfs_rename(xprt, req)
 
	SVCXPRT *xprt;
 
	struct svc_req *req;
 
{
 
	struct nfsrnmargs *args;		/* arguments to rename */
 
	enum nfsstat stat;			/* thr reply status */
 
	char *oldname, *newname;		/* old and new filenames */
 

 
	args = (struct nfsrnmargs *) malloc(sizeof(struct nfsrnmargs));
 
	if (args == NULL) {
 
		(void) fprintf(stderr, "nfs: out of memory\n");
 
		abort();
 
	}
 
	(void) bzero(args, sizeof(struct nfsrnmargs));
 
	if (! svc_getargs(xprt, xdr_rnmargs, args)) {
 
		(void) fprintf(stderr, "nfs_rename: cannot decode\n");
 
		svcerr_decode(xprt);
 
		(void) free(args);
 
		return;
 
	}
 
	if (NFS_READONLYFS) {
 
		nfserr_readonlyfs(xprt, xdr_rnmargs, args);
 
		return;
 
	}
 
	/* make old name from inode */
 
	oldname = intopn(args->rna_from.da_fhandle.f.fh_fno);
 
	newname = intopn(args->rna_to.da_fhandle.f.fh_fno);
 
	if (oldname == NULL || newname == NULL)	/* cannot find path for file */
 
		stat = NFSERR_STALE;		/* ==> stale file handle */
 
	else {
 
		/* complete specification for names */
 
		(void) strcat(oldname, "\\");
 
		(void) strcat(oldname, args->rna_from.da_name);
 
		(void) strcat(newname, "\\");
 
		(void) strcat(newname, args->rna_to.da_name);
 
		stat = (enum nfsstat) rename(oldname, newname);
 
	}
 
	/* reply to rename request */
 
	if (! svc_sendreply(xprt, xdr_enum, &stat))
 
		(void) fprintf(stderr, "nfs_rename: cannot reply\n");
 
	else if (NFS_VERBOSE)
 
		(void) fprintf(stderr, ">>> NFS_RENAME: %s to %s\n", 
 
				oldname, newname);
 
	
 
	/* free all data */
 
	svc_freeargs(xprt, xdr_rnmargs, args);
 
	(void) free(newname);
 
	(void) free(oldname);
 
}
 

 
/*
 
 *  void nfs_setattr(SVCXPRT *xprt, struct svc_req *req) --
 
 *      Set file attributes.
 
 */
 
void nfs_setattr(xprt, req)
 
	SVCXPRT *xprt;
 
	struct svc_req *req;
 
{
 
	struct nfssaargs *args;			/* arguments */
 
	struct nfsattrstat *res;		/* results */
 
	char *fullpath;				/* full path name */
 

 
	args = (struct nfssaargs *) malloc(sizeof(struct nfssaargs));
 
	if (args == NULL) {
 
		(void) fprintf(stderr, "nfs: out of memory\n");
 
		abort();
 
	}
 
	(void) bzero(args, sizeof(struct nfssattr));
 
	if (! svc_getargs(xprt, xdr_saargs, args)) {
 
		(void) fprintf(stderr, "nfs_setattr: cannot decode\n");
 
		svcerr_decode(xprt);
 
		(void) free(args);
 
		return;
 
	}
 
	if (NFS_READONLYFS) {
 
		nfserr_readonlyfs(xprt, xdr_saargs, args);
 
		return;
 
	}
 
	fullpath = intopn(args->saa_fh.f.fh_fno);	/* get file name */
 
	if (args->saa_sa.sa_mode & UCHK_WR)		/* if write perm.... */
 
		file_setperm(fullpath, UPERM_REG);	/* regular file */
 
	else
 
		file_setperm(fullpath, UPERM_RDONLY);	/* rd only file */
 

 
	if (args->saa_sa.sa_mtime.tv_sec != 0)
 
		file_settime(fullpath, args->saa_sa.sa_mtime.tv_sec);
 
	
 
	res = (struct nfsattrstat *) malloc(sizeof(struct nfsattrstat));
 
	if (res == NULL) {
 
		(void) fprintf(stderr, "nfs: out of memory\n");
 
		abort();
 
	}
 
	(void) bzero(res, sizeof(struct nfsattrstat));
 
	if (! file_getattr(fullpath, &(res->ns_attr))) {
 
#ifdef DEBUG
 
		(void) fprintf(stderr, "nfs_setattr: cannot get attr\n");
 
#endif
 
		res->ns_status = NFSERR_NOENT;
 
	}
 
	else
 
		res->ns_status = NFS_OK;
 

 
	if (! svc_sendreply(xprt, xdr_attrstat, res))
 
		(void) fprintf(stderr, "nfs_setattr: cannot reply\n");
 
	else if (NFS_VERBOSE)
 
		(void) printf(">>> NFS_SETATTR: %s\n", fullpath);
 
	
 
	/* free data */
 
	svc_freeargs(xprt, xdr_saargs, args);
 
	(void) free(res);
 
	if (fullpath != NULL)
 
		(void) free(fullpath);
 
}
 
	
 
/*
 
 *  void nfserr_readonlyfs(SVCXPRT *xprt, xdrproc_t xproc, void *args) --
 
 *      Return error status of NFSERR_ROFS.   Write attempted on read only
 
 *	file system.
 
 */
 
void nfserr_readonlyfs(xprt, xproc, args)
 
	SVCXPRT *xprt;
 
	xdrproc_t xproc;
 
	void *args;
 
{
 
	enum nfsstat err;
 

 
	err = NFSERR_ROFS;
 
	if (! svc_sendreply(xprt, xdr_enum, &err))
 
		(void) fprintf(stderr, "nfserr: cannot send reply\n");
 
	svc_freeargs(xprt, xproc, args);
 
}
 

 
