/*
 
 *  files.c --
 
 *      File manipulation procedures.  This module is highly machine
 
 *      dependent.
 
 *
 
 *  Author:
 
 *      See-Mong Tan
 
 */
 

 
#include "common.h"
 
#include "msc-dos.h"		/* for dos drive and file routines */
 
#include <direct.h>		/* for dos directory ops */
 
#include <fcntl.h>		/* these ... */
 
#include <sys/stat.h>		/* for ... */
 
#include <io.h>			/* low level DOS access */
 

 
#define DRIVE_SCALINGFACTOR 2	/* scaling factor for dos drive blocks */
 

 
/*
 
 *  bool_t file_getattr(char *path, struct nfsfattr *attr) --
 
 *      Gets attributes for designated file or directory and puts
 
 *      the information in *attr.  Returns TRUE if successful, FALSE otherwise.
 
 *	Adds file to directory tree if the file does not already exist in
 
 *	the tree.
 
 */
 
bool_t file_getattr(path, attr)
 
	char *path;
 
	struct nfsfattr *attr;
 
{
 
	struct find_t findbuf;
 
	u_int f_attribs, time, date;	/* attribs and time of last write */
 
	int handle;			/* DOS file handle */
 

 
	if (path == NULL)
 
		return FALSE;
 

 
	if (_dos_findfirst(path, _A_NORMAL | _A_SUBDIR | _A_RDONLY,
 
				&findbuf) != 0) {
 
		/* not successful */
 
#ifdef DEBUG
 
		(void) fprintf(stderr, 
 
			"file_getattr: cannot get attr, errno = %d\n", errno);
 
#endif
 
		return FALSE;
 
	}
 
	(void) bzero(attr, sizeof(struct nfsfattr));	/* zero return buf */
 

 
	/* get file attributes and time of the last write */
 
	if (_dos_getfileattr(path, &f_attribs) != 0) {
 
		(void) fprintf(stderr, "nfs: cannot get file attributes\n");
 
		abort();		/* must be io err then */
 
	}
 
	  /* file protection bits and type */
 
	if (f_attribs == _A_RDONLY) {		/* rdonly */
 
		attr->na_type = NFREG;
 
		attr->na_mode = UPERM_RDONLY;
 
		attr->na_nlink = 1;
 
	}
 
	else if (f_attribs == _A_SUBDIR) {   	/* subdirectory */
 
		attr->na_type = NFDIR;
 
		attr->na_mode = UPERM_SUBDIR;
 
		attr->na_nlink = file_nsubdirs(path) + 2; 
 
				/* # of subdirectories plus this one */
 
	}
 
	else {
 
		attr->na_type = NFREG;	  	/* normal file */
 
		attr->na_mode = UPERM_REG;
 
		attr->na_nlink = 1;
 
	}
 

 
	  /* file size in bytes */
 
	if (f_attribs == _A_SUBDIR) {        /* directory */
 
		attr->na_blocks = 1;         /* just say 1 block */
 
		attr->na_size = 1024;
 
	}
 
	else {
 
		attr->na_size = findbuf.size;
 
		attr->na_blocks = findbuf.size / BASIC_BLOCKSIZE + 
 
				(findbuf.size % BASIC_BLOCKSIZE == 0 ? 0 : 1);
 
	}
 
	  /* owner and group id - superuser */
 
	attr->na_uid = 0;
 
	attr->na_gid = 0;
 
	  /* preferred transfer size in blocks */
 
	attr->na_blocksize = NFS_MAXDATA;
 
	  /* device # == drive # */
 
	attr->na_fsid = (int) *path - (int) 'a' + 1;
 
	attr->na_rdev = attr->na_fsid;
 
	  /* inode # */
 
	if ((attr->na_nodeid = pntoin(path)) == -1)
 
		attr->na_nodeid = addpathtodirtree(path);
 
	  /* time of last access */
 
	attr->na_atime.tv_usec = 0;
 
	attr->na_atime.tv_sec = unixtime(findbuf.wr_time, findbuf.wr_date);
 
	  /* time of last write */
 
	attr->na_mtime = attr->na_atime;       /* note all times are the same */
 
	  /* time of last change */
 
	attr->na_ctime = attr->na_atime;
 

 
	return TRUE;
 
}
 

 
/*
 
 *  int file_nsubdirs(char *path) --
 
 *	Returns # of subdirectories in the given directory.
 
 */
 
int file_nsubdirs(path)
 
	char *path;
 
{
 
	int subdirs = 0;
 
	struct find_t ft;
 
	char name[MAXPATHNAMELEN];
 
#define VALID(ft) (strcmp((ft).name, ".") != 0 && strcmp((ft).name, "..") != 0)
 

 
	(void) strcpy(name, path);
 
	(void) strcat(name, "\\*.*");		/* append wildcard */
 
	if (_dos_findfirst(name, _A_SUBDIR, &ft) != 0)
 
		if (VALID(ft))
 
			subdirs++;
 
	else
 
		return 0;
 

 
	while(_dos_findnext(&ft) != 0)
 
		if (VALID(ft))
 
			subdirs++;
 

 
	return subdirs;
 
#undef VALID
 
}
 
	
 
/*
 
 *  long file_freeblocks(int drive, long *free, long *total) --
 
 *      Return # of free blocks in specified filesystem (ie. drive).
 
 */
 
long file_freeblocks(drive, free, total)
 
	int drive;
 
	long *free, *total;
 
{
 
	struct diskfree_t df;	/* disk free */
 

 
	if (_dos_getdiskfree(drive, &df) != 0) {	/* dos call */
 
		(void) fprintf(stderr, "freeblocks: err, cannot read\n");
 
		return 0;
 
	}
 
	*free = (df.avail_clusters * df.sectors_per_cluster) 
 
		/ DRIVE_SCALINGFACTOR;
 
	*total = (df.total_clusters * df.sectors_per_cluster) 
 
		 / DRIVE_SCALINGFACTOR;
 

 
	return *free;
 
}
 

 
/* a file pointer cache for read requests */
 
#define FRDCSIZ 5
 
static struct {
 
	char pn[MAXPATHNAMELEN];
 
	FILE *fp;
 
} frdc_cache[FRDCSIZ];		/* up to five cached file handles */
 

 
static int frdc_last;		/* last location saved */
 

 
/*
 
 *  void frdc_save(char *path, FILE *fp) --
 
 *	Cache read file pointers.
 
 */
 
void frdc_save(path, fp)
 
	char *path;
 
	FILE *fp;
 
{
 
	if (frdc_cache[frdc_last].fp != NULL)
 
		(void) fclose(frdc_cache[frdc_last].fp);  	/* throw away */
 
	(void) strcpy(frdc_cache[frdc_last].pn, path);
 
	frdc_cache[frdc_last].fp = fp;
 
	frdc_last += 1;
 
	if (frdc_last == FRDCSIZ)
 
		frdc_last = 0;
 
}
 

 
/*
 
 *  void frdc_del(char *path) --
 
 *	Delete saved file pointer from read cache.  No effect is file
 
 *	was not cached.  Closes file pointer also.
 
 */
 
void frdc_del(path)
 
	char *path;
 
{
 
	int i;
 

 
	for(i = 0; i < FRDCSIZ; i++) {
 
		if (frdc_cache[i].fp != NULL &&
 
		    strcmp(frdc_cache[i].pn, path) == 0) {
 
			(void) fclose(frdc_cache[i].fp);
 
			frdc_cache[i].fp = NULL;
 
			return;
 
		}
 
	}
 
}
 

 
/*
 
 *  FILE *frdc_find(char *path) --
 
 *	Finds cached file pointer corresponding to path, or NULL
 
 *	If no such file exists.
 
 */
 
FILE *frdc_find(path)
 
	char *path;
 
{
 
	int i;
 

 
	for(i = 0; i < FRDCSIZ; i++) {
 
		if (frdc_cache[i].fp != NULL && 
 
		    strcmp(frdc_cache[i].pn, path) == 0)
 
			return frdc_cache[i].fp;
 
	}
 

 
	return NULL;
 
}
 

 
/*
 
 *  int file_read(char *path, u_long offset, u_long count, char *buffer) --
 
 *	Reads count bytes at offset into buffer.  Returns # of bytes read,
 
 *      and -1 if an error occurs, or 0 for EOF or null file.
 
 */
 
int file_read(path, offset, count, buffer)
 
	char *path;
 
	u_long offset, count;
 
	char *buffer;
 
{
 
	FILE *fp;
 
	bool_t saved = FALSE;
 
	int bytes = 0;
 

 
	if ((fp = frdc_find(path)) != NULL) {
 
		saved = TRUE;
 
#ifdef DEBUG
 
		(void) printf("Saved\n");
 
#endif
 
	}
 
	else if ((fp = fopen(path, "rb")) == NULL)	/* read in binary mode */
 
		return -1;
 
	if (fseek(fp, (long) offset, SEEK_SET) != 0) {
 
		if (! feof(fp)) {
 
			(void) fclose(fp);
 
			return -1;
 
		}
 
		else
 
			return 0;
 
	}
 
	bytes = fread(buffer, sizeof(char), count, fp);
 
	if (! saved)
 
		frdc_save(path, fp);
 

 
	return bytes;
 
}
 
	
 
/*
 
 *  fhandle_t pntofh(char *path) --
 
 *      Converts path name to file handle.  DOS or UNIX style paths.
 
 */
 
fhandle_t pntofh(path)
 
	char *path;
 
{
 
	u_long inodeno;
 
	fhandle_t fh;
 

 
	(void) bzero(&fh, sizeof(fhandle_t));
 
	if ((inodeno = pntoin(path)) == -1)
 
		inodeno = addpathtodirtree(path);
 
	fh.f.fh_fno = inodeno;
 
	fh.p.fh_fno = parentinode(inodeno);
 
	if (isalpha(*path))
 
		fh.f.fh_fsid = fh.p.fh_fsid = (int) *path - (int) 'a' + 1;
 
	else
 
		fh.f.fh_fsid = fh.p.fh_fsid = (int) *(path + 1) - 'a' + 1;
 
	(void) strcpy(fh.fh_pn, intoname(inodeno));
 
	
 
	return fh;
 
}
 

 
/*
 
 *  int file_rddir(char *path, int offs, struct udirect *udp) --
 
 *      Put file information at offs in directory at path in nfs cookie 
 
 *	at *udp. Returns # of bytes in record, 0 if there are no more entries
 
 *   	or -1 for a read error.
 
 */
 
int file_rddir(path, offs, udp)
 
	char *path;
 
	int offs;
 
	struct udirect *udp;
 
{
 
	int i;
 
	char name[MAXPATHNAMELEN];
 
	static struct {
 
		char path[MAXPATHNAMELEN];	/* path name */
 
		struct find_t ft;		/* dos find struct */
 
		int offs;			/* it's offset */
 
	} ftsv;					/* saved in a cache */
 
#define SUD 	sizeof(struct udirect)		/* full udp cookie size */
 

 
	if (strncmp(path, ftsv.path, strlen(path)) == 0 && 
 
	    offs == ftsv.offs) {
 
		/* do nothing - return struct already read in */
 
	}
 
	else {
 
		(void) sprintf(ftsv.path, "%s\\*.*", path);  /* wildcard */
 
		if (_dos_findfirst(ftsv.path, _A_NORMAL | _A_SUBDIR |
 
          	      		   _A_RDONLY, &(ftsv.ft)) != 0)
 
			return -1;			/* error reading */
 
		for(i = 1; i <= offs / SUD; i++) {	/* cdr down to find */
 
			if (_dos_findnext(&(ftsv.ft)) != 0) {
 
				return -1;
 
			}
 
		}
 
	}
 
	/* stuff udp cookie with info */
 
	(void) strtolower(ftsv.ft.name);		/* to lower case */
 
	(void) sprintf(name, "%s\\%s", path, ftsv.ft.name);  /* full path */
 
	if ((udp->d_fileno = pntoin(name)) == -1)	/* get filenumber */
 
		udp->d_fileno = addpathtodirtree(name);
 
	udp->d_namlen = strlen(ftsv.ft.name);		/* copy name over */
 
	(void) strcpy(udp->d_name, ftsv.ft.name);
 
	udp->d_offset = offs + SUD;		/* this offset */
 

 
	udp->d_reclen = UDIRSIZ(udp);			/* size of this rec */
 

 
	/* now see if there is one more entry */
 
	if (_dos_findnext(&(ftsv.ft)) != 0)
 
		return 0;				/* is end of file */
 
	else {
 
		ftsv.offs = offs + SUD;
 
		return udp->d_reclen;			/* still more entries */
 
	}
 
#undef SUD
 
}
 
	
 
/*
 
 *  char *strtolower(char *s) --
 
 *	Converts all characters in s to lower case.  Returns s.
 
 */
 
char *strtolower(s)
 
	char *s;
 
{
 
	char *tmp;
 

 
	tmp = s;
 
	while(*s != '\0') {
 
		if (isalpha(*s) && isupper(*s))
 
			*s = tolower(*s);
 
		s++; 
 
	}
 

 
	return tmp;
 
}
 

 
/*
 
 *  int file_write(char *name, u_long offset, u_long count, char *buffer) --
 
 *      Write to file with name at offset, count bytes of data from buffer.
 
 *	File should already exist.  Returns 0 for success, or some error 
 
 *	code if not.
 
 */
 
int file_write(name, offset, count, buffer)
 
	char *name;
 
	u_long offset;
 
	u_long count;
 
	char *buffer;
 
{
 
	int handle;			/* write file handle */
 
	long newoff;
 

 
	frdc_del(name);			/* delete from read cache */
 
	/* open for writing only */
 
	handle = open(name, O_WRONLY | O_BINARY);
 
	if (handle == -1)
 
		return errno;		/* return error code */
 
	newoff = lseek(handle, offset, SEEK_SET);
 
	if (write(handle, buffer, count) == -1) {
 
		(void) close(handle);
 
		return errno;		/* some error */
 
	}
 
	(void) close(handle);
 

 
	return (int) NFS_OK;
 
}
 
		
 
/*
 
 *  int file_create(char *name, struct nfssattr *sattr, struct nfsfattr *fattr)
 
 *      Creates a file with full path name and attributes sattr, and returns
 
 *      the file attributes in *fattr.  Adds file to directory tree.
 
 *      Returns NFS_OK for success, or some error code for failure.
 
 */
 
int file_create(name, sattr, fattr)
 
	char *name;
 
	struct nfssattr *sattr;
 
	struct nfsfattr *fattr;
 
{
 
	int handle;			/* file handle */
 
	int sattribs = S_IREAD;		/* set attributes */
 

 
	if (name == NULL)
 
		return NFSERR_NOENT;
 

 
	if (sattr->sa_mode & UCHK_WR)	   /* file is writeable */
 
		sattribs |= S_IWRITE;	   /* set DOS file to be read & write */
 

 
	frdc_del(name);			   /* delete from read cache */
 
	 /* check if file already exists */
 
	if ((handle = open(name, O_CREAT | O_TRUNC, sattribs)) == EEXIST)
 
		return NFSERR_EXIST;
 
	close(handle);
 
	
 
	/* stuff info into file return attribute cookie and add to dir tree */
 
	if (! file_getattr(name, fattr)) 
 
		return (int) NFSERR_IO;	  /* just created but not found now! */
 

 
	return (int) NFS_OK;
 
}
 

 
/*
 
 *  int file_setperm(char *path, long perm) --
 
 *      Sets file permissions depending on perm.  Perm is of two types,
 
 *      UPERM_REG (regular file) or UPERM_RDONLY (read only).
 
 *	Returns 0 for success or some error code if an error occurs.
 
 */
 
int file_setperm(path, perm)
 
	char *path;
 
	long perm; /* Used to be int, incorrectly */
 
{
 
	int stat, attribs;
 

 
	if (perm == UPERM_RDONLY)
 
		attribs = _A_RDONLY;
 
	else
 
		attribs = _A_NORMAL;
 

 
	stat = _dos_setfileattr(path, attribs);
 
	if (stat == 0)
 
		return 0;
 
	else
 
		return errno;
 
}	
 
		
 
/*
 
 *  int file_settime(char *path, long secs) --
 
 *      Sets file time specified by secs (# of seconds since 0000, Jan 1, 1970.
 
 *	Returns 0 for success or some error code if an error occurs.
 
 */
 
int file_settime(path, secs)
 
	char *path;
 
	long secs;
 
{
 
	struct tm *t;
 
	int stat, handle;
 
	unsigned date, time;
 

 
	/* must open file to change perms */
 
	if ((handle = open(path, O_CREAT | O_EXCL, S_IWRITE)) == -1) {
 
		if (errno != EEXIST) 		/* file exists */
 
			return errno;
 
	}
 
	t = gmtime((time_t *) &secs);	/* convert secs to time struct */
 
	if (t == NULL)
 
		return 1;
 

 
	date = t->tm_mday + (t->tm_mon << 5) + (t->tm_year << 9);
 
	time = t->tm_sec + (t->tm_min << 5) + (t->tm_hour << 11);
 
	stat = _dos_setftime(handle, date, time);
 
	(void) close(handle);
 
	if (t != NULL)			/* free alloc'ed time struct */
 
		(void) free(t);
 
	if (stat != 0)
 
		return errno;
 
	else
 
		return 0;
 
}
 

 
/*
 
 *  int file_unlink(char *name) --
 
 *       Removes named file.
 
 */
 
int file_unlink(name)
 
	char *name;
 
{
 
	frdc_del(name);
 
	return unlink(name);
 
}
 

 
