//*****************************************************************************
//*****************************************************************************
//*****************************************************************************
//*****************************************************************************
//  Windows Sockets FTP Application
//
//  Based on the WS_FTP program written by: John A. Junod
//  Internet: <junodj@gordon-emh2.army.mil>
//
//  This program executable and all source code is released into the public
//  domain.  The primary purpose of this application was to learn what it
//  takes to write a Window Sockets Application and this is NOT a full 
//  implementation of an FTP client.
//
//  WINFTP written by:
//      Santanu Lahiri       Internet: <slahiri@magnus.acs.ohio-state.edu>
//
//  Reworked most of the code in this module into a format similar to 
//  the Microsoft Class Library Naming Convention.  Hopefully this will
//  reduce the effort needed to convert it to an MSVC/C++ app if ever.
//
//  Added the Configuration Entry in the Connect dialog box to allow
//  multiple configurations for the same host.  Also added the "Delete
//  Config" option to delete unwanted configs.  Checking the "Save Config"
//  will now save the current configuration or update an existing one.
//
//  Added the History option to the main window.  This control now tracks
//  the directories being visited for ease of browsing.  This is a Drop-Down
//  combo box where the list box shows the dirs.  Clicking on an entry in the
//  list causes a CHDIR command to be executed.
//
//  Added a Refresh button to allow refreshing the Directory listings.  Next
//  step is to add code to optionally suppress automatic listing of directories
//  and allowing listing of specified file types.
//
//  Converted all Command buttons to simulated buttons to reduce resources.
//  The remaining controls are the list boxes, combo boxes, radio
//  buttons, and static text boxes.  This reduces #controls by about 22.
//
//  MODULE: WS_MAIN.C  (main window functions and some dialog boxes)
//
//*****************************************************************************
//*****************************************************************************
#include "ws_glob.h"
#include "winftp.H"
#include <stdio.h>
#include <stdlib.h> 
#include <direct.h>
#include <dos.h>

#include <shellapi.h>

#define RB_SHOWCHECKS 8100
#define EN_TEXTCHANGE 0x4000

int nView=0;
int selects[256];

BOOL bDebugLog;
static HBRUSH hbrGray1, hbrGray2;

FARPROC lpfnOldLocal,  lpfnNewLocal;
FARPROC lpfnOldRemote, lpfnNewRemote;

LRESULT CALLBACK LocalDirProc  (HWND, UINT, WPARAM, LPARAM);
LRESULT CALLBACK RemoteDirProc (HWND, UINT, WPARAM, LPARAM);

#ifndef USEASYNC
SubProcessAsync (HWND hWnd, UINT Message, WPARAM wParam, LPARAM lParam)
{
  return(FALSE);
}
#endif

BOOL CALLBACK WS_DeleteFileProc (HWND hDlg, UINT Msg, WPARAM wParam, LPARAM lParam);

//*****************************************************************************
//  Main Program
//*****************************************************************************
int PASCAL WinMain(HANDLE hInstance, HANDLE hPrevInstance, LPSTR lpszCmdLine, int nCmdShow)
{
  int        nRc;           // return value from Register Classes
  int        nErr;

#ifdef WIN32
  lstrcpy (szAppName, "WINFTP32");
#else
  lstrcpy (szAppName, "WINFTP16");
#endif
  
  hInst = hInstance;
  hbrGray1 = CreateSolidBrush (RGB (192,192,192));
  hbrGray2 = CreateSolidBrush (RGB (128,128,128));

  if (!hPrevInstance)
  {
    if ((nRc = nCwRegisterClasses()) == -1)
    {
      MessageBox((HWND)NULL, "Window creation failed", NULL, MB_ICONEXCLAMATION);
      return nRc;    
    }
  }

  if ((nErr=InitInstance (hInst))!=IDOK) return nErr;
  if(strstr (lpszCmdLine, "auto")!=NULL) bAutoStart=TRUE;

  switch (nErr=WSAStartup( 0x0101, &WSAData))  // register with winsock tcp/ip API
  {
    case  0: nRc = ExecFTPApp (nCmdShow); break;
    default: ReportWSError ("WSAStartup", nErr);            
  }

  UnsetStatusLines();
  unlink (szCurrentDir);
  CwUnRegisterClasses();

  DeleteObject (hbrGray1);
  DeleteObject (hbrGray2);
  
  return nRc;
}

//*****************************************************************************
//  Initialize this instance of the application
//*****************************************************************************
InitInstance (HINSTANCE hInst)
{
  long       nWndunits;       // window units for size and location
  int        nWidth, nHeight; // resulting width and height for this window
  
  nWndunits = GetDialogBaseUnits();
  nWndx = LOWORD (nWndunits);
  nWndy = HIWORD (nWndunits);
  nWidth = ((250 * nWndx) / 4);
  nHeight = ((222 * nWndy) / 8);

  hWndMain = CreateWindow (szAppName, szDefaultHdr,  WS_OVERLAPPEDWINDOW,
    CW_USEDEFAULT, CW_USEDEFAULT, nWidth, nHeight, (HWND) NULL, (HMENU) NULL, hInst,  NULL);

  if (hWndMain == (HWND) NULL)
  {
    MessageBox((HWND)NULL, "Error registering class", NULL, MB_ICONEXCLAMATION);
    return IDCANCEL;
  }
  return IDOK;
}

//*****************************************************************************
//  Main message loop is in this routine
//*****************************************************************************
ExecFTPApp (int nCmdShow)
{
  MSG msg;
  HWND hLocal;
  HWND hRemote;
  
  GetLocalInfo();
  hLocal  = GetDlgItem (hWndMain, EDT_LFILETYPE);
  hRemote = GetDlgItem (hWndMain, EDT_RFILETYPE);
  lpfnOldLocal  = (FARPROC) GetWindowLong (hLocal,  GWL_WNDPROC);
  lpfnOldRemote = (FARPROC) GetWindowLong (hRemote, GWL_WNDPROC);
  lpfnNewLocal  = (FARPROC) MakeProcInstance ((FARPROC) LocalDirProc,  hInst);
  lpfnNewRemote = (FARPROC) MakeProcInstance ((FARPROC) RemoteDirProc,  hInst);
  SetWindowLong (hLocal,  GWL_WNDPROC, (LONG) lpfnNewLocal);
  SetWindowLong (hRemote, GWL_WNDPROC, (LONG) lpfnNewRemote);
  
  ShowWindow(hWndMain, nCmdShow);         // display main window
  if (bAutoStart) PostMessage (hWndMain, WM_COMMAND, BTN_CONNECT, 0L);
  while(GetMessage (&msg, (HWND) NULL, 0, 0))     // Until WM_QUIT message
  {
    TranslateMessage(&msg);
    DispatchMessage(&msg);
  }
  WSACleanup();
  ReleaseDisplayMem();
  return msg.wParam;
}

//*****************************************************************************
//  Check and despatch messages as necessary
//*****************************************************************************
CheckMsgQueue (HWND hWnd)
{
  MSG Msg;
  int i;
  
  for (i=0; i<10; i++)
  {
    if (!PeekMessage (&Msg, hWnd, 0, 0, PM_REMOVE)) return 0;
    TranslateMessage (&Msg);
    DispatchMessage (&Msg);
  }
  return 0;
}

//*****************************************************************************
// Clean up the files in the temp directory
//*****************************************************************************
void CleanupTempFiles (HWND hWnd)
{
  char szName[_MAX_PATH];
  char szFile[_MAX_PATH];
  int nMax, nI, nLen;
  char *lp;
  
#ifdef WIN32
  HANDLE hFile;
  WIN32_FIND_DATA f;
#else
  struct _find_t f;
#endif

  if (bRetain) return;
  memset (szName, '\0', _MAX_PATH);
  lstrcpy (szName, szTmpViewFile);
  lp = strstr (szName, "%04d");
  if (lp!=NULL)
  {
    lstrcpy (lp, "*.tmp");
    lp = szTempDir;
    if (lstrlen (szTempDir)>0) lp += lstrlen (szTempDir) - 1;
    if (*lp=='\\') *lp = '\0';

#ifdef WIN32
    while ((hFile=FindFirstFile (szName, &f))!=INVALID_HANDLE_VALUE)
    {
      wsprintf (szFile, "%s\\%s", szTempDir, f.cFileName);
      unlink (szFile);
      FindClose (hFile);
    }
#else
    while (_dos_findfirst (szName, _A_NORMAL, &f)==0)
    {
      wsprintf (szFile, "%s\\%s", szTempDir, f.name);
      unlink (szFile);
    }
#endif
  }
  nMax = (int) SendDlgItemMessage (hWnd, LST_DELFILES, LB_GETCOUNT, (WPARAM) 0, (LPARAM) 0L);
  if ((nMax==0)||(nMax==LB_ERR)) return;
  for (nI=0; nI<nMax; nI++)
  {
    nLen = (int) SendDlgItemMessage (hWndMain, LST_DELFILES, LB_GETTEXT, (WPARAM) nI, (LPARAM)(LPCSTR) szFile);
    if (nLen>0) unlink (szFile);
  }
}

//*****************************************************************************
//  If the current directory is not already in the list, add it in.
//*****************************************************************************
LPSTR FindViewerName (LPSTR lpName)
{
  char szBuf[_MAX_PATH];
  int nI;
  
  lstrcpy (szBuf, lpName);
  strupr (szBuf);
  for (nI=0; nI<nViewNum; nI++)
  {
    if (strstr (szBuf, lpVuExt(nI))!=NULL) return (LPSTR) lpVuPgm(nI);
  }
  return szViewer;
}

//*****************************************************************************
//  If the current directory is not already in the list, add it in.
//*****************************************************************************
int FindViewerIndex (LPSTR lpName)
{
  char szBuf[_MAX_PATH];
  int nI;
  
  lstrcpy (szBuf, lpName);
  strupr (szBuf);
  for (nI=0; nI<nViewNum; nI++)
  {
    if (strstr (szBuf, lpVuExt(nI))!=NULL) return nI;
  }
  return -1;
}

//*****************************************************************************
//  If the current directory is not already in the list, add it in.
//*****************************************************************************
AddDirToList (HWND hWnd, int nID, LPSTR lpDir)
{
  if (SendDlgItemMessage (hWnd, nID, CB_FINDSTRINGEXACT, (WPARAM) -1, (LPARAM)(LPCSTR) lpDir)==CB_ERR)
     SendDlgItemMessage (hWnd, nID, CB_ADDSTRING, (WPARAM) 0, (LPARAM)(LPCSTR) lpDir);
  return 0;
}

//*****************************************************************************
//  Directory selected from Local ComboBox dropdown list.  Switch to that dir.
//  This is the "Recall History" function.
//*****************************************************************************
OnCmdLocalChangeDirLst (HWND hWnd)
{
  int nIndex;
  char szMsgBuf[_MAX_PATH];

  if ((nIndex=(int) SendMessage (hLbxLDirLst, CB_GETCURSEL, 0, 0L))!=CB_ERR)
  {
    SendDlgItemMessage (hWnd, LST_LDIRLST, CB_GETLBTEXT, (WPARAM) nIndex, (LPARAM)(LPCSTR) szMsgBuf);
    if (chdir (szMsgBuf)==0) GetLocalDirForWnd (hWnd);
  }
  return 0;
}

//*****************************************************************************
//  Change to selected local directory from the Directory/Drive list box
//*****************************************************************************
OnCmdLocalChangeDir (HWND hWnd)
{
  int nIndex;
  FARPROC lpfnMsgProc; 
  char szMsgBuf[_MAX_PATH];

  if ((nIndex=(int) SendMessage (hLbxLDir, LB_GETCURSEL, 0, 0L))!=LB_ERR)
  {
    SendMessage (hLbxLDir, LB_GETTEXT, nIndex, (LPARAM)(LPCSTR) szMsgBuf);
    if (strncmp (szMsgBuf,"[-",2)==0) 
    {
      setdisk (szMsgBuf[2]-'a'+1);
      GetLocalDirForWnd (hWnd);
      return 0;
    } 
  }
  switch (chdir (szMsgBuf))
  {
    case 0 : _getcwd (szMsgBuf, 80);
             strlwr (szMsgBuf);
             AddDirToList (hWnd, LST_LDIRLST, szMsgBuf);
             GetLocalDirForWnd (hWnd); break;
    default: lstrcpy (szDlgPrompt,"Enter local directory name:");
             memset (szDlgEdit, '\0', sizeof (szDlgEdit));
             lpfnMsgProc = MakeProcInstance ((FARPROC) WS_InputMsgProc, hInst);
             DialogBox (hInst, (LPSTR) "DLG_INPUT", hWnd, lpfnMsgProc);
             FreeProcInstance (lpfnMsgProc);
             if (chdir(szDlgEdit)==0) 
             {
               _getcwd (szMsgBuf, 80);
               strlwr (szMsgBuf);
               AddDirToList (hWnd, LST_LDIRLST, szMsgBuf);
               GetLocalDirForWnd(hWnd);  
             }
  }
  return 0;
}

//*****************************************************************************
//*****************************************************************************
OnCmdLocalToRemote (HWND hWnd)
{
  u_char szTmp[150];
  char szLocalName[_MAX_PATH], szRemoteName[_MAX_PATH];
  int nCount, nRC, nIndex;

  if (!bConnected) return 0;
  bCancelXfer = FALSE;
  bOpInProgress = TRUE;
  nCount = (int) SendMessage (hLbxLFiles, LB_GETSELITEMS, 256, (LPARAM)(int far *) selects);
  if (nCount>0 && nCount!=LB_ERR) 
  {
    
    for (nIndex=0; (nIndex<nCount); nIndex++) 
         SendMessage (hLbxLFiles, LB_SETSEL, (WPARAM) FALSE, MAKELPARAM (selects[nIndex], 0));
    CreateXferWindow();
    for (nIndex=0; (nIndex<nCount)&(!bCancelXfer); nIndex++) 
    {
      SendMessage (hLbxLFiles, LB_GETTEXT, selects[nIndex], (LPARAM)(LPCSTR) szLocalName);
      SendMessage (hLbxLFiles, LB_SETSEL, (WPARAM) TRUE, MAKELPARAM (selects[nIndex], 0));
      lstrcpy (szRemoteName, szLocalName);
      if (bInteractive)
      {
        FARPROC lpfnMsgProc;
        
        wsprintf (szDlgPrompt, "Enter remote file name for %s:", szLocalName);
        lstrcpy (szDlgEdit, szRemoteName);
        lpfnMsgProc = MakeProcInstance ((FARPROC) WS_InputMsgProc, hInst);
        DialogBox (hInst, (LPSTR) "DLG_INPUT", hWnd, lpfnMsgProc);
        FreeProcInstance (lpfnMsgProc);
        lstrcpy (szRemoteName, szDlgEdit);
      }
      wsprintf (szTmp, "Sending %s as %s (%u of %u)", szLocalName, szRemoteName, nIndex+1, nCount);
      DoAddLine (szTmp);
      wsprintf (szTmp, "STOR %s", szRemoteName);
      nRC = SendFile (ctrl_socket, szTmp, szLocalName, fType);
      SendMessage (hLbxLFiles, LB_SETSEL, (WPARAM) FALSE, MAKELPARAM (selects[nIndex], 0));
    }
    DeleteXferWindow();
    GetRemoteDirForWnd(hWnd);
  }
  bOpInProgress = FALSE;
  SendMessage (hTxtLBytes, WM_SETTEXT, (WPARAM) 0, (LPARAM)(LPCSTR) "");
  return 0;
}

//*****************************************************************************
//*****************************************************************************
OnCmdLocalMakeDir (HWND hWnd)
{
  FARPROC lpfnMsgProc;

  lstrcpy(szDlgPrompt, "Enter new local directory name:");
  memset (szDlgEdit, '\0', sizeof (szDlgEdit));
  lpfnMsgProc = MakeProcInstance ((FARPROC) WS_InputMsgProc, hInst);
  DialogBox(hInst, (LPSTR) "DLG_INPUT", hWnd, lpfnMsgProc);
  FreeProcInstance (lpfnMsgProc);
  _mkdir (szDlgEdit);
  GetLocalDirForWnd (hWnd);
  return 0;
}

//*****************************************************************************
//*****************************************************************************
OnCmdLocalRemoveDir (HWND hWnd)
{
  int nIndex;

  if ((nIndex=(int) SendMessage (hLbxLDir, LB_GETCURSEL, 0, 0L))!=LB_ERR)
  {
    SendMessage(hLbxLDir, LB_GETTEXT, nIndex, (LPARAM)(LPCSTR) szMsgBuf);
    wsprintf (szString, "Are you sure you want to delete \"%s\"?", szMsgBuf);
    if (MessageBox (hWnd, szString, "Verify Deletion", MB_YESNO)==IDYES)
    {
      if (_rmdir (szMsgBuf)==0) GetLocalDirForWnd (hWnd);
    }
  }
  return 0;
}

//*****************************************************************************
//*****************************************************************************
OnCmdLocalDeleteFile (HWND hWnd)
{
  int nIndex, nCount, nNum, nRC;
  char szLocalName[_MAX_PATH];
  BOOL bConfirm = TRUE;
  FARPROC lpfnMsgProc;
  
  bCancelXfer = FALSE;
  bOpInProgress = TRUE;
  nCount = (int) SendMessage (hLbxLFiles, LB_GETSELITEMS, 256, (LPARAM) (int *) selects);
  if (nCount>0 && nCount!=LB_ERR)
  {
    for (nIndex=0; (nIndex<nCount); nIndex++) 
         SendMessage (hLbxLFiles, LB_SETSEL, (WPARAM) FALSE, MAKELPARAM (selects[nIndex], 0));
    for (nIndex=0; (nIndex<nCount)&(!bCancelXfer); nIndex++) 
    {
      SendMessage(hLbxLFiles, LB_GETTEXT, nNum=selects[nIndex], (LPARAM)(LPCSTR) szLocalName);
      SendMessage (hLbxLFiles, LB_SETSEL, (WPARAM) TRUE, MAKELPARAM (nNum, 0));
      nRC = DLG_DELETE;
      if (bConfirm)
      {
        lstrcpy (szDlgEdit, szLocalName);
        lpfnMsgProc = MakeProcInstance ((FARPROC) WS_DeleteFileProc, hInst);
        nRC = DialogBox (hInst, (LPSTR) "DLG_DELETEFILE", hWnd, lpfnMsgProc);
        FreeProcInstance (lpfnMsgProc);
      }
      switch (nRC)
      {
        case DLG_NOTDELETE: break;
        case IDCANCEL     : nIndex = nCount+1; break;
        case DLG_DELETEALL: bConfirm = FALSE;
        case DLG_DELETE   : _unlink (szLocalName); break;
      }
      SendMessage (hLbxLFiles, LB_SETSEL, (WPARAM) FALSE, MAKELPARAM (nNum, 0));
    }
    GetLocalDirForWnd (hWnd);
  }
  bOpInProgress = FALSE;
  return 0;
}

//*****************************************************************************
//*****************************************************************************
OnCmdLocalRenameFile (HWND hWnd)
{
  int nIndex;
  FARPROC lpfnMsgProc;

  if ((nIndex=(int) SendMessage (hLbxLFiles, LB_GETCURSEL, 0, 0L))!=LB_ERR)
  {
    SendMessage (hLbxLFiles, LB_GETTEXT, nIndex, (LPARAM)(LPCSTR) szMsgBuf);
    wsprintf (szDlgPrompt, "Enter new name for \"%s\":", szMsgBuf);
    memset (szDlgEdit, '\0', sizeof (szDlgEdit));
    lpfnMsgProc = MakeProcInstance ((FARPROC) WS_InputMsgProc, hInst);
    DialogBox (hInst, (LPSTR) "DLG_INPUT", hWnd, lpfnMsgProc);
    FreeProcInstance (lpfnMsgProc);
    if (rename (szMsgBuf, szDlgEdit)==0) GetLocalDirForWnd (hWnd);
  }
  return 0;
}

//*****************************************************************************
//*****************************************************************************
OnCmdLocalHistory (HWND hWnd)
{  
  return 0;
}

//*****************************************************************************
//*****************************************************************************
LPSTR GetViewer (LPSTR lpLocalName, LPSTR lpVuPgm, LPSTR lpExt)
{
  static char szTmpViewer[_MAX_PATH];
  char *lp;

  lp = strchr (lpLocalName, '.');
  if (lp!=NULL)
  {
    strncpy (lpExt, ++lp, 4); 
    GetProfileString ("Extensions", lpExt, "", szTmpViewer, sizeof (szTmpViewer)-5);
    if (lstrlen (szTmpViewer) > 0) 
    {
      lp = strchr (szTmpViewer, '^');
      if (lp!=NULL) *lp = '\0';
      lpVuPgm = szTmpViewer;
    }
  }
  return lpVuPgm;
}

//*****************************************************************************
//*****************************************************************************
OnCmdLocalDisplay (HWND hWnd)
{
  char szLocalName[_MAX_PATH], szMsgBuf[_MAX_PATH], szDir[_MAX_PATH], *lpVuPgm;
  char szExt[10];
  int nIndex;

  if((nIndex=(int) SendMessage(hLbxLFiles, LB_GETCURSEL, 0, 0L))!=LB_ERR)
  {
    SendMessage (hTxtLDir, WM_GETTEXT, (WPARAM) sizeof (szDir)-1, (LPARAM)(LPCSTR) szDir);
    SendMessage (hLbxLFiles, LB_GETTEXT, nIndex, (LPARAM)(LPCSTR) szLocalName);
    lpVuPgm = FindViewerName (szLocalName);
    if (lpVuPgm==szViewer) lpVuPgm = GetViewer (szLocalName, lpVuPgm, szExt);
    nIndex = lstrlen (szDir)-1;
    if (szDir[nIndex]=='\\') szDir[nIndex] = '\0';
    wsprintf(szMsgBuf, "%s %s\\%s", lpVuPgm, szDir, szLocalName);
    if ((nIndex=WinExec((LPCSTR) szMsgBuf, SW_SHOWNORMAL))<32)
    {
      wsprintf (szLocalName, "Error Code was %d", nIndex);
      MessageBox (hWnd, szMsgBuf, szLocalName, MB_OK);
    }
  }
  return 0;
}

//*****************************************************************************
//  Execute the Double Click command defined by current setting
//*****************************************************************************
OnCmdLButtonDblClkLocal (HWND hWnd)
{
  switch (bDblClkVu)
  {
    case FALSE: OnCmdLocalToRemote (hWnd); break;
    default   : OnCmdLocalDisplay (hWnd); break;
  }
  return 0;
}

//*****************************************************************************
//*****************************************************************************
OnCmdLocalRefresh (HWND hWnd)
{  
  GetLocalDirForWnd (hWnd);
  return 0;
}

//*****************************************************************************
//*****************************************************************************
OnCmdRemoteMakeDir (HWND hWnd)
{
  FARPROC lpfnMsgProc;
  int nRC;

  if (!bConnected) return 0;
  lstrcpy (szDlgPrompt, "Enter new remote directory name:");
  memset (szDlgEdit, '\0', sizeof (szDlgEdit));
  lpfnMsgProc = MakeProcInstance ((FARPROC) WS_InputMsgProc, hInst);
  DialogBox (hInst, (LPSTR) "DLG_INPUT", hWnd, lpfnMsgProc);
  FreeProcInstance (lpfnMsgProc);
  nRC = DoMKD (ctrl_socket, szDlgEdit);
  if (nRC==FTP_COMPLETE) GetRemoteDirForWnd (hWnd);
  return 0;
}

//*****************************************************************************
//*****************************************************************************
OnCmdRemoteRemoveDir (HWND hWnd)
{
  int nIndex, nRC;
  char szMsgBuf[80];

  if (!bConnected) return 0;
  if ((nIndex = (int) SendMessage (hLbxRDir, LB_GETCURSEL, 0, 0L))!=LB_ERR)
  {
    SendMessage (hLbxRDir, LB_GETTEXT, nIndex, (LPARAM)(LPCSTR) szMsgBuf);
    wsprintf (szString, "Are you sure you want to delete \"%s\"?", szMsgBuf);
    if (MessageBox (hWnd, szString, "Verify Deletion", MB_YESNO)==IDYES) 
    {
      nRC = DoRMD (ctrl_socket, szMsgBuf);
      if (nRC==FTP_COMPLETE) GetRemoteDirForWnd (hWnd);
    }
  }
  return 0;
}

//*****************************************************************************
//*****************************************************************************
OnCmdRemoteDeleteFile (HWND hWnd)
{
  int nIndex, nCount, nRC;
  char szRemoteName[80];
  BOOL bConfirm = TRUE;
  FARPROC lpfnMsgProc;
  
  bCancelXfer = FALSE;
  bOpInProgress = TRUE;
  nCount = (int) SendMessage (hLbxRFiles, LB_GETSELITEMS, 256, (LPARAM) (int *) selects);
  if (nCount>0 && nCount!=LB_ERR)
  {
    for (nIndex=0; (nIndex<nCount); nIndex++) 
         SendMessage (hLbxRFiles, LB_SETSEL, (WPARAM) FALSE, MAKELPARAM (selects[nIndex], 0));
    for (nIndex=0; (nIndex<nCount)&(!bCancelXfer); nIndex++) 
    {
      SendMessage(hLbxRFiles, LB_GETTEXT, selects[nIndex], (LPARAM)(LPCSTR) szRemoteName);
      SendMessage (hLbxRFiles, LB_SETSEL, (WPARAM) TRUE, MAKELPARAM (selects[nIndex], 0));
      nRC = DLG_DELETE;
      if (bConfirm)
      {
        lstrcpy (szDlgEdit, szRemoteName);
        lpfnMsgProc = MakeProcInstance ((FARPROC) WS_DeleteFileProc, hInst);
        nRC = DialogBox (hInst, (LPSTR) "DLG_DELETEFILE", hWnd, lpfnMsgProc);
        FreeProcInstance (lpfnMsgProc);
      }
      switch (nRC)
      {
        case DLG_NOTDELETE: break;
        case IDCANCEL     : nIndex = nCount+1; break;
        case DLG_DELETEALL: bConfirm = FALSE;
        case DLG_DELETE   : nRC = DoDELE (ctrl_socket, szRemoteName); break;
      }
      SendMessage (hLbxRFiles, LB_SETSEL, (WPARAM) FALSE, MAKELPARAM (selects[nIndex], 0));
    }
    GetRemoteDirForWnd (hWnd);
  }
  bOpInProgress = FALSE;
  return 0;
}

//*****************************************************************************
//*****************************************************************************
OnCmdRemoteRenameFile (HWND hWnd)
{
  int nIndex, nRC;
  FARPROC lpfnMsgProc;
  char szRemoteName[80];
 
  if (!bConnected) return 0;
  if ((nIndex=(int) SendMessage (hLbxRFiles, LB_GETCURSEL, 0, 0L))!=LB_ERR)
  {
    SendMessage (hLbxRFiles, LB_GETTEXT, nIndex, (LPARAM)(LPCSTR) szRemoteName);
    wsprintf (szDlgPrompt, "Enter new name for \"%s\":", szRemoteName);
    memset (szDlgEdit, '\0', sizeof (szDlgEdit));
    lpfnMsgProc = MakeProcInstance ((FARPROC) WS_InputMsgProc, hInst);
    nRC=DialogBox(hInst, (LPSTR) "DLG_INPUT", hWnd, lpfnMsgProc);
    FreeProcInstance (lpfnMsgProc);
    if (nRC==IDOK)
    {
      nRC = command(ctrl_socket, "RNFR %s", szRemoteName);
      if (nRC==FTP_CONTINUE) 
      {
        nRC=command (ctrl_socket, "RNTO %s", szDlgEdit);
        if (nRC==FTP_COMPLETE) GetRemoteDirForWnd (hWnd);
      }
    }
  }
  return 0;
}

//*****************************************************************************
//*****************************************************************************
OnCmdRemoteChangeDir (HWND hWnd)
{
  int nIndex, nRC;
  FARPROC lpfnMsgProc;
  char szMsgBuf[80];
  
  if (!bConnected) return 0;
  if ((nIndex=(int) SendMessage (hLbxRDir, LB_GETCURSEL, 0, 0L))!=LB_ERR)
  {
    SendMessage (hLbxRDir, LB_GETTEXT, nIndex, (LPARAM)(LPCSTR) szMsgBuf);
    nRC = DoCWD (ctrl_socket, szMsgBuf);
    if (nRC==FTP_COMPLETE) 
    {
      GetRemoteDirForWnd (hWnd);
    }
  }
  else
  {
    lstrcpy (szDlgPrompt, "Enter remote directory name:");
    memset (szDlgEdit, '\0', sizeof (szDlgEdit));
    lpfnMsgProc = MakeProcInstance ((FARPROC) WS_InputMsgProc, hInst);
    nRC = DialogBox (hInst, (LPSTR) "DLG_INPUT", hWnd, lpfnMsgProc);
    FreeProcInstance (lpfnMsgProc);
    if (nRC==IDOK)
    {
      nRC = DoCWD ((SOCKET) ctrl_socket, szDlgEdit);
      if (nRC==FTP_COMPLETE)
      {
        GetRemoteDirForWnd (hWnd);
      }
    }
  }
  return 0;
}

//*****************************************************************************
//*****************************************************************************
OnCmdRemoteChangeDirLst (HWND hWnd)
{
  int nIndex, nRC;
  char szMsgBuf[80];
  
  if (!bConnected) return 0;
  if ((nIndex=(int) SendMessage (hLbxRDirLst, CB_GETCURSEL, 0, 0L))!=CB_ERR)
  {
    SendDlgItemMessage (hWnd, LST_RDIRLST, CB_GETLBTEXT, nIndex, (LPARAM)(LPCSTR) szMsgBuf);
    nRC = DoCWD (ctrl_socket, szMsgBuf);
    if (nRC==FTP_COMPLETE) 
    {
      GetRemoteDirForWnd (hWnd);
    }
  }
  return 0;
}

//*****************************************************************************
//*****************************************************************************
OnCmdRemoteToLocal (HWND hWnd)
{
  char szTmp[80];
  char szRemoteName[80];
  char szLocalName[80];
  int nCount, nRC, nIndex;

  if (!bConnected) return 0;
  bCancelXfer = FALSE;
  bOpInProgress = TRUE;
  nCount = (int) SendMessage (hLbxRFiles, LB_GETSELITEMS, 256, (LPARAM) (int far *) selects);
  if (nCount>0 && nCount!=LB_ERR) 
  {
    memset (szRemoteName, '\0', 80);
    memset (szLocalName,  '\0', 80);
    for (nIndex=0; (nIndex<nCount); nIndex++) 
         SendMessage (hLbxRFiles, LB_SETSEL, (WPARAM) FALSE, MAKELPARAM (selects[nIndex], 0));
    for (nIndex=0; (nIndex<nCount)&(!bCancelXfer); nIndex++) 
    {
      SendMessage (hLbxRFiles, LB_GETTEXT, selects[nIndex], (LPARAM)(LPCSTR) szRemoteName);
      SendMessage (hLbxRFiles, LB_SETSEL, (WPARAM) TRUE, MAKELPARAM (selects[nIndex], 0));
      MakeLocalName (szLocalName, szRemoteName);
      if (bInteractive)
      { 
        FARPROC lpfnMsgProc;
        
        wsprintf (szDlgPrompt, "Enter local file name for %s:", szRemoteName);
        lstrcpy (szDlgEdit, szLocalName);
        lpfnMsgProc = MakeProcInstance ((FARPROC) WS_InputMsgProc, hInst);
        DialogBox (hInst, (LPSTR) "DLG_INPUT", hWnd, lpfnMsgProc);
        FreeProcInstance (lpfnMsgProc);
        lstrcpy (szLocalName, szDlgEdit);
      }
      DoPrintf ("receiving %s as %s (%u of %u)", szRemoteName, szLocalName, nIndex+1, nCount);
      wsprintf (szTmp, "RETR %s", szRemoteName);
      nRC = RetrieveFile ((SOCKET)ctrl_socket, (LPSTR) szTmp, (LPSTR) szLocalName, fType);
      if (nRC==2) 
      {
        SendMessage (hLbxLFiles, LB_ADDSTRING, 0, (LPARAM)(LPCSTR) szRemoteName);
        SendMessage (hLbxRFiles, LB_SETSEL, (WPARAM) FALSE, MAKELPARAM (selects[nIndex], 0));
      }
      else break;
    }
    DeleteXferWindow();
    GetLocalDirForWnd (hWnd);
  }
  bOpInProgress = FALSE;
  SendMessage (hTxtRBytes, WM_SETTEXT, (WPARAM) 0, (LPARAM)(LPCSTR) "");
  return 0;
}

//*****************************************************************************
//*****************************************************************************
OnCmdRemoteRefresh (HWND hWnd)
{  
  if (!bConnected) return 0;
  GetRemoteDirForWnd (hWnd);
  return 0;
}

//*****************************************************************************
//*****************************************************************************
OnCmdRemoteHistory (HWND hWnd)
{  
  return 0;
}

//*****************************************************************************
//*****************************************************************************
OnCmdRemoteDisplay (HWND hWnd)
{
  char szTmp[80], *lpVuExec, *lpViewDir;
  char szRemoteName[80];
  char szUniqViewFile[_MAX_PATH], szExt[_MAX_EXT];
  int nIndex, nRC, nViewer, nLen;
  char nXferType=TYPE_A;
  
  if (!bConnected) return 0;
  if ((nIndex=(int) SendMessage (hLbxRFiles, LB_GETCURSEL, 0, 0L))!=LB_ERR)
  {
    SendMessage (hLbxRFiles, LB_GETTEXT, nIndex, (LPARAM)(LPCSTR) szRemoteName);
    wsprintf (szTmp, "RETR %s", szRemoteName);
    nViewer = FindViewerIndex (szRemoteName);
    lpVuExec = (nViewer==-1) ? GetViewer (szRemoteName, szViewer, szExt) : lpVuPgm (nViewer);
    nXferType = (nViewer==-1) ? (lpVuExec==szViewer?TYPE_A:TYPE_I) : GetViewTyp (nViewer);
    switch (nViewer)
    {
      case -1: wsprintf (szUniqViewFile, szTmpViewFile, ++nView); break;
      default: lpViewDir = lpVuDir (nViewer);
               nLen = lpViewDir==NULL ? 0 : lstrlen (lpViewDir);
               if ((lpViewDir==NULL) || (nLen==0))
               {
                  wsprintf (szUniqViewFile, szTmpViewFile, ++nView); 
               }
               else 
               {
                 lstrcpy (szUniqViewFile, lpViewDir);
                 if (szUniqViewFile[nLen-1]!='\\') lstrcat (szUniqViewFile, "\\"), nLen++;
                 wsprintf (szUniqViewFile+nLen, "WFTP%04d%s", nView++, lpVuExt(nViewer));
                 SendDlgItemMessage (hWnd, LST_DELFILES, LB_ADDSTRING, (WPARAM) 0, (LPARAM)(LPCSTR) szUniqViewFile);
               }
    }
    DoPrintf ("[1] Retrieve %s as %s", szRemoteName, szUniqViewFile);
    bOpInProgress = TRUE;
    nRC = RetrieveFile ((SOCKET) ctrl_socket, (LPSTR) szTmp, (LPSTR) szUniqViewFile, nXferType);
    DeleteXferWindow();
    bOpInProgress = FALSE;
    if (nRC==2) 
    {
      DoPrintf ("[1] View %s as %s", szRemoteName, szUniqViewFile);
      wsprintf (szString, "%s %s", lpVuExec, szUniqViewFile);
      WinExec (szString, SW_SHOWNORMAL);
      if ((!bRetain)&&(nView>1))
      {
        wsprintf (szUniqViewFile, szTmpViewFile, nView-1);
        unlink (szUniqViewFile);
      }
    }
  }
  return 0;
}

//*****************************************************************************
//  Execute the Double Click command defined by current setting
//*****************************************************************************
OnCmdLButtonDblClkRemote (HWND hWnd)
{
  switch (bDblClkVu)
  {
    case FALSE: OnCmdRemoteToLocal (hWnd); break;
    default   : OnCmdRemoteDisplay (hWnd); break;
  }
  return 0;
}

//*****************************************************************************
//*****************************************************************************
OnShowTransferType (HWND hWnd)
{
  SendMessage (hRBascii,  BM_SETCHECK, fType==TYPE_A, 0L);
  SendMessage (hRBbinary, BM_SETCHECK, fType==TYPE_I, 0L);
  SendMessage (hRBl8,     BM_SETCHECK, fType==TYPE_L, 0L);
  return 0;
}

//*****************************************************************************
//*****************************************************************************
OnCmdConnectHost (HWND hWnd, WORD wNotifyCode)
{
  FARPROC lpfnMsgProc;
  int nRC;
  
  if (ctrl_socket==INVALID_SOCKET) 
  {
    lpfnMsgProc = MakeProcInstance ((FARPROC) WS_HostMsgProc, hInst);
    nRC = DialogBox (hInst, (LPSTR) "DLG_HOST", hWnd, lpfnMsgProc);
    FreeProcInstance (lpfnMsgProc);
    if (nRC) 
    {
      ctrl_socket = (SOCKET) DoConnect (szRemoteHost);
      if (ctrl_socket!=INVALID_SOCKET)
      {
        RetrieveDirList (hWnd);
        if (szInitDir[0]!=0) DoCWD (ctrl_socket, szInitDir);
        if (wNotifyCode==BTN_CONNECT) GetRemoteDirForWnd (hWnd);
        //if (szScript[0]!='\0') DoRunScript (hWnd, szScript);
      }
    }
  }
  else
  {
    DoAddLine ("Already connected");
  }
  return 0;
}

//*****************************************************************************
//*****************************************************************************
OnCmdListHost (HWND hWnd)
{
  char szMsgBuf[_MAX_PATH+80];
  
  if (bConnected)
  {
    wsprintf (szMsgBuf, "%s %s", szViewer, szTmpDirFile);
    WinExec (szMsgBuf, SW_SHOWNORMAL);
  }
  return 0;
}

//*****************************************************************************
//  Display the long dir listing
//*****************************************************************************
OnCmdNListHost (HWND hWnd, WORD wNotifyCode)
{
  char szMsgBuf[_MAX_PATH+80];
  
  if (wNotifyCode==CMD_NLST)
  {
    if(DoDirList ((SOCKET)ctrl_socket, "NLST")!=FTP_COMPLETE) return 0;
  }
  if (bConnected) 
  {
    wsprintf (szMsgBuf, "%s %s", szViewer, szCurrentDir);
    WinExec (szMsgBuf, SW_SHOWNORMAL);
  }
  return 0;  
}

//*****************************************************************************
//*****************************************************************************
OnCmdLogToFile (HWND hWnd, WORD wCtlID)
{
  HMENU hMenu = GetMenu (hWnd);
  UINT nState = GetMenuState (hMenu, IDM_LOGTOFILE, MF_BYCOMMAND);
                
  CheckMenuItem (hMenu, IDM_LOGTOFILE, nState=(nState==MF_CHECKED)?MF_UNCHECKED:MF_CHECKED);
  nLogFlag = nState;
  return 0;
}

//*****************************************************************************
//*****************************************************************************
OnCmdLogFileName (HWND hWnd, WORD wCtlID)
{
  FARPROC lpfnMsgProc;
  int nResult;

  lstrcpy (szDlgPrompt, "Enter Log File Name:");
  strcpy (szDlgEdit, szLogFile);
  lpfnMsgProc = MakeProcInstance ((FARPROC)WS_InputMsgProc, hInst);
  nResult = DialogBox (hInst, (LPSTR) "DLG_INPUT", hWnd, lpfnMsgProc);
  FreeProcInstance (lpfnMsgProc);
  if (nResult==IDOK) strcpy (szLogFile, szDlgEdit);
  return 0;
}

//*****************************************************************************
//*****************************************************************************
OnCmdHelpHost (HWND hWnd)
{
  command ((SOCKET) ctrl_socket, "HELP");
  return 0;
}

//*****************************************************************************
//*****************************************************************************
OnCmdStatusHost (HWND hWnd)
{ 
  FARPROC lpfnMsgProc;
  
  lpfnMsgProc = MakeProcInstance ((FARPROC) WS_StatMsgProc, hInst);
  DialogBox (hInst, (LPSTR) "DLG_STATUS", hWnd, lpfnMsgProc);
  FreeProcInstance (lpfnMsgProc);
  SendMessage (hWnd, WM_COMMAND, RB_SHOWCHECKS, 0l);
  return 0;
}

//*****************************************************************************
//*****************************************************************************
OnCmdPWDHost (HWND hWnd)
{
  DoPWD ((SOCKET) ctrl_socket);
  return 0;
}

//*****************************************************************************
//*****************************************************************************
OnCmdDirOpHost (HWND hWnd, WORD wCtlID)
{
  FARPROC lpfnMsgProc;
  int nResult;
  
  lstrcpy(szDlgPrompt,"Enter remote directory name:");
  memset (szDlgEdit, '\0',sizeof (szDlgEdit));
  lpfnMsgProc=MakeProcInstance((FARPROC)WS_InputMsgProc,hInst);
  nResult = DialogBox(hInst,(LPSTR)"DLG_INPUT",hWnd,lpfnMsgProc);
  FreeProcInstance (lpfnMsgProc);
  if (nResult==IDOK) switch (wCtlID)
  {
    case CMD_RMD : DoRMD (ctrl_socket, szDlgEdit); break;
    case CMD_MKD : DoMKD (ctrl_socket, szDlgEdit); break;
  }
  return 0;
}

//*****************************************************************************
//*****************************************************************************
OnCmdGetFileHost (HWND hWnd)
{ 
  FARPROC lpfnMsgProc;
  int nResult;
  
  lstrcpy (szDlgPrompt, "Enter remote file name:");
  memset (szDlgEdit, '\0', sizeof (szDlgEdit));
  lpfnMsgProc = MakeProcInstance ((FARPROC) WS_InputMsgProc, hInst);
  nResult = DialogBox (hInst, (LPSTR) "DLG_INPUT", hWnd, lpfnMsgProc);
  FreeProcInstance (lpfnMsgProc);
  if (nResult==IDOK)
  {
    wsprintf (szDlgPrompt, "RETR %s", szDlgEdit);
    RetrieveFile (ctrl_socket, szDlgPrompt, szDlgEdit, fType);
  }
  return 0;
}

//*****************************************************************************
//*****************************************************************************
OnCmdPutFileHost (HWND hWnd)
{ 
  FARPROC lpfnMsgProc;
  int nResult;
  
  lstrcpy (szDlgPrompt, "Enter local file name:");
  memset (szDlgEdit, '\0', sizeof (szDlgEdit));
  lpfnMsgProc = MakeProcInstance ((FARPROC) WS_InputMsgProc, hInst);
  nResult = DialogBox (hInst, (LPSTR) "DLG_INPUT", hWnd, lpfnMsgProc);
  FreeProcInstance (lpfnMsgProc);
  if (nResult==IDOK)
  {
    wsprintf (szDlgPrompt, "STOR %s", szDlgEdit);
    SendFile (ctrl_socket, szDlgPrompt, szDlgEdit, fType);
  }
  return 0;
}

//*****************************************************************************
//*****************************************************************************
OnCmdQuoteHost (HWND hWnd)
{ 
  FARPROC lpfnMsgProc;
  int nResult;
  
  lstrcpy (szDlgPrompt, "Enter command for remote host:");
  memset (szDlgEdit, '\0', sizeof (szDlgEdit));
  lpfnMsgProc = MakeProcInstance ((FARPROC) WS_InputMsgProc, hInst);
  nResult = DialogBox(hInst, (LPSTR) "DLG_INPUT", hWnd, lpfnMsgProc);
  FreeProcInstance (lpfnMsgProc);
  if (nResult==IDOK) DoQUOTE ((SOCKET) ctrl_socket, szDlgEdit);
  return 0;
}

//*****************************************************************************
//*****************************************************************************
OnCmdGetCwdHost (HWND hWnd)    
{ 
  FARPROC lpfnMsgProc;
  int nResult;

  lstrcpy (szDlgPrompt, "Enter remote directory name:");
  memset (szDlgEdit, '\0', sizeof (szDlgEdit));
  lpfnMsgProc = MakeProcInstance ((FARPROC) WS_InputMsgProc, hInst);
  nResult = DialogBox (hInst, (LPSTR) "DLG_INPUT", hWnd, lpfnMsgProc);
  FreeProcInstance (lpfnMsgProc);
  if (nResult==IDOK) DoCWD ((SOCKET) ctrl_socket, szDlgEdit);
  return 0;
}

//*****************************************************************************
//*****************************************************************************
OnCmdFireWallSetup (HWND hWnd)    
{ 
  FARPROC lpfnMsgProc;
  int nResult;

  lpfnMsgProc = MakeProcInstance ((FARPROC) WS_FireWallSetupProc, hInst);
  nResult = DialogBox (hInst, (LPSTR) "DLG_FIREWALL", hWnd, lpfnMsgProc);
  FreeProcInstance (lpfnMsgProc);
  return 0;
}

//*****************************************************************************
//*****************************************************************************
OnCmdAbort (HWND hWnd)
{ 
  if (!bOpInProgress) return 0;
  //ForcePacket (ctrl_socket, "ABOR");
  bCancelXfer = TRUE;
  bAborted = TRUE;
  //if (WSAIsBlocking()) WSACancelBlockingCall();
  //ForceCommand (ctrl_socket, "ABOR");
  if (nTimerID!=-1)
  {
    KillTimer (hWndMain, nTimerID);
    nTimerID = -1;
  }
  DoAddLine("Operation aborted by user");    
  return 0;
}

//*****************************************************************************
//*****************************************************************************
OnCmdAbout (HWND hWnd)
{ 
  FARPROC lpfnMsgProc;
  
  lpfnMsgProc = MakeProcInstance ((FARPROC) WS_AboutMsgProc, hInst); 
  DialogBox (hInst, (LPSTR)"DLG_ABOUT", hWnd, lpfnMsgProc);
  FreeProcInstance (lpfnMsgProc);     
  return 0;
}

//*****************************************************************************
//*****************************************************************************
OnCmdShowDebug (HWND hWnd, WORD wCtrlID)
{
  HMENU hMenu = GetMenu (hWnd);
  UINT nState = GetMenuState (hMenu, IDM_SHOWDEBUG, MF_BYCOMMAND);
                
  CheckMenuItem (hMenu, IDM_SHOWDEBUG, nState=(nState==MF_CHECKED)?MF_UNCHECKED:MF_CHECKED);
  ShowWindow (hWndDbg, nState==MF_CHECKED?SW_SHOWNORMAL : SW_HIDE);
  return 0;
}

//*****************************************************************************
//*****************************************************************************
OnCmdSaveDir (HWND hWnd, WORD wCtrlID)
{
  HMENU hMenu = GetMenu (hWnd);
  UINT nState = GetMenuState (hMenu, IDM_SAVEDIR, MF_BYCOMMAND);
                
  CheckMenuItem (hMenu, IDM_SAVEDIR, nState=(nState==MF_CHECKED)?MF_UNCHECKED:MF_CHECKED);
  UpdateDirListOpt (hWnd, (BOOL) (nState==MF_CHECKED)?TRUE:FALSE);
  return 0;
}

//*****************************************************************************
//*****************************************************************************
OnCmdGetHostAddr (HWND hWnd, WORD wCtlID)
{

#ifdef WIN32
  DialogBox (hInst, (LPCTSTR)"DLG_FINDHOST", hWnd, (FARPROC) WS_FindHostProc);
  return 0;
#else
  FARPROC lpfnMsgProc;
  
  lpfnMsgProc = MakeProcInstance ((FARPROC) WS_FindHostProc, hInst); 
  DialogBox (hInst, (LPSTR)"DLG_FINDHOST", hWnd, lpfnMsgProc);
  FreeProcInstance (lpfnMsgProc);     
  return 0;
#endif
  
}

//*****************************************************************************
//*****************************************************************************
OnCmdSortByType (HWND hWnd, WORD wCtlID)
{
  HMENU hMenu = GetMenu (hWnd);
  LONG lStyle;

  if (wSortType==wCtlID) return 0;
  CheckMenuItem (hMenu, wSortType, MF_UNCHECKED);
  CheckMenuItem (hMenu, wSortType=wCtlID, MF_CHECKED);
  lStyle = GetWindowLong (hLbxRFiles, GWL_STYLE);
  switch (wSortType)
  {
    case IDM_SORTBYNAME: SetWindowLong (hLbxRFiles, GWL_STYLE, lStyle | LBS_SORT); break;
    default            : SetWindowLong (hLbxRFiles, GWL_STYLE, lStyle & ~LBS_SORT); break;
  }
  GetRemoteDirForWnd (hWnd);
  return 0;
}

//*****************************************************************************
//*****************************************************************************
OnCmdDownload (HWND hWnd, WORD wCtlID)
{
//  FARPROC lpfnMsgProc;

//  lpfnMsgProc = MakeProcInstance ((FARPROC) WS_DownloadProc, hInst); 
//  DialogBox (hInst, (LPSTR)"DLG_DOWNLOAD", hWnd, lpfnMsgProc);
//  FreeProcInstance (lpfnMsgProc);     

//  GetLocalDirForWnd (hWnd);
  return 0;
}

//*****************************************************************************
//*****************************************************************************
OnCmdLFileType (HWND hWnd, WORD wCode)
{
  char szBuf[20];
  
  if (wCode!=EN_TEXTCHANGE) return 0;
  memset (szBuf, '\0', 20);
  SendDlgItemMessage (hWnd, EDT_LFILETYPE, WM_GETTEXT, (WPARAM) 19, (LPARAM)(LPCSTR) szBuf);
  if (lstrcmpi (szBuf, szLFileType)==0) return 0;
  GetLocalDirForWnd (hWnd);
  return 0;  
}

//*****************************************************************************
//*****************************************************************************
OnCmdRFileType (HWND hWnd, WORD wCode)
{
  char szBuf[20];
  
  if (wCode!=EN_TEXTCHANGE) return 0;
  memset (szBuf, '\0', 20);
  SendDlgItemMessage (hWnd, EDT_RFILETYPE, WM_GETTEXT, (WPARAM) 19, (LPARAM)(LPCSTR) szBuf);
  if (lstrcmp (szBuf, szRFileType)==0) return 0;
  GetRemoteDirForWnd (hWnd);
  return 0;  
}

LPSTR szDisconnect = "You have a transfer in progress.\n"
                     "Disconnecting now may cause a GPF\n"
                     "Are you sure you wish to disconnect?";

//*****************************************************************************
//*****************************************************************************
OnCmdCloseOrExit (HWND hWnd, WPARAM wParam)
{
  if (bConnected)
  {
    if (bOpInProgress)
    {
      if (MessageBox (hWnd, szDisconnect, "Verify Disconnect", MB_YESNO)!=IDYES)
          return 0;
    }
    bAborted = TRUE;
    bCancelXfer = TRUE;
    if (data_socket!=INVALID_SOCKET) data_socket = DoClose (data_socket);
    if (listen_socket!=INVALID_SOCKET) listen_socket = DoClose (listen_socket);
    if (bConnected) DoDisconnect (ctrl_socket);  // Perform Shutdown
    if (ctrl_socket!=INVALID_SOCKET) 
    {
      ctrl_socket = DoClose (ctrl_socket);
      bConnected=FALSE;
    }
    SaveDirList (hWnd);
    SendDlgItemMessage (hWnd, LST_RDIRLST, CB_RESETCONTENT, 0, 0L);
    SetWindowText (hWnd, szDefaultHdr);
  }
  switch (wParam)
  {
    case IDM_EXIT:
    case BTN_EXIT: CleanupTempFiles (hWnd);
                   SendMessage (hWnd, WM_CLOSE, 0, 0L); break;
    default: GetRemoteDirForWnd (hWnd);
  }
  return 0;
}

//*****************************************************************************
//*****************************************************************************
LRESULT OnWmCmd (HWND hWnd, UINT Msg, WPARAM wParam, LPARAM lParam)
{
  WORD wNotifyCode;
  WORD wCtlID;

#ifdef WIN32
  wNotifyCode = HIWORD (wParam);
  wCtlID      = LOWORD (wParam);
#else
  wNotifyCode = HIWORD (lParam);
  wCtlID      = wParam;
#endif

  bAborted = FALSE;
  switch (wCtlID)
  {
    case BTN_CLOSE:
    case BTN_EXIT :
    case IDM_EXIT : OnCmdCloseOrExit (hWnd, wCtlID); return 0;
    default       : if (bCmdInProgress) return FALSE;
  }
  switch (wCtlID)
  {
    //*******************************************************
    // Local Button Functions and Dialog Boxes
    //*******************************************************
    case LST_LDIRS   : if (wNotifyCode==LBN_DBLCLK) OnCmdLocalChangeDir (hWnd); break;
    case LST_LDIRLST : if (wNotifyCode==CBN_SELCHANGE) OnCmdLocalChangeDirLst (hWnd); break;
    case LST_LFILES  : if (wNotifyCode==LBN_DBLCLK) OnCmdLButtonDblClkLocal (hWnd); break;
    case BTN_LCHANGE : OnCmdLocalChangeDir (hWnd); break;
    case BTN_LMKDIR  : OnCmdLocalMakeDir (hWnd); break;
    case BTN_LRMDIR  : OnCmdLocalRemoveDir (hWnd); break;
    case BTN_LDELETE : OnCmdLocalDeleteFile (hWnd); break;
    case BTN_LRENAME : OnCmdLocalRenameFile (hWnd); break;
    case BTN_LREFRESH: OnCmdLocalRefresh (hWnd); break;
    case BTN_LDISPLAY: OnCmdLocalDisplay (hWnd); break;
    case BTN_LHISTORY: OnCmdLocalHistory (hWnd); break;
    case BTN_LOCAL_TO_REMOTE: OnCmdLocalToRemote (hWnd); break;

    //*******************************************************
    // Remote Button Functions and Dialog Boxes
    //*******************************************************
    case LST_RDIRS   : if (wNotifyCode==LBN_DBLCLK) OnCmdRemoteChangeDir (hWnd); break;
    case LST_RDIRLST : if (wNotifyCode==CBN_SELCHANGE) OnCmdRemoteChangeDirLst (hWnd); break;
    case LST_RFILES  : if (wNotifyCode==LBN_DBLCLK) OnCmdLButtonDblClkRemote (hWnd); break;
    case BTN_RCHANGE : OnCmdRemoteChangeDir (hWnd);    break;
    case BTN_RMKDIR  : OnCmdRemoteMakeDir (hWnd); break;
    case BTN_RRMDIR  : OnCmdRemoteRemoveDir (hWnd); break;
    case BTN_RDELETE : OnCmdRemoteDeleteFile (hWnd); break;
    case BTN_RRENAME : OnCmdRemoteRenameFile (hWnd); break;
    case BTN_RREFRESH: OnCmdRemoteRefresh (hWnd); break;
    case BTN_RDISPLAY: OnCmdRemoteDisplay (hWnd); break;
    case BTN_RHISTORY: OnCmdRemoteHistory (hWnd); break;
    case BTN_REMOTE_TO_LOCAL: OnCmdRemoteToLocal (hWnd); break;

    case RB_ASCII : fType=TYPE_A; OnShowTransferType (hWnd); break;
    case RB_BINARY: fType=TYPE_I; OnShowTransferType (hWnd); break;
    case RB_L8    : fType=TYPE_L; OnShowTransferType (hWnd); break;
    case RB_SHOWCHECKS:           OnShowTransferType (hWnd); break;

//    case CMD_TYPE_I : fType=TYPE_I; break;
//    case CMD_TYPE_A : fType=TYPE_A; break;
    
    case BTN_LONG   : OnCmdNListHost (hWnd, wCtlID); break;
    case BTN_OPTION : OnCmdStatusHost (hWnd); break;
    case BTN_CONNECT: OnCmdConnectHost (hWnd, wCtlID); break;
    case BTN_ABORT  : OnCmdAbort (hWnd); break;
    case BTN_ABOUT  : OnCmdAbout (hWnd); break;
    
    case EDT_LFILETYPE: OnCmdLFileType (hWnd, wNotifyCode); break;
    case EDT_RFILETYPE: OnCmdRFileType (hWnd, wNotifyCode); break;
    
    case IDM_SHOWDEBUG  : OnCmdShowDebug (hWnd, wCtlID); break;
    case IDM_SAVEDIR    : OnCmdSaveDir (hWnd, wCtlID); break;
    case IDM_LOGTOFILE  : OnCmdLogToFile (hWnd, wCtlID); break;
    case IDM_LOGFILENAME: OnCmdLogFileName (hWnd, wCtlID); break;
    case IDM_HOSTADDR   : OnCmdGetHostAddr (hWnd, wCtlID); break;
    case IDM_HOSTLIST   : OnCmdDisplayHostList (hWnd, wCtlID, wNotifyCode); break;
    case IDM_DROPFILES  : OnCmdDropFiles (hWnd, (LPSTR *)lParam); break;
    case IDM_ZOOMINFO   : OnCmdZoomInfo (hWnd); break;
    case IDM_FIREWALL   : OnCmdFireWallSetup (hWnd); break;
    case IDM_SORTBYDATE : OnCmdSortByType (hWnd, wCtlID); break;
    case IDM_SORTBYNAME : OnCmdSortByType (hWnd, wCtlID); break;
    case IDM_DOWNLOAD   : OnCmdDownload (hWnd, wCtlID); break;

    case BTN_CLOSE  : OnCmdCloseOrExit (hWnd, wCtlID); break;
    case IDM_EXIT   : OnCmdCloseOrExit (hWnd, wCtlID); break;
    case IDM_ABOUT  : OnCmdAbout (hWnd); break;
    
    case CMD_CONNECT: OnCmdConnectHost (hWnd, wCtlID); break;
    case CMD_LIST   : OnCmdListHost (hWnd); break;
    case CMD_NLST   : OnCmdNListHost (hWnd, wCtlID); break;
    case CMD_HELP   : OnCmdHelpHost (hWnd); break;
    case CMD_PWD    : OnCmdPWDHost (hWnd); break;
    case CMD_RMD    : OnCmdDirOpHost (hWnd, wCtlID); break;
    case CMD_MKD    : OnCmdDirOpHost (hWnd, wCtlID); break;
    case CMD_RETR   : OnCmdGetFileHost (hWnd); break;
    case CMD_STOR   : OnCmdPutFileHost (hWnd); break;
    case CMD_QUOTE  : OnCmdQuoteHost (hWnd); break;
    case CMD_CWD    : OnCmdGetCwdHost (hWnd); break;
    
    default         : if (!SubProcessAsync (hWnd, Msg, wParam, lParam))
                          return DefWindowProc (hWnd, Msg, wParam, lParam);
  }
  return 0;
}

//*****************************************************************************
//*****************************************************************************
LRESULT OnSetCursor (HWND hWnd,UINT Msg,WPARAM wParam, LPARAM lParam)
{
  switch (bCmdInProgress)
  {
    case TRUE: SetCursor (hWaitCursor); return TRUE;
    default  : return (LRESULT) DefWindowProc (hWnd, Msg, wParam, lParam); 
  }
}

//*****************************************************************************
//*****************************************************************************
LRESULT OnPaint (HWND hWnd, UINT Msg, WPARAM wParam, LPARAM lParam)
{
  DoMainPaint (hWnd); 
  return 0;
}

int nBtnNum=-1;
int nListView=0;

//*****************************************************************************
//*****************************************************************************
LRESULT OnRButtonDown (HWND hWnd, UINT Msg, WPARAM wParam, LPARAM lParam)
{
  if (nListView) return (LRESULT) 0;
  ShowWindow (GetDlgItem (hWnd, LST_DELFILES), SW_SHOWNORMAL);
  nListView = 1;
  return (LRESULT) NULL;
}

//*****************************************************************************
//*****************************************************************************
LRESULT OnRButtonUp (HWND hWnd, UINT Msg, WPARAM wParam, LPARAM lParam)
{
  if (!nListView) return (LRESULT) 0;
  ShowWindow (GetDlgItem (hWnd, LST_DELFILES), SW_HIDE);
  nListView = 0;
  return (LRESULT) NULL;
}

//*****************************************************************************
//*****************************************************************************
LRESULT OnLButtonDown (HWND hWnd, UINT Msg, WPARAM wParam, LPARAM lParam)
{
  int nBtn;
  //HWND hWndFocus, hFile;
  
  if ((nBtn=FindButtonClicked (LOWORD (lParam), HIWORD (lParam)))==-1) return (LRESULT) 0;
  if (!GetButtonEnabledStatus (nBtn)) return (LRESULT) 0;
  if (SetButtonStatus (nBtn, 1)) return (LRESULT) 0;
  DoPaintButton (hWnd, nBtnNum=nBtn, 1);
  return (LRESULT) NULL;
}

//*****************************************************************************
//*****************************************************************************
LRESULT OnLButtonUp (HWND hWnd, UINT Msg, WPARAM wParam, LPARAM lParam)
{
  int nBtn, nID;
  
  if (nBtnNum==-1) return (LRESULT) NULL;
  if ((nBtn=FindButtonClicked (LOWORD (lParam), HIWORD (lParam)))==-1) return (LRESULT) 0;
  if (!GetButtonEnabledStatus (nBtn)) return (LRESULT) 0;
  if (nBtn!=nBtnNum) return (LRESULT) NULL;
  if (!SetButtonStatus (nBtn, 0)) return (LRESULT) NULL;
  switch (nID=GetButtonID (nBtn))
  {
    case BTN_ABORT: OnCmdAbort (hWnd); break;

#ifdef WIN32
    default       : PostMessage (hWnd, WM_COMMAND, MAKEWPARAM (nID, WM_LBUTTONUP), 0L); break;
#else
    default       : PostMessage (hWnd, WM_COMMAND, (WPARAM) nID, 0L); break;
#endif
  }
  DoPaintButton (hWnd, nBtn, 0);
  nBtnNum = -1;
  return (LRESULT) NULL;
}

//*****************************************************************************
//*****************************************************************************
LRESULT OnMouseMove (HWND hWnd, UINT Msg, WPARAM wParam, LPARAM lParam)
{
  int nBtn;
  
  if ((nBtnNum==-1) || !(wParam & MK_LBUTTON)) return (LRESULT) NULL;
  if ((nBtn=FindButtonClicked (LOWORD (lParam), HIWORD (lParam)))==nBtnNum) return (LRESULT) 0;
  if (nBtnNum==-1) return (LRESULT) NULL;
  if (!SetButtonStatus (nBtnNum, 0)) return (LRESULT) NULL;
  DoPaintButton (hWnd, nBtnNum, 0);
  nBtnNum = -1;
  return (LRESULT) NULL;
}

//*****************************************************************************
//*****************************************************************************
LRESULT OnVScroll (HWND hWnd, UINT Msg, WPARAM wParam, LPARAM lParam)
{
  switch(wParam) 
  {
    case SB_LINEUP  : ScrollStatus (hWnd, -1); break;
    case SB_LINEDOWN: ScrollStatus (hWnd, 1) ; break;
  }
  return DefWindowProc(hWnd, Msg, wParam, lParam);
}

#ifdef WIN32

//*****************************************************************************
//*****************************************************************************
LRESULT OnCtlColorBtn (HWND hWnd, UINT Msg, WPARAM wParam, LPARAM lParam)
{
  if (lParam==(LPARAM) hRBascii || lParam==(LPARAM) hRBbinary || lParam==(LPARAM) hRBl8) 
  {
    SelectObject ((HDC)wParam, GetStockObject (ANSI_VAR_FONT));
    SetBkColor ((HDC) wParam, RGB (192,192,192));
    return (LRESULT) hbrGray1;
  }
  return (LRESULT) NULL;
}


//*****************************************************************************
//*****************************************************************************
LRESULT OnCtlColorStatic (HWND hWnd, UINT Msg, WPARAM wParam, LPARAM lParam)
{
  SelectObject ((HDC)wParam, GetStockObject (ANSI_VAR_FONT));
  SetBkColor ((HDC) wParam, RGB (192,192,192));
  return (LRESULT) hbrGray1;
}

#else

//*****************************************************************************
//*****************************************************************************
LRESULT OnCtlColor (HWND hWnd, UINT Msg, WPARAM wParam, LPARAM lParam)
{
  switch (HIWORD(lParam)) 
  {
    case CTLCOLOR_BTN: if (!(LOWORD(lParam)==hRBascii || LOWORD(lParam)==hRBbinary || LOWORD(lParam)==hRBl8)) 
      {
        SelectObject ((HDC)wParam, GetStockObject(ANSI_VAR_FONT));
        return (LRESULT) NULL;
      }
    case CTLCOLOR_STATIC:
                SelectObject ((HDC)wParam, GetStockObject (ANSI_VAR_FONT));
                SetBkColor ((HDC) wParam, RGB (192,192,192));
                return (LRESULT) hbrGray1;
  }
  return (LRESULT) NULL;
}

#endif

//*****************************************************************************
//*****************************************************************************
LRESULT OnClose (HWND hWnd, UINT Msg, WPARAM wParam, LPARAM lParam)
{
  if (hWnd == hWndMain) 
  {
    if (data_socket!=INVALID_SOCKET) data_socket = DoClose (data_socket);
    if (listen_socket!=INVALID_SOCKET) listen_socket = DoClose (data_socket);
    if (ctrl_socket!=INVALID_SOCKET) ctrl_socket = DoClose (ctrl_socket);
    DragAcceptFiles (hWnd, FALSE);
    DestroyWindow (hWnd);
    SaveUserInfo();
    PostQuitMessage (0);  // Quit the application
  }
  else
  {
    DestroyWindow (hWnd);
  }
  return 0;
}

//*****************************************************************************
//*****************************************************************************
LRESULT OnSetFocus (HWND hWnd, UINT Msg, WPARAM wParam, LPARAM lParam)
{
  SetFocus (GetDlgItem (hWnd, EDT_LFILETYPE));
  return 0;
}

//*****************************************************************************
//*****************************************************************************
LRESULT OnKillFocus (HWND hWnd, UINT Msg, WPARAM wParam, LPARAM lParam)
{
  
  return 0;
}

//*****************************************************************************
//*****************************************************************************
LRESULT OnDragDropFile (HWND hWnd, UINT Msg, WPARAM wParam, LPARAM lParam)
{
  UINT uiFiles, uiNum;
  LPSTR *lpPtr;

#ifdef WIN32
#define  IDDROPNUM  0xFFFFFFFF
#else
#define  IDDROPNUM  0xFFFF
#endif

  uiFiles = DragQueryFile ((HANDLE) wParam, IDDROPNUM, (LPSTR) NULL, 0);
  if (uiFiles==0) return 0;
  
  lpPtr = (LPSTR *) GlobalAllocPtr (GHND, (uiFiles+2)*sizeof (LPSTR));
  if (lpPtr!=NULL) 
  {
    for (uiNum = 0; uiNum < uiFiles; uiNum++) 
    {
      lpPtr[uiNum] = (LPSTR) GlobalAllocPtr (GHND, _MAX_PATH);
      if (lpPtr[uiNum]!=NULL) DragQueryFile ((HANDLE) wParam, uiNum, lpPtr[uiNum], _MAX_PATH);
    }
    lpPtr[uiNum] = (LPSTR) NULL;
  }
  DragFinish ((HANDLE) wParam);
  PostMessage (hWnd, WM_COMMAND, (WPARAM) IDM_DROPFILES, (LPARAM) lpPtr);
  return 0;
}


//*****************************************************************************
//*****************************************************************************
LRESULT OnCreate (HWND hWnd, UINT Msg, WPARAM wParam, LPARAM lParam)
{
  hStdCursor=LoadCursor((HINSTANCE)NULL,IDC_ARROW);
  hWaitCursor=LoadCursor((HINSTANCE)NULL,IDC_WAIT);
  LoadUserInfo();
  CreateDebugWindow(hWnd,hInst);
  CreateSubWindows(hWnd,hInst);
  GetLocalDirForWnd(hWnd);
  GetRemoteDirForWnd(hWnd);
  SendMessage(hWnd,WM_COMMAND,RB_SHOWCHECKS,0L);
  DragAcceptFiles (hWnd, TRUE);
  CheckMenuItem (GetMenu (hWnd), IDM_LOGTOFILE, nLogFlag);
  CheckMenuItem (GetMenu (hWnd), wSortType=IDM_SORTBYNAME, MF_CHECKED);
  return 0;
}

//*****************************************************************************
//*****************************************************************************
LRESULT OnTimer (HWND hWnd, UINT Msg, WPARAM wParam, LPARAM lParam)
{
  if ((int) wParam==10)
  {
    KillTimer (hWndMain, 10);
    nTimerID = -1;
    if(WSAIsBlocking()) 
    {
      DoAddLine("Timer cancelled blocking call");    
      bAborted=TRUE;
      WSACancelBlockingCall();
    }
  }
  return 0;
}

//*****************************************************************************
//*****************************************************************************
LRESULT OnParentNotify (HWND hWnd, UINT Msg, WPARAM wParam, LPARAM lParam)
{
  static WORD wChildID=0;
  WORD wTmpID;
  int nSel, nI;
  
  if (wParam!=WM_LBUTTONDOWN) return DefWindowProc (hWnd, Msg, wParam, lParam);
  wTmpID = GetChildWindowID (lParam);
  if ((wChildID==wTmpID)||(wTmpID==0)) return 0L;
  
  switch (wChildID)
  {
    case LST_LDIRS   : 
    case LST_RDIRS   : SendDlgItemMessage (hWnd, wChildID, LB_SETCURSEL, (WPARAM) -1, 0L); break;

    case LST_LFILES  : 
    case LST_RFILES  : nSel = (int) SendDlgItemMessage (hWnd, wChildID, LB_GETSELITEMS, 256, (LPARAM) (int far *) selects);
                       if (nSel>0 && nSel!=LB_ERR) 
                       {
                         for (nI=0; nI<nSel; nI++) 
                           SendDlgItemMessage (hWnd, wChildID, LB_SETSEL, (WPARAM) FALSE, MAKELPARAM (selects[nI], 0));
                       }
                       break;
    default          : wChildID = wTmpID;
                       return DefWindowProc (hWnd, Msg, wParam, lParam);
  }
  wChildID = wTmpID;
  return 0L;
}

/************************************************************************/
/* Main Window Procedure                                                */
/* This procedure provides service routines for the Windows events      */
/* (messages) that Windows sends to the window, as well as the user     */
/* initiated events (messages) that are generated when the user selects */
/* the action bar and pulldown menu controls or the corresponding       */
/* keyboard accelerators.                                               */
/************************************************************************/
LRESULT CALLBACK WndProc(HWND hWnd, UINT Msg, WPARAM wParam, LPARAM lParam)
{
  switch (Msg)
  {
    case WM_CREATE      : return OnCreate (hWnd, Msg, wParam, lParam); break;
    case WM_TIMER       : return OnTimer  (hWnd, Msg, wParam, lParam); break;
    case WM_COMMAND     : return OnWmCmd  (hWnd, Msg, wParam, lParam); break;
    case WM_SETCURSOR   : return OnSetCursor (hWnd, Msg, wParam, lParam); break;
    case WM_PAINT       : return OnPaint  (hWnd, Msg, wParam, lParam); break;
    case WM_LBUTTONDOWN : return OnLButtonDown (hWnd, Msg, wParam, lParam); break;
    case WM_LBUTTONUP   : return OnLButtonUp (hWnd, Msg, wParam, lParam); break;
    case WM_MOUSEMOVE   : return OnMouseMove (hWnd, Msg, wParam, lParam); break;
    case WM_VSCROLL     : return OnVScroll (hWnd, Msg, wParam, lParam); break;
    case WM_KILLFOCUS   : return OnKillFocus (hWnd, Msg, wParam, lParam); break;
    case WM_SETFOCUS    : return OnSetFocus (hWnd, Msg, wParam, lParam); break;
    case WM_PARENTNOTIFY: return OnParentNotify (hWnd, Msg, wParam, lParam); break;
    case WM_DROPFILES   : return OnDragDropFile (hWnd, Msg, wParam, lParam); break;
    case WM_CLOSE       : return OnClose  (hWnd, Msg, wParam, lParam); break;

    //case WM_RBUTTONDOWN : return OnRButtonDown (hWnd, Msg, wParam, lParam); break;
    //case WM_RBUTTONUP   : return OnRButtonUp (hWnd, Msg, wParam, lParam); break;

#ifdef WIN32
    case WM_CTLCOLORBTN   : return OnCtlColorBtn (hWnd, Msg, wParam, lParam); break; 
    case WM_CTLCOLORSTATIC: return OnCtlColorStatic (hWnd, Msg, wParam, lParam); break;
#else
    case WM_CTLCOLOR    : return OnCtlColor (hWnd, Msg, wParam, lParam); break;
#endif
    
    default             : return DefWindowProc (hWnd, Msg, wParam, lParam);
  }
  return 0L;
}

/************************************************************************/
/************************************************************************/
void CreateTempFileNames (LPSTR szTempDir)
{
  char szDir[100], *lp;
  
  lstrcpy (szDir, szTempDir);
  lp = szDir;
  if (lstrlen (lp)>0) lp += lstrlen (lp)-1;
  if (*lp=='\\') *lp='\0';
  wsprintf (szTmpDirFile,  "%s\\%s", szDir, szTmpDirFileName);
  wsprintf (szTmpViewFile, "%s\\%s", szDir, szTmpViewFileName);
}

/************************************************************************/
/* Misc Dialog Window Procedures                                        */
/************************************************************************/
BOOL CALLBACK WS_AboutMsgProc(HWND hDlg, UINT Msg, WPARAM wParam, LPARAM lParam)
{ 
  switch (Msg)
  {
    case WM_INITDIALOG: 
      {
        char szVer[80];
        LPSTR szStamp = __DATE__;

        wsprintf (szVer, "Version %s", szStamp);
        SetWindowText (GetDlgItem (hDlg, IDC_VERSION), szVer);
        cwCenter (hDlg, 0); 
      } break;
    case WM_CLOSE     : PostMessage (hDlg, WM_COMMAND, IDCANCEL, 0L); break;
    case WM_COMMAND   : switch (wParam)
      {
        case IDOK     : EndDialog(hDlg, TRUE); return TRUE;
        case IDCANCEL : EndDialog(hDlg, FALSE); return TRUE;
        default       : return TRUE;
      }
    default: return FALSE;
  }
}

/************************************************************************/
/************************************************************************/
BOOL CALLBACK WS_FireWallSetupProc(HWND hDlg, UINT Msg, WPARAM wParam, LPARAM lParam)
{ 
  switch (Msg)
  {
    case WM_INITDIALOG: 
      {
        SetDlgItemText (hDlg, IDC_FIREWALLHOST, szFireWallHost);
        SetDlgItemText (hDlg, IDC_FIREWALLUSER, szFireWallUserID);
        SetDlgItemText (hDlg, IDC_FIREWALLPASS, szFireWallUserPass);
      } break;
    case WM_CLOSE     : PostMessage (hDlg, WM_COMMAND, IDCANCEL, 0L); break;
    case WM_COMMAND   : switch (wParam)
      {
        case IDOK     : GetDlgItemText (hDlg, IDC_FIREWALLHOST, szFireWallHost, 80);
                        GetDlgItemText (hDlg, IDC_FIREWALLUSER, szFireWallUserID, 25);
                        GetDlgItemText (hDlg, IDC_FIREWALLPASS, szFireWallUserPass, 25);
                        EndDialog(hDlg, TRUE); return TRUE;
        case IDCANCEL : EndDialog(hDlg, FALSE); return TRUE;
        default       : return TRUE;
      }
    default: return FALSE;
  }
}

//*****************************************************************************
//*****************************************************************************
BOOL CALLBACK WS_InputMsgProc(HWND hDlg, UINT Msg, WPARAM wParam, LPARAM lParam)
{ 
  switch(Msg)
  {
    case WM_INITDIALOG: SetDlgItemText (hDlg, DLG_PROMPT, szDlgPrompt);
                        SetDlgItemText (hDlg, DLG_EDIT, szDlgEdit);
                        cwCenter (hDlg, 0);
                        break;
    //case WM_CLOSE     : PostMessage (hDlg, WM_COMMAND, IDCANCEL, 0L); break;
    case WM_COMMAND   : switch (wParam)
      {
        case IDOK     : GetDlgItemText (hDlg, DLG_EDIT, szDlgEdit, 70);
                        EndDialog (hDlg, TRUE);
                        break;
        case IDCANCEL : EndDialog (hDlg, FALSE);
                        break;
      } break;
      
    default: return FALSE;
  }
  return TRUE;
}

//*****************************************************************************
//*****************************************************************************
BOOL CALLBACK WS_DeleteFileProc (HWND hDlg, UINT Msg, WPARAM wParam, LPARAM lParam)
{ 
  switch(Msg)
  {
    case WM_INITDIALOG: SetDlgItemText (hDlg, DLG_EDIT, szDlgEdit); break;
    //case WM_CLOSE     : PostMessage (hDlg, WM_COMMAND, IDCANCEL, 0L); break;
    case WM_COMMAND   : switch (wParam)
      {
        case DLG_NOTDELETE: EndDialog (hDlg, DLG_NOTDELETE); break;
        case DLG_DELETEALL: EndDialog (hDlg, DLG_DELETEALL); break;
        case DLG_DELETE   : EndDialog (hDlg, DLG_DELETE); break;
        case IDCANCEL     : EndDialog (hDlg, IDCANCEL);   break;
      } break;
      
    default: return FALSE;
  }
  return TRUE;
}

static int nViewer=0;

//*****************************************************************************
//*****************************************************************************
void GetViewerInfo (HWND hDlg)
{
  char szExt[10], szPgm[80], szDir[80];
  LPSTR lp1, lp2, lp3;
  int nI, nTyp, nXfer;
  
  GetDlgItemText (hDlg, DLG_VIEWTYPE, szExt, 5);
  GetDlgItemText (hDlg, DLG_VIEWPGM,  szPgm, 75);
  GetDlgItemText (hDlg, DLG_VIEWDIR,  szDir, 75);
  strupr (szExt);
  
  nTyp = IsDlgButtonChecked (hDlg, DLG_VIEW_ASCII) ? TYPE_A : TYPE_I;
  lp1 = lpVuExt (nViewer);
  lp2 = lpVuPgm (nViewer);
  lp3 = lpVuDir (nViewer);
  
  nXfer = GetViewTyp (nViewer);
  if ((lstrcmp (lp1, szExt)==0)&&(lstrcmp (lp2, szPgm)==0)&&
      (lstrcmp (lp3, szDir)==0)&&(nXfer==nTyp)) return;

  for (nI=0; nI<nViewNum; nI++)
  {
    if (lstrcmp (lpVuExt (nI), szExt)==0)
    {
      lstrcpy (lpVuPgm (nI), szPgm);
      lstrcpy (lpVuDir (nI), szDir);
      SetViewTyp (nI, (char) nTyp);
      return;
    }
  }
  if (ReAllocViewer (nViewNum+1)!=NULL)
  {
    nI = nViewNum++;
    lstrcpy (lpVuExt (nI), szExt);
    lstrcpy (lpVuPgm (nI), szPgm);
    lstrcpy (lpVuDir (nI), szDir);
    SetViewTyp (nI, (char) nTyp);
  }
}

//*****************************************************************************
//*****************************************************************************
void SetViewerInfo (HWND hDlg, int nBtn)
{
  int nTyp;
  HWND hWndScroll = GetDlgItem (hDlg, DLG_SCROLLVIEWER);
        
  SetDlgItemText (hDlg, DLG_VIEWTYPE, lpVuExt (nBtn));
  SetDlgItemText (hDlg, DLG_VIEWPGM,  lpVuPgm (nBtn));
  SetDlgItemText (hDlg, DLG_VIEWDIR,  lpVuDir (nBtn));
  nTyp = GetViewTyp (nBtn);
  CheckDlgButton (hDlg, DLG_VIEW_ASCII, nTyp==TYPE_A);
  CheckDlgButton (hDlg, DLG_VIEW_BINARY, nTyp==TYPE_I);
  SetScrollRange (hWndScroll, SB_CTL, 0, nViewNum-1, FALSE);
  SetScrollPos (hWndScroll, SB_CTL, nBtn, TRUE);
}

//*****************************************************************************
//*****************************************************************************
BOOL CALLBACK WS_StatMsgProc(HWND hDlg, UINT Msg, WPARAM wParam, LPARAM lParam)
{ 
  switch(Msg)
  {
    case WM_INITDIALOG: switch (fType)
      {
        case TYPE_A : CheckRadioButton(hDlg,RB_ASCII,RB_L8,RB_ASCII); break;
        case TYPE_I : CheckRadioButton(hDlg,RB_ASCII,RB_L8,RB_BINARY); break;
        default     : CheckRadioButton(hDlg,RB_ASCII,RB_L8,RB_L8); break;
      }
      SetDlgItemText (hDlg, DLG_EDIT, szViewer);
      SetDlgItemText (hDlg, DLG_MAILADDR, szMailAddress);
      SetDlgItemText (hDlg, DLG_TEMPDIR,  szTempDir);
      
      CheckDlgButton(hDlg, DLG_DBLCLK_VIEW, bDblClkVu);
      CheckDlgButton(hDlg, DLG_DBLCLK_XFER, !bDblClkVu);

      if (nViewNum>0) SetViewerInfo (hDlg, nViewer=0);
      
      CheckDlgButton(hDlg, CKB_VERBOSE,     bVerbose);
      CheckDlgButton(hDlg, CKB_BELL,        bBell);
      CheckDlgButton(hDlg, CKB_GLOBBING,    bDoGlob);
      CheckDlgButton(hDlg, CKB_HASH,        bHash);
      CheckDlgButton(hDlg, CKB_PROMPT,      bInteractive);
      CheckDlgButton(hDlg, CKB_MCASE,       bMCase);
      CheckDlgButton(hDlg, CKB_PORT_CMDS,   bSendPort);
      CheckDlgButton(hDlg, CKB_RECV_UNIQUE, bRecvUniq);
      CheckDlgButton(hDlg, CKB_STOR_UNIQUE, bStorUniq);
      CheckDlgButton(hDlg, CKB_CRSTRIP,     bCRstrip);
      CheckDlgButton(hDlg, CKB_AUTOSTART,   bAutoStart);
      CheckDlgButton(hDlg, CKB_RETAINFILES, bRetain);
      CheckDlgButton(hDlg, CKB_DEBUGLOG,    bDebugLog);
      break;

    case WM_CLOSE: PostMessage (hDlg, WM_COMMAND, IDCANCEL, 0L);
                   break;

    case WM_SETCURSOR: if (bCmdInProgress) SetCursor (hWaitCursor);
                       else return FALSE;
                       break;

    case WM_VSCROLL: 
    {
#ifdef WIN32
      HWND hCtl = (HWND) lParam;
#else
      HWND hCtl = (HWND) HIWORD (lParam);
#endif
                       
      if (hCtl != GetDlgItem (hDlg, DLG_SCROLLVIEWER)) return FALSE;
      GetViewerInfo (hDlg);
      switch (wParam)
      {
        case SB_LINEUP  : if (nViewer>0) nViewer--; break;
        case SB_LINEDOWN: if (nViewer<(nViewNum-1)) nViewer++; break;
        default         : return FALSE;
      }
      SetViewerInfo (hDlg, nViewer);
      return TRUE;
    }
                       
    case WM_COMMAND: switch(wParam)
      {
        case IDOK:  fType = TYPE_L;
                    if (IsDlgButtonChecked (hDlg, RB_ASCII)) fType = TYPE_A;
                    else if (IsDlgButtonChecked (hDlg, RB_BINARY)) fType = TYPE_I;
                    bDblClkVu   = IsDlgButtonChecked (hDlg, DLG_DBLCLK_VIEW);
                    bVerbose    = IsDlgButtonChecked (hDlg, CKB_VERBOSE);
                    bBell       = IsDlgButtonChecked (hDlg, CKB_BELL);
                    bDoGlob     = IsDlgButtonChecked (hDlg, CKB_GLOBBING);
                    bHash       = IsDlgButtonChecked (hDlg, CKB_HASH);
                    bInteractive= IsDlgButtonChecked (hDlg, CKB_PROMPT);
                    bMCase      = IsDlgButtonChecked (hDlg, CKB_MCASE);
                    bSendPort   = IsDlgButtonChecked (hDlg, CKB_PORT_CMDS);
                    bCRstrip    = IsDlgButtonChecked (hDlg, CKB_CRSTRIP);
                    bRecvUniq   = IsDlgButtonChecked (hDlg, CKB_RECV_UNIQUE);
                    bStorUniq   = IsDlgButtonChecked (hDlg, CKB_STOR_UNIQUE);
                    bAutoStart  = IsDlgButtonChecked (hDlg, CKB_AUTOSTART);
                    bRetain     = IsDlgButtonChecked (hDlg, CKB_RETAINFILES);
                    bDebugLog   = IsDlgButtonChecked (hDlg, CKB_DEBUGLOG);
                    GetDlgItemText (hDlg, DLG_EDIT, szViewer, 70);
                    GetDlgItemText (hDlg, DLG_MAILADDR, szMailAddress, 127);
                    GetDlgItemText (hDlg, DLG_TEMPDIR,  szTempDir, 70);
                    GetViewerInfo  (hDlg);
                    CreateTempFileNames (szTempDir);
                    EndDialog (hDlg, TRUE);
                    break;

        case IDCANCEL: EndDialog(hDlg, FALSE); break;
      }  return TRUE;

    default: return FALSE;
  }
  return TRUE;
}

#ifdef WIN32
#define  WINFTPICON   "winftpnt"
#else
#define  WINFTPICON   "winftp"
#endif

/************************************************************************/
/* nCwRegisterClasses Function                                          */
/* The following function registers all the classes of all the windows  */
/* associated with this application. The function returns an error code */
/* if unsuccessful, otherwise it returns 0.                             */
/************************************************************************/
int nCwRegisterClasses(void)
{
  WNDCLASS   wc;    // struct to define a window class

  memset(&wc, 0x00, sizeof(wc));
  wc.style = CS_HREDRAW | CS_VREDRAW | CS_BYTEALIGNWINDOW;
  wc.lpfnWndProc = WndProc;
  wc.cbClsExtra = 0;
  wc.cbWndExtra = 0;
  wc.hInstance = hInst;
  wc.hIcon = LoadIcon (hInst, WINFTPICON);
  wc.hCursor = LoadCursor ((HINSTANCE) NULL, IDC_ARROW);
  wc.hbrBackground = hbrGray1;
  wc.lpszMenuName = "WINFTP";   /* Menu Name is App Name */
  wc.lpszClassName = szAppName; /* Class Name is App Name */
  if (RegisterClass (&wc)==0) return -1;

  wc.style = CS_HREDRAW | CS_VREDRAW;
  wc.lpfnWndProc   = WndXferProc;
  wc.hIcon         = LoadIcon (hInst, WINFTPICON);
  wc.hbrBackground = hbrGray1;
  wc.lpszMenuName  = NULL;      /* Menu Name is App Name */
  wc.lpszClassName = szXferWnd; /* Class Name is App Name */
  if (RegisterClass (&wc)==0) return -1;

  wc.style = CS_HREDRAW | CS_VREDRAW;
  wc.lpfnWndProc   = WndMsgProc;
  wc.hIcon         = LoadIcon ((HINSTANCE) NULL, IDI_ASTERISK);
  wc.hbrBackground = GetStockObject (WHITE_BRUSH);
  wc.lpszMenuName  = NULL;      /* Menu Name is App Name */
  wc.lpszClassName = szMsgWnd; /* Class Name is App Name */
  if (RegisterClass (&wc)==0) return -1;
  return 0;
}

/************************************************************************/
/*  cwCenter Function                                                   */
/*  centers a window based on the client area of its parent             */
/************************************************************************/
void cwCenter(HWND hWnd, int top)
{
  POINT      pt;
  RECT       swp;
  RECT       rParent;
  int        iwidth;
  int        iheight;

  // get the rectangles for the parent and the child
  GetWindowRect (hWnd, &swp);
  GetClientRect (hWndMain, &rParent);

  // calculate the height and width for MoveWindow
  iwidth = swp.right - swp.left;
  iheight = swp.bottom - swp.top;

  // find the center point and convert to screen coordinates
  pt.x = (rParent.right - rParent.left) / 2;
  pt.y = (rParent.bottom - rParent.top) / 2;
  ClientToScreen (hWndMain, &pt);

  // calculate the new x, y starting point
  pt.x = pt.x - (iwidth / 2);
  pt.y = pt.y - (iheight / 2);

  // top will adjust the window position, up or down
  if (top) pt.y = pt.y + top;

  // move the window
  MoveWindow(hWnd, pt.x, pt.y, iwidth, iheight, FALSE);
}

/************************************************************************/
/*  CwUnRegisterClasses Function                                        */
/*  Deletes any refrences to windows resources created for this         */
/*  application, frees memory, deletes instance, handles and does       */
/*  clean up prior to exiting the window                                */
/************************************************************************/
void CwUnRegisterClasses (void)
{
  WNDCLASS   wc;    // struct to define a window class
  
  memset (&wc, 0x00, sizeof(wc));
  UnregisterClass (szAppName, hInst);
  UnregisterClass (szXferWnd, hInst);
}

//******************************************************************************
//******************************************************************************
LRESULT CALLBACK LocalDirProc (HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
  if (msg==WM_KEYDOWN && wParam==VK_RETURN)
#ifdef WIN32
     SendMessage (hWndMain, WM_COMMAND, MAKEWPARAM (EDT_LFILETYPE, EN_TEXTCHANGE), MAKELPARAM (hwnd, 0));
#else
     SendMessage (hWndMain, WM_COMMAND, EDT_LFILETYPE, MAKELONG (hwnd, EN_TEXTCHANGE));
#endif
  return CallWindowProc (lpfnOldLocal, hwnd, msg, wParam, lParam);
}

//******************************************************************************
//******************************************************************************
LRESULT CALLBACK RemoteDirProc (HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
  if (msg==WM_KEYDOWN && wParam==VK_RETURN)
#ifdef WIN32
     SendMessage (hWndMain, WM_COMMAND, MAKEWPARAM (EDT_RFILETYPE, EN_TEXTCHANGE), MAKELPARAM (hwnd, 0));
#else
     SendMessage (hWndMain, WM_COMMAND, EDT_RFILETYPE, MAKELONG (hwnd, EN_TEXTCHANGE));
#endif
  return CallWindowProc (lpfnOldRemote, hwnd, msg, wParam, lParam);
}

