// ==========================================================================
// talk.c - A talk client for MS-DOS. Uses the WATTCP library.
//
// (C) 1993, 1994 by Michael Ringe, Institut fuer Theoretische Physik,
// RWTH Aachen, Aachen, Germany (michael@thphys.physik.rwth-aachen.de)
//
// This program is free software; see the file COPYING for details.
// ==========================================================================

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <conio.h>
#include <dos.h>
#include "..\include\tcp.h"

#if !defined(VERSION)
#define VERSION "0.0"
#endif
#define DEBUG 0		// Debugging
#define NONET 0		// Disable network

// Keyboard codes returned by readkey()

#define CR	0x000d
#define ESC	0x001b
#define ALT_S	0x1F00
#define ALT_L	0x2600
#define ALT_C	0x2E00
#define F1	0x3b00
#define HOME	0x4700
#define LEFT	0x4b00
#define DEL	0x5300
#define C_LEFT	0x7300

// Ports used by talk

#define MY_DATA_PORT 1050	// Our local data port
#define MY_CTL_PORT 1051	// Our local control port
#define TALK_PORT 518		// Where talkd is listening
#define MSG_ID 1224568

// Debugging

#if (DEBUG)
#define DB_puts(s) puts(s)
#define DB_printf(s) printf s
#define DB_ipsplit(ip) (int)((ip) >> 24) & 0xff,(int)((ip) >> 16) & 0xff,\
	(int)((ip) >> 8) & 0xff,(int)(ip) & 0xff
#else
#define DB_puts(s)
#define DB_printf(s)
#define DB_ipsplit(ip) 0
#endif

#if (NONET)
#define tcp_tick(s) 1
#define sock_init() tcp_config(NULL)
#define	sock_putc(s,ch)
#define sock_fastread(s,b,sz) 0
#define resolve(h) 0x13141516
#endif



// Global data

char version[] = "Talk version " VERSION ", written by Michael Ringe\r\n";
char usage[] = "Usage: talk [-l] [user@host]\r\n  or   talk [-l] alias\r\n"
	       "  or   talk -a\r\n";
char aamsg1[] = "Hello %s, %60s\n";
char *aamsg2[11] =
	{"This talk is running in answer mode.",
	 "If you don't mind talking to a machine you may leave",
	 "a message. Your input will be logged and read later.",
	 "-- %s",
	 NULL
	};
char *logfilename = "talk.log";	// Log file name
char
    *userarg = NULL,		// Command line arg, NULL = wait
    *localuser = "PC-User",	// My user name
    *remoteuser = NULL,		// Remote user name
    *remotehost = NULL;		// Remote host name
int answermode = 0;		// Operate as answering machine
int log = 0;			// Log session to log file
longword remoteip = 0;		// Remote IP address
tcp_Socket ds;			// Talk socket
char scrnmode = 0;		// Screen mode: 0=single window, 1=split
FILE *lf;			// Log file

// Screen colors

#define PAL_SIZE 5		// Palette size
#define P_LOCAL 0		// Palette indexes
#define P_REMOTE 1
#define P_STATUS 2
#define P_STATUSL 3
#define P_STATUSR 4
#define A_NORM 0x07

unsigned short
    a[2][PAL_SIZE] =
    {{0x0F,0x70,0x70,0x7F,0x70},	// Color set for single window mode
     {0x0F,0x0F,0x70,0x70,0x70}		// Color set for split screen mode
    };

// The following is used to handle multiple windows

static int x[2] = {1,1};		// Saved cursor position
static int y[2] = {1,1};
static int where = -1;			// Current window
static int ystatus;			// Status line position
unsigned char st[80][2];		// Status line buffer
int newcolors = 1;			// Color change flag

// Edit characters. By convention the first three characters each
// talk transmits after connection are the three edit characters.

#define EC_SIZE 3
#define EC_ERASE 0	// Erase char
#define EC_KILL 1	// Kill line
#define EC_WERASE 2	// Erase word

static char my_ec[3] = { 0x7f, 0x15, 0x17 };
static char his_ec[3] = { 0, 0, 0 };

// The following stuff is adapted from Linux's talkd and talk source

typedef struct {
	byte	 vers;
	byte	 type;
	byte	 answer;
	byte	 pad;
	longword id_num;
	struct	 sockaddr addr;
	short    pad1;		// because WATTCP's sockaddr is only 14 bytes
	struct	 sockaddr ctl_addr;
	short    pad2;		// see above
	long	 pid;
#define	NAME_SIZE	12
	char	 l_name[NAME_SIZE];
	char	 r_name[NAME_SIZE];
#define	TTY_SIZE	16
	char	 r_tty[TTY_SIZE];
} CTL_MSG;

typedef struct {
	byte	 vers;
	byte	 type;
	byte	 answer;
	byte	 pad;
	longword id_num;
	struct	 sockaddr addr;
	short    pad1;		// see above
} CTL_RESPONSE;

#define TALK_VERSION	1

#define LEAVE_INVITE	0
#define LOOK_UP		1
#define DELETE		2
#define ANNOUNCE	3

#define SUCCESS		0
#define NOT_HERE	1
#define FAILED		2
#define MACHINE_UNKNOWN	3
#define PERMISSION_DENIED 4
#define UNKNOWN_REQUEST	5
#define	BADVERSION	6
#define	BADADDR		7
#define	BADCTLADDR	8
#define MAXANSWER	8



static char *answermsg[MAXANSWER+2] = {
	"",
	"Your party is not logged on",
	"Target machine is too confused to talk to us",
	"Target machine does not recognize us",
	"Your party is refusing messages",
	"Target machine can not handle remote talk",
	"Target machine indicates protocol mismatch",
	"Target machine indicates protocol botch (addr)",
	"Target machine indicates protocol botch (ctl_addr)",

	// All error codes > MAXANSWER will be mapped to MAXANSWER + 1
	"Unknown error code from target machine"
};


// --------------------------------------------------------------------------
// Some utility functions:
// mystrncpy() - Safe strncpy
// setsockaddr() - Set socketaddr fields in network byte order
// --------------------------------------------------------------------------

static char *mystrncpy(char *dest, char *src, int len)

{
    strncpy(dest,src,len);
    dest[len-1] = 0;
    return dest;
}

static void setsockaddr(struct sockaddr *s, word port, longword ip)

{
    s->s_type = 0x0200;		/* AF_INET */
    s->s_port = intel16(port);
    s->s_ip = intel(ip);
    memset(s->s_spares,0,sizeof(s->s_spares));
}


// --------------------------------------------------------------------------
// dostime() - Return date and time in ascii format
// --------------------------------------------------------------------------

static char *dostime(void)

{
    struct dosdate_t d;
    struct dostime_t t;
    static char buf[25];
    static char dow[7][4] = {"Sun","Mon","Tue","Wed","Thu","Fri","Sat"};
    static char mon[12][4] = {"Jan","Feb","Mar","Apr","May","Jun","Jul",
			      "Aug","Sep","Oct","Nov","Dec"};

    _dos_gettime(&t);
    _dos_getdate(&d);
    sprintf(buf,"%s %s %2.2d %2.2d:%2.2d:%2.2d %4.4d",
	dow[d.dayofweek],mon[d.month-1],d.day,t.hour,t.minute,t.second,
	d.year);
    return buf;
}


// --------------------------------------------------------------------------
// DB_dumpctl() - Display control message (for debugging)
// DB_dumpans() - Display answer message (for debugging)
// --------------------------------------------------------------------------

#if (DEBUG)

static void DB_dumpctl( CTL_MSG *m)
{
    printf("CTL_MSG     : vers=%d type=%d id=%8.8lx addr=%d.%d.%d.%d:%u\n"
	   "              ctl_addr=%d.%d.%d.%d:%u\n"
	   "              l_name='%s' r_name='%s' r_tty='%s'\n",
	   (int)m->vers,(int)m->type,(unsigned long)m->id_num,
	   DB_ipsplit(intel(m->addr.s_ip)),intel16(m->addr.s_port),
	   DB_ipsplit(intel(m->ctl_addr.s_ip)),intel16(m->ctl_addr.s_port),
	   m->l_name,m->r_name,m->r_tty);
}

static void DB_dumpans(CTL_RESPONSE *m)
{
    printf("CTL_RESPONSE: vers=%d type=%d answer=%d id=%8.8lx"
	   " addr=%d.%d.%d.%d:%u\n", (int)m->vers,(int)m->type,
	   (int)m->answer,(unsigned long)m->id_num,
	   DB_ipsplit(intel(m->addr.s_ip)),intel16(m->addr.s_port));
}

#else

#define DB_dumpctl(m)
#define DB_dumpans(m)

#endif


// --------------------------------------------------------------------------
// Initialization stuff
// --------------------------------------------------------------------------

static unsigned int xtoi(char *c)

{
    int i = 0, k;
    for (k = 0; k < 2; ++k)
    {
	i <<= 4;
	if (*c >= '0' && *c <= '9') i += *c - '0';
	else if (*c >= 'A' && *c <= 'F') i += *c - 'A' + 10;
	else if (*c >= 'a' && *c <= 'f') i += *c - 'a' + 10;
	++c;
    }
    return i;
}

static void my_usr_init(char *key, char *val)

{
    if (*val == '$')
    {
#if (DEBUG)
	char *old = val;
#endif
	val = getenv(strupr(val+1));
	DB_printf(("%s: '%s' expands to '%s'\n",key,old,val));
	if (val == NULL) return;	// Use default
    }

    if (!stricmp(key,"talk_localname"))
    {
	localuser = strdup(val);
    }
    else if (!stricmp(key,"talk_screenmode"))
    {
	if (!stricmp(val,"split"))
	    scrnmode = 1;
    }
    else if (!stricmp(key,"talk_colors"))
    {
	unsigned short *x = a[0];
	int i,j;

	for (j = 0; j < 2 && *val != 0; ++j)
	{
	    for (i = PAL_SIZE; i > 0; --i)
	    {
		*x++ = xtoi(val);
		if (*++val != 0) ++val;
	    }
	    if (*val != 0) ++val;
	}
	if (j == 1)
	    memcpy(a+1,a,PAL_SIZE*sizeof(short));
    }
    else if (!stricmp(key,"talk_alias"))
    {
	char *c = strchr(val,':');
	if (c != NULL)
	{
	    *c++ = 0;
	    if (userarg != NULL && !strcmp(val,userarg))
		userarg = strdup(c);
	}
    }
    else if (!stricmp(key,"talk_logfile"))
    {
	logfilename = strdup(val);
    }
    else if (!stricmp(key,"talk_message"))
    {
	static int n = 0;

	if (n < 10)
	{
	    aamsg2[n++] = strdup(val);
	    aamsg2[n] = NULL;
	}
    }
}

static int init(int argc, char *argv[])

{
    int i;
    char *c;

    textattr(A_NORM);
    cputs(version);

    // Process options
    for (i = 1; i < argc && *(c = argv[i]) == '-'; ++i)
    {
	while (*++c != 0)
	{
	    if (*c == 'a') answermode = 1;
	    else if (*c == 'l') log = 1;
	}
    }

    // Process remaining arguments
    if (argc - i > 1 || (argc - i == 1 && answermode))
    {
	cputs(usage);
	return 1;
    }
    else if (argc - i == 1)
	userarg = argv[i];

    // Start up wattcp
    usr_init = my_usr_init;
    sock_init();

    // Enter message
    if (answermode)
    {
	int n;
	char tmp[100];
	userarg = NULL;
	log = 1;
	cputs("Enter Message (end with a single '.'):\r\n");
	for (n = 0; n < 10; ++n)
	{
	    *tmp = 79;
	    cgets(tmp);
	    cputs("\r\n");
	    if (tmp[1] == 1 && tmp[2] == '.' || n == 0 && tmp[1] == 0) break;
	    aamsg2[n] = strdup(tmp+2);
	    aamsg2[n+1] = NULL;
	}
    }

    // Split argument into user and host name
    if (userarg != NULL)
    {
	char *c = strchr(userarg,'@');
	if (c == NULL)
	{
	    cputs(usage);
	    return 1;
	}
	*c++ = 0;
	remoteuser = userarg;
	remotehost = c;

	// Look up remote IP address
	if (!(remoteip = resolve(remotehost)))
	{
	    cprintf("Error: Cannot resolve '%s'\r\n",remotehost);
	    return 1;
	}
	DB_printf(("Remote IP: %d.%d.%d.%d\n",DB_ipsplit(remoteip)));

    }
    DB_printf(("Local user: %s\nRemote user: %s\nRemote host: %s\n",
	localuser,remoteuser,remotehost));
    return 0;
}


// --------------------------------------------------------------------------
// wait_invite() - Wait for a remote invitation or for the remote party to
//                 reply.
// --------------------------------------------------------------------------

static int wait_invite(void)

{
    udp_Socket daemon;
    CTL_MSG ctl;
    CTL_RESPONSE ans;
    word status;
    static char localuserstr[NAME_SIZE];

    if (!udp_open(&daemon,TALK_PORT,0,0,NULL))
	goto sock_err;
    DB_printf(("Listening on port %d\n",TALK_PORT));

    // Wait for LOOK_UP
    cprintf("Waiting for %s... press ESC to quit\r\n", (userarg == NULL) ?
	"invitation" : "your party to respond");
    do
    {
	ctl.type = LOOK_UP + 1;
	if (kbhit() && getch() == 27)
	{
	    sock_close(&daemon);
	    return 27;
	}
	tcp_tick(&daemon);
	if (sock_fastread(&daemon,(byte *)&ctl,sizeof(ctl)) != sizeof(ctl))
	    continue;
	DB_dumpctl(&ctl);
    }
    while (ctl.vers != TALK_VERSION || ctl.type != LOOK_UP);
    sock_close(&daemon);

    // Reply with SUCCESS
    ans.vers = TALK_VERSION;
    ans.type = LOOK_UP;
    ans.answer = SUCCESS;
    ans.id_num = MSG_ID;
    setsockaddr(&ans.addr,MY_DATA_PORT,my_ip_addr);
    sock_write(&daemon,(char *)&ans,sizeof(ans));
    DB_dumpans(&ans);
    DB_puts("Replied with SUCCESS\n");

    // Wait for tcp connection
    tcp_listen(&ds,MY_DATA_PORT,0,0,NULL,0);
    sock_wait_established(&ds,0,NULL,&status);

    // Assume the requested user name
    mystrncpy(localuserstr,ctl.r_name,NAME_SIZE);
    localuser = localuserstr;

    // Set remote host name (sorry, no gethostbyaddr()...)
    if (userarg == NULL)
    {
	static char remoteipstr[16], remoteuserstr[NAME_SIZE];
	struct sockaddr sa;
	int l = sizeof(sa);

	getpeername(&ds,&sa,&l);
	inet_ntoa(remoteipstr,sa.s_ip);
	remotehost = remoteipstr;
	mystrncpy(remoteuserstr,ctl.l_name,NAME_SIZE);
	remoteuser = remoteuserstr;
    }
    return 0;

sock_err:
    return 1;
}


// --------------------------------------------------------------------------
// send_ctl() - Send a CTL_MSG to the remote talkd. Returns 0 on success.
// --------------------------------------------------------------------------

static int send_ctl(udp_Socket *cs, CTL_MSG *m, CTL_RESPONSE *r)

{
    int count;
    word status = 0;

    for (count = 10; status != 1 && count > 0; --count)
    {
	DB_puts("Sending message");
	DB_dumpctl(m);
	sock_write(cs,(byte *)m,sizeof(*m));
	if (_ip_delay1(cs,2,NULL,&status)) continue;
	if (sock_read(cs,(byte *)r,sizeof(*r)) != sizeof(*r)) continue;
	DB_dumpans(r);
	if (r->vers != TALK_VERSION || r->type != m->type) continue;
	DB_printf(("Valid response received, answer=%d (%s)\n",
	    r->answer,answermsg[r->answer > MAXANSWER ? MAXANSWER+1:
	    r->answer]));
	return 0;
    }
    cputs("No response from target machine, giving up\r\n");
    return 1;
}


// --------------------------------------------------------------------------
// invite() - Announce an invitation to the remote talkd.
// --------------------------------------------------------------------------

static int invite(void)

{
    udp_Socket cs;		// Control socket
    CTL_MSG ctl;		// Control message
    CTL_RESPONSE ans;		// Response from remote talkd
    word status;
    int a;
    int count;

    // Step I: Connect to remote talkd
    // -------------------------------
    if (!udp_open(&cs,MY_CTL_PORT,remoteip,TALK_PORT,NULL))
	goto sock_err;
    sock_wait_established(&cs,sock_delay,NULL,&status);
    DB_puts("Connection to remote talkd established");

    // Step II: Check for invitation on remote machine
    // -----------------------------------------------
    DB_puts("Lookup on remote machine");
    // Prepare the LOOK_UP message
    ctl.vers = TALK_VERSION;
    ctl.type = LOOK_UP;
    ctl.id_num = MSG_ID;
    setsockaddr(&ctl.addr,0,0);
    setsockaddr(&ctl.ctl_addr,MY_CTL_PORT,my_ip_addr);
    mystrncpy(ctl.l_name,localuser,NAME_SIZE);
    mystrncpy(ctl.r_name,remoteuser,NAME_SIZE);
    mystrncpy(ctl.r_tty,"",TTY_SIZE);
    // Send the LOOK_UP message
    if (send_ctl(&cs,&ctl,&ans)) return 1;
    // If the remote talkd had an invitation for us, connect now
    if (ans.answer == SUCCESS)
    {
	tcp_open(&ds,MY_DATA_PORT,intel(ans.addr.s_ip),
	    intel16(ans.addr.s_port),NULL);
	sock_wait_established( &ds, 0, NULL, &status);
	return 0;
    }

    // Step III: Announce our invitation to the remote talkd
    // -----------------------------------------------------
    DB_puts("Announcing invitation to remote talkd");
    // Prepare the ANNOUNCE message
    ctl.type = ANNOUNCE;
    setsockaddr(&ctl.addr,MY_DATA_PORT,my_ip_addr);
    // Send the ANNOUNCE message
    if (send_ctl(&cs,&ctl,&ans)) return 1;

    // If successfull, wait for remote party to reply. If this
    // fails, delete the invitation and exit.
    // -------------------------------------------------------
    a = ans.answer;
    if (a == SUCCESS)
    {
	if (wait_invite())
	{
	    DB_puts("Deleting invitation\r\n");
	    ctl.type = DELETE;
	    sock_write(&cs,(byte *)&ctl,sizeof(ctl));
	    return 1;
	}
	else
	    return 0;
    }

    // Invitation failed: Print error message from remote talkd and exit
    // -----------------------------------------------------------------
    if (a > MAXANSWER) a = MAXANSWER + 1;
    cprintf("%s\r\n",answermsg[a]);
    return 1;

sock_err:
    cputs("Error: Cannot open connection.\r\n");
    return 1;
}


// --------------------------------------------------------------------------
// lwrite() - Write char to log file. mode == 0: local, mode == 1: remote
// --------------------------------------------------------------------------

static void lwrite(int mode, char ch)

{
    static char buf[100];
    static int lmode = -1;
    static int lcount = 0;

    if (lf == NULL) return;
    if (mode != lmode || lcount >= 77 || ch == '\n')
    {
	if (lcount > 0) 		// Flush buffer
	{
	    buf[lcount] = '\n';
	    buf[lcount+1] = 0;
	    fputs(lmode ? ">" : "<",lf);
	    fputs(buf,lf);
	    lcount = 0;
	}
	lmode = mode;
    }
    if (ch == 0 || ch == '\n') return;
    if (ch == 8 || ch == 127)
    {
	if (lcount > 0) --lcount;
    }
    else
	buf[lcount++] = ch;
}


// --------------------------------------------------------------------------
// openlog() - Open and close log file. mode == 0: close; mode == 1: open;
//             mode == 2: toggle
// --------------------------------------------------------------------------

static void openlog(int mode)

{
    if (mode == 2) mode = (lf == NULL);
    if (lf != NULL)
    {
	lwrite(-1,0);	// Flush log file
	fclose(lf);
    }
    if (mode)
    {
	if ((lf = fopen(logfilename,"a+t")) != NULL)
	{
	    fprintf(lf,"\n***** %s@%s (%s) *****\n",remoteuser,
		  remotehost,dostime());
	    lwrite(-1,0);
	}
    }
    else
	lf = NULL;
}


// --------------------------------------------------------------------------
// exchange_edit_chars() - Exchange the three edit characters with the
//                         remote talk.
// --------------------------------------------------------------------------

static void exchange_edit_chars(void)

{
    sock_write(&ds,my_ec,3);
    sock_read(&ds,his_ec,3);
    DB_printf(("Remote edit characters: ERASE=%x, KILL=%x, WERASE=%x\n",
	    his_ec[0],his_ec[1],his_ec[2]));
}


// --------------------------------------------------------------------------
// aputs() - Fill buffer with char/attribute pairs
// --------------------------------------------------------------------------

static void aputs(int pos,int len, unsigned short attr, char *s)

{
    unsigned short *d = (unsigned short *) st + pos;
    while (*s != 0 && len-- > 0)
	*d++ = (attr << 8) | (unsigned char) *s++;
}


// --------------------------------------------------------------------------
// select() - Select window (0 = local, 1 = remote)
// --------------------------------------------------------------------------

static void select(int win)

{
    if (where == win) return;

    // Save cursor position
    if (where != -1)
    {
	x[scrnmode ? where : 0] = wherex();
	y[scrnmode ? where : 0] = wherey();
    }

    // Change window
    if (win == -1)
    {
	window(1,1,80,25);
	textattr(A_NORM);
    }
    else
    {
	textattr(a[scrnmode][win]);
	if (scrnmode == 0)
	    window(1,1,80,24);
	else
	    window(1,win ? 14 : 1,80,win ? 25 : 12);
	// Restore cursor position
	gotoxy(x[scrnmode ? win : 0],y[scrnmode ? win : 0]);
    }

    where = win;
}


// --------------------------------------------------------------------------
// cleanup() - Clean up screen
// --------------------------------------------------------------------------

static void cleanup(int i)

{
    select(-1);
    gotoxy(1,25);
    if (i == 1) cputs("\r\nYou closed.");
    else if (i == 2) cputs("\r\nConnection closed by remote user.");
    clreol();
    if (answermode) cputs("\n\r");
}


// --------------------------------------------------------------------------
// init_screen() - Initialize windows
// --------------------------------------------------------------------------

static void init_screen(void)

{
    int i;

    select(-1);
    switch (scrnmode)
    {
	case 0:
	    ystatus = 25;
	    select(0);
	    clrscr();
	    break;
	case 1:
	    ystatus = 13;
	    select(0);
	    clrscr();
	    select(1);
	    clrscr();
	    break;
    }
    newcolors = 1;
}


// --------------------------------------------------------------------------
// myputch() - Print a character to the current window. Handles special
//             characters.
// --------------------------------------------------------------------------

static int abswherey(void)	// returns absolute cursor position

{
    struct REGPACK r;

    r.r_bx = 0;
    r.r_ax = 0x0300;
    intr(0x10,&r);
    return (r.r_dx >> 8) + 1;
}


static void myputch(char ch)

{
    if (lf != NULL) lwrite(where == 1,ch);
    if (ch == '\n')			// Newline
    {
	cputs("\r\n");
    }
    else if (ch == my_ec[EC_ERASE])	// Erase character
    {
	cputs("\b \b");
    }
    else if (ch == my_ec[EC_KILL])	// Clear line
    {
	gotoxy(1,wherey());
	clreol();
    }
    else if (ch == my_ec[EC_WERASE])	// Erase word
    {
	static char b[80][2];
	int i;
	int y = abswherey();
	gettext(1,y,80,y,b);
	for (i = wherex()-1; i > 0 && b[i][0] == ' '; --i);
	while (i > 0 && b[i-1][0] != ' ') --i;
	gotoxy(i+1,wherey());
	clreol();
    }
    else if (ch == 9)			// Tab
    {
	int i = 8 - (wherex() - 1) % 8;
	while (i-- > 0) putch(' ');
    }
    else
	putch(ch);
}


// --------------------------------------------------------------------------
// update_screen() - Called periodically to update the status line
// --------------------------------------------------------------------------

static void update_screen(void)

{
    static unsigned char h = 0, m = 0;
    static char buf[80];
    struct dostime_t t;
    int i;

    _dos_gettime(&t);
    if (t.minute != m || newcolors)
    {
	sprintf(buf,"%s@%s",remoteuser,remotehost);
	aputs(0,80,a[scrnmode][P_STATUS],
	  "                                                 "
	  "F1-Help   ESC-Quit    hh:mm");
	if (scrnmode == 1)
	{
	    aputs(0,1,a[scrnmode][P_STATUSL],"\x18");
	    aputs(10,1,a[scrnmode][P_STATUSR],"\x19");
	}
	aputs(scrnmode,9-scrnmode,a[scrnmode][P_STATUSL],localuser);
	aputs(10+scrnmode,40-scrnmode,a[scrnmode][P_STATUSR],buf);
	h = t.hour;
	m = t.minute;
	if (lf != NULL) aputs(70,1,a[scrnmode][P_STATUS],"L");
	sprintf(buf,"%2.2d:%2.2d",h,m);
	aputs(75,5,a[scrnmode][P_STATUS],buf);
	puttext(1,ystatus,80,ystatus,st);
	newcolors = 0;
    }
}


// --------------------------------------------------------------------------
// help() - Display a help screen
// --------------------------------------------------------------------------

static void help(void)

{
    short screen[80*25];
    int x, y;

    select(-1);
    gettext(1,1,80,25,screen);
    window(1,1,80,25);
    textattr(15);
    clrscr();
    cputs(version);

    gotoxy(1,5);
    cputs(
	"CONTROL KEYS\r\n"
	"\r\n"
	"Close connection .......... ESC\r\n"
	"Display help .............. F1\r\n"
	"Toggle display mode ....... Alt-S\r\n"
	"Toggle log file ........... Alt-L\r\n"
	"\r\n"
	"EDIT KEYS\r\n"
	"\r\n"
	"Erase character ........... Del, Backspace, Left\r\n"
	"Erase word ................ Ctrl-Left, Ctrl-W\r\n"
	"Clear line ................ Home, Ctrl-U\r\n"
      );
    gotoxy(1,25);
    cputs("Press any key to return");
    getch();
    puttext(1,1,80,25,screen);
    select(0);
}


// --------------------------------------------------------------------------
// readkey() - Read one key from keyboard. Returns values > 0xFF for
//             extended keys.
// --------------------------------------------------------------------------

static unsigned short readkey(void)

{
    unsigned short ch;

    if ((ch = getch()) != 0) return ch;
    return (getch() << 8);
}


// --------------------------------------------------------------------------
// talk() - This function does the actual talking
// --------------------------------------------------------------------------

static int talk(void)

{
    static char buf[500];
    unsigned short ch;
    int i;

#if (!NONET)
    exchange_edit_chars();
#endif
    DB_puts("Entering talk(), press return:");
#if DEBUG
    gets(buf);
#endif
    sound(2000);
    delay(100);
    nosound();
    init_screen();

    // If in answer mode, write the message to the remote screen
    // Using sock_printf here would use another 3kB memory.
    if (answermode)
    {
	int n;
	sprintf(buf,aamsg1,remoteuser,dostime());
	sock_puts(&ds,buf);
	for (n = 0; n < 10 && aamsg2[n] != NULL; ++n)
	{
	    sprintf(buf,aamsg2[n],localuser);
	    sock_puts(&ds,buf);
	    sock_putc(&ds,'\n');
	}
    }

    // Main loop, repeat until ESC pressed or socket closed
    while (tcp_tick(&ds))
    {
	update_screen();

	// Process keyboard input
	if (kbhit())
	{
	    ch = readkey();
	    // Handle some special cases
	    switch (ch)
	    {
		case ESC:
		    return 1;
		case F1:
		    help();
		    continue;
		case ALT_S:
		    scrnmode = 1 - scrnmode;
		    init_screen();
		    continue;
		case ALT_L:
		    openlog(2);
		    newcolors = 1;	// Redraw status line
		    continue;
		case 8:
		case DEL:
		case LEFT:
		    ch = my_ec[EC_ERASE];
		    break;
		case C_LEFT:
		    ch = my_ec[EC_WERASE];
		    break;
		case HOME:
		    ch = my_ec[EC_KILL];
		    break;
		case CR: ch = 10; break;
	    }

	    // If it's a normal character, write it to both
	    // local screen and remote host.
	    if (ch < 0x100)
	    {
		select(0);
		myputch(ch);

		// Translate edit characters
		for (i = 0; i < EC_SIZE; ++i)
		    if (ch == my_ec[i])
		    {
			ch = his_ec[i];
			break;
		    }
		sock_putc(&ds,ch);
	    }
	}

	// Process input from remote host
	i = sock_fastread(&ds,buf,sizeof(buf));
#if (NONET)
	i = 0;
	if (ch == ALT_C)
	{ i = 17;
	  strcpy(buf," 123 4567 199 8xxx");
	  ch = 0;
	}
#endif
	if (i > 0)
	{
	    int k;
	    select(1);
	    for (k = 0; k < i; ++k)
		myputch(buf[k]);
	}
    }
    return 2;		// Closed by remote party
}


// --------------------------------------------------------------------------
// main()
// --------------------------------------------------------------------------

void main(int argc, char *argv[])

{
    int i = 0;
    word s;

    if (init(argc,argv))
	exit(1);

#if (NONET)
    talk();
    openlog(0);
    cleanup(1);
    exit(0);
#endif

    do
    {
	if (userarg == NULL)
	    i = wait_invite();
	else
	    i = invite();

	if (i == 0)
	{
	    if (log) openlog(1);
	    i = talk();
	    openlog(0);
	    sock_abort(&ds);
	    sock_close(&ds);
	    if (lf != NULL)
		fclose(lf);
	    cleanup(i);
	}
    }
    while (answermode && i != 27);
    exit(0);
}

