/*
 * Program:	Tenex Format Mail Delivery Module
 *
 * Author:	Mark Crispin
 *		Networks and Distributed Computing
 *		Computing & Communications
 *		University of Washington
 *		Administration Building, AG-44
 *		Seattle, WA  98195
 *		Internet: MRC@CAC.Washington.EDU
 *
 * Date:	5 April 1993
 * Last Edited:	16 December 1993
 *
 * Copyright 1993 by the University of Washington
 *
 *  Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appears in all copies and that both the
 * above copyright notice and this permission notice appear in supporting
 * documentation, and that the name of the University of Washington not be
 * used in advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.  This software is made
 * available "as is", and
 * THE UNIVERSITY OF WASHINGTON DISCLAIMS ALL WARRANTIES, EXPRESS OR IMPLIED,
 * WITH REGARD TO THIS SOFTWARE, INCLUDING WITHOUT LIMITATION ALL IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE, AND IN
 * NO EVENT SHALL THE UNIVERSITY OF WASHINGTON BE LIABLE FOR ANY SPECIAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM
 * LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, TORT
 * (INCLUDING NEGLIGENCE) OR STRICT LIABILITY, ARISING OUT OF OR IN CONNECTION
 * WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */


#include <stdio.h>
#include <pwd.h>
#include <errno.h>
extern int errno;		/* just in case */
#include <sysexits.h>
#include <sys/file.h>
#include <sys/stat.h>
#include <sys/syslog.h>
#include "mail.h"
#include "osdep.h"
#include "misc.h"

				/* driver we use */
extern DRIVER tenexdriver,dummydriver;
extern MAILSTREAM tenexproto;	/* prototype stream for our driver */

int critical = NIL;		/* flag saying in critical code */

/* Function prototypes */

void file_string_init (STRING *s,void *data,unsigned long size);
char file_string_next (STRING *s);
void file_string_setpos (STRING *s,unsigned long i);
void main (int argc,char *argv[]);
int deliver (FILE *f,unsigned long msglen,char *s);
int fail (char *string,int code);
char *getusername (char *s,char **t);


/* File string driver for file stringstructs */

STRINGDRIVER file_string = {
  file_string_init,		/* initialize string structure */
  file_string_next,		/* get next byte in string structure */
  file_string_setpos		/* set position in string structure */
};


/* Cache buffer for file stringstructs */

#define CHUNKLEN 16384
char chunk[CHUNKLEN];

/* Initialize file string structure for file stringstruct
 * Accepts: string structure
 *	    pointer to string
 *	    size of string
 */

void file_string_init (STRING *s,void *data,unsigned long size)
{
  s->data = data;		/* note fd */
  s->size = size;		/* note size */
  s->chunk = chunk;
  s->chunksize = (unsigned long) CHUNKLEN;
  SETPOS (s,0);			/* set initial position */
}


/* Get next character from file stringstruct
 * Accepts: string structure
 * Returns: character, string structure chunk refreshed
 */

char file_string_next (STRING *s)
{
  char c = *s->curpos++;	/* get next byte */
				/* move to next chunk */
  SETPOS (s,s->offset + s->chunksize);
  return c;			/* return the byte */
}


/* Set string pointer position for file stringstruct
 * Accepts: string structure
 *	    new position
 */

void file_string_setpos (STRING *s,unsigned long i)
{
  s->offset = i;		/* set new offset */
				/* set size of data */
  s->cursize = min ((long) CHUNKLEN,s->size);
  s->curpos = s->chunk;		/* reset position */
				/* move to that position in the file */
  fseek ((FILE *) s->data,s->offset,SEEK_SET);
  fread (s->curpos,sizeof (char),(unsigned int) s->cursize,(FILE *) s->data);
}

/* Main program */

void main (int argc,char *argv[])
{
  FILE *f;
  int pid,status,c,ret = 0;
  unsigned long msglen = 0;
  char *s,*help = "usage: tmail [-r from_name] [-d] user";
  openlog ("tmail",LOG_PID,LOG_MAIL);
  mail_link (&tenexdriver);	/* link in our c-client driver */
  mail_link (&dummydriver);	/* must be lunk */
  mailstd_proto = NIL;		/* no system standard prototype */
				/* make sure have some arguments */
  if (--argc < 1) _exit (fail (help,EX_USAGE));
				/* process all flags */
  while (argc && (*(s = *++argv)) == '-') {
    argc--;			/* gobble this argument */
    switch (s[1]) {		/* what is this flag? */
    case 'd':			/* flag meaning multiple users */
      break;
    case 'r':			/* flag giving origin */
      if (argc--) argv++;	/* skip argument */
      else _exit (fail ("missing argument to -r",EX_USAGE));
      break;
    default:			/* anything else */
      _exit (fail ("unknown switch",EX_USAGE));
    }
  }
  if (argc) {			/* have any recipients? */
    if (!(f = tmpfile ())) _exit (fail ("can't make temp file",EX_TEMPFAIL));
				/* copy text from standard input */
    while ((c = getchar ()) != EOF) {
      putc (c,f);
      msglen++;			/* count up message length */
    }
    fflush (f);			/* make sure all changes written out */
    do {			/* create daughter fork to do the work */
      if ((pid = fork ()) < 0) ret = fail (strerror (errno),EX_OSERR);
      else if (pid) {		/* mother process */
	while (pid != wait (&status));
				/* normal termination? */
	if (!ret) ret = (status & 0xff) ? EX_SOFTWARE : (status & 0xff00) >> 8;
      }
				/* daughter process */
      else _exit (deliver (f,msglen,*argv));
    } while (--argc && *argv++);
    mm_dlog (ret ? "error in delivery" : "all recipients delivered");
  }
  else ret = fail ("no recipients",EX_USAGE);
  fclose (f);			/* all done with temporary file */
  _exit (ret);			/* normal exit */
}

/* Deliver message to recipient list
 * Accepts: file description of message temporary file
 *	    size of message temporary file in bytes
 *	    recipient
 * Returns: NIL if success, else error code
 */

int deliver (FILE *f,unsigned long msglen,char *s)
{
  char *t,tmp[MAILTMPLEN],mailbox[MAILTMPLEN];
  STRING st;
  struct passwd *pwd;
  struct stat sbuf;
  uid_t owner;
  uid_t root = geteuid ();	/* our privileged user ID */
  setruid (root);		/* make sure real ID is root */
  sprintf (tmp,"delivering to %s",s);
  mm_dlog (tmp);
  if (*s == '/') {		/* absolute file name? */
    strcpy (mailbox,s);		/* yes, copy as is */
    *(strrchr (s,'/')) = '\0';	/* tie off at directory name */
    if (stat (s,&sbuf)) {	/* get its information */
      sprintf (tmp,"can't get directory for file %s",s);
      return fail (tmp,EX_CANTCREAT);
    }
    owner = sbuf.st_uid;	/* set owner as owner of superior directory */
  }
				/* else must be user name */
  else if (pwd = getpwnam (getusername (s,&t))) {
				/* build mail spool name */
    sprintf (tmp,MAILFILE,pwd->pw_name);
				/* is mail spool a directory? */
    if ((!stat (tmp,&sbuf)) && (sbuf.st_mode & S_IFDIR))
      sprintf (mailbox,"%s/%s",tmp,t ? t : "INBOX");
    else sprintf (mailbox,"%s/%s.TxT",pwd->pw_dir,t ? t : "mail");
    owner = pwd->pw_uid;	/* owner is that user */
  }
  else {
    sprintf (tmp,"can't determine home directory for user %s",s);
    return fail (tmp,EX_NOUSER);
  }

  INIT (&st,file_string,(void *) f,msglen);
  seteuid (owner);		/* become the owner */
				/* do the append */
  if (!mail_append (&tenexproto,mailbox,&st)) {
				/* unknown folders deliver to INBOX */
    if (t) return deliver (f,msglen,s);
				/* create INBOX */
    sprintf (tmp,"attempting to create INBOX for %s",s);
    mm_dlog (tmp);
				/* can we create INBOX? */
    if (!mail_create (&tenexproto,mailbox)) {
      seteuid (root);		/* revert to root */
      return fail ("create failed",EX_CANTCREAT);
    }
    seteuid (root);		/* revert to root */
    chown (mailbox,owner,-1);	/* make it owned by that user */
    chmod (mailbox,0600);	/* and protected */
    mm_dlog ("mailbox created");
    seteuid (owner);		/* become the owner */
    if (!mail_append (&tenexproto,mailbox,&st)) {
      seteuid (root);		/* revert to root */
      return fail ("delivery failed",EX_TEMPFAIL);
    }
  }
  seteuid (root);		/* revert to root */
  chown (mailbox,owner,-1);	/* make it owned by that user */
  mm_dlog ("delivered");
  return NIL;			/* return success */
}

/* Report an error
 * Accepts: string to output
 */

int fail (char *string,int code)
{
  mm_log (string,ERROR);	/* pass up the string */
  return code;			/* error code to return */
}


/* Get user name from username+mailbox specifier
 * Accepts: username/mailbox specifier
 *	    pointer to return location for mailbox specifier
 * Returns: user name, mailbox specifier value NIL if INBOX, patches out +
 */

char *getusername (char *s,char **t)
{
  char tmp[MAILTMPLEN];
  if (*t = strchr (s,'+')) {	/* have a mailbox specifier? */
    *(*t)++ = '\0';		/* yes, tie off user name */
				/* user+INBOX same as user */
    if (!strcmp ("INBOX",ucase (strcpy (tmp,*t)))) *t = NIL;
  }
  return s;			/* return user name */
}

/* Co-routines from MAIL library */


/* Message matches a search
 * Accepts: MAIL stream
 *	    message number
 */

void mm_searched (MAILSTREAM *stream,long msgno)
{
  fatal ("impossible mm_searched() call");
}


/* Message exists (i.e. there are that many messages in the mailbox)
 * Accepts: MAIL stream
 *	    message number
 */

void mm_exists (MAILSTREAM *stream,long number)
{
  fatal ("impossible mm_exists() call");
}


/* Message expunged
 * Accepts: MAIL stream
 *	    message number
 */

void mm_expunged (MAILSTREAM *stream,long number)
{
  fatal ("impossible mm_expunged() call");
}


/* Message flags update seen
 * Accepts: MAIL stream
 *	    message number
 */

void mm_flags (MAILSTREAM *stream,long number)
{
  fatal ("impossible mm_flags() call");
}


/* Mailbox found
 * Accepts: Mailbox name
 */

void mm_mailbox (char *string)
{
  fatal ("impossible mm_mailbox() call");
}


/* BBoard found
 * Accepts: BBoard name
 */

void mm_bboard (char *string)
{
  fatal ("impossible mm_bboard() call");
}

/* Notification event
 * Accepts: MAIL stream
 *	    string to log
 *	    error flag
 */

void mm_notify (MAILSTREAM *stream,char *string,long errflg)
{
  mm_log (string,errflg);	/* just do mm_log action */
}


/* Log an event for the user to see
 * Accepts: string to log
 *	    error flag
 */

void mm_log (char *string,long errflg)
{
  fprintf (stderr,"%s\n",string);
  switch (errflg) {  
  case NIL:			/* no error */
    syslog (LOG_MAIL|LOG_INFO,"%s",string);
    break;
  case PARSE:			/* parsing problem */
  case WARN:			/* warning */
    syslog (LOG_MAIL|LOG_WARNING,"%s",string);
    break;
  case ERROR:			/* error */
  default:
    syslog (LOG_MAIL|LOG_ERR,"%s",string);
    break;
  }
}


/* Log an event to debugging telemetry
 * Accepts: string to log
 */

void mm_dlog (char *string)
{
  fprintf (stderr,"%s\n",string);
  syslog (LOG_MAIL|LOG_DEBUG,"%s",string);
}

/* Get user name and password for this host
 * Accepts: host name
 *	    where to return user name
 *	    where to return password
 *	    trial count
 */

void mm_login (char *host,char *username,char *password,long trial)
{
  fatal ("impossible mm_login() call");
}


/* About to enter critical code
 * Accepts: stream
 */

void mm_critical (MAILSTREAM *stream)
{
  critical = T;			/* note in critical code */
}


/* About to exit critical code
 * Accepts: stream
 */

void mm_nocritical (MAILSTREAM *stream)
{
  critical = NIL;		/* note not in critical code */
}


/* Disk error found
 * Accepts: stream
 *	    system error code
 *	    flag indicating that mailbox may be clobbered
 * Returns: T if user wants to abort
 */

long mm_diskerror (MAILSTREAM *stream,long errcode,long serious)
{
  return T;
}


/* Log a fatal error event
 * Accepts: string to log
 */

void mm_fatal (char *string)
{
				/* shouldn't happen normally */
  syslog (LOG_MAIL|LOG_ALERT,"%s",string);
}
