/******************************************************************************
 * Module xpi_TLtransact.c - Transaction model test tool for xperfisis
 *
 * Written:	 06/19/91 by John H. Lee
 * Last Changed: 06/21/91 by John H. Lee
 * Version:	 0.2
 ******************************************************************************
 * Description:
 *
 *	This tool is designed to carry out performance test of various
 * transactions models.  Currently three models are simulated:  by Atomic
 * Broadcast, by token passing, and by quorum.
 ******************************************************************************
 * Exports:
 *      ToolClass transactToolClass	Tool class record pointer
 *
 * Imports:
 *      xpi_graph routines
 *      xpi_util routines
 ******************************************************************************
 * Revisions:
 *   0.1   06/19/91 JHL  Initial Creation
 *   0.2   06/21/91 JHL  Modified to use the BYPASS mechanism
 *****************************************************************************/

#include <isis.h>
#include <X11/Intrinsic.h>
#include <X11/IntrinsicP.h>
#include <Xm/Xm.h>
#include <Xm/CascadeB.h>
#include <Xm/DrawingA.h>
#include <Xm/Form.h>
#include <Xm/LabelG.h>
#include <Xm/MainW.h>
#include <Xm/MenuShell.h>
#include <Xm/MessageB.h>
#include <Xm/PushB.h>
#include <Xm/PushBG.h>
#include <Xm/RowColumn.h>
#include <Xm/ScrollBar.h>

#include "xperfisis.h"
#include "xpi_TLtransact.h"

/*** Private Preprocessor Defines ***/

#define	GRAPH_SCT	0	/* Server call time graph		*/
#define	GRAPH_NTT	1	/* Normalized transaction time graph	*/
#define	GRAPH_ARW	2	/* Average read/write time graph	*/


/*** Private Global Variable Definitions ***/

static struct {
	char	*name;
	int	transaction_entry;
} TransactionList[] = {		/* List of transaction methods		*/
	{"ABcast",		ENTRY_TRANSABCAST},
	{"Token Passing",	ENTRY_TRANSTOKEN}, 
	{"Quorum",		ENTRY_TRANSQUORUM}, 
	{NULL, NULL}
};

static XmString	SeparatorStr,	/* Compound string separator		*/
		EmptyStr,	/* Empty compound string		*/
		OKStr,		/* "OK" string				*/
		StopStr,	/* "Stop" string			*/
		BusyTitleStr,		/* Busy--working dialog title	*/
		NoServersTitleStr,	/* No servers dialog title	*/
		NoServersStr,	/* No servers selected dialog message	*/
		NumSamplesTitleStr,	/* Num of samples dialog title	*/
		NumSamplesStr,	/* Num of samples dialog message	*/
		NumReadsTitleStr,	/* Num of reads dialog title	*/
		NumReadsStr,	/* Num of reads dialog message		*/
		NumWritesTitleStr,	/* Num of writes dialog title	*/
		NumWritesStr,	/* Num of writes dialog message		*/
		PacketSizeTitleStr,	/* Packet Size dialog title	*/
		PacketSizeStr;	/* Packet Size dialog messag		*/

/*** Tool Class Functions ***/

static void	ClassInit(),
		Init(),
		Destroy();




/*** Tool Instance Record ***/

struct _TransactToolRec_Struct {
	ToolCommonPart	common;
	Widget		draw_area,
			ginfo_lbl,
			tinfo_lbl,
			menu_panes[10],
			menu_cscs[10],
			working_dlg;
	EvDialog	ev_dlg;
	LiDialog	li_dlg;
	Graph		*graph;
	GDatum		*data,
			*data_transtime,
			*data_rwtime;
	unsigned	data_sz,
			data_transtime_sz,
			data_rwtime_sz,
			num_data,
			num_transtime,
			num_rwtime;
	unsigned	num_test;
	int		transaction_type,
			num_reads,
			num_writes,
			packet_size;
	int		packet_sz;
	int		dm_size;
	int		num_servers;
	address		*target_list,
			target_servers[PG_MAXMEMB+1],
			*site_list[PG_MAXMEMB+1];
	NotifyList	test_NL,
			test_done_NL;
	Bool		run_test;
};
typedef struct _TransactToolRec_Struct	TransactToolRec;



/*** Tool Class Record ***/

ToolClassRec	transactToolClassRec = {
    /* tool_title		*/ "Transaction--Transaction model tests",
    /* menu_name		*/ "Transaction",
    /* class_init		*/ ClassInit,
    /* init			*/ Init,
    /* destroy			*/ Destroy,
    /* instance_size		*/ sizeof(TransactToolRec),
    /* class_initialized	*/ False
};

ToolClassRec	*transactToolClass = &transactToolClassRec;

/*** Private Functions ***/

static XmString	CreateToolInfo();
static void	SetToolInfo();
static void	StartCB(),
		StopCB(),
		SetTransactionTypeCB(),
		SetNumSamplesCB(),
		SetNumReadsCB(),
		SetNumWritesCB(),
		SetPacketSizeCB(),
		SetTargetServersCB(),
		SetTargetServersChangeCB(),
		SetTargetServersConfirmCB(),
		DisplayGraphCB();
static void	TestUpdateNP(),
		TestTaskDoneNP();
static void	TestTask();

/******************************************************************************
 *
 *	Class Initialization
 *
 *		This procedure is called exactly once for this class before
 *	any instance of this class is created.
 *
 *****************************************************************************/

static void ClassInit()
{
XmString	msg_str1,
		msg_str2,
		msg_str3;


	/*** Register tasks ***/

	isis_task(TestTask, "test_task");


	/*** Create static compound strings ***/
				/* Compound String separator		*/
	SeparatorStr = XmStringSeparatorCreate();

				/* Empty compound string		*/
	EmptyStr = XmStringCreateSimple("");

				/* "OK" string				*/
	OKStr = XmStringCreateSimple("OK");

				/* "Stop" string			*/
	StopStr = XmStringCreateSimple("Stop");

				/* Busy--working title message		*/
	BusyTitleStr = XmStringCreateSimple("Test Running");

				/* No Servers selected title message	*/
	NoServersTitleStr = XmStringCreateSimple("No Servers Alert");
				/* No Servers selected dialog message	*/
	NoServersStr = XmStringCreateSimple(
		"At least one target server must be selected.");

				/* Number of Reads title message	*/
	NumReadsTitleStr = XmStringCreateSimple("Set Number of Reads");
				/* Number of Reads dialog message	*/
	msg_str1 = XmStringCreateSimple(
		"Please set desired number of reads per transaction.");
	msg_str2 = XmStringConcat(msg_str1, SeparatorStr);
	XmStringFree(msg_str1);
	msg_str1 = XmStringCreateSimple("");
	msg_str3 = XmStringConcat(msg_str2, msg_str1);
	XmStringFree(msg_str1);
	XmStringFree(msg_str2);
	msg_str1 = XmStringConcat(msg_str3, SeparatorStr);
	XmStringFree(msg_str3);
	msg_str2 = XmStringCreateSimple("Reads per transaction:");
	NumReadsStr = XmStringConcat(msg_str1, msg_str2);
	XmStringFree(msg_str1);
	XmStringFree(msg_str2);

				/* Number of Writes title message	*/
	NumWritesTitleStr = XmStringCreateSimple("Set Number of Reads");
				/* Number of Writes dialog message	*/
	msg_str1 = XmStringCreateSimple(
		"Please set desired number of Writes per transaction.");
	msg_str2 = XmStringConcat(msg_str1, SeparatorStr);
	XmStringFree(msg_str1);
	msg_str1 = XmStringCreateSimple("");
	msg_str3 = XmStringConcat(msg_str2, msg_str1);
	XmStringFree(msg_str1);
	XmStringFree(msg_str2);
	msg_str1 = XmStringConcat(msg_str3, SeparatorStr);
	XmStringFree(msg_str3);
	msg_str2 = XmStringCreateSimple("Writes per transaction:");
	NumWritesStr = XmStringConcat(msg_str1, msg_str2);
	XmStringFree(msg_str1);
	XmStringFree(msg_str2);

				/* Number of Samples title message	*/
	NumSamplesTitleStr = XmStringCreateSimple("Set Number of Samples");
				/* Number of Samples dialog message	*/
	msg_str1 = XmStringCreateSimple(
		"Please set desired number of samples to take.");
	msg_str2 = XmStringConcat(msg_str1, SeparatorStr);
	XmStringFree(msg_str1);
	msg_str1 = XmStringCreateSimple("");
	msg_str3 = XmStringConcat(msg_str2, msg_str1);
	XmStringFree(msg_str1);
	XmStringFree(msg_str2);
	msg_str1 = XmStringConcat(msg_str3, SeparatorStr);
	XmStringFree(msg_str3);
	msg_str2 = XmStringCreateSimple("Number of Samples:");
	NumSamplesStr = XmStringConcat(msg_str1, msg_str2);
	XmStringFree(msg_str1);
	XmStringFree(msg_str2);

				/* Packet Size title message		*/
	PacketSizeTitleStr = XmStringCreateSimple("Set PacketSize");
				/* Packet Size dialog message		*/
	msg_str1 = XmStringCreateSimple(
		"Please set desired packet size of reads & writes.");
	msg_str2 = XmStringConcat(msg_str1, SeparatorStr);
	XmStringFree(msg_str1);
	msg_str1 = XmStringCreateSimple("");
	msg_str3 = XmStringConcat(msg_str2, msg_str1);
	XmStringFree(msg_str1);
	XmStringFree(msg_str2);
	msg_str1 = XmStringConcat(msg_str3, SeparatorStr);
	XmStringFree(msg_str3);
	msg_str2 = XmStringCreateSimple("Bytes per read or write:");
	PacketSizeStr = XmStringConcat(msg_str1, msg_str2);
	XmStringFree(msg_str1);
	XmStringFree(msg_str2);
} /* function ClassInit */

/******************************************************************************
 *
 *	Initialization
 *
 *		This procedure is called to initialized a new instance of
 *	this tool class.  It is called exactly once per new instance by
 *	CreateTool().
 *
 *****************************************************************************/

static void Init(t)
Tool	t;
{
AppWindow	*app_win = t->common.app_win;
TransactTool	tt = (TransactTool)t;
Widget		menu_pane,
		menu_ent,
		submenu_pane,
		button;
XmString	cancel_str,
		msg_str,
		title_str,
		header_str;
int		i,
		mp,
		mc;
Arg		args[16];
Cardinal	n;


	/*** Initalize instance data structure ***/

	tt->data = NULL;
	tt->data_sz = 0;
	tt->num_data = 0;
	tt->num_test = 1000;
	tt->transaction_type = 0;	/* Transaction by ABcast	*/
	tt->num_reads = 1;
	tt->num_writes = 1;
	tt->packet_size = 1;
	tt->run_test = False;
	tt->packet_sz = 0;
	tt->dm_size = 0;		/* Not yet implemented		*/
	tt->num_servers = 0;
	tt->target_list = NULL;
	memcpy(&tt->target_servers[0], &NULLADDRESS, sizeof(address));


	mp = 0;			/* Initialize number of menu panes	*/
	mc = 0;			/* Initialize number of menu csc buttons */


	/*** Add "Parameters" menu to menu bar ***/

	n = 0;
	menu_pane = XmCreatePulldownMenu(app_win->menu_bar,
		"menu_pane", args, n);
	tt->menu_panes[mp++] = menu_pane;

				/*** Transaction Type submenu 		***/
	n = 0;
	submenu_pane = XmCreatePulldownMenu(menu_pane,
		"submenu_pane", args, n);
	tt->menu_panes[mp++] = submenu_pane;

				/* Transaction submenu menu items	*/
	for (i=0; TransactionList[i].name != NULL; i++) {
	    n = 0;
	    XtSetArg(args[n], XmNuserData, i);				n++;
	    menu_ent = XmCreatePushButton(submenu_pane,
		    TransactionList[i].name, args, n);
	    XtManageChild(menu_ent);
	    XtAddCallback(menu_ent, XmNactivateCallback, SetTransactionTypeCB,
		    (caddr_t)t);
	} /* for */

	n = 0;
	XtSetArg(args[n], XmNsubMenuId, submenu_pane);			n++;
	tt->menu_cscs[mc] = XmCreateCascadeButton(menu_pane,
		"Transaction Type", args, n);
	XtManageChild(tt->menu_cscs[mc]);
	mc++;

				/* DM Group Size menu item		*/
				/*### NOT YET IMPLEMENTED ###		*/
	n = 0;
	XtSetArg(args[n], XmNsensitive, False);				n++;
	menu_ent = XmCreatePushButton(menu_pane, "DM Group...", args, n);
#ifdef	NOTIMPLEMENTED
	XtAddCallback(menu_ent, XmNactivateCallback, SetDMGroupCB,
		(caddr_t)t);
#endif	NOTIMPLEMENTED
	XtManageChild(menu_ent);

				/* Number of Samples menu item		*/
	n = 0;
	menu_ent = XmCreatePushButton(menu_pane, "Num Samples...", args, n);
	XtManageChild(menu_ent);
	XtAddCallback(menu_ent, XmNactivateCallback, SetNumSamplesCB,
		(caddr_t)t);

				/* Packet Size menu item		*/
	n = 0;
	menu_ent = XmCreatePushButton(menu_pane, "Packet Size...", args, n);
	XtManageChild(menu_ent);
	XtAddCallback(menu_ent, XmNactivateCallback, SetPacketSizeCB,
		(caddr_t)t);
				/* Number of Reads menu item		*/
	n = 0;
	menu_ent = XmCreatePushButton(menu_pane, "Num Reads...", args, n);
	XtManageChild(menu_ent);
	XtAddCallback(menu_ent, XmNactivateCallback, SetNumReadsCB,
		(caddr_t)t);
				/* Number of Writes menu item		*/
	n = 0;
	menu_ent = XmCreatePushButton(menu_pane, "Num Writes...", args, n);
	XtManageChild(menu_ent);
	XtAddCallback(menu_ent, XmNactivateCallback, SetNumWritesCB,
		(caddr_t)t);
				/* Target Servers menu item		*/
	n = 0;
	menu_ent = XmCreatePushButton(menu_pane, "Target Servers...", args, n);
	XtManageChild(menu_ent);
	XtAddCallback(menu_ent, XmNactivateCallback, SetTargetServersCB,
		(caddr_t)t);

	n = 0;
	XtSetArg(args[n], XmNsubMenuId, menu_pane);		n++;
	tt->menu_cscs[mc] = XmCreateCascadeButton(app_win->menu_bar,
		"Parameters", args, n);
	XtManageChild(tt->menu_cscs[mc]);
	mc++;


	/*** Add "Test" menu to menu bar ***/

	n = 0;
	tt->menu_panes[mp] = XmCreatePulldownMenu(app_win->menu_bar,
		"menu_pane", args, n);
				/* Start test menu item			*/
	menu_ent = XmCreatePushButton(tt->menu_panes[mp], "Start", args, n);
	XtManageChild(menu_ent);
	XtAddCallback(menu_ent, XmNactivateCallback, StartCB, (caddr_t)t);

	n = 0;
	XtSetArg(args[n], XmNsubMenuId, tt->menu_panes[mp]);		n++;
	tt->menu_cscs[mc] = XmCreateCascadeButton(app_win->menu_bar,
		"Test", args, n);
	XtManageChild(tt->menu_cscs[mc]);
	mc++;


	/*** Add "Display" menu to menu bar ***/

	n = 0;
	tt->menu_panes[mp] = XmCreatePulldownMenu(app_win->menu_bar,
		"menu_pane", args, n);
				/* Server Call Times menu item		*/
	n = 0;
	XtSetArg(args[n], XmNuserData, GRAPH_SCT);			n++;
	menu_ent = XmCreatePushButton(tt->menu_panes[mp],
		"Server Call Times", args, n);
	XtManageChild(menu_ent);
	XtAddCallback(menu_ent, XmNactivateCallback,
		DisplayGraphCB, (caddr_t)t);

				/* Norm. Transact. menu item		*/
	n = 0;
	XtSetArg(args[n], XmNuserData, GRAPH_NTT);			n++;
	menu_ent = XmCreatePushButton(tt->menu_panes[mp],
		"Norm. Tranact.", args, n);
	XtManageChild(menu_ent);
	XtAddCallback(menu_ent, XmNactivateCallback,
		DisplayGraphCB, (caddr_t)t);

				/* Avg. R/W menu item		*/
	n = 0;
	XtSetArg(args[n], XmNuserData, GRAPH_ARW);			n++;
	menu_ent = XmCreatePushButton(tt->menu_panes[mp],
		"Avg. R/W", args, n);
	XtManageChild(menu_ent);
	XtAddCallback(menu_ent, XmNactivateCallback,
		DisplayGraphCB, (caddr_t)t);

	n = 0;
	XtSetArg(args[n], XmNsubMenuId, tt->menu_panes[mp]);		n++;
	tt->menu_cscs[mc] = XmCreateCascadeButton(app_win->menu_bar,
		"Display", args, n);
	XtManageChild(tt->menu_cscs[mc]);
	mc++;


				/* End list of menus 			*/
	tt->menu_panes[mp] = NULL;
	tt->menu_cscs[mc] = NULL;


	/*** Create tool info field ***/

	msg_str = CreateToolInfo(tt);

	n = 0;
	XtSetArg(args[n], XmNbottomAttachment, XmATTACH_FORM);		n++;
	XtSetArg(args[n], XmNleftAttachment, XmATTACH_FORM);		n++;
	XtSetArg(args[n], XmNrightAttachment, XmATTACH_FORM);		n++;
	XtSetArg(args[n], XmNtopAttachment, XmATTACH_NONE);		n++;
	XtSetArg(args[n], XmNresizable, False);				n++;
	XtSetArg(args[n], XmNlabelString, msg_str);			n++;
	XtSetArg(args[n], XmNalignment, XmALIGNMENT_BEGINNING);		n++;
	tt->tinfo_lbl = XmCreateLabel(app_win->tool_form,
		"info", args, n);
	XtManageChild(tt->tinfo_lbl);
	XmStringFree(msg_str);


	/*** Create graph info field ***/

	msg_str = XmStringCreateSimple("No samples displayed.");

	n = 0;
	XtSetArg(args[n], XmNlabelString, msg_str);			n++;
	XtSetArg(args[n], XmNbottomWidget, tt->tinfo_lbl);		n++;
	XtSetArg(args[n], XmNbottomAttachment, XmATTACH_WIDGET);	n++;
	XtSetArg(args[n], XmNleftAttachment, XmATTACH_FORM);		n++;
	XtSetArg(args[n], XmNrightAttachment, XmATTACH_FORM);		n++;
	XtSetArg(args[n], XmNtopAttachment, XmATTACH_NONE);		n++;
	XtSetArg(args[n], XmNresizable, False);				n++;
	tt->ginfo_lbl = XmCreateLabel(app_win->tool_form,
		"info", args, n);
	XtManageChild(tt->ginfo_lbl);
	XmStringFree(msg_str);


	/*** Create "I'm Busy" dialog ***/

	cancel_str = XmStringCreateSimple("Stop");
	title_str = XmStringCreateSimple("Test Running");

	n = 0;
	XtSetArg(args[n], XmNcancelLabelString, cancel_str);		n++;
	XtSetArg(args[n], XmNdialogTitle, title_str);			n++;
	XtSetArg(args[n], XmNdialogStyle, XmDIALOG_FULL_APPLICATION_MODAL);
	XtSetArg(args[n], XmNdefaultButtonType, XmDIALOG_CANCEL_BUTTON); n++;
	tt->working_dlg = XmCreateWorkingDialog(app_win->main_win, "busy",
		args, n);
	XtAddCallback(tt->working_dlg, XmNcancelCallback, StopCB, (caddr_t)t);
	XmStringFree(title_str);
	XmStringFree(cancel_str);
				/* Remove OK & Help buttons		*/
	button = XmMessageBoxGetChild(tt->working_dlg, XmDIALOG_OK_BUTTON);
	XtUnmanageChild(button);
	button = XmMessageBoxGetChild(tt->working_dlg, XmDIALOG_HELP_BUTTON);
	XtUnmanageChild(button);


	/*** Create Set Num Reads/Num Writes/Packet Size dialog ***/

	tt->ev_dlg = CreateEvaluatorDialog(app_win->main_win);


	/*** Create Set Target Servers dialog ***/

	msg_str = XmStringCreateSimple(
		"Please select the desired target servers.");
	title_str = XmStringCreateSimple("Set Target Servers");
	header_str = XmStringCreateSimple("Serv-PID  Site");

	tt->li_dlg = CreateListDialog(app_win->main_win);
	SetListDialog(tt->li_dlg, app_win->shell, title_str,
		msg_str, header_str, NULL);

	XmStringFree(header_str);
	XmStringFree(title_str);
	XmStringFree(msg_str);
				/* Customize list box			*/
	n = 0;
	XtSetArg(args[n], XmNselectionPolicy, XmEXTENDED_SELECT);	n++;
	XtSetValues(tt->li_dlg->list, args, n);
				/* Set callbacks			*/
	XtAddCallback(tt->li_dlg->list, XmNextendedSelectionCallback,
		SetTargetServersChangeCB, (caddr_t)tt);
	XtAddCallback(tt->li_dlg->ok_btn, XmNactivateCallback,
		SetTargetServersConfirmCB, (caddr_t)tt);


	/*** Create graph display ***/
	n = 0;
	XtSetArg(args[n], XmNwidth, 400);				n++;
	XtSetArg(args[n], XmNheight, 200);				n++;
	XtSetArg(args[n], XmNbottomWidget, tt->ginfo_lbl);		n++;
	XtSetArg(args[n], XmNbottomAttachment, XmATTACH_WIDGET);	n++;
	XtSetArg(args[n], XmNleftAttachment, XmATTACH_FORM);		n++;
	XtSetArg(args[n], XmNrightAttachment, XmATTACH_FORM);		n++;
	XtSetArg(args[n], XmNtopAttachment, XmATTACH_FORM);		n++;
	XtSetArg(args[n], XmNresizePolicy, XmRESIZE_NONE);		n++;
	XtSetArg(args[n], XmNmarginWidth, 0);				n++;
	XtSetArg(args[n], XmNmarginHeight, 0);				n++;
	XtSetArg(args[n], XmNresizable, False);				n++;
	tt->draw_area = XmCreateDrawingArea(app_win->tool_form,
		"graph", args, n);

				/* Create simple graph			*/
	tt->graph = GraphCreate(tt->draw_area, NULL);
	tt->graph->radix_x = 1000;
	tt->graph->radix_y = 10;
	tt->graph->auto_scale = True;
	tt->graph->label_x = XmStringCreateSimple("Time (ms)");
	tt->graph->label_y = XmStringCreateSimple("Number of Samples");
	GraphSet(tt->graph);

	XtManageChild(tt->draw_area);

	/*** Create notify lists ***/

				/* Test progress update notify list	*/
	tt->test_NL = CreateNotifyList();
	AddNotify(tt, tt->test_NL, TestUpdateNP, NULL);
				/* Test complete notify list		*/
	tt->test_done_NL = CreateNotifyList();
	AddNotify(tt, tt->test_done_NL, TestTaskDoneNP, NULL);
} /* function Init */

/******************************************************************************
 *
 *	Destroy
 *
 *		This procedure is called once a tool instance is no longer
 *	needed.  It is called at most once per tool instance through
 *	DestroyTool().  All tool instances are destroyed before the the
 *	tool is exited normally.
 *		This procedure must destroy any widgets created for a tool
 *	instance that will not be automatically destroyed when the tool
 *	form is destroyed by DestroyTool(), and return any other resources
 *	allocated.  The tool instance record is automatically freed by
 *	DestroyTool().
 *
 *****************************************************************************/

static void Destroy(t)
Tool	t;
{
TransactTool	tt = (TransactTool)t;
int		m;


				/* Destroy graph			*/
	GraphDestroy(tt->graph);
	XtDestroyWidget(tt->draw_area);
				/* Destroy dialogs			*/
	XtDestroyWidget(tt->working_dlg);
	XtDestroyWidget(tt->ev_dlg->dialog);
	XtFree(tt->ev_dlg);
				/* Destroy our added menus		*/
	for (m=0; tt->menu_cscs[m] != NULL; m++)
		XtDestroyWidget(tt->menu_cscs[m]);
	for (m=0; tt->menu_panes[m] != NULL; m++)
		XtDestroyWidget(tt->menu_panes[m]);
				/* Free allocated arrays		*/
	if (tt->data) XtFree(tt->data);
				/* Destroy notify lists			*/
	DestroyNotifyList(tt->test_NL);
	DestroyNotifyList(tt->test_done_NL);
} /*  function Destroy */

/******************************************************************************
 *
 *	Private Procedures
 *
 *****************************************************************************/

static XmString CreateToolInfo(tt)
TransactTool	tt;
{
XmString	tinfo_str,
		tinfo_str1,
		tinfo_str2;
char		tinfo_buf[256],
		*t;
address		*s;


				/* Transaction type			*/
	strcpy(tinfo_buf, "Transaction Type:  ");
	strcat(tinfo_buf, TransactionList[tt->transaction_type].name);
	tinfo_str1 = XmStringCreateSimple(tinfo_buf);
	tinfo_str = XmStringConcat(tinfo_str1, SeparatorStr);
	XmStringFree(tinfo_str1);
				/* Number of samples			*/
	(void)sprintf(tinfo_buf, "Samples to be taken:  %d",
		tt->num_test);
	tinfo_str1 = XmStringCreateSimple(tinfo_buf);
	tinfo_str2 = XmStringConcat(tinfo_str, SeparatorStr);
	XmStringFree(tinfo_str);
	tinfo_str = XmStringConcat(tinfo_str2, tinfo_str1);
	XmStringFree(tinfo_str1);
	XmStringFree(tinfo_str2);
				/* Number of Reads/Writes		*/
	(void)sprintf(tinfo_buf, "Num Reads/Writes per Transaction:  %4d/%4d",
		tt->num_reads, tt->num_writes);
	tinfo_str1 = XmStringCreateSimple(tinfo_buf);
	tinfo_str2 = XmStringConcat(tinfo_str, SeparatorStr);
	XmStringFree(tinfo_str);
	tinfo_str = XmStringConcat(tinfo_str2, tinfo_str1);
	XmStringFree(tinfo_str1);
	XmStringFree(tinfo_str2);
				/* Packet size				*/
	(void)sprintf(tinfo_buf, "Packet size (bytes/bcast):  %d",
		tt->packet_size);
	tinfo_str1 = XmStringCreateSimple(tinfo_buf);
	tinfo_str2 = XmStringConcat(tinfo_str, SeparatorStr);
	XmStringFree(tinfo_str);
	tinfo_str = XmStringConcat(tinfo_str2, tinfo_str1);
	XmStringFree(tinfo_str1);
	XmStringFree(tinfo_str2);
				/* Target servers			*/
	strcpy(tinfo_buf, "Target Servers: ");
	t = tinfo_buf + strlen(tinfo_buf);
	if (!addr_isnull(tt->target_servers)) {
	    for (s=tt->target_servers; !addr_isnull(s); s++) {
		(void)sprintf(t, "%d@%d", s->addr_process, s->addr_site);
		t += strlen(t);
		if (!addr_isnull(s+1)) {
		    *t++ = ',';
		    *t++ = ' ';
		} /* if */
	    } /* for */
	} else strcat(tinfo_buf, "(None)");
	tinfo_str1 = XmStringCreateSimple(tinfo_buf);
	tinfo_str2 = XmStringConcat(tinfo_str, SeparatorStr);
	XmStringFree(tinfo_str);
	tinfo_str = XmStringConcat(tinfo_str2, tinfo_str1);
	XmStringFree(tinfo_str1);
	XmStringFree(tinfo_str2);


	return (tinfo_str);
} /* function SetToolInfo */





static void SetToolInfo(tt)
TransactTool	tt;
{
XmString	tinfo_str;
Arg		args[5];
int		n;


	tinfo_str = CreateToolInfo(tt);

	n = 0;
	XtSetArg(args[n], XmNlabelString, tinfo_str);			n++;
	XtSetValues(tt->tinfo_lbl, args, n);
	XmStringFree(tinfo_str);
} /* function SetToolInfo */

/******************************************************************************
 *
 *	Test Task functions
 *
 *****************************************************************************/

static void TestUpdateNP(t, client_data, call_data)
Tool	t;
caddr_t	client_data;
caddr_t	call_data;
{
TransactTool	tt = (TransactTool)t;
int		count = (int)call_data;
char		info_buf[64];
XmString	info_str;
Arg		args[5];
int		n;


				/* Update "busy" dialog			*/
	(void)sprintf(info_buf, "%5d samples left", tt->num_test - count);
	info_str = XmStringCreateSimple(info_buf);
	n = 0;
	XtSetArg(args[n], XmNmessageString, info_str);			n++;
	XtSetValues(tt->working_dlg, args, n);
	XmStringFree(info_str);
} /* function TestUpdateNP */





static void TestTaskDoneNP(t, client_data, call_data)
Tool	t;
caddr_t	client_data;
caddr_t	call_data;
{
TransactTool	tt = (TransactTool)t;
char		info_buf[64];
XmString	info_str;
Arg		args[5];
int		n;


				/* Set data to graph			*/
	GraphSetData(tt->graph, tt->data, tt->num_data);
				/* Update sample info			*/
	(void)sprintf(info_buf, "Time for call to TM, %d samples taken",
		tt->num_data);
	info_str = XmStringCreateSimple(info_buf);
	n = 0;
	XtSetArg(args[n], XmNlabelString, info_str);			n++;
	XtSetValues(tt->ginfo_lbl, args, n);
	XmStringFree(info_str);
				/* Popdown dialog			*/
	XtUnmanageChild(tt->working_dlg);
} /* function TestTaskDoneNP */





static void TestTask(arg)
void	*arg;
{
TransactTool	tt = (TransactTool)arg;
int		*status,
		*transtime_norm,
		*rwtime_avg;
address		*s;
int		i,
		nresp,
		elapsed;
struct timeval	t_start,
		t_end;


/*****WARNING:  DO NOT MAKE ANY X OR Xt CALLS FROM THIS FUNCTION
 * This is runs as separate task from the main X task.  All communications
 * must be done via shared variables or xperfisis' notification mechanism.
 */


				/* Allocate server return arrays	*/
	status = (int *)XtMalloc(sizeof(int) * tt->num_servers);
	transtime_norm = (int *)XtMalloc(sizeof(int) * tt->num_servers);
	rwtime_avg = (int *)XtMalloc(sizeof(int) * tt->num_servers);

	while (tt->run_test && (tt->num_test > tt->num_data)) {
				/* Get start time			*/
	    (void)gettimeofday(&t_start, NULL);

				/* Do test				*/
#ifdef	NOTDEF
	    nresp = abcast(tt->target_list,
		    TransactionList[tt->transaction_type].transaction_entry,
		    "%d,%d,%d,%d",
		    tt->dm_size, tt->num_reads, tt->num_writes,
		    tt->packet_size,
		    ALL, "%d,%d,%d", status, transtime_norm, rwtime_avg);
#else
	    nresp = abcast(tt->common.app_win->gaddr_p,
		    TransactionList[tt->transaction_type].transaction_entry,
		    "%d,%d,%d,%d",
		    tt->dm_size, tt->num_reads, tt->num_writes,
		    tt->packet_size,
		    ALL, "%d,%d,%d", status, transtime_norm, rwtime_avg);
#endif	NOTDEF
				/* Get end time			*/
	    (void)gettimeofday(&t_end, NULL);
				/* Check status				*/
	    if (nresp != tt->num_servers) {
				/* This *really* should be a popup	*/
		isis_perror("ERROR!  Not all servers responded!\n");
		goto ERROR;
	    } /* if */
	    for (i=tt->num_servers-1; i >= 0; i--) {
		if (status[i] != RET_OK) {
				/* This *really* should be a popup	*/
		    print("ERROR!  Response %d was error %d!\n", i, status[i]);
		    goto ERROR;
		} /* if */
	    } /* for */
				/* Calculate elapsed time		*/
	    elapsed = (t_end.tv_sec * 1000000 + t_end.tv_usec)
		    - (t_start.tv_sec * 1000000 + t_start.tv_usec);
				/* Store data items			*/
				/*   (simple enumeration format)	*/
	    tt->data[tt->num_data].x = elapsed;
	    tt->data[tt->num_data].cnt = tt->graph->radix_y;
	    tt->num_data++;
	    for (i=tt->num_servers - 1; i >= 0; i--) {
		tt->data_transtime[tt->num_transtime].x = transtime_norm[i];
		tt->data_transtime[tt->num_transtime].cnt = tt->graph->radix_y;
		tt->data_rwtime[tt->num_rwtime].x = rwtime_avg[i];
		tt->data_rwtime[tt->num_rwtime].cnt = tt->graph->radix_y;

		tt->num_transtime++;
		tt->num_rwtime++;
	    } /* for */

				/* Periodically update "busy" dialog	*/
	    if ((tt->num_data & 0x1f) == 0)
		    TriggerNotify(tt->test_NL, tt->num_data);

	} /* while */

ERROR:
				/* We're done, cleanup			*/
	XtFree(status);
	XtFree(rwtime_avg);
	XtFree(transtime_norm);
				/* Invoke finish-up routine		*/
	TriggerNotify(tt->test_done_NL, NULL);
} /* function TestTask */

/******************************************************************************
 *
 *	Callback Procedures
 *
 *****************************************************************************/
/******************************************************************************
 *
 *	Start/Stop Test Callback Procedures
 *
 *****************************************************************************/

static void StartCB(w, client_data, call_data)
Widget	w;		/* UNUSED */
caddr_t	client_data;	/* Tool */
caddr_t	call_data;	/* UNUSED */
{
TransactTool	tt = (TransactTool)client_data;
AppWindow	*app_win = tt->common.app_win;
char		msg_buf[64];
XmString	msg_str;

Arg		args[5];
int		n;


	if (tt->num_servers == 0) {
	    SetMessageDialog(app_win->warn_dlg, app_win->shell,
		    NoServersTitleStr, NoServersStr, OKStr, EmptyStr);
	    XtManageChild(app_win->warn_dlg);

	    return;
	} /* if */

				/* Reset data arrays			*/
	tt->num_data = 0;
	tt->num_transtime = 0;
	tt->num_rwtime = 0;
				/* Reallocate data arrays, if needed	*/
	if (tt->data_sz < tt->num_test) {
	    tt->data_sz = tt->num_test;
	    tt->data = (GDatum *)XtRealloc(tt->data,
		    sizeof(GDatum) * tt->data_sz);
	} /* if */
	if (tt->data_transtime_sz < tt->num_test * tt->num_servers) {
	    tt->data_transtime_sz = tt->num_test * tt->num_servers;
	    tt->data_transtime = (GDatum *)XtRealloc(tt->data_transtime,
		    sizeof(GDatum) * tt->data_transtime_sz);
	} /* if */
	if (tt->data_rwtime_sz < tt->num_test * tt->num_servers) {
	    tt->data_rwtime_sz = tt->num_test * tt->num_servers;
	    tt->data_rwtime = (GDatum *)XtRealloc(tt->data_rwtime,
		    sizeof(GDatum) * tt->data_rwtime_sz);
	} /* if */

				/* Put up "busy" message		*/
	(void)sprintf(msg_buf, "%5d samples left", tt->num_test);
	msg_str = XmStringCreateSimple(msg_buf);
	SetMessageDialog(tt->working_dlg, app_win->shell, BusyTitleStr,
		msg_str, EmptyStr, StopStr);
	XtManageChild(tt->working_dlg);
	XmStringFree(msg_str);

				/* Fork test task			*/
	tt->run_test = True;
	t_fork(TestTask, tt);
} /* function StartCB */






static void StopCB(w, client_data, call_data)
Widget	w;		/* UNUSED */
caddr_t	client_data;	/* UNUSED */
caddr_t	call_data;	/* UNUSED */
{
TransactTool	tt = (TransactTool)client_data;


				/* Tell test task to stop		*/
	tt->run_test = False;
} /* function StopCB */

/******************************************************************************
 *
 *	Set Transaction Type Callback Procedure
 *
 *****************************************************************************/

static void SetTransactionTypeCB(w, client_data, call_data)
Widget	w;		/* UNUSED */
caddr_t	client_data;	/* UNUSED */
caddr_t	call_data;	/* UNUSED */
{
TransactTool	tt = (TransactTool)client_data;
int		transaction_type;
Arg		args[5];
int		n;


					/* Get transaction type		*/
	n = 0;
	XtSetArg(args[n], XmNuserData, &transaction_type);		n++;
	XtGetValues(w, args, n);

					/* Set transaction type		*/
	tt->transaction_type = transaction_type;
					/* Set tool info		*/
	SetToolInfo(tt);
} /* function SetTransactionTypeCB */

/******************************************************************************
 *
 *	Set Num Samples Callback Procedures
 *
 *****************************************************************************/
static void	SetNumSamplesChangeCB(),
		SetNumSamplesConfirmCB(),
		SetNumSamplesCancelCB();



static void SetNumSamplesCB(w, client_data, call_data)
Widget	w;		/* UNUSED */
caddr_t	client_data;	/* TransactTool */
caddr_t	call_data;	/* UNUSED */
{
TransactTool	tt = (TransactTool)client_data;

static Arg	args[] = {
	{XmNminimum,	(XtArgVal)1},
	{XmNmaximum,	(XtArgVal)(10000 + 100)}
};


				/* Set evalutor dialog			*/
	XtAddCallback(tt->ev_dlg->scroll_bar, XmNdragCallback,
		SetNumSamplesChangeCB, (caddr_t)tt);
	XtAddCallback(tt->ev_dlg->scroll_bar, XmNvalueChangedCallback,
		SetNumSamplesChangeCB, (caddr_t)tt);
	XtAddCallback(tt->ev_dlg->ok_btn, XmNactivateCallback,
		SetNumSamplesConfirmCB, (caddr_t)tt);
	XtAddCallback(tt->ev_dlg->cancel_btn, XmNactivateCallback,
		SetNumSamplesCancelCB, (caddr_t)tt);

				/* Set number of samples slider		*/
	XmScrollBarSetValues(tt->ev_dlg->scroll_bar, 1,
		1, 0, 0, True);
	XtSetValues(tt->ev_dlg->scroll_bar, args, XtNumber(args));
	XmScrollBarSetValues(tt->ev_dlg->scroll_bar, tt->num_test,
		100, 1, 100, True);

	SetEvaluatorDialog(tt->ev_dlg, tt->common.app_win->shell,
		NumSamplesTitleStr, NumSamplesStr, NULL);

				/* Popup number of samples popup	*/
	XtManageChild(tt->ev_dlg->dialog);
} /* function SetNumSamplesCB */





static void SetNumSamplesChangeCB(w, client_data, call_data)
Widget	w;		/* UNUSED */
caddr_t	client_data;	/* TransactTool */
caddr_t	call_data;	/* UNUSED */
{
TransactTool	tt = (TransactTool)client_data;
int		num_samples,
		slider_size,
		increment,
		page_increment;
char		msg_buf[16];
XmString	msg_str;
Arg		arg;


				/* Get packet size slider value		*/
	XmScrollBarGetValues(w, &num_samples, &slider_size,
		&increment, &page_increment);
				/* Display number of samples value	*/
	(void)sprintf(msg_buf, "%5d", num_samples);
	msg_str = XmStringCreateSimple(msg_buf);
	XtSetArg(arg, XmNlabelString, msg_str);
	XtSetValues(tt->ev_dlg->value_lbl, &arg, 1);
	XmStringFree(msg_str);
} /* function SetNumSamplesChangeCB */





static void SetNumSamplesCancelCB(w, client_data, call_data)
Widget	w;		/* UNUSED */
caddr_t	client_data;	/* TransactTool */
caddr_t	call_data;	/* UNUSED */
{
TransactTool	tt = (TransactTool)client_data;


				/* Remove callbacks			*/
	XtRemoveCallback(tt->ev_dlg->scroll_bar, XmNdragCallback,
		SetNumSamplesChangeCB, (caddr_t)tt);
	XtRemoveCallback(tt->ev_dlg->scroll_bar, XmNvalueChangedCallback,
		SetNumSamplesChangeCB, (caddr_t)tt);
	XtRemoveCallback(tt->ev_dlg->ok_btn, XmNactivateCallback,
		SetNumSamplesConfirmCB, (caddr_t)tt);
	XtRemoveCallback(tt->ev_dlg->cancel_btn, XmNactivateCallback,
		SetNumSamplesCancelCB, (caddr_t)tt);
} /* function SetNumSamplesCancelCB */





static void SetNumSamplesConfirmCB(w, client_data, call_data)
Widget	w;		/* UNUSED */
caddr_t	client_data;	/* TransactTool */
caddr_t	call_data;	/* UNUSED */
{
TransactTool	tt = (TransactTool)client_data;
int		packet_size,
		slider_size,
		increment,
		page_increment;


				/* Get packet size slider value		*/
	XmScrollBarGetValues(tt->ev_dlg->scroll_bar, &tt->num_test,
		&slider_size, &increment, &page_increment);
				/* Update tool info display		*/
	SetToolInfo(tt);
				/* Remove callbacks			*/
	SetNumSamplesCancelCB(w, client_data, call_data);
} /* function SetNumSamplesConfirmCB */

/******************************************************************************
 *
 *	Set Packet Size Callback Procedures
 *
 *****************************************************************************/
static void	SetPacketSizeChangeCB(),
		SetPacketSizeConfirmCB(),
		SetPacketSizeCancelCB();



static void SetPacketSizeCB(w, client_data, call_data)
Widget	w;		/* UNUSED */
caddr_t	client_data;	/* TransactTool */
caddr_t	call_data;	/* UNUSED */
{
TransactTool	tt = (TransactTool)client_data;

static Arg	args[] = {
	{XmNminimum,	(XtArgVal)1},
	{XmNmaximum,	(XtArgVal)(4096 + 512)}
};


				/* Set evaluator dialog			*/
	XtAddCallback(tt->ev_dlg->scroll_bar, XmNdragCallback,
		SetPacketSizeChangeCB, (caddr_t)tt);
	XtAddCallback(tt->ev_dlg->scroll_bar, XmNvalueChangedCallback,
		SetPacketSizeChangeCB, (caddr_t)tt);
	XtAddCallback(tt->ev_dlg->ok_btn, XmNactivateCallback,
		SetPacketSizeConfirmCB, (caddr_t)tt);
	XtAddCallback(tt->ev_dlg->cancel_btn, XmNactivateCallback,
		SetPacketSizeCancelCB, (caddr_t)tt);

				/* Set packet size slider 		*/
	XmScrollBarSetValues(tt->ev_dlg->scroll_bar, 1,
		1, 0, 0, True);
	XtSetValues(tt->ev_dlg->scroll_bar, args, XtNumber(args));
	XmScrollBarSetValues(tt->ev_dlg->scroll_bar, tt->packet_size,
		512, 1, 512, True);

	SetEvaluatorDialog(tt->ev_dlg, tt->common.app_win->shell,
		PacketSizeTitleStr, PacketSizeStr, NULL);

				/* Popup packet size popup		*/
	XtManageChild(tt->ev_dlg->dialog);
} /* function SetPacketSizeCB */





static void SetPacketSizeChangeCB(w, client_data, call_data)
Widget	w;		/* UNUSED */
caddr_t	client_data;	/* TransactTool */
caddr_t	call_data;	/* UNUSED */
{
TransactTool	tt = (TransactTool)client_data;
int		packet_size,
		slider_size,
		increment,
		page_increment;
char		msg_buf[16];
XmString	msg_str;
Arg		arg;


				/* Get packet size slider value		*/
	XmScrollBarGetValues(w, &packet_size, &slider_size,
		&increment, &page_increment);
				/* Display new packet size value	*/
	(void)sprintf(msg_buf, "%4d", packet_size);
	msg_str = XmStringCreateSimple(msg_buf);
	XtSetArg(arg, XmNlabelString, msg_str);
	XtSetValues(tt->ev_dlg->value_lbl, &arg, 1);
	XmStringFree(msg_str);
} /* function SetPacketSizeChangeCB */





static void SetPacketSizeCancelCB(w, client_data, call_data)
Widget	w;		/* UNUSED */
caddr_t	client_data;	/* TransactTool */
caddr_t	call_data;	/* UNUSED */
{
TransactTool	tt = (TransactTool)client_data;


				/* Remove callbacks			*/
	XtRemoveCallback(tt->ev_dlg->scroll_bar, XmNdragCallback,
		SetPacketSizeChangeCB, (caddr_t)tt);
	XtRemoveCallback(tt->ev_dlg->scroll_bar, XmNvalueChangedCallback,
		SetPacketSizeChangeCB, (caddr_t)tt);
	XtRemoveCallback(tt->ev_dlg->ok_btn, XmNactivateCallback,
		SetPacketSizeConfirmCB, (caddr_t)tt);
	XtRemoveCallback(tt->ev_dlg->cancel_btn, XmNactivateCallback,
		SetPacketSizeCancelCB, (caddr_t)tt);
} /* function SetPacketSizeCancelCB */





static void SetPacketSizeConfirmCB(w, client_data, call_data)
Widget	w;		/* UNUSED */
caddr_t	client_data;	/* TransactTool */
caddr_t	call_data;	/* UNUSED */
{
TransactTool	tt = (TransactTool)client_data;
int		packet_size,
		slider_size,
		increment,
		page_increment;


				/* Get packet size slider value		*/
	XmScrollBarGetValues(tt->ev_dlg->scroll_bar, &tt->packet_size,
		&slider_size, &increment, &page_increment);
				/* Update tool info display		*/
	SetToolInfo(tt);
				/* Remove callbacks			*/
	SetPacketSizeCancelCB(w, client_data, call_data);
} /* function SetPacketSizeConfirmCB */

/******************************************************************************
 *
 *	Set Number of Reads Callback Procedures
 *
 *****************************************************************************/
static void	NumReadsChangeCB(),
		NumReadsConfirmCB(),
		NumReadsCancelCB();





static void SetNumReadsCB(w, client_data, call_data)
Widget	w;		/* UNUSED */
caddr_t	client_data;	/* TransactTool */
caddr_t	call_data;	/* UNUSED */
{
TransactTool	tt = (TransactTool)client_data;

static Arg	args[] = {
	{XmNminimum,	(XtArgVal)1},
	{XmNmaximum,	(XtArgVal)(100 + 10)}
};


				/* Set evalutor dialog			*/
	XtAddCallback(tt->ev_dlg->scroll_bar, XmNdragCallback,
		NumReadsChangeCB, (caddr_t)tt);
	XtAddCallback(tt->ev_dlg->scroll_bar, XmNvalueChangedCallback,
		NumReadsChangeCB, (caddr_t)tt);
	XtAddCallback(tt->ev_dlg->ok_btn, XmNactivateCallback,
		NumReadsConfirmCB, (caddr_t)tt);
	XtAddCallback(tt->ev_dlg->cancel_btn, XmNactivateCallback,
		NumReadsCancelCB, (caddr_t)tt);

				/* Set number of reads slider 		*/
	XmScrollBarSetValues(tt->ev_dlg->scroll_bar, 1,
		1, 0, 0, True);
	XtSetValues(tt->ev_dlg->scroll_bar, args, XtNumber(args));
	XmScrollBarSetValues(tt->ev_dlg->scroll_bar, tt->num_reads,
		10, 1, 10, True);

	SetEvaluatorDialog(tt->ev_dlg, tt->common.app_win->shell,
		NumReadsTitleStr, NumReadsStr, NULL);

				/* Popup number of reads popup		*/
	XtManageChild(tt->ev_dlg->dialog);
} /* function SetNumReadsCB */





static void NumReadsChangeCB(w, client_data, call_data)
Widget	w;		/* UNUSED */
caddr_t	client_data;	/* TransactTool */
caddr_t	call_data;	/* UNUSED */
{
TransactTool	tt = (TransactTool)client_data;
int		num_reads,
		slider_size,
		increment,
		page_increment;
char		msg_buf[16];
XmString	msg_str;
Arg		arg;


				/* Get num_reads slider value		*/
	XmScrollBarGetValues(w, &num_reads, &slider_size,
		&increment, &page_increment);
				/* Display new num_reads value	*/
	(void)sprintf(msg_buf, "%2d", num_reads);
	msg_str = XmStringCreateSimple(msg_buf);
	XtSetArg(arg, XmNlabelString, msg_str);
	XtSetValues(tt->ev_dlg->value_lbl, &arg, 1);
	XmStringFree(msg_str);
} /* function NumReadsChangeCB */





static void NumReadsCancelCB(w, client_data, call_data)
Widget	w;		/* UNUSED */
caddr_t	client_data;	/* TransactTool */
caddr_t	call_data;	/* UNUSED */
{
TransactTool	tt = (TransactTool)client_data;


				/* Remove callbacks			*/
	XtRemoveCallback(tt->ev_dlg->scroll_bar, XmNdragCallback,
		NumReadsChangeCB, (caddr_t)tt);
	XtRemoveCallback(tt->ev_dlg->scroll_bar, XmNvalueChangedCallback,
		NumReadsChangeCB, (caddr_t)tt);
	XtRemoveCallback(tt->ev_dlg->ok_btn, XmNactivateCallback,
		NumReadsConfirmCB, (caddr_t)tt);
	XtRemoveCallback(tt->ev_dlg->cancel_btn, XmNactivateCallback,
		NumReadsCancelCB, (caddr_t)tt);
} /* function NumReadsCancelCB */





static void NumReadsConfirmCB(w, client_data, call_data)
Widget	w;		/* UNUSED */
caddr_t	client_data;	/* TransactTool */
caddr_t	call_data;	/* UNUSED */
{
TransactTool	tt = (TransactTool)client_data;
int		num_reads,
		slider_size,
		increment,
		page_increment;


				/* Get num_reads slider value		*/
	XmScrollBarGetValues(tt->ev_dlg->scroll_bar, &tt->num_reads,
		&slider_size, &increment, &page_increment);
				/* Update tool info display		*/
	SetToolInfo(tt);
				/* Remove callbacks			*/
	NumReadsCancelCB(w, client_data, call_data);
} /* function NumReadsConfirmCB */

/******************************************************************************
 *
 *	Set Number of Writes Callback Procedures
 *
 *****************************************************************************/
static void	NumWritesChangeCB(),
		NumWritesConfirmCB(),
		NumWritesCancelCB();





static void SetNumWritesCB(w, client_data, call_data)
Widget	w;		/* UNUSED */
caddr_t	client_data;	/* TransactTool */
caddr_t	call_data;	/* UNUSED */
{
TransactTool	tt = (TransactTool)client_data;

static Arg	args[] = {
	{XmNminimum,	(XtArgVal)1},
	{XmNmaximum,	(XtArgVal)(100 + 10)}
};


				/* Set evalutor dialog			*/
	XtAddCallback(tt->ev_dlg->scroll_bar, XmNdragCallback,
		NumWritesChangeCB, (caddr_t)tt);
	XtAddCallback(tt->ev_dlg->scroll_bar, XmNvalueChangedCallback,
		NumWritesChangeCB, (caddr_t)tt);
	XtAddCallback(tt->ev_dlg->ok_btn, XmNactivateCallback,
		NumWritesConfirmCB, (caddr_t)tt);
	XtAddCallback(tt->ev_dlg->cancel_btn, XmNactivateCallback,
		NumWritesCancelCB, (caddr_t)tt);

				/* Set number of writes slider 		*/
	XmScrollBarSetValues(tt->ev_dlg->scroll_bar, 1,
		1, 0, 0, True);
	XtSetValues(tt->ev_dlg->scroll_bar, args, XtNumber(args));
	XmScrollBarSetValues(tt->ev_dlg->scroll_bar, tt->num_writes,
		10, 1, 10, True);

	SetEvaluatorDialog(tt->ev_dlg, tt->common.app_win->shell,
		NumWritesTitleStr, NumWritesStr, NULL);

				/* Popup number of writes popup		*/
	XtManageChild(tt->ev_dlg->dialog);
} /* function SetNumWritesCB */





static void NumWritesChangeCB(w, client_data, call_data)
Widget	w;		/* UNUSED */
caddr_t	client_data;	/* TransactTool */
caddr_t	call_data;	/* UNUSED */
{
TransactTool	tt = (TransactTool)client_data;
int		num_writes,
		slider_size,
		increment,
		page_increment;
char		msg_buf[16];
XmString	msg_str;
Arg		arg;


				/* Get num_writes slider value		*/
	XmScrollBarGetValues(w, &num_writes, &slider_size,
		&increment, &page_increment);
				/* Display new num_writes value		*/
	(void)sprintf(msg_buf, "%2d", num_writes);
	msg_str = XmStringCreateSimple(msg_buf);
	XtSetArg(arg, XmNlabelString, msg_str);
	XtSetValues(tt->ev_dlg->value_lbl, &arg, 1);
	XmStringFree(msg_str);
} /* function NumWritesChangeCB */





static void NumWritesCancelCB(w, client_data, call_data)
Widget	w;		/* UNUSED */
caddr_t	client_data;	/* TransactTool */
caddr_t	call_data;	/* UNUSED */
{
TransactTool	tt = (TransactTool)client_data;


				/* Remove callbacks			*/
	XtRemoveCallback(tt->ev_dlg->scroll_bar, XmNdragCallback,
		NumWritesChangeCB, (caddr_t)tt);
	XtRemoveCallback(tt->ev_dlg->scroll_bar, XmNvalueChangedCallback,
		NumWritesChangeCB, (caddr_t)tt);
	XtRemoveCallback(tt->ev_dlg->ok_btn, XmNactivateCallback,
		NumWritesConfirmCB, (caddr_t)tt);
	XtRemoveCallback(tt->ev_dlg->cancel_btn, XmNactivateCallback,
		NumWritesCancelCB, (caddr_t)tt);
} /* function NumWritesCancelCB */





static void NumWritesConfirmCB(w, client_data, call_data)
Widget	w;		/* UNUSED */
caddr_t	client_data;	/* TransactTool */
caddr_t	call_data;	/* UNUSED */
{
TransactTool	tt = (TransactTool)client_data;
int		num_writes,
		slider_size,
		increment,
		page_increment;


				/* Get num_writes slider value		*/
	XmScrollBarGetValues(tt->ev_dlg->scroll_bar, &tt->num_writes,
		&slider_size, &increment, &page_increment);
				/* Update tool info display		*/
	SetToolInfo(tt);
				/* Remove callbacks			*/
	NumWritesCancelCB(w, client_data, call_data);
} /* function NumWritesConfirmCB */

/******************************************************************************
 *
 *	Set Target Servers Callback Procedures
 *
 *****************************************************************************/

static void SetTargetServersCB(w, client_data, call_data)
Widget	w;		/* UNUSED */
caddr_t	client_data;	/* TransactTool */
caddr_t	call_data;	/* UNUSED */
{
TransactTool	tt = (TransactTool)client_data;
AppWindow	*app_win = tt->common.app_win;
int		i;
groupview	*gv;
char		str_buf[128];
XmString	value_str,
		site_array[PG_MAXMEMB],
		site_sel_array[PG_MAXMEMB];
int		num_sel = 0;
address		*s;
Arg		args[5];
int		n;


				/* Get current group view		*/
	gv = pg_getview(app_win->gaddr_p);
				/* Create arrays of sites		*/
	for (i=0; i < gv->gv_nmemb; i++) {
	    (void)sprintf(str_buf, "%8d  (%3d)%s",
		    gv->gv_members[i].addr_process,
		    gv->gv_members[i].addr_site,
		    site_names[gv->gv_members[i].addr_site]);
	    site_array[i] = XmStringCreateSimple(str_buf);
	    tt->site_list[i] = &gv->gv_members[i];

				/* Currently selected?			*/
	    for (s=tt->target_servers; !addr_isnull(s); s++) {
		if (addr_isequal(s, &gv->gv_members[i])) {
		       site_sel_array[num_sel++] = site_array[i];
		       break;
		} /* if */
	    } /* for */

	} /* for */

				/* Set dialog's list			*/
	n = 0;
	XtSetArg(args[n], XmNitems, site_array);			n++;
	XtSetArg(args[n], XmNitemCount, gv->gv_nmemb);			n++;
	XtSetArg(args[n], XmNselectedItemCount, num_sel);		n++;
				/* This is a bug in Motif v1.1 (well,	*/
	if (num_sel > 0) {	/*   *I* think it's a bug...)		*/
	    XtSetArg(args[n], XmNselectedItems, site_sel_array);	n++;
	} else {
	    XtSetArg(args[n], XmNselectedItems, NULL);			n++;
	} /* else */
	XtSetValues(tt->li_dlg->list, args, n);
				/* Free site strings (list makes copy)	*/
	for (i=0; i < gv->gv_nmemb; i++) XmStringFree(site_array[i]);

				/* Set dialog's value string		*/
	(void)sprintf(str_buf, "%d server%s selected.", num_sel,
		(num_sel == 1 ? " " : "s "));
	value_str = XmStringCreateSimple(str_buf);
	SetListDialog(tt->li_dlg, app_win->shell,
		NULL, NULL, NULL, value_str);
	XmStringFree(value_str);

				/* Popup dialog				*/
	XtManageChild(tt->li_dlg->dialog);
} /* function SetTargetServersCB */





static void SetTargetServersChangeCB(w, client_data, call_data)
Widget	w;		/* UNUSED */
caddr_t	client_data;	/* TransactTool */
caddr_t	call_data;	/* XmListCallbackStruct */
{
TransactTool	tt = (TransactTool)client_data;
AppWindow	*app_win = tt->common.app_win;
XmListCallbackStruct	*cb_info = (XmListCallbackStruct *)call_data;
char		value_buf[64];
XmString	value_str;

	
	(void)sprintf(value_buf, "%d server%sselected.",
		cb_info->selected_item_count,
		(cb_info->selected_item_count == 1 ? " " : "s "));
	value_str = XmStringCreateSimple(value_buf);
	SetListDialog(tt->li_dlg, NULL, NULL, NULL, NULL, value_str);
	XmStringFree(value_str);
} /* function SetTargetServersChangeCB */





static void SetTargetServersConfirmCB(w, client_data, call_data)
Widget	w;		/* UNUSED */
caddr_t	client_data;	/* TransactTool */
caddr_t	call_data;	/* UNUSED */
{
TransactTool	tt = (TransactTool)client_data;
AppWindow	*app_win = tt->common.app_win;
int		*sel_pos,
		sel_cnt,
		i=0,
		*s;


				/* Get list of selected servers		*/
	if (XmListGetSelectedPos(tt->li_dlg->list, &sel_pos, &sel_cnt)) {
	    tt->num_servers = sel_cnt;
				/* Keep only selected servers		*/
	    for (s=sel_pos; i < sel_cnt; i++,s++) {
		(void)memcpy(&tt->target_servers[i],  tt->site_list[*s - 1],
			sizeof(address));
	    } /* for */

	    XtFree(sel_pos);

	} else tt->num_servers = 0;

				/* Terminate list of targets		*/
	(void)memcpy(&tt->target_servers[i], &NULLADDRESS, sizeof(address));

				/* Create process list for BYPASS	*/
	if (tt->target_list != NULL) pl_delete(tt->target_list);
	tt->target_list = pl_create(app_win->gaddr_p, tt->target_servers);

				/* Update info display			*/
	SetToolInfo(tt);
} /* function SetTargetServersConfirmCB */

/******************************************************************************
 *
 *	Set Display Graph Callback Procedure
 *
 *****************************************************************************/

/*** Set graph to display specified data ***/

static void DisplayGraphCB(w, client_data, call_data)
Widget	w;		/* UNUSED */
caddr_t	client_data;	/* TransactTool */
caddr_t	call_data;	/* UNUSED */
{
TransactTool	tt = (TransactTool)client_data;
AppWindow	*app_win = tt->common.app_win;
char		info_buf[128];
XmString	info_str;
int		graph_type;
Arg		args[5];
int		n;


				/* No data to display?			*/
	if (tt->data == NULL) return;

				/* Get graph info			*/
	n = 0;
	XtSetArg(args[n], XmNuserData, &graph_type);			n++;
	XtGetValues(w, args, n);
	switch (graph_type) {
	  case GRAPH_SCT:
	    GraphSetData(tt->graph, tt->data, tt->num_data);
	    (void)sprintf(info_buf, "Time for call to TM, %d samples taken",
		    tt->num_data);
	    break;
	  case GRAPH_NTT:
	    GraphSetData(tt->graph, tt->data_transtime, tt->num_transtime);
	    (void)sprintf(info_buf, "Transaction Time Normalized by \
# of Links, %d samples taken",
		    tt->num_transtime);
	    break;
	  case GRAPH_ARW:
	    GraphSetData(tt->graph, tt->data_rwtime, tt->num_rwtime);
	    (void)sprintf(info_buf, "Average Read/Write Time, \
%d samples taken",
		    tt->num_rwtime);
	    break;
	  default:
	    XtError("DisplayGraph:  Internal error--unknown graph type");
	    exit(1);
	} /* switch */

				/* Set info label			*/
	info_str = XmStringCreateSimple(info_buf);
	n = 0;
	XtSetArg(args[n], XmNlabelString, info_str);			n++;
	XtSetValues(tt->ginfo_lbl, args, n);
	XmStringFree(info_str);
} /* function DisplayGraphCB */

/*************************** End of xpi_TLtransact.c *************************/
