/******************************************************************************
 * Program xperfisis - ISIS performance tool for the X Window System
 *
 * Written:	 05/14/91 by John H. Lee
 * Last Changed: 06/21/91 by John H. Lee
 * Version:	 0.3
 ******************************************************************************
 * Description:
 *
 *	This program provides a merged environment of ISIS and the X Window
 * System for tools that perform performance testing.  Emphasis has been
 * placed on making it easy to add tools to the program while provided a
 * somewhat common framework for all tools.
 ******************************************************************************
 * Exports:
 *	Display *AppDisplay		Application X server display connection
 *	char *AppName			Application name
 *	char *AppClass			Application resource class
 *	XtAppContext AppContext		Application context
 *	address *GAddr_P		performance server ISIS group address
 *	AppWindow **AppWindows		Array of application windows
 *	int AppWindowsSize		Allocated size of AppWindows array
 *	int AppWindowsOpen		Total number of windows open
 *
 * Imports:
 *	Widget CreateApplication()	Create initial application window
 * 
 ******************************************************************************
 * Compilation Flags:
 *	None
 ******************************************************************************
 * Modules:
 *	xperfisis.c		Main function
 *	xpi_graph.c		Graph psuedo-widget
 *	xpi_notify.c		Notification mechanism routines
 *	xpi_shells.c		Application shell routines
 *	xpi_tools.c		Tool management routines
 *	xpi_tooldefs.c		Tool declarations
 *	xpi_utils.c		Miscellaneous utility & convenience routines
 *	xpi_TLdefault.c		Initial default tool
 *	xpi_TLxxxxx.c		Tool modules
 *
 * Compilation and Linking:
 *	This program was developed for ISIS v3.0, the X Window System V11R4,
 * and Motif v1.1.
 *	The ISIS include files must be in the local search path for the
 * compiler, and the X11 and Motif files in the system search path.  This
 * program must be linked with the ISIS, X11, and Motif libraries.
 ******************************************************************************
 * Revisions:
 *   0.1    05/14/91 JHL  Initial Creation
 *   0.2    06/12/91 JHL  Changed invocation of isis_accept_events() to timeout
 *   0.3    06/21/91 JHL  Improved ISIS performance by using XtAppAddInput()
 *			 and ISIS internal event queue
 *****************************************************************************/


#include "isis.h"

#include <X11/Intrinsic.h>

#include "xperfisis.h"
#include "xpi_shells.h"
#include "xpi_notify.h"





/*** Public global variables ***/

Display	*AppDisplay;		/* Application display connection	*/
char	*AppName;		/* Application name			*/
char	*AppClass = "XPerfISIS";/* Application Class			*/
XtAppContext AppContext;	/* Application context			*/
address	*GAddr_P;		/* perfserver ISIS group address	*/

AppWindow	**AppWindows;	/* Application windows			*/
int		AppWindowsSize = 10,	/* Size of array		*/
		AppWindowsOpen = 0;	/* Number of open windows	*/


/*** Private global variables ***/

				/* ISIS Timeout handler ID		*/
static XtIntervalId	ISISTimeOutID = NULL;


/*** Private functions ***/

static void	AddNextISISTimeOut(),
		ProcessISISInput();





main(argc, argv)
int	argc;
char	**argv;
{
Widget		toplevel;


	/*** Initialization ***/

	AppName = argv[0];	/* Application instance name		*/
				/* Application windows array		*/
	AppWindows = (AppWindow **)
		XtCalloc(AppWindowsSize, sizeof(AppWindow *));


				/* Initialize X Toolkit			*/
	XtToolkitInitialize();
				/* Open connection to X server		*/
	AppContext = XtCreateApplicationContext();
	AppDisplay = XtOpenDisplay(AppContext, NULL, AppName, AppClass,
		NULL, 0, &argc, argv);
	if (!AppDisplay) {
	    XtError("xperfisis: Can't open display");
	    exit(1);
	} /* if */


				/* Initialize ISIS			*/
	isis_init(PERFPORTNO);

	if (addr_isnull(GAddr_P = pg_lookup(PERFSERVNAME))) {
	    XtError("xperfisis: \
unable to locate performance server group");
	    exit(1);
	} /* if */

	if (pg_client(GAddr_P, "") == -1) {
	    isis_perror("Error: xperfisis: \
unable to become client of performance server group");
	    exit(1);
        } /* if */

				/* Register handlers to be called	*/
				/*   for ISIS				*/
	AddNextISISTimeOut(&AppContext);
	XtAppAddInput(AppContext, isis_socket, XtInputReadMask,
		ProcessISISInput, &AppContext);
	XtAppAddInput(AppContext, intercl_socket, XtInputReadMask,
		ProcessISISInput, &AppContext);


	/*** Create initial application window ***/

	AppWindows[0] = XtNew(AppWindow);
	AppWindows[0]->win_index = 0;
	AppWindows[0]->app_context = AppContext;
	AppWindows[0]->gaddr_p = GAddr_P;
	toplevel = CreateApplication(AppDisplay, AppName, AppClass,
		AppWindows[0]);
	XtRealizeWidget(toplevel);

	isis_start_done();	/* Done with isis initialization	*/


	/*** Main Loop ***/

	XtAppMainLoop(AppContext);

} /* main */





/*** Invoked ISIS to handle the next ISIS input */

static void ProcessISISInput(client_data, source, input_id)
XtPointer	client_data;	/* Application context */
int		*source;	/* UNUSED */
XtInputId	*input_id;	/* UNUSED */
{
XtAppContext	*appcontext = (XtAppContext *)client_data;


				/* Give ISIS some time now		*/
	isis_accept_events(ISIS_ASYNC);
				/* Schedule next timeout		*/
	AddNextISISTimeOut(appcontext);
				/* Any triggered notify lists?  Handle now */
	HandleNotify();
} /* function ProcessISISInput */





/*** Invoked ISIS to handle the next internal ISIS event */

static void ProcessISISTimeOut(client_data, timer_id)
XtPointer	client_data;
XtIntervalId	*timer_id;	/* UNUSED */
{
XtAppContext	*appcontext = (XtAppContext *)client_data;


				/* Timeout expired			*/
	ISISTimeOutID = NULL;
				/* Give ISIS some time now		*/
	isis_accept_events(ISIS_ASYNC);
				/* Schedule next timeout		*/
	AddNextISISTimeOut(appcontext);
				/* Any triggered notify lists?  Handle now */
	HandleNotify();
} /* function ProcessISISTimeOut */





/*** Check ISIS internal event queue to see when next ISIS internal event
 * is to occur, and schedule a timeout for it, if any.  Kids, don't do this
 * at home!
 */
static void AddNextISISTimeOut(appcontext)
XtAppContext	*appcontext;
{
qnode		*qevent;

extern qnode	*isis_time_queue;
unsigned long	timeout;


				/* Get next internal ISIS event		*/
	qevent = qu_head(isis_time_queue);
				/* Is there a new event?  Schedule it.	*/
	if (qevent != NULL) {
				/* Remove outdated timeout first	*/
	    if (ISISTimeOutID != NULL) XtRemoveTimeOut(ISISTimeOutID);
				/* Determine timeout (+slop factor)	*/
	    timeout = (unsigned long)(qevent->qu_time - ISIS_TIME + 100);
	    if (timeout > 1000) timeout = 1000;

	} else timeout = 1000;

	ISISTimeOutID = XtAppAddTimeOut(*appcontext, timeout,
	    ProcessISISTimeOut, (caddr_t)appcontext);
} /* function AddNextISISTimeOut */

/***************************** End of xperfisis.c ***************************/
