/******************************************************************************
 * Module xpi_shells.c - Top-level window routines for xperfisis
 *
 * Written:	 05/14/91 by John H. Lee
 * Last Changed: 05/25/91 by John H. Lee
 * Version:	 0.2
 ******************************************************************************
 * Description:
 *
 *	This module is responsible for creating the top-level application
 * shells and for implementing the non-tool menu items, including the
 * creation and destruction of windows and tool, and for exiting the program.
 ******************************************************************************
 * Exports:
 *	Widget CreateApplication()	Create initial application window
 *
 * Imports:
 * 	xpi_tools routines
 *	xperfisis application-wide globals
 *	ToolClass *ToolsList		List of tools available
 *	ToolClass defaultToolClass	Default tool class record
 ******************************************************************************
 * Revisions:
 *   0.1   05/14/91 JHL  Initial Creation
 *   0.2   05/25/91 JHL  Window size no longer changes when tool changed
 *****************************************************************************/

#include <isis.h>
#include <X11/IntrinsicP.h>
#include <X11/CoreP.h>
#include <Xm/Xm.h>
#include <Xm/CascadeB.h>
#include <Xm/MainW.h>
#include <Xm/MenuShell.h>
#include <Xm/Form.h>
#include <Xm/Frame.h>
#include <Xm/LabelG.h>
#include <Xm/MessageB.h>
#include <Xm/PushB.h>
#include <Xm/RowColumn.h>

#include "xperfisis.h"
#include "xpi_shells.h"

#include "xpi_tooldefs.h"
#include "xpi_tools.h"
#include "xpi_TLdefault.h"





static XmString SeparatorStr,
		QuitTitleStr,
		QuitMsgStr,
		QuitStr,
		DeleteTitleStr,
		DeleteMsgStr,
		DeleteStr;

static void	QuitCB(),
		NewWindowCB(),
		DeleteWindowCB(),
		SetToolCB();

static Widget	CreateNewApplication();





/***** Create initial application window *****/

#ifdef	_NO_PROTO
Widget CreateApplication(display, app_name, app_class, app_win)
Display		*display;
char		*app_name,
		*app_class;
AppWindow	*app_win;
#else
Widget CreateApplication(Display *display, char *app_name, char *app_class,
	 char *app_win)
#endif	_NO_PROTO
{
Widget		app_shell;
int		n;
Arg		args[25];



				/* Create application shell		*/
	app_shell = CreateNewApplication(display, app_name, app_class,
		app_win);
	AppWindowsOpen = 1;
				/* Create common static strings		*/
	SeparatorStr = XmStringSeparatorCreate();
	QuitTitleStr = XmStringCreateSimple("Quit Warning");
	QuitMsgStr = XmStringCreateSimple("Do you wish to quit?");
	QuitStr = XmStringCreateSimple("Quit");
	DeleteTitleStr = XmStringCreateSimple("Delete Window Warning");
	DeleteStr = XmStringCreateSimple("Delete");
	DeleteMsgStr = XmStringCreateSimple("Really delete this window?");

		    		/* Return initial shell			*/
	return (app_shell);
} /* function CreateApplication */





/*** Change tool for Application Window ***/

static void ChangeWindowTool(app_win, tool_class)
AppWindow	*app_win;
ToolClass	tool_class;
{
char		info_buf[256];
XmString	info_str;
Dimension	frame_height = 1,
		frame_width = 1,
		form_height = 0,
		form_width = 0;
int		n;
Arg		args[10];


	/*** Cleanup old tool ***/
	/* NOTE:  A bug in the Motif v1.1 xmFrame widget doesn't permit us
	 * to destroy the current tool form and create a new one immediately.
	 * XtDestroyWidget doesn't actually remove the destroyed widget and
	 * update the parent's children count until we return up to toolkit
	 * event dispatcher, and the xmFrame widget incorrectly checks only
	 * the number of children instead of the number of _managed_ children.
	 * *Sigh*.  Our work around, then, it to change the frame as well
	 * and reset the xmMainWindow areas here.  Keep this in mind when
	 * looking at CreateNewApplication().
	 */
				/* Destroy old tool			*/
	if (app_win->tool != NULL)
		DestroyTool(app_win->tool);
				/* Destroy old frame & form		*/
	if (app_win->tool_frame != NULL) {
				/* Get current dimensions		*/
		n = 0;
		XtSetArg(args[n], XmNheight, &frame_height);		n++;
		XtSetArg(args[n], XmNwidth, &frame_width);		n++;
		XtGetValues(app_win->tool_frame, args, n);
		n = 0;
		XtSetArg(args[n], XmNheight, &form_height);		n++;
		XtSetArg(args[n], XmNwidth, &form_width);		n++;
		XtGetValues(app_win->tool_form, args, n);

		XtDestroyWidget(app_win->tool_frame);
	} /* if */


	/*** Create new tool area frame ***/

	n = 0;
	XtSetArg(args[n], XmNheight, frame_height);			n++;
	XtSetArg(args[n], XmNwidth, frame_width);			n++;
	XtSetArg(args[n], XmNbottomAttachment, XmATTACH_NONE);		n++;
	XtSetArg(args[n], XmNleftAttachment, XmATTACH_NONE);		n++;
	XtSetArg(args[n], XmNrightAttachment, XmATTACH_NONE);		n++;
	XtSetArg(args[n], XmNtopWidget, app_win->info_frame);		n++;
	XtSetArg(args[n], XmNtopAttachment, XmATTACH_WIDGET);		n++;
	XtSetArg(args[n], XmNresizable, True);				n++;
	app_win->tool_frame = XmCreateFrame(app_win->main_form,
		"tool_frame", args, n);
	XtManageChild(app_win->tool_frame);


	/*** Create new tool form ***/

				/* Create new Form for tool		*/
	n = 0;
	XtSetArg(args[n], XmNheight, form_height);			n++;
	XtSetArg(args[n], XmNwidth, form_width);			n++;
	app_win->tool_form = XmCreateForm(app_win->tool_frame,
		"tool_form", args, n);
	XtManageChild(app_win->tool_form);


	/*** Create default tool ***/

	app_win->tool = CreateTool(tool_class, app_win);

	(void)sprintf(info_buf, "Analysis Tool: %s",
		app_win->tool->common.class_rec->title);
	info_str = XmStringCreateSimple(info_buf);

	n = 0;
	XtSetArg(args[n], XmNlabelString, info_str);			n++;
	XtSetValues(app_win->info_lbl, args, n);
	XmStringFree(info_str);


	n = 0;
	XtSetArg(args[n], XmNbottomAttachment, XmATTACH_FORM);		n++;
	XtSetArg(args[n], XmNleftAttachment, XmATTACH_FORM);		n++;
	XtSetArg(args[n], XmNrightAttachment, XmATTACH_FORM);		n++;
	XtSetArg(args[n], XmNresizable, False);				n++;
	XtSetValues(app_win->tool_frame, args, n);
} /* function ChangeWindowTool */





/*** Create New Application Shell ***/

static Widget CreateNewApplication(display, app_name, app_class, app_win)
Display		*display;
char		*app_name,
		*app_class;
AppWindow	*app_win;
{
Widget		menu_pane,
		menu_ent,
		menu_csc,
		button;
ToolClass	*t_ent;
char		shell_title[64];

int		n;
Arg		args[25];

Tool		t;



				/* Create application shell		*/
	n = 0;
	XtSetArg(args[n], XmNallowShellResize, True);			n++;
	if (app_win->win_index > 0) {
	    (void)sprintf(shell_title, "%s Window #%d", app_name,
		    app_win->win_index + 1);
	    XtSetArg(args[n], XmNtitle, shell_title);			n++;
	} /* if */
	app_win->shell = XtAppCreateShell(app_name, app_class,
		applicationShellWidgetClass, display, args, n);

				/* Create main window			*/
	n = 0;
	app_win->main_win = XmCreateMainWindow(app_win->shell, "main",
		args, n);
	XtManageChild(app_win->main_win);
				/* Create menu bar			*/
	n = 0;
	app_win->menu_bar = XmCreateMenuBar(app_win->main_win, "menu_bar",
		args, n);
	XtManageChild(app_win->menu_bar);


	/*** Create "File" menu ***/

	n = 0;
	menu_pane = XmCreatePulldownMenu(app_win->menu_bar, "menu_pane",
		args, n);

	n = 0;
	menu_ent = XmCreatePushButton(menu_pane, "Quit", args, n);
	XtManageChild(menu_ent);
	XtAddCallback(menu_ent, XmNactivateCallback,
		QuitCB, (caddr_t)app_win);

	n = 0;
	XtSetArg(args[n], XmNsubMenuId, menu_pane);			n++;
	menu_csc = XmCreateCascadeButton(app_win->menu_bar, "File", args, n);
	XtManageChild(menu_csc);


	/*** Create "Window" menu ***/

	n = 0;
	menu_pane = XmCreatePulldownMenu(app_win->menu_bar, "menu_pane",
		args, n);

	n = 0;
	menu_ent = XmCreatePushButton(menu_pane, "New", args, n);
	XtManageChild(menu_ent);
	XtAddCallback(menu_ent, XmNactivateCallback, NewWindowCB, NULL);

	n = 0;
	menu_ent = XmCreatePushButton(menu_pane, "Delete", args, n);
	XtManageChild(menu_ent);
	XtAddCallback(menu_ent, XmNactivateCallback, DeleteWindowCB,
		(caddr_t)app_win);

	n = 0;
	XtSetArg(args[n], XmNsubMenuId, menu_pane);			n++;
	menu_csc = XmCreateCascadeButton(app_win->menu_bar, "Window",
		args, n);
	XtManageChild(menu_csc);


	/*** Create "Tools" menu ***/

	n = 0;
	menu_pane = XmCreatePulldownMenu(app_win->menu_bar, "menu_pane",
		args, n);
				/* Add entry for each tool		*/
	for (t_ent=ToolsList; *t_ent != NULL; t_ent++) {
	    n = 0;
	    XtSetArg(args[n], XmNuserData, *t_ent);			n++;
	    menu_ent = XmCreatePushButton(menu_pane, (*t_ent)->menu_name,
		    args, n);
	    XtManageChild(menu_ent);
	    XtAddCallback(menu_ent, XmNactivateCallback, SetToolCB,
		    (caddr_t)app_win);
	} /* for */

	n = 0;
	XtSetArg(args[n], XmNsubMenuId, menu_pane);			n++;
	menu_csc = XmCreateCascadeButton(app_win->menu_bar, "Tools",
		args, n);
	XtManageChild(menu_csc);


	/*** Create work area form ***/

	n = 0;
	XtSetArg(args[n], XmNresizePolicy, XmRESIZE_NONE);		n++;
	app_win->main_form = XmCreateForm(app_win->main_win, "main_form",
		args, n);
	XtManageChild(app_win->main_form);

				/* Information area label		*/
	n = 0;
	XtSetArg(args[n], XmNbottomAttachment, XmATTACH_NONE);		n++;
	XtSetArg(args[n], XmNleftAttachment, XmATTACH_FORM);		n++;
	XtSetArg(args[n], XmNrightAttachment, XmATTACH_FORM);		n++;
	XtSetArg(args[n], XmNtopAttachment, XmATTACH_FORM);		n++;
	XtSetArg(args[n], XmNresizable, False);				n++;
	app_win->info_frame = XmCreateFrame(app_win->main_form,
		"info_frame", args, n);
	XtManageChild(app_win->info_frame);

	n = 0;
	XtSetArg(args[n], XmNalignment, XmALIGNMENT_BEGINNING);		n++;
	app_win->info_lbl = XmCreateLabelGadget(app_win->info_frame,
		"infoLabel", args, n);
	XtManageChild(app_win->info_lbl);


				/* Set areas of main window		*/
	XmMainWindowSetAreas(app_win->main_win, app_win->menu_bar,
		NULL, NULL, NULL, app_win->main_form);


	/*** Create Warning dialog ***/

	n = 0;
	XtSetArg(args[n], XmNdefaultButtonType, XmDIALOG_CANCEL_BUTTON); n++;
	XtSetArg(args[n], XmNdialogStyle,
		XmDIALOG_FULL_APPLICATION_MODAL);			n++;
	app_win->warn_dlg = XmCreateWarningDialog(app_win->shell, "warn",
		args, n);
				/* Remove help button			*/
	button = XmMessageBoxGetChild(app_win->warn_dlg,
		XmDIALOG_HELP_BUTTON);
	XtUnmanageChild(button);


	/*** Init other data on the newly create application window */

	app_win->tool_frame = NULL;
	app_win->tool_form = NULL;
	app_win->title = app_name;
	app_win->tool = NULL;


	/*** Create default tool ***/
	/* "Hey!  Where's the XmMainWindowSetAreas?"
	 * "It's in ChangeWindowTool().  The 'why' is in there."
	 */
	ChangeWindowTool(app_win, defaultToolClass);


	return (app_win->shell);
} /* function CreateNewApplication */




/*****************************************************************************
 *
 *	Callback Routines
 *
 ****************************************************************************/
/*****************************************************************************
 *
 *	Quit Application Callback Routines
 *
 ****************************************************************************/

static void	QuitConfirmCB(),
		QuitCancelCB();

/*** Quit Application Callback ***/
static void QuitCB(w, client_data, call_data)
Widget	w;		/* UNUSED */
caddr_t	client_data;	/* Application window */
caddr_t	call_data;	/* UNUSED */
{
AppWindow	*app_win = (AppWindow *)client_data;
char		msg_buf[256];
XmString	msg_str1,
		msg_str2,
		msg_str;


				/* Create warning message		*/
	if (AppWindowsOpen > 1) {
				/* Multiple windows open		*/
	    (void)sprintf(msg_buf, "There are %d windows still open.",
		    AppWindowsOpen);
	    msg_str1 = XmStringCreateSimple(msg_buf);
	    msg_str2 = XmStringConcat(msg_str1, SeparatorStr);
	    XmStringFree(msg_str1);
	    msg_str = XmStringConcat(msg_str2, QuitMsgStr);
	    XmStringFree(msg_str2);
	} else {
	    msg_str = QuitMsgStr;
	} /* else */

				/* Set message & position warning dialog */
	SetMessageDialog(app_win->warn_dlg, app_win->shell, QuitTitleStr,
		msg_str, QuitStr, NULL);
	XtAddCallback(app_win->warn_dlg, XmNokCallback, QuitConfirmCB,
		client_data);
	XtAddCallback(app_win->warn_dlg, XmNcancelCallback, QuitCancelCB,
		client_data);

	if (msg_str != QuitMsgStr) XmStringFree(msg_str);

				/* Popup warning dialog			*/
	XtManageChild(app_win->warn_dlg);
} /* function QuitCB */





/*** Cancel quit callback ***/
static void QuitCancelCB(w, client_data, call_data)
Widget	w;		/* UNUSED */
caddr_t	client_data;	/* Application Window index */
caddr_t	call_data;	/* UNUSED */
{
AppWindow	*app_win = (AppWindow *)client_data;


				/* Remove callbacks			*/
	XtRemoveCallback(app_win->warn_dlg, XmNokCallback, QuitConfirmCB,
		client_data);
	XtRemoveCallback(app_win->warn_dlg, XmNcancelCallback, QuitCancelCB,
		client_data);
} /* function QuitCancelCB */





/*** Confirm quit callback ***/
static void QuitConfirmCB(w, client_data, call_data)
Widget	w;		/* UNUSED */
caddr_t	client_data;	/* UNUSED */
caddr_t	call_data;	/* UNUSED */
{
int	i;


				/* Destroy all active tools		*/
	for (i=AppWindowsSize-1; i >= 0; i--) {
	    if (AppWindows[i] != NULL) DestroyTool(AppWindows[i]->tool);
	} /* for */

	exit(0);		/* Exit application			*/
} /* function QuitConfirmCB */

/*****************************************************************************
 *
 *	Create/Delete Application Window Callback Routines
 *
 ****************************************************************************/

static void	DeleteWindowConfirmCB(),
		DeleteWindowCancelCB();

/*** Create new window callback ***/
static void NewWindowCB(w, client_data, call_data)
Widget	w;		/* UNUSED */
caddr_t	client_data;	/* UNUSED */
caddr_t	call_data;	/* UNUSED */
{
int		win_index;
AppWindow	*app_win;
Widget		shell;


				/* Allocate unused window entry		*/
	for (win_index=0; win_index < AppWindowsSize; win_index++)
		if (AppWindows[win_index] == NULL) break;
				/* None free?  Resize window table	*/
	if (win_index >= AppWindowsSize) {
	    win_index = AppWindowsSize;
	    AppWindowsSize += 10;
	    AppWindows = (AppWindow **)XtRealloc(AppWindows,
		    AppWindowsSize * sizeof(AppWindow *));
	} /* if */
	    
				/* Create & install new window entry	*/
	app_win = XtNew(AppWindow);
	AppWindows[win_index] = app_win;
				/* Initialize application window values	*/
	app_win->win_index = win_index;
	app_win->app_context = AppContext;
	app_win->gaddr_p = GAddr_P;
				/* Create shell				*/
	shell = CreateNewApplication(AppDisplay, AppName, AppClass, app_win);
	XtRealizeWidget(shell);

	AppWindowsOpen++;
} /* function NewWindowCB */





/*** Delete existing window callback ***/
static void DeleteWindowCB(w, client_data, call_data)
Widget	w;		/* UNUSED */
caddr_t	client_data;	/* Application window */
caddr_t	call_data;	/* UNUSED */
{
AppWindow	*app_win = (AppWindow *)client_data;
Arg	args[5];
int	n;


	if (AppWindowsOpen == 1) QuitCB(w, client_data, call_data);
	else {
				/* Set warning dialog 			*/
	    SetMessageDialog(app_win->warn_dlg, app_win->shell,
		    DeleteTitleStr, DeleteMsgStr, DeleteStr, NULL);
	    XtAddCallback(app_win->warn_dlg, XmNokCallback,
		    DeleteWindowConfirmCB, client_data);
	    XtAddCallback(app_win->warn_dlg, XmNcancelCallback,
		    DeleteWindowCancelCB, client_data);

				/* Popup warning dialog			*/
	    XtManageChild(app_win->warn_dlg);
	} /* else */
} /* function DeleteWindowCB */





/*** Confirm delete window callback ***/
static void DeleteWindowConfirmCB(w, client_data, call_data)
Widget	w;		/* UNUSED */
caddr_t	client_data;	/* Application window */
caddr_t	call_data;	/* UNUSED */
{
AppWindow	*app_win = (AppWindow *)client_data;


				/* Destroy tool				*/
	DestroyTool(app_win->tool);
				/* Destroy shell & all children		*/
	XtDestroyWidget(app_win->shell);
				/* Mark entry free for reuse		*/
	AppWindows[app_win->win_index] = NULL;
	AppWindowsOpen--;
				/* Free window				*/
	XtFree(app_win);
} /* function DeleteWindowConfirmCB */





/*** Cancel delete window callback ***/
static void DeleteWindowCancelCB(w, client_data, call_data)
Widget	w;		/* UNUSED */
caddr_t	client_data;	/* Application window */
caddr_t	call_data;	/* UNUSED */
{
AppWindow	*app_win = (AppWindow *)client_data;


				/* Remove callbacks			*/
	XtRemoveCallback(app_win->warn_dlg, XmNokCallback,
		DeleteWindowConfirmCB, client_data);
	XtRemoveCallback(app_win->warn_dlg, XmNcancelCallback,
		DeleteWindowCancelCB, client_data);
} /* function DeleteWindowCancelCB */

/*****************************************************************************
 *
 *	Set Window Tool Callback Routine
 *
 ****************************************************************************/

/*** Set tool in existing window callback ***/
static void SetToolCB(w, client_data, call_data)
Widget	w;		/* Menu pushbutton widget */
caddr_t	client_data;	/* Application window */
caddr_t	call_data;	/* UNUSED */
{
AppWindow	*app_win = (AppWindow *)client_data;
ToolClass	tool_class;
int		n;
Arg		args[1];


				/* Get tool class 			*/
	n = 0;
	XtSetArg(args[n], XmNuserData, &tool_class);			n++;
	XtGetValues(w, args, n);

				/* Different class?  Change tool	*/
#ifdef	NOTDEF
	if (tool_class != win_entry->tool->common.class_rec)
#endif	NOTDEF
		ChangeWindowTool((int)client_data, tool_class);
} /* function SetToolCB */

/**************************** End of xpi_shells.c ****************************/
