/******************************************************************************
 * Module xpi_graph.c - Graph display handler for ISIS performance tool for X
 *
 * Written:	 05/15/91 by John H. Lee
 * Last Changed: 06/17/91 by John H. Lee
 * Version:	 0.3
 ******************************************************************************
 * Description:
 *	This set of routines is used to create a graph and maintain a graph
 * display.  This is more properly done as a graph widget.
 ******************************************************************************
 * Exports:
 *	Graph GraphCreate()		Create & initialize a graph instance
 *	void GraphDestroy()		Destroy a graph instance
 *	void GrapSet()			Change graph parameters
 *	void GraphSetData()		Change data displayed by graph
 *	void GraphAddPoint()		Add data point to graph
 *	void GraphDeleteData()		Delete current data points from graph
 *
 * Imports:
 * 	None
 ******************************************************************************
 * Notes:
 *	This really, Really, REALLY should be done as a graph widget.
 *	DrawCB should use the expose region when copying from the pixmap.
 ******************************************************************************
 * Revisions:
 *   0.1   05/15/91 JHL  Initial Creation
 *   0.2   05/30/91 JHL  Graph now coalesces coincidental data points on screen
 *   0.3   06/17/91 JHL  Added function prototypes
 *****************************************************************************/

#include <limits.h>
#include <math.h>

#include <X11/Intrinsic.h>
#include <X11/IntrinsicP.h>
#include <X11/CoreP.h>
#include <Xm/Xm.h>
#include <Xm/Label.h>

#include "xpi_graph.h"






/*** Callback routines ***/

static void	DrawCB(),
		ResizeCB(),
		DestroyCB();


/*** Private Routines ***/

static void	ValToWinPoint(),
		WinToValPoint(),
		DrawAxis(),
		DrawData(),
		DrawPoint(),
		SetRedraw(),
		AddDataPoint(),
		GraphResize(),
		GraphSetSize();
/*****************************************************************************
 *
 *	Creation/Destruction Routines
 *
 ****************************************************************************/

#ifdef	_NO_PROTO
Graph *GraphCreate(w, g)
Widget	w;
Graph		*g;
#else
Graph *GraphCreate(Widget w, Graph *g)
#endif	_NO_PROTO
{
Widget		tic_lbl;
XmFontList	fl;
XmFontContext	fcontext;
XmStringCharSet	charset;
XFontStruct	*def_font;
Pixel		fg_color,
		bg_color;
XmString	test_str;
char		test_txt[256];

int		n;
Arg		arg[5];

XGCValues	values;
XSetWindowAttributes	attributes;


	if (g == NULL) {
	    g = (Graph *)XtMalloc(sizeof(Graph));
				/* Initialize public fields		*/
	    g->view_min_x = 0;
	    g->view_min_y = 0;
	    g->interval_x = 10;
	    g->interval_base_x = 0;
	    g->interval_y = 10;
	    g->interval_base_y = 0;
	    g->radix_x = 1;
	    g->radix_y = 1;
	    g->scale_x = 1;
	    g->scale_y = 1;
	    g->label_x = NULL;
	    g->label_y = NULL;
	    g->format_x = "%g";
	    g->format_y = "%g";
	    g->auto_scale = True;
	} /* if */
				/* Initialize private or read-only fields */
	g->draw_area = w;
	g->draw_pixmap = NULL;
	g->data = NULL;
	g->num_data = 0;
	g->data_sz = 0;
	g->graph_data = NULL;
	g->num_graph = 0;
	g->graph_sz = 0;
	g->view_max_x = 100;
	g->view_max_y = 100;
	g->min_x = LONG_MAX;
	g->max_x = LONG_MIN;
	g->min_y = LONG_MAX;
	g->max_y = LONG_MIN;
	g->not_realized = True;
				/* Use dummy widget to get resources	*/
	tic_lbl = XmCreateLabel(w, "tic_label", NULL, 0);
				/*   Get fontlist			*/
	n = 0;
	XtSetArg(arg[n], XmNfontList, &fl);				n++;
	XtSetArg(arg[n], XmNforeground, &fg_color);			n++;
	XtSetArg(arg[n], XmNbackground, &bg_color);			n++;
	XtGetValues(tic_lbl, arg, n);
	g->fontlist = XmFontListCopy(fl);
	XtDestroyWidget(tic_lbl);

				/* Calculate axis tic mark spacing	*/
	(void)sprintf(test_txt, g->format_x, 9999.9);
	test_str = XmStringCreateSimple(test_txt);
	XmStringExtent(g->fontlist, test_str,
		&g->tic_spacing_x, &g->tic_spacing_y);
	XmStringFree(test_str);
	g->tic_spacing_x += 5;
	g->tic_spacing_y = 2 * g->tic_spacing_y;
				/* Calculate axis margins		*/
	(void)sprintf(test_txt, g->format_y, 999.9);
	test_str = XmStringCreateSimple(test_txt);
	XmStringExtent(g->fontlist, test_str,
		&g->tic_margin_width, &g->tic_margin_height);
	XmStringFree(test_str);
	g->tic_margin_width += 5;
	g->tic_margin_height += 5;

				/* Create label widgets			*/
	n = 0;
	XtSetArg(arg[n], XmNlabelString, g->label_x);			n++;
	g->x_lbl = XmCreateLabel(w, "x_label", arg, n);
	XtManageChild(g->x_lbl);

	n = 0;
	XtSetArg(arg[n], XmNx, 0);					n++;
	XtSetArg(arg[n], XmNy, 0);					n++;
	XtSetArg(arg[n], XmNlabelString, g->label_y);			n++;
	g->y_lbl = XmCreateLabel(w, "y_label", arg, n);
	XtManageChild(g->y_lbl);

				/* Allocate (modifiable) GC for drawing	*/
	(void)XmFontListInitFontContext(&fcontext, g->fontlist);
	(void)XmFontListGetNextFont(fcontext, &charset, &def_font);
	(void)XmFontListFreeFontContext(fcontext);
	XtFree(charset);
	values.font = def_font->fid;
	values.foreground = fg_color;
	g->draw_gc = XCreateGC(XtDisplay(w), RootWindowOfScreen(XtScreen(w)),
		GCFont | GCForeground, &values);

				/* Allocate GC for erasing		*/
	values.function = GXcopy;
	values.foreground = bg_color;
	values.fill_style = FillSolid;
	g->erase_gc = XtGetGC(w,
		GCFunction | GCForeground | GCFillStyle,
		&values);

				/* Attach our draw routine		*/
	XtAddCallback(w, XmNexposeCallback, DrawCB, (caddr_t)g);
				/* Attach our resize routine		*/
	XtAddCallback(w, XmNresizeCallback, ResizeCB, (caddr_t)g);
				/* Attach our destroy routine		*/
	XtAddCallback(w, XmNdestroyCallback, DestroyCB, (caddr_t)g);

				/* Do window-size dependent things	*/
	GraphResize(w, g);
	g->redraw = True;

				/* Return "Graph ID" to caller		*/
	return(g);
} /* function GraphCreate */





#ifdef	_NO_PROTO
void GraphDestroy(g)
Graph	*g;
#else
void GraphDestroy(Graph *g)
#endif	_NO_PROTO
{
				/* Remove callbacks			*/
	XtRemoveCallback(g->draw_area, XmNexposeCallback, DrawCB,
		(caddr_t)g);
	XtRemoveCallback(g->draw_area, XmNresizeCallback, ResizeCB,
		(caddr_t)g);
	XtRemoveCallback(g->draw_area, XmNdestroyCallback, DestroyCB,
		(caddr_t)g);
				/* Destroy auxillary widgets		*/
	XtDestroyWidget(g->x_lbl);
	XtDestroyWidget(g->y_lbl);
				/* Free draw pixmap, GC			*/
	XFreePixmap(XtDisplay(g->draw_area), g->draw_pixmap);
	XFreeGC(XtDisplay(g->draw_area), g->draw_gc);
	XtReleaseGC(g->draw_area, g->erase_gc);
} /* function GraphDestroy */





static void DestroyCB(w, client_data, call_data)
Widget	w;
caddr_t client_data;
caddr_t	call_data;
{
Graph	*g = (Graph *)client_data;


	GraphDestroy(g);
} /* function DestroyCB */
/*****************************************************************************
 *
 *	Expose/Draw routines
 *
 ****************************************************************************/

static void DrawCB(w, client_data, call_data)
Widget	w;
caddr_t	client_data;
caddr_t	call_data;
{
XmDrawingAreaCallbackStruct	*cbinfo = (XmDrawingAreaCallbackStruct *)
					call_data;
Graph	*g = (Graph *)client_data;


	if (g->not_realized) {	/* Do Graph "Realization-time" stuff	*/
	    XSetWindowAttributes	attributes;

				/* Change bit gravity			*/
	    attributes.bit_gravity = ForgetGravity;
	    XChangeWindowAttributes(XtDisplay(w), XtWindow(w),
		    CWBitGravity, &attributes);

				/* Recompute internal values		*/
	    GraphResize(w, g);
	    g->redraw = True;

				/* We'll do this only once		*/
	    g->not_realized = False;
	} /* if */

	if (g->redraw) {	/* Reconstruct image bitmap		*/
	    XFillRectangle(XtDisplay(w), g->draw_pixmap, g->erase_gc,
			0, 0, w->core.width, w->core.height);
	    DrawAxis(g);
	    DrawData(g);

	    g->redraw = False;
	} /* if */
				/* Copy pixmap into window		*/
	XCopyArea(XtDisplay(w), g->draw_pixmap, cbinfo->window, g->draw_gc,
		0, 0, w->core.width-1, w->core.height-1,
		0, 0);
} /* function DrawCB */



static void ValToWinPoint(g, x, y, plot_x, plot_y)
Graph		*g;
long		x,		/* Coordinates in data space		*/
		y;
Position	*plot_x,	/* Coordinates in X11 window space	*/
		*plot_y;
{
	*plot_x = (Position)((x - g->view_min_x)
		/ g->scale_x + g->tic_margin_width + 0.5);
	*plot_y = (Position)(g->draw_area->core.height
		- g->x_lbl->core.height - 2 * g->x_lbl->core.border_width
		- g->tic_margin_height
		- (y - g->view_min_y) / g->scale_y + 0.5);
} /* function ValToWinPoint */



static void WinToValPoint(g, plot_x, plot_y, x, y)
Graph	*g;
int	plot_x,			/* Coordinates in X11 window space	*/
	plot_y;
long	*x,			/* Coordinates in data space		*/
	*y;
{
	*x = (plot_x - g->tic_margin_width) * g->scale_x + g->view_min_x;
	*y = (g->draw_area->core.height
		- g->x_lbl->core.height - g->x_lbl->core.border_width
		- g->tic_margin_height - plot_y) * g->scale_y + g->view_min_y;
} /* function WinToValPoint */



static float CalcValueX(g, val)
Graph		*g;
long		val;		/* Coordinates in data space		*/
{
	return ((float)val / (float)g->radix_x);
} /* function CalcValueX */



static float CalcValueY(g, val)
Graph		*g;
long		val;		/* Coordinates in data space		*/
{
	return ((float)val / (float)g->radix_y);
} /* function CalcValueY */



static void DrawAxis(g)
Graph	*g;
{
XPoint		axis[3];
long		val;
Position	plot_x,
		plot_y;
Dimension	tic_width,
		tic_baseln;
char		tic_buf[32];
XmString	tic_string;

Arg		arg[5];
int		n;



				/* Figure location of axis lines	*/
	ValToWinPoint(g, g->view_min_x, g->view_min_y, &plot_x, &plot_y);
	axis[0].x = plot_x - 1;
	axis[0].y = g->y_lbl->core.height + 2*g->y_lbl->core.border_width + 5;
	axis[1].x = axis[0].x;
	axis[1].y = plot_y + 1;
	axis[2].x = g->draw_area->core.width - 10;
	axis[2].y = axis[1].y;
				/* Draw axis lines			*/
	XDrawLines(XtDisplay(g->draw_area), g->draw_pixmap, g->draw_gc,
		 axis, XtNumber(axis), CoordModeOrigin);

	/*** Draw x-axis tic marks ***/
				/* Start tic marks on but not before axis */
	val = g->interval_base_x;
	while (val < g->view_min_x)
		val += g->interval_x;
	while (val >= g->view_min_x)
		val -= g->interval_x;

	ValToWinPoint(g, val, 0, &plot_x, &plot_y);
	if (plot_x < axis[0].x) {
	    val += g->interval_x;
	    ValToWinPoint(g, val, 0, &plot_x, &plot_y);
	} /* if */
	while (plot_x <= axis[2].x) {
				/* Draw mark				*/
	    XDrawLine(XtDisplay(g->draw_area), g->draw_pixmap, g->draw_gc,
		    plot_x, axis[1].y + 1, plot_x, axis[1].y + 3);
				/* Draw mark value			*/
	    (void)sprintf(tic_buf, g->format_x, CalcValueX(g, val));
	    tic_string = XmStringCreateSimple(tic_buf);
	    tic_width = XmStringWidth(g->fontlist, tic_string);
	    XmStringDraw(XtDisplay(g->draw_area), g->draw_pixmap,
		    g->fontlist, tic_string, g->draw_gc,
		    plot_x - tic_width / 2, axis[1].y + 3, tic_width,
		    XmALIGNMENT_BEGINNING, XmSTRING_DIRECTION_L_TO_R, NULL);
	    XmStringFree(tic_string);

	    val += g->interval_x;
	    ValToWinPoint(g, val, 0, &plot_x, &plot_y);
	} /* while */


	/*** Draw y-axis tic marks ***/
				/* Start tics on but not below axis	*/
	val = g->interval_base_y;
	while (val < g->view_min_y)
		val += g->interval_y;
	while (val >= g->view_min_y)
		val -= g->interval_y;

	ValToWinPoint(g, 0, val, &plot_x, &plot_y);
	if (plot_y > axis[1].y) {
		val += g->interval_y;
		ValToWinPoint(g, 0, val, &plot_x, &plot_y);
	} /* if */
	while (plot_y >= axis[0].y) {
				/* Draw mark				*/
	    XDrawLine(XtDisplay(g->draw_area), g->draw_pixmap, g->draw_gc,
		    axis[1].x - 3, plot_y, axis[1].x - 1, plot_y);
				/* Draw mark value			*/
	    (void)sprintf(tic_buf, g->format_y, CalcValueY(g, val));
	    tic_string = XmStringCreateSimple(tic_buf);
	    tic_width = XmStringWidth(g->fontlist, tic_string);
	    tic_baseln = XmStringBaseline(g->fontlist, tic_string) / 2;
	    XmStringDraw(XtDisplay(g->draw_area), g->draw_pixmap,
		    g->fontlist, tic_string, g->draw_gc,
		    axis[1].x - tic_width - 4, plot_y - tic_baseln, tic_width,
		    XmALIGNMENT_BEGINNING, XmSTRING_DIRECTION_L_TO_R, NULL);
	    XmStringFree(tic_string);

	    val += g->interval_y;
	    ValToWinPoint(g, 0, val, &plot_x, &plot_y);
	} /* while */

} /* function DrawAxis */



static void DrawData(g)
Graph	*g;
{
int		i;
Position	plot_x,
		plot_y,
		base_y;
GDatum		*d;


				/* Nothing to plot?			*/
	if (g->num_graph == 0) return;

				/* Get baseline plot values		*/
	ValToWinPoint(g, g->view_min_x, g->view_min_y, &plot_x, &base_y);

				/* Loop through array, plotting data	*/
	for (i=g->num_graph, d=g->graph_data; i > 0 && d->x < g->view_max_x;
		d++, i--) {
	    ValToWinPoint(g, d->x, d->cnt, &plot_x, &plot_y);
	    XDrawLine(XtDisplay(g->draw_area), g->draw_pixmap,
		    g->draw_gc, plot_x, base_y, plot_x, plot_y);
	} /* for */
} /* function DrawData */





#ifdef	NOTDEF
static void DrawData(g)
Graph	*g;
{
int	i;
Position	last_x = SHRT_MIN,
		last_y,
		plot_x,
		plot_y,
		x;
GDatum		*d;


				/* Nothing to plot?			*/
	if (g->num_data == 0) return;

				/* Locate first visible datum		*/
	for (d=g->data, i=g->num_data; i > 0; d++, i++)
		if (d->x >= g->view_min_x) break;
				/* Not at left visible edge?  Get preceding */
	if ((g->view_min_x < d->x) && (d > g->data)) d--;

				/* Loop through array, plotting line	*/
	for (i=g->num_data - (d - g->data) - 1, x=d->x; i > 0; d++, i--) {
				/* Break loop if last pt. not visible	*/
	    if (d->x >= g->view_max_x) break;
	    else {		/* Plot line to y at each x		*/
		ValToWinPoint(g, d->x, d->cnt, &plot_x, &plot_y);
		if (last_x != SHRT_MIN) XDrawLine(
			XtDisplay(g->draw_area), g->draw_pixmap, g->draw_gc,
			last_x, last_y, plot_x, plot_y);
				/* Save location of last point		*/
		last_x = plot_x;
		last_y = plot_y;
	    } /* else */
	} /* for */
				/* Plot to last point			*/
	ValToWinPoint(g, d->x, d->cnt, &plot_x, &plot_y);
	if (last_x != SHRT_MIN) {
	    XDrawLine(XtDisplay(g->draw_area), g->draw_pixmap, g->draw_gc,
		    last_x, last_y, plot_x, plot_y);
	} /* if */
} /* function DrawData */
#endif	NOTDEF
/*****************************************************************************
 *
 *	Resize routines
 *
 ****************************************************************************/

static void GeneratePlotData(g)
Graph	*g;
{
int	i;
Position	last_x,
		plot_x,
		junk_y;
long		x,
		cnt;
GDatum		*d;


				/* Reset graph data			*/
	g->num_graph = 0;

				/* Nothing to plot?			*/
	if (g->num_data == 0) return;

				/* Locate first visible datum		*/
	for (d=g->data, i=g->num_data; i > 0; d++, i++)
		if (d->x >= g->view_min_x) break;

	x = d->x;		/* Get first datum			*/
	cnt = d->cnt;
	ValToWinPoint(g, x, cnt, &last_x, &junk_y);
	d++;
				/* Loop through array, merging data	*/
	for (i=g->num_data-(d - g->data); i > 0; d++, i--) {
	    ValToWinPoint(g, d->x, d->cnt, &plot_x, &junk_y);
				/* Same window x as last point?		*/
	    if (plot_x == last_x) {
				/* Then combine points			*/
		cnt += d->cnt;
	    } else {		/* Else store plot point		*/
				/* Check array for room			*/
		if (g->num_graph == g->graph_sz) {
				/* Allocate more room			*/
		    g->graph_sz += 100;
		    g->graph_data = (GDatum *)XtRealloc(g->graph_data,
			    sizeof(GDatum) * g->graph_sz);
		} /* if */

		g->graph_data[g->num_graph].x = x;
		g->graph_data[g->num_graph++].cnt = cnt;

		x = d->x;
		last_x = plot_x;
		cnt = d->cnt;
	    } /* else */
	} /* for */

				/* Check array for room			*/
	if (g->num_graph == g->graph_sz) {
				/* Allocate more room			*/
	    g->graph_sz += 100;
	    g->graph_data = (GDatum *)XtRealloc(g->graph_data,
		    sizeof(GDatum) * g->graph_sz);
	} /* if */

	g->graph_data[g->num_graph].x = x;
	g->graph_data[g->num_graph++].cnt = cnt;
} /* function GeneratePlotData */





/*** Find new axis "tic" interval
 * Choosing a tic interval can be tricky; if we're not careful, they'll
 * overlap or be to far apart.  Using a heuristic, we'll look at how the
 * tics are spaced using a "every 10", "every 5", or "every 2" interval
 * (at the appropriate power of ten) and choose the one coming closest
 * to the desired window coordinates spacing.  It won't be perfect, but
 * we can't do better without using odd-ball intervals like "every 3" that
 * no human will appreciate.
 */
static long DetermineInterval(scale, radix, spacing)
long		scale,
		radix;
Dimension	spacing;
{
double	ratio,
	power,
	multiple;
long	interval;


	ratio = (double)radix / ((double)scale * (double)spacing);
	power = floor(log10(ratio));
	multiple = ratio / pow((double)10.0, power);

	if (multiple < 1.5)	/* ~1.0-1.5 x spacing @1x10^(-power) units */
		interval = (long)(pow((double)10.0, -power) * radix + 0.5);

	else if (multiple < 3)	/* ~.75-1.5 x spacing @5x10^(-power-1) units */
		interval = (long)(5.0 * pow((double)10.0, -power - 1.0)
			* radix + 0.5);

	else if (multiple < 7)	/* ~0.6-1.4 x spacing @2x10^(-power-1) units */
		interval = (long)(2.0 * pow((double)10.0, -power - 1.0)
			* radix + 0.5);

	else			/* ~0.7-1.0 x spacing @1x10^(-power-1) units */
		interval = (long)(pow((double)10.0, -power - 1.0)
			* radix + 0.5);

	return (interval);
} /* function DetermineInterval */




static void GraphResize(w, g)
Widget	w;
Graph	*g;
{
unsigned long	valuemask;
XGCValues	values;
XRectangle	clip_rect;
int		i;
Position	x,
		y;
GDatum		*d;

int		n;
Arg		arg[5];


				/* Change size of pixmap		*/
	if (g->draw_pixmap) XFreePixmap(XtDisplay(w), g->draw_pixmap);
	g->draw_pixmap = XCreatePixmap(
		XtDisplay(w), RootWindowOfScreen(XtScreen(w)),
		w->core.width, w->core.height, w->core.depth);

				/* Place x-axis label widget		*/
	n = 0;
	XtSetArg(arg[n], XmNx, (w->core.width - g->x_lbl->core.width
		- 2 * g->x_lbl->core.border_width) / 2);		n++;
	XtSetArg(arg[n], XmNy,
		w->core.height - g->x_lbl->core.height
		- 2 * g->x_lbl->core.border_width - 2);			n++;
	XtSetValues(g->x_lbl, arg, n);

				/* Set scaling, min-max, etc. based on data */
	GraphSetSize(w, g);
} /* function GraphResize */





static void GraphSetSize(w, g)
Widget	w;
Graph	*g;
{
unsigned long	valuemask;
XGCValues	values;
XRectangle	clip_rect;
int		i;
Position	x,
		y;
GDatum		*d;


				/* Generate plot data			*/
	if (!g->auto_scale) {	/* Determine new graph limits		*/
	    GeneratePlotData(g);
	    WinToValPoint(g, w->core.width - 20, g->y_lbl->core.height
		    + 2 * g->y_lbl->core.border_width + 2,
		    &g->view_max_x, &g->view_max_y);
	    if (g->view_max_x < g->view_min_x) g->view_max_x = g->view_min_x;
	    if (g->view_max_y < g->view_min_y) g->view_max_y = g->view_min_y;

				/* No data?  Assume reasonable defaults	*/
	} else if (g->num_data == 0) {
	    g->scale_x = g->radix_x;
	    g->scale_y = g->radix_y;
	    g->view_min_x = 0;
	    g->view_min_y = 0;
	    WinToValPoint(g, w->core.width - 20, g->y_lbl->core.height
		    + 2 * g->y_lbl->core.border_width + 2,
		    &g->view_max_x, &g->view_max_y);
	    g->interval_x = DetermineInterval(g->scale_x, g->radix_x,
		    g->tic_spacing_x);
	    g->interval_y = DetermineInterval(g->scale_y, g->radix_y,
		    g->tic_spacing_y);

	} else {		/* Size up graph & calculate nice scale	*/
	    /*** Chicken and Egg Problem
	     * What we want to do here is to determine new view limits and
	     * scale factors so that we can display all the graph data.
	     * We also want to merge data points if they coincide in the
	     * output window (i.e., they have the same x in window space,)
	     * but we can't know what the min & max y values are until we
	     * have a scale factor first.  Fortunately, all we need is the
	     * x scale factor to merge data-points, and the min & max x
	     * values are fixed, directly determinable from the data.
	     */
				/* Set x scale-factor			*/
	    g->view_min_x = (long)((int)((float)g->data[0].x
		    / (float)g->radix_x / 10.0) * 10 * g->radix_y);
	    g->view_max_x = (long)ceil(g->data[g->num_data - 1].x / g->radix_x
		    / 10.0) * 10 * g->radix_x;
	    ValToWinPoint(g, g->view_min_x, 0, &x, &y);
	    g->scale_x = (long)ceil((double)(g->view_max_x - g->view_min_x)
		    / (double)(w->core.width - 10 - x));
	    if (g->scale_x < 1) g->scale_x = 1;

				/* Now generate graph w/merged points	*/
	    GeneratePlotData(g);

	    d = g->graph_data;	/* Now determine max y's (cnt's)	*/
	    g->view_min_y = 0;	/* We always want to start at 0		*/
	    g->view_max_y = d->cnt;
	    d++;
	    for (i=g->num_graph-1; i > 0; d++, i--) {
		if (d->cnt > g->view_max_y) g->view_max_y = d->cnt;
	    } /* for */
				/* Determine new y scale-factor		*/
	    ValToWinPoint(g, g->view_min_x, g->view_min_y, &x, &y);
	    g->scale_y = (long)ceil((ceil((double)g->view_max_y
		    / (double)g->radix_y / 10.0) * g->radix_y * 10)
		    / (y - g->y_lbl->core.height - 5
		    - 2 * g->y_lbl->core.border_width - g->tic_margin_height));
	    if (g->scale_y < 1) g->scale_y = 1;

				/* Determine new axis tic intervals	*/
	    g->interval_x = DetermineInterval(g->scale_x, g->radix_x,
		    g->tic_spacing_x);
	    g->interval_y = DetermineInterval(g->scale_y, g->radix_y,
		    g->tic_spacing_y);
	} /* else */
	    
#ifdef	NOTDEF
				/* Determine new clipping region	*/
	ValToWinPoint(g, g->view_min_x, g->view_max_y, &clip_rec.x, &clip_rec.y);
	ValToWinPoint(g, g->view_max_x, g->view_min_y,
		&clip_rec.width, &clip_rec.height);
	clip_rec.width -= clip_rec.x - 1;
	clip_rec.height -= clip_rec.y - 1;
	XSetClipRectangle(XtDisplay(w), g->draw_gc,
		0, 0, &clip_rect, 1, Unsorted);
#endif	NOTDEF
} /* function GraphResize */



static void ResizeCB(w, client_data, call_data)
Widget	w;
caddr_t	call_data;
caddr_t	client_data;
{
Graph	*g = (Graph *)client_data;


	GraphResize(w, g);

				/* We'll need to reconstruct the window	*/
	g->redraw = True;
} /* function Resize */
/*****************************************************************************
 *
 *	Other private routines
 *
 ****************************************************************************/

static void SetRedraw(g)
Graph	*g;
{
				/* Rescale stuff, if we're autoscaling	*/
	if (g->auto_scale) GraphSetSize(g->draw_area, g);
				/* Erase current graph, redraw w/new info */
	if (!g->redraw && XtIsRealized(g->draw_area)) XClearArea(
		XtDisplay(g->draw_area), XtWindow(g->draw_area),
		0, 0, 0, 0, True);
	g->redraw = True;
}





static void AddDataPoint(g, pt)
Graph	*g;
GDatum	*pt;
{
int	l = 0,
	u = 0,
	m = 0;


				/* Search for existing data bucket	*/
	if (g->data != NULL) {
	    l = 0;		/* Do binary search			*/
	    u = g->num_data;
	    while (l < u) {
		m = (l + u) / 2;
		if (g->data[m].x < pt->x) l = m + 1;
		else if (g->data[m].x > pt->x) u = m;
		else {
		    u = m;
		    l = u + 1;
		    break;
		} /* else */
	    } /* while */
	} /* if */
				/* Count datum				*/
	if (l == u + 1) g->data[m].cnt += pt->cnt;
	else {			/* Insert new bucket			*/
	    g->num_data++;	/* Resize the array if necessary	*/
	    if (g->num_data > g->data_sz) {
		g->data_sz += 100;
		if (g->data != NULL) g->data = (GDatum *)XtRealloc(
			(char *)g->data, sizeof(GDatum) * g->data_sz);
		else g->data = (GDatum *)XtMalloc(sizeof(GDatum) * g->data_sz);
	    } /* if */

				/* Make room for new bucket		*/
	    if (m < u) m++;
	    if (m < g->num_data) bcopy(&(g->data[m]), &(g->data[m+1]),
			sizeof(GDatum) * (g->num_data - m - 1));

				/* Add datum to new bucket		*/
	    g->data[m].x = pt->x;
	    g->data[m].cnt = pt->cnt;
	} /* else */
} /* function AddDataPoint */

/*****************************************************************************
 *
 *	Public routines
 *
 ****************************************************************************/

#ifdef	_NO_PROTO
void GraphSet(g)
Graph	*g;
#else
void GraphSet(Graph *g)
#endif	_NO_PROTO
{
int	n;
Arg	arg[5];

				/* Set axis labels			*/
	n = 0;
	XtSetArg(arg[n], XmNlabelString, g->label_x);			n++;
	XtSetValues(g->x_lbl, arg, n);

	n = 0;
	XtSetArg(arg[n], XmNlabelString, g->label_y);			n++;
	XtSetValues(g->y_lbl, arg, n);
				/* Redraw graph w/new info		*/
	SetRedraw(g);
} /* function GraphSet */





#ifdef	_NO_PROTO
void GraphSetData(g, data, num_data)
Graph	*g;
GDatum	*data;
unsigned num_data;
#else
void GraphSetData(Graph *g, GDatum *data, unsigned num_data)
#endif	_NO_PROTO
{
GDatum	*p;
int	i;
				/* Reset data array			*/
	g->num_data = 0;
				/* Count up the data points		*/
	for (p=data, i=0; i < num_data; p++, i++)
		AddDataPoint(g, p);
				/* Redraw with new data			*/
	SetRedraw(g);
} /* function GraphSetData */





#ifdef	_NO_PROTO
void GraphAddPoint(g, val)
Graph	*g;
long	val;
#else
void GraphAddPoint(Graph *g, long val)
#endif	_NO_PROTO
{
	AddDataPoint(g, val);
				/* Do we need to redisplay?		*/
	SetRedraw(g);
} /* function GraphAddPoint */





#ifdef	_NO_PROTO
void GraphDeleteData(g)
Graph	*g;
#else
void GraphDeleteData(Graph *g)
#endif	_NO_PROTO
{
				/* Zero number of data items		*/
	g->num_data = 0;
				/* Redraw graph window			*/
	SetRedraw(g);
} /* function GraphDeleteData */

/*************************** End of xpi_graph.c ******************************/
