/******************************************************************************
 * Module xpi_TLroundtrip.c - Round-trip tool for xperfisis
 *
 * Written:	 06/06/91 by John H. Lee
 * Last Changed: 06/10/91 by John H. Lee
 * Version:	 0.2
 ******************************************************************************
 * Description:
 *
 *	This tool performs a basic broadcast with reply and determines how
 * long it takes.  The samples are displayed on a histogram.
 ******************************************************************************
 * Exports:
 *	ToolClass roundTripToolClass	Tool class record pointer
 *
 * Imports:
 * 	xpi_graph routines
 *	xpi_util routines
 ******************************************************************************
 * Revisions:
 *   0.1   06/06/91 JHL  Initial Creation
 *   0.2   06/10/91 JHL  Cleaned up internal structure
 *****************************************************************************/

#include <isis.h>
#include <X11/Intrinsic.h>
#include <X11/IntrinsicP.h>
#include <Xm/Xm.h>
#include <Xm/CascadeB.h>
#include <Xm/DrawingA.h>
#include <Xm/Form.h>
#include <Xm/LabelG.h>
#include <Xm/MainW.h>
#include <Xm/MenuShell.h>
#include <Xm/MessageB.h>
#include <Xm/PushB.h>
#include <Xm/PushBG.h>
#include <Xm/RowColumn.h>
#include <Xm/ScrollBar.h>

#include "xperfisis.h"
#include "xpi_TLroundtrip.h"
#include "xpi_graph.h"




/*** Private Global Variable Definitions ***/

extern int	abcast_l(),	/* Why aren't these extern defs in isis.h??? */
		cbcast_l(),
		gbcast_l(),
		fbcast_l(),
		mbcast_l();

#define	BCAST_ABCAST	0
#define	BCAST_CBCAST	1
#define	BCAST_GBCAST	2
#define	BCAST_FBCAST	3
#define	BCAST_MBCAST	4
static struct {
	char	*name;
	int	(*bcast_func)();
} BcastList[] = {		/* List of broadcast methods		*/
	{"ABcast", abcast_l},
	{"CBcast", cbcast_l}, 
	{"GBcast", gbcast_l}, 
	{"FBcast", fbcast_l}, 
	{"MBcast", mbcast_l}, 
	{NULL, NULL}
};

static XmString	SeparatorStr,	/* Compound string separator		*/
		EmptyStr,	/* Empty compound string		*/
		OKStr,		/* "OK" string				*/
		StopStr,	/* "Stop" string			*/
		BusyTitleStr,		/* Busy--working dialog title	*/
		GranularityTitleStr,	/* Granularity dialog title	*/
		GranularityStr,	/* Granularity dialog message		*/
		NumSamplesTitleStr,	/* Num of samples dialog title	*/
		NumSamplesStr,	/* Num of samples dialog message	*/
		NoServersTitleStr,	/* No servers dialog title	*/
		NoServersStr,	/* No servers selected dialog message	*/
		PacketSizeTitleStr,	/* Packet Size dialog title	*/
		PacketSizeStr;	/* Packet Size dialog messag		*/

/*** Tool Class Functions ***/

static void	ClassInit(),
		Init(),
		Destroy();




/*** Tool Instance Record ***/

struct _RoundTripToolRec_Struct {
	ToolCommonPart	common;
	Widget		draw_area,
			ginfo_lbl,
			tinfo_lbl,
			menu_panes[6],
			menu_cscs[6],
			working_dlg;
	EvDialog	ev_dlg;
	LiDialog	li_dlg;
	Graph		*graph;
	GDatum		*data;
	unsigned	data_sz,
			num_data;
	unsigned	num_test;
	int		bcast_type,
			granularity,
			packet_size;
	char		*packet_data;
	int		packet_sz;
	address		target_servers[PG_MAXMEMB+1],
			*site_list[PG_MAXMEMB+1];
	Bool		run_test;
};
typedef struct _RoundTripToolRec_Struct	RoundTripToolRec;



/*** Tool Class Record ***/

ToolClassRec	roundTripToolClassRec = {
    /* tool_title		*/ "Round-Trip--Broadcast+Reply Message Times",
    /* menu_name		*/ "RoundTrip",
    /* class_init		*/ ClassInit,
    /* init			*/ Init,
    /* destroy			*/ Destroy,
    /* instance_size		*/ sizeof(RoundTripToolRec),
    /* class_initialized	*/ False
};

ToolClassRec	*roundTripToolClass = &roundTripToolClassRec;

/*** Private Functions ***/

static XmString	CreateToolInfo();
static void	SetToolInfo();
static void	StartCB(),
		StopCB(),
		SetBroadcastTypeCB(),
		SetNumSamplesCB(),
		SetPacketSizeCB(),
		SetGranularityCB(),
		SetTargetServersCB(),
		SetTargetServersChangeCB(),
		SetTargetServersConfirmCB();

/******************************************************************************
 *
 *	Class Initialization
 *
 *		This procedure is called exactly once for this class before
 *	any instance of this class is created.
 *
 *****************************************************************************/

static void ClassInit()
{
XmString	msg_str1,
		msg_str2,
		msg_str3;


	/*** Create static compound strings ***/
				/* Compound String separator		*/
	SeparatorStr = XmStringSeparatorCreate();

				/* Empty compound string		*/
	EmptyStr = XmStringCreateSimple("");

				/* "OK" string				*/
	OKStr = XmStringCreateSimple("OK");

				/* "Stop" string			*/
	StopStr = XmStringCreateSimple("Stop");

				/* Busy--working title message		*/
	BusyTitleStr = XmStringCreateSimple("Test Running");

				/* Granularity title message		*/
	GranularityTitleStr = XmStringCreateSimple("Set Granularity");
				/* Granularity dialog message		*/
	msg_str1 = XmStringCreateSimple(
		"Please set desired granularity of samples.");
	msg_str2 = XmStringConcat(msg_str1, SeparatorStr);
	XmStringFree(msg_str1);
	msg_str1 = XmStringCreateSimple("");
	msg_str3 = XmStringConcat(msg_str2, msg_str1);
	XmStringFree(msg_str1);
	XmStringFree(msg_str2);
	msg_str1 = XmStringConcat(msg_str3, SeparatorStr);
	XmStringFree(msg_str3);
	msg_str2 = XmStringCreateSimple("Broadcasts per sample:");
	GranularityStr = XmStringConcat(msg_str1, msg_str2);
	XmStringFree(msg_str1);
	XmStringFree(msg_str2);

				/* No Servers selected title message	*/
	NoServersTitleStr = XmStringCreateSimple("No Servers Alert");
				/* No Servers selected dialog message	*/
	NoServersStr = XmStringCreateSimple(
		"At least one target server must be selected.");

				/* Number of Samples title message	*/
	NumSamplesTitleStr = XmStringCreateSimple("Set Number of Samples");
				/* Number of Samples dialog message	*/
	msg_str1 = XmStringCreateSimple(
		"Please set desired number of samples to take.");
	msg_str2 = XmStringConcat(msg_str1, SeparatorStr);
	XmStringFree(msg_str1);
	msg_str1 = XmStringCreateSimple("");
	msg_str3 = XmStringConcat(msg_str2, msg_str1);
	XmStringFree(msg_str1);
	XmStringFree(msg_str2);
	msg_str1 = XmStringConcat(msg_str3, SeparatorStr);
	XmStringFree(msg_str3);
	msg_str2 = XmStringCreateSimple("Number of Samples:");
	NumSamplesStr = XmStringConcat(msg_str1, msg_str2);
	XmStringFree(msg_str1);
	XmStringFree(msg_str2);

				/* Packet Size title message		*/
	PacketSizeTitleStr = XmStringCreateSimple("Set PacketSize");
				/* Packet Size dialog message		*/
	msg_str1 = XmStringCreateSimple(
		"Please set desired packet size of broadcasts.");
	msg_str2 = XmStringConcat(msg_str1, SeparatorStr);
	XmStringFree(msg_str1);
	msg_str1 = XmStringCreateSimple("");
	msg_str3 = XmStringConcat(msg_str2, msg_str1);
	XmStringFree(msg_str1);
	XmStringFree(msg_str2);
	msg_str1 = XmStringConcat(msg_str3, SeparatorStr);
	XmStringFree(msg_str3);
	msg_str2 = XmStringCreateSimple("Bytes per broadcast:");
	PacketSizeStr = XmStringConcat(msg_str1, msg_str2);
	XmStringFree(msg_str1);
	XmStringFree(msg_str2);
} /* function ClassInit */

/******************************************************************************
 *
 *	Initialization
 *
 *		This procedure is called to initialized a new instance of
 *	this tool class.  It is called exactly once per new instance by
 *	CreateTool().
 *
 *****************************************************************************/

static void Init(t)
Tool	t;
{
AppWindow	*app_win = t->common.app_win;
RoundTripTool	rt = (RoundTripTool)t;
Widget		menu_pane,
		menu_ent,
		submenu_pane,
		button;
XmString	cancel_str,
		msg_str,
		title_str,
		header_str;
int		i,
		mp,
		mc;
Arg		args[16];
Cardinal	n;


	/*** Initalize instance data structure ***/

	rt->data = NULL;
	rt->data_sz = 0;
	rt->num_data = 0;
	rt->num_test = 1000;
	rt->bcast_type = BCAST_ABCAST;
	rt->granularity = 1;
	rt->packet_size = 1;
	rt->run_test = False;
	rt->packet_data = NULL;
	rt->packet_sz = 0;
	memcpy(rt->target_servers, &NULLADDRESS, sizeof(address));


	mp = 0;			/* Initialize number of menu panes	*/
	mc = 0;			/* Initialize number of menu csc buttons */


	/*** Add "Parameters" menu to menu bar ***/

	n = 0;
	menu_pane = XmCreatePulldownMenu(app_win->menu_bar,
		"menu_pane", args, n);
	rt->menu_panes[mp++] = menu_pane;

				/*** Broadcast Type submenu 		***/
	n = 0;
	submenu_pane = XmCreatePulldownMenu(menu_pane,
		"submenu_pane", args, n);
	rt->menu_panes[mp++] = submenu_pane;

				/* Broadcast submenu menu items		*/
	for (i=0; BcastList[i].name != NULL; i++) {
	    n = 0;
	    XtSetArg(args[n], XmNuserData, i);				n++;
	    menu_ent = XmCreatePushButton(submenu_pane,
		    BcastList[i].name, args, n);
	    XtManageChild(menu_ent);
	    XtAddCallback(menu_ent, XmNactivateCallback, SetBroadcastTypeCB,
		    (caddr_t)t);
	} /* for */

	n = 0;
	XtSetArg(args[n], XmNsubMenuId, submenu_pane);			n++;
	rt->menu_cscs[mc] = XmCreateCascadeButton(menu_pane,
		"Broadcast Type", args, n);
	XtManageChild(rt->menu_cscs[mc]);
	mc++;

				/* Number of Samples menu item		*/
	n = 0;
	menu_ent = XmCreatePushButton(menu_pane, "Num Samples...", args, n);
	XtManageChild(menu_ent);
	XtAddCallback(menu_ent, XmNactivateCallback, SetNumSamplesCB,
		(caddr_t)t);

				/* Packet Size menu item		*/
	n = 0;
	menu_ent = XmCreatePushButton(menu_pane, "Packet Size...", args, n);
	XtManageChild(menu_ent);
	XtAddCallback(menu_ent, XmNactivateCallback, SetPacketSizeCB,
		(caddr_t)t);
				/* Granularity menu item		*/
	n = 0;
	menu_ent = XmCreatePushButton(menu_pane, "Granularity...", args, n);
	XtManageChild(menu_ent);
	XtAddCallback(menu_ent, XmNactivateCallback, SetGranularityCB,
		(caddr_t)t);
				/* Target Servers menu item		*/
	n = 0;
	menu_ent = XmCreatePushButton(menu_pane, "Target Servers...", args, n);
	XtManageChild(menu_ent);
	XtAddCallback(menu_ent, XmNactivateCallback, SetTargetServersCB,
		(caddr_t)t);

	n = 0;
	XtSetArg(args[n], XmNsubMenuId, menu_pane);		n++;
	rt->menu_cscs[mc] = XmCreateCascadeButton(app_win->menu_bar,
		"Parameters", args, n);
	XtManageChild(rt->menu_cscs[mc]);
	mc++;


	/*** Add "Test" menu to menu bar ***/

	n = 0;
	rt->menu_panes[mp] = XmCreatePulldownMenu(app_win->menu_bar,
		"menu_pane", args, n);
				/* Start test menu item			*/
	menu_ent = XmCreatePushButton(rt->menu_panes[mp], "Start", args, n);
	XtManageChild(menu_ent);
	XtAddCallback(menu_ent, XmNactivateCallback, StartCB, (caddr_t)t);

	n = 0;
	XtSetArg(args[n], XmNsubMenuId, rt->menu_panes[mp]);		n++;
	rt->menu_cscs[mc] = XmCreateCascadeButton(app_win->menu_bar,
		"Test", args, n);
	XtManageChild(rt->menu_cscs[mc]);
	mc++;


				/* End list of menus 			*/
	rt->menu_panes[mp] = NULL;
	rt->menu_cscs[mc] = NULL;


	/*** Create tool info field ***/

	msg_str = CreateToolInfo(rt);

	n = 0;
	XtSetArg(args[n], XmNbottomAttachment, XmATTACH_FORM);		n++;
	XtSetArg(args[n], XmNleftAttachment, XmATTACH_FORM);		n++;
	XtSetArg(args[n], XmNrightAttachment, XmATTACH_FORM);		n++;
	XtSetArg(args[n], XmNtopAttachment, XmATTACH_NONE);		n++;
	XtSetArg(args[n], XmNresizable, False);				n++;
	XtSetArg(args[n], XmNlabelString, msg_str);			n++;
	XtSetArg(args[n], XmNalignment, XmALIGNMENT_BEGINNING);		n++;
	rt->tinfo_lbl = XmCreateLabel(app_win->tool_form,
		"info", args, n);
	XtManageChild(rt->tinfo_lbl);
	XmStringFree(msg_str);


	/*** Create graph info field ***/

	msg_str = XmStringCreateSimple("No samples displayed.");

	n = 0;
	XtSetArg(args[n], XmNlabelString, msg_str);			n++;
	XtSetArg(args[n], XmNbottomWidget, rt->tinfo_lbl);		n++;
	XtSetArg(args[n], XmNbottomAttachment, XmATTACH_WIDGET);	n++;
	XtSetArg(args[n], XmNleftAttachment, XmATTACH_FORM);		n++;
	XtSetArg(args[n], XmNrightAttachment, XmATTACH_FORM);		n++;
	XtSetArg(args[n], XmNtopAttachment, XmATTACH_NONE);		n++;
	XtSetArg(args[n], XmNresizable, False);				n++;
	rt->ginfo_lbl = XmCreateLabel(app_win->tool_form,
		"info", args, n);
	XtManageChild(rt->ginfo_lbl);
	XmStringFree(msg_str);


	/*** Create "I'm Busy" dialog ***/

	cancel_str = XmStringCreateSimple("Stop");
	title_str = XmStringCreateSimple("Test Running");

	n = 0;
	XtSetArg(args[n], XmNcancelLabelString, cancel_str);		n++;
	XtSetArg(args[n], XmNdialogTitle, title_str);			n++;
	XtSetArg(args[n], XmNdialogStyle, XmDIALOG_FULL_APPLICATION_MODAL);
	XtSetArg(args[n], XmNdefaultButtonType, XmDIALOG_CANCEL_BUTTON); n++;
	rt->working_dlg = XmCreateWorkingDialog(app_win->main_win, "busy",
		args, n);
	XtAddCallback(rt->working_dlg, XmNcancelCallback, StopCB, (caddr_t)t);
	XmStringFree(title_str);
	XmStringFree(cancel_str);
				/* Remove OK & Help buttons		*/
	button = XmMessageBoxGetChild(rt->working_dlg, XmDIALOG_OK_BUTTON);
	XtUnmanageChild(button);
	button = XmMessageBoxGetChild(rt->working_dlg, XmDIALOG_HELP_BUTTON);
	XtUnmanageChild(button);


	/*** Create Set Granularity/Packet Size dialog ***/

	rt->ev_dlg = CreateEvaluatorDialog(app_win->main_win);


	/*** Create Set Target Servers dialog ***/

	msg_str = XmStringCreateSimple(
		"Please select the desired target servers.");
	title_str = XmStringCreateSimple("Set Target Servers");
	header_str = XmStringCreateSimple("Serv-PID  Site");

	rt->li_dlg = CreateListDialog(app_win->main_win);
	SetListDialog(rt->li_dlg, app_win->shell, title_str,
		msg_str, header_str, NULL);

	XmStringFree(header_str);
	XmStringFree(title_str);
	XmStringFree(msg_str);
				/* Customize list box			*/
	n = 0;
	XtSetArg(args[n], XmNselectionPolicy, XmEXTENDED_SELECT);	n++;
	XtSetValues(rt->li_dlg->list, args, n);
				/* Set callbacks			*/
	XtAddCallback(rt->li_dlg->list, XmNextendedSelectionCallback,
		SetTargetServersChangeCB, (caddr_t)rt);
	XtAddCallback(rt->li_dlg->ok_btn, XmNactivateCallback,
		SetTargetServersConfirmCB, (caddr_t)rt);


	/*** Create graph display ***/
	n = 0;
	XtSetArg(args[n], XmNwidth, 400);				n++;
	XtSetArg(args[n], XmNheight, 200);				n++;
	XtSetArg(args[n], XmNbottomWidget, rt->ginfo_lbl);		n++;
	XtSetArg(args[n], XmNbottomAttachment, XmATTACH_WIDGET);	n++;
	XtSetArg(args[n], XmNleftAttachment, XmATTACH_FORM);		n++;
	XtSetArg(args[n], XmNrightAttachment, XmATTACH_FORM);		n++;
	XtSetArg(args[n], XmNtopAttachment, XmATTACH_FORM);		n++;
	XtSetArg(args[n], XmNresizePolicy, XmRESIZE_NONE);		n++;
	XtSetArg(args[n], XmNmarginWidth, 0);				n++;
	XtSetArg(args[n], XmNmarginHeight, 0);				n++;
	XtSetArg(args[n], XmNresizable, False);				n++;
	rt->draw_area = XmCreateDrawingArea(app_win->tool_form,
		"graph", args, n);

				/* Create simple graph			*/
	rt->graph = GraphCreate(rt->draw_area, NULL);
	rt->graph->radix_x = 1000;
	rt->graph->radix_y = 10;
	rt->graph->auto_scale = True;
	rt->graph->label_x = XmStringCreateSimple("Time (ms)");
	rt->graph->label_y = XmStringCreateSimple("Number of Broadcasts");
	GraphSet(rt->graph);

	XtManageChild(rt->draw_area);
} /* function Init */

/******************************************************************************
 *
 *	Destroy
 *
 *		This procedure is called once a tool instance is no longer
 *	needed.  It is called at most once per tool instance through
 *	DestroyTool().  All tool instances are destroyed before the the
 *	tool is exited normally.
 *		This procedure must destroy any widgets created for a tool
 *	instance that will not be automatically destroyed when the tool
 *	form is destroyed by DestroyTool(), and return any other resources
 *	allocated.  The tool instance record is automatically freed by
 *	DestroyTool().
 *
 *****************************************************************************/

static void Destroy(t)
Tool	t;
{
RoundTripTool	rt = (RoundTripTool)t;
int		m;


				/* Destroy graph			*/
	GraphDestroy(rt->graph);
	XtDestroyWidget(rt->draw_area);
				/* Destroy dialogs			*/
	XtDestroyWidget(rt->working_dlg);
	XtDestroyWidget(rt->ev_dlg->dialog);
	XtFree(rt->ev_dlg);
				/* Destroy our added menus		*/
	for (m=0; rt->menu_cscs[m] != NULL; m++)
		XtDestroyWidget(rt->menu_cscs[m]);
	for (m=0; rt->menu_panes[m] != NULL; m++)
		XtDestroyWidget(rt->menu_panes[m]);
				/* Free allocated arrays		*/
	XtFree(rt->data);
	XtFree(rt->packet_data);
} /*  function Destroy */

/******************************************************************************
 *
 *	Private Procedures
 *
 *****************************************************************************/

static XmString CreateToolInfo(rt)
RoundTripTool	rt;
{
XmString	tinfo_str,
		tinfo_str1,
		tinfo_str2;
char		tinfo_buf[256],
		*t;
address		*s;


				/* Broadcast type			*/
	strcpy(tinfo_buf, "Broadcast Type:  ");
	strcat(tinfo_buf, BcastList[rt->bcast_type].name);
	tinfo_str1 = XmStringCreateSimple(tinfo_buf);
	tinfo_str = XmStringConcat(tinfo_str1, SeparatorStr);
	XmStringFree(tinfo_str1);
				/* Number of samples			*/
	(void)sprintf(tinfo_buf, "Samples to be taken:  %d",
		rt->num_test);
	tinfo_str1 = XmStringCreateSimple(tinfo_buf);
	tinfo_str2 = XmStringConcat(tinfo_str, SeparatorStr);
	XmStringFree(tinfo_str);
	tinfo_str = XmStringConcat(tinfo_str2, tinfo_str1);
	XmStringFree(tinfo_str1);
	XmStringFree(tinfo_str2);
				/* Sample granularity			*/
	(void)sprintf(tinfo_buf, "Granularity (bcast/sample):  %d",
		rt->granularity);
	tinfo_str1 = XmStringCreateSimple(tinfo_buf);
	tinfo_str2 = XmStringConcat(tinfo_str, SeparatorStr);
	XmStringFree(tinfo_str);
	tinfo_str = XmStringConcat(tinfo_str2, tinfo_str1);
	XmStringFree(tinfo_str1);
	XmStringFree(tinfo_str2);
				/* Packet size				*/
	(void)sprintf(tinfo_buf, "Packet size (bytes/bcast):  %d",
		rt->packet_size);
	tinfo_str1 = XmStringCreateSimple(tinfo_buf);
	tinfo_str2 = XmStringConcat(tinfo_str, SeparatorStr);
	XmStringFree(tinfo_str);
	tinfo_str = XmStringConcat(tinfo_str2, tinfo_str1);
	XmStringFree(tinfo_str1);
	XmStringFree(tinfo_str2);
				/* Target servers			*/
	strcpy(tinfo_buf, "Target Servers: ");
	t = tinfo_buf + strlen(tinfo_buf);
	if (!addr_isnull(rt->target_servers)) {
	    for (s=rt->target_servers; !addr_isnull(s); s++) {
		(void)sprintf(t, "%d@%d", s->addr_process, s->addr_site);
		t += strlen(t);
		if (!addr_isnull(s+1)) {
		    *t++ = ',';
		    *t++ = ' ';
		} /* if */
	    } /* for */
	} else strcat(tinfo_buf, "(None)");
	tinfo_str1 = XmStringCreateSimple(tinfo_buf);
	tinfo_str2 = XmStringConcat(tinfo_str, SeparatorStr);
	XmStringFree(tinfo_str);
	tinfo_str = XmStringConcat(tinfo_str2, tinfo_str1);
	XmStringFree(tinfo_str1);
	XmStringFree(tinfo_str2);


	return (tinfo_str);
} /* function SetToolInfo */





static void SetToolInfo(rt)
RoundTripTool	rt;
{
XmString	tinfo_str;
Arg		args[5];
int		n;


	tinfo_str = CreateToolInfo(rt);

	n = 0;
	XtSetArg(args[n], XmNlabelString, tinfo_str);			n++;
	XtSetValues(rt->tinfo_lbl, args, n);
	XmStringFree(tinfo_str);
} /* function SetToolInfo */

/******************************************************************************
 *
 *	Test Working Procedure
 *
 *****************************************************************************/

static Boolean	TestWP(client_data)
XtPointer	client_data;
{
RoundTripTool	rt = (RoundTripTool)client_data;
struct timeval	t_start,
		t_end;
long		elapsed;
address		*s;
char		info_buf[64];
XmString	info_str;
Arg		args[5];
int		n,
		i;


	if (rt->run_test) {	/* If the test is still running...	*/
				/* Set entry point			*/
	    for (s=rt->target_servers; !addr_isnull(s); s++)
		    s->addr_entry = ENTRY_JUSTREPLY;

				/* Get start time			*/
	    (void)gettimeofday(&t_start, NULL);

				/* Do broadcast				*/
	    for (i=rt->granularity; i > 0; i--) {
		(*BcastList[rt->bcast_type].bcast_func)
			("l", rt->target_servers, "%C", rt->packet_data,
			rt->packet_size,
			1, "");
	    } /* for */

				/* Get end time			*/
	    (void)gettimeofday(&t_end, NULL);
				/* Calculate elapsed time		*/
	    elapsed = (t_end.tv_sec * 1000000 + t_end.tv_usec)
		    - (t_start.tv_sec * 1000000 + t_start.tv_usec);
				/* Store average time per broadcast	*/
				/*   (simple enumeration format)	*/
	    rt->data[rt->num_data].x = elapsed / rt->granularity;
	    rt->data[rt->num_data].cnt = rt->graph->radix_y;
	    rt->num_data++;

				/* Update "busy" dialog			*/
	    if ((rt->num_data & 0x3f) == 0) {
		(void)sprintf(info_buf, "%5d broadcasts left",
			(rt->num_test - rt->num_data) * rt->granularity);
		info_str = XmStringCreateSimple(info_buf);
		n = 0;
		XtSetArg(args[n], XmNmessageString, info_str);		n++;
		XtSetValues(rt->working_dlg, args, n);
		XmStringFree(info_str);
	    } /* if */

				/* Stored enough data?  Signal end of test */
	    if (rt->num_data == rt->num_test) rt->run_test = False;
	} /* if */

	if (rt->run_test)	/* Not done yet...			*/
		return (False);	/*   ...please don't delete us!		*/

	else {			/* Test complete			*/
				/* Set data to graph			*/
	    GraphSetData(rt->graph, rt->data, rt->num_data);
				/* Update sample info			*/
	    (void)sprintf(info_buf, "%d samples taken", rt->num_data);
	    info_str = XmStringCreateSimple(info_buf);
	    n = 0;
	    XtSetArg(args[n], XmNlabelString, info_str);		n++;
	    XtSetValues(rt->ginfo_lbl, args, n);
	    XmStringFree(info_str);
				/* Popdown "busy" dialog		*/
	    XtUnmanageChild(rt->working_dlg);
				/* We're done, OK to delete		*/
	    return (True);
	} /* else */

} /* function TestWP */

/******************************************************************************
 *
 *	Callback Procedures
 *
 *****************************************************************************/
/******************************************************************************
 *
 *	Start/Stop Test Callback Procedures
 *
 *****************************************************************************/

static void StartCB(w, client_data, call_data)
Widget	w;		/* UNUSED */
caddr_t	client_data;	/* Tool */
caddr_t	call_data;	/* UNUSED */
{
RoundTripTool	rt = (RoundTripTool)client_data;
AppWindow	*app_win = rt->common.app_win;
char		msg_buf[64];
XmString	msg_str;

Arg		args[5];
int		n;


	if (addr_isnull(rt->target_servers)) {
	    SetMessageDialog(app_win->warn_dlg, app_win->shell,
		    NoServersTitleStr, NoServersStr, OKStr, EmptyStr);
	    XtManageChild(app_win->warn_dlg);

	    return;
	} /* if */

				/* Reset data array			*/
	rt->num_data = 0;
				/* Allocate array, if needed		*/
	if (rt->data_sz < rt->num_test) {
	    rt->data_sz = rt->num_test;
	    rt->data = (GDatum *)XtRealloc(rt->data,
		    sizeof(GDatum) * rt->data_sz);
	} /* if */

				/* Allocate packet, if needed		*/
	if (rt->packet_sz < rt->packet_size) {
	    rt->packet_sz = rt->packet_size;
	    rt->packet_data = (char *)XtRealloc(rt->packet_data,
		    rt->packet_sz);
	} /* if */

				/* Put up "busy" message		*/
	(void)sprintf(msg_buf, "%5d broadcasts left",
		rt->num_test * rt->granularity);
	msg_str = XmStringCreateSimple(msg_buf);
	SetMessageDialog(rt->working_dlg, app_win->shell, BusyTitleStr,
		msg_str, EmptyStr, StopStr);
	XtManageChild(rt->working_dlg);
	XmStringFree(msg_str);

				/* Register work procedure		*/
	rt->run_test = True;
	XtAppAddWorkProc(app_win->app_context, TestWP, (XtPointer)rt);
} /* function StartCB */






static void StopCB(w, client_data, call_data)
Widget	w;		/* UNUSED */
caddr_t	client_data;	/* UNUSED */
caddr_t	call_data;	/* UNUSED */
{
RoundTripTool	rt = (RoundTripTool)client_data;


				/* Tell work procedure to stop		*/
	rt->run_test = False;
} /* function StopCB */

/******************************************************************************
 *
 *	Set Broadcast Type Callback Procedure
 *
 *****************************************************************************/

static void SetBroadcastTypeCB(w, client_data, call_data)
Widget	w;		/* UNUSED */
caddr_t	client_data;	/* UNUSED */
caddr_t	call_data;	/* UNUSED */
{
RoundTripTool	rt = (RoundTripTool)client_data;
int		bcast_type;
Arg		args[5];
int		n;


					/* Get broadcast type		*/
	n = 0;
	XtSetArg(args[n], XmNuserData, &bcast_type);			n++;
	XtGetValues(w, args, n);

					/* Set broadcast type		*/
	rt->bcast_type = bcast_type;
					/* Set tool info		*/
	SetToolInfo(rt);
} /* function SetBroadcastTypeCB */

/******************************************************************************
 *
 *	Set Num Samples Callback Procedures
 *
 *****************************************************************************/
static void	SetNumSamplesChangeCB(),
		SetNumSamplesConfirmCB(),
		SetNumSamplesCancelCB();



static void SetNumSamplesCB(w, client_data, call_data)
Widget	w;		/* UNUSED */
caddr_t	client_data;	/* RoundTripTool */
caddr_t	call_data;	/* UNUSED */
{
RoundTripTool	rt = (RoundTripTool)client_data;

static Arg	args[] = {
	{XmNminimum,	(XtArgVal)1},
	{XmNmaximum,	(XtArgVal)(10000 + 100)}
};


				/* Set evalutor dialog			*/
	XtAddCallback(rt->ev_dlg->scroll_bar, XmNdragCallback,
		SetNumSamplesChangeCB, (caddr_t)rt);
	XtAddCallback(rt->ev_dlg->scroll_bar, XmNvalueChangedCallback,
		SetNumSamplesChangeCB, (caddr_t)rt);
	XtAddCallback(rt->ev_dlg->ok_btn, XmNactivateCallback,
		SetNumSamplesConfirmCB, (caddr_t)rt);
	XtAddCallback(rt->ev_dlg->cancel_btn, XmNactivateCallback,
		SetNumSamplesCancelCB, (caddr_t)rt);

				/* Set number of samples slider		*/
	XmScrollBarSetValues(rt->ev_dlg->scroll_bar, 1,
		1, 0, 0, True);
	XtSetValues(rt->ev_dlg->scroll_bar, args, XtNumber(args));
	XmScrollBarSetValues(rt->ev_dlg->scroll_bar, rt->num_test,
		100, 1, 100, True);

	SetEvaluatorDialog(rt->ev_dlg, rt->common.app_win->shell,
		NumSamplesTitleStr, NumSamplesStr, NULL);

				/* Popup number of samples popup	*/
	XtManageChild(rt->ev_dlg->dialog);
} /* function SetNumSamplesCB */





static void SetNumSamplesChangeCB(w, client_data, call_data)
Widget	w;		/* UNUSED */
caddr_t	client_data;	/* RoundTripTool */
caddr_t	call_data;	/* UNUSED */
{
RoundTripTool	rt = (RoundTripTool)client_data;
int		num_samples,
		slider_size,
		increment,
		page_increment;
char		msg_buf[16];
XmString	msg_str;
Arg		arg;


				/* Get packet size slider value		*/
	XmScrollBarGetValues(w, &num_samples, &slider_size,
		&increment, &page_increment);
				/* Display number of samples value	*/
	(void)sprintf(msg_buf, "%5d", num_samples);
	msg_str = XmStringCreateSimple(msg_buf);
	XtSetArg(arg, XmNlabelString, msg_str);
	XtSetValues(rt->ev_dlg->value_lbl, &arg, 1);
	XmStringFree(msg_str);
} /* function SetNumSamplesChangeCB */





static void SetNumSamplesCancelCB(w, client_data, call_data)
Widget	w;		/* UNUSED */
caddr_t	client_data;	/* RoundTripTool */
caddr_t	call_data;	/* UNUSED */
{
RoundTripTool	rt = (RoundTripTool)client_data;


			/* Remove callbacks			*/
	XtRemoveCallback(rt->ev_dlg->scroll_bar, XmNdragCallback,
		SetNumSamplesChangeCB, (caddr_t)rt);
	XtRemoveCallback(rt->ev_dlg->scroll_bar, XmNvalueChangedCallback,
		SetNumSamplesChangeCB, (caddr_t)rt);
	XtRemoveCallback(rt->ev_dlg->ok_btn, XmNactivateCallback,
		SetNumSamplesConfirmCB, (caddr_t)rt);
	XtRemoveCallback(rt->ev_dlg->cancel_btn, XmNactivateCallback,
		SetNumSamplesCancelCB, (caddr_t)rt);
} /* function SetNumSamplesCancelCB */





static void SetNumSamplesConfirmCB(w, client_data, call_data)
Widget	w;		/* UNUSED */
caddr_t	client_data;	/* RoundTripTool */
caddr_t	call_data;	/* UNUSED */
{
RoundTripTool	rt = (RoundTripTool)client_data;
int		packet_size,
		slider_size,
		increment,
		page_increment;


				/* Get packet size slider value		*/
	XmScrollBarGetValues(rt->ev_dlg->scroll_bar, &rt->num_test,
		&slider_size, &increment, &page_increment);
				/* Update tool info display		*/
	SetToolInfo(rt);
				/* Remove callbacks			*/
	SetNumSamplesCancelCB(w, client_data, call_data);
} /* function SetNumSamplesConfirmCB */

/******************************************************************************
 *
 *	Set Packet Size Callback Procedures
 *
 *****************************************************************************/
static void	SetPacketSizeChangeCB(),
		SetPacketSizeConfirmCB(),
		SetPacketSizeCancelCB();



static void SetPacketSizeCB(w, client_data, call_data)
Widget	w;		/* UNUSED */
caddr_t	client_data;	/* RoundTripTool */
caddr_t	call_data;	/* UNUSED */
{
RoundTripTool	rt = (RoundTripTool)client_data;

static Arg	args[] = {
	{XmNminimum,	(XtArgVal)1},
	{XmNmaximum,	(XtArgVal)(4096 + 512)}
};


				/* Set evalutor dialog			*/
	XtAddCallback(rt->ev_dlg->scroll_bar, XmNdragCallback,
		SetPacketSizeChangeCB, (caddr_t)rt);
	XtAddCallback(rt->ev_dlg->scroll_bar, XmNvalueChangedCallback,
		SetPacketSizeChangeCB, (caddr_t)rt);
	XtAddCallback(rt->ev_dlg->ok_btn, XmNactivateCallback,
		SetPacketSizeConfirmCB, (caddr_t)rt);
	XtAddCallback(rt->ev_dlg->cancel_btn, XmNactivateCallback,
		SetPacketSizeCancelCB, (caddr_t)rt);

				/* Set packet size slider 		*/
	XmScrollBarSetValues(rt->ev_dlg->scroll_bar, 1,
		1, 0, 0, True);
	XtSetValues(rt->ev_dlg->scroll_bar, args, XtNumber(args));
	XmScrollBarSetValues(rt->ev_dlg->scroll_bar, rt->packet_size,
		512, 1, 512, True);

	SetEvaluatorDialog(rt->ev_dlg, rt->common.app_win->shell,
		PacketSizeTitleStr, PacketSizeStr, NULL);

				/* Popup packet size popup		*/
	XtManageChild(rt->ev_dlg->dialog);
} /* function SetPacketSizeCB */





static void SetPacketSizeChangeCB(w, client_data, call_data)
Widget	w;		/* UNUSED */
caddr_t	client_data;	/* RoundTripTool */
caddr_t	call_data;	/* UNUSED */
{
RoundTripTool	rt = (RoundTripTool)client_data;
int		packet_size,
		slider_size,
		increment,
		page_increment;
char		msg_buf[16];
XmString	msg_str;
Arg		arg;


				/* Get packet size slider value		*/
	XmScrollBarGetValues(w, &packet_size, &slider_size,
		&increment, &page_increment);
				/* Display new packet size value	*/
	(void)sprintf(msg_buf, "%4d", packet_size);
	msg_str = XmStringCreateSimple(msg_buf);
	XtSetArg(arg, XmNlabelString, msg_str);
	XtSetValues(rt->ev_dlg->value_lbl, &arg, 1);
	XmStringFree(msg_str);
} /* function SetPacketSizeChangeCB */





static void SetPacketSizeCancelCB(w, client_data, call_data)
Widget	w;		/* UNUSED */
caddr_t	client_data;	/* RoundTripTool */
caddr_t	call_data;	/* UNUSED */
{
RoundTripTool	rt = (RoundTripTool)client_data;

				/* Remove callbacks			*/
	XtRemoveCallback(rt->ev_dlg->scroll_bar, XmNdragCallback,
		SetPacketSizeChangeCB, (caddr_t)rt);
	XtRemoveCallback(rt->ev_dlg->scroll_bar, XmNvalueChangedCallback,
		SetPacketSizeChangeCB, (caddr_t)rt);
	XtRemoveCallback(rt->ev_dlg->ok_btn, XmNactivateCallback,
		SetPacketSizeConfirmCB, (caddr_t)rt);
	XtRemoveCallback(rt->ev_dlg->cancel_btn, XmNactivateCallback,
		SetPacketSizeCancelCB, (caddr_t)rt);
} /* function SetPacketSizeCancelCB */





static void SetPacketSizeConfirmCB(w, client_data, call_data)
Widget	w;		/* UNUSED */
caddr_t	client_data;	/* RoundTripTool */
caddr_t	call_data;	/* UNUSED */
{
RoundTripTool	rt = (RoundTripTool)client_data;
int		packet_size,
		slider_size,
		increment,
		page_increment;


				/* Get packet size slider value		*/
	XmScrollBarGetValues(rt->ev_dlg->scroll_bar, &rt->packet_size,
		&slider_size, &increment, &page_increment);
				/* Update tool info display		*/
	SetToolInfo(rt);
				/* Remove callbacks			*/
	SetPacketSizeCancelCB(w, client_data, call_data);
} /* function SetPacketSizeConfirmCB */

/******************************************************************************
 *
 *	Set Sample Granularity Callback Procedures
 *
 *****************************************************************************/
static void	GranularityChangeCB(),
		GranularityConfirmCB(),
		GranularityCancelCB();





static void SetGranularityCB(w, client_data, call_data)
Widget	w;		/* UNUSED */
caddr_t	client_data;	/* RoundTripTool */
caddr_t	call_data;	/* UNUSED */
{
RoundTripTool	rt = (RoundTripTool)client_data;

static Arg	args[] = {
	{XmNminimum,	(XtArgVal)1},
	{XmNmaximum,	(XtArgVal)(25 + 5)}
};


				/* Set evalutor dialog			*/
	XtAddCallback(rt->ev_dlg->scroll_bar, XmNdragCallback,
		GranularityChangeCB, (caddr_t)rt);
	XtAddCallback(rt->ev_dlg->scroll_bar, XmNvalueChangedCallback,
		GranularityChangeCB, (caddr_t)rt);
	XtAddCallback(rt->ev_dlg->ok_btn, XmNactivateCallback,
		GranularityConfirmCB, (caddr_t)rt);
	XtAddCallback(rt->ev_dlg->cancel_btn, XmNactivateCallback,
		GranularityCancelCB, (caddr_t)rt);

				/* Set granularity slider 		*/
	XmScrollBarSetValues(rt->ev_dlg->scroll_bar, 1,
		1, 0, 0, True);
	XtSetValues(rt->ev_dlg->scroll_bar, args, XtNumber(args));
	XmScrollBarSetValues(rt->ev_dlg->scroll_bar, rt->granularity,
		5, 1, 5, True);

	SetEvaluatorDialog(rt->ev_dlg, rt->common.app_win->shell,
		GranularityTitleStr, GranularityStr, NULL);

				/* Popup granularity popup		*/
	XtManageChild(rt->ev_dlg->dialog);
} /* function SetGranularityCB */





static void GranularityChangeCB(w, client_data, call_data)
Widget	w;		/* UNUSED */
caddr_t	client_data;	/* RoundTripTool */
caddr_t	call_data;	/* UNUSED */
{
RoundTripTool	rt = (RoundTripTool)client_data;
int		granularity,
		slider_size,
		increment,
		page_increment;
char		msg_buf[16];
XmString	msg_str;
Arg		arg;


				/* Get granularity slider value		*/
	XmScrollBarGetValues(w, &granularity, &slider_size,
		&increment, &page_increment);
				/* Display new granularity value	*/
	(void)sprintf(msg_buf, "%2d", granularity);
	msg_str = XmStringCreateSimple(msg_buf);
	XtSetArg(arg, XmNlabelString, msg_str);
	XtSetValues(rt->ev_dlg->value_lbl, &arg, 1);
	XmStringFree(msg_str);
} /* function GranularityChangeCB */





static void GranularityCancelCB(w, client_data, call_data)
Widget	w;		/* UNUSED */
caddr_t	client_data;	/* RoundTripTool */
caddr_t	call_data;	/* UNUSED */
{
RoundTripTool	rt = (RoundTripTool)client_data;


				/* Remove callbacks			*/
	XtRemoveCallback(rt->ev_dlg->scroll_bar, XmNdragCallback,
		GranularityChangeCB, (caddr_t)rt);
	XtRemoveCallback(rt->ev_dlg->scroll_bar, XmNvalueChangedCallback,
		GranularityChangeCB, (caddr_t)rt);
	XtRemoveCallback(rt->ev_dlg->ok_btn, XmNactivateCallback,
		GranularityConfirmCB, (caddr_t)rt);
	XtRemoveCallback(rt->ev_dlg->cancel_btn, XmNactivateCallback,
		GranularityCancelCB, (caddr_t)rt);
} /* function GranularityCancelCB */





static void GranularityConfirmCB(w, client_data, call_data)
Widget	w;		/* UNUSED */
caddr_t	client_data;	/* RoundTripTool */
caddr_t	call_data;	/* UNUSED */
{
RoundTripTool	rt = (RoundTripTool)client_data;
int		granularity,
		slider_size,
		increment,
		page_increment;


				/* Get granularity slider value		*/
	XmScrollBarGetValues(rt->ev_dlg->scroll_bar, &rt->granularity,
		&slider_size, &increment, &page_increment);
				/* Update tool info display		*/
	SetToolInfo(rt);
				/* Remove callbacks			*/
	GranularityCancelCB(w, client_data, call_data);
} /* function GranularityConfirmCB */

/******************************************************************************
 *
 *	Set Target Servers Callback Procedures
 *
 *****************************************************************************/

static void SetTargetServersCB(w, client_data, call_data)
Widget	w;		/* UNUSED */
caddr_t	client_data;	/* RoundTripTool */
caddr_t	call_data;	/* UNUSED */
{
RoundTripTool	rt = (RoundTripTool)client_data;
AppWindow	*app_win = rt->common.app_win;
int		i;
groupview	*gv;
char		str_buf[128];
XmString	value_str,
		site_array[PG_MAXMEMB],
		site_sel_array[PG_MAXMEMB];
int		num_sel = 0;
address		*s;
Arg		args[5];
int		n;


				/* Get current group view		*/
	gv = pg_getview(app_win->gaddr_p);
				/* Create arrays of sites		*/
	for (i=0; i < gv->gv_nmemb; i++) {
	    (void)sprintf(str_buf, "%8d  (%3d)%s",
		    gv->gv_members[i].addr_process,
		    gv->gv_members[i].addr_site,
		    site_names[gv->gv_members[i].addr_site]);
	    site_array[i] = XmStringCreateSimple(str_buf);
	    rt->site_list[i] = &gv->gv_members[i];

				/* Currently selected?			*/
	    for (s=rt->target_servers; !addr_isnull(s); s++) {
		if (addr_isequal(s, &gv->gv_members[i])) {
		       site_sel_array[num_sel++] = site_array[i];
		       break;
		} /* if */
	    } /* for */

	} /* for */

				/* Set dialog's list			*/
	n = 0;
	XtSetArg(args[n], XmNitems, site_array);			n++;
	XtSetArg(args[n], XmNitemCount, gv->gv_nmemb);			n++;
	XtSetArg(args[n], XmNselectedItemCount, num_sel);		n++;
				/* This is a bug in Motif v1.1 (well,	*/
	if (num_sel > 0) {	/*   *I* think it's a bug...)		*/
	    XtSetArg(args[n], XmNselectedItems, site_sel_array);	n++;
	} else {
	    XtSetArg(args[n], XmNselectedItems, NULL);			n++;
	} /* else */
	XtSetValues(rt->li_dlg->list, args, n);
				/* Free site strings (list makes copy)	*/
	for (i=0; i < gv->gv_nmemb; i++) XmStringFree(site_array[i]);

				/* Set dialog's value string		*/
	(void)sprintf(str_buf, "%d server%s selected.", num_sel,
		(num_sel == 1 ? " " : "s "));
	value_str = XmStringCreateSimple(str_buf);
	SetListDialog(rt->li_dlg, app_win->shell,
		NULL, NULL, NULL, value_str);
	XmStringFree(value_str);

				/* Popup dialog				*/
	XtManageChild(rt->li_dlg->dialog);
} /* function SetTargetServersCB */





static void SetTargetServersChangeCB(w, client_data, call_data)
Widget	w;		/* UNUSED */
caddr_t	client_data;	/* RoundTripTool */
caddr_t	call_data;	/* XmListCallbackStruct */
{
RoundTripTool	rt = (RoundTripTool)client_data;
AppWindow	*app_win = rt->common.app_win;
XmListCallbackStruct	*cb_info = (XmListCallbackStruct *)call_data;
char		value_buf[64];
XmString	value_str;

	
	(void)sprintf(value_buf, "%d server%sselected.",
		cb_info->selected_item_count,
		(cb_info->selected_item_count == 1 ? " " : "s "));
	value_str = XmStringCreateSimple(value_buf);
	SetListDialog(rt->li_dlg, NULL, NULL, NULL, NULL, value_str);
	XmStringFree(value_str);
} /* function SetTargetServersChangeCB */





static void SetTargetServersConfirmCB(w, client_data, call_data)
Widget	w;		/* UNUSED */
caddr_t	client_data;	/* RoundTripTool */
caddr_t	call_data;	/* UNUSED */
{
RoundTripTool	rt = (RoundTripTool)client_data;
AppWindow	*app_win = rt->common.app_win;
int		*sel_pos,
		sel_cnt,
		i=0,
		*s;


				/* Get list of selected servers		*/
	if (XmListGetSelectedPos(rt->li_dlg->list, &sel_pos, &sel_cnt)) {
				/* Keep only selected servers		*/
	    for (s=sel_pos; i < sel_cnt; i++,s++) {
		(void)memcpy(&rt->target_servers[i],  rt->site_list[*s - 1],
			sizeof(address));
	    } /* for */

	    XtFree(sel_pos);
	} /* if */
				/* Terminate list of targets		*/
	(void)memcpy(&rt->target_servers[i], &NULLADDRESS, sizeof(address));

				/* Update info display			*/
	SetToolInfo(rt);
} /* function SetTargetServersConfirmCB */

/************************** End of xpi_TLroundtrip.c *************************/
