/******************************************************************************
 * Module xpi_notify.c - Notification routines for xperfisis
 *
 * Written:	 06/18/91 by John H. Lee
 * Last Changed: 06/18/91 by John H. Lee
 * Version:	 0.1
 ******************************************************************************
 * Description:
 *
 *	This modules provides the means for a convenient mechanism for
 * functions in one ISIS task to invoke functions in another task in a
 * callback-style.  Specifically, it can be used for a task not in the
 * X toolkit task to notify the X toolkit task about things such as a
 * test completing.  This allows the main task to fork tasks to perform
 * calls that may block and yet be notified when the task is complete.
 *	This module is necessary since the toolkit and Xlib are not
 * reentrant, so only the main task may use toolkit and Xlib calls.
 ******************************************************************************
 * Exports:
 *	NotifyList CreateNotifyList()	Allocate & initialize a notify list
 *	void DestroyNotifyList()	Deallocate a notify list
 *	void SetNotifyDoneProc()	Set procedure to call after any clients
 *	void AddNotify()		Add client to notify list
 *	void RemoveNotify()		Remove client from notify list
 *	void TriggerNotify()		Cause the calling of clients when safe
 *	void HandleNextNotify()		Handle next triggered notify list
 *	void HandleNotify()		Handle all triggered notify lists
 *
 * Imports:
 * 	None
 ******************************************************************************
 * Revisions:
 *   0.1   06/18/91 JHL  Initial Creation
 *****************************************************************************/

#include "isis.h"
#include "xperfisis.h"
#include "xpi_notify.h"




/*** Private Type Definitions ***/

typedef struct {
	void	(*callback_proc)();
	Tool	t;
	caddr_t	client_data;
} NotifyEntry;

typedef struct _TriggerEntry_Struct {
	struct _TriggerEntry_Struct	*next;
	caddr_t				call_data;
} TriggerEntry;

struct _NotifyList_Struct {
	struct _NotifyList_Struct	*trig_next;
	NotifyEntry	*clients;
	int		num_notify;
	TriggerEntry	*triggers;
	void		(*notify_done_proc)();
	caddr_t		notify_done_arg;
	Bool		please_destroy;
};


/*** Private Global Definitions ***/

static NotifyList	TriggeredNotify = NULL,
			TriggeredNotifyEnd = NULL,
			CurrentNotify;



/******************************************************************************
 *
 *	Private Functions
 *
 *****************************************************************************/
/*** Enqueue notify list onto list of triggered notifies ***/
static void EnqueueTriggered(nl)
NotifyList	nl;
{
	if (TriggeredNotify == NULL) {
	    TriggeredNotify = nl;
	    TriggeredNotifyEnd = nl;

	} else {
	    nl->trig_next = NULL;
	    TriggeredNotifyEnd->trig_next = nl;
	    TriggeredNotifyEnd = nl;
	} /* else */
} /* function EnqueueTriggered */





/*** Destroy and Free Notify List ***/
static void DestroyNL(nl)
NotifyList	nl;
{
TriggerEntry	*p,
		*q;
int		i;


				/* Free trigger list			*/
	p = nl->triggers; 
	while (p != NULL) {
	    q = p;
	    p = p->next;
	    XtFree(q);
	} /* while */
	XtFree(nl->clients);	/* Free client list			*/
	XtFree(nl);		/* Free notify list itself		*/
} /* function DestroyNotifyList */

/******************************************************************************
 *
 *	Public Functions
 *
 *****************************************************************************/
/******************************************************************************
 *
 *	Initialize Notify List
 *
 *****************************************************************************/

#ifdef	_NO_PROTO
NotifyList CreateNotifyList()
#else
NotifyList CreateNotifyList(void);
#endif	_NO_PROTO
{
NotifyList	nl;


	nl = XtNew(struct _NotifyList_Struct);
	nl->trig_next = NULL;
	nl->clients = NULL;
	nl->num_notify = 0;
	nl->triggers = NULL;
	nl->notify_done_proc = NULL;
	nl->notify_done_arg = NULL;
	nl->please_destroy = False;

	return (nl);
} /* function CreateNotifyList */

/******************************************************************************
 *
 *	Destroy Notify List
 *
 *****************************************************************************/

#ifdef	_NO_PROTO
void DestroyNotifyList(nl)
NotifyList	nl;
#else
void DestroyNotifyList(NotifyList nl);
#endif	_NO_PROTO
{
NotifyList	p,
		q;


				/* Is this notify list the active one?	*/
	if (nl == CurrentNotify) {
				/* Mark and destroy when safe		*/
	    nl->please_destroy = True;

				/* Has this list been triggered?	*/
	} else {
	    if (nl == TriggeredNotify) {
		TriggeredNotify = TriggeredNotify->trig_next;
	    } else if (nl->trig_next != NULL) {
				/* Locate place in triggered list	*/
		p = TriggeredNotify;
		q = NULL;
		while ((p != NULL) && (p != nl)) {
		    q = p;
		    p = p->trig_next;
		} /* while */
				/* Remove from list			*/
		q->trig_next = p->trig_next;
	    } /* else if */
				/* Destroy list				*/
	    nl->trig_next = NULL;
	    DestroyNL(nl);
	} /* else */
} /* function DestroyNotifyList */

/******************************************************************************
 *
 *	Set Notify Done Proc
 *		The NotifyDoneProc() is called when all clients of the notify
 *	list has been called.
 *
 *****************************************************************************/

#ifdef	_NO_PROTO
void SetNotifyDoneProc(nl, done_proc, done_proc_arg)
NotifyList	nl;
void		(*done_proc)();
caddr_t		done_proc_arg;
#else
void SetNotifyDoneProc(NotifyList nl,
	void (*done_proc)(), caddr_t done_proc_arg);
#endif	_NO_PROTO
{
	nl->notify_done_proc = done_proc;
	nl->notify_done_arg = done_proc_arg;
} /* function SetNotifyDoneProc */

/******************************************************************************
 *
 *	Add Client to Notify List
 *
 *****************************************************************************/

#ifdef	_NO_PROTO
void AddNotify(t, nl, callback_proc, client_data)
Tool		t;
NotifyList	nl;
void		(*callback_proc)();
caddr_t		client_data;
#else
void AddNotify(Tool t, NotifyList nl,
	void (*callback_proc)(), caddr_t client_data);
#endif	_NO_PROTO
{
int	i;


				/* Allocate room for another client	*/
	nl->num_notify++;
	nl->clients = (NotifyEntry *)XtRealloc(nl->clients,
		nl->num_notify * sizeof(NotifyEntry));
				/* Add new client to notify list	*/
	nl->clients[nl->num_notify - 1].t = t;
	nl->clients[nl->num_notify - 1].callback_proc = callback_proc;
	nl->clients[nl->num_notify - 1].client_data = client_data;
} /* function AddNotify */

/******************************************************************************
 *
 *	Delete Client from Notify List
 *
 *****************************************************************************/

#ifdef	_NO_PROTO
void RemoveNotify(t, nl, callback_proc, client_data)
Tool		t;
NotifyList	nl;
void		(*callback_proc)();
caddr_t		client_data;
#else
void RemoveNotify(Tool t, NotifyList nl,
	void (*callback_proc)(), caddr_t client_data);
#endif	_NO_PROTO
{
int	i;


				/* Locate client in client array	*/
	for (i=nl->num_notify - 1; i > 0; i--) {
	    if ((nl->clients[i].t == t)
		    && (nl->clients[i].callback_proc == callback_proc)
		    && (nl->clients[i].client_data == client_data)) {
				/* Remove client, compact array		*/
		nl->num_notify--;
		if (i < nl->num_notify) bcopy(
			&nl->clients[i+1], &nl->clients[i],
			(nl->num_notify - i) * sizeof(NotifyEntry));
		break;
	    } /* if */
	} /* for */
} /* function RemoveNotify */

/******************************************************************************
 *
 *	TriggerNotify
 *		Enable invoking of clients when safe to do so.
 *
 *****************************************************************************/

#ifdef	_NO_PROTO
void TriggerNotify(nl, call_data)
NotifyList	nl;
caddr_t		call_data;
#else
void TriggerNotify(NotifyList nl, caddr_t call_data);
#endif	_NO_PROTO
{
TriggerEntry	*trigger,
		*t;

				/* Allow trigger only if not to be destroyed */
	if (nl->please_destroy) return;
				/* Enqueue if not already on triggered list */
	if (nl->triggers == NULL) EnqueueTriggered(nl);
				/* Create trigger entry with call_data	*/
	trigger = XtNew(TriggerEntry);
	trigger->call_data = call_data;
	trigger->next = NULL;
				/* Enqueue trigger on trigger list	*/
	if (nl->triggers == NULL) nl->triggers = trigger;
	else {
	    for (t=nl->triggers; t->next != NULL; t=t->next);
	    t->next = trigger;
	} /* else */
} /* function TriggerNotify */

/******************************************************************************
 *
 *	HandleNextNotify
 *		Invoke clients of Notify List--caller is responsible for
 *	ensuring it is safe to do so.
 *
 *****************************************************************************/

#ifdef	_NO_PROTO
Bool HandleNextNotify()
#else
Bool HandleNextNotify(void);
#endif	_NO_PROTO
{
NotifyEntry	*client_list;
NotifyList	nl;
int		i,
		num_clients;
void		(*notify_done_proc)();
caddr_t		notify_done_arg;
TriggerEntry	*trigger;


	if (TriggeredNotify == NULL) return (False);

				/* Get next triggered notify list	*/
	CurrentNotify = TriggeredNotify;
	TriggeredNotify = TriggeredNotify->trig_next;
	nl = CurrentNotify;
	nl->trig_next = NULL;	/* Mark as not being on TriggeredList, in */
				/*   case a client calls DeleteNotifyList() */

				/* Copy list so we can iterate safely	*/
	client_list = (NotifyEntry *)XtMalloc(sizeof(NotifyEntry)
		* nl->num_notify);
	memcpy(client_list, nl->clients, nl->num_notify * sizeof(NotifyEntry));
	num_clients = nl->num_notify;
				/* Copy notify_done function		*/
	notify_done_proc = nl->notify_done_proc;
	notify_done_arg = nl->notify_done_arg;
				/* Get call_data			*/
	trigger = nl->triggers;

				/* Iterate through list & invoke each client */
	CurrentNotify = nl;
	for (i=0; i < num_clients; i++) (*client_list[i].callback_proc)
		(client_list[i].t, client_list[i].client_data,
		trigger->call_data);
				/* Invoke notify_done function		*/
	if (notify_done_proc != NULL) (*notify_done_proc)(notify_done_arg);
				/* Update trigger list (do it after when */
				/*   clients might call TriggerNotify()) */
	nl->triggers = trigger->next;
	XtFree(trigger);

				/* More triggers?  Re-enque		*/
	if (nl->triggers != NULL) EnqueueTriggered(nl);
				/* Destroy if marked as so		*/
	else if (nl->please_destroy) DestroyNL(nl);

	XtFree(client_list);	/* Free safe copy of clients		*/

	CurrentNotify = NULL;	/* Not currently notifying clients	*/

	return (True);		/* There was a triggered notify list	*/
} /* function HandleNextNotify */

/******************************************************************************
 *
 *	HandleNotify
 *		Invoke clients of all triggered Notify Lists--caller is
 *	responsible for	ensuring it is safe to do so.
 *
 *****************************************************************************/

#ifdef	_NO_PROTO
Bool HandleNotify()
#else
Bool HandleNotify(void);
#endif	_NO_PROTO
{
NotifyEntry	*client_list;
NotifyList	nl;
int		i,
		num_clients;
void		(*notify_done_proc)();
caddr_t		notify_done_arg;
TriggerEntry	*trigger;


	if (TriggeredNotify == NULL) return (False);

				/* Do all triggered notify lists	*/
	while (TriggeredNotify != NULL) HandleNextNotify();

	return (True);		/* There was a triggered notify list	*/
} /* function HandleNotify */

/*************************** End of xpi_notify.c *****************************/
