/*

Copyright 1990 by M. Wood and K. Marzullo
Rights to use this source in unmodified form granted for all
commercial and research uses.  Rights to develop derivative
versions reserved by the authors.

*/

#define ISSOURCE
#include "isis.h"
#include "types.h"
#include <ctype.h>

typedef char STRING16[16];

static int base_type_size[] = {sizeof(int),sizeof(double),sizeof(STRING16)};

static int longcmp(),realcmp(),type_strcmp();
extern int strcmp();

static int (*base_type_cmp[])() = {longcmp,realcmp,type_strcmp};

static char * longput(), * realput(), * strput();
static char * (*base_type_put[])() = {longput,realput,strput};

char *base_type_name[] = {"integer","real","string"};

static int longcmp3(),realcmp3(),strcmp3();

int (*basetypecmp3[])() = {longcmp3,realcmp3,strcmp3};

extern ISET *set_index();

int type_relop(s)
char *s;

{
    int relation;

    if (*s == '?') {
	s++;
	relation = R_POSSIBLE;
    } else if (*s == '#') {
	s++;
	relation = R_CARD;
    } else
      relation = 0;

    if (*s == '<') {
	if (*++s == 0) return relation | R_LT;
	else if (*s == '>') return relation | R_NE;
	else if (*s == '=') return relation | R_LE;
    } else if (*s == '>') {
	if (*++s == 0) return relation | R_GT;
	else if (*s == '=') return relation | R_GE;
    } else if (*s == '=') {
	if (*++s == 0) return relation | R_EQ;
    } else if (*s == '!') {
	if (*++s == '=') return relation | R_NE;
    } else if (*s == '*') {
	if (*++s == 0) return relation | R_CHANGE;
    }
    fprintf(stderr,"Bad relop\n");
    exit(-1);
}



char * relop_name(rel)
int rel;

{
    static char *rels[] = {"CMP","=",">",">=","<","<=","<>","*"};
    static char buffer[8];
    char *p = buffer;
    int base = rel & R_MASK;
    
    if (rel & R_CARD) 
      *p++ = '#';
    if (rel & R_POSSIBLE)
      *p++ = '?';
    strcpy(p,rels[base]);
    return buffer;
}



void relop_fprint(fh,rel)
FILE *fh;
int rel;

{
    fputs(relop_name(rel),fh);
}



/* Type comparison routines */
/* could have function converting (-1,0,1) to T,F but that would
   require always doing two compares so will do it this way to make
   it faster */

static int longcmp(relation,a,b)
int relation;
long *a,*b;

{
    switch (relation) {
      case R_CMP : return (*a - *b);
      case R_EQ : return *a == *b;
      case R_LT : return *a < *b;
      case R_LE : return *a <= *b;
      case R_GT : return *a > *b;
      case R_GE : return *a >= *b;
      case R_NE :
      case R_CHANGE : return *a != *b;
      default : fprintf(stderr,"bad relational op!\n");
    }
}



static int realcmp(relation,a,b)
int relation;
double *a,*b;

{
    switch (relation) {
      case R_CMP : return (*a==*b) ? 0 : ((*a<*b) ? -1 : 1);
      case R_EQ : return *a == *b;
      case R_LT : return *a < *b;
      case R_LE : return *a <= *b;
      case R_GT : return *a > *b;
      case R_GE : return *a >= *b;
      case R_NE :
      case R_CHANGE : return *a != *b;
      default : fprintf(stderr,"bad relational op!\n");
    }
}



static int type_strcmp(relation,a,b)
int relation;
char *a,*b;

{
    int cmpval = strcmp(a,b);

    switch (relation) {
      case R_CMP : return cmpval;
      case R_EQ : return cmpval == 0;
      case R_LT : return cmpval < 0;
      case R_LE : return cmpval <= 0;
      case R_GT : return cmpval > 0;
      case R_GE : return cmpval >= 0;
      case R_NE :
      case R_CHANGE : return cmpval != 0;
      default : fprintf(stderr,"bad relational op!\n");
    }
}



static int longcmp3(a,b)
int *a,*b;

{
    return *a - *b;
}



static int realcmp3(a,b)
double *a,*b;

{
    return (*a==*b) ? 0 : ((*a<*b) ? -1 : 1);
}

static int strcmp3(a,b)
char **a,**b;

{
    return strcmp(*a,*b);
}



static char * longput(s,val)
char *s;
long * val;

{
    sprintf(s,"%d",*val);
    return s + strlen(s);

}


static char * realput(s,val)
char *s;
double * val;

{
    sprintf(s,"%f",*val);
    return s + strlen(s);
}



static char * strput(s,val)
char *s;
char * val;

{
    *s = '"';
    strcpy(s + 1,val);
    strcat(s,"\"");
    return s + strlen(s);
}


int type_size(type)
int type;

{
    if (type & TYPE_SET)
      return sizeof(message *);
    else if (type & TYPE_INTERVAL)
      return 2 * base_type_size[type & TYPE_BASE];
    else
      return base_type_size[type];
}


char * type_isisfmt(type,external)
int type, external;

{
    static char * isis_fmtcodes[4][3] = {
	{ "%D[1]","%G[1]","%s"},
	{ "%-d","%-g","%-s"},
	{ "%D[2]","%G[2]","%S[2]"},
	{ "%-D[2]","%-G[2]","%-S[2]"}
    };

    char * t;

    t = (type & TYPE_SET) ? 
      ((external) ? "%m" : "%-m") :
      isis_fmtcodes[((external) ? 0 : 1) +
		    ((type & TYPE_INTERVAL) ? 2 : 0)]
	[type & TYPE_BASE];

    return t;
}



int cardinality(set,type)
message *set;
int type;

{
    int count;
    PTR p;
    char * fmt;

    type &= TYPE_BASE;
    fmt = type_isisfmt(type,FALSE);
    msg_rewind(set);
    for (count = 0;
	 msg_get(set,fmt,&p) > 0;
	 count++) ;
    msg_rewind(set);
    return count;
}



type_cmp(type,relation,a,b)
int type;
int relation;
PTR a,b;

{
    PTR width;

    if (type & TYPE_SET) {
	if (relation & R_CARD) {
	    fprintf(stderr,"some code needs changing somewhere\n");
	} else {
	    int cval;
	    ISET *isa,*isb;

	    isa = set_index(type,*(message **) a);
	    isb = set_index(type,*(message **) b);
	    cval = isetcmp(type,relation,isa,isb);
	    free_iset(isa);
	    free_iset(isb);
	    return cval;
	}
    } else if (type & TYPE_INTERVAL) {
	int base_type = type & TYPE_BASE;
	int elmsize = base_type_size[base_type];

	if (relation & R_CARD) {
	    int ival;
	    double dval;

	    if (base_type == TYPE_REAL) {
		dval = (* (double *)((double *)a + elmsize)) -
		  (* (double *)a);
		width = (PTR) &dval;
		return realcmp(relation & R_MASK,width, (double *) b);
	    } else {
		/* type INT */
		ival = (* (long *)((long *)a + elmsize)) -
		  (* (long *) a);
		width = (PTR) &ival;
		return longcmp(relation & R_MASK,width, (long *) b);
	    }
	} else {
	    /* compare two intervals, first taking into account
	       the possibly operator.  For relation R, POSSIBLY R is
	       the same as NOT DEFINITELY NOT R (?R = ~!~R). 

	       Note that R_CMP and R_POSSIBLE are mutually exclusive
	       options. */

	    int base_relation;
	    int result;

	    if (relation & R_POSSIBLE) 
	      base_relation = (~relation) & R_MASK;
	    else
	      base_relation = relation & R_MASK;

	    switch (base_relation) {
	      case R_CMP : 
		if (type_cmp(type,R_LT,a,b))
		  return -1;
		else if (type_cmp(type,R_GT,a,b))
		  return 1;
		else
		  return 0;
	      case R_EQ : 
		result =
		  (*base_type_cmp[base_type])(R_EQ,a,b) &&
		  (*base_type_cmp[base_type])
		    (R_EQ,(char *)a + elmsize,(char *)b + elmsize);
		break;
	      case R_NE : 
		result =
		  (*base_type_cmp[base_type])
		    (R_GT,b,(char *)a+elmsize) ||
		  (*base_type_cmp[base_type])
		    (R_LT,(char *)b+elmsize,a);
		break;
	      case R_LT :
		result =
		  (*base_type_cmp[base_type])
		    (R_LT,(char *)a+elmsize,b);
		break;
	      case R_LE :
		result =
		  (*base_type_cmp[base_type])
		    (R_LE,(char *)a+elmsize,b);
		break;
	      case R_GT :
		result =
		  (*base_type_cmp[base_type])
		    (R_GT,a,(char *)b+elmsize);
		break;
	      case R_GE :
		result =
		  (*base_type_cmp[base_type])
		    (R_GE,a,(char *)b+elmsize);
		break;
	    }
	    if (relation & R_POSSIBLE)
	      result = !result;
	    return result;
	}
    } else
      return (*base_type_cmp[type])(relation & R_MASK,a,b);
}


      
char *type_to_ascii(type, val)
int type;
PTR val;

{
    int base_type;
    static char buffer[8192];
    char *p = buffer;

    base_type = type & TYPE_BASE;

    if (type & TYPE_INTERVAL) {
	*p++ = '[';
	p = (*base_type_put[base_type])(p,val);
	*p++ = ',';
	p = (*base_type_put[base_type])
	  (p,(char *)val+base_type_size[base_type]);
	*p++ = ']';
    } else if (type & TYPE_SET) {
	PTR vp;
	int num_elmts = cardinality(*(message **) val,base_type);
	char * fmt = type_isisfmt(base_type, FALSE);

	msg_rewind( * (message **) val);
	*p++ = '{';
	while (num_elmts--) {
	    msg_get(*(message **) val,fmt,&vp);
	    p = (*base_type_put[base_type])(p,vp);
	    if (num_elmts) *p++ = ' ';
	}
	*p++ = '}';
	msg_rewind( * (message **) val);
    } else
      p = (*base_type_put[base_type])(p,val);

    *p = (char) 0;
    if (p-buffer > sizeof(buffer)) {
	fprintf(stderr,"type buffer overflow!\n");
	exit(-1);
    }
    return buffer;
}
   


type_fprint(fh,type,val)
FILE *fh;
int type;
PTR val;

{
    fputs(type_to_ascii(type,val),fh);
}




int packrel(type,relation,m1,col1,m2,col2)
int type;
int relation;
int col1,col2;
message *m1,*m2;

{
    PTR m1_p,m2_p;


    if (type & TYPE_SET) {
	fprintf(stderr,"Packrel on sets not implemented");
	exit(-1);
    }
    m1_p = m2_p = NULL;
    type_unpack(type,m1,col1,&m1_p);
    type_unpack(type,m2,col2,&m2_p);
    if (!m1_p)
      msg_printaccess(m1_p);
    if (!m2_p)
      msg_printaccess(m2_p);

    return type_cmp(type,relation,m1_p,m2_p);
}

int packcmp(type,m1,col1,m2,col2)
int type;
int col1,col2;
message *m1,*m2;
{
    return packrel(type, R_CMP, m1, col1, m2, col2);
}

int copy_col(type,dsttuple, dstcol, srctuple,srccol)
int type;
message *dsttuple;
int dstcol;
message *srctuple;
int srccol;

{
   PTR val = NULL;

   if (type_unpack(type,srctuple,srccol,&val) < 0)
     return -1;
   return type_pack(type,dsttuple,dstcol,val);
}



type_unpack(type,msg,column,val)
int type;
message *msg;
int column;
PTR *val;

{
    int status;

    if (type & TYPE_SET) {
	static message * m;
	status = msg_getfld(msg,column + 1,NULL,
			    type_isisfmt(type,*val), &m);
	if (*val) {
	    m = msg_copy(m);
	    ** (message ***) val = m;
	} else
	  (*val) = (PTR) &m;
    } else 
      status = msg_getfld(msg,column + 1,NULL,
			  type_isisfmt(type,*val),
			  (*val) ? *val : (PTR) val);

    if (status < 0) {
	fprintf(stderr,"type_unpack: type mismatch\n");
	msg_printaccess(msg);
    }
    return status;
}



type_pack(type,msg,column,val)
int type;
message *msg;
int column;
PTR val;

{
    return msg_putfld(msg,column+1,type_isisfmt(type,TRUE),
		      (type & TYPE_SET) ? (PTR ) * (message **) val :
		      val);
}



static char *strtos(b,bp)
char *b;
char **bp;

/* this routine takes a pointer into a string and returns the next
   substring, which is defined as follows:  skip past all white space,
   then if character is a quote, return string up to end quote else
   return up to next white space.  The substring is formed by
   overwriting the ending character (white space or ") with a
   (char) 0.  The out parm bp is set to point to where the next
   substring begins (although there may be leading white space. */
   
{
    char *rv;
    extern char *strdup();

    while (isspace(*b)) b++;
    if (*b == '"') {
	for (*bp = rv = b + 1;
	     **bp && **bp != '"';
	     (*bp)++) ;
    } else { 
	for (*bp = rv = b;
	     **bp && !(isspace(**bp));
	     (*bp)++) ;
    }
    if (**bp) {
	/* don't do this if already at end of string */
	**bp = (char) 0;
	(*bp)++;
    }
    return rv;
}



ascii_to_type(type,bp,rp,val)
int type;
char *bp;
char **rp;
PTR * val;

{
    extern double strtod();
    extern long strtol();
    static int_interval ival;
    static real_interval dval;
    int type_base = type & TYPE_BASE;

    if (rp) *rp = bp;

    if (type & TYPE_INTERVAL) {
	if (*bp != '[') {
	    /* do a type promotion */
	    if (type_base == TYPE_INT) {
		ival[0] = ival[1] = strtol(bp,&bp,0);
		*val = (PTR) ival;
	    } else if (type_base == TYPE_REAL) {
		dval[0] = dval[1] = strtod(bp,&bp);
		*val = (PTR) dval;
	    } else
	      return -1;
	} else {
	    bp++;		/* get past '[' */
	    if (type_base == TYPE_INT)
	      ival[0] = strtol(bp,&bp,0);
	    else
	      dval[0] = strtod(bp,&bp);
	    if (*bp++ != ',')
	      return -1;
	    if (type_base == TYPE_INT) {
		ival[1] = strtol(bp,&bp,0);
		*val = (PTR) ival;
	    } else if (type_base == TYPE_REAL) {
		dval[1] = strtod(bp,&bp);
		*val = (PTR ) dval;
	    } else
	      return -1;
	}
    } else if (type & TYPE_SET) {
	static message * set;
	char *new_bp;
	PTR val1;

	set = msg_newmsg();
	*val = (PTR) &set;
	if (*bp == '{') 
	  bp++;			/* get past '{' */
	while (*bp && (*bp != '}')) {
	    if (!ascii_to_type(type_base,bp,&new_bp,&val1)) {
		set_pack(type,&set,val1);
	    } else 
	      return -1;
	    bp = new_bp;
	    while (isspace(*bp)) bp++;
	}
    } else {
	if (type_base == TYPE_INT) {
	    ival[0] = strtol(bp,&bp,0);
	    *val = (PTR) ival;
	} else if (type_base == TYPE_REAL) {
	    dval[0] = strtod(bp,&bp);
	    *val = (PTR) dval;
	} else if (type_base == TYPE_STRING) {
	    *val = (PTR) strtos(bp,&bp);
	} else
	  return -1;
    }
    if (rp) *rp = bp;
    return 0;
}



int name_to_type(name)
char * name;

{
    int type;
    int len = strlen(name);
    int i;

    if ((*name=='{') || (*name=='[')) {
	type = (*name == '{') ? TYPE_SET : TYPE_INTERVAL;
	len -= 2;		/* subtract out the braces */
	/* check for corresponding closing brace, which in
	   ASCII is two away. */
	if (*name != (char) (*(name + len + 1) - 2))
	  return -1;
	name++;
    } else {
	type = 0;
    }

    for (i=-1;++i<NUM_TYPES;) {
	if (!strncmp(name,base_type_name[i],len)) {
	    type |= i;
	    break;
	}
    }
    if (i == NUM_TYPES) {
	return -1;
    } else
      return type;
}


char * type_to_name(type)
int type;

{
    static char typename[16];
    char *p = typename;

    if (type & TYPE_INTERVAL)
      *p++ = '[';
    else if (type & TYPE_SET)
      *p++ = '{';

    strcpy(p,base_type_name[type & TYPE_BASE]);

    if (type & TYPE_INTERVAL)
      strcat(p,"]");
    else if (type & TYPE_SET)
      strcat(p,"}");
    return typename;
}
