/*
 *      Originally coded by Robbert van Renesse
 *
 *
 *      ISIS release V2.0, May 1990
 *      Export restrictions apply
 *
 *      The contents of this file are subject to a joint, non-exclusive
 *      copyright by members of the ISIS Project.  Permission is granted for
 *      use of this material in unmodified form in commercial or research
 *      settings.  Creation of derivative forms of this software may be
 *      subject to restriction; obtain written permission from the ISIS Project
 *      in the event of questions or for special situations.
 *      -- Copyright (c) 1990, The ISIS PROJECT
 */

/*
 * This file contains a lot of trivial routines for mathematical
 * operations.
 */

#include "magic.h"
#include <math.h>
#include "value.h"
#include "expr.h"

struct value *x_pop();

int m_i1, m_i2, m_seq;
double m_d1, m_d2;

/* Fetch one integer value of the stack.  If the top of the stack is null,
 * push it back and return null.
 */
m_int1(c)
struct context *c;
{
	struct value *v = x_pop(c);

	if (val_null(v)) {
		x_dirpush(v);
		return 0;
	}
	m_seq = val_seq(v);
	m_i1 = val_int(v);
	val_free(v);
	return 1;
}

/* Pop two integers of the stack, and store them in i1 and i2.  An
 * operation will be applied to them.  In m_seq we put the sequence
 * number of the result of this operation, being the maximum of the
 * sequence numbers of the operands.  If any of the operands is null,
 * push a null and return 0.
 */
m_int2(c)
struct context *c;
{
	struct value *v1, *v2;

	v2 = x_pop(c);
	v1 = x_pop(c);
	if (val_null(v1) || val_null(v2)) {
		val_free(v1);
		val_free(v2);
		x_null(c);
		return 0;
	}
	m_seq = m_max(val_seq(v1), val_seq(v2));
	m_i1 = val_int(v1);
	m_i2 = val_int(v2);
	val_free(v1);
	val_free(v2);
	return 1;
}

/* Like m_int1, but now with a double.
 */
m_dbl1(c)
struct context *c;
{
	struct value *v = x_pop(c);

	if (val_null(v)) {
		x_dirpush(v);
		return 0;
	}
	m_seq = val_seq(v);
	m_d1 = val_dbl(v);
	val_free(v);
	return 1;
}

/* Like m_int2, but now with doubles.
 */
m_dbl2(c)
struct context *c;
{
	struct value *v1, *v2;

	v2 = x_pop(c);
	v1 = x_pop(c);
	if (val_null(v1) || val_null(v2)) {
		val_free(v1);
		val_free(v2);
		x_null(c);
		return 0;
	}
	m_seq = m_max(val_seq(v1), val_seq(v2));
	m_d1 = val_dbl(v1);
	m_d2 = val_dbl(v2);
	val_free(v1);
	val_free(v2);
	return 1;
}

/* Apply a function like sin() to a single operand.  If the operand is null,
 * push it back.
 */
m_do1(c, f)
struct context *c;
double (*f)();
{
	struct value *v = x_pop(c);

	if (val_null(v))
		x_dirpush(v);
	else
		x_dblpush((*f)(val_dbl(v)), val_seq(v));
	val_free(v);
}

m_exp(c)	struct context *c; { m_do1(c, exp); }
m_log(c)	struct context *c; { m_do1(c, log); }
m_sin(c)	struct context *c; { m_do1(c, sin); }
m_cos(c)	struct context *c; { m_do1(c, cos); }
m_tan(c)	struct context *c; { m_do1(c, tan); }
m_asin(c)	struct context *c; { m_do1(c, asin); }
m_acos(c)	struct context *c; { m_do1(c, acos); }
m_atan(c)	struct context *c; { m_do1(c, atan); }
m_sinh(c)	struct context *c; { m_do1(c, sinh); }
m_cosh(c)	struct context *c; { m_do1(c, cosh); }
m_tanh(c)	struct context *c; { m_do1(c, tanh); }
m_asinh(c)	struct context *c; { m_do1(c, asinh); }
m_acosh(c)	struct context *c; { m_do1(c, acosh); }
m_atanh(c)	struct context *c; { m_do1(c, atanh); }
m_round(c)	struct context *c; { m_do1(c, rint); }
m_floor(c)	struct context *c; { m_do1(c, floor); }
m_ceil(c)	struct context *c; { m_do1(c, ceil); }
m_aint(c)	struct context *c; { m_do1(c, aint); }
m_anint(c)	struct context *c; { m_do1(c, anint); }
m_j0(c)		struct context *c; { m_do1(c, j0); }
m_j1(c)		struct context *c; { m_do1(c, j1); }
m_y0(c)		struct context *c; { m_do1(c, y0); }
m_y1(c)		struct context *c; { m_do1(c, y1); }

m_sign(c)
struct context *c;
{
	if (m_dbl1(c))
		x_numpush(m_d1 < 0 ? -1 : (m_d1 == 0 ? 0 : 1), m_seq);
}

m_absolute(c)
struct context *c;
{
	if (m_dbl1(c))
		x_dblpush(m_abs(m_d1), m_seq);
}

m_neg(c)
struct context *c;
{
	if (m_dbl1(c))
		x_dblpush(-m_d1, m_seq);
}

m_minimum(c)
struct context *c;
{
	if (m_dbl2(c))
		x_dblpush(m_min(m_d1, m_d2), m_seq);
}

m_maximum(c)
struct context *c;
{
	if (m_dbl2(c))
		x_dblpush(m_max(m_d1, m_d2), m_seq);
}

m_plus(c)
struct context *c;
{
	if (m_dbl2(c))
		x_dblpush(m_d1 + m_d2, m_seq);
}

m_minus(c)
struct context *c;
{
	if (m_dbl2(c))
		x_dblpush(m_d1 - m_d2, m_seq);
}

m_times(c)
struct context *c;
{
	if (m_dbl2(c))
		x_dblpush(m_d1 * m_d2, m_seq);
}

m_div(c)
struct context *c;
{
	if (m_dbl2(c))
		x_dblpush(m_d1 / m_d2, m_seq);
}

m_intdiv(c)
struct context *c;
{
	if (m_int2(c))
		if (m_i2 == 0)
			x_dblpush(m_i1 / (double) m_i2, m_seq);
		else
			x_numpush(m_i1 / m_i2, m_seq);
}

m_mod(c)
struct context *c;
{
	if (m_int2(c))
		if (m_i2 == 0)
			x_dblpush(m_i2 / (double) m_i2, m_seq);
		else
			x_numpush(m_i1 % m_i2, m_seq);
}

m_pow(c)
struct context *c;
{
	if (m_dbl2(c))
		x_dblpush(pow(m_d1, m_d2), m_seq);
}

m_e(){
	x_dblpush(2.71828, 0);
}

m_pi(){
	x_dblpush(3.14159, 0);
}

m_atan2(c)
struct context *c;
{
	if (m_dbl2(c))
		x_dblpush(atan2(m_d1, m_d2), m_seq);
}

m_lt(c)
struct context *c;
{
	if (m_dbl2(c))
		x_numpush(m_d1 < m_d2, m_seq);
}

m_le(c)
struct context *c;
{
	if (m_dbl2(c))
		x_numpush(m_d1 <= m_d2, m_seq);
}

m_eq(c)
struct context *c;
{
	if (m_dbl2(c))
		x_numpush(m_d1 == m_d2, m_seq);
}

m_ne(c)
struct context *c;
{
	if (m_dbl2(c))
		x_numpush(m_d1 != m_d2, m_seq);
}

m_gt(c)
struct context *c;
{
	if (m_dbl2(c))
		x_numpush(m_d1 > m_d2, m_seq);
}

m_ge(c)
struct context *c;
{
	if (m_dbl2(c))
		x_numpush(m_d1 >= m_d2, m_seq);
}

m_not(c)
struct context *c;
{
	if (m_int1(c))
		x_numpush(!m_i1, m_seq);
}

m_and(c)
struct context *c;
{
	if (m_int2(c))
		x_numpush(m_i1 && m_i2, m_seq);
}

m_or(c)
struct context *c;
{
	if (m_int2(c))
		x_numpush(m_i1 || m_i2, m_seq);
}

m_trunc(c)
struct context *c;
{
	if (m_dbl1(c))
		x_numpush((int) m_d1, m_seq);
}

/* Register all the functions.
 */
m_start(){
	x_register("$sign", m_sign);
	x_register("$abs", m_absolute);	x_register("$neg", m_neg);
	x_register("$min", m_minimum);	x_register("$max", m_maximum);
	x_register("+", m_plus);	x_register("-", m_minus);
	x_register("*", m_times);	x_register("/", m_div);
	x_register("$div", m_intdiv);
	x_register("$mod", m_mod);	x_register("%", m_mod);
	x_register("$pow", m_pow);
	x_register("$exp", m_exp);	x_register("$log", m_log);
	x_register("$e", m_e);		x_register("$pi", m_pi);
	x_register("$sin", m_sin);	x_register("$cos", m_cos);
	x_register("$tan", m_tan);
	x_register("$asin", m_asin);	x_register("$acos", m_acos);
	x_register("$atan", m_atan);	x_register("$atan2", m_atan2);
	x_register("$sinh", m_sinh);	x_register("$cosh", m_cosh);
	x_register("$tanh", m_tanh);	x_register("$atanh", m_atanh);
	x_register("$asinh", m_asinh);	x_register("$acosh", m_acosh);
	x_register("<", m_lt);		x_register("<=", m_le);
	x_register("=", m_eq);		x_register("==", m_eq);
	x_register("<>", m_ne);		x_register("!=", m_ne);
	x_register(">", m_gt);		x_register(">=", m_ge);
	x_register("!", m_not);		x_register("$not", m_not);
	x_register("&&", m_and);	x_register("$and", m_and);
	x_register("||", m_or);		x_register("$or", m_or);
	x_register("$trunc", m_trunc);
	x_register("$round", m_round);	x_register("$rint", m_round);
	x_register("$floor", m_floor);	x_register("$ceil", m_ceil);
	x_register("$aint", m_aint);	x_register("$anint", m_anint);
	x_register("$j0", m_j0);	x_register("$j1", m_j1);
	x_register("$y0", m_y0);	x_register("$y1", m_y1);
}
