/*
 *      Originally coded by Robbert van Renesse
 *
 *
 *      ISIS release V2.0, May 1990
 *      Export restrictions apply
 *
 *      The contents of this file are subject to a joint, non-exclusive
 *      copyright by members of the ISIS Project.  Permission is granted for
 *      use of this material in unmodified form in commercial or research
 *      settings.  Creation of derivative forms of this software may be
 *      subject to restriction; obtain written permission from the ISIS Project
 *      in the event of questions or for special situations.
 *      -- Copyright (c) 1990, The ISIS PROJECT
 */

/*
 * This file contains the routines that maintain a flexible data base
 * consisting of records.  Each record consists of a set of named attributes.
 * The name is an arbitrary ASCII string.  The value of the attribute is
 * a sequence of values, the first one being the current one.  Each value
 * consists of an ASCII string and a global sequence number.
 */

#include <stdio.h>
#include "magic.h"
#include "value.h"
#include "expr.h"

/* Often the attributes of a record are small integers.  For efficiency sake
 * these attributes are stored separately from the other attributes.
 * MAX_NUMBER is the maximum number of attributes stored this way.
 */
#define MAX_NUMBER	100

/* Records and attributes are hashed for efficiency.  These are the sizes of
 * the hash tables.  Note that there is a hash table for attributes per
 * record.
 */
#define REC_HASH	547
#define ATTR_HASH	32

/* Often many operations occur on a small number of records.  Instead of
 * each time calculating a hash function of the string name of the record,
 * the pointers to the names are used to quickly find the record.  However,
 * this is unreliable since the data where the pointer is pointing may be
 * overwritten.  This hash table is used as a cache that must be cleared
 * in case data gets overwritten.
 */
#define PTR_HASH	32

/* Function evaluations are cached in a binary tree.  To control the size
 * of the cache the height of this tree is limited to MAX_HEIGHT.
 */
#define MAX_HEIGHT	6

#define LEFT	0	/* left child in binary tree */
#define RIGHT	1	/* right child in binary tree */

/* Nodes in a binary tree have the structure defined below.  The sort order
 * is according to history (sequence number).  The nodes are also kept in a
 * linked list to facilitate traveling down in history order.  A node stores
 * a value plus associated sequence number, and till when it had this value.
 * Normally this is the sequence number of the next value in history, but in
 * the case of the function cache this is not necessarily true.
 */
struct node {
	struct node *n_prev;		/* linked list in history order */
	struct value *n_value;		/* value stored in this node */
	int n_seq;			/* till this seqno it had this value */
	short n_balance;		/* for AVL balancing */
	short n_height;			/* height of the tree below */
	struct node *n_child[2];	/* the two children */
};

/* This structure points to a value tree.
 */
struct vtree {
	struct node *vt_root;		/* the root of the tree */
	struct node *vt_current;	/* the current (last) node */
	struct node *vt_last;		/* the last node accessed */
};

/* This structure points to the contents of an attribute in a record.
 */
struct contents {
	struct vtree c_values;		/* the binary tree of values */
	struct vtree c_cache;		/* cache of function evaluations */
};

/* This structure points to the name and contents of an attribute:
 */
struct attr {
	struct attr *a_next;		/* link to next attribute */
	char *a_name;			/* name of attribute */
	struct contents a_contents;
};

/* This structure stores a record.  It contains the name, the attributes
 * 0 through MAX_NUMBER (called index attributes), and the other attributes
 * (maintained in a hash table).
 */
struct record {
	struct record *r_next;			/* next record */
	char *r_name;				/* name of record */
	struct contents r_array[MAX_NUMBER];	/* index attributes */
	struct attr *r_attrs[ATTR_HASH];	/* hash table of attributes */
};

/* This structure is an entry in a hash table of (name, record) pairs, which
 * is used to make record lookup more efficient.
 */
struct cache {
	char *c_name;
	struct record *c_rec;
};

struct value *x_expr(), *x_pop();
FILE *fopen();

/* Test whether s points to "name".
 */
#define is_name(s)	(s[0]=='n'&&s[1]=='a'&&s[2]=='m'&&s[3]=='e'&&s[4]==0)

int db_seq;				/* global sequence number */
int db_change;				/* set if data base updated */
struct record *db_hashtab[REC_HASH];	/* hash table of records */
FILE *db_file;				/* dump file */
char db_filename[64];			/* the name of the dump file */
struct value *db_null;			/* null value */
struct cache db_cache[PTR_HASH];	/* db_getrec cache */

/* Increase the sequence number only if changes have been made.  If so,
 * call the watch module to see if any messages need to be sent.  This
 * is the place where a record could be written to a log file for complete
 * recoverability.
 */
db_commit(){
	if (db_change >= 0) {
		db_seq++;
		db_change = -1;
		watch_evaluate();
	}
}

/* Write a string to the data base file.  It is represented as n:s, where
 * n is the number of bytes, and s the string itself.
 */
db_sdump(s)
char *s;
{
	fprintf(db_file, "%d:%s", strlen(s), s);
}

/* This dumps an integer.
 */
db_idump(i){
	char buf[16];

	mag_strnum(buf, i);
	fprintf(db_file, "%d:%s", strlen(buf), buf);
}

/* This dumps a double float.
 */
db_ddump(d)
double d;
{
	char buf[32];

	sprintf(buf, V_FMT_DBL, d);
	fprintf(db_file, "%d:%s", strlen(buf), buf);
}

/* Dump a tree of value records.
 */
db_cdump(c)
struct contents *c;
{
	struct value *v;
	struct node *n;

	fprintf(db_file, ":");
	for (n = c->c_values.vt_current; n != 0; n = n->n_prev) {
		v = n->n_value;
		fprintf(db_file, "%d:", val_seq(v));
		switch (v->v_assign) {
		case V_STR:	db_sdump(v->v_str);	break;
		case V_DBL:	db_ddump(v->v_dbl);	break;
		case V_INT:	db_idump(v->v_int);	break;
		case 0:		fprintf(db_file, "-1");	break;
		default:	mag_panic("db_cdump");
		}
		fprintf(db_file, "\n");
	}
}

/* Write the whole data base to db_file
 */
db_checkpoint(){
	struct record **rp, *r;
	struct attr **ap, *a;
	int i;

	if ((db_file = fopen(db_filename, "w")) == 0) {
		printf("can't checkpoint\n");
		return;
	}
	db_commit();	/* make sure the sequence number is correct */
	fprintf(db_file, "%d\n", db_seq);
	for (rp = db_hashtab; rp < &db_hashtab[REC_HASH]; rp++)
		for (r = *rp; r != 0; r = r->r_next) {
			fprintf(db_file, "#4:name:0:");
			db_sdump(r->r_name);
			fprintf(db_file, "\n");
			for (i = 0; i < MAX_NUMBER; i++)
				if (r->r_array[i].c_values.vt_current != 0) {
					fprintf(db_file, "#");
					db_idump(i);
					db_cdump(&r->r_array[i]);
				}
			for (ap = r->r_attrs; ap < &r->r_attrs[ATTR_HASH]; ap++)
				for (a = *ap; a != 0; a = a->a_next)
					if (a->a_contents.c_values.vt_current
									!= 0) {
						fprintf(db_file, "#");
						db_sdump(a->a_name);
						db_cdump(&a->a_contents);
					}
		}
	fclose(db_file);
}

/* Free all nodes in the linked list pointed to by n.
 */
db_prune(n)
struct node *n;
{
	struct node *tmp;

	while ((tmp = n) != 0) {
		n = tmp->n_prev;
		val_free(tmp->n_value);
		MAG_FREE(tmp);
	}
}

/* Reduce the amount of memory used by this attribute.  Call val_reduce on
 * all values, and remove the cache.
 */
db_creduce(c)
struct contents *c;
{
	struct node *n;

	for (n = c->c_values.vt_current; n != 0; n = n->n_prev)
		val_reduce(n->n_value);
	db_prune(c->c_cache.vt_current);
	c->c_cache.vt_root = c->c_cache.vt_current = c->c_cache.vt_last = 0;
}

/* Reduce the amount of memory used in this data base.  Call db_creduce on
 * all attributes.
 */
db_reduce(){
	struct record **rp, *r;
	struct attr **ap, *a;
	int i;

	for (rp = db_hashtab; rp < &db_hashtab[REC_HASH]; rp++)
		for (r = *rp; r != 0; r = r->r_next) {
			for (i = 0; i < MAX_NUMBER; i++)
				db_creduce(&r->r_array[i]);
			for (ap = r->r_attrs; ap < &r->r_attrs[ATTR_HASH]; ap++)
				for (a = *ap; a != 0; a = a->a_next)
					db_creduce(&a->a_contents);
		}
}

/* Allocate a value node.
 */
struct node *db_node_alloc(){
	return MAG_ALLOC(1, struct node);
}

/* Do an AVL tree insert.  This requires rebalancing the tree if the heights
 * of subtrees differ more than one node.  *np points to the root node of
 * the tree where (v, t) needs to be inserted.  v is the value, and t is
 * till when this value is valid.  prev points to the pointer that holds
 * the previous node.  *prev must be updated to point to the inserted node.
 * db_avl returns 1 if the subtree has grown higher.  Look in AVL tree
 * literature to understand this routine.
 */
db_avl(np, prev, v, t)
struct node **np, **prev;
struct value *v;
{
	int side, delta;
	struct node *n, *n1, *n2;

	/* If bottom of tree, insert here.
	 */
	if ((n = *np) == 0) {
		n = *np = db_node_alloc();
		n->n_value = v;
		n->n_seq = t;
		n->n_prev = *prev;
		*prev = n;
		return 1;
	}

	/* If value with same sequence number here already, overwrite.
	 */
	if (val_seq(v) == val_seq(n->n_value)) {
		val_free(n->n_value);
		n->n_value = v;
		n->n_seq = t;
		return 0;
	}

	/* Recursively descend the tree.  Return if the height of the
	 * subtree hasn't changed.
	 */
	if (val_seq(v) < val_seq(n->n_value)) {
		if (!db_avl(&n->n_child[LEFT], &n->n_prev, v, t))
			return 0;
		side = LEFT;
		delta = -1;
	}
	else {
		if (!db_avl(&n->n_child[RIGHT], prev, v, t))
			return 0;
		side = RIGHT;
		delta = 1;
	}

	/* Adjust the balance.
	 */
	if (n->n_balance == 0) {
		n->n_balance = delta;
		n->n_height++;
		return 1;
	}
	if (n->n_balance == -delta)
		n->n_balance = 0;
	else {
		/* Out of balance.  Rebalance.
		 */
		n1 = n->n_child[side];
		if (n1->n_balance == delta) {	/* single rotate */
			n->n_child[side] = n1->n_child[!side];
			n1->n_child[!side] = n;
			n->n_balance = n1->n_balance = 0;
			*np = n1;
		}
		else {				/* double rotate */
			n2 = n1->n_child[!side];
			n1->n_child[!side] = n2->n_child[side];
			n->n_child[side] = n2->n_child[!side];
			n2->n_child[!side] = n;
			n2->n_child[side] = n1;
			n->n_balance = n2->n_balance == delta ? -delta : 0;
			n1->n_balance = n2->n_balance == -delta ? delta : 0;
			n2->n_balance = 0;
			n1->n_height--;
			n2->n_height++;
			*np = n2;
		}
		n->n_height--;
	}
	return 0;
}

/* Insert (v, t) in the tree pointed to by vt.  If prune is set, the height
 * of the tree is restricted to MAX_HEIGHT (true in the case of caches).
 * If the tree grows to big, all but the right branch of the tree is
 * removed, effectively reducing it to about half the size.
 */
db_insert(vt, v, t, prune)
struct vtree *vt;
struct value *v;
{
	struct node *n;

	if (db_avl(&vt->vt_root, &vt->vt_current, v, t) && prune &&
					vt->vt_root->n_height >= MAX_HEIGHT) {
		n = vt->vt_root->n_child[RIGHT];
		db_prune(vt->vt_root);
		if ((vt->vt_root = n) == 0)
			vt->vt_current = 0;
		vt->vt_last = 0;
		while (n->n_child[LEFT] != 0)
			n = n->n_child[LEFT];
		n->n_prev = 0;
	}
}

/* Add v to the end of the history of this attribute.  The n_seq field in
 * the current node has to be updated to the current time (val_seq(v)).
 * db_change is set to this time to reflect that the current value in the
 * cache should not be trusted.
 */
db_append(c, v)
struct contents *c;
struct value *v;
{
	struct vtree *vt = &c->c_values;
	int seq = val_seq(v);

	db_change = seq;
	if (vt->vt_current != 0)
		vt->vt_current->n_seq = seq;
	db_insert(vt, v, seq, 0);
}

/* This macro checks whether t falls within the node defined by n.
 */
#define db_match(t, n)	((t) >= val_seq((n)->n_value) && (t) <= (n)->n_seq)

/* Look for a node at time t in the tree vt.  Update the vt_last pointer
 * if a matching node is located.
 */
struct node *db_seek(vt, t)
struct vtree *vt;
{
	struct node *n;

	/* First see if the current node matches.  If there is no current
	 * node, there is no tree either, so no match can be made.
	 */
	if ((n = vt->vt_current) == 0)
		return 0;
	if (db_match(t, n))
		return vt->vt_last = n;

	/* Now check whether either the last node is accessed again, or
	 * the one prior to the last node.  Since history is traversed
	 * sequentially regularly, this will often produce a match.
	 */
	if ((n = vt->vt_last) != 0) {
		if (db_match(t, n))
			return n;
		if ((n = n->n_prev) != 0 && db_match(t, n))
			return vt->vt_last = n;
	}

	/* Last resort:  traverse the tree.
	 */
	for (n = vt->vt_root; n != 0;
			n = n->n_child[t < val_seq(n->n_value) ? LEFT : RIGHT])
		if (db_match(t, n))
			return vt->vt_last = n;

	return 0;
}

/* Find the node in attribute c at time t.  The sequence number of the
 * current node is first updated to be t.
 */
struct node *db_lookup(c, t)
struct contents *c;
{
	struct node *n;

	if (c == 0 || (n = c->c_values.vt_current) == 0)
		return 0;
	if (n->n_seq < t)
		n->n_seq = t;
	return db_seek(&c->c_values, t);
}

/* Find the tree of values associated with the attribute.  If it isn't there,
 * allocate it if create is set.
 */
struct contents *db_attr_find(r, attr, create)
struct record *r;
char *attr;
{
	unsigned n, m;
	struct attr **ap, *a;
	char *s1, *s2;

	/* First check whether the attribute is a number between 0 and 100.
	 */
	if ((n = (unsigned) (attr[0] - '0')) < 10) {
		if ((m = attr[1]) == 0)
			return &r->r_array[n];
		if ((m -= '0') < 10 && attr[2] == 0)
			return &r->r_array[n * 10 + m];
	}

	/* Now look through the hash table of attributes.  First compute
	 * a hash value for the attribute.
	 */
	if ((n = attr[0]) != 0)
		n = (n << 1) + attr[1];
	for (ap = &r->r_attrs[n % ATTR_HASH]; (a = *ap) != 0; ap = &a->a_next) {
		s1 = a->a_name;
		s2 = attr;
		while (*s1 == *s2++)
			if (*s1++ == 0)
				return &a->a_contents;
	}

	/* The attribute doesn't exist yet.  Create it if necessary.
	 */
	if (!create)
		return 0;
	a = *ap = MAG_ALLOC(1, struct attr);
	a->a_name = mag_copy(attr);
	return &a->a_contents;
}

/* Allocate a record structure.
 */
struct record *db_rec_alloc(){
	return MAG_ALLOC(1, struct record);
}

/* db_cache maintains (char *name, struct record) pairs to make record
 * lookup more efficient.  Unfortunately, the memory that name points to
 * may be overwritten.  In that case db_clrcache should be called with
 * this pointer argument.  db_clrcache is called from MAG_FREE such that
 * freed and subsequently re-allocated memory doesn't cause problems.
 */
db_clrcache(p)
char *p;
{
	db_cache[((unsigned) p >> 2) % PTR_HASH].c_name = 0;
}

/* Get the record structure associated with name rec.  If create is set,
 * the record should be created if it doesn't exist already.  First look
 * in the cache.  If not there, call db_doget().
 */
#define db_getrec(rec, create)	\
	(cache = &db_cache[((unsigned) (rec) >> 2) % PTR_HASH], \
	((rec) == cache->c_name ? cache->c_rec : db_doget(rec, create, cache)))

/* Get the record structure for rec.  If create is set, create the record
 * if it doesn't exist already.  Update the cache structure pointed to by c.
 */
struct record *db_doget(rec, create, c)
char *rec;
struct cache *c;
{
	struct record **rp, *r;
	char *p = rec, *q;
	int n = 0;

	/* Calculate a hash value.
	 */
	while (*p != 0) {
		n <<= 1;
		n ^= *p++;
	}
	if (n < 0)
		n = -n;
	rp = &db_hashtab[n % REC_HASH];

	/* Look in the linked list pointed to by the hash entry.
	 */
	while ((r = *rp) != 0) {
		p = r->r_name;
		q = rec;
		for (q = rec;; q++)
			if (*p != *q) {
				rp = &r->r_next;
				break;
			}
			else if (*p++ == 0) {
				c->c_name = rec;
				return c->c_rec = r;
			}
	}

	/* Record doesn't exist yet.
	 */
	if (!create)
		return 0;
	r = *rp = db_rec_alloc();
	r->r_name = mag_copy(rec);
	c->c_name = rec;
	return c->c_rec = r;
}

/* Store "value" in the attribute denoted by record "rec" and attribute
 * "attr".  If the record doesn't exist yet, create it.  If the attribute
 * doesn't exist yet, create that too.  The value may be null.
 */
db_put(rec, attr, val)
char *rec, *attr;
struct value *val;
{
	struct cache *cache;

	if (!is_name(attr))
		/* Somebody tries to update the name of a record.  What
		 * the semantics of this operation should be is unclear.
		 * What if the new name exists already?  We're currently
		 * going to ignore the operation, which makes more sense
		 * than anything else.
		 */
		db_append(db_attr_find(db_getrec(rec, 1), attr, 1),
					val_pseq(val, db_seq, V_COPY));
}

/* Like db_put, but the value is a string.  Copy the string into a value
 * structure and call db_put.
 */
db_store(rec, attr, val)
char *rec, *attr, *val;
{
	struct cache *cache;

	if (!is_name(attr))
		db_append(db_attr_find(db_getrec(rec, 1), attr, 1),
					val_cstr(val, db_seq));
}

/* Like db_store, only the value is an double.
 */
db_dblstore(rec, attr, val)
char *rec, *attr;
double val;
{
	struct cache *cache;

	if (!is_name(attr))
		db_append(db_attr_find(db_getrec(rec, 1), attr, 1),
					val_pdbl(val, db_seq));
}

/* Like db_store, only the value is an integer.
 */
db_numstore(rec, attr, val)
char *rec, *attr;
{
	struct cache *cache;

	if (!is_name(attr))
		db_append(db_attr_find(db_getrec(rec, 1), attr, 1),
					val_pint(val, db_seq));
}

/* Retrieve the value of an attribute at ``time'' t.
 */
struct value *db_find(rec, attr, t)
char *rec, *attr;
{
	int seq;
	struct record *r;
	struct contents *c;
	struct node *n, *m;
	struct value *v;
	struct cache *cache;

	/* First get the record.
	 */
	if ((r = db_getrec(rec, 0)) == 0)
		return 0;
	
	/* If attr is "name", return the record's name.  It's not stored
	 * explicitly in the record.
	 */
	if (is_name(attr))
		return val_cstr(rec, 0);

	/* Locate the attribute and value node.
	 */
	c = db_attr_find(r, attr, 1);
	n = db_lookup(c, t);

	/* If not found, or the value is currently null, try first the
	 * $number attribute if attr starts with a digit.  If this fails,
	 * or if the attribute does not start with a digit, try the $default
	 * attribute.
	 */
	if (n == 0 || val_null(n->n_value)) {
		if ((unsigned) (*attr - '0') <= 10)
			m = db_lookup(db_attr_find(r, "$number", 0), t);
		else
			m = 0;
		if (m == 0 || val_null(m->n_value)) {
			m = db_lookup(db_attr_find(r, "$default", 0), t);
			if (m == 0 || val_null(m->n_value))
				return n == 0 ? 0 : val_ref(n->n_value);
		}
		n = m;
	}

	/* If the value is not a function, or the attribute's name starts
	 * with a '$', return the value.
	 */
	if (*attr == '$' || n->n_value->v_assign != V_STR ||
						*n->n_value->v_str != '#')
		return val_ref(n->n_value);

	/* The value is a function.  If no changes were made to the data
	 * base currently, look in the cache of function evaluations.
	 */
	if (t != db_change && (m = db_seek(&c->c_cache, t)) != 0)
		return val_ref(m->n_value);
	
	/* The function needs to be evaluated.  Set the time (db_seq) back
	 * to t, and call x_expr to evaluate the postfix expression.  At
	 * the end restore db_seq.
	 */
	seq = db_seq;
	db_seq = t;
	if ((v = x_expr(rec, attr, n->n_value)) == 0)
		v = val_pstr((char *) 0, t);
	db_seq = seq;

	/* Store the function result in the cache and return it.
	 */
	db_insert(&c->c_cache, val_ref(v), t, 1);
	return v;
}

/* Retrieve the value of an attribute at ``time'' t and the corresponding
 * sequence number.
 */
struct value *db_get(rec, attr, t)
char *rec, *attr;
{
	struct value *v = db_find(rec, attr, t > db_seq ? db_seq : t);

	return v == 0 ? val_ref(db_null) : v;
}

/* Retrieve the current value of an attribute.
 */
db_retrieve(rec, attr, valp)
char *rec, *attr, **valp;
{
	struct value *v = db_find(rec, attr, db_seq);

	if (val_null(v)) {
		*valp = 0;
		val_free(v);
		return 0;
	}
	*valp = mag_copy(val_str(v));
	val_free(v);
	return 1;
}

/* Like db_retrieve, but returns a double value.
 */
db_dblretrieve(rec, attr, valp)
char *rec, *attr;
double *valp;
{
	struct value *v = db_find(rec, attr, db_seq);

	if (val_null(v)) {
		*valp = 0;
		val_free(v);
		return 0;
	}
	*valp = val_dbl(v);
	val_free(v);
	return 1;
}

/* Like db_retrieve, but returns an integer value.
 */
db_numretrieve(rec, attr, valp)
char *rec, *attr;
int *valp;
{
	struct value *v = db_find(rec, attr, db_seq);

	if (val_null(v)) {
		*valp = 0;
		val_free(v);
		return 0;
	}
	*valp = val_int(v);
	val_free(v);
	return 1;
}

/* Get the i-th attribute of record r at time t and return it in buf.
 */
char *db_getattr(r, i, t)
struct record *r;
{
	struct attr **ap, *a;
	struct node *n;

	for (ap = r->r_attrs; ap < &r->r_attrs[ATTR_HASH]; ap++)
		for (a = *ap; a != 0; a = a->a_next)
			if ((n = db_lookup(&a->a_contents, t)) != 0 &&
					n->n_value != 0 && i-- == 0)
				return a->a_name;
	return 0;
}

/* This postfix function returns the current sequence number.
 */
db_x_seq(){
	x_numpush(db_seq, db_seq);
}

/* This function is used to retrieve the i-th attribute of a record.
 */
db_x_attr(c)
struct context *c;
{
	struct record *r;
	int N, i = x_numpop(c);
	struct value *rval = x_pop(c);
	char *rec, *attr;
	struct cache *cache;

	if (i >= 0 && (rec = val_str(rval)) != 0) {
		if (i == 0)
			x_dirpush(val_sstr("name", 0));
		else {
			db_numretrieve(rec, "N", &N);
			if (N < 0)
				N = 0;
			if (--i < N)
				x_numpush(i, db_seq);
			else if ((r = db_getrec(rec, 0)) != 0 && (attr =
					db_getattr(r, i - N, db_seq)) != 0)
				x_dirpush(val_cstr(attr, db_seq));
			else
				x_null(c);
		}
	}
	else
		x_null(c);
	val_free(rval);
}

/* Count the number of attributes.  If there is an attribute called N,
 * count all attribute 0 thru N - 1.
 */
db_x_count(c)
struct context *c;
{
	struct record *r;
	struct attr **ap, *a;
	struct node *n;
	int N;
	struct value *rval = x_pop(c);
	char *rec;
	struct cache *cache;

	if ((rec = val_str(rval)) != 0) {
		db_numretrieve(rec, "N", &N);
		if (N < 0)
			N = 0;
		N++;		/* name of record */
		if ((r = db_getrec(rec, 0)) != 0)
			for (ap = r->r_attrs; ap < &r->r_attrs[ATTR_HASH]; ap++)
				for (a = *ap; a != 0; a = a->a_next) {
					n = db_lookup(&a->a_contents, db_seq);
					if (n != 0 && n->n_value != 0)
						N++;
				}
		x_numpush(N, db_seq);
	}
	else
		x_null(c);
	val_free(rval);
}

/* This function returns the ``attr''-rd record in the data base.
 */
db_x_datadict(c)
struct context *c;
{
	struct record **rp, *r;
	int i = x_numpop(c);

	for (rp = db_hashtab; rp < &db_hashtab[REC_HASH]; rp++)
		for (r = *rp; r != 0; r = r->r_next)
			if (i-- == 0) {
				x_dirpush(val_cstr(r->r_name, db_seq));
				return;
			}
}

/* This function returns the number of records in the data base.
 */
db_x_ddcount(c)
struct context *c;
{
	struct record **rp, *r;
	int N = 0;

	for (rp = db_hashtab; rp < &db_hashtab[REC_HASH]; rp++)
		for (r = *rp; r != 0; r = r->r_next)
			N++;
	x_numpush(N, db_seq);
}

/* Get a string from the data base file.
 */
char *db_getstring(){
	int length;
	char *s;

	fscanf(db_file, "%d", &length);
	if (length < 0)
		s = 0;
	else {
		getc(db_file);
		s = MAG_ALLOC(length + 1, char);
		fread(s, 1, length, db_file);
		s[length] = 0;
	}
	getc(db_file);
	return s;
}

/* Compute a hash value out of a string for a record string.
 */
db_rechash(s)
char *s;
{
	int n = 0;

	while (*s != 0) {
		n <<= 1;
		n ^= *s++;
	}
	if (n < 0)
		n = -n;
	return n % REC_HASH;
}

/* Restore the whole data base from the specified file.  Currently only
 * get the most recent value though.
 */
db_restore(){
	int seq;
	char *s;
	struct record **rp, *r;
	struct contents *c;

	if ((db_file = fopen(db_filename, "r")) == 0)
		mag_panic("can't get data base file");
	fscanf(db_file, "%d\n", &db_seq);
	while (getc(db_file) == '#') {
		s = db_getstring();
		if (is_name(s)) {
			r = db_rec_alloc();
			if (fscanf(db_file, "%d:", &seq) != 1)
				mag_panic("data base format error");
			r->r_name = db_getstring();
			rp = &db_hashtab[db_rechash(r->r_name)];
			r->r_next = *rp;
			*rp = r;
		}
		else {
			c = db_attr_find(r, s, 1);
			while (fscanf(db_file, "%d:", &seq) == 1)
				db_append(c, val_pstr(db_getstring(), seq));
		}
		MAG_FREE(s);
	}
	fclose(db_file);
	x_register("$db_seq", db_x_seq);
	x_register("$db_datadict", db_x_datadict);
	x_register("$db_ddcount", db_x_ddcount);
	x_register("$db_count", db_x_count);
	x_register("$db_attr", db_x_attr);
}

/* Cold start of the data base.
 */
db_init(name)
char *name;
{
	sprintf(db_filename, "db/%s.db", name);
	db_null = val_sstr((char *) 0, 0);
}
