/*

Copyright 1990 by M. Wood and K. Marzullo
Rights to use this source in unmodified form granted for all
commercial and research uses.  Rights to develop derivative
versions reserved by the authors.

*/

#include <stdio.h>
#include <strings.h>
#include "sm.h"
#include "sm_err.h"
#include "meta_entries.h"


/* Sensor library - routines sn_init, sn_newsensor, sn_chkcnd and
   support routines.

   By Mark D. Wood
*/


struct condition {
    struct condition *next,*prev;      /* next,prev in linked list */
    EVENT *eid;		        /* event number, for client's sake */
    address client;		/* address of client */
    int relation;		/* relation to be applied */
    int wastrue;		/* was true last time */
    int init;			/* for R_CHANGE */
    ISET * isvalue;		/* indexed value, if set type */
    double value;		/* value, note */
				/* that we bypass the C-type and */
				/* allocate ourselves the necessary space */
				/* this is a double to ensure proper */
				/* alignment */
};

typedef struct condition CONDITION;


#define MAX_NUMSENSORS 64
#define MAX_NOTIFIES 32

/* library globals */

extern short active_replica;
extern address * troupe_addr;
extern void (*meta_poller)();
extern short numcohorts;
extern address *meta_name();
void sn_chkcnd();

/* globals used by sensors */


static char my_name[INSTANCE_LENGTH];
static char *s_name[MAX_NUMSENSORS];
static int (*s_func[MAX_NUMSENSORS])();
static int s_type[MAX_NUMSENSORS];
static int s_pollfreq[MAX_NUMSENSORS];
static CONDITION *s_cond[MAX_NUMSENSORS];
static int num_sensors = 0;
static int sn_pollfreq = 7;
static int num_conds = 0;
static int nch;
static condition cond_checking = 0;
static int checking = FALSE;

GN_STRING sn_class;		/* also used by sn_rusage */

extern short numcohorts;

static void poll_task();
static void oneshot_died();


extern ISET * set_index();



static int lookup(name)
char *name;

{
    register s_num;
    for (s_num = 0; s_num < num_sensors; s_num++)
      if (!strcmp(s_name[s_num],name)) return s_num;
    return -1;
}



static void sn_poll(request)
message *request;

{
    message *m_value;
    char *sensor;
    char *inst;
    int s_num;
    long status;
    PTR  value;

#ifdef DEBUG
    extern int msg_namsgs, msg_nfmsgs;
    printf("mdiff1 = %d\n", msg_namsgs - msg_nfmsgs);
#endif 

    m_value = msg_newmsg();
    msg_get(request,"%-s%-s",&sensor,&inst);
#ifdef  DEBUG
    printf("sn_poll: %s %s\n",sensor,inst);
#endif

    s_num = lookup(sensor);

    if (s_num < 0) 
      status = SN_NOTKNOWN;
    else {
	status = (*s_func[s_num])(&value);
	if (!status)
	  type_pack(s_type[s_num],m_value,SM_VALFLD,value);
    }
    reply(request,"%d%s%m",(long) status,my_name,m_value);

    if (s_type[s_num] & TYPE_SET) 
      msg_delete(*(message **) value);

    msg_delete(m_value);
}


static void add_condition(queue,element)
CONDITION **queue, *element;

{
    element -> prev = NULL;
    element -> next = (*queue);
    if ((*queue))
      (*queue) -> prev = element;
    (*queue) = element;
    num_conds++;
    sn_chkcnd(nch);
}



static void delete_condition(queue,element)
CONDITION **queue,*element;

{
    if (element->next)
      element->next->prev = element->prev;
    if (element->prev)
      element->prev->next = element->next;
    if (*queue == element)
      *queue = element->next;
    free_iset(element->isvalue);
    free(element);
    num_conds--;
    sn_chkcnd(nch);
}



static void sn_alert(request)
message *request;

{
    char *sensor, *instance;
    int s_num,relation;
    EVENT * eid;
    CONDITION *newcond;
    PTR a_val;
    message *m_value;
    int type;

    msg_get(request,"%-s%-s%d%d%m",
	    &sensor,&instance,&eid,&relation,&m_value);
    if ((s_num = lookup(sensor)) < 0)
      return;

    type = s_type[s_num];
    if (relation & R_CARD) {
	if (type & TYPE_SET)
	  type = TYPE_INT;
	else
	  type = type & TYPE_BASE;
    } 

    newcond = (CONDITION *) malloc(sizeof(CONDITION)
				   + type_size(type));
    a_val = (PTR) &(newcond->value);
    newcond->eid = eid;
    newcond->relation = relation;
    newcond->client = *msg_getsender(request);
    newcond->wastrue = !(newcond->relation & SM_ALERTINIT);
    newcond->init = (((relation & R_MASK) == R_CHANGE) &&
		     (relation & SM_ALERTINIT)) ? TRUE : FALSE;

    type_unpack(type,m_value,SM_VALFLD,&a_val);

    if (type & TYPE_SET)
      newcond->isvalue = set_index(type,* (message **) a_val);
    else
      newcond->isvalue = NULL;

#ifdef DEBUG
    printf("Waiting on %s with relation %d (eid = %d)\n",
	   sensor,newcond->relation,newcond->eid);
    printf("value is ");
    type_fprint(stdout,type,a_val);
    printf("\n");
#endif

    add_condition(&s_cond[s_num],newcond);

    if (relation & SM_ALERTONCE) 
      proc_watch(newcond->client,oneshot_died,0);

    if (!s_pollfreq[s_num] || (relation & SM_ALERTINIT))
      sn_chkcnd(s_num);
}



static void sn_can_alert(request)
message *request;

{
    char *sensor, *instance;
    int s_num;
    EVENT *eid;
    address *addr;
    CONDITION *cp;

    if (checking)
      t_wait_l(&cond_checking,"sn_can_alert on checking");

    addr = msg_getsender(request);
    msg_get(request,"%-s%-s%d", &sensor,&instance,&eid);
    if ((s_num = lookup(sensor)) < 0)
      return;
    for (cp = s_cond[s_num]; cp; cp = cp -> next) {
	/* we can get away with setting cp to cp->next since once
	   we delete the element we break out of the loop */
	if ((cp->eid == eid) && addr_isequal(addr,&(cp->client))) {
	    delete_condition(&s_cond[s_num],cp);
	    break;
	}
    }
    flush();
    reply(request,"");
#ifdef DEBUG
    printf("canceled %d\n",eid);
#endif
}



static void check_conditions(s_num)
int s_num;

{
    CONDITION **condlist, *next;
    register CONDITION *condp;
    message *m_value;
    PTR value;
    int type,settype;
    int status;
    address clients[MAX_NOTIFIES+1];
    address *ap = clients;
    int numnotifies = 0;
    register message *packet = NULL;
    ISET * isvalue;
    int anychanges = FALSE;

    /* we need to lock out the routines that delete conditions
       during the execution of this routine, namely, sn_can_alert
       and somebody_died.  Adding a condition, as in sn_alert
       does not cause a problem. */

    checking = TRUE;

/* Make a packet containing
   the stuff to be sent out, if anything is indeed to be sent out.
   A packet is sent out if any events have happened or (if there are
   any changes to the "wastrue" variable and there are other cohorts).
   The packet is sent to all those clients who had events occur and
   also to the sensor's process group if there are other members.

   A word about the init field of CONDITION.  This mechanism's sole
   purpose is to handle the case R_CHANGE | SM_ALERTINIT.  If you
   specify R_CHANGE without the SM_ALERTINIT, the value you pass is
   used as the initial "old" value to compare against.  However, with
   SM_ALERTINIT, the system will always return the first value; this
   unfortunate mechanism is used to achieve that behaviour. */

    condlist = &s_cond[s_num];
    if (!*condlist)
      return;			/* nothing to do */

    type = s_type[s_num];
    settype = type & TYPE_SET;
    status = (*s_func[s_num])(&value);
    if (status) return;
    if (settype)
      isvalue = set_index(type, * (message **) value);
    
    for (condp = *condlist; condp; condp = next) {
	next = condp->next;
	if ( ((settype) ? 
	      ((condp->relation & R_CARD) ?
	       type_cmp(TYPE_INT,condp->relation,
			&(isvalue->size),(PTR) &(condp->value)) :
	       isetcmp(type,condp->relation,isvalue,condp->isvalue)) :
	      type_cmp(type,condp->relation,
		       value,(PTR) &(condp->value)))
	    || condp->init) {

	    if (condp->wastrue)
	      continue;
#ifdef DEBUG
            printf("False to True: ");
	    type_fprint(stdout,type,value);
	    relop_fprint(stdout,condp->relation);
	    if (!(condp->relation & R_CARD)) 
	      type_fprint(stdout,type,(PTR) &(condp->value));
	    putc('\n',stdout);
#endif
	    anychanges = TRUE;
	    if (!packet) {
		m_value = msg_newmsg();
		type_pack(s_type[s_num],m_value,
			  SM_VALFLD,value);
		packet = msg_gen("%d%d%s%d%m",
				 META_SENSOR,0,my_name,s_num,m_value);
	    }

	    if (condp->relation & R_MASK == R_CHANGE) {
		/* save current value as old.  Note that
		   we never set condp->wastrue in this case. */
		condp->init = FALSE;
		if (settype) {
		    if (condp->relation & R_CARD)
		      *(long *) &condp->value = isvalue->size;
		    else {
			free_iset(condp->isvalue);
			condp->isvalue = isvalue;
			isvalue->refcnt++;
			* (message **) &condp->value =
			  *(message **) value;
		    }
		} else {
		    bcopy(value,&(condp->value),type_size(type));
		}
	    } else {
		condp->wastrue = TRUE;
	    }

            if (++numnotifies < MAX_NOTIFIES) {
		*ap = condp->client;
		ap->addr_entry  = SM_ALERT;
		ap++;
		msg_put(packet,"%A[1]%d",&condp->client,
			(unsigned long) condp->eid);
#ifdef DEBUG
		printf("%d ",condp->eid);
		paddr(&condp->client);
		printf("\n");
#endif
		if (condp->relation & SM_ALERTONCE) {
		    delete_condition(condlist,condp);
		}
	    } else 
	      fprintf(stderr,"too many notifies!\n");
	} else {
	    if (condp->wastrue) {
		anychanges = TRUE;
#ifdef DEBUG
		printf("TRUE to FALSE: ");
		type_fprint(stdout,type,value);
		relop_fprint(stdout,condp->relation);
		type_fprint(stdout,type,(PTR) &(condp->value));
		putc('\n',stdout);
#endif
	    }
	    condp->wastrue = FALSE;
	}
    }

    if (anychanges && numcohorts && !packet) {
	m_value = msg_newmsg();
	type_pack(s_type[s_num],m_value,
		  SM_VALFLD,value);
	packet = msg_gen("%d%d%s%d%m",
			 META_SENSOR,0,my_name,s_num,m_value);
    }

    if (packet) {
	if (numcohorts) {	
	    /* only notify sensor group if it */
	    /* has others in it besides me */
	    *ap = *troupe_addr;
	    ap->addr_entry = SN_COHORT;
	    ap++;
	}
	*ap = NULLADDRESS;
	abcast_l("lxs",clients,packet,0);
	msg_delete(m_value);
	msg_delete(packet);
#ifdef DEBUG	
	printf("notifying %d clients\n",numnotifies);
#endif
    }
    if (settype) {
	free_iset(isvalue);
    }

    checking = FALSE;
    t_sig_all(&cond_checking,0);
}




static void poll_task(dummy)
int dummy;

{
    int s_num;

    if (!active_replica)
      return;

    while (1) {
	for (s_num = 0; s_num < num_sensors; s_num++) {
	    isis_accept_events(0);
	    if (s_cond[s_num] && s_pollfreq[s_num]) {
		check_conditions(s_num);
	    }
	}
	sleep(sn_pollfreq);
    }
}



static void sn_cohort_alert(msg)
message *msg;

{
    CONDITION **condlist;
    register CONDITION *condp,*next;
    message *m_value;
    PTR value;
    ISET * isvalue;
    int type,settype;
    int status;
    char * instance;
    int s_num;
    int variety;

    msg_get(msg,"%d%d%-s%d%m",
	    &variety,&status,&instance,&s_num,&m_value);

    if (variety == META_SENSOR)
      return;

    checking = TRUE;

    condlist = &s_cond[s_num];
    type = s_type[s_num];
    settype = type & TYPE_SET;
    value = NULL;
    type_unpack(type,m_value,SM_VALFLD,&value);
    if (settype)
      isvalue = set_index(type,value);

    for (condp = *condlist; condp; condp = next) {
	next = condp->next;
	if ( ((settype) ? 
	      ((condp->relation & R_CARD) ?
	       type_cmp(TYPE_INT,condp->relation,
			&(isvalue->size),(PTR) &(condp->value)) :
	       isetcmp(type,condp->relation,isvalue,condp->isvalue)) :
	      type_cmp(type,condp->relation,
		       value,(PTR) &(condp->value)))
	    || condp->init) {

	    if (condp->relation & R_MASK == R_CHANGE) {
		/* save current value as old.  Note that
		   we never set condp->wastrue in this case. */
		condp->init = FALSE;
		if (settype) {
		    if (condp->relation & R_CARD)
		      * (long *) &condp->value = isvalue->size;
		    else {
			free_iset(condp->isvalue);
			condp->isvalue = isvalue;
			isvalue->refcnt++;
		    }
		} else 
		  bcopy(value,&(condp->value),type_size(type));

	    } else
	      condp->wastrue = TRUE;

	    if (condp->relation & SM_ALERTONCE) {
		delete_condition(condlist,condp);
	    }
	} else {
	    condp->wastrue = FALSE;
	}
    }
    if (settype) {
	msg_delete(* (message**) value);
	free_iset(isvalue);
    }
    checking = FALSE;
    t_sig_all(&cond_checking,0);
}



static int alive_sensor(value)
long **value;

{
    static long alive_val = TRUE;
    *value = &alive_val;
    return 0;
}


static int num_conds_sensor(value)
long **value;

{
    static long val;
    val = num_conds;
    *value = &val;
    return 0;
}



int sn_newsensor(function,name,type,pollfreq)
int (*function)();
char *name;
int type;
int pollfreq;

{
    extern char *strdup();

    if (num_sensors >= MAX_NUMSENSORS)
      return -1;
    s_func[num_sensors] = function;
    s_type[num_sensors] = type;
    s_name[num_sensors] = strdup(name);
    s_cond[num_sensors] = NULL;
    s_pollfreq[num_sensors] = pollfreq;
    if (pollfreq && (pollfreq < sn_pollfreq))
      sn_pollfreq = pollfreq;
    return num_sensors++;
}



static void oneshot_died(paddr_p,event,arg)
address * paddr_p;
int event;
int arg;


{
    int s;
    CONDITION *cp;

    if (checking)
      t_wait_l(&cond_checking,"client failed; waiting on checking");

    for (s = 0; s < num_sensors; s++) {
	for (cp = s_cond[s]; cp; cp = cp->next) {
	    if (addr_isequal(paddr_p,&(cp->client))) {
		delete_condition(&s_cond[s],cp);
	    }
	}
    }
}



static void somebody_died(gview_p,arg)
groupview *gview_p;
int arg;

{
    if (addr_isnull(&gview_p->gv_departed))
      /* only worry about deaths */
      return;

    oneshot_died(&gview_p->gv_departed,W_FAIL,arg);
}



sn_init(class,isisport)
char *class;
int isisport;

{
    static donebefore = 0;
    address * gaddr;

    if (donebefore) return;
    donebefore = 1;

    strcpy(sn_class,class);
    meta_poller = poll_task;

    isis_init(isisport);
    sm_shortname(my_name,my_host,sizeof(my_name));
    isis_entry(SN_POLL,sn_poll,"sn_poll");
    isis_entry(SN_ALERT,sn_alert,"sn_alert");
    isis_entry(SN_COHORT,sn_cohort_alert,"sn_cohort_alert");
    isis_entry(SN_CANALERT,sn_can_alert,"sn_can_alert");
    isis_task(poll_task,"poll_task");
    isis_task(oneshot_died,"oneshot_died");
    isis_task(somebody_died,"somebody_died");
    sn_newsensor(alive_sensor,"alive",TYPE_INT,0);
    nch = sn_newsensor(num_conds_sensor,"numconds",TYPE_INT,0);
    gaddr = meta_name(class,my_name);

    if (pg_monitor(gaddr,somebody_died,0) <=0 ) {
	fprintf(stderr,"pg_watch failure!\n");
    }

    gaddr = pg_lookup("/meta/sm");
    if (!addr_isnull(gaddr))
      cbcast(gaddr,SM_NEWSENSOR,"%s%s",class,my_name,0);

}



void sn_chkcnd(s_num)
int s_num;

{
    if (s_num>=num_sensors || !active_replica)
      return;

    if (s_num == SN_CHK_ALL) {
	int i;
	for (i=0; i<num_sensors; i++)
	  check_conditions(i);
    } else
      check_conditions(s_num);
    isis_accept_events(0);
}
