/*

Copyright 1990 by M. Wood and K. Marzullo
Rights to use this source in unmodified form granted for all
commercial and research uses.  Rights to develop derivative
versions reserved by the authors.

*/

#include <stdio.h>
#include <sm.h>
#include <sm_err.h>
#include <table.h>
#include <classes.h>
#include <strings.h>

/*
  By Mark D. Wood
 */


#define SDFTABLE "/meta/sdf"

SENSOR **sensors;
int sm_numsensors = 0;
int sm_numclasses = 0;

extern EVENT *sm_eventlist;
extern int sm_clientof[];

extern char * strdup();

char *class_name[MAX_NUM_CLASSES];

int class_lookup(class)
char *class;

/* maps class name to class number, where the class number is the
   position of the name within the class number table.  Returns
   -1 if class is not found.
*/

{
    int i;

    for (i=0;i<sm_numclasses;i++) {
	if (!strcmp(class_name[i],class))
	  return i;
    }
    return -1;
}



static int pstrcmp(a,b)
SENSOR **a,**b;

{
    if ((*a)->class < (*b)->class) 
      return -1;
    else if ((*a)->class > (*b)->class) 
      return 1;
    else
      return strcmp((*a)->name,(*b)->name);
}



static getval(tuple,type,column,p)
message *tuple;
int type,column;
char **p;

{
    *p = NULL;
    type_unpack(type,tuple,column,p);
}



static getsensor(sp,state)
SENSOR **sp;
PTR state;

{
    char *sval;
    PTR val;
    message *tuple;
    extern char *strdup();

    /* if sp is not null, use the space it points to; otherwise
       allocate some space */

    if (!*sp)
      *sp = (SENSOR *) malloc(sizeof(SENSOR));

    if ((t_enumerate(state,&tuple)
	 != TBL_SUCCESS) ||
	(tuple == NULL))
      return 0;
    /* really should be stored in table in proper fashion */
    getval(tuple,TYPE_STRING,1,&sval);
    ((*sp)->name) = strdup(sval);
    getval(tuple,TYPE_STRING,2,&sval);
    if (((*sp)->type = name_to_type(sval)) < 0) {
	fprintf(stderr,
		"Error in type for %s\n",(*sp)->name);
	exit(-1);
    }
    getval(tuple,TYPE_STRING,0,&sval);
    if (((*sp)->class = class_lookup(sval)) <0) {
	if (sm_numclasses == MAX_NUM_CLASSES) {
	    perror("Internal class table overflow!\n");
	    exit(-1);
	} else {
	    (*sp) -> class = sm_numclasses++;
	    class_name[(*sp)->class] = strdup(sval);
	}
    }
    getval(tuple,TYPE_STRING,3,&sval);
    ascii_to_type((*sp)->type,sval,NULL,&val);
    (*sp)->default_val = msg_newmsg();
    type_pack((*sp)->type,
	      (*sp)->default_val,SM_VALFLD,val);
    return 1;
}



static freesensor(sp)
SENSOR *sp;

{
    fprintf(stderr,"freeing sensor %s\n",sp->name);
    free(sp->name);
    msg_delete(sp->default_val);
    free(sp);
}

    

static can_events(sp,error)
SENSOR *sp;
int error;

{
    EVENT *ep;
    message *msg;

    for (ep=sm_eventlist; ep; ep=ep->next) {
	if (ep->sp == sp) {
	    ep->relation |= SM_STOPONFAIL;
	    msg = msg_gen("%d%d%s%d%m",
			  ep,error,ep->instance,
			  (ep->sp->type & TYPE_SET) ? 0 : 1,
			  ep->sp->default_val);
	    msg_increfcount(ep->sp->default_val);
	    msg_rewind(ep->sp->default_val);
	    meta_alert(msg);
	}
    }
    
}



chk_sensor_ovf()

{
    if (sm_numsensors + 1 >= MAXSENSORS) {
	fprintf(stderr,"Sensor table overflow!\n");
	exit(-1);
    }
}



int update()

{
    static int oldver  = -1;	/* so we will get a copy 1st time */
    AttDes *schema;
    TableInfo info;

    SENSOR **oldsp,**oldsensors;
    SENSOR **newsp;
    SENSOR *new;
    int numoldleft;
    PTR state;

    
    if (t_info(SDFTABLE,&info,NULL) != TBL_SUCCESS) {
/*	fprintf(stderr,"SDF table does not exist!\n");
	exit(-1); */
	return 0;		/* may be in middle of updating */
    }

    if (info.version == oldver)
      return 0;			/* no changes! */
    oldver = info.version;

    numoldleft = sm_numsensors;
    oldsp = oldsensors = sensors;
    newsp = sensors =(SENSOR **) malloc(sizeof(SENSOR *) * MAXSENSORS);
    sm_numsensors = 0;
    new = NULL;
    
    /* go through list of old and new together.  Both are sorted.
       We expect most of the new to already exist, in which case
       we don't need to allocate new space.  So getsensor takes the
       address of a pointer to a SENSOR structure; only if the pointer
       is null is new space allocated.  We therefore set new to
       null accordingly */

    t_startenum(SDFTABLE,TBL_SORTALL,NULL,&info,&schema,&state);

    while (getsensor(&new,state)) {
	while(numoldleft && (strcmp(new->name,(*oldsp)->name) > 0)) {
	    /* old ones not present in new list */
	    if (!sm_clientof[(*oldsp)->class]) {
		can_events(*oldsp,SM_NOTAVAIL);
		freesensor(*oldsp);
	    } else {
		/* old one still active, so keep it around */
		chk_sensor_ovf();
		*(newsp++) = *oldsp;
		sm_numsensors++;
	    }
	    oldsp++;
	    numoldleft--;
	}

	chk_sensor_ovf();
	if (numoldleft) {
	    /* (*old)->name is <= to new->name */
	    if (!strcmp(new->name,(*oldsp)->name)) {
		/* old name equals new name */
		if ((*oldsp)->type == new->type) {
		    /* replace by overwriting; transparent to existing
		       conditions */
		    **oldsp = *new;
		    *(newsp++) = *oldsp;
		} else {
		    if (sm_clientof[(*oldsp)->class]) {
			/* must skip new as we still have active
			   old sensors alive */
			*(newsp++) = *oldsp;
			new = NULL;
		    } else {
			/* replace new */
			can_events(*oldsp,SM_TYPEHASCHANGED);
			freesensor(*oldsp);
			*(newsp++) = new;
			new = NULL;
		    }
		}
		oldsp++;
		numoldleft--;
	    } else {
		/* new name less than old, so it's a new one */
		*(newsp++) = new;
		new = NULL;
	    }
	} else {
	    /* no old ones left, so just copy */
	    *(newsp++) = new;
	    new = NULL;
	}
	sm_numsensors++;
    }

    while (numoldleft) {
	chk_sensor_ovf();
	*(newsp++) = *(oldsp++);
	sm_numsensors++;
	numoldleft--;
    }
    t_endenum(state);
    free(oldsensors);
    realloc(sensors,sm_numsensors * sizeof(SENSOR *));
    if (!sm_numsensors) sensors = NULL;
    if (new) free(new);
    return 1;			/* things have changed */
}



SENSOR *sm_sensorlookup(sensor)
char *sensor;

{
    register SENSOR **sp;
    SENSOR s_struct, *sp1;
    GN_STRING class;
    char *delim;
    int len;

    delim = index(sensor,CLASS_DELIM);
    len = delim - sensor;
    if (!delim || (len > sizeof(class)))
      return NULL;
    strncpy(class,sensor,len);
    class[len] = (char) 0;

    s_struct.name = delim + 1;
    s_struct.class = class_lookup(class);
    sp1 = &s_struct;

    sp = (sensors) ? (SENSOR **) bsearch((char *) &sp1,
					 (char *) sensors,
					 sm_numsensors,
					 sizeof(PTR),pstrcmp)
      : NULL;

    if (!sp) {
	if (update()) {
	    /* get it again as things may have changed */
	    if (!sensors) return NULL;
	    if (s_struct.class < 0)
	      s_struct.class = class_lookup(class);
	    sp = (SENSOR **) bsearch((char *) &sp1,
				     (char *) sensors,
				     sm_numsensors,
				     sizeof(PTR),pstrcmp);
	}
    }
    if (sp) return *sp;
    else return NULL;
}

