/*

Copyright 1990 by M. Wood and K. Marzullo
Rights to use this source in unmodified form granted for all
commercial and research uses.  Rights to develop derivative
versions reserved by the authors.

*/

#include <stdio.h>
#include "sm.h"
#include "sm_err.h"
#include "meta_entries.h"

  /*  The sensor/actuator monitor/manager library 
      Routines to handle async notifies of events.

      By Mark D. Wood.
   */

/* Library globals */

extern char *class_name[];
extern int sm_clientof[]; 
extern char * running_classes;
extern address * sm_sensor_address();

EVENT *sm_eventlist = NULL;/* list of all active events */
void free_event();
OCCURRENCE *waiting_q = NULL; /* events waiting to be handled */


/* Module globals */

static OCCURRENCE *running_q = NULL; /* those events running */


static okay_to_run(op)
OCCURRENCE *op;

{
    if (set_member(running_classes,op->eid->class)) {
	return 0;
    }
    if ((waiting_q && waiting_q->eid->priority > op->eid->priority) ||
        (running_q && running_q->eid->priority > op->eid->priority)) {
	return 0;
    }
    return 1;
}



static void process_event(op)
OCCURRENCE *op;

{
    EVENT *eid;
    int highest_priority;
    OCCURRENCE *p,*next;
    SENSOR * sp;
    GN_STRING name;

    eid = op->eid;		/* for convenience */
    sp = eid->sp;
    strcpy(name,class_name[sp->class]);
    strcat(name,":");
    strcat(name,sp->name);

#ifdef DEBUG
    printf("processing %d\n",(unsigned) eid);
#endif

    if (eid->variety == META_SENSOR) {
	if (!op->status || (eid->relation & SM_ALERTONFAIL)) {
	    (*eid->func)(op->status,name,op->instance,
			 op->m_value,sp->type,eid,eid->uid);
    
	    if ((eid->relation & SM_ALERTONCE) ||
		(op->status && (SM_STOPONFAIL & ~SM_ALERTONFAIL &
				eid->relation) &&
		 !(InstanceIsClass(eid->instance) &&
		   sm_clientof[sp->class]))) {

		/* delete condition under following conditions:
		   we are only alerting once or
		   we had an error and
		   we are stopping on error and
		   the condition was on a specific instance or
		   the condition was the whole class and all members
		   of the class failed
		   (implied by sm_clientof being false)
		   */

		eid->delete = TRUE;
		/* if wildcard, tell everyone else to stop too */
		if (InstanceIsClass(eid->instance) &&
		    !sm_clientof[sp->class])
		  sm_stop_watch(eid);
	    }
	}
    } else {
	/* an actuator notify */
	(*eid->func)(op->status,name,op->instance,eid->uid);
	eid->delete = TRUE;
    }

    /* these must be done after the stop_watch call - particularly
       the decrementing of num_running */

    pq_delete(&running_q,op);
    set_delete(running_classes,eid->class);
    msg_delete(op->msg);
    free((char *) op);
    eid->num_running--;

    if (eid->num_running)	/* following shouldn't happen */
      fprintf(stderr,"num running is greater than zero\n");

#ifdef DEBUG
    printf("processing finished of %d\n",(unsigned) eid);
#endif

    if (eid->delete && !(eid->num_running)) {
	el_delete(&sm_eventlist,eid);
	free_event(eid);
    }

    highest_priority = (running_q) ? running_q->eid->priority :
      ((waiting_q) ? waiting_q->eid->priority : 0);

    for(p = waiting_q;
        p && (p->eid->priority >= highest_priority);
	) {
	next  = p->next;
	if (okay_to_run(p)) {
	    pq_delete(&waiting_q,p);
	    wq_delete(&(p->eid->waiting),p);
	    p->eid->num_running++;
	    set_add(running_classes,p->eid->class);
	    pq_insert(&running_q,p);
	    t_fork(process_event,(PTR ) p);
	}
	p = next;
	/* check to see if we have emptied the queue or
	   wrapped around */
	if (!waiting_q || (p == waiting_q))
	  break;
    }
}



void schedule_occurrence(op)
OCCURRENCE *op;

{
    msg_increfcount(op->msg);
    if (okay_to_run(op)) {
	op->eid->num_running++;
	set_add(running_classes,op->eid->class);
	pq_insert(&running_q,op);
	t_fork(process_event,(PTR) op);
    } else {
	/* mark as being waiting, add to waiting queue and to
	   list of waiting occurrences for this condition */
	pq_insert(&waiting_q,op);
	wq_insert(&(op->eid->waiting),op);
    }
}



void meta_alert(msg)
message *msg;

{
    EVENT *eid,*eid1;
    OCCURRENCE *op,*occurrence;
    address *addr,*addr1;
    int s_num;			/* for benefit of replicas of sensor */
    int variety;

    occurrence = (OCCURRENCE *) malloc(sizeof(OCCURRENCE));
    occurrence->msg = msg;

    msg_get(msg,"%d%d%-s",
	    &variety,
	    &occurrence->status,
	    &occurrence->instance);

    if (variety == META_SENSOR) {
	msg_get(msg,"%d%-m%-a%d",
		&s_num,
		&occurrence->m_value,
		&addr1,&eid1); /* we'll do this one last */

	while (msg_get(msg,"%-a%d",&addr,&eid)>0) {
	    if (addr_isequal(&my_address,addr)) {
#ifdef DEBUG
		printf("`%d'\n",eid);
#endif
		op = (OCCURRENCE *) malloc(sizeof(OCCURRENCE));
		*op = *occurrence;
		op->eid = eid;
		/* for convenience of pq.c */
		op->priority = eid->priority;
		op->m_value = msg_copy(occurrence->m_value);
		schedule_occurrence(op);
	    }
	}

	if (addr_isequal(&my_address,addr1)) {
#ifdef DEBUG
	    paddr(&my_address);paddr(addr1);
	    printf("`%d'\n",eid1);
#endif
	    occurrence->priority = eid1->priority;
	    occurrence->eid = eid1;
	} else {
	    free(occurrence);
	    return;
	}
    } else {
	/* variety == META_ACTUATOR */

	msg_get(msg,"%d",&eid);
	occurrence->eid = eid;
	occurrence->priority = eid->priority;
    }

    /* optimized for the case where there is only one event; in that
       case no forking necessary,  no memory copying.  And where there
       are more than one in the packet, these benefits apply to
       the last */

    schedule_occurrence(occurrence);
}



void free_event(eid)
EVENT * eid;

{
#ifdef DEBUG
    printf("deleting eid %d\n",(int) eid);
#endif
    msg_delete(eid->await_val);
    free(eid->instance);
    free(eid);
}



/*  Failure handling mechanism:

Each sensor class joins two groups:  /meta/class and
/meta/class/instance.  The sensor monitor becomes a client of
/meta/class for each class that a watch is set on; a pg_monitor is
then done on that group, so that a failure by a member of that group
can be detected.  Restarts can not be handled in the same fashion,
because if every member of the group dies, the group then ceases to
exist.  So instead, when restarting, a sensor broadcasts its identity
to each sensor monitor (after the sensor has joined the appropriate
groups.)  Sensor monitors join the group /meta/sm for the sake of
receiving these messages.

To take advantage of the bypass facility, both the sensor and the
clients of a sensor must belong to the same group.  Thus we join the
group /Meta-BY/class but only if the watch is continuous (i.e., not
SM_ALERTONCE)---in that case it is not worth the overhead of joining
the group.

*/


void sensor_failed(gview_p,arg)
groupview *gview_p;
int arg;

{
    int class;
    message *msg;
    EVENT * ep, *next;
    char instance[INSTANCE_LENGTH];
    int site;
    int i;

    class = class_lookup(rindex(gview_p->gv_name,'/')+1);
    if (class < 0)	/* not a class we know about */
      return;

    if (addr_isnull(&gview_p->gv_departed))
      /* only worry about deaths */
      return;

    site = gview_p->gv_departed.addr_site;

    /* need to see if anybody else is alive at same site.  If so,
       then it was merely one of a set of replicas that died and
       we should ignore the failure. */

    for (i=0; i<gview_p->gv_nmemb; i++) {
	if (gview_p->gv_members[i].addr_site == site)
	  return;
    }

    if (!gview_p->gv_nmemb)
      sm_clientof[class] = FALSE;

    sm_shortname(instance,site_names[site],sizeof(instance));

#ifdef DEBUG
    paddr(&gview_p->gv_departed);
    printf(" (%s) has failed!\n",class_name[class]);
    printf("failed instance is %s\n",instance);
#endif
    for (ep = sm_eventlist; ep; ep=next) {
	next = ep->next;
	if ((ep->sp->class == class) &&
	    (ep->relation & SM_ALERTONFAIL) &&
	    (InstanceIsClass(ep->instance) ||
	     !strcmp(ep->instance,instance))) {
	    if (ep->variety == META_SENSOR) {
		msg = msg_gen("%d%d%s%d%m%A[1]%d",
			      META_SENSOR,SM_FAILED,instance,
			      0,	/* dummy sensor number */
			      ep->sp->default_val,
			      &my_address,ep);
		msg_increfcount(ep->sp->default_val);
		msg_rewind(ep->sp->default_val);
	    } else {
		/* actuator */
		msg = msg_gen("%d%d%s%d",
			      META_ACTUATOR,SM_FAILED,instance,ep);
	    }
	    t_fork(meta_alert,msg);
	}
    }
}



void newsensor(msg)
message *msg;

{
    EVENT *ep;
    address *gaddr,*caddr;
    char *class, *instance;
    int class_num;

    msg_get(msg,"%-s%-s",&class,&instance);

#ifdef DEBUG
    printf("%s came on line (%s)\n",instance,class);
#endif

    class_num = class_lookup(class);
    if (class_num < 0)
      return;

    gaddr = msg_getsender(msg);
    for (ep=sm_eventlist; ep; ep = ep->next) {
	if  ((ep->sp->class == class_num) &&
	     (InstanceIsClass(ep->instance) ||
	      !strcmp(ep->instance,instance)) &&
	     !(ep->relation & SM_STOPONFAIL & ~SM_ALERTONFAIL)) {
	    ep->sn_addr = *gaddr;
	    if (!sm_clientof[class_num]) {
		caddr = sm_sensor_address(class_num,"*");
		pg_client(caddr,"");
		if(pg_monitor(caddr,sensor_failed,0)<0)
		  return;
		sm_clientof[class_num] = TRUE;
	    }
	    if (!(ep->relation & SM_ALERTONCE))
	      bypass_enable(ep->sp->class);
	    cbcast(gaddr,SN_ALERT,"%s%s%d%d%m",
		   ep->sp->name,instance,(unsigned) ep,
		   ep->relation,ep->await_val,0);
	}
    }
}
