/*
 *      Originally coded by Robbert van Renesse
 *
 *
 *      ISIS release V2.0, May 1990
 *      Export restrictions apply
 *
 *      The contents of this file are subject to a joint, non-exclusive
 *      copyright by members of the ISIS Project.  Permission is granted for
 *      use of this material in unmodified form in commercial or research
 *      settings.  Creation of derivative forms of this software may be
 *      subject to restriction; obtain written permission from the ISIS Project
 *      in the event of questions or for special situations.
 *      -- Copyright (c) 1990, The ISIS PROJECT
 */

/*
 * This module takes care of the window management.  It creates a window,
 * and handles the events.
 */

#include <stdio.h>
#include "magic.h"
#include "value.h"
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xatom.h>
#include "win.h"

#define MAX_WINDOWS	64
#define NKEY		2

#define KEY1		0
#define KEY2		1

#define EVMASK	(	KeyPressMask | \
			ButtonPressMask | \
			ButtonMotionMask | \
			ButtonReleaseMask | \
			ExposureMask | \
			StructureNotifyMask )

#define b(x)		(1 << (x))
#define last(t)		(&(t)[sizeof(t) / sizeof((t)[0])])

extern char *mag_me;
extern mag_done;
extern db_seq, bar_seq;

struct value *db_get();

extern bar_draw(), bar_press(), bar_track();
extern cont_draw(), cont_press(), cont_track();
extern pic_draw(), pic_press(), pic_track(), pic_release();
extern graph_draw(), tbl_draw();
extern tbl_press(), tbl_track(), tbl_input(), tbl_getsel(), tbl_endsel();
extern menu_press(), menu_track(), menu_release(), menu_draw();
extern scroll_draw(), scroll_press(), scroll_track();
extern line_draw(), arrow_draw(), circle_draw(), hand_draw();

/* This table has one entry for each window.  See win.h.
 */
struct window win_windows[MAX_WINDOWS];

/* The event dispatch table.
 */
struct dispatch {
	char *d_type;		/* type of widget */
	int d_keymask;		/* which mouse keys are accepted */
	int (*d_draw)();	/* draw routine for this type */
	int (*d_press)(), (*d_track)(), (*d_release)();	/* other routines */
	int (*d_input)(), (*d_getsel)(), (*d_endsel)();	/* ditto */
} win_dispatch[] = {
	"bar",			b(KEY1),		bar_draw,
	bar_press,		bar_track,		bar_track,
	0,			0,			0,

	"picture",		b(KEY1),		pic_draw,
	pic_press,		pic_track,		pic_release,
	0,			0,			0,

	"table",		b(KEY1) | b(KEY2),	tbl_draw,
	tbl_press,		tbl_track,		tbl_track,
	tbl_input,		tbl_getsel,		tbl_endsel,

	"container",		b(KEY1),		cont_draw,
	cont_press,		cont_track,		cont_track,
	0,			0,			0,

	"menu",			b(KEY1),		menu_draw,
	menu_press,		menu_track,		menu_release,
	0,			0,			0,

	"scroll",		b(KEY1),		scroll_draw,
	scroll_press,		scroll_track,		scroll_track,
	0,			0,			0,

	"graph",		0,			graph_draw,
	0,			0,			0,
	0,			0,			0,

	"line",			0,			line_draw,
	0,			0,			0,
	0,			0,			0,

	"arrow",		0,			arrow_draw,
	0,			0,			0,
	0,			0,			0,

	"circle",		0,			circle_draw,
	0,			0,			0,
	0,			0,			0,

	"hand",			0,			hand_draw,
	0,			0,			0,
	0,			0,			0,
};

Display *x_dpy;			/* the connection to the X server */
Window x_win;			/* the communication window */
XFontStruct *x_fs;		/* info about font */
Atom x_select;			/* get selections though this atom */

FILE *win_fp;			/* for dumping windows */

struct object win_select[NKEY];	/* holder of current selection */
int (*win_received)();		/* receive function */
int win_current = -1;		/* interactive window */

/* Print an object structure.
 */
win_print(o)
struct object *o;
{
	printf("%s:%s:%s:%d:%d:%d:%d", o->o_name, o->o_type, o->o_record,
			o->o_x, o->o_y, o->o_width, o->o_height);
}

/* Draw a line from (x1, y1) to (x2, y2).  If win_fp is set, send the
 * information to that file pointer.
 */
win_line(w, x1, y1, x2, y2)
struct window *w;
{
	if (win_fp != 0)
		fprintf(win_fp, "l %d %d %d %d\n", x1, y1, x2, y2);
	else
		XDrawLine(x_dpy, w->w_pm, w->w_gc, x1, y1, x2, y2);
}

/* Draw a circle.
 */
win_circle(w, x, y, width, height)
struct window *w;
{
	if (win_fp != 0)
		fprintf(win_fp, "c %d %d %d %d\n", x, y, width, height);
	else
		XDrawArc(x_dpy, w->w_pm, w->w_gc, x, y, width, height,
								0, 360 * 64);
}

/* Invert a rectangle.
 */
win_fillrect(w, x, y, width, height)
struct window *w;
{
	if (win_fp != 0)
		fprintf(win_fp, "r %d %d %d %d\n", x, y, width, height);
	else
		XFillRectangle(x_dpy, w->w_pm, w->w_invgc, x, y, width, height);
}

/* Fill a polygon.
 */
win_fillpoly(w, p, n)
struct window *w;
XPoint p[];
{
	if (win_fp != 0) {
		int i;

		fprintf(win_fp, "p %d", n);
		for (i = 0; i < n; i++)
			fprintf(win_fp, " %d %d", p[i].x, p[i].y);
		fprintf(win_fp, "\n");
	}
	else {
		XFillPolygon(x_dpy, w->w_pm, w->w_gc, p, n,
						Convex, CoordModeOrigin);
		XDrawLines(x_dpy, w->w_pm, w->w_gc, p, 3, CoordModeOrigin);
	}
}

/* Draw a string. 
 */
win_string(w, x, y, width, height, s, fmt)
struct window *w;
char *s, *fmt;
{
	XCharStruct xcs;
	int len, dir, fam, fdm;

	if (win_fp != 0)
		fprintf(win_fp, "s %d %d %d %d %s %s\n", x, y, width, height,
					fmt == 0 ? "l" : fmt, s);
	else {
		len = strlen(s);
		XTextExtents(x_fs, s, len, &dir, &fam, &fdm, &xcs);
		if (fmt == 0 || *fmt == 'c')
			x += (width - xcs.width) / 2;
		else if (*fmt == 'l')
			x += 4;
		else
			x += (width - xcs.width) - 4;
		XSync(x_dpy, 0); /* Bug in X--Let X catch up */
		XDrawImageString(x_dpy, w->w_pm, w->w_gc, x,
			y - (height + xcs.descent - xcs.ascent) / 2, s, len);
	}
}

/* Find the appropriate entry for the named widget in the table win_dispatch.
 */
struct dispatch *win_type(o)
struct object *o;
{
	struct dispatch *d;

	if (o->o_type != 0)
		for (d = win_dispatch; d < last(win_dispatch); d++)
			if (scmp(d->d_type, o->o_type))
				return d;
	return 0;
}

/* Some process is trying to get the current selection.  Cooperate.
 */
win_getselect(xs)
XSelectionRequestEvent *xs;
{
	struct dispatch *d;

	if ((d = win_type(&win_select[KEY1])) != 0 && d->d_getsel != 0)
		(*d->d_getsel)(&win_select[KEY1], xs);
}

/* Some process got the ownership over the PRIMARY selection.  Unselect
 * the current selection.
 */
win_endselect(key){
	struct dispatch *d;

	if ((d = win_type(&win_select[key])) != 0 && d->d_endsel != 0 &&
						(d->d_keymask & b(key))) {
		(*d->d_endsel)(&win_select[key]);
		win_free(&win_select[key]);
	}
}

/* Button 1 has been pressed count times in the recent past.  If the mouse
 * is within a widget, call the appropriate routine.
 */
win_buttonpress(w, x, y, key, count)
struct window *w;
{
	struct dispatch *d;

	win_endselect(key);
	if (pic_find(&w->w_object, x, y, &win_select[key]) &&
	    (d = win_type(&win_select[key])) != 0 && d->d_press != 0 &&
						(d->d_keymask & b(key))) {
		x -= win_select[key].o_x;
		y -= win_select[key].o_y;
		(*d->d_press)(&win_select[key], x, y, key, count);
	}
}

/* The mouse is being moved with button depressed.
 */
win_buttonmotion(x, y, key){
	struct dispatch *d;

	if ((d = win_type(&win_select[key])) != 0 && d->d_track != 0 &&
						(d->d_keymask & b(key))) {
		x -= win_select[key].o_x;
		y -= win_select[key].o_y;
		(*d->d_track)(&win_select[key], x, y, key);
	}
}

/* Button is released.  Call the appropriate routine.
 */
win_buttonrelease(x, y, key){
	struct dispatch *d;

	if ((d = win_type(&win_select[key])) != 0 && d->d_release != 0 &&
						(d->d_keymask & b(key))) {
		x -= win_select[key].o_x;
		y -= win_select[key].o_y;
		(*d->d_release)(&win_select[key], x, y, key);
	}
}

/* A key has been depressed.  If a widget is currently selected, send the
 * character there.
 */
win_keyevent(xk)
XKeyEvent *xk;
{
	char buf[16], *p;
	int n;
	struct dispatch *d;

	if ((d = win_type(&win_select[KEY1])) != 0 && d->d_input != 0) {
		n = XLookupString(xk, buf, sizeof(buf),
				(KeySym *) 0, (XComposeStatus *) 0);
		for (p = buf; p < &buf[n]; p++)
			(*d->d_input)(&win_select[KEY1], *p);
	}
}

/* The window is reconfigured.  We're only interested in position and size.
 * We have to create a new shadow pixmap, and reconfigure the window.
 * Retrieving the actual position is tricky because of the window managers's
 * fiddling around with decorations and such.  Here I use the coordinates
 * of the parent's window, and hope that that's ok.  It works for twm, but
 * I don't know how it works for other managers.
 */
win_reconf(w)
struct window *w;
{
	int nchildren;
	Window root, parent, *children;
	XWindowAttributes wa;

	XFreePixmap(x_dpy, w->w_pm);
	XQueryTree(x_dpy, w->w_win, &root, &parent, &children, &nchildren);
	XFree((char *) children);
	XGetWindowAttributes(x_dpy, w->w_win, &w->w_wa);
	XGetWindowAttributes(x_dpy, parent, &wa);
	w->w_pm = XCreatePixmap(x_dpy, w->w_win,
			w->w_wa.width, w->w_wa.height, w->w_wa.depth);
	db_numstore(w->w_record, "x", wa.x);
	db_numstore(w->w_record, "y", wa.y);
	db_numstore(w->w_record, "width", w->w_wa.width);
	db_numstore(w->w_record, "height", w->w_wa.height);
}

/* Find out which mouse button this is.
 */
win_button(button)
unsigned button;
{
	switch (button) {
	case Button1:	return KEY1;
	case Button3:	return KEY2;
	default:	return -1;
	}
}

/* An event has been received.  Do the appropriate thing for every event.
 */
win_event(w, xev)
struct window *w;
XEvent *xev;
{
	static long lasttime[NKEY];	/* to detect double clicking */
	static count[NKEY];		/* how many time clicked? */
	XEvent out;
	XSelectionRequestEvent *req;
	XSelectionEvent *rep;
	XButtonEvent *xb;
	XMotionEvent *xm;
	long *l;
	int key;

	switch (xev->type) {
	case KeyPress:
		win_current = w - win_windows;
		win_keyevent(&xev->xkey);
		break;
	case ButtonPress:
		win_current = w - win_windows;
		xb = &xev->xbutton;
		if ((key = win_button(xb->button)) < 0)
			break;
		if (xb->time - lasttime[key] < 250)
			count[key]++;
		else
			count[key] = 1;
		lasttime[key] = xb->time;
		win_buttonpress(w, xb->x, xb->y, key, count[key]);
		break;
	case MotionNotify:
		win_current = w - win_windows;
		xm = &xev->xmotion;
		if (xm->state & Button1Mask)
			win_buttonmotion(xm->x, xm->y, KEY1);
		if (xm->state & Button3Mask)
			win_buttonmotion(xm->x, xm->y, KEY2);
		break;
	case ButtonRelease:
		win_current = w - win_windows;
		xb = &xev->xbutton;
		if ((key = win_button(xb->button)) < 0)
			break;
		win_buttonrelease(xb->x, xb->y, key);
		if (key == 0)
			XSetSelectionOwner(x_dpy,
				XA_PRIMARY, w->w_win, xb->time);
		break;
	case ClientMessage:
		if (win_received != 0) {
			l = xev->xclient.data.l;
			(*win_received)(l[0], l[1], l[2], l[3]);
		}
		break;
	case Expose:
		win_current = w - win_windows;
		break;
	case SelectionRequest:
		req = &xev->xselectionrequest;
		if (req->selection == XA_PRIMARY)
			win_getselect(req);
		else
			com_got_request(req->requestor, req->property);
		rep = &out.xselection;
		rep->type = SelectionNotify;
		rep->requestor = req->requestor;
		rep->selection = req->selection;
		rep->target = req->target;
		rep->property = req->property;
		rep->time = req->time;
		XSendEvent(x_dpy, req->requestor, False, NoEventMask, &out);
		break;
	case SelectionNotify:
		rep = &xev->xselection;
		com_got_reply(rep->property);
		break;
	case SelectionClear:
		if (xev->xselectionclear.selection != XA_PRIMARY)
			mag_panic("somebody stole my address");
		win_endselect(KEY1);
		break;
	case ConfigureNotify:
		win_current = w - win_windows;
		win_reconf(w);
		break;
	case MapNotify:
		win_current = w - win_windows;
		w->w_mapped = 1;
		w->w_locked = 0;
		break;
	case UnmapNotify:
		win_current = w - win_windows;
		w->w_mapped = w->w_locked = 0;
		break;
	case ReparentNotify:
	case 0:
		break;
	default:
		fprintf(stderr, "Event? <%d>\n", xev->type);
	}
}

/* This routine creates a window as described in the data base by
 * control#<win>.  With it it creates the shadow pixmap of the window,
 * and the graphical contexts (which cannot be shared).
 */
win_create(win){
	XSizeHints xsh;
	XGCValues xgcv;
	char *window, *rec, buf[64], *descr;
	int x, y, width, height;
	struct window *w = &win_windows[win];

	/* Find the width and the height of the window.
	 */
	mag_strnum(buf, win);
	db_retrieve("control", buf, &window);
	if (window == 0)
		return;
	db_retrieve(window, "record", &rec);
	db_numretrieve(rec, "x", &x);
	db_numretrieve(rec, "y", &y);
	db_numretrieve(rec, "width", &width);
	db_numretrieve(rec, "height", &height);
	MAG_FREE(window);
	MAG_FREE(rec);

	/* Create the window and all other stuff.
	 */
	w->w_win = XCreateSimpleWindow(x_dpy, DefaultRootWindow(x_dpy), x, y,
		width, height, 2, BlackPixel(x_dpy, 0), WhitePixel(x_dpy, 0));
	db_retrieve(window, "descr", &descr);
	sprintf(buf, "%s (%s)", mag_me, descr);
	MAG_FREE(descr);
	XStoreName(x_dpy, w->w_win, buf);
	xsh.flags = USPosition | USSize;
	xsh.x = x;
	xsh.y = y;
	xsh.width = width;
	xsh.height = height;
	XSetNormalHints(x_dpy, w->w_win, &xsh);
	xgcv.background = WhitePixel(x_dpy, 0);
	xgcv.graphics_exposures = False;
	xgcv.foreground = BlackPixel(x_dpy, 0);
	xgcv.font = x_fs->fid;
	w->w_gc = XCreateGC(x_dpy, w->w_win, GCForeground | GCBackground |
			GCFont | GCGraphicsExposures, &xgcv);
	xgcv.function = GXclear;
	w->w_clrgc = XCreateGC(x_dpy, w->w_win,
		GCFunction | GCFont | GCGraphicsExposures, &xgcv);
	xgcv.function = GXinvert;
	w->w_invgc = XCreateGC(x_dpy, w->w_win,
		GCFunction | GCFont | GCGraphicsExposures, &xgcv);
	XSetSelectionOwner(x_dpy,
		XInternAtom(x_dpy, mag_me, False), w->w_win, CurrentTime);
	XGetWindowAttributes(x_dpy, w->w_win, &w->w_wa);
	w->w_pm = XCreatePixmap(x_dpy, w->w_win,
			w->w_wa.width, w->w_wa.height, w->w_wa.depth);
	XSelectInput(x_dpy, w->w_win, EVMASK);
}

/* Lookup the window structure belonging to the X window identifier win.
 */
struct window *win_lookup(win)
Window win;
{
	struct window *w;

	for (w = win_windows; w < &win_windows[MAX_WINDOWS]; w++)
		if (w->w_win == win)
			return w;
	printf("event on unknown window\n");
	/*NOTREACHED*/
}

/* X initialize.  Open a connection to the server, and create the first
 * window.  This window is needed by the communication module and some
 * other stuff.
 */
win_init(){
	if ((x_dpy = XOpenDisplay((char *) 0)) == 0)
		mag_panic("can't open display");
	if ((x_fs = XLoadQueryFont(x_dpy, "9x15")) == 0)
		mag_panic("can't get font info");
	x_select = XInternAtom(x_dpy, "selection", False);
	win_create(0);
	x_win = win_windows[0].w_win;	/* need that for communication */
}

/* Send a client message to the window.
 */
win_message(a0, a1, a2, a3)
long a0, a1, a2, a3;
{
	XEvent xev;

	xev.xclient.type = ClientMessage;
	xev.xclient.window = x_win;
	xev.xclient.format = 32;
	xev.xclient.data.l[0] = a0;
	xev.xclient.data.l[1] = a1;
	xev.xclient.data.l[2] = a2;
	xev.xclient.data.l[3] = a3;
	XSendEvent(x_dpy, x_win, False, NoEventMask, &xev);
	XFlush(x_dpy);
}

/* Specify the routine to be called on receiving client messages.
 * Also clear the queue.
 */
win_receive(f)
int (*f)();
{
	win_received = f;
	XFlush(x_dpy);
}

/* Free all the stuff in an object structure.
 */
win_free(o)
struct object *o;
{
	val_free(o->o_valname);
	val_free(o->o_valtype);
	val_free(o->o_valrecord);
	o->o_name = o->o_type = o->o_record = 0;
}

/* Retrieve the win.h stuff.
 */
win_retrieve(rec, win, o)
char *rec;
struct object *o;
{
	char buf[16];

	mag_strnum(buf, win);
	o->o_valname = db_get(rec, buf, db_seq);
	if ((o->o_name = val_str(o->o_valname)) == 0) {
		val_free(o->o_valname);
		return 0;
	}
	o->o_valtype = db_get(o->o_name, "type", db_seq);
	if ((o->o_type = val_str(o->o_valtype)) == 0) {
		val_free(o->o_valname);
		val_free(o->o_valtype);
		return 0;
	}
	o->o_valrecord = db_get(o->o_name, "record", db_seq);
	o->o_record = val_str(o->o_valrecord);
	db_numretrieve(o->o_name, "x", &o->o_x);
	db_numretrieve(o->o_name, "y", &o->o_y);
	db_numretrieve(o->o_name, "width", &o->o_width);
	db_numretrieve(o->o_name, "height", &o->o_height);
	return 1;
}

/* Draw the specified widget.  If frame is true, then frame it.
 */
win_draw(w, o)
struct window *w;
struct object *o;
{
	int val;
	struct dispatch *d;

	if (o->o_width == 0 && o->o_height == 0)
		return;
	db_numretrieve(o->o_name, "frame", &val);
	if (val != 0) {
		win_line(w, o->o_x, o->o_y,
			o->o_x + o->o_width, o->o_y);
		win_line(w, o->o_x + o->o_width, o->o_y,
			o->o_x + o->o_width, o->o_y + o->o_height);
		win_line(w, o->o_x + o->o_width, o->o_y + o->o_height,
			o->o_x, o->o_y + o->o_height);
		win_line(w, o->o_x, o->o_y + o->o_height,
			o->o_x, o->o_y);
	}
	if ((d = win_type(o)) != 0 && d->d_draw != 0)
		(*d->d_draw)(w, o);
}

/* Dump the contents of a window onto a file.
 */
win_dump(win, file)
char *file;
{
	int width, height;
	struct window *w = &win_windows[win];

	if (!win_retrieve("control", win, &w->w_object) ||
					w->w_record == 0 || w->w_locked)
		return 0;
	db_numretrieve(w->w_record, "width", &width);
	db_numretrieve(w->w_record, "height", &height);

	if ((win_fp = fopen(file, "w")) == 0)
		return 0;
	fprintf(win_fp, "%d %d\n", width, height);
	win_draw(w, &w->w_object);
	fclose(win_fp);
	win_fp = 0;

	win_free(&w->w_object);
	return 1;
}

/* Clean up all windows and close the connection to the window server.
 */
win_close(){
	struct window *w;
	int N;

	if (x_dpy == 0)
		return;
	db_numretrieve("control", "N", &N);
	if (N <= 0 || N > MAX_WINDOWS)
		N = MAX_WINDOWS;
	for (w = win_windows; w < &win_windows[N]; w++)
		if (w->w_win != 0)
			XDestroyWindow(x_dpy, w->w_win);
	XCloseDisplay(x_dpy);
}

/* Get the next event and handle it.
 */
win_handle(){
	XEvent xev;
	struct window *w;

	XNextEvent(x_dpy, &xev);
	w = win_lookup(xev.xany.window);
	if (!win_retrieve("control", w - win_windows, &w->w_object))
		printf("no window %d\n", w - win_windows);
	else {
		win_event(w, &xev);
		win_free(&w->w_object);
		db_commit();
	}
}

/* Update the named window.
 */
win_update(win){
	struct window *w = &win_windows[win];
	int seq, mapped;

	if (!win_retrieve("control", win, &w->w_object))
		return;
	if (w->w_record == 0 || w->w_locked) {
		win_free(&w->w_object);
		return;
	}
	db_numretrieve(w->w_record, "mapped", &mapped);
	if (!w->w_mapped) {
		if (mapped) {
			if (w->w_win == 0)
				win_create(win);
			if (w->w_win == 0)
				return;
			XMapRaised(x_dpy, w->w_win);
			w->w_locked = 1;
		}
		win_free(&w->w_object);
		return;
	}
	if (w->w_win == 0) {
		win_free(&w->w_object);
		return;
	}
	if (!mapped) {
		if (w->w_win != 0) {
			XUnmapWindow(x_dpy, win_windows[win].w_win);
			XSync(x_dpy, 0);	/* Bug in X? */
			w->w_locked = 1;
		}
		win_free(&w->w_object);
		return;
	}
	XFillRectangle(x_dpy, w->w_pm, w->w_clrgc, 0, 0,
				w->w_wa.width, w->w_wa.height);
	seq = db_seq;
	if (bar_seq != 0)
		db_seq = bar_seq;
	win_draw(w, &w->w_object);
	db_seq = seq;
	XCopyArea(x_dpy, w->w_pm, w->w_win, w->w_gc, 0, 0,
				w->w_wa.width, w->w_wa.height, 0, 0);
	win_free(&w->w_object);
}

/* Handle any pending events and redraw all windows.
 */
win_sync(){
	int N, win;

	do {
		while (XPending(x_dpy) != 0)
			win_handle();
		if (win_current >= 0) {
			win_update(win_current);
			XFlush(x_dpy);
		}
		db_numretrieve("control", "N", &N);
		if (N <= 0 || N > MAX_WINDOWS)
			N = MAX_WINDOWS;
		for (win = 0; win < N; win++)
			if (win != win_current)
				win_update(win);
		win_current = -1;
	} while (XPending(x_dpy) != 0);
	mag_checkmem();
}

/* In a loop, update the windows, get the next event, and handle it.
 */
win_main_loop(){
	for (;;) {
		win_sync();
		if (mag_done)
			return;
		win_handle();		/* block */
	}
}
