/*
 *      Originally coded by Robbert van Renesse
 *
 *
 *      ISIS release V2.0, May 1990
 *      Export restrictions apply
 *
 *      The contents of this file are subject to a joint, non-exclusive
 *      copyright by members of the ISIS Project.  Permission is granted for
 *      use of this material in unmodified form in commercial or research
 *      settings.  Creation of derivative forms of this software may be
 *      subject to restriction; obtain written permission from the ISIS Project
 *      in the event of questions or for special situations.
 *      -- Copyright (c) 1990, The ISIS PROJECT
 */

/*
 * This module detects events.  An event is a state that changes value.
 * The state is a function (without side-effects) that is evaluated after
 * each screen update.  When an event has been detected, messages get sent
 * to interested parties.
 */

#include "magic.h"
#include "value.h"

#define MAX_WATCH	100

struct site {
	struct value *s_addr;	/* address of watcher */
	char *s_cmd;		/* command to watcher */
	struct site *s_next;	/* next watcher in list */
};

struct watch {
	char *w_rec, *w_attr;	/* specify the record and attribute */
	struct value *w_val;	/* its current value */
	struct site *w_sites;	/* interested parties */
} watch_tab[MAX_WATCH], *watch_last = watch_tab;

extern db_seq;
extern char *mag_me;

struct value *db_get(), **mes_alloc();

watch_reply(station, rep)
struct value *station, **rep;
{
	extern char *mag_me;
	struct watch *w;
	struct site **sp, *s;

	if (rep != 0)
		return;
	/* Destination crashed, remove subscriptions.  Later something
	 * should be done about freeing s_cmd, but currently it's
	 * always allocated with MAG_MALLOC().
	 */
	printf("%s: %s crashed\n", mag_me, val_str(station));
	for (w = watch_tab; w < watch_last; w++) {
		sp = &w->w_sites;
		while ((s = *sp) != 0)
			if (val_cmp(s->s_addr, station)) {
				val_free(s->s_addr);
				*sp = s->s_next;
			}
			else
				sp = &s->s_next;
	}
}

/* Send a watch message.
 */
watch_send_message(w, s)
struct watch *w;
struct site *s;
{
	struct value **mes;

	mes = mes_alloc(5);
	mes[1] = val_cstr(s->s_cmd, 0);
	mes[2] = val_cstr(w->w_rec, 0);
	mes[3] = val_cstr(w->w_attr, 0);
	mes[4] = val_ref(w->w_val);
	mes_send_request(val_ref(s->s_addr), mes, watch_reply);
}

/* Specify a new watch.
 */
watch_specify(station, cmd, rec, attr)
struct value *station;
char *cmd, *rec, *attr;
{
	struct watch *w;
	struct site *s;

	for (w = watch_tab; w < watch_last; w++) {
		if (scmp(w->w_rec, rec) && scmp(w->w_attr, attr))
			break;
	}
	if (w == &watch_tab[MAX_WATCH]) {
		printf("too many watches (should panic)\n");
		return;
	}
	if (w == watch_last) {
		watch_last++;
		w->w_rec = rec;
		w->w_attr = attr;
	}
	s = MAG_ALLOC(1, struct site);
	s->s_addr = val_ref(station);
	s->s_cmd = cmd;
	s->s_next = w->w_sites;
	w->w_sites = s;
	w->w_val = db_get(rec, attr, db_seq);
	watch_send_message(w, s);
}

/* See if any attributes have changed, and, if so, send messages.
 */
watch_evaluate(){
	static struct value null;
	struct value *v;
	struct watch *w;
	struct site *s;

	for (w = watch_tab; w < watch_last; w++) {
		v = db_get(w->w_rec, w->w_attr, db_seq);
		if (val_cmp(v, w->w_val))
			val_free(v);
		else {
			val_free(w->w_val);
			w->w_val = v;
			for (s = w->w_sites; s != 0; s = s->s_next)
				watch_send_message(w, s);
		}
	}
}

/* Make sure that this station at init time sends a watch to station at
 * init time.
 */
watch_cold(station, cmd, rec, attr)
char *station, *cmd, *rec, *attr;
{
	struct value **mes = mes_alloc(5);

	mes[1] = val_sstr("watch", 0);
	mes[2] = val_cstr(cmd, 0);
	mes[3] = val_cstr(rec, 0);
	mes[4] = val_cstr(attr, 0);
	mag_initmes(val_sstr(station, 0), mes);
}
