/*
 *      Originally coded by Robbert van Renesse
 *
 *
 *      ISIS release V2.0, May 1990
 *      Export restrictions apply
 *
 *      The contents of this file are subject to a joint, non-exclusive
 *      copyright by members of the ISIS Project.  Permission is granted for
 *      use of this material in unmodified form in commercial or research
 *      settings.  Creation of derivative forms of this software may be
 *      subject to restriction; obtain written permission from the ISIS Project
 *      in the event of questions or for special situations.
 *      -- Copyright (c) 1990, The ISIS PROJECT
 */

#include "magic.h"
#include "value.h"

#define assert(c)	if (!(c)) abort()

/* Allocate a value of specified type and sequence number.
 */
struct value *val_alloc(type, seq){
	struct value *v = MAG_ALLOC(1, struct value);

	v->v_refcnt = 1;
	v->v_assign = v->v_present = type;
	v->v_seq = seq;
	return v;
}

/* Free a value.  If there is a mag_malloc-ed string, free that too.
 */
val_dofree(v)
struct value *v;
{
	assert(v->v_refcnt == 0);
	if ((v->v_present & V_STR) && ! (v->v_present & V_STATIC))
		MAG_FREE(v->v_str);
	MAG_FREE(v);
}

/* Generate a string for this value structure.
 */
val_getstr(v)
struct value *v;
{
	char buf[64];

	assert(v->v_refcnt > 0 && v->v_refcnt < 64);
	switch (v->v_assign) {
	case V_INT:
		mag_strnum(buf, v->v_int);
		break;
	case V_DBL:
		sprintf(buf, V_FMT_DBL, v->v_dbl);
		break;
	default:
		mag_panic("val_str");
	}
	v->v_str = mag_copy(buf);
	v->v_present |= V_STR;
}

/* Generate a double for this value structure.
 */
val_getdbl(v)
struct value *v;
{
	double atof();
	unsigned n;

	assert(v->v_refcnt > 0 && v->v_refcnt < 64);
	switch (v->v_assign) {
	case V_INT:
		v->v_dbl = v->v_int;
		break;
	case V_STR:
		/* If the string consists of a single digit, be smart.
		 */
		if ((n = v->v_str[0] - '0') < 10 && v->v_str[1] == 0)
			v->v_dbl = n;
		else
			v->v_dbl = atof(v->v_str);
		break;
	default:
		mag_panic("val_dbl");
	}
	v->v_present |= V_DBL;
}

/* Generate an integer for this value structure.  If there is already a
 * double present, it is sufficient to convert the double, even if the
 * value was assigned as a string.  This may be more efficient, although
 * I'm not sure.
 */
val_getint(v)
struct value *v;
{
	assert(v->v_refcnt > 0 && v->v_refcnt < 64);
	v->v_int = v->v_present & V_DBL ? v->v_dbl : atoi(v->v_str);
	v->v_present |= V_INT;
}

/* Put string x in a value structure at time seq.
 */
struct value *val_pstr(x, seq)
char *x;
{
	struct value *v = val_alloc(x == 0 ? 0 : V_STR, seq);

	v->v_str = x;
	return v;
}

/* Like val_pstr, but copy the string x into new memory.
 */
struct value *val_cstr(x, seq)
char *x;
{
	struct value *v = val_alloc(x == 0 ? 0 : V_STR, seq);

	v->v_str = x == 0 ? 0 : mag_copy(x);
	return v;
}

/* Like val_pstr, but the string is static and should not be freed if
 * the value is.
 */
struct value *val_sstr(x, seq)
char *x;
{
	struct value *v = val_alloc(x == 0 ? 0 : V_STR, seq);

	v->v_str = x;
	v->v_present |= V_STATIC;
	return v;
}

/* Put a double in a value structure at time seq.
 */
struct value *val_pdbl(x, seq)
double x;
{
	struct value *v = val_alloc(V_DBL, seq);

	v->v_dbl = x;
	return v;
}

/* Put an integer in a value structure at time seq.
 */
struct value *val_pint(x, seq){
	struct value *v = val_alloc(V_INT, seq);

	v->v_int = x;
	return v;
}

/* Copy value v, but not its sequence number.  Instead use seq.  If free is
 * set v can be removed afterwards.  If its reference count = 1, we can
 * just use it.
 */
struct value *val_pseq(v, seq, free)
struct value *v;
{
	struct value *new;

	if (v == 0)
		return val_pstr((char *) 0, seq);
	assert(v->v_refcnt > 0 && v->v_refcnt < 64);
	if (free) {
		if (v->v_seq == seq)
			return v;
		if (--v->v_refcnt == 0) {
			v->v_refcnt++;
			v->v_seq = seq;
			return v;
		}
	}
	if (v->v_seq == seq) {
		v->v_refcnt++;
		return v;
	}
	new = val_alloc(v->v_assign, seq);
	new->v_present = v->v_present;
	if (v->v_assign == V_STR)
		if (v->v_present & V_STATIC)
			new->v_str = v->v_str;
		else
			new->v_str = mag_copy(v->v_str);
	else
		new->v_present &= ~V_STR;
	new->v_dbl = v->v_dbl;
	new->v_int = v->v_int;
	return new;
}

/* Compare values v1 and v2.
 */
val_docmp(v1, v2)
struct value *v1, *v2;
{
	if (v1 == 0)
		return val_null(v2);
	if (v2 == 0)
		return val_null(v1);
	assert(v1->v_refcnt > 0 && v1->v_refcnt < 64);
	assert(v2->v_refcnt > 0 && v2->v_refcnt < 64);
	switch (m_min(v1->v_assign, v2->v_assign)) {
	case 0:
		return v1->v_assign == v2->v_assign;
	case V_STR:
		if (!(v1->v_present & V_STR))
			val_getstr(v1);
		if (!(v2->v_present & V_STR))
			val_getstr(v2);
		return scmp(v1->v_str, v2->v_str);
	case V_DBL:
		if (!(v1->v_present & V_DBL))
			val_getdbl(v1);
		if (!(v2->v_present & V_DBL))
			val_getdbl(v2);
		return v1->v_dbl == v2->v_dbl;
	case V_INT:
		return v1->v_int == v2->v_int;
	default:
		mag_panic("val_cmp");
		/*NOTREACHED*/
	}
}

/* Reduce the amount of memory used by value v.  If assigned as a string,
 * see whether it could have been assigned as either a double or an
 * integer.  If not assigned as a string (anymore), free the string if
 * it's present.
 */
val_reduce(v)
struct value *v;
{
	char buf[64];

	if (v == 0)
		return;
	assert(v->v_refcnt > 0 && v->v_refcnt < 64);
	if (v->v_assign == V_STR) {
		if (!(v->v_present & V_INT)) {
			v->v_int = atoi(v->v_str);
			v->v_present |= V_INT;
		}
		mag_strnum(buf, v->v_int);
		if (scmp(buf, v->v_str))
			v->v_assign = V_INT;
		else {
			if (!(v->v_present & V_DBL)) {
				v->v_dbl = atof(v->v_str);
				v->v_present |= V_DBL;
			}
			sprintf(buf, V_FMT_DBL, v->v_dbl);
			if (scmp(buf, v->v_str))
				v->v_assign = V_DBL;
		}
	}
	if (v->v_assign != V_STR && (v->v_present & V_STR) &&
					!(v->v_present & V_STATIC)) {
		v->v_present &= ~V_STR;
		MAG_FREE(v->v_str);
	}
}
