/*
 *      Originally coded by Robbert van Renesse
 *
 *
 *      ISIS release V2.0, May 1990
 *      Export restrictions apply
 *
 *      The contents of this file are subject to a joint, non-exclusive
 *      copyright by members of the ISIS Project.  Permission is granted for
 *      use of this material in unmodified form in commercial or research
 *      settings.  Creation of derivative forms of this software may be
 *      subject to restriction; obtain written permission from the ISIS Project
 *      in the event of questions or for special situations.
 *      -- Copyright (c) 1990, The ISIS PROJECT
 */

/*
 * This file contains some handy routines.
 */

#include <stdio.h>
#include "magic.h"
#include "value.h"
#include <X11/Xlib.h>
#include "win.h"

#define MAG_DELTA	(256 * 1024)

extern db_seq;
char *malloc();
struct value *db_get();
struct value **mes_alloc(), **mes_ok(), **mes_error();

char *mag_me;			/* name of this station */
char *mag_end, *mag_limit;	/* memory limits */
int mag_done;			/* set when done */

/* Self-explanatory.
 */
mag_panic(s)
char *s;
{
	fprintf(stderr, "%s panic: %s\n", mag_me, s);
	win_close();
	exit(1);
}

/* Produce a bleep on the terminal.
 */
mag_bleep(){
	fprintf(stderr, "\7");
}

#ifdef MAG_DEBUG

/* This version of MAG_MALLOC and MAG_ALLOC includes a lot of error
 * checks.  Each block of memory contains the size of that block, a
 * magic number, and which structures it contains and how many of them.
 * Furthermore, the blocks are maintained in a doubly linked list so
 * they can be inspected if a memory leak is suspected.  At the end of
 * the block an extra character is added (0xA) which checks for overwriting
 * boundaries.
 */

struct header {
	long h_size;
	long h_magic;
	int h_n;
	char *h_descr;
	struct header *h_next, *h_prev;
} *mag_mem;		/* points to last one of doubly linked list */

int mag_max;		/* points to largest size ever allocated */

/* Reduce memory by clearing caches and representing values in a smart way.
 */
mag_reduce(){
	db_reduce();
	x_reduce();
}

/* Allocate a block of specified size with n structures of the specified
 * structure descr.  Allocate the header and the extra character at the
 * end of the block.
 */
char *mag_malloc(n, size, descr)
char *descr;
{
	char *p;
	struct header *h;

	if (size <= 0 || n <= 0)
		abort();
	size += sizeof(struct header) + 1;
	if ((p = malloc(size)) == 0)
		mag_panic("out of memory");
	h = (struct header *) p;
	h->h_size = size;
	h->h_magic = 0x5555;
	h->h_n = n;
	h->h_descr = descr;
	h->h_prev = 0;
	if ((h->h_next = mag_mem) != 0)
		mag_mem->h_prev = h;
	mag_mem = h;
	p[size - 1] = 0xA;
	if (size > mag_max)
		mag_max = size;
	p += sizeof(struct header);
	if (p + size > mag_end)
		mag_end = p + size;
	return p;
}

/* Like mag_malloc, but clear the memory.
 */
char *mag_alloc(n, size, descr)
char *descr;
{
	char *p = mag_malloc(n, size, descr);

	bzero(p, size);
	return p;
}

/* Free memory allocated by mag_malloc().  Call db_clrcache();  see db.c
 * for comments.  Check the header and magic numbers.  Remove the block
 * from the doubly linked list.  Note that mag_free allows null pointers.
 */
mag_free(p)
char *p;
{
	struct header *h;

	if (p == 0)
		return;
	db_clrcache(p);
	p -= sizeof(struct header);
	h = (struct header *) p;
	if (h->h_size < sizeof(struct header) + 2 || h->h_size > mag_max)
		abort();
	if (h->h_magic != 0x5555)
		abort();
	if (p[h->h_size - 1] != 0xA)
		abort();
	h->h_magic = 0xAAAA;
	p[h->h_size - 1] = 0x5;
	if (h->h_next != 0)
		h->h_next->h_prev = h->h_prev;
	if (h->h_prev != 0)
		h->h_prev->h_next = h->h_next;
	else
		mag_mem = h->h_next;
	free(p);
}

/* Print n blocks of the doubly linked list, starting at the last one
 * allocated.
 */
mag_memdump(n){
	struct header *h = mag_mem;

	for (h = mag_mem; h != 0 && n-- != 0; h = h->h_next) {
		printf("%8d: %d %s %x", h->h_size, h->h_n, h->h_descr, h + 1);
		if (h->h_magic != 0x5555 || ((char *) h)[h->h_size - 1] != 0xA)
			printf(" bad magic number");
		printf("\n");
	}
}

#else

/* This is a fast version of mag_malloc().  Allocated memory of limited
 * size is kept in a separate cache which is faster than calling malloc
 * and free.  The first word in an allocated block is set to the size of
 * the block.
 */

#define MAG_CACHE	128		/* maximum size */

char *malloc_cache[MAG_CACHE];		/* a cache entry per size */

/* Reduce memory by clearing caches and representing values in a smart way.
 * Afterwards clear our own malloc cache.
 */
mag_reduce(){
	char *p, **mc;

	db_reduce();
	x_reduce();
	for (mc = malloc_cache; mc < &malloc_cache[MAG_CACHE]; mc++)
		while ((p = *mc) != 0) {
			*mc = * (char **) p;
			free(p);
		}
}

/* Allocate a block of the specified size.  If the size is not too large,
 * check the cache first.
 */
char *mag_malloc(size){
	char *p, **mc;

	size += m_max(sizeof(int), sizeof(char *));
	if (size < MAG_CACHE) {
		mc = &malloc_cache[size];
		if ((p = *mc) != 0) {
			*mc = * (char **) p;
			* (* (int **) &p)++ = size;
			return p;
		}
	}
	if ((p = malloc(size)) == 0)
		mag_panic("out of memory");
	* (* (int **) &p)++ = size;
	if (p + size > mag_end)
		mag_end = p + size;
	return p;
}

/* Free memory.  If the size is not too large, put the block in the cache.
 * Note that mag_free allows null pointers.
 */
mag_free(p)
char *p;
{
	int size;
	char **mc;

	if (p == 0)
		return;
	db_clrcache(p);
	if ((size = * --(* (int **) &p)) >= MAG_CACHE) {
		free(p);
		return;
	}
	mc = &malloc_cache[size];
	* (char **) p = *mc;
	*mc = p;
}

/* Like mag_malloc, but clear the memory.
 */
char *mag_alloc(size){
	char *p = mag_malloc(size);

	bzero(p, size);
	return p;
}

#endif

/* This routine is called regularly by win_sync().  If more than
 * MAG_DELTA bytes have been allocated since the last time, call
 * mag_reduce to free some memory.  This makes sure that no stray
 * cache entries are being maintained all of the time.
 */
mag_checkmem(){
	if (mag_end > mag_limit) {
		mag_reduce();
		mag_limit = mag_end + MAG_DELTA;
	}
}

/* This does an sprintf(buf, "%d", n) but somewhat faster.
 */
mag_strnum(buf, n)
char *buf;
{
	char num[32], *p = &num[sizeof(num) - 1];
	int sign;

	if ((sign = n) < 0)
		n = -n;
	*p = 0;
	do
		*--p = '0' + n % 10;
	while ((n /= 10) != 0);
	if (sign < 0)
		*--p = '-';
	while ((*buf++ = *p++) != 0)
		;
}

/* Return a copy of the given string.
 */
char *mag_copy(s)
char *s;
{
	char *r = s, *t;

	while (*r++ != 0)
		;
	r = t = MAG_MALLOC(r - s, char);
	while ((*r++ = *s++) != 0)
		;
	return t;
}

/* Add a new entry in the record named by name.  The attribute N contains
 * the number of entries it currently has.  The entry is generated by
 * taking the name of the record and adding .<N> to it.  N is incremented
 * afterwards.
 */
char *mag_new(name)
char *name;
{
	char buf[64], *s;
	int N;

	db_numretrieve(name, "N", &N);
	sprintf(buf, "%s.%d", name, N);
	s = mag_copy(buf);
	mag_strnum(buf, N);
	db_store(name, buf, s);
	db_numstore(name, "N", N + 1);
	return s;
}

/* Return entry number i in record name.
 */
struct value *mag_index(name, i)
char *name;
{
	char buf[16];

	mag_strnum(buf, i);
	return db_get(name, buf, db_seq);
}

/* Append value to the record name.  If true is set, this is a true set.
 * This means that multiple entries of the same value are not allowed.
 */
mag_append(name, value, true)
char *name, *value;
{
	struct value *v_ent;
	char buf[16], *ent;
	int N, i;

	db_numretrieve(name, "N", &N);
	if (true)
		for (i = 0; i < N; i++) {
			v_ent = mag_index(name, i);
			if ((ent = val_str(v_ent)) != 0 && scmp(ent, value)) {
				val_free(v_ent);
				return;
			}
			val_free(v_ent);
		}
	mag_strnum(buf, N);
	db_store(name, buf, value);
	db_numstore(name, "N", N + 1);
}

/* Delete value from the set name.
 */
mag_delete(name, value)
char *name, *value;
{
	struct value *v_ent;
	char buf[16], *ent;
	int N, i;

	db_numretrieve(name, "N", &N);
	for (i = 0; i < N; i++) {
		v_ent = mag_index(name, i);
		if ((ent = val_str(v_ent)) != 0 && scmp(ent, value)) {
			val_free(v_ent);
			v_ent = mag_index(name, --N);
			mag_strnum(buf, i);
			db_put(name, buf, v_ent);
			db_numstore(name, "N", N);
			val_free(v_ent);
			return;
		}
		val_free(v_ent);
	}
}

/* A get command has been received.  Call db_get().
 */
struct value **mag_get(req)
struct value **req;
{
	struct value **rep;
	int t, n;
	
	if ((n = mes_count(req)) < 4)
		return mes_error("Usage: get record attribute [seq]");
	t = n == 4 ? db_seq : val_int(req[4]);
	rep = mes_alloc(3);
	rep[1] = val_sstr("ok", db_seq);
	rep[2] = db_get(val_str(req[2]), val_str(req[3]), t);
	return rep;
}

/* A put command has been received.  Call db_store().
 */
struct value **mag_put(req)
struct value **req;
{
	int cnt = mes_count(req);

	if (cnt != 5)
		return mes_error("Usage: put record attribute value");
	db_put(val_str(req[2]), val_str(req[3]), req[4]);
	return mes_ok();
}

/* Set a watch on req[3], req[4] for station req[0].  Use req[2] as a
 * command if the watch fires.
 */
struct value **mag_watch(req)
struct value **req;
{
	if (mes_count(req) != 5)
		return mes_error("Usage: watch command record attribute");
	watch_specify(req[0], mag_copy(val_str(req[2])),
		mag_copy(val_str(req[3])), mag_copy(val_str(req[4])));
	return mes_ok();
}

/* Dump the data base.
 */
struct value **mag_dump(req)
struct value **req;
{
	db_checkpoint();
	return mes_ok();
}

/* Dump the contents window req[2] onto file req[3].
 */
struct value **mag_windump(req)
struct value **req;
{
	if (mes_count(req) != 4)
		return mes_error("Usage: windump window file");
	if (!win_dump(val_int(req[2]), val_str(req[3])))
		return mes_error("can't dump");
	return mes_ok();
}

/* This function is called when the menu is selected.
 */
struct value **mag_select(req)
struct value **req;
{
	char *win, *wid, *rec, *sel, *attr;
	int map;

	if (!val_null(req[4])) {
		rec = val_str(req[2]);
		sel = val_str(req[3]);
		attr = val_str(req[4]);
		db_retrieve(rec, attr, &win);
		db_retrieve(win, "record", &wid);
		db_store(rec, sel, (char *) 0);
		db_numretrieve(wid, "mapped", &map);
		map = !map;
		db_numstore(wid, "mapped", map);
		mag_free(wid);
		mag_free(win);
		db_numstore("menu", "mapped", 0);
	}
	return mes_ok();
}

mag_reply(station, reply)
struct value *station, **reply;
{
	if (reply == 0)
		printf("%s: destination %s is down\n",
					mag_me, val_str(station));
	else if (strcmp(val_str(reply[1]), "ok") != 0)
		printf("%s: %s %s: %s\n", mag_me, val_str(station),
				val_str(reply[1]), val_str(reply[2]));
}

/* Store val in record rec, attribute attr at the specified station.
 */
mag_store(station, rec, attr, val)
struct value *station, *val;
char *rec, *attr;
{
	struct value **mes;

	mes = mes_alloc(5);
	mes[1] = val_sstr("put", db_seq);
	mes[2] = val_cstr(rec, db_seq);
	mes[3] = val_cstr(attr, db_seq);
	mes[4] = val_ref(val);
	mes_send_request(val_ref(station), mes, mag_reply);
}

/* Send the message described in record rec.
 */
mag_sendmes(rec)
char *rec;
{
	struct value **mes;
	int N, i;

	db_numretrieve(rec, "N", &N);
	mes = mes_alloc(N + 1);
	for (i = 0; i < N; i++)
		mes[i + 1] = mag_index(rec, i);
	mes_send_request(db_get(rec, "dest", db_seq), mes, mag_reply);
}

/* This list magic.mes contains messages to be sent.  Send them.
 */
mag_messages(){
	struct value *v_rec;
	char *rec;
	int N, i;

	db_numretrieve("magic.mes", "N", &N);
	for (i = 0; i < N; i++) {
		v_rec = mag_index("magic.mes", i);
		if ((rec = val_str(v_rec)) != 0)
			mag_sendmes(rec);
		val_free(v_rec);
	}
}

/* This routine handles a watch on control/status.  If this becomes other
 * than run, checkpoint the data base and exit.
 */
struct value **mag_term(req)
struct value **req;
{
	if (val_null(req[4]) || !scmp(val_str(req[4]), "run"))
		mag_done = 1;
	return mes_ok();
}

/* Call all init routines and update the history bar.
 */
mag_init(name)
char *name;
{
	mag_me = name;
	db_init(name);
	db_restore();
	x_start();
	win_init();
	com_init();
	dump_start();
	temp_start();
	ww_start();
	clk_register();
	mes_subscribe("get", mag_get);
	mes_subscribe("put", mag_put);
	mes_subscribe("watch", mag_watch);
	mes_subscribe("dump", mag_dump);
	mes_subscribe("windump", mag_windump);
	mes_subscribe("select", mag_select);
	mes_subscribe("mag_term", mag_term);
	db_store("control", "status", "run");
	db_numstore("quit", "mapped", 0);
	watch_specify(V_NULL, "mag_term", "control", "status");
	watch_specify(V_NULL, "select", "control", "select");
	mag_messages();
}

mag_exit(nochk){
	win_close();
	if (!nochk)
		db_checkpoint();
	exit(0);
}
