/*
 *      Originally coded by Robbert van Renesse
 *
 *
 *      ISIS release V2.0, May 1990
 *      Export restrictions apply
 *
 *      The contents of this file are subject to a joint, non-exclusive
 *      copyright by members of the ISIS Project.  Permission is granted for
 *      use of this material in unmodified form in commercial or research
 *      settings.  Creation of derivative forms of this software may be
 *      subject to restriction; obtain written permission from the ISIS Project
 *      in the event of questions or for special situations.
 *      -- Copyright (c) 1990, The ISIS PROJECT
 */

/*
 * This module allows you to interface to meta sensors and actuators.
 */

#include <X11/Xlib.h>
#include "sm.h"
#include "sm_err.h"
#include "magic.h"
#include "value.h"
#include "win.h"

#define WIDTH		600
#define HEIGHT		150

struct value **mes_ok(), **mes_error(), **mes_alloc();
struct value *db_get();
char *type_to_ascii();

/* Get a set element out of *ps which is delimited by delim, and return
 * it in elt.
 */
meta_element(ps, delim, elt)
char **ps, *elt;
{
	char *s = *ps;

	while (*s != 0 && *s != delim) {
		if (*s == '\\') {
			if ((*elt = *++s) == 0)
				break;
		}
		else
			*elt = *s;
		s++;
		elt++;
	}
	*elt = 0;
	*ps = s;
}

/* This routine parses the string val, which is of the form "{ a b c ... }",
 * where a, b, and c are possibly quoted strings.  Put the elements of this
 * set into a new record, whose name is attr/rec.  Assign this name to
 * rec/attr.  It is more convenient to store sensors under their instances
 * than vice versa if you see what I mean.
 */
meta_set(rec, attr, val)
char *rec, *attr, *val;
{
	char *p = val + 1, list[64], elt[128];

	sprintf(list, "%s/%s", attr, rec);
	db_store(rec, attr, list);
	db_numstore(list, "N", 0);
	while (*p != '}') {
		while (*p == ' ')
			p++;
		if (*p == '"') {
			p++;
			meta_element(&p, '"', elt);
			if (*p == '"')
				p++;
		}
		else
			meta_element(&p, ' ', elt);
		mag_append(list, elt, 0);
	}
}

/* The meta notify routine is called from an isis task.  Check whether
 * there is enough stack space left to invoke our routines.
 */
meta_checkstack(){
	char buf[8192];

	t_scheck();
}

/* This routine is invoked by meta when something happens to a sensor.
 * If status equals 0, a new value has been received.  Copy the value
 * into our own database.
 */
void meta_notify(status, sensor, inst, msg, type, eid)
char *sensor, *inst;
message *msg;
EVENT *eid;
{
	PTR value = NULL;
	char *s_val;

	meta_checkstack();
	switch (status) {
	case 0:
		mag_append("instances", inst, 1);
		break;

	case SM_FAILED:
		printf("sensor %s/%s failed\n", sensor, inst);
		mag_delete("instances", inst);
		break;
		
	default:
		printf("meta_notify: bad status %d\n", status);
		return;
	}

	type_unpack(type, msg, SM_VALFLD, &value);
	if (value == 0) {
		printf("meta_notify: bad value\n");
		return;
	}
	s_val = type_to_ascii(type, value);
	if (*s_val == '{')
		meta_set(inst, sensor, s_val);
	else
		db_store(inst, sensor, s_val);
	db_commit();
	msg_delete(msg);
}

/* Return in *type the type of the named sensor or actuator.  This return
 * may fail if sm_sensorlookup fails.
 */
meta_type(sensor, type)
char *sensor;
int *type;
{
	SENSOR *cp, *sm_sensorlookup();

	if ((cp = sm_sensorlookup(sensor)) == 0)
		return SM_NOTKNOWN;
	*type = cp->type;
	return 0;
}

/* Set a meta watch on sensor/inst.
 */
meta_setwatch(v_sensor, v_inst, v_relation, v_value)
struct value *v_sensor, *v_inst, *v_relation, *v_value;
{
	EVENT *eid;
	PTR value;
	message *msg, *msg_newmsg();
	int type, rel, status;
	char *inst, *relation, *sensor = val_str(v_sensor);

	if ((status = meta_type(sensor, &type)) < 0)
		return status;
	if ((relation = val_str(v_relation)) == 0 || *relation == 0)
		rel = R_CHANGE;
	else
		rel = type_relop(relation);
	rel |= SM_ALERTINIT | SM_ALERTONFAIL;
	if (val_null(v_value))
		msg = 0;
	else {
		ascii_to_type(type, val_str(v_value), NULL, &value);
		msg = msg_newmsg();
		type_pack(type, msg, SM_VALFLD, value);
	}
	if ((inst = val_str(v_inst)) == 0 || *inst == 0)
		inst = "*";
	status = sm_watch(sensor, inst, rel, msg, meta_notify, 0, 0, &eid);
	if (status == SM_NOTAVAIL) {
		printf("no %s/%s sensor currently available\n", sensor, inst);
		return 0;
	}
	return status;
}

/* Trigger a meta actuator.
 */
meta_actuate(v_actuator, v_inst, v_value)
struct value *v_actuator, *v_inst, *v_value;
{
	message *msg, *msg_newmsg();
	int type, status;
	PTR value;
	char *actuator = val_str(v_actuator);

	if ((status = meta_type(actuator, &type)) < 0)
		return status;
	ascii_to_type(type, val_str(v_value), NULL, &value);
	msg = msg_newmsg();
	type_pack(type, msg, SM_VALFLD, value);
	status = sm_actuate(actuator, val_str(v_inst), msg, 0, 0);
	flush();
	return status;
}

/* Put an error message in the error field of the window.
 */
meta_error(name, status)
char *name;
{
	char buf[64];

	switch (status) {
	case SM_INTERNAL:	strcpy(buf, "internal meta error");	break;
	case SM_NOTAVAIL:	strcpy(buf, "not available");		break;
	case SM_NOTKNOWN:	strcpy(buf, "not known");		break;
	case SM_NAMETOOLONG:	strcpy(buf, "name too long");		break;
	case SM_FAILED:		strcpy(buf, "failed");			break;
	case SM_TYPEHASCHANGED:	strcpy(buf, "type has changed");	break;
	default:		sprintf(buf, "meta error %d", status);
	}
	db_store(name, "error", buf);
}

/* This routine is invoked when the client specifies a sensor using the
 * sensor window.  Call meta_setwatch to actually set the watch, and
 * mag_initmes to make sure that the watch is automatically started on
 * startups.
 */
struct value **meta_sensor(req)
struct value **req;
{
	extern db_seq;
	struct value *sensor, *instance, *relation, *value, **mes;
	int status;

	if (val_null(req[4]))
		return mes_ok();
	sensor = db_get("meta.sensor", "sensor", db_seq);
	instance = db_get("meta.sensor", "instance", db_seq);
	relation = db_get("meta.sensor", "relation", db_seq);
	value = db_get("meta.sensor", "value", db_seq);
	if (val_null(sensor))
		db_store("meta.sensor", "error", "fill in sensor");
	else {
		status = meta_setwatch(sensor, instance, relation, value);
		if (status != 0)
			meta_error("meta.sensor", status);
		else {
			db_store("meta.sensor", "error", "fill in table");
			mes = mes_alloc(6);
			mes[1] = val_sstr("meta_watch", db_seq);
			mes[2] = val_ref(sensor);
			mes[3] = val_ref(instance);
			mes[4] = val_ref(relation);
			mes[5] = val_ref(value);
			mag_initmes(V_NULL, mes);
		}
	}
	val_free(sensor);
	val_free(instance);
	val_free(relation);
	val_free(value);
	db_store("meta.sensor.sel", "select", (char *) 0);
	return mes_ok();
}

/* This routine is invoked when a watch on a Magic Lantern sensor fires.
 * It has to trigger a Meta actuator in response.
 */
struct value **meta_update(req)
struct value **req;
{
	char buf[128];
	struct value *actuator, *instance;
	int status;

	if (val_null(req[4]))
		return mes_ok();
	sprintf(buf, "%s/%s", val_str(req[2]), val_str(req[3]));
	actuator = db_get(buf, "actuator", db_seq);
	instance = db_get(buf, "instance", db_seq);
	status = meta_actuate(actuator, instance, req[4]);
	val_free(actuator);
	val_free(instance);
	if (status == 0)
		return mes_ok();
	sprintf(buf, "%d", status);
	return mes_error(buf);
}

/* This routine is invoked when a client wants to specify an actuator.
 * The client specifies an ML watch and a META actuator.  When the
 * watch is invoked, the actuator gets triggered.
 */
struct value **meta_actuator(req)
struct value **req;
{
	char *rec, *attr, *actuator, *instance, buf[128];

	if (val_null(req[4]))
		return mes_ok();
	db_retrieve("meta.actuator", "record", &rec);
	db_retrieve("meta.actuator", "attr", &attr);
	db_retrieve("meta.actuator", "actuator", &actuator);
	db_retrieve("meta.actuator", "instance", &instance);
	if (rec == 0 || attr == 0 || actuator == 0 || instance == 0) {
		db_store("meta.actuator", "error", "fill in everything");
		MAG_FREE(rec);
		MAG_FREE(attr);
	}
	else {
		sprintf(buf, "%s/%s", rec, attr);
		db_store(buf, "actuator", actuator);
		db_store(buf, "instance", instance);
		watch_specify(V_NULL, "meta_update", rec, attr);
	}
	MAG_FREE(actuator);
	MAG_FREE(instance);
	db_store("meta.actuator.sel", "select", (char *) 0);
	return mes_ok();
}

/* This routine is invoked when a client wants to set a watch on something.
 */
struct value **meta_watch(req)
struct value **req;
{
	int status;
	char buf[16];

	if (mes_count(req) != 6)
		return mes_error("needs 4 arguments");
	if ((status = meta_setwatch(req[2], req[3], req[4], req[5])) == 0)
		return mes_ok();
	sprintf(buf, "%d", status);
	return mes_error(buf);
}

/* Initialized meta, and set watches for the sensor and actuators windows.
 * Also catch messages from clients who want to set watches or invoke
 * actuators.
 */
meta_init(name)
char *name;
{
	char *getenv(), *port = getenv("ISISPORT");

	sm_init(port == 0 ? 1603 : atoi(port));
	mag_init(name);
	sel_init();
	mes_subscribe("meta_sensor", meta_sensor);
	watch_specify(V_NULL, "meta_sensor", "meta.sensor.sel", "select");
	mes_subscribe("meta_actuator", meta_actuator);
	watch_specify(V_NULL, "meta_actuator", "meta.actuator.sel", "select");
	mes_subscribe("meta_watch", meta_watch);
	mes_subscribe("meta_update", meta_update);
}
