/*
 *      Originally coded by Robbert van Renesse
 *
 *
 *      ISIS release V2.0, May 1990
 *      Export restrictions apply
 *
 *      The contents of this file are subject to a joint, non-exclusive
 *      copyright by members of the ISIS Project.  Permission is granted for
 *      use of this material in unmodified form in commercial or research
 *      settings.  Creation of derivative forms of this software may be
 *      subject to restriction; obtain written permission from the ISIS Project
 *      in the event of questions or for special situations.
 *      -- Copyright (c) 1990, The ISIS PROJECT
 */

/*
 * This file contains the code for ``container'' widgets.  A container is
 * like a scrollbar.  The "record" and "attr" attributes specify where to
 * get the data from.  "interval" specifies an upperbound on this data.
 * "direction" (north, south, east, west) specifies in which direction
 * the scrollbar should grow.
 */

#include "magic.h"
#include "value.h"
#include <X11/Xlib.h>
#include "win.h"

#define NORTH	0
#define SOUTH	1
#define WEST	2
#define EAST	3

/* This value (0 or 1) specifies how the container should follow the mouse.
 * If 0, the level of the container is the position of the mouse.  If 1,
 * the level is 1 higher than the position of the mouse.  cont_above is
 * set to 1 if the user depresses the mouse above the level of the container.
 * This gives a more natural feeling to mouse tracking, especially if the
 * granularity of the steps is high.  In one specific case, where interval
 * equals 1, it means that if you depress the mouse in the container, the
 * value is inverted.  Such a container can be used as a switch.
 */
int cont_above;

/* Return a number (0, 1, 2, 3) corresponding to the direction.
 */
cont_dir(name)
char *name;
{
	extern db_seq;
	struct value *db_get(), *v_dir = db_get(name, "direction", db_seq);
	char *direction = val_str(v_dir);
	int dir;

	if (direction == 0)
		dir = NORTH;
	else
		switch (*direction) {
		case 's':	dir = SOUTH;	break;
		case 'w':	dir = WEST;	break;
		case 'e':	dir = EAST;	break;
		default:	dir = NORTH;
		}
	val_free(v_dir);
	return dir;
}

/* The mouse has been depressed.  Set cont_above, and figure out what the
 * new value of the attribute is.
 */
cont_press(o, x, y, count)
struct object *o;
{
	int interval, value, new, dir;
	char *attr;

	db_numretrieve(o->o_name, "interval", &interval);
	if (interval == 0)
		return;
	db_retrieve(o->o_name, "attr", &attr);
	if (attr == 0)
		return;
	dir = cont_dir(o->o_name);
	db_numretrieve(o->o_record, attr, &value);
	switch (dir) {
	case NORTH:
		new = (o->o_height - y) * interval / o->o_height;
		break;
	case SOUTH:
		new = y * interval / o->o_height;
		break;
	case WEST:
		new = (o->o_width - x) * interval / o->o_width;
		break;
	case EAST:
		new = x * interval / o->o_width;
	}
	cont_above = new >= value ? 1 : 0;
	db_numstore(o->o_record, attr, new + cont_above);
	MAG_FREE(attr);
}

/* The mouse has been moved (or released).  Calculate the new value of the
 * attribute.
 */
cont_track(o, x, y)
struct object *o;
{
	int interval, value, dir;
	char *attr;

	db_numretrieve(o->o_name, "interval", &interval);
	if (interval == 0)
		return;
	db_retrieve(o->o_name, "attr", &attr);
	if (attr == 0)
		return;
	dir = cont_dir(o->o_name);
	switch (dir) {
	case NORTH:
		if (y < 0)
			value = interval;
		else if (y >= o->o_height)
			value = 0;
		else
			value = cont_above +
				(o->o_height - y) * interval / o->o_height;
		break;
	case SOUTH:
		if (y < 0)
			value = 0;
		else if (y >= o->o_height)
			value = interval;
		else
			value = cont_above + y * interval / o->o_height;
		break;
	case WEST:
		if (x < 0)
			value = interval;
		else if (x >= o->o_width)
			value = 0;
		else
			value = cont_above +
				(o->o_width - x) * interval / o->o_width;
		break;
	case EAST:
		if (x < 0)
			value = 0;
		else if (x >= o->o_width)
			value = interval;
		else
			value = cont_above + x * interval / o->o_width;
	}
	db_numstore(o->o_record, attr, value);
	MAG_FREE(attr);
}

/* Draw the container.
 */
cont_draw(w, o)
struct window *w;
struct object *o;
{
	int interval, value, dir, length;
	char *attr;

	db_numretrieve(o->o_name, "interval", &interval);
	if (interval == 0)
		return;
	db_retrieve(o->o_name, "attr", &attr);
	if (attr == 0)
		return;
	dir = cont_dir(o->o_name);
	db_numretrieve(o->o_record, attr, &value);
	switch (dir) {
	case NORTH:
		length = value * o->o_height / interval;
		win_fillrect(w, o->o_x, o->o_y + o->o_height - length,
							o->o_width, length);
		break;
	case SOUTH:
		length = value * o->o_height / interval;
		win_fillrect(w, o->o_x, o->o_y, o->o_width, length);
		break;
	case WEST:
		length = value * o->o_width / interval;
		win_fillrect(w, o->o_x + o->o_width - length, o->o_y,
							length, o->o_height);
		break;
	case EAST:
		length = value * o->o_width / interval;
		win_fillrect(w, o->o_x, o->o_y, length, o->o_height);
	}
	MAG_FREE(attr);
}

/* This shows information about container widgets, and is used in the
 * edit window.
 */
cont_edit(){
	char *wid, *pic_new();

	db_store("edit.buffer", "container", "edit.buffer.container");

	mag_append("edit.buffer.cont", "edit.buffer.cont.schema", 0);
	mag_append("edit.buffer.cont", "#edit.buffer selection #", 0);
	db_store("edit.buffer.cont.schema", "attr", "attribute");
	db_store("edit.buffer.cont.schema", "interval", "interval");
	db_store("edit.buffer.cont.schema", "direction", "direction");
	wid = pic_new("edit.buffer.container", "table", "edit.buffer.cont",
						0, 0, 580, 40, 1, 1);
	tbl_column(wid, "attr", 1, 0, "c", (char *) 0);
	tbl_column(wid, "interval", 1, 0, "c", (char *) 0);
	tbl_column(wid, "direction", 1, 0, "c", (char *) 0);
	MAG_FREE(wid);
}
