/*
 *      Originally coded by Robbert van Renesse
 *
 *
 *      ISIS release V2.0, May 1990
 *      Export restrictions apply
 *
 *      The contents of this file are subject to a joint, non-exclusive
 *      copyright by members of the ISIS Project.  Permission is granted for
 *      use of this material in unmodified form in commercial or research
 *      settings.  Creation of derivative forms of this software may be
 *      subject to restriction; obtain written permission from the ISIS Project
 *      in the event of questions or for special situations.
 *      -- Copyright (c) 1990, The ISIS PROJECT
 */

/*
 * This is the communication module.  Communication is done via X11
 * selections.  A selection is an atom, owned by a certain window.  You
 * can send a SelectionRequest to the owner of a selection, specifying
 * a property atom and a target atom.  The property atom identifies the
 * data object through which you want to communicate, and the target atom
 * the type in which you want to have the reply.
 *
 * The way in which we are going to use these is somewhat unorthodox.
 * Each station will have its own selection, used as an address.  To get
 * hold of this atom you can call XInternAtom with as argument the ASCII
 * name of the station.  Now you can send a request to any station.  The
 * property will contain the command and arguments of the request.  This
 * is the unorthodox part, since the owner is not supposed to look into
 * the current contents of the property.  The owner overwrites the atom
 * with the reply.  The target atom will be unused, as the request will
 * identify the type of the reply.
 *
 * Note that communication is asynchronous in nature.  You send a request,
 * and some time later comes back a reply.  Although we could simulate
 * synchrony by waiting for the reply, we have chosen not to do so to
 * preserve as much as possible of real-time characteristics, and to avoid
 * deadlock (e.g., X requesting something of Y and vice versa).
 *
 * The asynchrony also introduces a problem, namely that the properties
 * will become a scarce resource.  When you want to send a message, you
 * have to allocate a property.  The same property can be used for the
 * reply, so that the client knows when it is free again.  The properties
 * will be named "mag_mess_0" through "mag_mess_<N - 1>", where N is the
 * number of properties available for sending messages.  When you want to
 * send a message, and there is no property available, the message gets
 * queued.
 */

#include <X11/Xlib.h>
#include "value.h"
#include "magic.h"

#define MAX_PROPS	16	/* max. # properties */
#define MAX_LENGTH	1024	/* max. message length */
#define MAX_STATIONS	17	/* size of hash table of stations */

/* This data structure maintains a queue of messages to be sent.
 */
struct message {
	struct message *m_next;		/* next message in queue */
	struct value **m_mes;		/* contents */
	struct value *m_dest;		/* whereto? */
	int (*m_reply)();		/* reply procedure */
} *com_in, *com_out;	/* queue pointers */

/* This data structure maintains a pool of properties.
 */
struct property {
	int p_alloc;			/* allocated or not? */
	Atom p_prop;			/* atom of property */
	struct value *p_dest;		/* whereto? */
	int (*p_reply)();		/* reply procedure */
} com_props[MAX_PROPS];
int com_lost;	/* the number of ``lost'' properties (to be recovered) */

/* This data structure maintains of cache of stations and their selection
 * properties.
 */
struct station {
	struct value *s_name;		/* name of the station */
	Atom s_sel;			/* its selection property */
} com_stations[MAX_STATIONS];	/* cache of stations */

extern Display *x_dpy;
extern Window x_win;
extern db_seq;

struct value **mes_error();

Atom com_type;			/* dummy target/type atom */

/* Put message into prop, and free it.
 */
com_put_prop(w, prop, mes)
Window w;
Atom prop;
struct value **mes;
{
	static char buf[8192];
	struct value **vp, *v;
	char *p = buf, *s;

/*
	db_put("mess_out", "value", mes[0]);
*/
	sprintf(p, "%d\n", mes_count(mes));
	while (*++p != 0)
		;
	for (vp = mes; (v = *vp) != 0; vp++) {
		sprintf(p, "%d:", v->v_seq);
		while (*++p != 0)
			;
		if (val_null(v))
			strcat(p, "-1\n");
		else {
			s = val_str(v);
			sprintf(p, "%d:%s\n", strlen(s), s);
		}
		while (*++p != 0)
			;
		val_free(v);
	}
	XChangeProperty(x_dpy, w, prop, com_type, 8,
				PropModeReplace, buf, p - buf);
	MAG_FREE(mes);
}

/* Get the message out of the specified property of the specified window.
 */
struct value **com_get_prop(w, prop)
Window w;
Atom prop;
{
	Atom actual_type;
	int actual_format, nvals, seq, len;
	unsigned long nitems, bytes_after;
	char *buf = 0, *p, *q, *s;
	struct value **mes, **v;

	XGetWindowProperty(x_dpy, w, prop, 0L, (long) MAX_LENGTH,
		False, com_type, &actual_type, &actual_format,
		&nitems, &bytes_after, &buf);
	if (buf == 0) {
		printf("couldn't get property (pretty fatal actually)\n");
		return mes_error("bogus message");
	}

	p = buf;
	nvals = atoi(p);
	while (*p++ != '\n')
		;
	v = mes = MAG_MALLOC(nvals + 1, struct value *);
	while (nvals--) {
		sscanf(p, "%d:%d", &seq, &len);
		while (*p++ != ':')
			;
		if (*p == '-') {
			*v++ = val_pstr((char *) 0, seq);
			p += 3;
		}
		else {
			while (*p++ != ':')
				;
			q = s = MAG_MALLOC(len + 1, char);
			while (len--)
				*q++ = *p++;
			*q = 0;
			*v++ = val_pstr(s, seq);
		}
	}
	*v = 0;
	free(buf);
/*		should remember seqno as well
	db_put("mess_in", "value", mes[0]);
*/
	return mes;
}

/* Flush the station cache.
 */
com_flush(){
	struct station *s;

	for (s = com_stations; s < &com_stations[MAX_STATIONS]; s++)
		if (s->s_name != 0) {
			val_free(s->s_name);
			s->s_name = 0;
		}
}

/* Get the station selection number.  If it isn't in the cache, get it
 * using XInternAtom.
 */
Atom com_sel(station)
struct value *station;
{
	int n = 0;
	char *p = val_str(station);
	struct station *s;

	while (*p != 0)
		n ^= *p++;
	s = &com_stations[n % MAX_STATIONS];
	for (n = 0; n < MAX_STATIONS; n++) {
		if (s->s_name == 0)
			break;
		if (val_cmp(s->s_name, station))
			break;
		if (++s == &com_stations[MAX_STATIONS])
			s = com_stations;
	}
	if (n == MAX_STATIONS)
		com_flush();
	if (s->s_name == 0) {
		s->s_name = val_ref(station);
		s->s_sel = XInternAtom(x_dpy, val_str(station), False);
	}
	return s->s_sel;
}

/* Put message in to prop, and send it off to station.
 */
com_send_prop(p, dest, mes, reply)
struct property *p;
struct value *dest, **mes;
int (*reply)();
{
	p->p_dest = dest;
	p->p_reply = reply;
	com_put_prop(x_win, p->p_prop, mes);
	XConvertSelection(x_dpy, com_sel(dest), com_type,
					p->p_prop, x_win, CurrentTime);
}

/* Asynchronous RPC.  Call station with the specified request message.
 * The reply procedure is called when a reply arrives, with the station
 * and the reply message as arguments.  First we check whether there is an
 * outstanding RPC to this particular site.  If so, we enqueue the message
 * for later transmission.  If not, we have to allocate a property to contain
 * the message.  If this fails, we still have to queue the message.  If not,
 * we can put the message in the property and send it off.
 */
com_send_request(dest, mes, reply)
struct value *dest, **mes;
int (*reply)();
{
	struct property *p;
	struct message *m;

	for (m = com_out; m != 0; m = m->m_next)
		if (val_cmp(m->m_dest, dest))
			break;
	if (m == 0)
		for (p = com_props; p < &com_props[MAX_PROPS]; p++)
			if (!p->p_alloc) {
				p->p_alloc = 1;
				com_send_prop(p, dest, mes, reply);
				return;
			}
	m = MAG_ALLOC(1, struct message);
	m->m_dest = dest;
	m->m_mes = mes;
	m->m_reply = reply;
	if (com_in == 0)
		com_in = com_out = m;
	else {
		com_in->m_next = m;
		com_in = m;
	}
}

/* Received a request.  Get the message out of the property, call
 * mes_recv_request, and put the reply back in the property.
 */
com_got_request(w, prop)
Window w;
Atom prop;
{
	struct value **mes, **mes_recv_request();

	mes = com_get_prop(w, prop);
	com_put_prop(w, prop, mes_recv_request(mes));
	mes_free(mes);
}

/* Recover lost properties.  com_lost contains the number of lost
 * properties.  If this is equal to the number of outstanding properties
 * we can just free the outstanding properties.  We generate error (null)
 * replies for the lost properties.
 */
com_recover(){
	struct property *p;
	int n = MAX_PROPS;

	for (p = com_props; p < &com_props[MAX_PROPS]; p++)
		if (!p->p_alloc)
			n--;
	if (n != com_lost)
		return;
	for (p = com_props; p < &com_props[MAX_PROPS]; p++)
		if (p->p_alloc) {
			(*p->p_reply)(p->p_dest, (struct value **) 0);
			val_free(p->p_dest);
			db_commit();
			p->p_alloc = 0;
		}
	com_lost = 0;
}

/* Received a reply from the owner of the selection.  Call the reply procedure
 * with the message in the property as argument.  Then see if there are more
 * messages to be sent and send them.
 */
com_got_reply(prop)
Atom prop;
{
	struct message *m;
	struct property *p;
	struct value **mes;

	if (prop == None) {
		com_lost++;
		com_recover();
	}
	else {
		for (p = com_props; p < &com_props[MAX_PROPS]; p++)
			if (p->p_prop == prop)
				break;
		if (p == &com_props[MAX_PROPS] || !p->p_alloc) {
			printf("received strange property\n");
			return;
		}
		mes = com_get_prop(x_win, prop);
		(*p->p_reply)(p->p_dest, mes);
		val_free(p->p_dest);
		mes_free(mes);
		db_commit();
		p->p_alloc = 0;
		if (com_lost != 0)
			com_recover();
	}
	if ((m = com_out) == 0)
		return;
	for (p = com_props; p < &com_props[MAX_PROPS]; p++)
		if (!p->p_alloc) {
			com_send_prop(p, m->m_dest, m->m_mes, m->m_reply);
			if ((com_out = m->m_next) == 0)
				com_in = 0;
			MAG_FREE(m);
			p->p_alloc = 1;
			if ((m = com_out) == 0)
				return;
		}
}

/* Initialize the list of properties.
 */
com_init(){
	struct property *p;
	char buf[16];

	for (p = com_props; p < &com_props[MAX_PROPS]; p++) {
		sprintf(buf, "mag_mess%d", p - com_props);
		if ((p->p_prop = XInternAtom(x_dpy, buf, False)) == None)
			mag_panic("can't create message property");
	}
	com_type = XInternAtom(x_dpy, "dummy target/type", False);
}
