.so tmac.mine
.TL
Magic Documentation
.AU
Robbert van Renesse
Computer Science Department
4124 Upson Hall
Cornell University
Ithaca, NY 14853
.AE
.NH
Generally Magic Things
.PP
This document describes the Magic Lantern Distributed Management
system, or just Magic in short.
Magic Lantern is a highly advanced graphical toolkit which is
especially suitable for the management of possibly distributed
systems.
At the heart of the Magic Lantern is a data base which stores
functions, and which retains a complete history of updates.
The data base maintains the configuration of the system under
management, sensor values, and the contents of the windows on the
screen.
The graphical interface interacts directly with the data base.
Magic Lantern also provides watches, communication mechanisms, and
can be connected to the Meta system for further functionality.
.PP
The document starts out with a concise description of all the
functionality as it is currently implemented.
Later, starting with the section called ``Magic Example,'' we will
indulge in cute applications which will serve as examples of this
description.
In fact, we would suggest that you first only browse through the
first part, then read the ``Magic Example,'' which lets you play
with some applications, and then, if you wish, read the first part
more thoroughly.
This document is very new, and many parts will be reorganized or
rewritten;  in particular this very sentence will disappear in later
versions.
.NH 2
Installation
.PP
To be able to run this package you need at least 10 Megabytes of
disk space.
You can find out whether you have this much or not by running the
command "df .".
Install the Magic software.
This should create the following entries:
.DS
.ta \w'README   'u
README	explanatory comments
bin	some scripts
config	ISIS and META definitions
cookie	cookie factory demo sources
db	data base subdirectory
doc	documentation directory
files	files installed
include	include files
lib	library routine sources
make	makefile templates
util	utility sources
.DE
First edit the file "config" such that ISIS and META are defined as
their respective home directories.
To compile the whole system, run bin/install.
It will automatically create a subdirectory named after the
architecture on which you run.
You can call the script from several architectures to get versions for
each one.
.PP
You will have to run X11 windows, preferably release 4.
The system was developed on a SUN 4 SPARC station 1 running SUN OS.
It should be portable enough, but it might be very slow on
architectures with less than 10 MIPS and 10 Megabytes of memory.
In this document we assume that the architecture is a SUN 4.
.NH 2
Running Magic
.PP
The Magic system is initialized by either of the following routines:
.DS
mag_init(char *name);
mag_cold(char *name, int width, int height, int mapped);
.DE
.I Mag_init()
reads the data base from the file "db/name.db" and initialize internal
data structures.
.I Mag_cold()
is used if the data base should not be read.
This will create a new data base by the specified name, and
creates a window of the given size.
.I Mapped
specifies whether the main window should be mapped or not.
.PP
The utility
.DS
create name
.DE
creates a data base called
.I name ,
initialized with several data that will be described later in
this document.
.PP
Magic is normally running a main loop, which is invoked after
initialization:
.DS
extern int mag_done;
win_main_loop();
.DE
This loop can be terminated by setting
.I mag_done .
The program can be terminated by
.DS
mag_exit(int nochk);
.DE
If
.I nochk
is non-zero, the internal Magic data base will not be checkpointed.
.NH 2
Memory Allocation
.PP
The Magic system uses its own memory allocator.
It can be used by including the file "magic.h" and loading the
Magic library.
This will provide the following three routines:
.DS
type *MAG_ALLOC(n, type);
type *MAG_MALLOC(n, type);
MAG_FREE(type *p);
.DE
.I Type
can be any C type.
.I MAG_ALLOC()
and
.I MAG_MALLOC()
both allocate
.I n
contiguous memory units for that type,
the difference being that
.I MAG_ALLOC()
zeroes the memory allocated.
.I MAG_FREE()
frees memory allocated by either
.I MAG_ALLOC()
or
.I MAG_MALLOC() .
Declarations and casts, such as necessary for the standard C memory
allocator, are unnecessary.
.SH
Notes
.PP
This memory allocator can be used together with the standard C
.I malloc()/free()
library.
However, memory allocated by
.I MAG_ALLOC()
can only be freed by
.I MAG_FREE() ,
and memory allocated by
.I malloc()
can only be freed by
.I free() .
All memory in the Magic system is allocated by either
.I MAG_ALLOC()
or
.I MAG_MALLOC() .
.PP
There are two version of the memory allocator.
Normally
.I MAG_DEBUG
is not defined.
In the other case, when MAG_DEBUG is defined, the memory allocator
does some sanity checks each time one of the procedures is invoked.
The routine
.DS
mag_memdump(int n);
.DE
pretty prints the last
.I n
memory allocations that were not freed.
This is handy for finding memory leaks.
.NH 2
Values
.PP
Values are ASCII strings (possibly null strings) with associated
sequence numbers.
The historical data base stores postfix functions on values that return
values.
The sequence number is a measure of time in the data base, but is not
associated with real-time.
Instead the sequence number is increased on each commit operation in
the data base.
In practice, strings are often interpreted as floating point or as
integer number.
For example, the string "3.4" denotes the number 3.4.
Values can be created, copied, and deleted, but not changed.
.PP
The ``value'' module implements the ``struct value'' abstract data type.
Programs can use this data type by including the "value.h" include
file and loading the Magic library.
The module provides for efficient copying and interpretation of
values.
.PP
Values can be created by any of the following functions:
.DS
struct value *val_sstr(char *s, int seq);
struct value *val_pstr(char *s, int seq);
struct value *val_cstr(char *s, int seq);
struct value *val_pdbl(double d, int seq);
struct value *val_pint(int i, int seq);
.DE
.I Val_sstr()
is used for static strings, that is, strings that do not change value.
The string cannot be freed when the value is deleted.
.I Val_pstr()
is used for strings whose storage has been allocated using MAG_MALLOC
or MAG_ALLOC, and whose storage has to be freed when the value is
deleted.
The string storage may not be overwritten, nor freed other than by
deleting the value.
.I Val_cstr()
is used for dynamic strings, that is, strings which may be overwritten
or freed.
.I Val_cstr()
creates an internal copy of the string, and is thus more expensive than
.I val_sstr()
or
.I val_pstr() .
Each of these routines accept a null string.
.PP
Values can be copied resp.\& deleted by
.DS
struct value *val_ref(struct value *v);
val_free(struct value *v);
.DE
Each time a value is referenced by another pointer, use
.I val_ref()
to increment the internal reference counter.
To decrement the reference counter, use
.I val_free() .
.I Val_free()
will automatically release storage associated with the value when the
reference counter reaches zero.
.PP
To copy a value, while changing its sequence number, use
.DS
struct value *val_pseq(struct value *v, int seq, int copy_or_free);
.DE
This returns a value like
.I v ,
but with the sequence number
.I seq .
.I Copy_or_free
is a boolean flag.
If non-zero (V_FREE), the value
.I v
is freed automatically.
Otherwise (V_COPY),
.I val_pseq()
returns a new value and leaves
.I v
intact.
.I 
.PP
Values can be accessed using the following functions:
.DS
int val_null(struct value *v);
char *val_str(struct value *v);
double val_dbl(struct value *v);
int val_int(struct value *v);
int val_seq(struct value *v);
.DE
.I Val_null()
returns 0 if and only if the value contains a null string.
.I Val_str()
returns a string representation of the value, and null if the value
contains a null string.
.I Val_dbl()
and
.I val_int()
return a floating point resp.\& and integer representation of the
value, and zero if the value contains a null string.
.I Val_seq()
returns the sequence number of the value.
.PP
Values can be compared for equality using the function
.DS
val_cmp(struct value *v, struct value *w);
.DE
.I Val_cmp
returns non-zero if and only if the values pointed to by
.I v
and
.I w
represent the same ASCII string.
.I Val_cmp
ignores the sequence number.
.SH
Notes
.PP
The module is used by including the line
.DS
#include "value.h"
.DE
The routines above are declared in this file, and should not be declared
in the program, as some of the routines may be macros.
Also, because the routines may be macros, it is advised not to use
arguments that have side-effects.
Instead we advise to use only simple expressions such as constants and
variables for arguments to these routines.
.PP
A null value pointer is taken as a value containing a null string and
a sequence number equal to zero.
V_NULL has been defined as a null value pointer.
Thus all routines above accept and return null pointers.
A null pointer return does not signify an error.
The value module implements several sanity checks, and invokes
.I abort()
if a real error is detected.
.PP
The procedure
.DS
val_reduce(struct value *v);
.DE
can be used to decrease the storage used by the value
.I v .
Otherwise it is non-operational.
.NH 2
Postfix Functions
.PP
The Magic system provides postfix functions on values.
For example, the expression { 2 4 + } evaluates to the value "6".
(In this document we will use braces to quote postfix expressions.)
The functions keep track of the maximum sequence number so that
the time at which a function acquired a certain value can be
retrieved.
We represent a postfix expression by a list of so-called
.I tokens .
In our previous example, the tokens are ``2,'' ``4,'' and ``+.''
.PP
The expressions are evaluated strictly from left to right using a
stack.
Tokens that are not operators are pushed onto the stack.
Operator tokens replace zero or more tokens from the top of the stack
by an arbitrary amount of other tokens.
For example, ``+'' replaces two tokens from the stack by a single
token containing the sum.
Operators may be quoted by double quotes to remove their special
meaning, so that they may be pushed onto the stack unevaluated.
Special characters in a quoted string may be quoted by a `\e'.
If one of the operators of an operation is null, then the result will
be null as well.
.PP
The basic operators are listed in Appendix A.
They are divided into mathematical operators, boolean operators,
string operators, special operators, and general operators.
Mathematical operators include operations like addition or the sine
of a value.
Boolean operators like ``<'' and ``or'' represent
.I false
by zero, and
.I true
by non-zero values.
The ``?'' operator accepts a null condition, which is treated as if
false.
.PP
String operators apply to ASCII strings.
An example of such an operation is string comparison.
General operators apply to the postfix machine.
Examples are discarding or duplicating tokens on the top of the stack.
One general operator is assignment, called ":=".
This operator takes a value and a name of the stack, and binds the
value to that name.
Later, when it sees the token "%name" it will substitute the value
in its place.
For example, { x a := %a } results in the value "x".
If it sees a variable token "%name" to which no value was bound, it
pushes a null value on the stack.
.PP
Special operators have specialized functions.
An example of a special operator is ``#'', which takes a record and an
attribute as input tokens, and produces the value that is retrieved
from the the data base as output token.
These operators will be described later when the data base has been
described.
.PP
Postfix expressions are evaluated in
.I contexts .
Each context has its own stack and variables.
A context may also have a parent context.
The variables of the parent context may be read, but not overwritten.
If assigned, only the child context gets updated.
Child contexts are used by list operators and their usage will be
discussed later.
.SH
Notes
.PP
The postfix functions are true functions, in that they do not have any
side-effects to the data base when evaluated.
Side-effects can be implemented by
.I watches ,
and will be described later.
.PP
.NH
The Magic Historical Data Base
.PP
The Magic data base is organized in
.I records .
Each record may contain an arbitrary collection of
.I attributes .
An attribute has an list of values, ordered on sequence number.
Not two values in the list can have the same sequence number.
The last value in the list, the value with the highest sequence number,
is considered the current value.
Initially, all attributes are null and have a zero sequence number.
If the string representation of a value starts with the character ``#'',
then the rest of the string is a postfix function on other values in
the data base, and returns either an ASCII string or null.
.PP
Associated with the data base is the
.I "current sequence number" .
Values updated in the data base are stored under the current sequence
number.
By increasing the current sequence number, the changes in the data
base are committed atomically.
It is not possible to update the history of the data base.
.PP
Records and attributes are named by ASCII strings.
Thus string values can be retrieved by specifying two strings and a
sequence number.
The current sequence number is globally available to allow retrieving
the current value of an attribute.
.PP
The basic interface to the data base is the following:
.DS
extern int db_seq;
db_put(char *record, char *attribute, struct value *v);
struct value *db_get(char *record, char *attribute, int seq);
db_commit();
db_checkpoint();
.DE
.I Db_seq
is the current sequence number, and should not be updated directly.
.I Db_put()
updates the value of an attribute by adding it to the end of the list.
.I Record
specifies the string name of the record, and
.I attribute
the name of the attribute.
.I V
is a value, which sequence number gets overwritten to be
.I db_seq .
.I Db_get
returns that value in the list of attributes that has the highest
sequence number lower or equal to
.I seq .
Db_get never returns a null pointer, but may return a null value.
The value can be freed using
.I val_free() .
.PP
Db_commit() is used to increment the sequence number if any changes
were made to the data base since the last commit.
If not, db_commit() does not do anything.
If changes were made, they may possibly be logged for crash recovery,
but this depends on the implementation.
.PP
.I Db_checkpoint()
write the whole data base to a stable file, and, if necessary,
truncates the log file.
.I Db_checkpoint()
is an expensive operation which should not be invoked often.
However, not invoking it at all may cause all information to be lost
if no log file is maintained, or may cause the log file to grow too
large if it is.
.PP
Records have three special attributes.
The first is the "name" attribute, containing the name of the record.
The second is the "$default" attribute.
If the value of an attribute in the record is null,
.I db_get()
will try to evaluate the "$default" attribute instead.
If this isn't null, that value is returned instead.
The last special attribute is "$number".
If an attribute name starts with a digit, and evaluates to null, then
the "$number" attribute is tried in the same way as the "$default"
attribute.
If this evaluates to null as well, the "$default" attribute is tried
afterwards.
.PP
For convenience the following routines have been defined as well:
.DS
db_store(char *record, char *attribute, char *s);
db_dblstore(char *record, char *attribute, double d);
db_numstore(char *record, char *attribute, int i);
db_retrieve(char *record, char *attribute, char **s);
db_dblretrieve(char *record, char *attribute, double *d);
db_numretrieve(char *record, char *attribute, int *i);
.DE
.I Db_store ,
.I db_dblstore ,
and
.I db_numstore
immediately store strings, doubles, or integers into the data base.
.I Db_retrieve ,
.I db_dblretrieve ,
and
.I db_numretrieve
retrieve strings, doubles, or integers from the data base.
The sequence number used is
.I db_seq ,
so that the current values of attributes are retrieved.
The string returned by
.I db_retrieve()
can be freed using
.I MAG_FREE() .
.SH
Notes
.PP
To initialize the data base, use the functions:
.DS
db_init(char *name);
db_restore();
.DE
These functions are automatically called from
.I mag_init() ,
and are rarely invoked by user applications directly.
.I Db_init()
initializes the data structures and remembers the given name for
data base restoration and checkpointing.
In the current implementation, the second storage of a data base
is in the file "db/name.db".
.I Db_restore()
reads this file to restore the data base in main memory.
The file starts with the current sequence number.
After that each ``#length:attribute:'' starts a new attribute.
.I Length
is the length of the string name of
.I attribute .
If the attribute is "name", then it start a new record.
.PP
After this follows a list of values of the form ``seq:length:value'',
each on a new line.
.I Seq
is the sequence number of the value, and
.I length
the string length of the value.
Null values have the different form ``seq:\-1''.
.PP
For performance reasons, the Magic data base maintains an internal cache
of records pointers.
This may lead to problems if the data pointed to by record pointers
is overwritten and re-used.
The routine
.DS
db_clrcache(char *record);
.DE
should be called by users after the data pointed to by a record pointer
is overwritten.
This should be considered a performance kludge, and not an elegant
feature.
.I Db_clrcache
is automatically invoked from
.I MAG_FREE() ,
so that reallocating memory does not pose problems.
.NH 2
Lists and Tables
.PP
The data base can be used to store the values of sensors and
actuators, the configuration of a system, the contents of the visual
display, information about communication between modules, and other
things like, for example, a data dictionary.
The data dictionary is provided by a set of special functions.
.PP
Data in the data base is represented by values, which conveniently
stores ASCII strings, floating point numbers, and integers.
If other types of data need be stored, this has to be encoded in some
way.
A record type is directly representable as a data base record.
A list is stored in a record, with the attributes serving as indexes.
For example, a list of ten elements could be stored in a record with
attributes ``0'' through ``9.''
The attribute ``N'' stores the number of elements in the list.
.PP
A list can be manipulated through the following routines:
.DS
mag_append(char *name, char *value, int uni);
mag_delete(char *name, char *value);
struct value *mag_index(char *name, int index);
.DE
.I Mag_append
adds
.I value
to the list which is represented by the record named by
.I name .
If
.I uni
is non-zero, the list is a set, and if the value is already in the
set, it is not appended.
Otherwise the value is just added to the end of the list.
.I Mag_delete
deletes the value once from the list, if it exists.
These two routines use string values for convenience.
.I Mag_index
returns the value at the specified index in the list.
Note that
.I mag_index
returns a value data type, and not a string.
.PP
A number of postfix operators work on these lists.
For example, the "$list" operation takes a list name and a quoted
operation as arguments, and applies the operation on all operators
in the list.
The operation is evaluated in a new context.
In this context, the "$entry" operator evaluates to the name of
the entry in the list.
As an example, { name "$entry" $list } pushes all the entries in
the list ``name'' onto the stack.
Note that the $entry operator has to be quoted so that it is not
evaluated before $list is.
.PP
One of the mostly used structures is the table structure.
A table is built from a list of record names.
Each record represents a row in the table.
Associated with each of the columns is an attribute in the record.
More complex structures such as graphs can be formed by storing the name
of a record in an attribute of another.
Most commonly the record name is constructed by concatenating the name
of the list and the index in the list, separated by a single dot.
For example, if the name is "x", then the record containing the third
row is named "x.2".
To add a row to a table, use:
.DS
char *mag_new(char *name);
.DE
This routines generates the name for a new row, adds it to the list
called
.I name ,
and returns the name of the new row.
.NH 2
Templates
.PP
Often it is necessary to build several instances of some data
structure in the data base.
The Magic system provides a module that
.I "stamps out"
data structures in the data base according to some
.I template .
The template can be configured using a
.I "variables record" .
.I Variables
are named by ASCII strings starting with an `@'-sign.
The variables record maps these variables to their values.
The records in a template are named by variables.
.PP
The routine
.DS
temp_stamp(char *template, char *variables_record);
.DE
takes the name of a template structure and the name the
variables record.
.I Temp_stamp
will generate a unique value for all variables in the template
that have a null value in the variables record.
The variable "@" is mapped to the name of the variables record.
Otherwise the template is copied literally.
.PP
Templates may best be explained by an example.
Say I want to create a template consisting of two records, each with
an attribute "other" pointing to the other record.
Then I create template records named, say, "@x" and "@y", with
attribute "other" in record "@x" set to value "@y", and attribute
"other" in record "@y" to value "@x".
Say "vars" is an ``empty'' record,
then \f2temp_stamp("@x", "vars")\fP will create an instance of the
template "@x".
The name for "@x" and "@y" in the template can be retrieved from the
"vars" record in attributes "@x" and "@y" respectively.
.NH
Magic Operators
.NH 2
Special and General Operators
.PP
In this section we will discuss the operators in Appendix A that need
extra explanation.
.PP
The ``#'' and ``$get'' operators allow data base retrieval in
postfix expression.
The expression { record attribute # } returns the current value of the
named record and attribute in the data base.
This can be used, for example, to make a ``symbolic link'' to
some value in the data base.
By assigning the string ``#a b #" to some attribute, this attribute
will always display the same value as the attribute ``b'' in record
``a''.
The ``$get'' operator takes an extra sequence number so that the data
base history can be accessed.
.PP
The ``$record'' and ``$attr'' operators evaluate to the name of the
record, resp.\& the name of the attribute being evaluated.
For example, in a table we can assign the value
"# $record $attr . $concat3" to the $number attribute.
If the table's name is "table", and we retrieve the value of the
attribute named "2", this will automatically result in the value
"table.2".
.PP
Data base selection can be done using the "$selcnt" and "$select"
operators.
These operators work on a list and select certain values depending
on some selection criterion.
The selection criterion is described by a postfix expression, which
is evaluated in a child context.
In this context, the "$entry" operator evaluates to the name of the
row, in the same way as the list operator.
$selcnt returns the number of entries for which the selection criterion
is true, and $select returns a specific entry in the selection.
.PP
For example, the list is a table that has a column called "load", and
the selection criterion is all the rows that have a
"load" attribute smaller than three, the postfix expression for the
selection would be { $entry load # 2 < }.
If the name of the table is "machines", we can create a new table
"lightload" by assigning the postfix expression
{ machines "$entry load # 2 <" $selcnt } to the "N" attribute, and
the expression { machines $attr "$entry load # 2 <" $select } to the
"$number" attribute of "lightload".
.PP
Instead of specifying the selection criterion twice, we can store it
in a separate attribute and use the $eval operator to evaluate it.
In this case, if we assign "$entry load # 2 <" to the attribute "sel"
of lightload, we can assign { machines "$record sel # $eval" } to the
"N" attribute and { machines $attr "$record sel # $eval" } to the
"$number" attribute.
Now the selection criterion can be easily changed to any other.
.PP
The ``$sort'' operator can be used to sort a list or table.
It takes the name of the list, an index, and a comparison function
as arguments.
The comparison function is evaluated in a child context, in which the
variables %1 and %2 are set to the two entries being compared.
It should return smaller than zero if %1 < %2, zero if %1 is %2, and
greater than zero if %1 > %2.
For example, to create a new table called "sorted" which is to contain
the "machines" table sorted on load, we assign { machines N # } to
the "N" attribute of "sorted", and
{ machines $attr "%1 load # %2 load # \-" $sort } to the "$number"
attribute.
.PP
If the list "lightload" is to be sorted, we can do this in two ways.
The first one is to substitute "lightload" for "machines" in the
expressions above.
The second one is to do it the other way around, by first sorting
machines as above, and then selecting the entries.
However, this last way is more expensive, since the sort function
will have to operate on more entries.
.PP
The ``$iter'' takes an operator and a number
.I n
as arguments, and evaluates the operator
.I n
times.
We can use this operator to calculate the average load on the machines
using the expression
{ machines "$entry load #" $list machines N # N := %N 1 \- "+" $iter %N / }.
Note the mixed use of "N", both as the name of an attribute and as a
variable.
In this example, "$list" pushes all the loads on the stack, "#"
retrieves the number of loads, and ":=" assigns this number to
.I N .
Then the "+" operator is evaluated
.I N
\- 1 times, adding all the loads.
This total is divided over
.I N
to generate the average load.
.SH
Notes
.PP
The "$db_ddcount", "$db_datadict", "$db_count", and "$db_attr"
operations can be used to implement a data dictionary.
In fact,
.I mag_cold()
initialized the attribute "N" in record "datadict" to { $db_ddcount },
and the attribute "$number" to { $attr $db_datadict }.
Thus the "datadict" record contains a list of all the record names
in the data base, including "datadict" itself.
.NH 2
Standard Functions
.PP
We have seen how the $eval operator can be used to store postfix
expressions in the data base and evaluate them.
In the Magic data base a set of standard function have been made
available this way in the record "std.func".
These are functions over real time, that is, time measured in seconds.
There are function to differentiate or integrate over time, taking
the minimum or maximum over time, or getting the value of an attribute
at some time.
These are, resp., stored in the attributes "diff", "int", "min", "max",
and "get" of the std.func record.
.PP
As time may also be provided as a value in the data base, we can match
sequence number to real time.
This way it is possible to see at what time an attributed changed its
value.
The functions in "std.func" assume that the "value" attribute of the
record "clock" is incremented by one every second.
Later we will discuss how this can be achieved in the Magic system.
.PP
For example, if the "load" attribute of the record "mach" is updated
regularly to contain the load of that machine, we can calculate the
average load over the last ten seconds using the expression
{ mach load 10 std.func int # $eval }.
.NH
Magic Communication
.PP
There can be only one Magic data base per process, and vice versa.
We call such a Magic process a
.I station .
The Magic system provides communication between stations by
asynchronous request/response exchanges.
A request or response message is a null-terminated list of values.
The first value in the list contains the name of the source station
of the message, and the sequence number that was current when the
message was sent.
In case of a request message, the second value contains the name of a
function to be invoked in the destination station.
The name of the station is specified using the
.I mag_init()
routine.
.PP
The communication interface comprises the following two procedures:
.DS
mes_subscribe(char *command, int (*request_proc)(struct value **message))
mes_send_request(struct value *station, struct value **message,
.ta \w'mes_send_req'u
	int (*reply_proc)(struct value *station, struct value **message));
.DE
.LP
.I Mes_subscribe()
specifies that when a request message is received with
.I command
as its second element,
.I request_proc()
is to be invoked with the request message as its argument.
This procedure returns a reply message which is sent back subsequently.
.PP
.I Mes_send_request()
sends
.I message
to the named station.
Both the station and message are given away to
.I mes_send_request() ,
and it will free them.
When a reply message is received,
.I reply_proc
is invoked with the name of the station and the reply message as its
arguments.
If
.I station
does not exist, or crashes during the handling of the request, the
procedure is invoked with a null reply message.
The reply procedure does not return any value.
.PP
Both request and reply procedures run uninterruptedly to completion.
At the end of these procedures, db_commit() is invoked automatically,
atomically committing changes to the data base.
At the start of these procedures, the name of the source station and
its sequence number can be stored in the data base, thus maintaining
a partial (or causal) order between the event histories of the
different stations.
The procedures are not allowed to update or free the message (or station),
but can copy the values using
.I val_ref() .
.SH
Notes
.PP
There is a standard reply procedure,
.DS
mag_reply(struct value *station, struct value **message);
.DE
that will print an appropriate error message if the message is null
or the second value is not "ok".
.NH 2
Message Allocation
.PP
Messages can be allocated and freed using
.DS
struct value **mes_alloc(int n);
mes_free(struct value **m);
.DE
.I Mes_alloc
returns a list of
.I n
+ 1 entries.
The first entry points to a value containing the name of this station
and its current sequence number
.I db_seq ;
the other entries are zero-filled.
.I Mes_free
frees the message.
.PP
The routine
.DS
mes_count(struct message **m);
.DE
returns the number of values in the list.
The routines
.DS
struct message **mes_ok();
struct message **mes_error(char *s);
.DE
can be used to generate standard reply messages.
.I Mes_ok()
creates a message containing the string "ok" as its second element.
.I Mes_error()
creates a message containing the string "error" as its second, and
.I s
as its third element.
.PP
Using the routine
.DS
mag_initmes(struct value *station, struct value **message);
.DE
it is possible to have messages sent automatically at
station start-up.
The station and message are stored in the data base, and retrieved
and sent by
.I mag_init() .
Like in
.I mes_send_request() ,
.I station
may be null, and the station and message are freed.
.NH 2
Standard Messages
.PP
The Magic system provides a set of standard request procedures, of
which some will be presented here. 
These are "put", "get", and "dump".
The "put" procedure takes a message of five elements: the source
station, the command name "put", a record name, an attribute, and
a value, and stores this value in the named record and attribute.
The "get" procedure takes a message of four or five elements,
containing a record name, an attribute, and possibly a sequence
number.
It retrieves a value from the named record and attribute, and uses
the current sequence number if there are only four elements.
The resulting value is put in the third entry of the reply.
.PP
The "dump" procedure takes no arguments.
It invokes
.I db_checkpoint()
to save the current contents of the data base.
.SH
Notes
.PP
The procedure
.DS
mes_print(struct value **m);
.DE
prints the list of values in the message
.I m
on standard output.
.PP
The utility ``magic'' can be used to send request messages to stations.
Magic copies its complete argument vector into the message.
The replies are printed using
.I mes_print() .
For example, the command "magic machines get mach load" will retrieve
the current value of the attribute "load" in the record "mach" in the
station called "machines".
.NH 2
Automatic Message Generation
.PP
It is possible to have requests sent automatically when the data base
is updated, by setting a
.I watch
on an attribute in the data base.
Each time the attribute changes, the watch sends a request message to a
specified station.
The routine to set a watch is:
.DS
watch_specify(char *station, char *command, char *record, char *attribute);
.DE
.LP
Each time
.I attribute
in
.I record
changes, a message containing
.I command ,
.I record ,
.I attribute ,
and the new value is sent to station.
.I Station
may be null (V_NULL has been defined for this purpose),
in which case the message is sent to the station itself.
.PP
By setting watches, stations can keep copies of values in multiple
data bases.
For example, to keep station "station" up-to-date on record "record"
and attribute "attribute", call
watch_specify("station", "put", "record", "attribute").
This will invoke the standard command "put" automatically in "station"
each time the attribute changes its current value.
Often the station specified in a watch is the station that holds the
record itself.
This way a request message task is started each time the attribute
changes, so that the station can take appropriate control actions.
.PP
For efficiency it may be sufficient if a watch reports updates at
most, say, once every ten seconds.
For example, if we have an attribute "x", we can assign a new attribute
"y" the function { $record x clock value # 10 % std.func get # $eval }.
.SH
Notes
.PP
.I Mag_init()
provides for the standard request "watch", which takes a command, a
record, and an attribute as arguments.
This request will put a watch on the specified attribute, using the
source station of the request as the destination station of updates.
The routine
.DS
watch_cold(char *station, char *cmd, char *record, char *attribute);
.DE
sends a "watch" message to the specified station at cold start.
.PP
There is a standard watch on the "status" attribute of the "control"
record.
.I Mag_init
initializes this attribute to "run".
If the status is no longer equal to "run", the station process will
set the flag
.I mag_done
to break the main loop.
Therefore it is possible to terminate a station externally using
the command "magic station put control status stop".
.NH
The Magic Visual Interface
.PP
The Magic system provides a visual interface that is connected
directly to the data base.
With this we mean that anything displayed has been described in the
data base, and any manual updates of the visual display directly
result in updates in the data base.
.PP
The visual interface consists of a set of windows, each of which may
or may not be mapped.
The information whether the window is mapped or not is also stored in
the data base.
The windows contain so-called
.I widgets ,
which are visual objects representing parts of the data base.
Data connected with the widget, such as position and size, are also
stored in the data base.
In the next section these will be explained using an example.
This section will describe the ideas concisely.
.NH 2
Widgets
.PP
Widgets are described by records.
Each widget has the following attributes:
.DS
.ta \w'width, height    'u
type	the type of the widget
x, y	position
width, height	size
frame	rectangular frame if non-zero
select	selectable by mouse if non-zero
.DE
The
.I type
attribute describes the type of the widget, such as "menu",
"graph", or "table".
The position and size of the attribute is described by
.I x ,
.I y ,
.I width ,
and
.I height .
The units of these are in screen pixels, but can be scaled.
The position (0, 0) is in the top-left corner, thus height extends
downwards.
Some widgets accept negative widths and heights.
.PP
If the
.I frame
attribute is non-zero, a rectangular frame will be drawn around the
widget.
This works for widgets with an unknown type, and this feature can be
used to draw boxes.
.PP
If the
.I select
attribute is non-zero, the widget can be selected by the mouse.
Mouse selection is done by pointing with the mouse cursor at the
widget, and depressing the first (left-most) button.
Widgets can also track the mouse while the button is depressed.
The table widget can also track the last (right-most) button.
.PP
In the following sections we will discuss each of the widgets types.
.NH 3
Simple Widgets
.PP
The following widgets are not selectable.
They draw simple things.
They are
.DS
line
arrow
circle
hand
.DE
Line and arrow widgets are drawn from the (x, y) position to
(x + width, y + height).
The circle widget draws an ellipse filling the box defined by
the frame of the widget.
A hand widget is like a line or arrow widget, but is shaped
for the hand of a clock or dial.
The thickness of the hand can be controlled by the attribute
"thickness".
.NH 3
Graph Widgets
.PP
These widgets display data in some format, and can update the data
by mouse selection and/or keyboard input.
They are
.DS
container
graph
table
scroll
menu
.DE
.SH
Container
.PP
A container widget is like a scroll bar.
The height of the scroll bar is connected to an attribute in the data
base.
A container widget has the following attributes:
.DS
.ta \w'direction      'u
record	the record involved
attr	attribute in this record
interval	integer interval of values
direction	north, south, east, or west
.DE
The
.I record
and
.I attr
attributes define to which attribute in the data this scroll bar connects.
.I Interval
defines the range of values, being any integer between 0 and
.I interval
\- 1.
Direction defines in which direction the scroll bar grows.
If null (default), the direction is to above (north).
The scroll bar inverts black and white of whatever it covers.
.PP
By making an interval of size 1, the container widget will be binary,
and in effect behave as a binary switch.
Clicking in the switch will flip the value of the attribute.
.SH
Graph
.PP
A graph widget shows the value of an attribute over some time
interval in history.
To retrieve time, it uses the "value" attribute of the "clock"
record, which should be incremented at even intervals, usually
seconds.
Graph widgets are not selectable.
A graph widget has the following attributes:
.DS
.ta \w'direction      'u
interval	time interval
min	minimum value displayed
max	maximum value displayed
style	normal or histogram
fill	blank or black filled
.DE
.I Interval
defines the period in time.
.I Min
and
.I max
define the range of values displayed.
By setting
.I style
to "hist" a histogram is displayed instead of a normal graph.
By setting
.I fill
to "black", all space below the graph is blackened.
.SH
Table
.PP
Table widgets are used to display text.
The text is stored in table structures.
A table widget has the following attributes:
.DS
.ta \w'direction      'u
record	table record
start	first entry to be displayed
number	number of entries displayed
N	# attributes or columns displayed
0..<N \- 1>	description of columns
direct	display not a list, but a record
transpose	transpose rows and columns
.DE
.I Record
is the name of the table.
The table has attributes "N" and "0" through
.I N
\- 1,
one for each entry in the table.
The table widget displays a row for each entry in the table.
If
.I start
and
.I number
are defined, the table widget only shows
.I number
entries, starting with entry
.I start .
.PP
The table widget itself is a table.
Each entry in the table widget describes a column in the table
displayed.
.I N
in the table widget is the number of columns.
Each record in the table widget has the following attributes:
.DS
.ta \w'format    'u
attr	name of attribute displayed
width	relative width of the column
offset	offset in column
format	left, center, or right adjusted
mode	ro (read-only) or blank (read-write)
.DE
.I Attr
names which attribute is displayed in this column.
.I Width
is the relative width of the column.
For example, if there are two columns, one having width 1, and the
other width 2, the second column is twice as wide as the first one.
.I Offset
is the offset (the width of the margin) in the column.
Using the
.I format
attribute, the adjustment of the strings in the columns can be
specified.
Default is centered.
The routine
.DS
tbl_column(char *name, char *attr, int width, int offset,
.ta \w'tbl_column('u
	char *format, char *mode);
.DE
adds a new column to a table widget.
.PP
Text in tables is selectable by the mouse.
By clicking and dragging the mouse it is possible to select pieces
of text, and, unless the
.I mode
is set to "ro" (read-only), the selection can be overwritten.
Single clicking does character selection, double clicking word
selection, and triple clicking complete entry selection.
If the text being displayed is null, the whole entry will be
blackened.
Typing a character overwrites the selection with that character.
A selection may be between two characters, used for insertion.
Certain control keys have special effects:
.DS
.ta \w'^N or ^J <Line Feed>    'u
<DEL>	delete the selection
^H or <Back Space>	also delete the character before the selection
^W	also delete the word before the selection
^U	delete the whole entry
^M or <Return>	go to the first entry in the next row
^N or ^J <Line Feed>	go to the next row
^F, ^I, or ^L	go to the next column
^K or ^P	go to the previous row
^B		go to the previous column
^G		select the whole entry
.DE
If
.I direct
is non-zero, there is only one row, and the
.I record
attribute points to the record to be displayed.
This is useful for displaying simply text, and not tables.
If
.I transpose
is non-zero, records and attributes are reversed.
This means that the rows display attributes, and the columns rows.
The table record draws lines between the rows and columns.
.SH
Scroll
.PP
The scroll widget has been especially designed to be used in
conjunction with table widgets.
It has only one attribute,
.I record ,
which usually points to a table widget.
The scroll widget manipulates the
.I start
and
.I number
attributes in the table widget so that only part of the table is
displayed.
This is useful for large tables.
The scroll widget is usually placed directly to the left of the
table, and is of equal height.
.PP
The scroll widget displays a black bar, corresponding to the part of
the table that is displayed.
The bar can be moved up and down with the mouse, and this way
different parts of the table become visible.
.SH
Menu
.PP
It has two attributes:  "record" and "horiz".
The
.I record
attribute points to a list record.
Besides having
.I N
and index attributes, the list record has an attribute
.I select
that selects one entry in the list.
.PP
A menu widget contains
.I N
even sized slots.
The slot selected by the
.I select
attribute is inverted.
The
.I select
attribute may be null, in which case no slot is selected.
By depressing the mouse in a slot, the
.I select
attribute is updated.
.PP
Usually the slots are stacked on top of each other, the highest slot
corresponding to zero.
If the attribute
.I horiz
is non-zero ,
the slots are positioned next to each other, the left-most one
corresponding to zero.
Normally menus are placed on top of (unselectable) table widgets,
displaying the text in the list record.
.NH 3
Picture Widget
.PP
A
.I picture
widget is a compound widget built out of other widgets.
Using mouse selection the placement and size of the widgets can be
controlled.
A picture widget has only one attribute,
.I record ,
pointing to a list of widget records to be displayed.
The widgets in the list are position relative to the origin of the
picture widget, and are scaled relative to the width and height of
the picture widget.
The normalized size of picture widgets is 8192 by 8192 units.
If unscaled, each unit corresponds to one pixel on the screen.
.PP
Using the mouse, widgets in a picture widget can be created, moved,
resized, deleted, or edited in some other way.
This is controlled by four records:
.DS
.ta \w'edit.template    'u
edit.mode	new, move, resize, edit, or delete
edit.type	creation type of widget, or template
edit.template	creation template
edit.raster	restrict positioning
.DE
.I Edit.mode
is a list of operations.
The "select" attribute in
.I edit.mode
selects an operation in this list to be started when the mouse is
depressed in a selectable widget in the picture.
If the mode is "new", a new widget is created within the picture.
.I Edit.type
is a list of widget types.
The "select" attribute in
.I edit.type
selects which widget type is to be created.
When depressed, the user can drag a rubber band to define the size
of the widget.
The
.I edit
mode will be described later when the editor is discussed.
.PP
One special type, "template", is reserved for stamping out a
template, using the
.I temp_stamp()
primitive.
It is possible to create a picture template of a set of widgets.
This is convenient if a set of similar pictures is to be created.
If
.I edit.type
is "template", the template name of the widget and the variables
record name will be retrieved from the record
.I edit.template ,
in the attributes "record" and "vars" respectively.
The "frame" attribute of this record is copied to the "frame"
attribute of the picture widget that is stamped out.
.PP
The "raster" attribute in
.I edit.raster
restricts the placement of objects to five pixels, if non-zero.
This makes it easier to arrange widgets manually in a picture.
.PP
The routine
.DS
char *pic_new(char *name, char *type, char *record,
.ta \w'char *pic_new('u
	int x, int y, int width, int height, int frame, int select);
.DE
creates a new widget record in picture
.I name
with the specified attributes, and returns the name of the new widget
record.
.PP
The routine
.DS
line_append(char *name, int x1, int y1, int x2, int y2);
.DE
creates a new line widget in the picture widget
.I name .
The line is drawn between (x1, y1) and (x2, y2).
.NH 3
Bar Widget
.PP
The bar widget is another type of scroll bar.
However, it is not connected to any data in the data base, but
controls the time of the data which is being displayed.
It is therefore often called the ``history bar.''
It is commonly placed to the extreme left of windows.
By depressing the mouse in the history bar, the time of data being
displayed is set relative to the height of the bar and the length of
history.
There is a small box in the bottom of the history bar.
By depressing the mouse there, the current values of data are
displayed.
.NH 2
Windows
.PP
Each station may have one or more windows.
Associated with each window is a picture widget defining the contents
of the window.
The "control" record maintains a list of these pictures.
The records defining the contents of the window have the following
extra attributes:
.DS
.ta \w'width, height    'u
x, y	position on screen
width, height	size of the window
mapped	mapped or not
.DE
These attributes may be manipulated directly using the window manager
functions of the window system you are using.
However, currently the
.I mapped
attribute only works by direct manipulation in the data base.
(That is,
.I iconifying
has no lasting effect.)
.PP
The routine
.DS
mag_picture(char *pic, char *record, char *descr,
.ta \w'mag_picture('u
	int width, int height, int mapped);
.DE
adds a picture widget record
.I pic
to the control record.
Moreover, it sets the
.I record
attribute, and the
.I width ,
.I height ,
and
.I mapped
attributes therein.
.I Descr
is put in the
.I descr
attribute of the picture widget, and is used by the editor as
described later.
It describes the contents of the window.
.NH 2
Dumping Windows
.PP
The contents of windows can be dumped on files, and subsequently
printed.
The contents of a window is dumped by
.DS
win_dump(int window, char *file);
.DE
.I Win_dump
dumps the contents of the window defined by index
.I window
in the "control" record onto the specified file in some internal
encoding.
There is also a standard request message to do so, called
"windump", which takes the window and file parameters as
arguments.
For example, "magic clock windump 0 file" dumps window 0 in the
"clock" station onto the file "file".
.PP
The file can be converted to
.I PostScript
or to
.I pic
format using the utilities "tops" and "topic" respectively.
These utilities read the internal encoding on standard input, and
produce the intended format on standard output.
The internal encoding consists of lines of the following format:
.DS
.ta \w'p 'u
l	x1 y1 x2 y2
c	x y width height
r	x y width height
p	N x1 y1 ... xN yN
s	x y width height format string
.DE
The first character defines the object to be drawn (line, circle,
rectangle, polygon, or string respectively).
The rectangle should make the contents of the rectangle stand out
against the background, for example by inverting it.
The polygon should be filled.
The format of the string defines how it should be adjusted in the
defined rectangle.
.NH
A Magic Example
.PP
In this section we will describe an example, which may make the
ideas described above somewhat clear.
We encourage you to run the example for real, instead of just reading
the text below.
The example we are going to discuss is the
.I clock
utility.
You can start one up yourself using the commands
.DS
sun4/bin/clock_init
sun4/bin/clock&
.DE
.I Clock_init
initialized the clock data base, and
.I clock
displays a window with a clock widget, as shown in Fig.\|\*(Fn.
Position the clock widget in a place on the screen where you would
like to have it.
.F1
.ps -2
.PS 3
line from (0, 100) to (12, 100)
line from (12, 100) to (12, 0)
line from (12, 0) to (0, 0)
line from (0, 0) to (0, 100)
line from (0, 12) to (12, 12)
line from (17, 95) to (42, 95)
line from (42, 95) to (42, 70)
line from (42, 70) to (17, 70)
line from (17, 70) to (17, 95)
line from (17, 70) to (42, 70) to (42, 70) to (17, 70) to (17, 70) to (42, 70)
line from (17, 70) to (42, 70)
line from (17, 30) to (232, 30)
line from (232, 30) to (232, 10)
line from (232, 10) to (17, 10)
line from (17, 10) to (17, 30)
"Fri Jun 15 17:14:16 1990" at (124, 20) center
line from (210, 70) to (229, 68)
line from (207, 72) to (226, 71) to (208, 67) to (207, 72)
line from (211, 73) to (214, 59) to (207, 71) to (211, 73)
line from (212, 93) to (212, 94)
line from (214, 93) to (214, 94)
line from (217, 92) to (217, 93)
line from (219, 92) to (219, 93)
line from (223, 89) to (223, 90)
line from (225, 88) to (226, 89)
line from (227, 86) to (228, 87)
line from (228, 84) to (229, 84)
line from (231, 80) to (232, 80)
line from (231, 78) to (232, 78)
line from (232, 75) to (233, 75)
line from (232, 73) to (233, 73)
line from (232, 68) to (233, 68)
line from (232, 66) to (233, 66)
line from (231, 63) to (232, 63)
line from (231, 61) to (232, 61)
line from (228, 57) to (229, 57)
line from (227, 55) to (228, 54)
line from (225, 53) to (226, 52)
line from (223, 52) to (223, 51)
line from (219, 49) to (219, 48)
line from (217, 49) to (217, 48)
line from (214, 48) to (214, 47)
line from (212, 48) to (212, 47)
line from (207, 48) to (207, 47)
line from (205, 48) to (205, 47)
line from (202, 49) to (202, 48)
line from (200, 49) to (200, 48)
line from (196, 52) to (196, 51)
line from (194, 53) to (193, 52)
line from (192, 55) to (191, 54)
line from (191, 57) to (190, 57)
line from (188, 61) to (187, 61)
line from (188, 63) to (187, 63)
line from (187, 66) to (186, 66)
line from (187, 68) to (186, 68)
line from (187, 73) to (186, 73)
line from (187, 75) to (186, 75)
line from (188, 78) to (187, 78)
line from (188, 80) to (187, 80)
line from (191, 84) to (190, 84)
line from (192, 86) to (191, 87)
line from (194, 88) to (193, 89)
line from (196, 89) to (196, 90)
line from (200, 92) to (200, 93)
line from (202, 92) to (202, 93)
line from (205, 93) to (205, 94)
line from (207, 93) to (207, 94)
line from (210, 93) to (210, 95)
line from (220, 90) to (221, 92)
line from (229, 82) to (231, 83)
line from (231, 70) to (233, 70)
line from (229, 60) to (231, 59)
line from (220, 51) to (221, 49)
line from (209, 49) to (209, 47)
line from (198, 51) to (197, 49)
line from (190, 60) to (188, 59)
line from (187, 71) to (185, 71)
line from (190, 82) to (188, 83)
line from (198, 90) to (197, 92)
.PE
.ps +2
.F2
The main
.I clock
window.
.F3
.PP
In the upper left corner of the window there is a small box.
When you click in it with the mouse, a new window will appear
showing a menu of selections.
The box, in fact, is a container widget with an interval of size one,
and connected to the "mapped" attribute of the menu window.
Select the "edit" entry in the menu with the mouse.
(If you accidentally select something else, just leave it there and
try again.)
The menu window disappears, and a new window appears as shown in
Fig.\|\*(Fn.
This window is the editor window, and using it we will make a tour
of the clock data base.
.F1
.ps -2
.PS 5
line from (10, 590) to (40, 590)
line from (40, 590) to (40, 560)
line from (40, 560) to (10, 560)
line from (10, 560) to (10, 590)
"\s25"
line from (10, 560) to (40, 560) to (40, 560) to (10, 560) to (10, 560)
"\s0"
line from (50, 590) to (380, 590)
line from (380, 590) to (380, 560)
line from (380, 560) to (50, 560)
line from (50, 560) to (50, 590)
line from (440, 590) to (440, 560)
line from (490, 590) to (490, 560)
line from (540, 590) to (540, 560)
"rec" at (415, 575) center
"tbl" at (465, 575) center
"wid" at (515, 575) center
"pic" at (565, 575) center
line from (390, 590) to (590, 590)
line from (590, 590) to (590, 560)
line from (590, 560) to (390, 560)
line from (390, 560) to (390, 590)
"\s25"
line from (390, 590) to (440, 590) to (440, 560) to (390, 560) to (390, 590)
"\s0"
line from (22, 550) to (162, 550)
line from (162, 550) to (162, 530)
line from (162, 530) to (22, 530)
line from (22, 530) to (22, 550)
line from (162, 550) to (590, 550)
line from (590, 550) to (590, 530)
line from (590, 530) to (162, 530)
line from (162, 530) to (162, 550)
line from (22, 520) to (162, 520)
line from (162, 520) to (162, 120)
line from (162, 120) to (22, 120)
line from (22, 120) to (22, 520)
line from (22, 500) to (162, 500)
line from (22, 480) to (162, 480)
line from (22, 460) to (162, 460)
line from (22, 440) to (162, 440)
line from (22, 420) to (162, 420)
line from (22, 400) to (162, 400)
line from (22, 380) to (162, 380)
line from (22, 360) to (162, 360)
line from (22, 340) to (162, 340)
line from (22, 320) to (162, 320)
line from (22, 300) to (162, 300)
line from (22, 280) to (162, 280)
line from (22, 260) to (162, 260)
line from (22, 240) to (162, 240)
line from (22, 220) to (162, 220)
line from (22, 200) to (162, 200)
line from (22, 180) to (162, 180)
line from (22, 160) to (162, 160)
line from (22, 140) to (162, 140)
line from (162, 520) to (590, 520)
line from (590, 520) to (590, 120)
line from (590, 120) to (162, 120)
line from (162, 120) to (162, 520)
line from (162, 500) to (590, 500)
line from (162, 480) to (590, 480)
line from (162, 460) to (590, 460)
line from (162, 440) to (590, 440)
line from (162, 420) to (590, 420)
line from (162, 400) to (590, 400)
line from (162, 380) to (590, 380)
line from (162, 360) to (590, 360)
line from (162, 340) to (590, 340)
line from (162, 320) to (590, 320)
line from (162, 300) to (590, 300)
line from (162, 280) to (590, 280)
line from (162, 260) to (590, 260)
line from (162, 240) to (590, 240)
line from (162, 220) to (590, 220)
line from (162, 200) to (590, 200)
line from (162, 180) to (590, 180)
line from (162, 160) to (590, 160)
line from (162, 140) to (590, 140)
line from (10, 520) to (22, 520)
line from (22, 520) to (22, 120)
line from (22, 120) to (10, 120)
line from (10, 120) to (10, 520)
.PE
.ps +2
.F2
The
.I editor
window.
.F3
.PP
The top left box in the editor window is a container widget that
is linked to the "mapped" attribute of the editor window itself.
Thus by selecting it we can unmap the editor window.
If you do so, you will have to pop up the menu and select the editor
again.
.PP
Next to it is a rectangular box that will hold the name of the record
to be edited.
If you click in it it will turn black, meaning that the selection is
currently null.
Now type the string "control".
The window will then have changed as displayed in Fig.\|\*(Fn.
.F1
.ps -2
.PS 5
line from (10, 590) to (40, 590)
line from (40, 590) to (40, 560)
line from (40, 560) to (10, 560)
line from (10, 560) to (10, 590)
"\s25"
line from (10, 590) to (40, 590) to (40, 560) to (10, 560) to (10, 590)
"\s0"
line from (50, 590) to (380, 590)
line from (380, 590) to (380, 560)
line from (380, 560) to (50, 560)
line from (50, 560) to (50, 590)
"control" at (215, 575) center
"\s25"
line from (245, 583) to (246, 583) to (246, 567) to (245, 567) to (245, 583)
"\s0"
line from (440, 590) to (440, 560)
line from (490, 590) to (490, 560)
line from (540, 590) to (540, 560)
"rec" at (415, 575) center
"tbl" at (465, 575) center
"wid" at (515, 575) center
"pic" at (565, 575) center
line from (390, 590) to (590, 590)
line from (590, 590) to (590, 560)
line from (590, 560) to (390, 560)
line from (390, 560) to (390, 590)
"\s25"
line from (390, 590) to (440, 590) to (440, 560) to (390, 560) to (390, 590)
"\s0"
line from (22, 550) to (162, 550)
line from (162, 550) to (162, 530)
line from (162, 530) to (22, 530)
line from (22, 530) to (22, 550)
line from (162, 550) to (590, 550)
line from (590, 550) to (590, 530)
line from (590, 530) to (162, 530)
line from (162, 530) to (162, 550)
line from (22, 520) to (162, 520)
line from (162, 520) to (162, 120)
line from (162, 120) to (22, 120)
line from (22, 120) to (22, 520)
line from (22, 500) to (162, 500)
"name" at (92, 510) center
line from (22, 480) to (162, 480)
"0" at (92, 490) center
line from (22, 460) to (162, 460)
"1" at (92, 470) center
line from (22, 440) to (162, 440)
"2" at (92, 450) center
line from (22, 420) to (162, 420)
"3" at (92, 430) center
line from (22, 400) to (162, 400)
"4" at (92, 410) center
line from (22, 380) to (162, 380)
"5" at (92, 390) center
line from (22, 360) to (162, 360)
"6" at (92, 370) center
line from (22, 340) to (162, 340)
"7" at (92, 350) center
line from (22, 320) to (162, 320)
"select" at (92, 330) center
line from (22, 300) to (162, 300)
"status" at (92, 310) center
line from (22, 280) to (162, 280)
"N" at (92, 290) center
line from (22, 260) to (162, 260)
line from (22, 240) to (162, 240)
line from (22, 220) to (162, 220)
line from (22, 200) to (162, 200)
line from (22, 180) to (162, 180)
line from (22, 160) to (162, 160)
line from (22, 140) to (162, 140)
line from (162, 520) to (590, 520)
line from (590, 520) to (590, 120)
line from (590, 120) to (162, 120)
line from (162, 120) to (162, 520)
line from (162, 500) to (590, 500)
"control" at (376, 510) center
line from (162, 480) to (590, 480)
"window" at (376, 490) center
line from (162, 460) to (590, 460)
"temp.win" at (376, 470) center
line from (162, 440) to (590, 440)
"edit.win" at (376, 450) center
line from (162, 420) to (590, 420)
"dump.win" at (376, 430) center
line from (162, 400) to (590, 400)
"quit.win" at (376, 410) center
line from (162, 380) to (590, 380)
"watch.win" at (376, 390) center
line from (162, 360) to (590, 360)
"calc.win" at (376, 370) center
line from (162, 340) to (590, 340)
"menu.win" at (376, 350) center
line from (162, 320) to (590, 320)
line from (162, 300) to (590, 300)
"run" at (376, 310) center
line from (162, 280) to (590, 280)
"8" at (376, 290) center
line from (162, 260) to (590, 260)
line from (162, 240) to (590, 240)
line from (162, 220) to (590, 220)
line from (162, 200) to (590, 200)
line from (162, 180) to (590, 180)
line from (162, 160) to (590, 160)
line from (162, 140) to (590, 140)
line from (10, 520) to (22, 520)
line from (22, 520) to (22, 120)
line from (22, 120) to (10, 120)
line from (10, 120) to (10, 520)
"\s25"
line from (11, 519) to (22, 519) to (22, 120) to (11, 120) to (11, 519)
"\s0"
.PE
.ps +2
.F2
The editor window, now showing the "control" record.
.F3
.PP
The "control" record, as discussed earlier, contains the list of
windows that the "clock" station defines.
The attribute "N" holds the number of windows (8).
Attribute "0" holds the name of the picture widget containing the main
window ("window").
Let's have a look at it.
Double click in the selection box, and overwrite the string "control"
with "window".
Now the "window" record is being displayed.
As discussed, it describes a picture widget.
The contents of the picture widget is described in the "widget"
record, as denoted by the "record" attribute of the "window" record.
Have a look at it by overwriting the selection with "widget".
.PP
The "widget" record is a list of widgets defining the main clock
window.
There are four widgets:  the history bar, the menu container, the
clock (a picture widget), and a table holding the current time.
The attribute
.I x ,
.I y ,
.I width ,
.I height ,
and
.I mapped
hold window information.
Notice that the window position on the screen is maintained here.
.PP
Let's now look at the vertical menu in the top right corner of the
editor window.
It has four selections.
Currently the "rec" selection signifies that a record is being
displayed.
Select the "tbl" selection, and the "widget" list is being
displayed.
Skip the "wid" selection currently, and try the "pic" selection.
This displays Fig.\|\*(Fn.
.F1
.ps -2
.PS 5
line from (10, 590) to (40, 590)
line from (40, 590) to (40, 560)
line from (40, 560) to (10, 560)
line from (10, 560) to (10, 590)
"\s25"
line from (10, 590) to (40, 590) to (40, 560) to (10, 560) to (10, 590)
"\s0"
line from (50, 590) to (380, 590)
line from (380, 590) to (380, 560)
line from (380, 560) to (50, 560)
line from (50, 560) to (50, 590)
"widget" at (215, 575) center
line from (440, 590) to (440, 560)
line from (490, 590) to (490, 560)
line from (540, 590) to (540, 560)
"rec" at (415, 575) center
"tbl" at (465, 575) center
"wid" at (515, 575) center
"pic" at (565, 575) center
line from (390, 590) to (590, 590)
line from (590, 590) to (590, 560)
line from (590, 560) to (390, 560)
line from (390, 560) to (390, 590)
"\s25"
line from (540, 590) to (590, 590) to (590, 560) to (540, 560) to (540, 590)
"\s0"
"scale" at (55, 540) center
line from (10, 550) to (100, 550)
line from (100, 550) to (100, 530)
line from (100, 530) to (10, 530)
line from (10, 530) to (10, 550)
"\s25"
line from (10, 530) to (100, 530) to (100, 530) to (10, 530) to (10, 530)
"\s0"
"raster" at (55, 510) center
line from (10, 520) to (100, 520)
line from (100, 520) to (100, 500)
line from (100, 500) to (10, 500)
line from (10, 500) to (10, 520)
"\s25"
line from (10, 520) to (100, 520) to (100, 500) to (10, 500) to (10, 520)
"\s0"
line from (10, 450) to (100, 450)
"new" at (55, 460) center
line from (10, 430) to (100, 430)
"move" at (55, 440) center
line from (10, 410) to (100, 410)
"resize" at (55, 420) center
line from (10, 390) to (100, 390)
"edit" at (55, 400) center
"delete" at (55, 380) center
line from (10, 470) to (100, 470)
line from (100, 470) to (100, 370)
line from (100, 370) to (10, 370)
line from (10, 370) to (10, 470)
"\s25"
line from (10, 470) to (100, 470) to (100, 450) to (10, 450) to (10, 470)
"\s0"
line from (10, 305) to (100, 305)
"graph" at (55, 315) center
line from (10, 285) to (100, 285)
"container" at (55, 295) center
line from (10, 265) to (100, 265)
"table" at (55, 275) center
line from (10, 245) to (100, 245)
"menu" at (55, 255) center
line from (10, 225) to (100, 225)
"scroll" at (55, 235) center
line from (10, 205) to (100, 205)
"picture" at (55, 215) center
line from (10, 185) to (100, 185)
"line" at (55, 195) center
line from (10, 165) to (100, 165)
"arrow" at (55, 175) center
line from (10, 145) to (100, 145)
"box" at (55, 155) center
line from (10, 125) to (100, 125)
"circle" at (55, 135) center
"template" at (55, 115) center
line from (10, 325) to (100, 325)
line from (100, 325) to (100, 105)
line from (100, 105) to (10, 105)
line from (10, 105) to (10, 325)
"\s25"
line from (10, 325) to (100, 325) to (100, 305) to (10, 305) to (10, 325)
"\s0"
line from (110, 550) to (560, 550)
line from (560, 550) to (560, 525)
line from (560, 525) to (110, 525)
line from (110, 525) to (110, 550)
line from (302, 550) to (302, 525)
line from (495, 550) to (495, 525)
line from (110, 490) to (122, 490)
line from (122, 490) to (122, 390)
line from (122, 390) to (110, 390)
line from (110, 390) to (110, 490)
line from (110, 402) to (122, 402)
line from (127, 485) to (152, 485)
line from (152, 485) to (152, 460)
line from (152, 460) to (127, 460)
line from (127, 460) to (127, 485)
"\s25"
line from (127, 460) to (152, 460) to (152, 460) to (127, 460) to (127, 460)
"\s0"
line from (127, 420) to (342, 420)
line from (342, 420) to (342, 400)
line from (342, 400) to (127, 400)
line from (127, 400) to (127, 420)
"Tue Jun 19 17:08:01 1990" at (234, 410) center
line from (320, 460) to (322, 479)
line from (317, 460) to (332, 471) to (320, 457) to (317, 460)
line from (321, 463) to (325, 449) to (317, 461) to (321, 463)
line from (322, 483) to (322, 484)
line from (324, 483) to (324, 484)
line from (327, 482) to (327, 483)
line from (329, 482) to (329, 483)
line from (333, 479) to (333, 480)
line from (335, 478) to (336, 479)
line from (337, 476) to (338, 477)
line from (338, 474) to (339, 474)
line from (341, 470) to (342, 470)
line from (341, 468) to (342, 468)
line from (342, 465) to (343, 465)
line from (342, 463) to (343, 463)
line from (342, 458) to (343, 458)
line from (342, 456) to (343, 456)
line from (341, 453) to (342, 453)
line from (341, 451) to (342, 451)
line from (338, 447) to (339, 447)
line from (337, 445) to (338, 444)
line from (335, 443) to (336, 442)
line from (333, 442) to (333, 441)
line from (329, 439) to (329, 438)
line from (327, 439) to (327, 438)
line from (324, 438) to (324, 437)
line from (322, 438) to (322, 437)
line from (317, 438) to (317, 437)
line from (315, 438) to (315, 437)
line from (312, 439) to (312, 438)
line from (310, 439) to (310, 438)
line from (306, 442) to (306, 441)
line from (304, 443) to (303, 442)
line from (302, 445) to (301, 444)
line from (301, 447) to (300, 447)
line from (298, 451) to (297, 451)
line from (298, 453) to (297, 453)
line from (297, 456) to (296, 456)
line from (297, 458) to (296, 458)
line from (297, 463) to (296, 463)
line from (297, 465) to (296, 465)
line from (298, 468) to (297, 468)
line from (298, 470) to (297, 470)
line from (301, 474) to (300, 474)
line from (302, 476) to (301, 477)
line from (304, 478) to (303, 479)
line from (306, 479) to (306, 480)
line from (310, 482) to (310, 483)
line from (312, 482) to (312, 483)
line from (315, 483) to (315, 484)
line from (317, 483) to (317, 484)
line from (320, 483) to (320, 485)
line from (330, 480) to (331, 482)
line from (339, 472) to (341, 473)
line from (341, 460) to (343, 460)
line from (339, 450) to (341, 449)
line from (330, 441) to (331, 439)
line from (319, 439) to (319, 437)
line from (308, 441) to (307, 439)
line from (300, 450) to (298, 449)
line from (297, 461) to (295, 461)
line from (300, 472) to (298, 473)
line from (308, 480) to (307, 482)
.PE
.ps +2
.F2
The editor window, showing the "widget" record in picture mode.
.F3
.ds pI \*(Fp
.PP
The
.I picture
mode lets you edit pictures directly.
For example, select "move" in the top left menu, pick up the clock
in the editor window with the mouse, and put it wherever you want it.
Similarly, if you select "resize", you can resize the clock and give
it a multitude of shapes, invert it, put it upside down, or whatever
you want to do with it.
Notice that in the main window the updates are displayed directly.
Now select edit, and click on the clock.
.PP
The editor is now in "wid" mode, allowing you to edit widget records.
In this case, it is a picture widget, described by the record called
"clock".
Each line in the clock, and each hand is another widget.
The record "clock.0" describes the second hand, which is a line of
which the width and height are functions of the time.
Overwrite the selection with "clock.0" and you'll see it more
clearly.
Overwrite, for example, the "line" type with "arrow", and see how it
changes the second hand into an arrow in the main window.
.PP
Now let's try something funny.
Overwrite the type of the widget with "picture", and the "record"
field with "clock".
Thus, we have made the second hand the clock itself, and this works
recursively.
This is not very clear from the main window, as it will be too small.
.PP
So let's look at the clock record itself.
Delete the ".0" part of the selection.
You can do that by overwriting the whole selection, but also by typing
two backspaces.
Then select the "pic" mode.
Now you'll see only white space, because we are looking at the top
left corner of the clock widget.
We have to scale it down, and can do so by clicking in the top left
button (container widget) labeled "scale".
Now the whole clock will fill the editor window.
See how neatly the second hand is a changing clock image, as shown in
Fig.\|\*(Fn.
.F1
.ps -2
.PS 5
line from (10, 590) to (40, 590)
line from (40, 590) to (40, 560)
line from (40, 560) to (10, 560)
line from (10, 560) to (10, 590)
"\s25"
line from (10, 590) to (40, 590) to (40, 560) to (10, 560) to (10, 590)
"\s0"
line from (50, 590) to (380, 590)
line from (380, 590) to (380, 560)
line from (380, 560) to (50, 560)
line from (50, 560) to (50, 590)
"clock" at (215, 575) center
line from (440, 590) to (440, 560)
line from (490, 590) to (490, 560)
line from (540, 590) to (540, 560)
"rec" at (415, 575) center
"tbl" at (465, 575) center
"wid" at (515, 575) center
"pic" at (565, 575) center
line from (390, 590) to (590, 590)
line from (590, 590) to (590, 560)
line from (590, 560) to (390, 560)
line from (390, 560) to (390, 590)
"\s25"
line from (540, 590) to (590, 590) to (590, 560) to (540, 560) to (540, 590)
"\s0"
"scale" at (55, 540) center
line from (10, 550) to (100, 550)
line from (100, 550) to (100, 530)
line from (100, 530) to (10, 530)
line from (10, 530) to (10, 550)
"\s25"
line from (10, 550) to (100, 550) to (100, 530) to (10, 530) to (10, 550)
"\s0"
"raster" at (55, 510) center
line from (10, 520) to (100, 520)
line from (100, 520) to (100, 500)
line from (100, 500) to (10, 500)
line from (10, 500) to (10, 520)
"\s25"
line from (10, 520) to (100, 520) to (100, 500) to (10, 500) to (10, 520)
"\s0"
line from (10, 450) to (100, 450)
"new" at (55, 460) center
line from (10, 430) to (100, 430)
"move" at (55, 440) center
line from (10, 410) to (100, 410)
"resize" at (55, 420) center
line from (10, 390) to (100, 390)
"edit" at (55, 400) center
"delete" at (55, 380) center
line from (10, 470) to (100, 470)
line from (100, 470) to (100, 370)
line from (100, 370) to (10, 370)
line from (10, 370) to (10, 470)
"\s25"
line from (10, 470) to (100, 470) to (100, 450) to (10, 450) to (10, 470)
"\s0"
line from (10, 305) to (100, 305)
"graph" at (55, 315) center
line from (10, 285) to (100, 285)
"container" at (55, 295) center
line from (10, 265) to (100, 265)
"table" at (55, 275) center
line from (10, 245) to (100, 245)
"menu" at (55, 255) center
line from (10, 225) to (100, 225)
"scroll" at (55, 235) center
line from (10, 205) to (100, 205)
"picture" at (55, 215) center
line from (10, 185) to (100, 185)
"line" at (55, 195) center
line from (10, 165) to (100, 165)
"arrow" at (55, 175) center
line from (10, 145) to (100, 145)
"box" at (55, 155) center
line from (10, 125) to (100, 125)
"circle" at (55, 135) center
"template" at (55, 115) center
line from (10, 325) to (100, 325)
line from (100, 325) to (100, 105)
line from (100, 105) to (10, 105)
line from (10, 105) to (10, 325)
"\s25"
line from (10, 325) to (100, 325) to (100, 305) to (10, 305) to (10, 325)
"\s0"
line from (110, 550) to (560, 550)
line from (560, 550) to (560, 525)
line from (560, 525) to (110, 525)
line from (110, 525) to (110, 550)
line from (302, 550) to (302, 525)
line from (495, 550) to (495, 525)
line from (110, 490) to (560, 490)
line from (560, 490) to (560, 40)
line from (560, 40) to (110, 40)
line from (110, 40) to (110, 490)
line from (425, 185) to (424, 183) to (425, 185) to (425, 185)
line from (425, 185) to (425, 184) to (425, 185) to (425, 185)
line from (422, 190) to (416, 179) to (419, 191) to (422, 190)
line from (421, 191) to (422, 182) to (418, 190) to (421, 191)
line from (436, 195) to (437, 195)
line from (437, 194) to (438, 194)
line from (438, 193) to (439, 193)
line from (438, 191) to (439, 191)
line from (438, 188) to (439, 188)
line from (438, 186) to (439, 186)
line from (437, 185) to (438, 185)
line from (436, 184) to (437, 184)
line from (404, 184) to (403, 184)
line from (403, 185) to (402, 185)
line from (402, 186) to (401, 186)
line from (402, 188) to (401, 188)
line from (402, 191) to (401, 191)
line from (402, 193) to (401, 193)
line from (403, 194) to (402, 194)
line from (404, 195) to (403, 195)
line from (420, 204) to (420, 205)
line from (429, 202) to (430, 203)
line from (435, 197) to (437, 197)
line from (437, 189) to (439, 189)
line from (435, 182) to (437, 182)
line from (429, 177) to (430, 176)
line from (420, 175) to (420, 174)
line from (411, 177) to (410, 176)
line from (405, 182) to (403, 182)
line from (403, 190) to (401, 190)
line from (405, 197) to (403, 197)
line from (411, 202) to (410, 203)
line from (409, 209) to (388, 168) to (397, 213) to (409, 209)
line from (405, 212) to (408, 176) to (394, 209) to (405, 212)
line from (407, 260) to (407, 263)
line from (414, 259) to (414, 262)
line from (420, 258) to (421, 261)
line from (426, 256) to (427, 259)
line from (437, 250) to (439, 252)
line from (442, 247) to (444, 249)
line from (446, 242) to (448, 244)
line from (450, 238) to (453, 240)
line from (457, 228) to (460, 229)
line from (459, 223) to (462, 224)
line from (461, 217) to (464, 217)
line from (462, 211) to (465, 211)
line from (462, 200) to (465, 200)
line from (461, 194) to (464, 194)
line from (459, 188) to (462, 187)
line from (457, 183) to (460, 182)
line from (450, 173) to (453, 171)
line from (446, 169) to (448, 167)
line from (442, 164) to (444, 162)
line from (437, 161) to (439, 159)
line from (426, 155) to (427, 152)
line from (420, 153) to (421, 150)
line from (414, 152) to (414, 149)
line from (407, 151) to (407, 148)
line from (395, 151) to (395, 148)
line from (388, 152) to (388, 149)
line from (382, 153) to (381, 150)
line from (376, 155) to (375, 152)
line from (365, 161) to (363, 159)
line from (360, 164) to (358, 162)
line from (356, 169) to (354, 167)
line from (351, 173) to (348, 171)
line from (345, 183) to (342, 182)
line from (343, 188) to (340, 187)
line from (341, 194) to (338, 194)
line from (340, 200) to (337, 200)
line from (340, 211) to (337, 211)
line from (341, 217) to (338, 217)
line from (343, 223) to (340, 224)
line from (345, 228) to (342, 229)
line from (351, 238) to (348, 240)
line from (356, 242) to (354, 244)
line from (360, 247) to (358, 249)
line from (365, 250) to (363, 252)
line from (376, 256) to (375, 259)
line from (382, 258) to (381, 261)
line from (388, 259) to (388, 262)
line from (395, 260) to (395, 263)
line from (401, 258) to (401, 265)
line from (430, 251) to (433, 257)
line from (452, 232) to (458, 235)
line from (460, 205) to (467, 205)
line from (452, 179) to (458, 176)
line from (430, 160) to (433, 154)
line from (401, 153) to (401, 146)
line from (372, 160) to (369, 154)
line from (350, 179) to (344, 176)
line from (342, 206) to (335, 206)
line from (350, 232) to (344, 235)
line from (372, 251) to (369, 257)
line from (363, 279) to (289, 123) to (321, 293) to (363, 279)
line from (352, 292) to (361, 156) to (308, 282) to (352, 292)
line from (356, 471) to (357, 484)
line from (378, 468) to (380, 481)
line from (398, 462) to (402, 474)
line from (419, 455) to (424, 467)
line from (456, 433) to (463, 443)
line from (473, 419) to (482, 429)
line from (488, 404) to (498, 413)
line from (502, 387) to (512, 394)
line from (524, 350) to (536, 355)
line from (531, 329) to (543, 333)
line from (537, 309) to (550, 311)
line from (540, 287) to (553, 288)
line from (540, 244) to (553, 243)
line from (537, 222) to (550, 220)
line from (531, 202) to (543, 198)
line from (524, 181) to (536, 176)
line from (502, 144) to (512, 137)
line from (488, 127) to (498, 118)
line from (473, 112) to (482, 102)
line from (456, 98) to (463, 88)
line from (419, 76) to (424, 64)
line from (398, 69) to (402, 57)
line from (378, 63) to (380, 50)
line from (356, 60) to (357, 47)
line from (313, 60) to (312, 47)
line from (291, 63) to (289, 50)
line from (271, 69) to (267, 57)
line from (250, 76) to (245, 64)
line from (213, 98) to (206, 88)
line from (196, 112) to (187, 102)
line from (181, 127) to (171, 118)
line from (167, 144) to (157, 137)
line from (145, 181) to (133, 176)
line from (138, 202) to (126, 198)
line from (132, 222) to (119, 220)
line from (129, 244) to (116, 243)
line from (129, 287) to (116, 288)
line from (132, 309) to (119, 311)
line from (138, 329) to (126, 333)
line from (145, 350) to (133, 355)
line from (167, 387) to (157, 394)
line from (181, 404) to (171, 413)
line from (196, 419) to (187, 429)
line from (213, 433) to (206, 443)
line from (250, 455) to (245, 467)
line from (271, 462) to (267, 474)
line from (291, 468) to (289, 481)
line from (313, 471) to (312, 484)
line from (335, 464) to (335, 490)
line from (433, 437) to (446, 460)
line from (506, 365) to (529, 378)
line from (532, 265) to (558, 265)
line from (506, 167) to (529, 154)
line from (433, 94) to (446, 71)
line from (334, 68) to (334, 42)
line from (235, 94) to (222, 71)
line from (163, 167) to (140, 154)
line from (136, 266) to (110, 266)
line from (163, 365) to (140, 378)
line from (235, 437) to (222, 460)
.PE
.ps +2
.F2
The editor window, showing a ``recursive'' clock.
.F3
.PP
Recursion is a powerful mechanism in Magic.
A clear example is the following.
The editor itself is no different from other windows, and is described
by the "edit" record.
Overwrite the "clock" selection with "edit", and, switch of the
"scale" button.
Now the editor will be displayed by the editor, and so on.
It is in fact possible to edit the editor from the editor, but you
have to be careful not to destroy the editor while doing it.
Ignore the warnings that are printed in the window where you started
the clock.
A problem is that the Magic system doesn't know when to stop, and uses
a crude maximum recursion level to break the infinite recursion.
.PP
We will now go through the editor in a more organized way.
Select "clock" again, and select the "rec" mode.
The "clock" record is too large too fit in the table.
To the left of the table, a scroll widget makes it possible to scan
through the table.
This way, find the "value" attribute, which contains the number of
seconds since the epoch.
The clock station updates this value every second.
The "hour", "minute", "second", and "ctime" attributes are functions
of this value.
For example, the "ctime" attribute is { clock value # $time }, and is
used in the table widget in the main clock window.
.PP
The small table in the editor window above the record being displayed
can be used to display or add individual attributes.
Just type the name of the attribute in the left part, and, if you wish
to update that attribute, its value in the right part.
You can browse through the whole data base using the "datadict"
record, which is a list of all record names in the data base.
.PP
Now go to the "wid" mode, and overwrite the selection with "window".
This will describe the "window" widget, which is a picture consisting
of four other widgets.
The basic attributes of the records describing these widgets will be
displayed.
Now go to the picture mode, and select the record "mypic".
Make sure the "scale" button is switched on, and the "new" mode is
selected.
Now, using the "line", "arrow", "box", and "circle" types, create
your own picture.
.PP
We are going to add a new widget to the main window.
Select "widget" and switch the scale button off.
Select the "picture" type.
Now pull up a square box above the date bar, by clicking with the mouse
where you want to top left corner and dragging it with the mouse button
depressed to the bottom right corner.
Select "edit" and click in the box.
Change the record field to mypic, and see how your picture is being
displayed in the main window.
.PP
Now leave the editor by clicking in the box in the top left corner of
the editor window.
Click the mouse in the history bar on the left side of the clock
window, and pull the mouse up and down with the button depressed.
You can see exactly how the main window has changed since you started
the clock window.
You can put the clock back into normal operation by clicking in the
small box in the bottom of the history bar.
.PP
Now we are going to terminate the clock.
Pop up the menu again by clicking in the box in the top left corner of
the main clock window, and select "quit".
A new window pops up, asking whether you really want to quit or not.
Click in the "yes" box, and the clock station will checkpoint its
data base and exit.
Now start it up again, using "sun4/bin/clock&", and you'll see that
everything, including the position of the window, has been
remembered.
.NH
The Menu
.PP
The menu that appears when you click in the container widget in the
top left corner of the main clock window is just another window, like
the main clock windown itself and the editor window.
The menu window has two overlapping widgets:  a table widget and a
menu widget.
The table widget is not selectable, and contains the text.
It shows the "descr" attributes in the "control" list of windows.
Similarly, the menu widget is tied to the "control" record.
There is a watch on the "select" attribute of the "control" record.
When a menu entry is selected, this attribute is updated, and a
message is sent to a function that will consequently map or unmap
the selected window.
.PP
Thus when you select an entry in the menu, the corresponding window
either gets mapped or unmapped, depending on what its current status
is.
The menu usually contains at least the following entries:
.DS C
window
template
edit
dump
quit
watch
calc
cancel
.DE
The first entry in the menu is the main window.
If you select that, the main window will disappear.
There is another entry called "cancel", which is actually tied to the
menu itself.
You can select that if you accidentally popped up the menu, or changed
your mind.
Another way of doing this is to click in the container widget again in
the main window.
.PP
The "template" window lets you stamp out templates.
You have to specify the template record and the variables record, and
hit the done button.
The "edit" entry pops up the editor.
The "dump" window is empty and will either not appear or disappear
quickly.
As a side-effect, the data base gets checkpointed.
The "quit" window enables you to terminate a station.
.PP
The "watch" window lets you set a watch on an attribute in the data
base.
The "calc" window contains two vertical bars.
In the top bar you can enter postfix expressions, and the result will
be displayed in the lower bar.
Try it.
.NH
ISIS
.PP
ISIS applications the following three initialization routines in
the following order:
.DS
isis_init(int port);
mag_init(char *name);
sel_init();
.DE
Instead of
.I win_main_loop
or
.I isis_mainloop() ,
applications invoke
.I sel_loop()
as their main loop.
.PP
The utility
.DS
sun4/bin/isisstat
.DE
uses the
.I isisstat
data base, and collects ISIS statistics every 10 seconds.
To time this it uses a watch on the clock station.
It retrieves the information using the ISIS port defined by the
environment variable ISISPORT, or, if absent, the default port
1603.
.PP
The information is stored in the "isis" record.
The isisstat database defines a new record "diff" which contains the
differentials of the attributes in "isis".
Another new record called "flag" contains two attributes "cfault" and
"vchange" which are set to "1" whenever the corresponding attributes
in "isis" change.
.PP
The distribution contains a default "isisstat" data base that contains
a number of widgets displaying some of the values in the "diff"
record.
However, this is very immature, and not to much can be expected of it.
You can play with the data base to make it more useful.
.NH
Meta
.PP
The Meta interface within the Magic system consists of the following
routines:
.DS
.ta \w'meta_setwatch(struct val'u
meta_init(char *name);
meta_cold();
meta_setwatch(struct value *sensor, struct value *instance,
	struct value *relation, struct value *value);
meta_actuate(struct value *actuator, struct value *instance,
	struct value *value);
.DE
.I Meta_init()
automatically initializes the ISIS, Meta, and Magic subsystems, thus
other initialization routines need be called.
The ISIS port is retrieved from the environment variable
.I ISISPORT ,
If absent, the system uses the default port 1603.
.I Name
is the station's name.
The Meta application should invoke
.I sel_loop()
as its main loop.
.I Meta_cold()
initializes certain Meta related data structures in the data base,
and is automatically invoked by the
.I create
utility.
.PP
The Magic system can start Meta sensors and invoke Meta actuators.
The routine
.I meta_setwatch()
starts a Meta watch on the specified
.I sensor
and
.I instance .
The
.I relation
may be null, in which case any change in the sensor will be detected.
The Magic system will put any value received from the Meta actuator
in the record called
.I instance
under the attribute called
.I sensor .
If the sensor is of a set type, the attribute will point to a list
record containing the values in the order received.
The Magic record called "instances" will be updated to contain a
list of all Meta instances that are believed to be up and running.
The routine
.I meta_actuate()
will poke a Meta actuator with the specified value.
.NH 2
Standard Messages
.PP
There are three standard request messages:
.DS
meta_watch
meta_update
meta_actuator
.DE
.PP
.I Meta_watch
takes four arguments, and invokes
.I meta_setwatch() .
.PP
The standard message "meta_update" can be used to invoke
.I meta_actuate() .
.I Meta_update
gets three parameters":
.I record ,
.I attribute
and
.I value.
It inspects the Magic record called
.I record/attribute
and retrieves the attributes "actuator" and "instance".
Meta_update uses these and the
.I value
parameter in the message to invoke
.I meta_actuate .
.PP
This can be used to tie a Meta actuator to a Magic attribute as
follows.
Put a watch on the record and attribute to invoke "meta_actuate".
Furthermore, create a record
.I record/attribute
containing the type and instance of the actuator in the attributes
"actuator" and "instance".
Now each time the attribute is updated, the Meta actuator is invoked
automatically with the new value.
.PP
.I Meta_actuator
takes four arguments:  a record, an attribute, an actuator, and an
instance.
It sets up a record called
.I record/attribute
with the actuator and instance,
and puts a
.I meta_update
watch on the attribute
.NH 2
Visual Meta invocation
.PP
If
.I meta_cold()
has been invoked, the data base will contain two windows which will
let you invoke Meta watches and actuators.
If you pop up the main menu, there will be two entries:
.DS
meta sensor
meta actuator
.DE
The Meta sensor window lets you specify a sensor, an instance, a
relation, and a value.
It invokes
.I meta_setwatch()
in response.
If you don't specify an instance, a Meta watch will be set on all
instances.
If you don't specify a relation, any change to the sensor will be
reported.
In addition to setting the watch, the window remembers the watch in
the data base, so that each time you start the station again, the
watch will be set automatically.
.PP
The Meta actuator window lets you specify an actuator, an instance, a
record, and an attribute.
The value of the record and attribute are automatically propagated to
the actuator.
This is currently not saved in the data base, so that it has to be
done each time you start the station again.
.NH 2
Example
.PP
This example assumes that you have Meta running.
This includes the ISIS system, a table manager, and at least one
machine sensor.
Create a new data base called "meta", and start the
.I display
utility as follows:
.DS
sun4/bin/create meta
sun4/bin/display meta
.DE
.I Display
is a generic station that will display any data base.
A small window pops up with a history bar and a container widget to
pop up a menu.
Pop up the menu, and select "meta sensor".
.PP
Fill out the window that pops up.
Specify only the sensor, and make it "machine:load".
Then press the "go" button.
If anything goes wrong, an error message will appear.
We assume here that this works, and press the "done" button so that
the window disappears again.
.PP
Now go into the editor and look at the instances record.
This will display the set of instances currently running a
machine:load sensor.
When you look at a record named by one of these instances, you will
find an attribute called "machine:load" displaying the load on that
system, which is regularly updated.
To find the average load on all the machines, pop up the "calc" window,
click in the top vertical bar, and type in the following postfix
expressions (concatenated):
.DS
instances "$entry machine:load #" $list
instances N # N :=
%N 1 \- "+" $iter %N /
.DE
The first line retrieves all the loads, the second the number of
loads, and the last adds them up and divides to obtain the average.
.NH
The Cookie Factory
.PP
The cookie factory is a simple simulation.
It consists of five stations:  a mixer, an oven, two conveyor belts,
and a central manager station.
Furthermore, the cookie factory uses the clock station.
You can start it up using the command "bin/rundemo".
Two windows will pop up:  the clock window, and the manager's
window.
(If there was a clock running already, it will notice and panic.  This
is a feature of the Magic system:  it will let only one station use
the same name.)
Now put the windows where you want them.
.PP
The manager's window contains a matrix of two by four (square) switches.
The columns correspond to the machines:
.DS
.ta \w'm2   'u
m1	mixer
b1	conveyor belt
m2	oven
b2	conveyor belt
.DE
The top row indicates whether the machine is turned of or on.
Only the conveyor belts should be turned on.
(This may not be immediately, but should be true after a while.)
.PP
There is a station associated with each machine.
The bottom row in the switch matrix indicates whether the main window
of the machines are mapped or not.
Click in all the bottom four switches, and four windows should pop
up.
Place them next to each other using the window manager, in the order
"m1" (leftmost), "b1", "m2", and "b2" (rightmost).
Each window has a square indicator in the top right corner indicating
whether the machine is turned on or off.
Again, only the conveyor belts should be on.
.PP
The "m1" window has three containers, corresponding to a flour,
butter, and sugar container.
The machine uses these ingredients in the ratio 1 : 1 : 2.
You can fill the containers using the mouse.
Click with the mouse somewhere in the top of each of the three
containers.
.PP
The manager has sensors on each of the containers, and notices this.
In response it switches on the mixer.
After a while the first batch of batter will be places on the first
conveyor belt on its way to the oven, and it will be placed in the
batter container there.
The manager also has a sensor on the batter container, and as soon as
it thinks there is enough batter it will switch on the oven.
Finally, the cookies are places on the second conveyor belt where they
will be for ever.
.PP
The manager switches machines on and off depending on the level in
each of the containers.
The machines do not decide this by themselves.
You will have to refill the flour, butter and sugar containers
regularly to keep the factory in operation.
You can terminate the factory by terminating the manager.
Click in the top left button of the manager window, and select the
quit window.
The clock stays alive, but will complain about the machines that have
disappeared.
.NH
Advanced Magic
.PP
Using the template facility in picture editing.
.bp
.NH
Appendix A
.sp 2
.LP
This appendix lists the operators along with the number of arguments.
In their description the arguments on top of the stack are denoted by
$1, $2, ... .
.de OP
.ta 0.3i 1.5i 2i
.br
	\\$2	\\$1	\\$3
..
.NH 2
Mathematical Operators
.LP
.OP 1 $sign "sign of $1 (\-1, 0, or 1)"
.OP 1 $abs "absolute value of $1"
.OP 1 $neg "\- $1"
.OP 2 $min "minimum of $1 and $2"
.OP 2 $max "maximum of $1 and $2"
.OP 2 + "$1 + $2"
.OP 2 \-  "$1 \- $2"
.OP 2 * "$1 * $2"
.OP 2 / "$1 / $2"
.OP 2 $div "integer divide: $1 / $2"
.OP 2 "$mod or %" "integer modulo: $1 % $2"
.OP 1 $trunc "remove mantissa"
.OP 1 $floor "round down"
.OP 1 $ceil "round up"
.OP 1 $round "round to nearest integer value"
.OP 1 "$rint or $aint" ditto
.OP 1 $anint "ditto, but halfway cases rounded up"
.OP 2 $pow "$1 ^ $2"
.OP 1 $exp "exp($1)"
.OP 1 $log "log($1)"
.OP 1 $sin "sin($1)"
.OP 1 $cos "cos($1)"
.OP 1 $tan "tan($1)"
.OP 1 $asin "asin($1)"
.OP 1 $acos "acos($1)"
.OP 1 $atan "atan($1)"
.OP 2 $atan2 "atan($2 / $1)"
.OP 1 $sinh "sinh($1)"
.OP 1 $cosh "cosh($1)"
.OP 1 $tanh "tanh($1)"
.OP 1 $asinh "asinh($1)"
.OP 1 $acosh "acosh($1)"
.OP 1 $atanh "atanh($1)"
.OP 1 $j0 "Bessel j0($1)"
.OP 1 $j1 "Bessel j1($1)"
.OP 1 $y0 "Bessel y0($1)"
.OP 1 $y1 "Bessel y1($1)"
.NH 2
Boolean Operators 
.LP
.OP 2 < "$1 < $2"
.OP 2 <= "$1 <= $2"
.OP 2 "= or ==" "$1 == $2"
.OP 2 "<> or !=" "$1 != $2"
.OP 2 > "$1 > $2"
.OP 2 >= "$1 >= $2"
.OP 2 "$and or &&" "$1 and $2 are both true"
.OP 2 "$or or ||" "at least one of $1 and $2 are true"
.OP 1 "$not or !" "$1 is false"
.OP 3 ? "if $1 then $2 else $3"
.NH 2
String Operators
.LP
.OP 1 $quote: "put necessary quotes and escapes in $1 to quote it"
.OP 2 $cmp: "string compare of $1 and $2 (returns \-1, 0, or 1)"
.OP 1 $length: "string length of $1"
.OP 2 $concat: "concatenate $1 and $2"
.OP 3 $concat3: "concatenate $1, $3, and $2"
.NH 2
Special Operators
.LP
.OP 1 $time "convert time $1 to printable string (a la ctime)"
.OP 1 $clk_hour "convert time $1 to which hour it is"
.OP 2 # "retrieve attribute $2 in record $1"
.OP 3 $get "retrieve value and seq # of $2 in $1 at seq $3"
.OP 0 $db_ddcount "return number of records in database"
.OP 1 $db_datadict "return $1-th record"
.OP 1 $db_count "return number of attributes in record $1"
.OP 2 $db_attr "record = $1.  Return $2-th attr"
.NH 2
General Operators
.LP
.OP 0 $nop "do nothing"
.OP 1 $pop "discard $1"
.OP 2 := "assign $1 to $2"
.OP 1 $dup "duplicate $1"
.OP 2 $swap "exchange $1 and $2"
.OP 1 $eval "evaluate $1"
.OP 2 $iter "evaluate $2 $1 times"
.OP 2 $list "evaluate $2 on all entries in record $1"
.OP 3 $select "return $2th attribute in list $1 for which $3 is non-zero"
.OP 2 $selcnt "return # attributes in list $1 for which $2 is non-zero"
.OP 3 $sort "return $2th attribute in list $1 if compare function is $3"
.NH 2
Constants and Variables
.LP
.OP 0 $null "push a null value"
.OP 0 $e "exp(1)"
.OP 1 $pi "3.14159..."
.OP 0 $db_seq "current sequence number"
.OP 0 $record "the record that contains the function"
.OP 0 $attr "the attribute that contains the function"
.OP 0 $entry "the entry name in a $list and $select operations"
.OP 0 %1,%2 "the entries that are being compared in $sort"
.NH 2
Standard Functions
.LP
These functions have to be retrieved from the record "std.func" first.
.LP
.OP 2 diff "differentiate <$1, $2> over time"
.OP 3 int "integrate <$1, $2> over the last $3 seconds"
.OP 3 min "take the minimum of <$1, $2> over the last $3 seconds"
.OP 3 max "take the maximum of <$1, $2> over the last $3 seconds"
.OP 3 get "get the value of <$1, $2> $3 seconds ago"
