/*

Copyright 1990 by M. Beck and K. Marzullo
Rights to use this source in unmodified form granted for all
commercial and research uses.  Rights to develop derivative
versions reserved by the authors.

*/

#include <stdio.h>
#include <search.h>
#include "isis.h"
#include "types.h"
#include "table.h"
#include "table_mgr.h" 

#define VERSION_MOD 0x1000000

int T_create(rel_des, stable, repl_factor,
		num_attributes, attributes, delete, randomize)
RelDes *rel_des;		/* Relation pointer			*/
int stable;			/* type of relation: TRUE  if stable	*/
				/*		     FALSE if not	*/
int	repl_factor;		/* replication factor 			*/
int	num_attributes;		/* number of attributes 		*/
AttDes	*attributes;		/* attributes information 		*/
int	delete;			/* delete attribute if it already exists */
int	randomize;		/* version number generator		*/
{
  message *mp;
  RELNAME relation;

  if (rel_des->exist) {

	if (delete) {
	  	strncpy(relation, rel_des->name, sizeof(RELNAME)); 
		T_destroy(rel_des);
		rel_des  = rel_lookup(relation, 1);
	} else
		return TBL_TABLE_EXIST;
  }

  rel_des->stable = stable;
  rel_des->repl_factor = repl_factor;
  rel_des->num_attributes = num_attributes;
  rel_des->num_tuples = 0;
  rel_des->version = randomize%VERSION_MOD;
  rel_des->exist =  1;
  rel_des->attributes = attributes;
  rel_des->index = (Index *)calloc(num_attributes+2, sizeof(Index));

  /* NOTE: insertion into sysrel MUST come after creation of relation 	*/
  /* in order for creation of sysrel to work!			 	*/
  
  if (sys_insert(rel_des->name) != TBL_SUCCESS) return TBL_FAILURE;
  return TBL_SUCCESS;
}

int T_destroy(rel_des)
RelDes *rel_des;		/* Relation pointer */
{
  int v, i;
  message *mp, **mpp;
 
  if (!rel_des) return TBL_TABLE_NOEXIST;
  
  for (mpp = INDEX(rel_des, TBL_SORTALL);
		mpp; mpp = INDEX(rel_des, TBL_SORTALL))
  {
    	for (i = TBL_SORTALL; i < rel_des->num_attributes; i++)
    		if ((i == TBL_SORTALL) || INDEX(rel_des, i))
	  	{
  			T_cmp_init(rel_des, i);
		  	tdelete(*mpp, &INDEX(rel_des, i), T_cmp);
		}
	msg_delete(*mpp);
  }

  free(rel_des->attributes);
  tdelete(rel_des, &root_rel, rel_cmp);
  free(rel_des);

  mp = msg_newmsg();
  type_pack(TYPE_STRING,mp, 0, rel_des->name);
  v = T_remove(sys_rel, mp);

  return (v == TBL_SUCCESS ? TBL_SUCCESS : TBL_FAILURE);
}

static int hardness;

int T_reset(hard)
int hard;
{
  void rel_reset();

  hardness = hard;
  twalk(root_rel, rel_reset);
  return TBL_SUCCESS;
}

void rel_reset(rel_des)
RelDes **rel_des;
{
  if ((hardness || !(*rel_des)->stable) && *rel_des != sys_rel)
	T_destroy(*rel_des);
}

int T_info(rel_des, info, attributes)
RelDes *rel_des;		/* Relation pointer */
TableInfo *info;
AttDes **attributes;
{
  info->stable = rel_des->stable;
  info->repl_factor = rel_des->repl_factor;
  info->num_attr = rel_des->num_attributes;
  info->num_tuples = rel_des->num_tuples;
  info->version = rel_des->version;
  *attributes = rel_des->attributes;

  return TBL_SUCCESS;
}

int T_rename_rel(rel_des, new_relation)
RelDes *rel_des;		/* Relation pointer 		*/
RELNAME new_relation;		/* New relation name		*/
{
  if (sys_remove(rel_des->name) != TBL_SUCCESS) return TBL_FAILURE;
  if (sys_insert(new_relation)  != TBL_SUCCESS) return TBL_FAILURE;
  strncpy(rel_des->name, new_relation, sizeof(RELNAME)); 
  return TBL_SUCCESS;
}

int T_rename_attr(rel_des, attr_col, new_attr)
RelDes *rel_des;			/* Relation pointer */
int	attr_col;			/* Attribute column		*/
RELNAME	new_attr;			/* New attribute name		*/
{
  if (rel_des->num_attributes <= attr_col  || attr_col < TBL_SORTALL)
	return TBL_COL_NOEXIST;

  strncpy(rel_des->attributes[attr_col].AttName, new_attr, sizeof(RELNAME));
  return TBL_SUCCESS;
}

int T_remove_dups(rel_des)
RelDes	*rel_des;			/* Relation pointer 		*/
{
  int val, i, t;
  AttDes *attrs;
  TableInfo info;
  message **arr, **old_index;

  rel_des->version += 1;
  rel_des->version %= VERSION_MOD;

  val = T_list(rel_des, TBL_SORTALL, NULL, &info, &attrs, &arr);
  if (val != TBL_SUCCESS) return val;
  if (info.num_tuples == 0) return TBL_SUCCESS;

  for (i=TBL_SORTALL; i < rel_des->num_attributes; i++)
  {
	T_cmp_init(rel_des, i);
    	while(INDEX(rel_des,i))
		tdelete(*INDEX(rel_des, i), &INDEX(rel_des, i), T_cmp);
  }

  T_cmp_init(rel_des, TBL_SORTALL);
  for(i = 0; i < info.num_tuples; i++)
	if (i == 0 || tuple_cmp(rel_des, arr[i], arr[i-1], TBL_SORTALL))
		tsearch(arr[i], &INDEXALL(rel_des), T_cmp);
  	else
	{
		msg_delete(arr[i]);
		rel_des->num_tuples -= 1;
	}
  free(arr);

  return TBL_SUCCESS;
}

int T_insert(rel_des, tuple, suppress_dups)
RelDes *rel_des;		/* Relation pointer */
message *tuple;			/* tuple to insert/delete		*/
int suppress_dups;		/* TRUE means don't insert duplicated	*/
				/* FALSE means do 			*/
{
  int i;
  struct node *node, *ep;

  rel_des->version = (rel_des->version + 1) % VERSION_MOD;

  T_cmp_init(rel_des, TBL_SORTALL); 
  if (suppress_dups && tfind(tuple, &INDEXALL(rel_des), T_cmp))
	      	return TBL_TUPLE_DUPLICATE;
  for (i=TBL_SORTALL; i < rel_des->num_attributes; i++)
    	if ((i == TBL_SORTALL) || INDEX(rel_des, i))
	{
		T_cmp_init(rel_des, i);
		tsearch(tuple, &INDEX(rel_des, i), T_dcmp);
	}
 
  rel_des->num_tuples++;
  return TBL_SUCCESS;
}


int T_remove(rel_des, tuple)
RelDes *rel_des;		/* Relation pointer */
message *tuple;
{
  int i;
  message **p;

  if (!rel_des)  return TBL_FAILURE;
  rel_des->version = (rel_des->version + 1) % VERSION_MOD;

  T_cmp_init(rel_des, TBL_SORTALL);
  p = (message **)tfind(tuple, &INDEX(rel_des, TBL_SORTALL), T_cmp);
  if (!p) return TBL_TUPLE_NOEXIST;

  for (i = TBL_SORTALL; i < rel_des->num_attributes; i++)
    	if ((i == TBL_SORTALL) || INDEX(rel_des, i))
  	{
  		T_cmp_init(rel_des, i);
	  	tdelete(tuple, &INDEX(rel_des, i), T_cmp);
	}

  msg_delete(*p);
  rel_des->num_tuples--;
  return TBL_SUCCESS;

}

static int T_dump_count;
static RelDes *T_dump_rel;
static message *T_dump_filter, **T_dump_array;

T_dump_init(rel, filt, arr)
RelDes *rel;
message *filt, **arr;
{
	T_dump_count = 0;
	T_dump_rel = rel;
	T_dump_filter = filt;
	T_dump_array = arr;
}

T_dump(tuple, order)
message **tuple;
VISIT order;
{
  if (POST(order) &&
      (!T_dump_filter || tuple_test(T_dump_rel, T_dump_filter, *tuple)))
  {
	T_dump_array[T_dump_count] = *tuple;
	T_dump_count++;
  }
}

static Index *T_build_index;

T_build_init(index)
Index *index;
{
	T_build_index = index;
}

T_build(tuple, order)
message **tuple;
VISIT order;
{
  if (POST(order))
	tsearch(*tuple, T_build_index, T_dcmp);
}

int T_list(rel_des, column_num, filter, info, attrs, arr)
RelDes *rel_des;		/* Relation pointer */
int column_num;			/* column (attribute) number to sort by	  */
				/* TBL_NOSORT means order is unimportant  */
				/* TBL_SORTALL means sort lexigraphically */
				/* on all columns			  */

message *filter;		/* filter on tuples returned		  */
				/* list of triples (conjunction):	  */
				/*	column number to compare,	  */
				/*	value to compare,		  */
				/*	sense of comparison		  */

TableInfo *info;		/* output: current table info 		  */
AttDes **attrs;			/* output: table schema			  */

message ***arr;			/* output: pointer to tuple array	  */
				/* 	   NULL if no array returned	  */
{
  int i;

  if (rel_des->num_attributes <= column_num || column_num < TBL_SORTALL)
	return TBL_COL_NOEXIST;

  if (rel_des->num_tuples > 0)
  {
  	if (column_num == TBL_NOSORT) column_num = TBL_SORTALL;
  	if (!INDEX(rel_des, column_num))
  	{
  		T_build_init(&INDEX(rel_des,column_num));
		T_cmp_init(rel_des, column_num);
  		twalk(INDEXALL(rel_des), T_build);
  	}

  	*arr = (message **)malloc(sizeof(message *)*(rel_des->num_tuples));
  	if (!*arr) return TBL_FAILURE;

  	T_dump_init(rel_des, filter, *arr);
  	twalk(INDEX(rel_des, column_num), T_dump);
  } else
	T_dump_count = 0;

  T_info(rel_des, info, attrs);  
  info->num_tuples = T_dump_count;

  return TBL_SUCCESS;
}



