/*

Copyright 1990 by M. Wood and K. Marzullo
Rights to use this source in unmodified form granted for all
commercial and research uses.  Rights to develop derivative
versions reserved by the authors.

*/

/*  Machine sensor statistics.

    By Mark D. Wood
*/



#include <stdio.h>
#include <strings.h>
#include <fcntl.h>
#include <utmp.h>
#include <sys/param.h>          /* def of FSCALE */
#include <sys/time.h>
#include <sys/wait.h>

#ifdef SUNOS4
#include <kvm.h>
#endif

#include <nlist.h>
#include <stdio.h>
#include <fcntl.h>

#include "sm.h"
#include "sm_err.h"


#define AVENRUN 0
#define BOOTTIME 1
#define POLL_FREQ 11

/* structure used for Meta rexec facility */

struct jobtype {
char * name;
struct jobtype *prev,*next;
};

typedef struct jobtype JOB;


/* globals used by sensors */

JOB *job_queue = NULL;
int num_jobs = 0;

int sn_jobs, sn_numjobs;	/* sensor handles */

int fd_utmp;                    /* descriptor for /etc/utmp - login */
int kernel_access = 1;


#ifdef SUNOS4
static kvm_t *kd;
#else
static int fd_kmem;			/* descriptor for /dev/kmem */
#endif

#define FIXLEN 8



static struct nlist sys_name_list[] = { /* to get name list */
    {"_avenrun"},
    {"_boottime"},
    {""},
};



init_kernel()

{
#ifdef SUNOS4
    if ((kd = kvm_open(NULL,NULL,NULL,O_RDONLY,"kvm_open")) == NULL) {
	kernel_access = 0;
    } else if (kvm_nlist(kd,sys_name_list)) {
	fprintf(stderr,"kvm_nlist failed");
	kernel_access = 0;
    }
#else
    if (nlist("/vmunix",sys_name_list)) {
	fprintf(stderr,"nlist failed");
	kernel_access = 0;
    } else if ((fd_kmem = open("/dev/kmem",O_RDONLY)) < 0) {
	kernel_access = 0;
    }
#endif
    if (!kernel_access)
      fprintf(stderr,
	      "No kernel access, using system calls instead.\n");
}      



int kernel_value(name,buffer,amt)
int name;
char * buffer;
int amt;

{

#ifdef SUNOS4
    printf("before kvm_read\n");
    if (kvm_read(kd,(unsigned long) sys_name_list[name].n_value,
		 buffer,amt) < 0) 
      return -1;
    else
      return 0;

#else

    lseek(fd_kmem,(unsigned long) sys_name_list[name].n_value,0);
    read(fd_kmem,buffer,amt);

#endif

}



char *fixtostr(fix)
char *fix;

{
    static char temp[FIXLEN + 1];

    memcpy(temp,fix,FIXLEN);
    temp[FIXLEN] = (char) 0;
    return temp;
}



/* stuff for maintaining jobs that we do */
/* The event lists are noncircular doubly-linked lists; each end is
   terminated with a null pointer */


job_insert(queue,element)
JOB **queue, *element;

{
    element -> prev = NULL;
    element -> next = (*queue);
    if ((*queue))
      (*queue) -> prev = element;
    (*queue) = element;
}



job_delete(queue,element)
JOB **queue,*element;

{
    if (element->next)
      element->next->prev = element->prev;
    if (element->prev)
      element->prev->next = element->next;
    if (*queue == element)
      *queue = element->next;
}



/* 
  The sensors themselves.  Sensors take as a parameter the address of
  a pointer which is set to point to the value; the value itself is
  stored in a statically allocated piece of memory.  The reason for
  this is to conceal from the caller the type of the result; the caller
  need not allocate space for it.  All functions return as a result the
  status of the operation, the error code (supposed to be a negative
  number).
*/



int login(value)
message ***value;

{
    struct utmp utmp_buf;
    static message * m;
    
    m = msg_newmsg();
    *value = &m;

    if (lseek(fd_utmp,0L,0)) return SM_NOUTMP;

    while (read(fd_utmp,&utmp_buf,sizeof(utmp_buf))
           == sizeof(utmp_buf)) {
        if (utmp_buf.ut_name[0]) {
	    set_pack(TYPE_STRING,&m,fixtostr(utmp_buf.ut_name));
        }
    }
    return 0;
}



int load(value)
double ** value;

{
    static double r_load;


    if (kernel_access) {
#ifdef sun
	long i_load;
	kernel_value(AVENRUN,(char *) &i_load,sizeof(i_load));
	r_load = (double) i_load / FSCALE;
#else
	kernel_value(AVENRUN,(char *) &r_load,sizeof(r_load));
#endif
    } else {
        FILE *popen(), *fp;
        char buf[128], *q;

        fp = popen("uptime", "r");
        fgets(buf, sizeof(buf), fp);
        pclose(fp);
	q = rindex(buf,':');
	if (!q) 
	  return -1;
	q++;
        sscanf(q, "%lf", &r_load);
    }

    *value = &r_load;
    return 0;
}



int mytime(value)
long **value;

{
    static long thetime;

    time(&thetime);
    *value = &thetime;
    return 0;
}




int boottime(value)
long **value;

{
    static struct timeval t;

    if (kernel_access) {
	kernel_value(BOOTTIME,(char *) &t,sizeof(t));
	*value = &t.tv_sec;
	return 0;
    } else {
      return -1;
  }
}




int numjobs(value)
int **value;

{
    *value = &num_jobs;
    return 0;
}



int jobs(value)
message ***value;

{
    JOB * jp;
    static message * m;
    
    m = msg_newmsg();
    *value = &m;

    for (jp = job_queue; jp; jp = jp->next) {
	set_pack(TYPE_STRING,&m,jp->name);
    }
    return 0;
}


condition waitforchild = 0;


void childfinished(dummy)
int dummy;

{
    t_sig_all(&waitforchild,0);
}

int exec(cmd)
char *cmd;

{
    int pid;
    int status;
    JOB job;

    job.name = cmd;
    job_insert(&job_queue,&job);
    num_jobs++;
    sn_chkcnd(sn_jobs);
    sn_chkcnd(sn_numjobs);

    pid = fork();
    if (!pid) {
	int sys_stat;

	isis_disconnect();
#ifdef DEBUG
	printf("running %s\n",cmd);
#endif
	sys_stat = system(cmd);
#ifdef DEBUG
	printf("system status = %d\n",sys_stat);
#endif
	exit(sys_stat);
    } else {
	do {
	    t_wait(&waitforchild);
	} while (!(wait4(pid,&status,WNOHANG,NULL)));
	job_delete(&job_queue,&job);
#ifdef DEBUG
	printf("finished %s (%d)\n",cmd,status);
#endif
	num_jobs--;
	sn_chkcnd(sn_jobs);
	sn_chkcnd(sn_numjobs);
	if (status & 0x177)
	  return SM_FAILED;
	else
	  return (status >> 8) & 0x177;
    }
}



main(argc,argv)
int argc;
char *argv[];

{
    char *portnum;
    static int isisport = 0;
    static int pollfreq = POLL_FREQ;
    extern char * getenv();

    if (argc == 2) {
	pollfreq = atoi(argv[1]);
	if (pollfreq <=0 ) {
	    fprintf(stderr,"Poll frequency must be nonzero!\n");
	    exit(-1);
	}
    }
    portnum = getenv(ISISPORT);
    if (portnum) isisport = atoi(portnum);

    init_kernel();
    fd_utmp = open("/etc/utmp",O_RDONLY);

    sn_newsensor(login,"login",TYPE_SET|TYPE_STRING,pollfreq);
    sn_newsensor(load,"load",TYPE_REAL,pollfreq);
    sn_newsensor(mytime,"localtime",TYPE_INT,pollfreq);
    sn_newsensor(boottime,"boottime",TYPE_INT,pollfreq);

    act_newactuator(exec,"exec",TYPE_STRING,FALSE);
    isis_chwait(childfinished,0);
    sn_jobs = sn_newsensor(jobs,"jobs",TYPE_SET|TYPE_STRING,0);
    sn_numjobs = sn_newsensor(numjobs,"numjobs",TYPE_INT,0);

    sn_init("machine",isisport);
    act_init("machine",isisport);
  
    isis_mainloop(0);
}
