/*      Originally coded by Robbert van Renesse
 *
 *
 *      ISIS release V2.0, May 1990
 *      Export restrictions apply
 *
 *      The contents of this file are subject to a joint, non-exclusive
 *      copyright by members of the ISIS Project.  Permission is granted for
 *      use of this material in unmodified form in commercial or research
 *      settings.  Creation of derivative forms of this software may be
 *      subject to restriction; obtain written permission from the ISIS Project
 *      in the event of questions or for special situations.
 *      -- Copyright (c) 1990, The ISIS PROJECT
 */

/*
 * This module evaluates postfix expressions.  The input is represented in
 * a null-terminated ASCII string.  Tokens in the string are separated by
 * spaces.  Tokens including spaces can be quoted using "".  Characters in
 * quotes can be escaped by \.  Operators are non-quoted strings, which
 * replace themselves plus zero or more tokens by zero or more new tokens.
 * Tokens that are not operators can be seen as operators that replace
 * themselves with themselves.
 *
 * Expression are evaluated in a certain context.  Contexts include the top
 * of the expression stack, and a set of temporary variables.
 */

#include "magic.h"
#include "value.h"
#include "expr.h"

#define MAX_LEVEL	 16	/* Maximum nesting level */
#define STACK_SIZE	128	/* Size of expression stack */
#define OPER_HASH	 47	/* Size of operator hash table */
#define PROG_HASH	 64	/* Size of postfix programs hash table */

/* Test whether c falls between a and z inclusive.
 */
#define between(a, c, z)	((unsigned) ((c) - (a)) < (z) - (a))

struct value *db_get(), *mag_index();

/* This structure contains the names and function pointers of all operators.
 * It is hashed for efficiency.
 */
struct operator {
	struct operator *op_next;
	char *op_name;			/* name of operator */
	int (*op_func)();		/* function pointer */
} *x_opers[OPER_HASH];

/* Types of token:
 */
#define T_VALUE		0
#define T_VARIABLE	1
#define T_OPERATOR	2

/* A token in a postfix expression is either a value, a variable, or an
 * operator.  This structure is a union of those types.
 */
struct token {
	int t_type;			/* type of token */
	union {
		struct value *tu_value;
		char *tu_variable;
		int (*tu_oper)();
	} t_u;
};

#define t_value		t_u.tu_value
#define t_variable	t_u.tu_variable
#define t_oper		t_u.tu_oper

/* x_progs is a cache of postfix programs, implemented by a hash table.
 * The postfix programs are represented by strings (p_prog), and parsed
 * and compiled into a list of tokens (p_tokens).
 */
struct prog {
	struct prog *p_next;
	char *p_prog;			/* postfix program string */
	struct token *p_tokens;		/* token list */
	struct token *p_last;		/* last token */
} *x_progs[PROG_HASH];

struct value *x_stack[STACK_SIZE];	/* the expression stack */
struct value **x_sp;			/* the expr stack pointer */

/* This routine is called to reduce the amount of memory used.  Clear
 * the program cache.
 */
x_reduce(){
	struct prog **pp, *p;
	struct token *t;

	for (pp = x_progs; pp < &x_progs[PROG_HASH]; pp++)
		while ((p = *pp) != 0) {
			*pp = p->p_next;
			MAG_FREE(p->p_prog);
			for (t = p->p_tokens; t < p->p_last; t++)
				switch (t->t_type) {
				case T_VALUE:
					val_free(t->t_value);
					break;
				case T_VARIABLE:
					MAG_FREE(t->t_variable);
				}
			MAG_FREE(p);
		}
}

/* Within context c, bind/assign val to name.
 */
x_bind(c, name, val)
struct context *c;
char *name;
struct value *val;
{
	struct variable *v;

	for (v = c->c_vars; v != 0; v = v->v_next)
		if (scmp(v->v_name, name)) {
			val_free(v->v_value);
			v->v_value = val_ref(val);
			return;
		}
	v = MAG_MALLOC(1, struct variable);
	v->v_name = mag_copy(name);
	v->v_value = val_ref(val);
	v->v_next = c->c_vars;
	c->c_vars = v;
}

/* Cleanup the variables in a context.
 */
x_cleanup(c)
struct context *c;
{
	struct variable *v;

	while ((v = c->c_vars) != 0) {
		MAG_FREE(v->v_name);
		val_free(v->v_value);
		c->c_vars = v->v_next;
		MAG_FREE(v);
	}
}

/* Push a <value, seq> pair onto the stack.
 */
x_dirpush(v)
struct value *v;
{
	if (x_sp == x_stack)
		mag_panic("expression stack overflow");
	if (v != 0 && v->v_refcnt <= 0)
		abort();
	*--x_sp = v;
}

/* Like x_dirpush, but copy the value.
 */
x_push(v)
struct value *v;
{
	x_dirpush(val_ref(v));
}

/* Pop a value of the stack and return its seqno in seq.
 */
struct value *x_pop(c)
struct context *c;
{
	if (x_sp == c->c_top)
		return 0;
	else {
		if (*x_sp != 0 && c->c_seq < val_seq(*x_sp))
			c->c_seq = val_seq(*x_sp);
		return *x_sp++;
	}
}

/* Push a double.
 */
x_dblpush(x, seq)
double x;
{
	x_dirpush(val_pdbl(x, seq));
}

/* Push an integer.
 */
x_numpush(x, seq){
	x_dirpush(val_pint(x, seq));
}

/* Pop a floating point number.
 */
double x_dblpop(c)
struct context *c;
{
	struct value *v = x_pop(c);
	double x = val_dbl(v);

	val_free(v);
	return x;
}

/* Pop a number.
 */
x_numpop(c)
struct context *c;
{
	struct value *v = x_pop(c);
	int x = val_int(v);

	val_free(v);
	return x;
}

/* Duplicate the top of the stack.
 */
x_dup(c)
struct context *c;
{
	if (x_sp == c->c_top)
		return;
	x_push(*x_sp);
}

/* Swap the top two stack entries.
 */
x_swap(c)
struct context *c;
{
	struct value *tmp, **s;

	if (x_sp == c->c_top)
		return;
	s = c->c_top - 1;
	if (x_sp == s) {
		x_dirpush(*s);
		*s = 0;
		return;
	}
	tmp = x_sp[0];
	x_sp[0] = x_sp[1];
	x_sp[1] = tmp;
}

/* Remove the top of the stack.
 */
x_pop1(c)
struct context *c;
{
	val_free(*x_sp++);
}

/* Push a null.
 */
x_null(c)
struct context *c;
{
	x_dirpush((struct value *) 0);
}

/* Scan *ps until delim is found, and return the substring in token.
 */
x_token(ps, delim, token)
char **ps, *token;
{
	char *s = *ps;

	while (*s != 0 && *s != delim) {
		if (*s == '\\') {
			if ((*token = *++s) == 0)
				break;
		}
		else
			*token = *s;
		s++;
		token++;
	}
	*token = 0;
	*ps = s;
}

/* Compute a hash value out of a string.
 */
x_hashoper(s)
char *s;
{
	int n = 0;

	while (*s != 0) {
		n <<= 1;
		n ^= *s++;
	}
	if (n < 0)
		n = -n;
	return n % OPER_HASH;
}

/* Look at token s.  If it's an operator, add an operator token to the
 * compiled program.  If it's a variable, add a variable token.  Otherwise
 * add a value token.
 */
x_operator(t, s)
struct token *t;
char *s;
{
	struct operator *op;

	for (op = x_opers[x_hashoper(s)]; op != 0; op = op->op_next)
		if (scmp(s, op->op_name)) {
			t->t_type = T_OPERATOR;
			t->t_oper = op->op_func;
			return;
		}
	if (*s == '%') {
		t->t_type = T_VARIABLE;
		t->t_variable = mag_copy(s + 1);
	}
	else {
		t->t_type = T_VALUE;
		t->t_value = val_cstr(s, 0);
	}
}

/* Compile p->p_prog into p->p_tokens.  Split the program into string tokens
 * using x_token, and then decide what kind of token it is using x_operator.
 * Tokens starting with '"' are quoted value tokens.
 */
x_compile(p)
struct prog *p;
{
	char token[256], *prog = p->p_prog;
	struct token tl[100], *t = tl;

	while (*prog != 0) {
		if (t == &tl[100])
			mag_panic("too many tokens");
		while (*prog == ' ')
			prog++;
		if (*prog == 0)
			break;
		if (*prog == '"') {	/* quoted value token */
			prog++;
			x_token(&prog, '"', token);
			if (*prog == '"')
				prog++;
			t->t_type = T_VALUE;
			t->t_value = val_cstr(token, 0);
		}
		else {
			x_token(&prog, ' ', token);
			x_operator(t, token);
		}
		t++;
	}
	p->p_tokens = MAG_MALLOC(t - tl, struct token);
	p->p_last = &p->p_tokens[t - tl];
	bcopy((char *) tl, (char *) p->p_tokens, (char *) t - (char *) tl);
}

/* Lookup the value of variable s and push it.  If there is no such variable,
 * push null.
 */
x_variable(c, s)
struct context *c;
char *s;
{
	struct context *ctx;
	struct variable *v;

	for (ctx = c; ctx != 0; ctx = ctx->c_parent)
		for (v = ctx->c_vars; v != 0; v = v->v_next)
			if (scmp(s, v->v_name)) {
				x_push(v->v_value);
				return;
			}
	x_null(c);
}

/* Lookup prog in the cache of compiled programs.  If it isn't there,
 * compile it and install it in the cache.
 */
struct prog *x_lookup(prog)
char *prog;
{
	int l, n = 0, m = 16;
	char *s = prog;
	struct prog *p, **pp;

	while (--m && (l = *s++) != 0)
		n ^= l;
	pp = &x_progs[n % PROG_HASH];
	for (p = *pp; p != 0; p = p->p_next)
		if (scmp(p->p_prog, prog))
			return p;
	p = MAG_MALLOC(1, struct prog);
	p->p_prog = mag_copy(prog);
	p->p_next = *pp;
	*pp = p;
	x_compile(p);
	return p;
}

/* Evaluate prog.  Look it up in the cache (where it will be installed and
 * compiled if it wasn't there), and run through the list of tokens.
 */
x_run(c, prog)
struct context *c;
char *prog;
{
	static level;
	struct prog *p;
	struct token *t;

	if (level == MAX_LEVEL) {
		printf("expression nested too deep (should panic)\n");
		return;
	}
	level++;
	p = x_lookup(prog);
	for (t = p->p_tokens; t < p->p_last; t++)
		switch (t->t_type) {
		case T_VALUE:
			x_push(t->t_value);
			break;
		case T_VARIABLE:
			x_variable(c, t->t_variable);
			break;
		case T_OPERATOR:
			(*t->t_oper)(c);
			break;
		default:
			mag_panic("bad token");
		}
	--level;
}

/* Evaluate the program in the string value v.  If it's not a string value,
 * just push the value.
 */
x_do(c, v)
struct context *c;
struct value *v;
{
	if (v == 0 || v->v_assign != V_STR)
		x_push(v);
	else
		x_run(c, v->v_str);
}

/* Evaluate the value on top of the stack using x_do.
 */
x_eval(c)
struct context *c;
{
	struct value *v = x_pop(c);

	x_do(c, v);
	val_free(v);
}

/* Apply an operator N times.
 */
x_iter(c)
struct context *c;
{
	struct value *op = x_pop(c);
	int n = x_numpop(c);

	while (n-- > 0)
		x_do(c, op);
	val_free(op);
}

/* Copy context c into c2.  Variables aren't copied, but can be accessed
 * as the parent context is accessible through c_parent.
 */
x_copy(c, c2)
struct context *c, *c2;
{
	c2->c_parent = c;
	c2->c_seq = c->c_seq;
	c2->c_record = c->c_record;
	c2->c_attr = c->c_attr;
	c2->c_entry = c->c_entry;
	c2->c_top = x_sp;
	c2->c_vars = 0;
}

/* This takes a record argument and a quoted operator and applies the
 * operator to each value in the list.  The operator is evaluated in a
 * new context.
 */
x_list(c)
struct context *c;
{
	struct value *op, *rec, *ent;
	char *r;
	int N, i;
	struct context ctx;

	op = x_pop(c);
	rec = x_pop(c);
	if ((r = val_str(rec)) != 0) {
		db_numretrieve(r, "N", &N);
		x_copy(c, &ctx);
		for (i = 0; i < N; i++) {
			ent = mag_index(r, i);
			ctx.c_entry = ent;
			x_do(&ctx, op);
			x_cleanup(&ctx);
			val_free(ent);
		}
	}
	val_free(rec);
	val_free(op);
}

/* Do nothing.
 */
x_nop(){
}

/* Retrieve something from the data base.
 */
x_doget(c, seq)
struct context *c;
{
	struct value *rec, *attr;
	char *r, *a;

	attr = x_pop(c);
	rec = x_pop(c);
	if ((r = val_str(rec)) == 0 || (a = val_str(attr)) == 0) {
		val_free(rec);
		val_free(attr);
		x_null(c);
		return;
	}
	x_dirpush(db_get(r, a, seq));
	val_free(rec);
	val_free(attr);
}

/* Retrieve something from the data base at a specific time.
 */
x_get(c)
struct context *c;
{
	x_doget(c, x_numpop(c));
}

/* Like x_get, but use current sequence number.
 */
x_retrieve(c)
struct context *c;
{
	extern db_seq;

	x_doget(c, db_seq);
}

/* Push the sequence number of the top stack entry.
 */
x_seq(c)
struct context *c;
{
	struct value *seq = x_pop(c), *result;

	if (seq == 0)
		x_null(c);
	else {
		result = val_pint(val_seq(seq), val_seq(seq));
		val_free(seq);
		x_dirpush(result);
	}
}

/* Convert the top of the stack to a string a la ctime.
 */
x_time(c)
struct context *c;
{
	char *ctime(), *s;
	struct value *now = x_pop(c);
	long t;

	if (val_null(now))
		x_dirpush(now);
	else {
		t = val_int(now);
		s = ctime(&t);
		s[24] = 0;		/* get rid of newline */
		x_dirpush(val_cstr(s, val_seq(now)));
		val_free(now);
	}
}

/* Push the name of the record containing the function being evaluated.
 */
x_record(c)
struct context *c;
{
	x_dirpush(val_cstr(c->c_record, c->c_seq));
}

/* Push the name of the attribute being evaluated.
 */
x_attr(c)
struct context *c;
{
	x_dirpush(val_cstr(c->c_attr, c->c_seq));
}

/* Push the name of the entry in x_list and x_select operators.
 */
x_entry(c)
struct context *c;
{
	x_push(c->c_entry);
}

/* Top of stack contains condition and two values.  Remove one of the two
 * according to the condition.
 */
x_if(c)
struct context *c;
{
	struct value *true, *false, *cond;

	false = x_pop(c);
	true = x_pop(c);
	cond = x_pop(c);
	if (val_int(cond)) {
		if (val_seq(cond) < val_seq(true))
			true = val_pseq(true, val_seq(cond), V_FREE);
		x_dirpush(true);
		val_free(false);
	}
	else {
		if (val_seq(cond) < val_seq(false))
			false = val_pseq(false, val_seq(cond), V_FREE);
		x_dirpush(false);
		val_free(true);
	}
	val_free(cond);
}

/* Assign to a temporary variable.
 */
x_assign(c)
struct context *c;
{
	struct value *var, *v;
	char *name;

	var = x_pop(c);
	v = x_pop(c);
	if ((name = val_str(var)) != 0)
		x_bind(c, name, v);
	val_free(v);
	val_free(var);
}

/* Evaluate attribute attr in record rec which currently has value v.
 * Update the sequence number in the result so that it is not lower than
 * the sequence number of v.  Cleanup the stack afterwards.
 */
struct value *x_expr(rec, attr, v)
char *rec, *attr;
struct value *v;
{
	struct context ctx;

	ctx.c_top = x_sp;
	ctx.c_parent = 0;
	ctx.c_seq = val_seq(v);
	ctx.c_vars = 0;
	ctx.c_record = rec;
	ctx.c_attr = attr;
	x_run(&ctx, val_str(v) + 1);
	if ((v = x_pop(&ctx)) != 0 && val_seq(v) < ctx.c_seq)
		v = val_pseq(v, ctx.c_seq, V_FREE);
	while (x_sp != ctx.c_top)
		val_free(*x_sp++);
	x_cleanup(&ctx);
	return v;
}

/* Register an operator.
 */
x_register(name, func)
char *name;
int (*func)();
{
	struct operator *op, **hash;

	hash = &x_opers[x_hashoper(name)];
	op = MAG_MALLOC(1, struct operator);
	op->op_name = name;
	op->op_func = func;
	op->op_next = *hash;
	*hash = op;
}

/* Register all operators.  Call other start routines that register
 * operators.
 */
x_start(){
	x_sp = &x_stack[STACK_SIZE];
	x_register("$pop", x_pop1);
	x_register("$dup", x_dup);
	x_register("$iter", x_iter);
	x_register("$swap", x_swap);
	x_register("$eval", x_eval);
	x_register("$list", x_list);
	x_register("$nop", x_nop);
	x_register("$null", x_null);
	x_register("$get", x_get);
	x_register("#", x_retrieve);
	x_register("$seq", x_seq);
	x_register("$time", x_time);
	x_register("$record", x_record);
	x_register("$attr", x_attr);
	x_register("$entry", x_entry);
	x_register("?", x_if);
	x_register(":=", x_assign);
	m_start();
	str_start();
	sort_start();
	sel_start();
}
