/*
 * Copyright 1992 Petcom Industries Limited.  All rights reserved.
 *
 * As far as I'm concerned, you can do anything at all you want with
 * this code.  You must, however, retain this and all other copyright
 * notices in the source code, and abide by this and all other copyright
 * notices.  In particular, you are forbidden to market this code, or
 * derived code or binaries.
 */

/*
 *            Copyright 1991, Intel Corporation
 *                  All rights reserved.
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby granted,
 * provided that the above copyright notice appear in all copies and
 * that both the copyright notice appear in all copies and that both
 * the copyright notice and this permission notice appear in
 * supporting documentation, and that the name of Intel Corporation
 * not be used in advertising or publicity pertaining to distribution
 * of the software without specific, written prior premission.
 * 
 * COMPANY AND/OR INTEL DISCLAIM ALL WARRANTIES WITH REGARD TO
 * THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE. IN NO
 * EVENT SHALL COMPANY NOR INTEL BE LIABLE FOR ANY SPECIAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER
 * RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE
 * OF THIS SOFTWARE.
 */


/*
 * $Header: /u3/src/slip/driver/io/RCS/slp.c,v 1.4 92/05/08 12:02:31 root Exp $
 * $Author: root $
 *
 * REVISION HISTORY:
 *
 * $Log:	slp.c,v $
 * Revision 1.4  92/05/08  12:02:31  root
 * Compression seems to be functioning well now.  Increased the
 * hi and lo water marks to attempt to get as much traffic localized
 * as possible, in order to increase the priority band effect.
 * 
 * Revision 1.3  92/05/01  16:41:40  root
 * VJ header compression installed, testing is proceeding.  Seems
 * to be working nicely.  Need a bit of work on either the
 * hi-water mark or the qband stuff.
 * 
 * Revision 1.2  92/04/28  09:49:46  root
 * Fixed a number of bugs.  Changed b_datap->db_base references
 * to b_rptr;  fixed bug in slip_dl_cmds that never freed
 * an allocated message block on unitdata requests;  handling
 * of ifstats chain fixed.
 * 
 * Revision 1.1  92/04/28  09:46:34  root
 * Initial revision
 *
 */

static char RCS_header[] = "$Header: /u3/src/slip/driver/io/RCS/slp.c,v 1.4 92/05/08 12:02:31 root Exp $";
static char RCS_author[] = "$Author: root $";


/*
 * SLIP6 - serial line internet protocol streams multiplexor driver
 *	   for SVR4.  Van-Jacobssen header compression implemented.
 *
 *	 - the basic module came from the Intel PD slip release.  A
 *	   number of bugs were fixed, and then the 386BSD release of
 *	   slip header compression was added.   See comments below.
 */

/*
 * Serial Line Internet Protocol (SLIP) streams multiplexor driver for
 * Intel Unix System V/386 Release 4.0.
 *
 * The upper streams is supposed to be linked to the ip driver (/dev/ip)
 * and optionally linked to the slip hangup daemon. The lower streams can
 * linked to any number of serial driver.
 *
 * The slattach command builds the ip to slip to serial device links and
 * The slhangupd command is a daemon for receiving M_HANGUP message from
 * the serial driver.
 *
 * The packet framing protocol code, in the upper write and the lower
 * read service routine (slip_uwsrv and slip_lrsrv) is based from
 * tty_slip.c written by Rayan Zachariassen <rayan@ai.toronto.edu> and
 * Doug Kingston <dpk@morgan.com>.
 *
 * Author:
 *     Sudji Husodo <sudji@indo.intel.com> 1/9/91
 */

#include <sys/types.h>
#include <sys/kmem.h>
#include <sys/stream.h>
#include <sys/stropts.h>
#include <sys/dlpi.h>
#include <sys/syslog.h>
#include <sys/strlog.h>
#include <sys/socket.h>
#include <sys/sockio.h>
#include <sys/log.h>
#include <sys/errno.h>
#include <sys/ioctl.h>
#include <sys/param.h>
#include <sys/ddi.h>

#include <net/if.h>
#include <netinet/in.h>
#include <netinet/in_systm.h>
#include <netinet/ip.h>
#include <sys/slp.h>

#define DL_PRIM_SIZE	sizeof (union DL_primitives)

/*
 * SLMAX is a hard limit on input packet size.  To simplify the code
 * and improve performance, we require that packets fit in an mbuf
 * cluster, and if we get a compressed packet, there's enough extra
 * room to expand the header into a max length tcp/ip header (128
 * bytes).  So, SLMAX can be at most
 *	MCLBYTES - 128
 *
 * SLMTU is a hard limit on output packet size.  To insure good
 * interactive response, SLMTU wants to be the smallest size that
 * amortizes the header cost.  (Remember that even with
 * type-of-service queuing, we have to wait for any in-progress
 * packet to finish.  I.e., we wait, on the average, 1/2 * mtu /
 * cps, where cps is the line speed in characters per second.
 * E.g., 533ms wait for a 1024 byte MTU on a 9600 baud line.  The
 * average compressed header size is 6-8 bytes so any MTU > 90
 * bytes will give us 90% of the line bandwidth.  A 100ms wait is
 * tolerable (500ms is not), so want an MTU around 296.  (Since TCP
 * will send 256 byte segments (to allow for 40 byte headers), the
 * typical packet size on the wire will be around 260 bytes).  In
 * 4.3tahoe+ systems, we can set an MTU in a route so we do that &
 * leave the interface MTU relatively high (so we don't IP fragment
 * when acting as a gateway to someone using a stupid MTU).
 *
 * Similar considerations apply to SLIP_HIWAT:  It's the amount of
 * data that will be queued 'downstream' of us (i.e., in clists
 * waiting to be picked up by the tty output interrupt).  If we
 * queue a lot of data downstream, it's immune to our t.o.s. queuing.
 * E.g., if SLIP_HIWAT is 1024, the interactive traffic in mixed
 * telnet/ftp will see a 1 sec wait, independent of the mtu (the
 * wait is dependent on the ftp window size but that's typically
 * 1k - 4k).  So, we want SLIP_HIWAT just big enough to amortize
 * the cost (in idle time on the wire) of the tty driver running
 * off the end of its clists & having to call back slstart for a
 * new packet.  For a tty interface with any buffering at all, this
 * cost will be zero.  Even with a totally brain dead interface (like
 * the one on a typical workstation), the cost will be <= 1 character
 * time.  So, setting SLIP_HIWAT to ~100 guarantees that we'll lose
 * at most 1% while maintaining good interactive response.
 */
#define BUFOFFSET	128		/* space required at START of inbuf */
#define	SLMTU		296		/* slip MTU - for 9600 baud */

/*
 * The following disgusting hack gets around the problem that IP TOS
 * can't be set yet.  We want to put "interactive" traffic on a high
 * priority queue.  To decide if traffic is interactive, we check that
 * a) it is TCP and b) one of its ports is telnet, rlogin or ftp control.
 */
static u_int interactive_ports[8] = {
	0,	513,	0,	0,
	0,	21,	0,	23,
};
#define INTERACTIVE(p) (interactive_ports[((p) >> 8) & 7] == (p))

interactive(p)
u_short p;
{
	p = ntohs(p);
	if(interactive_ports[p & 7] == p) return(1);
		else return(0);
}

int slp_devflag = 0;		/* V4.0 style driver */

#ifndef DEBUG
static
#endif
int slip_open(),  slip_close(), slip_uwput(), slip_uwsrv(),
    slip_lwput(), slip_lwsrv(), slip_lrput(), slip_lrsrv();

#define KBYTE 1024

static struct module_info minfo[5] = {
	SLIPM_ID, "slp", 0, 8192, 64*KBYTE, 32*KBYTE,
	SLIPM_ID, "slp", 0, 8192, 64*KBYTE, 32*KBYTE,
	SLIPM_ID, "slp", 0, 8192, 64*KBYTE, 32*KBYTE,
	SLIPM_ID, "slp", 0, 8192, 64*KBYTE, 32*KBYTE,
	SLIPM_ID, "slp", 0, 8192, 64*KBYTE, 32*KBYTE,
};

static struct qinit urinit = {
	NULL,       NULL,       slip_open, slip_close, NULL, &minfo[0], NULL };

static struct qinit uwinit = {
	slip_uwput, slip_uwsrv, slip_open, slip_close, NULL, &minfo[1], NULL };

static struct qinit lrinit = {
	slip_lrput, slip_lrsrv, slip_open, slip_close, NULL, &minfo[3], NULL };

static struct qinit lwinit = {
	slip_lwput, slip_lwsrv, slip_open, slip_close, NULL, &minfo[4], NULL };

struct streamtab slp_info = {
	&urinit, &uwinit, &lrinit, &lwinit };

slip_t *slip_hup = (slip_t *) 0;

extern	struct	ifstats	*ifstats;
extern	u_int	slip_num;
extern	slip_t	slip_data[];

int npackets = 0;

/*
 * slip_open
 */

#ifndef DEBUG
static
#endif
slip_open (q, devp, flag, sflag, credp)
queue_t	*q;
dev_t	*devp;
int		flag;
int		sflag;
struct	cred	*credp;
{
	register slip_t *p_slip;
	dev_t	 dev;
	int		 oldpri;
	mblk_t	 *bp;
	major_t  major = getmajor (*devp);
	minor_t  minor = getminor (*devp);

	STRLOG (SLIPM_ID,0,0,SL_TRACE,"slip_open: major %d minor %d",major,minor);

	/* find an unused entry in slip_data */

	if (sflag == CLONEOPEN) {
		for (dev=0, p_slip=&slip_data[0]; dev<slip_num; dev++, p_slip++)
			if (!p_slip->buf)
				break;
		minor = (minor_t) dev;
	}

	/* if there's no more free entry, return No Space error code */

	if (minor >= slip_num) {
		STRLOG (SLIPM_ID, 1, 0, SL_TRACE, "slip open: can't allocate device");
		return ENXIO;
	}

	/* initialized slip information */

	oldpri = splstr ();
	p_slip->state    = DL_UNBOUND;
	p_slip->qtop     = q;
	p_slip->qbot     = NULL;
	p_slip->buf      = (u_char *) kmem_alloc (SLIPMTU+BUFOFFSET, KM_SLEEP);
	p_slip->bptr	 = p_slip->buf + BUFOFFSET;
	p_slip->qt_blocked = 0;
	drv_getparm (PPID, &p_slip->pid);		/* keep process id */

	p_slip->escape   = p_slip->overrun = p_slip->inlen = 0;
	p_slip->flags    = IFF_UP | IFF_POINTOPOINT;
	p_slip->uname[0] = '\0';

	/* initialized interface and its statistics */

	p_slip->stats.ifs_name       = (char *) p_slip->uname;
	p_slip->stats.ifs_unit       = 0;
	p_slip->stats.ifs_active     = 0;
	p_slip->stats.ifs_mtu	     = SLMTU;
	p_slip->stats.ifs_ipackets   = p_slip->stats.ifs_opackets = 0;
	p_slip->stats.ifs_ierrors    = p_slip->stats.ifs_oerrors  = 0;
	p_slip->stats.ifs_collisions = 0;

	sl_compress_init(&(p_slip->sc_comp));
	p_slip->sc_flags = SC_AUTOCOMP;
	p_slip->flags |= IFF_NOARP;
	p_slip->xmit_hiband = p_slip->recv_hiband = 0;
	p_slip->uw_blocked = 0;

	splx (oldpri);

	/* initialize read and write queue pointers to private data */

	q->q_ptr = (caddr_t) p_slip;
	WR(q)->q_ptr = (caddr_t) p_slip;

	/* set up the correct stream head flow control parameters */

	if (bp = allocb (sizeof (struct stroptions), BPRI_MED)) {
		struct stroptions *sop = (struct stroptions *) bp->b_rptr;
		bp->b_datap->db_type = M_SETOPTS;
		bp->b_wptr += sizeof (struct stroptions);
		sop->so_flags = SO_HIWAT | SO_LOWAT;
		sop->so_hiwat = minfo [2].mi_hiwat;
		sop->so_lowat = minfo [2].mi_lowat;
		putnext (q, bp);
	}
	*devp = makedevice (major, minor);
	STRLOG (SLIPM_ID, 0, 0, SL_TRACE, "slip open: device %d coming up", minor);
	return (0);
}

dumpq(label, q)
char *label;
queue_t *q;
{
	queue_t *qp;

	for(qp=q; qp; qp=qp->q_next) {
		printf("%s: name <%s>, hiwat %d, lowat %d",
			label,
			qp->q_qinfo->qi_minfo->mi_idname,
			qp->q_hiwat,
			qp->q_lowat);
		if(qp->q_qinfo->qi_srvp) printf(" SRV\n");
			else printf("\n");
	}
}

/*
 * slip_close ()
 */

#ifndef DEBUG
static
#endif
slip_close (q)
queue_t	*q;
{
	slip_t	*p_slip;
	dev_t	 dev;
	int		oldpri;
	struct ifstats *ifp, *pifp;

	STRLOG (SLIPM_ID, 0, 0, SL_TRACE, "slip_close: going down ...");
	p_slip = (slip_t *) q->q_ptr;
	oldpri = splstr ();

	if (p_slip->buf)
		kmem_free (p_slip->buf, SLIPMTU+BUFOFFSET);

	p_slip->state = DL_UNATTACHED;
	p_slip->buf   = 0;
	p_slip->bptr  = 0;
	p_slip->inlen = 0;
	p_slip->qtop  = 0;
	p_slip->pid   = 0;

	for(ifp = ifstats, pifp = NULL; ifp; ifp = ifp->ifs_next) {
		if(ifp == &p_slip->stats) {
			if(pifp) pifp->ifs_next = ifp->ifs_next;
				else ifstats = ifp->ifs_next;
			break;
		}
		pifp = ifp;
	}
	splx (oldpri);
}

/*
 *	slip_ioctl ()
 */

#ifndef DEBUG
static
#endif
slip_ioctl (q, mp)
queue_t	*q;
mblk_t	*mp;
{
	struct iocblk *iocp;
	struct ifreq  *ifr;
	slip_t *p_slip;
	int		oldpri;
	struct	linkblk	*lp;
	slip_t *pp;
	u_char *p;
	int n;

	p_slip  = (slip_t *) q->q_ptr;
	iocp = (struct iocblk *) mp->b_rptr;

	STRLOG (SLIPM_ID, 1, 0, SL_TRACE, "slip_ioctl: enter: case ('%c',%d)", 0x00FF&(iocp->ioc_cmd>>8), 0x00FF&iocp->ioc_cmd);
	oldpri = splstr ();

	switch (iocp->ioc_cmd) {

	case REG_SLHUP:
		STRLOG (SLIPM_ID, 1, 0, SL_TRACE, "slip_ioctl: REG_SLHUP");
		if (slip_hup) {
			splx (oldpri);
			mp->b_datap->db_type = M_IOCNAK;
			qreply (q, mp);
			return (0);
		}
		else
			slip_hup = p_slip;
		break;

	case UNREG_SLHUP:
		STRLOG (SLIPM_ID, 1, 0, SL_TRACE, "slip_ioctl: UNREG_SLHUP");
		slip_hup = 0;
		break;

	case DO_COMPRESS:
		p_slip->sc_flags |= SC_COMPRESS;
		p_slip->flags |= IFF_NOTRAILERS;
		break;

	case DO_AUTOCOMP:
		p_slip->sc_flags |= SC_AUTOCOMP;
		p_slip->flags |= IFF_NOARP;
		break;

	case I_LINK:
		iocp->ioc_error = iocp->ioc_rval = iocp->ioc_count = 0;

		lp = (struct linkblk *) mp->b_cont->b_rptr;
		p_slip->qbot = lp->l_qbot;
		p_slip->qbot->q_ptr = (char *) p_slip;
		OTHERQ (p_slip->qbot)->q_ptr = (char *) p_slip;
		break;

	case I_UNLINK:
		iocp->ioc_error = iocp->ioc_rval = iocp->ioc_count = 0;
		p_slip->qbot = NULL;
		break;

	case SIOCSIFNAME:
		ifr = (struct ifreq *) mp->b_cont->b_rptr;

		/* copy interface name to local slip structure */

		/*
		 * interface name (ifr->ifr_name) contains the name and unit, e.g.
		 * "sl0", "sl12", "emd0", "wd1", etc. Store the name in slip->uname
		 * and unit in slip->unit. If unit is not supplied, e.g. "slip",
		 * then unit number is assumed to be zero.
		 */

		strncpy (p_slip->uname, ifr->ifr_name, IFNAMSIZ);	/* copy name */
		n = 0;

		/* starting from the last char, find the first non-digit char */

		p = p_slip->uname + strlen(p_slip->uname) - 1;
		while ('0' <= *p && *p <= '9')
			p--;

		/* calculate integer, replace them with nulls */

		while (*++p) {
			n = 10*n + (*p-'0');
			*p = '\0';
		}
		p_slip->stats.ifs_unit = n;				/* set ifs unit number  */

		/* search for matching interface name and unit */

		for (n=0, pp=&slip_data[0]; n<slip_num; n++, pp++)
			if (pp != p_slip && pp->buf && !strcmp (pp->uname,p_slip->uname) &&
				pp->stats.ifs_unit == p_slip->stats.ifs_unit)
				break;

		if (n < slip_num) {							/* found matching ifname */
			splx (oldpri);
			mp->b_datap->db_type = M_IOCNAK;		/* Negative Ack reply */
			qreply (q, mp);
			return (0);
		}
		else {										/* ifname is unique */
			p_slip->stats.ifs_next = ifstats;		/* set ifstats pointers */
			ifstats = &p_slip->stats;				/* used for statistics */
		}											/* by netstat command */
		break;

	case SIOCGIFFLAGS:
		((struct iocblk_in *)iocp)->ioc_ifflags = p_slip->flags;
		break;

	case SIOCSIFFLAGS:
		p_slip->flags = ((struct iocblk_in *)iocp)->ioc_ifflags;
		break;

	case SIOCSIFADDR:
		p_slip->flags |= IFF_RUNNING;
		((struct iocblk_in *)iocp)->ioc_ifflags |= IFF_RUNNING;
		break;

	default:
		break;
	}
	splx (oldpri);
	mp->b_datap->db_type = M_IOCACK;
	qreply (q, mp);
}

/*
 * slip_uwput ()
 */

#ifndef DEBUG
static
#endif
slip_uwput (q, mp)
queue_t	*q;
mblk_t	*mp;
{
	STRLOG (SLIPM_ID, 0, 0, SL_TRACE, "slip_uwput: case 0x%2.2x", mp->b_datap->db_type);

	switch (mp->b_datap->db_type) {

	case M_FLUSH:
		if (*mp->b_rptr & FLUSHW) {
			flushq(q, FLUSHALL);
			*mp->b_rptr &= ~FLUSHW;
		}
		if (*mp->b_rptr & FLUSHR)
			qreply(q, mp);
		else
			freemsg(mp);
		break;

	case M_IOCTL:
		slip_ioctl (q, mp);
		break;

	case M_PROTO:
	case M_PCPROTO:
		slip_dl_cmds (q, mp);
		break;

	default:
		STRLOG (SLIPM_ID,1,0,SL_TRACE,"slip_uwput: unknown message type, passing message to the next queue");
		putnext (((slip_t *)q->q_ptr)->qbot, mp);
		break;
	} 
}

/*
 * slip_uwsrv ()
 */

#ifndef DEBUG
static
#endif
slip_uwsrv (q)
register queue_t *q;
{
	register mblk_t	*mp, *mpd, *mp2;
	register u_char	*cp;
	register int	pktlen, num;
	register slip_t *p_slip;
	register struct ip *ip;

	STRLOG (SLIPM_ID, 0, 0, SL_TRACE, "slip_uwsrv: enter");

	while ((mp = getq(q)) != NULL) {
		STRLOG (SLIPM_ID,1,0,SL_TRACE,"slip_uwsrv: got message from q");
		p_slip = (slip_t *) q->q_ptr;

		pktlen = 0;

		if((ip=(struct ip *)mp->b_cont->b_rptr)->ip_p == IPPROTO_TCP) {
			register mblk_t *mptr;
			u_short tmp;
			register int p = ((int *)ip)[ip->ip_hl];

			tmp = ntohs(ip->ip_len);

			/* CORRECT the output byte wptr */
			for(mptr=mp->b_cont; mptr->b_cont; mptr=mptr->b_cont) {
				tmp -= (u_short)(mptr->b_wptr - mptr->b_rptr);
			}
			mptr->b_wptr = mptr->b_rptr + tmp;

			if (interactive(p >> 16) || interactive(p & 0xffff)) {
				p = 1;
				mp->b_band = 1;
			} else {
				p = 0;
			}
			if (p_slip->sc_flags & SC_COMPRESS) {
				/*
				 * The last parameter turns off connection id
				 * compression for background traffic:  Since
				 * fastq traffic can jump ahead of the background
				 * traffic, we don't know what order packets will
				 * go on the line.
				 */
				p = sl_compress_tcp(mp->b_cont, ip,
					&p_slip->sc_comp, p);
				*((u_char *) mp->b_cont->b_rptr) |= p;
			}
		} else if (p_slip->sc_flags & SC_NOICMP && ip->ip_p == IPPROTO_ICMP) {
			freemsg(mp);
			continue;
		}
		if (!bcanput(p_slip->qbot,mp->b_band)) {
			STRLOG (SLIPM_ID,1,0,SL_TRACE,"slip_uwsrv: can't put message to qbot");
			putbq (q, mp);
			p_slip->qt_blocked = 1;
			p_slip->uw_blocked++;
			return;
		}
		/*
		 * count the number of special characters (END & ESC)
		 */

		num = 2;		/* END char is put at the start and end of packet */

		for (mpd = mp->b_cont; mpd != 0; mpd = mpd->b_cont) {
			pktlen += (mpd->b_wptr - mpd->b_rptr);

			for (cp = mpd->b_rptr; cp < mpd->b_wptr; cp++) {
				if (*cp == END || *cp == ESC)
					num++;
			}
		}
		STRLOG (SLIPM_ID, 1, 0, SL_TRACE, "slip_uwsrv: # of bytes in packet = %d; # of special character in packet: %d", pktlen, num);
		/*
		 * allocate message block to be sent down stream
		 */

		if ((mp2 = allocb (pktlen + num, BPRI_MED)) == NULL) {
			STRLOG (SLIPM_ID,1,0,SL_TRACE,"slip_uwsrv: can't allocate message block - dropping outgoing message");
			p_slip->stats.ifs_oerrors++;
			freemsg (mp);
			return;
		}
		/*
		 * frame packet, escape special characters ESC and END
		 */

		*mp2->b_wptr++ = END;

		for (mpd = mp->b_cont; mpd != 0; mpd = mpd->b_cont) {
			for (cp = mpd->b_rptr; cp < mpd->b_wptr; cp++) {
				if (*cp == END) {
					*mp2->b_wptr++ = ESC;
					*mp2->b_wptr++ = ESC_END;
				}
				else if (*cp == ESC) {
					*mp2->b_wptr++ = ESC;
					*mp2->b_wptr++ = ESC_ESC;
				}
				else
					*mp2->b_wptr++ = *cp;
			}
		}
		*mp2->b_wptr++ = END;

		mp2->b_datap->db_type = M_DATA;
		mp2->b_band = mp->b_band;
		p_slip->xmit_hiband += mp->b_band;
		p_slip->stats.ifs_opackets++;
		freemsg (mp);
		slip_lwput (p_slip->qbot, mp2);
	}
}

dump_slstats(p_slip)
slip_t *p_slip;
{
	struct slcompress *sc = &p_slip->sc_comp;
	printf("packets %d compr %d search %d miss %d\n",
		sc->sls_packets, sc->sls_compressed,
		sc->sls_searches, sc->sls_misses);
	printf("uncompin %d, compin %d, errorin %d, tossedin %d\n",
		sc->sls_uncompressedin,
		sc->sls_compressedin,
		sc->sls_errorin,
		sc->sls_tossed);
	printf("hisend %d hirecv %d uwblocked %d\n",p_slip->xmit_hiband,
		p_slip->recv_hiband,p_slip->uw_blocked);
}
/*
 * slip_lwput
 */

#ifndef DEBUG
static
#endif
slip_lwput (q, mp)
queue_t *q;
mblk_t  *mp;
{
	STRLOG (SLIPM_ID, 0, 0, SL_TRACE, "slip_lwput: enter %x, %x",q , mp);

	if (bcanput(q->q_next,mp->b_band)) {
		STRLOG (SLIPM_ID, 0, 0, SL_TRACE, "slip_lwput: putnext");
		putnext(q, mp);
	}
	else {
		STRLOG (SLIPM_ID, 0, 0, SL_TRACE, "slip_lwput: putq");
		putq(q, mp);
	}
}

/*
 * slip_lwsrv (q)
 */

#ifndef DEBUG
static
#endif
slip_lwsrv (q)
queue_t *q;
{
	mblk_t  *mp;
	slip_t *p_slip;

	STRLOG (SLIPM_ID, 0, 0, SL_TRACE,"slip_lwsrv: enter");

	while (mp = getq(q)) {
		STRLOG (SLIPM_ID, 0, 0, SL_TRACE,"slip_lwsrv: getq (%x) = %x",q, mp);
		if (bcanput(q->q_next,mp->b_band)) {
			STRLOG (SLIPM_ID, 0, 0, SL_TRACE,"slip_lwsrv: putnext");
			putnext(q, mp);
		}
		else {
			STRLOG (SLIPM_ID, 0, 0, SL_TRACE,"slip_lwsrv: putbq");
			putbq(q, mp);
			return;
		}
	}
	p_slip = (slip_t *) q->q_ptr;

	if (p_slip->qt_blocked) {
		STRLOG (SLIPM_ID,1,0,SL_TRACE,"slip_lwsrv: slip_uwsrv is blocked, enable it");
		/*
		 * qtop is the upper read q and
		 * we want to enable the upper write q
		 */
		qenable (WR(p_slip->qtop));
		p_slip->qt_blocked = 0;
	}
}

/*
 * slip_lrput ()
 */

#ifndef DEBUG
static
#endif
slip_lrput (q, mp)
queue_t	*q;
mblk_t	*mp;
{
	slip_t 	*p_slip;
	mblk_t	*resp;
	int		device;
	int		oldpri;

	STRLOG (SLIPM_ID, 0, 0, SL_TRACE, "slip_lrput: enter: case 0x%2.2x", mp->b_datap->db_type);
	p_slip = (slip_t *) q->q_ptr;

	switch (mp->b_datap->db_type) {

	case M_DATA:
		STRLOG (SLIPM_ID,2,0,SL_TRACE,"slip_lrput: putq (%x, %x)", q, mp);
		putq (q, mp);
		break;

	case M_FLUSH:
		STRLOG (SLIPM_ID,2,0,SL_TRACE,"slip_lrput: M_FLUSH type = %x",*mp->b_rptr);
		if (*mp->b_rptr & FLUSHR)
			flushq(q, FLUSHALL);
		if (*mp->b_rptr & FLUSHW) {
			*mp->b_rptr &= ~FLUSHR;
			flushq(WR(q), FLUSHALL);
			qreply(q, mp);
		} else
			freemsg(mp);
		return;

	case M_HANGUP:
		oldpri = splstr ();
		/*
		 * if pid is set, ignore message
		 */
		if (p_slip->pid == 0)
			freemsg (mp);
		/*
		 * else if slip hangup daemon exists send pid to the daemon
		 */
		else if (slip_hup && (resp = allocb (sizeof(pid_t),BPRI_MED))) {
			STRLOG (SLIPM_ID,1,0,SL_TRACE,
				"slip_lrput: sending pid %d to hangup daemon", p_slip->pid);
			*(pid_t *) resp->b_wptr = p_slip->pid;
			resp->b_wptr += sizeof (pid_t);
			resp->b_datap->db_type = M_PCPROTO;
			putnext (slip_hup->qtop, resp);
			freemsg (mp);
		}
		/*
		 * else pass the message upstream
		 */
		else
			putnext (p_slip->qtop, mp);

		splx (oldpri);
		break;
			
	default:
		putnext (p_slip->qtop, mp);
		break;
	}
}

/*
 * slip_lrsrv ()
 */

#ifndef DEBUG
static
#endif
slip_lrsrv (q)
queue_t	*q;
{
	register	u_char 	*cp;
	register	slip_t 	*p_slip;
	mblk_t		*bp, *mp, *mp1, *mp2;

	STRLOG (SLIPM_ID, 0, 0, SL_TRACE, "slip_lrsrv: enter");

	while ((mp = getq(q)) != NULL) {
		p_slip = (slip_t *) q->q_ptr;

		if (!bcanput (p_slip->qtop->q_next,mp->b_band)) {
			putbq (q, mp);
			return;
		}
		p_slip = (slip_t *) q->q_ptr;

		for (bp = mp; bp != 0; bp = bp->b_cont) {
			for (cp = bp->b_rptr; cp < bp->b_wptr; cp++) {
				if (*cp == END) {
					if (p_slip->inlen < 3);	/* ignore packet */
					else if (p_slip->overrun) {
						printf("SLIP OVERRUN size %d!\n",p_slip->inlen);
						p_slip->stats.ifs_ierrors++;
					} else
						slp_gotpacket(p_slip,q);
				}
				else if (p_slip->inlen >= SLIPMTU)
					p_slip->overrun = 1;

				else if (*cp == ESC)					/* if data is ESC */
					p_slip->escape = 1;

				else if (p_slip->escape) {
					p_slip->escape = 0;

					if (*cp == ESC_END)
						*(p_slip->bptr + p_slip->inlen++) = END;
					else if (*cp == ESC_ESC)
						*(p_slip->bptr + p_slip->inlen++) = ESC;
					else
						*(p_slip->bptr + p_slip->inlen++) = *cp;
				}
				else
					*(p_slip->bptr + p_slip->inlen++) = *cp;
			}
			bp->b_rptr = cp;
		}
		freemsg (mp);
	}
}

slp_gotpacket(p_slip,q)
register slip_t *p_slip;
queue_t *q;
{
	mblk_t *mp1, *mp2;
	register u_int type;
	int oldpri,band;
	register struct ip *ip;

	oldpri = splstr();
	if ((type = (*p_slip->bptr & 0xf0)) != (IPVERSION << 4)) {
		if (type & 0x80) {
			type = TYPE_COMPRESSED_TCP;
		} else if (type == TYPE_UNCOMPRESSED_TCP) {
			*p_slip->bptr &= 0x4f;
		}
		/*
		 * We've got something that's not an IP packet.
		 * If compression is enabled, try to decompress it.
		 * Otherwise, if `auto-enable' compression is on and
		 * it's a reasonable packet, decompress it and then
		 * enable compression.  Otherwise, drop it.
		 */
		if(p_slip->sc_flags & SC_COMPRESS) {
			p_slip->inlen = sl_uncompress_tcp(&p_slip->bptr,
				p_slip->inlen, type, &p_slip->sc_comp);
			if (p_slip->inlen <= 0) {
				p_slip->stats.ifs_ierrors++;
				goto done;
			}
		} else if ((p_slip->sc_flags & SC_AUTOCOMP) &&
		    type == TYPE_UNCOMPRESSED_TCP && p_slip->inlen >= 40) {
			p_slip->inlen = sl_uncompress_tcp(&p_slip->bptr, p_slip->inlen,
						type, &p_slip->sc_comp);
			if (p_slip->inlen <= 0) {
				p_slip->stats.ifs_ierrors++;
				goto done;
			}
			p_slip->sc_flags |= SC_COMPRESS;
			p_slip->flags |= IFF_NOTRAILERS;
		} else {
			p_slip->stats.ifs_ierrors++;
			goto done;
		}
	}
	band = 0;
	if((ip=(struct ip *)p_slip->bptr)->ip_p == IPPROTO_TCP) {
		register int p = ((int *)ip)[ip->ip_hl];

		if (interactive(p >> 16) || interactive(p & 0xffff)) {
			band = 1;
			p_slip->recv_hiband++;
		}
	}

	if ((mp1=allocb (DL_UNITDATA_IND_SIZE,BPRI_MED))==NULL)
		p_slip->stats.ifs_ierrors++;
	else if ((mp2 = allocb (p_slip->inlen, BPRI_MED)) == NULL) {
		p_slip->stats.ifs_ierrors++;
		freemsg (mp1);
	} else {
		p_slip->stats.ifs_ipackets++;	/* send unit data */
		linkb (mp1, mp2);		/* indication up */
		slip_dl_unitdata_ind (q, mp1);	/* stream */
		mp1->b_band = band;
		putnext (p_slip->qtop, mp1);
	}
done:
	p_slip->inlen = 0;			/* reset info for */
	p_slip->bptr = p_slip->buf + BUFOFFSET;
	p_slip->overrun = 0;			/* receiving data */
	p_slip->escape = 0;
	spl(oldpri);
}

/*
 * slip_dl_cmds ()
 */

#ifndef DEBUG
static
#endif
slip_dl_cmds (q, mp)
queue_t	*q;
mblk_t	*mp;
{
	mblk_t				*response;
	slip_t				*p_slip;
	union DL_primitives		*p_dl;
	int		oldpri;

	p_slip = (slip_t *)q->q_ptr;

	p_dl = (union DL_primitives *) mp->b_rptr;

	if(p_dl->dl_primitive == DL_UNITDATA_REQ) {
		if (p_slip->state == DL_IDLE) {
			STRLOG (SLIPM_ID,2,0,SL_TRACE,"slip_dl_cmd: putq (%x, %x)", q, mp);
			putq (q, mp);
			return;
		}
	}

	if ((response = allocb (DL_PRIM_SIZE, BPRI_MED)) == NULL) {
		STRLOG (SLIPM_ID,1,0,SL_TRACE,"slip_dl_cmd: can't allocate response buffer");
		freemsg (mp);
		return;
	}

	switch (p_dl->dl_primitive) {

	case DL_INFO_REQ:
		STRLOG (SLIPM_ID,1,0,SL_TRACE,"slip_dl_cmd: DL_INFO_REQ");
		slip_dl_info_ack (q, mp, response);
		break;

	case DL_BIND_REQ:
		STRLOG (SLIPM_ID,1,0,SL_TRACE,"slip_dl_cmd: DL_BIND_REQ");
		if (p_slip->state == DL_UNBOUND) {
			oldpri = splstr ();
			p_slip->sap   = ((dl_bind_req_t *)p_dl)->dl_sap;
			p_slip->state = DL_IDLE;
			p_slip->stats.ifs_active = 1;
			splx (oldpri);

			slip_dl_bind_ack (q, mp, response);
		}
		else
			slip_dl_error_ack (q, mp, response, DL_OUTSTATE);
		break;

	case DL_UNBIND_REQ:
		STRLOG (SLIPM_ID,1,0,SL_TRACE,"slip_dl_cmd: DL_UNBIND_REQ");
		if (p_slip->state == DL_IDLE) {
			oldpri = splstr ();
			p_slip->state = DL_UNBOUND;
			p_slip->stats.ifs_active = 0;
			splx (oldpri);

			flushq (q, FLUSHDATA);						/* Flush both q's */
			flushq (RD(q), FLUSHDATA);

			slip_dl_ok_ack (q, mp, response);
		}
		else
			slip_dl_error_ack (q, mp, response, DL_OUTSTATE);

		break;

	case DL_UNITDATA_REQ:
		slip_dl_error_ack (q, mp, response, DL_OUTSTATE);
		break;

	default:
		STRLOG (SLIPM_ID,1,0,SL_TRACE,"slip_dl_cmd: default 0x%2.2x",p_dl->dl_primitive);
		slip_dl_error_ack (q, mp, response, DL_UNSUPPORTED);
		break;
	}
	freemsg (mp);
	putnext (RD(q), response);
}

/*
 *  slip_dl_info_ack ()
 */

#ifndef DEBUG
static
#endif
slip_dl_info_ack (q, mp, response)
queue_t *q;
mblk_t	*mp;
mblk_t	*response;
{
	dl_info_ack_t	*p_info_ack;
	slip_t			*p_slip;

	STRLOG (SLIPM_ID, 0, 0, SL_TRACE, "slip_dl_info_ack: enter");

	p_slip = (slip_t *) q->q_ptr;

	p_info_ack = (dl_info_ack_t *) response->b_wptr;
	p_info_ack->dl_primitive     = DL_INFO_ACK;
	p_info_ack->dl_max_sdu       = SLMTU;

	p_info_ack->dl_min_sdu       = 46;                  /* ????? */
	p_info_ack->dl_addr_length   = 0;					/* ????? MAC_ADD_SIZE*/
	p_info_ack->dl_mac_type      = DL_CHAR;				/* ????? */

	p_info_ack->dl_current_state = p_slip->state;
	p_info_ack->dl_service_mode  = DL_CLDLS;			/* connecionless DL */

	p_info_ack->dl_qos_length       = 0;				/* ???? */
	p_info_ack->dl_qos_offset       = 0;				/* ???? */
	p_info_ack->dl_qos_range_length = 0;				/* ???? */
	p_info_ack->dl_qos_range_offset = 0;				/* ???? */

	p_info_ack->dl_provider_style = DL_STYLE1;

	p_info_ack->dl_addr_offset = 0;						/* ???? */
	p_info_ack->dl_growth = 0;

	response->b_datap->db_type = M_PCPROTO;
	response->b_wptr           += DL_INFO_ACK_SIZE;
}

/*
 *  slip_dl_bind_ack ()
 */

#ifndef DEBUG
static
#endif
slip_dl_bind_ack (q, mp, response)
queue_t *q;
mblk_t	*mp;
mblk_t	*response;
{
	dl_bind_req_t	*p_dl;
	dl_bind_ack_t	*p_bind;

	STRLOG (SLIPM_ID, 0, 0, SL_TRACE, "slip_dl_bind_ack: enter");

	p_dl = (dl_bind_req_t *) mp->b_rptr;

	p_bind = (dl_bind_ack_t *) response->b_wptr;
	p_bind->dl_primitive   = DL_BIND_ACK;
	p_bind->dl_sap         = p_dl->dl_sap;
	p_bind->dl_addr_length = 0;
	p_bind->dl_addr_offset = 0;

	response->b_wptr           += DL_BIND_ACK_SIZE;
	response->b_datap->db_type = M_PCPROTO;
}

/*
 *  slip_dl_ok_ack ()
 */

#ifndef DEBUG
static
#endif
slip_dl_ok_ack (q, mp, response)
queue_t *q;
mblk_t	*mp;
mblk_t	*response;
{
	union DL_primitives	*p_dl;
	dl_ok_ack_t			*p_ok_ack;

	STRLOG (SLIPM_ID, 0, 0, SL_TRACE, "slip_dl_ok_ack: enter");

	p_dl = (union DL_primitives *) mp->b_rptr;

	p_ok_ack = (dl_ok_ack_t *)(response->b_wptr);
	p_ok_ack->dl_primitive         = DL_OK_ACK;
	p_ok_ack->dl_correct_primitive = p_dl->dl_primitive;

	response->b_wptr           += DL_OK_ACK_SIZE;
	response->b_datap->db_type = M_PCPROTO;
}

/*
 * slip_dl_error_ack
 */

#ifndef DEBUG
static
#endif
slip_dl_error_ack (q, mp, response, dl_errno)
queue_t *q;
mblk_t	*mp;
mblk_t	*response;
ulong	dl_errno;
{
	union DL_primitives	*p_dl;
	dl_error_ack_t		*p_error;

	STRLOG (SLIPM_ID, 0, 0, SL_TRACE, "slip_dl_error_ack: enter");

	p_dl = (union DL_primitives *) mp->b_rptr;

	p_error = (dl_error_ack_t *) response->b_wptr;
	p_error->dl_primitive       = DL_ERROR_ACK;
	p_error->dl_error_primitive = p_dl->dl_primitive;
	p_error->dl_errno           = dl_errno;
	p_error->dl_unix_errno      = 0;

	response->b_wptr           += DL_ERROR_ACK_SIZE;
	response->b_datap->db_type = M_PCPROTO;
}

/*
 * slip_dl_unitdata_ind ()
 */

#ifndef DEBUG
static
#endif
slip_dl_unitdata_ind (q, mp)
queue_t *q;
mblk_t	*mp;
{
	dl_unitdata_ind_t	*p_dl;
	slip_t				*p_slip;

	STRLOG (SLIPM_ID, 0, 0, SL_TRACE, "slip_dl_unitdata_ind: enter");

	p_dl = (dl_unitdata_ind_t *) mp->b_wptr;
	p_dl->dl_primitive = DL_UNITDATA_IND;
	p_dl->dl_dest_addr_length = 0;
	p_dl->dl_dest_addr_offset = DL_UNITDATA_IND_SIZE;
	p_dl->dl_src_addr_length = 0;
	p_dl->dl_src_addr_offset = p_dl->dl_dest_addr_offset + p_dl->dl_dest_addr_length;

	mp->b_wptr += DL_UNITDATA_IND_SIZE;
	mp->b_datap->db_type = M_PROTO;

	/* copy packet received to the next message block */

	p_slip = (slip_t *) q->q_ptr;

	bcopy ((caddr_t)p_slip->bptr, (caddr_t)mp->b_cont->b_wptr, p_slip->inlen);

	mp->b_cont->b_wptr += p_slip->inlen;
}
