/*
 * if_ppp.c - Point-to-Point Protocol (PPP) Asynchronous driver.
 *
 * Copyright (c) 1989 Carnegie Mellon University.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that the above copyright notice and this paragraph are
 * duplicated in all such forms and that any documentation,
 * advertising materials, and other materials related to such
 * distribution and use acknowledge that the software was developed
 * by Carnegie Mellon University.  The name of the
 * University may not be used to endorse or promote products derived
 * from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 *
 * Drew D. Perkins
 * Carnegie Mellon University
 * 4910 Forbes Ave.
 * Pittsburgh, PA 15213
 * (412) 268-8576
 * ddp@andrew.cmu.edu
 *
 * Based on:
 *	@(#)if_sl.c	7.6.1.2 (Berkeley) 2/15/89
 *
 * Copyright (c) 1987 Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that the above copyright notice and this paragraph are
 * duplicated in all such forms and that any documentation,
 * advertising materials, and other materials related to such
 * distribution and use acknowledge that the software was developed
 * by the University of California, Berkeley.  The name of the
 * University may not be used to endorse or promote products derived
 * from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 *
 * Serial Line interface
 *
 * Rick Adams
 * Center for Seismic Studies
 * 1300 N 17th Street, Suite 1450
 * Arlington, Virginia 22209
 * (703)276-7900
 * rick@seismo.ARPA
 * seismo!rick
 *
 * Pounded on heavily by Chris Torek (chris@mimsy.umd.edu, umcp-cs!chris).
 * Converted to 4.3BSD Beta by Chris Torek.
 * Other changes made at Berkeley, based in part on code by Kirk Smith.
 */

/* $Header: if_sl.c,v 1.12 85/12/20 21:54:55 chris Exp $ */
/* from if_sl.c,v 1.11 84/10/04 12:54:47 rick Exp */

#define KERNEL 1
#define KERNEL_FEATURES 1
#define INET 1
#define SC_IF (*sc->sc_if)

#include "sys/param.h"
#include "sys/systm.h"
#include "sys/dir.h"
#include "sys/user.h"
#include "sys/ioctl.h"
#include "sys/tty.h"
#include "sys/proc.h"
#include "sys/file.h"
#include "sys/conf.h"
#include "sys/buf.h"
#include "sys/dk.h"
#include "sys/uio.h"
#include "sys/errno.h"
#include "sys/mbuf.h"
#include "sys/socket.h"

#include "net/if.h"
#include "net/route.h"
#if INET
#include "netinet/in.h"
#include "netinet/in_systm.h"
#include "netinet/in_var.h"
#include "netinet/ip.h"

#endif

#define VJC 1
#ifdef	VJC
#include "slcompress.h"
#endif

#include "if_ppp.h"
#include "ppp.h"

#include <kernserv/kern_server_types.h>
#include <sys/linedisc.h>
kern_server_t instance;

int pppcontrol();
netbuf_t pppgetbuf();
extern int nodev(), ttymodem(struct tty*, int);

struct ppp_softc ppp_softc[NPPP];

int pppoutput(), pppioctl();
void pppstart();

static inline void
packet_dropped(int i)
{
}

static inline int
nbq_full(struct nb_queue* nbq)
{
    if (nbq->head == 0)
	    return nbq->tail == QUEUE_SIZE - 1;
    else
	    return nbq->head - 1 == nbq->tail;
}

static inline int
nbq_empty(struct nb_queue* nbq)
{
    return nbq->head == nbq->tail;
}

static inline int
nbq_low(struct nb_queue* nbq)
{
    int size = nbq->tail - nbq->head;
    if (size < 0) size += QUEUE_SIZE;
    return (size < QUEUE_SIZE / 2);
}

static inline netbuf_t
nbq_dequeue(struct nb_queue* nbq)
{
    int s = splimp();
    if (nbq_empty(nbq))
    {
	return NULL;
    }
    else
    {
	netbuf_t ret = nbq->queue[nbq->head];
	if (nbq->head == QUEUE_SIZE - 1)
		nbq->head = 0;
	else
		nbq->head++;
	return ret;
    }
    splx(s);
}

static inline int
nbq_enqueue(struct nb_queue* nbq, netbuf_t nb)
{
    int s = splimp();
    if (nbq_full(nbq))
    {
	return -1;
    }
    else
    {
	nbq->queue[nbq->tail] = nb;
	if (nbq->tail == QUEUE_SIZE - 1)
		nbq->tail = 0;
	else
		nbq->tail++;
    }
    splx(s);
}


static inline void
nb_write_byte(netbuf_t nb, char b)
{
    int ret;
    if (nb == NULL)
	    panic("nb_write_byte");
    nb_grow_bot(nb, 1);
    if ((ret = nb_write(nb, nb_size(nb) - 1, 1, &b)) < 0)
	    printf("nb_write failed: ret = %x\n", ret);
}

static inline void
nb_write_bytes(netbuf_t nb, char* bs, int n)
{
    int ret;
    if (nb == NULL)
	    panic("nb_write_byte");
    nb_grow_bot(nb, n);
    if ((ret = nb_write(nb, nb_size(nb) - n, n, bs)) < 0)
	    printf("nb_write failed: ret = %x\n", ret);
}

netbuf_t
pppgetbuf(netif_t ifp)
{
    struct ifnet* ifn = (struct ifnet*)ifp;
    int len = MAX(ifn->if_mtu, PPP_MTU) + sizeof (struct ifnet *) +
	    sizeof (struct ppp_header) + sizeof (u_short);
    return nb_alloc(len);
}

static netbuf_t
pppgetinbuf(netif_t ifp)
{
    struct ifnet* ifn = (struct ifnet*)ifp;
    netbuf_t nb;
    int len = MAX(ifn->if_mtu, PPP_MTU) + sizeof (struct ifnet *) +
#ifdef VJC
	    128 +
#endif
	    sizeof (struct ppp_header) + sizeof (u_short);
    nb =  nb_alloc(len);
#ifdef VJC
    nb_shrink_top(nb, 128);
#endif
    return nb;
}

static void
pppalloc(void* scp)
{
    struct ppp_softc* sc = (struct ppp_softc*)scp;
    netbuf_t nb;
    
    /*printf("PPPalloc\n");*/
    while(!nbq_full(&sc->inq))
    {
	nb = pppgetinbuf((netif_t)sc->sc_if);
	nb_shrink_bot(nb, nb_size(nb));
	nbq_enqueue(&sc->inq, nb);
    }
}

inline void
possibly_alloc(struct ppp_softc* sc)
{
    if (nbq_low(&sc->inq))
	    kern_serv_callout(&instance, pppalloc, (void*)sc);
}

int
pppdetach()
{
    if (!tty_ld_remove(PPPDISC))
	    printf("Could not remove PPP discipline\n");
}

/*
 * Line specific open routine.
 * Attach the given tty to the first available ppp unit.
 */
/* ARGSUSED */
int pppopen(dev_t dv, struct tty* tp)
{
	register struct ppp_softc *sc;
	register int nppp;

	/*if (!suser())
		return (EPERM);*/
	if (tp->t_line == PPPDISC)
		return (EBUSY);

	for (nppp = 0, sc = ppp_softc; nppp < NPPP; nppp++, sc++)
		if (sc->sc_ttyp == NULL) {
			sc->sc_flags = 0;
			sc->in_nb = NULL;
			sc->sc_asyncmap = 0xffffffff;
			sc->inq.head = 0;
			sc->inq.tail = 0;
			sc->sndq.head = 0;
			sc->sndq.tail = 0;
			if (pppinit(sc) == 0) {
				SC_IF.if_flags &= ~(IFF_UP|IFF_RUNNING);
				return (ENOBUFS);
			}
			tp->t_sc = (caddr_t)sc;
			sc->sc_ttyp = tp;
			ttyflush(tp, FREAD | FWRITE);
#ifdef VJC
			sl_compress_init(&sc->comp);
#endif
			SC_IF.if_flags |= IFF_RUNNING;
			return (0);
		}

	return (ENXIO);
}

/*
 * Line specific close routine.
 * Detach the tty from the ppp unit.
 * Mimics part of ttyclose().
 */
void
pppclose(tp)
	struct tty *tp;
{
	register struct ppp_softc *sc;
	int s;

	ttywflush(tp);
	tp->t_line = 0;
	s = splimp();		/* paranoid; splnet probably ok */
	sc = (struct ppp_softc *)tp->t_sc;
	if (sc != NULL) {
		if_down(&SC_IF);
		sc->sc_ttyp = NULL;
		tp->t_sc = NULL;
		if (sc->in_nb != NULL)
		{
		    nb_free(sc->in_nb);
		    sc->in_nb = NULL;
		}
		if (sc->tin_nb != NULL)
			nb_shrink_bot(sc->tin_nb, nb_size(sc->tin_nb));
		SC_IF.if_flags &= ~(IFF_UP|IFF_RUNNING);
	}
	splx(s);
}

/*
 * Line specific (tty) read routine.
 */
pppread(tp, uio)
	register struct tty *tp;
	struct uio *uio;
{
	register struct ppp_softc *sc = (struct ppp_softc *)tp->t_sc;
	netbuf_t nb;
	register int s;
	int error;
	int temp;

	if ((tp->t_state&TS_CARR_ON)==0)
		return (EIO);
	s = splimp();
	while (sc->in_nb == NULL && tp->t_line == PPPDISC)
		if (tp->t_state&TS_NBIO) {
			splx(s);
			return (EWOULDBLOCK);
		}
		else
			sleep((caddr_t)&tp->t_rawq, TTIPRI);
	if (tp->t_line != PPPDISC) {
		splx(s);
		return (-1);
	}
	nb = sc->in_nb; /* Dequeue */
	sc->in_nb = NULL;
	splx(s);
	error = uiomove(nb_map(nb), nb_size(nb), UIO_READ, uio);
	nb_free(nb);
	return (error);
}

/*
 * Line specific (tty) write routine.
 */
pppwrite(tp, uio)
	register struct tty *tp;
	struct uio *uio;
{
	register struct ppp_softc *sc = (struct ppp_softc *)tp->t_sc;
	struct sockaddr dst;
	netbuf_t nb;
	struct ppp_header *ph1;
	int len, error;

	if (sc == NULL || tp == NULL || sc->sc_if == NULL)
		panic("pppwrite");
	
	if ((tp->t_state&TS_CARR_ON)==0)
		return (EIO);
	if (tp->t_line != PPPDISC)
		return (-1);
	if (uio->uio_resid > SC_IF.if_mtu + sizeof (struct ppp_header) ||
	    uio->uio_resid < sizeof (struct ppp_header))
		return (EMSGSIZE);
	nb = pppgetbuf((netif_t)sc->sc_if);
	len = uio->uio_resid;
	if (error = uiomove(nb_map(nb), nb_size(nb), UIO_WRITE, uio))
	{
	    nb_free(nb);
	    return error;
	}
	nb_shrink_bot(nb, nb_size(nb) - len);
	dst.sa_family = AF_UNSPEC;
	ph1 = (struct ppp_header *) &dst.sa_data;
	nb_read(nb, 0, sizeof(*ph1), (void*)ph1);
	nb_shrink_top(nb, sizeof(*ph1));
	return (pppoutput(&SC_IF, nb, &dst));
}

/*
 * Line specific (tty) select routine.
 */
pppselect(dev, rw)
	struct tty* dev;
	int rw;
{
	register struct tty *tp = dev;
	register struct ppp_softc *sc = (struct ppp_softc *)tp->t_sc;
	int s = spltty();

	if (sc == NULL)
		panic("ttselect1");
	if (tp == 0)
		panic("ttselect");
	switch (rw) {

	case FREAD:
		if (sc->in_nb != NULL || ((tp->t_state & TS_CARR_ON) == 0))
			goto win;
#if 0
		if (tp->t_rsel && tp->t_rsel->p_wchan == (caddr_t)&selwait)
			tp->t_state |= TS_RCOLL;
		else
#endif
			tp->t_rsel = u.u_procp;
		break;

	case FWRITE:
		if (!nbq_full(&sc->sndq))
			goto win;
#if 0
		if (tp->t_wsel && tp->t_wsel->p_wchan == (caddr_t)&selwait)
			tp->t_state |= TS_WCOLL;
		else
#endif
			tp->t_wsel = u.u_procp;
		break;
	}
	splx(s);
	return (0);
win:
	splx(s);
	return (1);
}

/*
 * Line specific (tty) ioctl routine.
 * Provide a way to get the ppp unit number.
 * This discipline requires that tty device drivers call
 * the line specific l_ioctl routine from their ioctl routines.
 */
/* ARGSUSED */
ppptioctl(tp, cmd, data, flag)
	struct tty *tp;
	void* data;
{
	register struct ppp_softc *sc = (struct ppp_softc *)tp->t_sc;
	int s;

	switch (cmd) {
	case TIOCGETD:
		*(int *)data = ((struct ppp_softc *)tp->t_sc)->sc_if->if_unit;
		return (0);
	case FIONREAD:
		s = splimp();		/* paranoid; splnet probably ok */
		if (sc->in_nb != NULL)
		{
		    *(int *)data = nb_size(sc->in_nb);
		    splx(s);
		}
		else
		{
		    splx(s);
		    *(int *)data = 0;
		}
		return (0);
	}
	return (-1);
}

/*
 * FCS lookup table as calculated by genfcstab.
 */
static u_short fcstab[256] = {
	0x0000,	0x1189,	0x2312,	0x329b,	0x4624,	0x57ad,	0x6536,	0x74bf,
	0x8c48,	0x9dc1,	0xaf5a,	0xbed3,	0xca6c,	0xdbe5,	0xe97e,	0xf8f7,
	0x1081,	0x0108,	0x3393,	0x221a,	0x56a5,	0x472c,	0x75b7,	0x643e,
	0x9cc9,	0x8d40,	0xbfdb,	0xae52,	0xdaed,	0xcb64,	0xf9ff,	0xe876,
	0x2102,	0x308b,	0x0210,	0x1399,	0x6726,	0x76af,	0x4434,	0x55bd,
	0xad4a,	0xbcc3,	0x8e58,	0x9fd1,	0xeb6e,	0xfae7,	0xc87c,	0xd9f5,
	0x3183,	0x200a,	0x1291,	0x0318,	0x77a7,	0x662e,	0x54b5,	0x453c,
	0xbdcb,	0xac42,	0x9ed9,	0x8f50,	0xfbef,	0xea66,	0xd8fd,	0xc974,
	0x4204,	0x538d,	0x6116,	0x709f,	0x0420,	0x15a9,	0x2732,	0x36bb,
	0xce4c,	0xdfc5,	0xed5e,	0xfcd7,	0x8868,	0x99e1,	0xab7a,	0xbaf3,
	0x5285,	0x430c,	0x7197,	0x601e,	0x14a1,	0x0528,	0x37b3,	0x263a,
	0xdecd,	0xcf44,	0xfddf,	0xec56,	0x98e9,	0x8960,	0xbbfb,	0xaa72,
	0x6306,	0x728f,	0x4014,	0x519d,	0x2522,	0x34ab,	0x0630,	0x17b9,
	0xef4e,	0xfec7,	0xcc5c,	0xddd5,	0xa96a,	0xb8e3,	0x8a78,	0x9bf1,
	0x7387,	0x620e,	0x5095,	0x411c,	0x35a3,	0x242a,	0x16b1,	0x0738,
	0xffcf,	0xee46,	0xdcdd,	0xcd54,	0xb9eb,	0xa862,	0x9af9,	0x8b70,
	0x8408,	0x9581,	0xa71a,	0xb693,	0xc22c,	0xd3a5,	0xe13e,	0xf0b7,
	0x0840,	0x19c9,	0x2b52,	0x3adb,	0x4e64,	0x5fed,	0x6d76,	0x7cff,
	0x9489,	0x8500,	0xb79b,	0xa612,	0xd2ad,	0xc324,	0xf1bf,	0xe036,
	0x18c1,	0x0948,	0x3bd3,	0x2a5a,	0x5ee5,	0x4f6c,	0x7df7,	0x6c7e,
	0xa50a,	0xb483,	0x8618,	0x9791,	0xe32e,	0xf2a7,	0xc03c,	0xd1b5,
	0x2942,	0x38cb,	0x0a50,	0x1bd9,	0x6f66,	0x7eef,	0x4c74,	0x5dfd,
	0xb58b,	0xa402,	0x9699,	0x8710,	0xf3af,	0xe226,	0xd0bd,	0xc134,
	0x39c3,	0x284a,	0x1ad1,	0x0b58,	0x7fe7,	0x6e6e,	0x5cf5,	0x4d7c,
	0xc60c,	0xd785,	0xe51e,	0xf497,	0x8028,	0x91a1,	0xa33a,	0xb2b3,
	0x4a44,	0x5bcd,	0x6956,	0x78df,	0x0c60,	0x1de9,	0x2f72,	0x3efb,
	0xd68d,	0xc704,	0xf59f,	0xe416,	0x90a9,	0x8120,	0xb3bb,	0xa232,
	0x5ac5,	0x4b4c,	0x79d7,	0x685e,	0x1ce1,	0x0d68,	0x3ff3,	0x2e7a,
	0xe70e,	0xf687,	0xc41c,	0xd595,	0xa12a,	0xb0a3,	0x8238,	0x93b1,
	0x6b46,	0x7acf,	0x4854,	0x59dd,	0x2d62,	0x3ceb,	0x0e70,	0x1ff9,
	0xf78f,	0xe606,	0xd49d,	0xc514,	0xb1ab,	0xa022,	0x92b9,	0x8330,
	0x7bc7,	0x6a4e,	0x58d5,	0x495c,	0x3de3,	0x2c6a,	0x1ef1,	0x0f78
};

/*
 * Calculate a new FCS given the current FCS and the new data.
 */
u_short pppfcs(fcs, cp, len)
    register u_short fcs;
    register u_char *cp;
    register int len;
{
    while (len--)
	fcs = PPP_FCS(fcs, *cp++);
    return (fcs);
}

/*
 * Queue a packet.  Start transmission if not active.
 * Packet is placed in normally Information field of PPP frame.
 */
pppoutput(ifp, nb, dst)
	struct ifnet *ifp;
	netbuf_t nb;
	struct sockaddr *dst;
{
	register struct ppp_softc *sc = &ppp_softc[ifp->if_unit];
	struct ppp_header *ph;
	u_short protocol, fcs;
	u_char address, control, *cp;
	int s, error, compac, compprot;

	if (sc == NULL)
		panic("pppoutput");

	if (sc->sc_ttyp == NULL ||
	    (ifp->if_flags & IFF_RUNNING) != IFF_RUNNING) {
		error = ENETDOWN;	/* sort of */
		goto bad;
	}
	if ((sc->sc_ttyp->t_state & TS_CARR_ON) == 0) {
		error = EHOSTUNREACH;
		goto bad;
	}
	
	/*
	 * Compute PPP header.
	 */
	address = PPP_ALLSTATIONS;
	control = PPP_UI;
	switch (dst->sa_family) {
#ifdef INET
	    case AF_INET:
		protocol = PPP_IP;
		break;
#endif
#ifdef NS
	    case AF_NS:
		protocol = PPP_XNS;
		break;
#endif
	    case AF_UNSPEC:
		ph = (struct ppp_header *) dst->sa_data;
		address = ph->ph_address;
		control = ph->ph_control;
		protocol = ntohs(ph->ph_protocol);
		break;
	    default:
		printf("ppp%d: af%d not supported\n", ifp->if_unit,
			dst->sa_family);
		error = EAFNOSUPPORT;
		goto bad;
	}
#ifdef	VJC
	if((protocol == PPP_IP) && (ifp->if_flags & IFF_VJC)) {
		register struct ip *ip;
		ip = (struct ip*)nb_map(nb);
		if(ip->ip_p == IPPROTO_TCP) {
			u_char type = sl_compress_tcp(nb, ip, &sc->comp, 1);
			switch (type) {
		
				case TYPE_UNCOMPRESSED_TCP :
					protocol = PPP_VJC_UNCOMP;
					break;
				case TYPE_COMPRESSED_TCP :
					protocol = PPP_VJC_COMP;
					break;	
				default :;
			}
		}
	}
#endif
	compac = SC_IF.if_flags & IFF_COMPAC &&
		 address == PPP_ALLSTATIONS &&
		 control == PPP_UI &&
		 protocol != PPP_ALLSTATIONS;
	compprot = SC_IF.if_flags & IFF_COMPPROT &&
		 protocol < 0x100;

	/*
	 * Add PPP header.  If no space in first mbuf, allocate another.
	 */
	nb_grow_top(nb, (compac ? 0 : 2) + (compprot ? 1 : 2));

	cp = (u_char*)nb_map(nb);
	if (!compac) {
	    *cp++ = address;
	    *cp++ = control;
	}
	if (!compprot) {
	    *cp++ = protocol >> 8;
	}
	*cp++ = protocol & 0xff;

	/*
	 * Add PPP trailer.  Compute one's complement of FCS over frame
	 * and attach to mbuf chain least significant byte first.
	 */
	fcs = PPP_INITFCS;
	fcs = pppfcs(fcs, (char*)nb_map(nb), nb_size(nb));
	fcs ^= 0xffff;

	cp = (u_char*)nb_map(nb) + nb_size(nb);
	nb_grow_bot(nb, sizeof(short));
	*cp++ = fcs & 0xff;
	*cp++ = fcs >> 8;

	s = splimp();
	if (nbq_enqueue(&sc->sndq, nb) < 0) {
		packet_dropped(-1);
		SC_IF.if_oerrors++;
		splx(s);
		error = ENOBUFS;
		goto bad;
	}
	if (sc->sc_ttyp->t_outq.c_cc == 0) {
		splx(s);
		pppstart(sc->sc_ttyp);
	} else
		splx(s);
	return (0);

bad:
	nb_free(nb);
	return (error);
}

/*
 * Start output on interface.  Get another datagram
 * to send from the interface queue and map it to
 * the interface before starting output.
 */
void
pppstart(tp)
	register struct tty *tp;
{
	register struct ppp_softc *sc = (struct ppp_softc *)tp->t_sc;
	register int len;
	register u_char *start, *stop, *cp;
	int n, s;
	extern int cfreecount;
	netbuf_t nb;

	for (;;) {
		/*
		 * If there is more in the output queue, just send it now.
		 * We are being called in lieu of ttstart and must do what
		 * it would.
		 */
		if (tp->t_outq.c_cc > 0)
			ttstart(tp);
		if (tp->t_outq.c_cc > PPP_HIWAT)
			return;

		/*
		 * This happens briefly when the line shuts down.
		 */
		if (sc == NULL)
			return;

		/*
		 * If system is getting low on clists
		 * and we have something running already, stop here.
		 */
		if (cfreecount < CLISTRESERVE +
		    SC_IF.if_mtu + sizeof (struct ppp_header) &&
		    tp->t_outq.c_cc)
			return;

		/*
		 * Get a packet and send it to the interface.
		 */
		s = splimp();
		nb = nbq_dequeue(&sc->sndq);
		splx(s);
		if (nb == NULL)
			return;

		/*
		 * The extra PPP_FLAG will start up a new packet, and thus
		 * will flush any accumulated garbage.  We do this whenever
		 * the line may have been idle for some time.
		 */
		if (tp->t_outq.c_cc == 0)
			(void) putc(PPP_FLAG, &tp->t_outq);

		start = (u_char*)nb_map(nb);
		len = nb_size(nb);
		stop = start + len;
		while (len > 0) {
		    /*
		     * Find out how many bytes in the string we can
		     * handle without doing something special.
		     */
		    for (cp = start; cp < stop; cp++)
			    if ((*cp == PPP_FLAG) ||
				(*cp == PPP_ESCAPE) ||
				(*cp < 0x20 &&
				 (sc->sc_asyncmap & (1 << *cp))))
				    break;
		    n = cp - start;
		    if (n) {
			/*
			 * Put n characters at once
			 * into the tty output queue.
			 */
			if (b_to_q((char *)start, n,
				   &tp->t_outq))
				break;
			len -= n;
			start = cp;
		    }
		    
		    /*
		     * If there are characters left in the mbuf,
		     * the first one must be special..
		     * Put it out in a different form.
		     */
		    if (len) {
			if (putc(PPP_ESCAPE, &tp->t_outq))
				break;
			if (putc(*start ^ PPP_TRANS,
				 &tp->t_outq)) {
			    (void) unputc(&tp->t_outq);
			    break;
			}
			start++;
			len--;
		    }
		}
		
		nb_free(nb);

		if (putc(PPP_FLAG, &tp->t_outq)) {
			/*
			 * Not enough room.  Remove a char to make room
			 * and end the packet normally.
			 * If you get many collisions (more than one or two
			 * a day) you probably do not have enough clists
			 * and you should increase "nclist" in param.c.
			 */
			(void) unputc(&tp->t_outq);
			(void) putc(PPP_FLAG, &tp->t_outq);
			SC_IF.if_collisions++;
		} else
			SC_IF.if_opackets++;
	}

	return;
}

/*
 * Allocate enough mbuf to handle current MTU.
 */
pppinit(sc)
	register struct ppp_softc *sc;
{
    netbuf_t nb = pppgetinbuf((netif_t)sc->sc_if);
    netbuf_t t;
    int s;
    
    if (sc == NULL)
	    panic("pppinit");
    s = splimp();
    /* Throw the old ones */
    while((t = nbq_dequeue(&sc->inq)) != NULL)
	    nb_free(t);
    pppalloc((void*) sc);
    
    /* Copy old terminal in buffer */
    if (sc->tin_nb != NULL && nb_size(sc->tin_nb) <= nb_size(nb))
    {
	nb_write(nb, 0, nb_size(sc->tin_nb), nb_map(sc->tin_nb));
	nb_shrink_bot(nb, nb_size(nb) - nb_size(sc->tin_nb));
    }
    else
	    nb_shrink_bot(nb, nb_size(nb));
    
    if (sc->tin_nb != NULL)
	    nb_free(sc->tin_nb);
    sc->tin_nb = nb;
    splx(s);
}

/*
 * tty interface receiver interrupt.
 */
void
pppinput(c, tp)
	int c;
	register struct tty *tp;
{
	register struct ppp_softc *sc;
	u_char* cp;
	int top_grown = 0;
	struct ppp_header *ph;
	int s;

	if (tp == NULL)
		panic("pppinput");
	tk_nin++;
	sc = (struct ppp_softc *)tp->t_sc;
	if (sc == NULL)
		return;
	
	if (sc->tin_nb == NULL)
	{
	    sc->tin_nb = nbq_dequeue(&sc->inq);
	    if (sc->tin_nb != NULL)
		    nb_shrink_bot(sc->tin_nb, nb_size(sc->tin_nb));
	    possibly_alloc(sc);
	}
	
	if (sc->tin_nb == NULL)
	{
	    /* Out of input buffers */
	    if (c == PPP_FLAG)
	    {
		packet_dropped(1);
		sc->sc_flags &= ~SC_FLUSH;
		SC_IF.if_ierrors++;
	    }
	    else
		    sc->sc_flags |= SC_FLUSH;
	    return;
	}

	c &= 0xff;
	if (c == PPP_FLAG) {
		sc->sc_flags &= ~SC_FLUSH;

		if (nb_size(sc->tin_nb) >= 2)
			nb_shrink_bot(sc->tin_nb, 2);

		if (nb_size(sc->tin_nb) < sizeof (struct ppp_header)) {
			if (nb_size(sc->tin_nb))
			{
			    packet_dropped(2);
			    nb_shrink_bot(sc->tin_nb, nb_size(sc->tin_nb));
			    SC_IF.if_ierrors++;
			}
			return;
		}
		if (sc->sc_fcs != PPP_GOODFCS) {
			    packet_dropped(3);
			SC_IF.if_ierrors++;
			nb_shrink_bot(sc->tin_nb, nb_size(sc->tin_nb));
			return;
		}

		SC_IF.if_ipackets++;
		ph = (struct ppp_header *)nb_map(sc->tin_nb);
		
#ifdef VJC
		switch (ntohs(ph->ph_protocol)) 
		{
		case PPP_VJC_COMP :
			if(SC_IF.if_flags & IFF_VJC)
			{
			    int xlen;
			    xlen = nb_size(sc->tin_nb) - sizeof(struct ppp_header);
			    cp = (u_char*)nb_map(sc->tin_nb) + sizeof(struct ppp_header);
			    xlen = sl_uncompress_tcp(&cp,xlen, TYPE_COMPRESSED_TCP, &sc->comp);
			    if(!xlen)
			    {
				printf("ppp: sl_uncompress failed on type Compressed");
				goto reject;
			    }
			    top_grown = (u_char*)nb_map(sc->tin_nb) - cp + sizeof(struct ppp_header);
			    nb_grow_top(sc->tin_nb, top_grown);
			    ph = (struct ppp_header *)nb_map(sc->tin_nb);
			    ((struct ppp_header *) nb_map(sc->tin_nb))->ph_protocol = htons(PPP_IP);
			    break;
			}
			
		    case PPP_VJC_UNCOMP :
			    if(SC_IF.if_flags & IFF_VJC)
			    {
				cp = (unsigned char *) nb_map(sc->tin_nb) + sizeof(struct ppp_header);
				if(sl_uncompress_tcp(&cp, 1, TYPE_UNCOMPRESSED_TCP, &sc->comp)) {
				    ph->ph_protocol = htons(PPP_IP);
				    break;
				}
				printf("ppp: sl_uncompress failed on type Uncompresed\n");
			    reject:;
				nb_shrink_top(sc->tin_nb, top_grown);
				nb_shrink_bot(sc->tin_nb, nb_size(sc->tin_nb));
			    packet_dropped(4);
				SC_IF.if_ierrors++;
				return;				
			    }
		    default:;
		    }
#endif
			
		switch (ntohs(ph->ph_protocol)) {
#ifdef INET
		    case PPP_IP:
			nb_shrink_top(sc->tin_nb, sizeof(struct ppp_header));
			inet_queue(sc->sc_if, sc->tin_nb);
			sc->tin_nb = nbq_dequeue(&sc->inq);
			if (sc->tin_nb != NULL)
				nb_shrink_bot(sc->tin_nb, nb_size(sc->tin_nb));
			possibly_alloc(sc);
			break;
#endif
		    default:
			ttwakeup(tp);
			s = splimp();
			/* Is input 'queue' full? */
			if (sc->in_nb != NULL) {
			    /* Drop the packet */
			    packet_dropped(5);
			    SC_IF.if_ierrors++;
#ifdef VJC
			    nb_shrink_top(sc->tin_nb, top_grown);
#endif
			    nb_shrink_bot(sc->tin_nb, nb_size(sc->tin_nb));
			} else
			{
			    sc->in_nb =  sc->tin_nb;
			    sc->tin_nb = nbq_dequeue(&sc->inq);
			    if (sc->tin_nb != NULL)
				    nb_shrink_bot(sc->tin_nb, nb_size(sc->tin_nb));
			    possibly_alloc(sc);
			}

			splx(s);
		}
		return;
	}
	else if (sc->sc_flags & SC_FLUSH)
		return;
	else if (c == PPP_ESCAPE) {
		sc->sc_flags |= SC_ESCAPED;
		return;
	}

	if (sc->sc_flags & SC_ESCAPED) {
		sc->sc_flags &= ~SC_ESCAPED;
		c ^= PPP_TRANS;
	}

	/*
	 * Initialize buffer on first octet received.
	 * First octet could be address or protocol (when compressing
	 * address/control).
	 * Second octet is control.
	 * Third octet is first or second (when compressing protocol)
	 * octet of protocol.
	 * Fourth octet is second octet of protocol.
	 */
	if (nb_size(sc->tin_nb) == 0) {
		struct ifnet* ptr = &SC_IF;
		sc->sc_fcs = PPP_INITFCS;
		if (c != PPP_ALLSTATIONS) {
			if (SC_IF.if_flags & IFF_COMPAC) {
				nb_write_byte(sc->tin_nb, PPP_ALLSTATIONS);
				nb_write_byte(sc->tin_nb, PPP_UI);
			}
			else {
				printf("<5(%x)>", c);
				SC_IF.if_ierrors++;
				sc->sc_flags |= SC_FLUSH;
				return;
			}
		}
	}
	if (nb_size(sc->tin_nb) == 1 && c != PPP_UI) {
		    packet_dropped(6);
		SC_IF.if_ierrors++;
		sc->sc_flags |= SC_FLUSH;
		return;
	}
	if (nb_size(sc->tin_nb) == 2 && (c & 1) == 1) {
		if (SC_IF.if_flags & IFF_COMPPROT) {
			nb_write_byte(sc->tin_nb, 0);
		}
		else {
		    packet_dropped(7);
			SC_IF.if_ierrors++;
			sc->sc_flags |= SC_FLUSH;
			return;
		}
	}
	if (nb_size(sc->tin_nb) == 3 && (c & 1) == 0) {
		    packet_dropped(8);
		SC_IF.if_ierrors++;
		sc->sc_flags |= SC_FLUSH;
		return;
	}

	if (nb_size(sc->tin_nb) >= MAX(SC_IF.if_mtu, PPP_MTU) +
	    sizeof (struct ppp_header) + sizeof (u_short)) {
		    packet_dropped(9);
		SC_IF.if_ierrors++;
		sc->sc_flags |= SC_FLUSH;
		return;
	}
	
	nb_write_byte(sc->tin_nb, c);
	sc->sc_fcs = PPP_FCS(sc->sc_fcs, c);
}

int
pppcontrol(ifp, cmd, data)
netif_t ifp;
const char* cmd;
void* data;
{
    if (!strcmp(cmd, "unix-ioctl"))
    {
	if_ioctl_t* ctl = (if_ioctl_t*)data;
	return pppioctl((struct ifnet*)ifp,
			ctl->ioctl_command,
			ctl->ioctl_data);
    }
    else if (!strcmp(cmd, "setaddr"))
    {
	struct sockaddr_in *sin = (struct sockaddr_in *)data;
	if (sin->sin_family != AF_INET)
		return EAFNOSUPPORT;
	if_flags_set(ifp, if_flags(ifp) | IFF_UP);
	return 0;
    }
    else if (!strcmp(cmd, "setflags"))
    {
	return 0;
    }
    else
    {
	printf("Invalid ppp control %s\n", cmd);
	return EINVAL;
    }
}

/*
 * Process an ioctl request.
 */
pppioctl(ifp, cmd, data)
	register struct ifnet *ifp;
	int cmd;
	caddr_t data;
{
	register struct ppp_softc *sc = &ppp_softc[ifp->if_unit];
	register struct ifaddr *ifa = (struct ifaddr *)data;
	register struct ifreq *ifr = (struct ifreq *)data;
	int s = splimp(), error = 0;

	switch (cmd) {
	case SIOCSIFFLAGS:
		if ((ifp->if_flags & IFF_RUNNING) == 0)
			ifp->if_flags &= ~IFF_UP;
		break;

	case SIOCSIFADDR:
		if (ifa->ifa_addr.sa_family != AF_INET)
			error = EAFNOSUPPORT;
		break;

	case SIOCSIFDSTADDR:
		if (ifa->ifa_addr.sa_family != AF_INET)
			error = EAFNOSUPPORT;
		break;

	case SIOCSIFMTU:
		if (!suser())
			return (EPERM);
		SC_IF.if_mtu = ifr->ifr_mtu;
		if (pppinit(sc) == 0)
			error = ENOBUFS;
		break;

	case SIOCGIFMTU:
		ifr->ifr_mtu = SC_IF.if_mtu;
		break;

	case SIOCSIFASYNCMAP:
		if (!suser())
			return (EPERM);
		sc->sc_asyncmap = ifr->ifr_asyncmap;
		break;

	case SIOCGIFASYNCMAP:
		ifr->ifr_asyncmap = sc->sc_asyncmap;
		break;

	case SIOCSIFVJCOMP:	/* enable or disable VJ compression */
#ifdef VJC
		if( ifr->ifr_data )
			ifp->if_flags |= IFF_VJC;
		else
			ifp->if_flags &= ~IFF_VJC;
		break;
#else
		error = EINVAL;
		break;
#endif

	default:
		error = EINVAL;
	}
	splx(s);
	return (error);
}


void
ppprend(char *cp, int n, struct tty *tp)
{
    register struct ppp_softc *sc;
    int s;
    
    if (tp == NULL)
	    panic("pppinput");
    sc = (struct ppp_softc *)tp->t_sc;
    if (sc == NULL)
	    return;

    while (n)
    {
	if (sc->tin_nb == NULL)
	{
	    sc->tin_nb = nbq_dequeue(&sc->inq);
	    possibly_alloc(sc);

	    if (sc->tin_nb == NULL)
	    {
		/* Out of input buffers */
		sc->sc_flags |= SC_FLUSH;
		SC_IF.if_ierrors++;
		return;
	    }
	    else
		    nb_shrink_bot(sc->tin_nb, nb_size(sc->tin_nb));
	}
    
	if (sc->sc_flags & SC_FLUSH)
	{
	    do
	    {
		if (*(cp++) == PPP_FLAG)
		{
		    pppinput(PPP_FLAG, tp);
		    --n;
		    break;
		}
	    }
	    while(--n);
	}
	else if (nb_size(sc->tin_nb) > 3 &&
		 nb_size(sc->tin_nb) + n < MAX(SC_IF.if_mtu, PPP_MTU) +
		 sizeof (struct ppp_header) + sizeof (u_short) &&
		 *cp != PPP_FLAG &&
		 *cp != PPP_ESCAPE)
	{
	    char* cp1 = cp;
	    if (sc->sc_flags & SC_ESCAPED)
	    {
		sc->sc_flags &= ~SC_ESCAPED;
		*cp ^= PPP_TRANS;
	    }
	    
	    do
	    {
		sc->sc_fcs = PPP_FCS(sc->sc_fcs, *(cp++));
	    } while(--n && *cp != PPP_FLAG && *cp != PPP_ESCAPE);
	    
	    tk_nin += cp - cp1;
	    nb_write_bytes(sc->tin_nb, cp1, cp - cp1);
	}
	else
	{
	    --n;
	    pppinput(*(cp++), tp);
	}
    }
}

extern int bla(dev_t dev, struct tty* tp);
int install_ld()
{
    return tty_ld_install(PPPDISC, NORMAL_LDISC, pppopen,
			  pppclose, pppread, pppwrite, ppptioctl,
			  pppinput, ppprend, pppstart, ttymodem,
			  pppselect);
}

/*
 * Called from boot code to establish ppp interfaces.
 */
pppattach()
{
	register struct ppp_softc *sc;
	register int i = 0;
	extern int install_ld();
	extern int ipforwarding;
	extern int ipsendredirects;
	
	if (install_ld() < 0)
	{
	    printf("Could not install line discipline");
	    return 0;
	}

	for (sc = ppp_softc; i < NPPP; sc++, i++) {
		sc->sc_if = (struct ifnet*)
			if_attach(NULL, NULL, pppoutput, 
				  pppgetbuf, pppcontrol, "ppp", i, "Serial line PPP", 
				  PPP_MTU, IFF_POINTOPOINT, NETIFCLASS_VIRTUAL, NULL);
		/* Fucked up docs.  They don't say that NETIFCLASS_REAL are
		not visible to ioctl's! */
	}
	ipforwarding = 1;
	ipsendredirects = 1;
	return 1;
}

