/* 
** Copyright 1986, 1987, 1988, 1989, 1990, 1991 by the Condor Design Team
** 
** Permission to use, copy, modify, and distribute this software and its
** documentation for any purpose and without fee is hereby granted,
** provided that the above copyright notice appear in all copies and that
** both that copyright notice and this permission notice appear in
** supporting documentation, and that the names of the University of
** Wisconsin and the Condor Design Team not be used in advertising or
** publicity pertaining to distribution of the software without specific,
** written prior permission.  The University of Wisconsin and the Condor
** Design Team make no representations about the suitability of this
** software for any purpose.  It is provided "as is" without express
** or implied warranty.
** 
** THE UNIVERSITY OF WISCONSIN AND THE CONDOR DESIGN TEAM DISCLAIM ALL
** WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES
** OF MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE UNIVERSITY OF
** WISCONSIN OR THE CONDOR DESIGN TEAM BE LIABLE FOR ANY SPECIAL, INDIRECT
** OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS
** OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE
** OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE
** OR PERFORMANCE OF THIS SOFTWARE.
** 
** Author:  Michael J. Litzkow
** 	         University of Wisconsin, Computer Sciences Dept.
** 
*/ 

#include <stdio.h>
#include <sys/types.h>
#include <sys/file.h>
#include <sys/param.h>
#include "condor_sys.h"
#include "ckpt_file.h"
#include "debug.h"

double	get_time();


/*
** Transfer a local file to a remote machine.
*/
send_file( local, remote, mode )
char	*local;
char	*remote;
int		mode;
{

	int		scm;
	int		remote_fd, local_fd;
	int		answer;
	char	buf[4096];
	int		nbytes;
	int		count = 0;
	double	start, finish;

	dprintf( D_ALWAYS, "Entering send_file( %s, %s, 0%o )\n",local,remote,mode);

	remote_fd = try_via_nfs( remote, O_WRONLY|O_CREAT|O_TRUNC, mode );
	if( remote_fd < 0 ) {
		dprintf( D_ALWAYS, "Transferring file via PSEUDO_send_file\n" );
		answer =  REMOTE_syscall( PSEUDO_send_file, local, remote, mode );
		return answer;
	}


	dprintf( D_ALWAYS, "Transferring file via NFS\n" );
	scm = SetSyscalls( SYS_LOCAL | SYS_RECORDED );

	start = get_time();


	if( (local_fd=open(local,O_RDONLY,0)) < 0 ) {
		dprintf( D_FULLDEBUG, "Failed to open \"%s\" locally, errno = %d\n",
														local, errno);
		(void)SetSyscalls( scm );
		return -1;
	}

	while( (nbytes=read(local_fd,buf,sizeof(buf))) > 0 ) {
		if( write(remote_fd,buf,nbytes) == nbytes ) {
			count += nbytes;
		} else {
			dprintf( D_ALWAYS, "Can't write fd %d, errno = %d\n",
														remote_fd, errno );
			(void)SetSyscalls( scm );
			return -1;
		}
	}

	(void)close( local_fd );
	(void)close( remote_fd );
	finish = get_time();

	if( nbytes < 0 ) {
		dprintf( D_ALWAYS, "Can't read fd %d, errno = %d\n", local_fd, errno );
	}
	if( nbytes == 0 ) {
		dprintf( D_ALWAYS,"Send_file() transferred %d bytes, %f bytes/second\n",
										count, count / (finish - start) );
	}

	(void)SetSyscalls( scm );
	return nbytes;
}

/*
** Transfer a remote file to the local machine.
*/
get_file( remote, local, mode )
char	*remote;
char	*local;
int		mode;
{
	int		scm;
	int		remote_fd, local_fd;
	int		answer;
	char	buf[4096];
	int		nbytes;
	int		count = 0;
	double	start, finish;
	char	external[ MAXPATHLEN ];

	dprintf( D_ALWAYS, "Entering get_file( %s, %s, 0%o )\n", remote,local,mode);

	remote_fd = try_via_nfs( remote, O_RDONLY, 0 );
	if( remote_fd < 0 ) {
		dprintf( D_ALWAYS, "Transferring file via PSEUDO_get_file\n" );
		answer =  REMOTE_syscall( PSEUDO_get_file, remote, local, mode );
		return answer;
	}

	dprintf( D_ALWAYS, "Transferring file via NFS\n" );
	scm = SetSyscalls( SYS_LOCAL | SYS_RECORDED );

	start = get_time();

	if( (local_fd=open(local,O_WRONLY|O_CREAT|O_TRUNC,mode)) < 0 ) {
		dprintf( D_FULLDEBUG, "Failed to open \"%s\" locally, errno = %d\n",
														local, errno);
		(void)SetSyscalls( scm );
		return -1;
	}

	while( (nbytes=read(remote_fd,buf,sizeof(buf))) > 0 ) {
		if( write(local_fd,buf,nbytes) == nbytes ) {
			count += nbytes;
		} else {
			dprintf( D_ALWAYS, "Can't write fd %d, errno = %d\n",
														local_fd, errno );
			(void)SetSyscalls( scm );
			return -1;
		}
	}

	(void)close( local_fd );
	(void)close( remote_fd );

	finish = get_time();

	if( nbytes < 0 ) {
		dprintf( D_ALWAYS, "Can't read fd %d, errno = %d\n", remote_fd, errno );
	}
	if( nbytes == 0 ) {
		dprintf( D_ALWAYS,"Get_file() transferred %d bytes, %f bytes/second\n",
										count, count / (finish - start) );
	}


	(void)SetSyscalls( scm );
	return nbytes;
}

#include <sys/time.h>
double
get_time()
{
	struct timeval	tv;
	int				scm;

	scm = SetSyscalls( SYS_LOCAL | SYS_RECORDED );

	if( gettimeofday( &tv, 0 ) < 0 ) {
		perror( "gettimeofday()" );
		exit( 1 );
	}

	(void)SetSyscalls( scm );
	return (double)tv.tv_sec + (double)tv.tv_usec / 1000000.0;
}

/*
** Here we try to open the file via NFS, but avoid an obvious cases where
** NFS would be a loss.  If the file is physically located on the submitting
** machine, block transferring the file via the shadow will be better than
** NFS.  This will be even more true if we are writing the file. 
**
** On the other hand, if the file is on the submitting machine, and we're
** that machine, then what appears to be NFS access would really be local.
** In that case we're better off than using the shadow, so do it locally,
** which is still called "NFS".
*/
try_via_nfs( path, flags, mode )
char	*path;
int		flags;
int		mode;
{
	int		scm;
	char	external[ MAXPATHLEN ];
	static char	submitting_host[ MAXPATHLEN ];
	static char	this_host[ MAXPATHLEN ];
	int		submitting_host_len;
	int		answer = 0;
	char	*file_host, *find_physical_host();

	dprintf( D_FULLDEBUG, "Entering try_via_nfs\n" );

	scm = SetSyscalls( SYS_REMOTE | SYS_RECORDED );

	if( submitting_host[0] == '\0' ) {

		gethostname( submitting_host, sizeof(submitting_host) );
		dprintf( D_FULLDEBUG, "submitting_host = \"%s\"\n", submitting_host );

		(void)SetSyscalls( SYS_LOCAL | SYS_RECORDED );
		gethostname( this_host, sizeof(this_host) );
		dprintf( D_FULLDEBUG, "this_host = \"%s\"\n", this_host );

		(void)SetSyscalls( SYS_REMOTE | SYS_RECORDED );
		submitting_host_len = strlen( submitting_host );
	}

	if( (file_host = find_physical_host(path,flags)) == NULL ) {
		dprintf( D_ALWAYS, "Can't find physical location of file \"%s\"\n",
																		path );
		(void)SetSyscalls( scm );
		return -1;
	}
	dprintf( D_FULLDEBUG, "at point 2\n" );
	dprintf( D_FULLDEBUG, "this_host = \"%s\"\n", this_host );
	dprintf( D_FULLDEBUG, "submitting_host = \"%s\"\n", submitting_host );
	dprintf( D_FULLDEBUG, "file_host = \"%s\"\n", file_host );

	if( strcmp(submitting_host,file_host) == 0 ) {
		dprintf(D_FULLDEBUG,"File is physically located on submitting host\n" );
		if( strcmp(submitting_host,this_host) == 0 ) {
			dprintf( D_FULLDEBUG, "But we are the submitting host\n" );
			answer = nfs_open( path, flags, mode );
			dprintf( D_FULLDEBUG, "answer=%d\n", answer );
		} else {
			answer = -1;	/* Avoid NFS */
		}
	} else {
		dprintf( D_FULLDEBUG,
			 "File is NOT physically located on submitting host\n" );
		answer = nfs_open( path, flags, mode );
		dprintf( D_FULLDEBUG, "answer=%d\n", answer );
	}
	if( answer >= 0 ) {
		answer = MarkFileOpen( answer, path, flags, FI_NFS );
		dprintf( D_FULLDEBUG, "now answer=%d\n", answer );
	}

	(void)SetSyscalls( scm );
	return answer;
}

char *
find_physical_host( path, flags )
char	*path;
int		flags;
{
	static char answer[ MAXPATHLEN ];
	char		dir[ MAXPATHLEN ];
	char		*ptr, *index(), *rindex();

		/* Try to find the pathname as given */
	if( extern_name(path,answer,sizeof(answer)) >= 0 ) {
		if( ptr=index(answer,':') ) {
			*ptr = '\0';
		}
		return answer;
	}

	if( !(flags & O_CREAT) ) {
		return NULL;
	}

		/* He's trying to creat the file, look for the parent directory */
	strcpy( dir, path );
    if( ptr=rindex(dir,'/') ) {
		if( ptr == dir ) {
			strcpy( dir, "/" );
		} else {
			*ptr = '\0';
		}
	} else {
		REMOTE_syscall( PSEUDO_getwd, dir );
	}

	if( extern_name(dir,answer,sizeof(answer)) >= 0 ) {
		if( ptr=index(answer,':') ) {
			*ptr = '\0';
		}
		return answer;
	} else {
		return NULL;
	}
}
