/* 
** Copyright 1992 by the Condor Design Team
** 
** Permission to use, copy, modify, and distribute this software and its
** documentation for any purpose and without fee is hereby granted,
** provided that the above copyright notice appear in all copies and that
** both that copyright notice and this permission notice appear in
** supporting documentation, and that the names of the University of
** Wisconsin and the Condor Design Team not be used in advertising or
** publicity pertaining to distribution of the software without specific,
** written prior permission.  The University of Wisconsin and the Condor
** Design Team make no representations about the suitability of this
** software for any purpose.  It is provided "as is" without express
** or implied warranty.
** 
** THE UNIVERSITY OF WISCONSIN AND THE CONDOR DESIGN TEAM DISCLAIM ALL
** WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES
** OF MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE UNIVERSITY OF
** WISCONSIN OR THE CONDOR DESIGN TEAM BE LIABLE FOR ANY SPECIAL, INDIRECT
** OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS
** OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE
** OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE
** OR PERFORMANCE OF THIS SOFTWARE.
** 
** Author:  Michael J. Litzkow,
** 	         University of Wisconsin, Computer Sciences Dept.
** 
*/ 

/*
** This program is designed to test the condor checkpointing mechanism's
** ability to correctly save and restore the stack for programs where
** the SP doesn't happen to point into the traditional "stack" area
** at the time of the checkpoint.  Programs written in languages that
** support co-routines or threads such as Modula 2 will often exhibit
** this kind of behavior.  In most versions of UNIX, this would not
** cause any problem, but in SunOS the core file does not directly
** specify the size of the stack segment at the time of the core dump.
** In the Sun case the checkpointing mechanism resorts "tricks" to get
** this information, so this test is relevant on those systems.  (Of
** course it should work on all the systems.
**
** Two checkpoint/restarts are attempted.  Known data is stored on the
** stack (in the real stack area), then the SP is moved into the
** "data" area before the first checkpoint.  After the first checkpoint
** known data is stored on the new stack and another checkpoint/restart
** is executed.  After this, the contents of the known data areas on
** both stacks are checked for consistency.
*/


#include <stdio.h>
#include <signal.h>

#define LARGE_BUF_SIZE 40960
#define SMALL_BUF_SIZE 100

int		N;

#define STACKSIZE (0x4000)
char	TmpStack[ STACKSIZE ];

main( argc, argv )
int		argc;
char	*argv[];
{
	setbuf( stdout, NULL );
	foo( 1, 10 );
	printf( "Normal End Of Job\n" );
	exit( 0 );
}

foo( n, lim )
int		n;
int		lim;
{
	int		buf[LARGE_BUF_SIZE];
	int		i;

	N = n;
	fill_buf( buf, LARGE_BUF_SIZE, n );

	if( n < lim ) {
		foo( n + 1, lim );
	} else {
		move_sp_and_ckpt();
	}

	if( N != n ) {
		fprintf( stderr, "Detected error in n's\n" );
		fprintf( stderr, "n = %d, N = %d\n", n, N );
		exit( 1 );
	}
	printf( "Foo Level %d: ", n );
	printf( "N ok, " );

	check_buf( buf, LARGE_BUF_SIZE, n );
	printf( "Buffer ok\n" );

	N -= 1;
}

move_sp_and_ckpt()
{
	struct sigvec	vec;
	void		usr1_handler();
	struct sigstack	stk;


		/* Change handler and stack for USR1 */
	vec.sv_handler = usr1_handler;
	vec.sv_mask = 0;
	vec.sv_flags = SV_ONSTACK;
		
	if( sigvec(SIGUSR1,&vec,(struct sigvec *)0) < 0 ) {
		perror( "sigvec" );
		exit( 1 );
	}

		/* Change to our signal stack */
	stk.ss_onstack = 0;
	stk.ss_sp = &TmpStack[ STACKSIZE ];

	if( sigstack( &stk, (struct sigstack *)0 ) < 0 ) {
		perror( "sigstack" );
		exit( 1 );
	}

		/* Invoke the handler */
	kill( getpid(), SIGUSR1 );
}

	
void
usr1_handler()
{
	printf( "About to checkpoint\n" );
	ckpt();
	printf( "Returned from checkpoint\n" );
	bar( 1, 5 );
}

bar( n, lim )
int		n;
int		lim;
{
	int		buf[SMALL_BUF_SIZE];
	int		i;

	fill_buf( buf, SMALL_BUF_SIZE, n );

	if( n < lim ) {
		bar( n + 1, lim );
	} else {
		printf( "About to checkpoint\n" );
		ckpt();
		printf( "Returned from checkpoint\n" );
	}

	printf( "Bar Level %d: ", n );
	check_buf( buf, SMALL_BUF_SIZE, n );
	printf( "Buffer ok\n" );
}

fill_buf( buf, len, val )
int		buf[];
int		len;
int		val;
{
	int		i;

	for( i=0; i<len; i++ ) {
		buf[i] = val;
	}
}

check_buf( buf, len, val )
int		buf[];
int		len;
int		val;
{
	int		i;

	for( i=0; i<len; i++ ) {
		if( buf[i] != val ) {
			fprintf( stderr, "Detected error at buffer element %d\n", i );
			fprintf( stderr, "Should be %d, but is %d\n", val, buf[i] );
			exit( 1 );
		}
	}
}
