/* 
** Copyright 1986, 1987, 1988, 1989, 1990, 1991 by the Condor Design Team
** 
** Permission to use, copy, modify, and distribute this software and its
** documentation for any purpose and without fee is hereby granted,
** provided that the above copyright notice appear in all copies and that
** both that copyright notice and this permission notice appear in
** supporting documentation, and that the names of the University of
** Wisconsin and the Condor Design Team not be used in advertising or
** publicity pertaining to distribution of the software without specific,
** written prior permission.  The University of Wisconsin and the Condor
** Design Team make no representations about the suitability of this
** software for any purpose.  It is provided "as is" without express
** or implied warranty.
** 
** THE UNIVERSITY OF WISCONSIN AND THE CONDOR DESIGN TEAM DISCLAIM ALL
** WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES
** OF MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE UNIVERSITY OF
** WISCONSIN OR THE CONDOR DESIGN TEAM BE LIABLE FOR ANY SPECIAL, INDIRECT
** OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS
** OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE
** OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE
** OR PERFORMANCE OF THIS SOFTWARE.
** 
** Authors:  Allan Bricker and Michael J. Litzkow,
** 	         University of Wisconsin, Computer Sciences Dept.
** 
*/ 


#include <stdio.h>
#include <pwd.h>
#include <errno.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <sys/file.h>
#include <ctype.h>
#include <sys/param.h>
#include "except.h"
#include "proc.h"
#include "expr.h"

#ifdef NDBM
#include <ndbm.h>
#else NDBM
#include "ndbm_fake.h"
#endif NDBM

static char *_FileName_ = __FILE__;		/* Used by EXCEPT (see except.h)     */

#define STAR	-1
#define TRUE	1
#define FALSE	0

DBM		*Q, *OpenJobQueue();

int		Long;
char	*MyName;
char	*Spool;

int		UnexpandedJobs;
int		IdleJobs;
int		RunningJobs;

PROC_ID	Filter[100];
int		N_Filters;

usage()
{
	fprintf( stderr, "Usage: %s [-l] [cluster | cluster.proc] ...\n", MyName );
	exit( 1 );
}


/* ARGSUSED */
main( argc, argv)
int		argc;
char	*argv[];
{
	char	queue_name[MAXPATHLEN];
	int		display();
	char	*arg;

	config( *argv, (CONTEXT *)0 );
	init_params();

	for( MyName = *argv++; arg = *argv; argv++ ) {
		if( arg[0] == '-' && arg[1] == 'l' ) {
			Long = 1;
		} else if( isdigit(arg[0]) ) {
			add_to_filter( arg );
		} else {
			usage();
		}
	}

		/* Open job queue */
	(void)sprintf( queue_name, "%s/job_queue", Spool );
	if( (Q=OpenJobQueue(queue_name,O_RDONLY,0)) == NULL ) {
		EXCEPT( "OpenJobQueue(%s)", queue_name );
	}

	/*	This could hang up everything if the user stops condor_q
		(e.g. condor_q | more).

	if( LockJobQueue(Q,READER) < 0 ) {
		EXCEPT( "LockJobQueue()" );
	}
	*/

	if( !Long ) {
		printf( "%-7s %-14s %-11s %-12s %-2s %-3s %-4s %-18s\n\n",
		"   ID", " OWNER", "  SUBMITTED", "   CPU_USAGE",
		" ST", "PRI", "SIZE", "COMMAND" );
	}
	ScanJobQueue( Q, display );

	if( N_Filters == 0 && !Long ) {
		print_summary();
	}

	exit( 0 );
}

SetSyscalls( foo )
int		foo;
{
	return foo;
}

display( proc )
PROC	*proc;
{
	if( Long ) {
		if( !filter(&proc->id) ) {
			display_proc_long( proc );
		}
		return;
	}

	switch( proc->status ) {
		case REMOVED:
		case COMPLETED:
		case SUBMISSION_ERR:
			return;
		case UNEXPANDED:
			UnexpandedJobs += 1;
			break;
		case IDLE:
			IdleJobs += 1;
			break;
		case RUNNING:
			RunningJobs += 1;
			break;
		default:
			EXCEPT( "Job %d.%d, unknown status (%d)\n",
							proc->id.cluster, proc->id.proc, proc->status );
			return;
	}
	if( filter(&proc->id) )
		return;

	display_proc_short( proc );
}


add_to_filter( arg )
char	*arg;
{
	char	*p, *index();
	int		cluster, proc;

	if( p=index(arg,'.') ) {
		cluster = atoi( arg );
		proc = atoi( ++p );
	} else {
		cluster = atoi( arg );
		proc = STAR;
	}

	if( cluster < 1 || proc < STAR )
		usage();

	Filter[N_Filters].cluster = cluster;
	Filter[N_Filters].proc = proc;
	N_Filters += 1;
}

/*
** Return true if this PROC_ID should be filtered from the output (not
** printed). Otherwise return false.
*/
filter( pid )
PROC_ID		*pid;
{
	int		i;

	if( N_Filters == 0 )
		return FALSE;

	for( i=0; i<N_Filters; i++ ) {
		if( pid->cluster == Filter[i].cluster &&
					(pid->proc == Filter[i].proc || Filter[i].proc == STAR) )
			return FALSE;
	}
	return TRUE;
}

init_params()
{
	char	*param();

	Spool = param("SPOOL");
	if( Spool == NULL ) {
		EXCEPT( "SPOOL not specified in config file\n" );
	}
}

print_summary()
{
	int		total;

	total = UnexpandedJobs + IdleJobs + RunningJobs;
	if( total == 0 ) {
		return;
	}

	printf( "\n%d %s in queue %d unexpanded, %d idle, %d running.\n",
	total, total>1 ? "jobs" : "job", UnexpandedJobs, IdleJobs, RunningJobs );
}
