/* 
** Copyright 1992 by the Condor Design Team
** 
** Permission to use, copy, modify, and distribute this software and its
** documentation for any purpose and without fee is hereby granted,
** provided that the above copyright notice appear in all copies and that
** both that copyright notice and this permission notice appear in
** supporting documentation, and that the names of the University of
** Wisconsin and the Condor Design Team not be used in advertising or
** publicity pertaining to distribution of the software without specific,
** written prior permission.  The University of Wisconsin and the Condor
** Design Team make no representations about the suitability of this
** software for any purpose.  It is provided "as is" without express
** or implied warranty.
** 
** THE UNIVERSITY OF WISCONSIN AND THE CONDOR DESIGN TEAM DISCLAIM ALL
** WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES
** OF MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE UNIVERSITY OF
** WISCONSIN OR THE CONDOR DESIGN TEAM BE LIABLE FOR ANY SPECIAL, INDIRECT
** OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS
** OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE
** OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE
** OR PERFORMANCE OF THIS SOFTWARE.
** 
** Author:  Michael J. Litzkow,
** 	        University of Wisconsin, Computer Sciences Dept.
** 
*/ 

#include <stdio.h>

#include <sys/file.h>
#include <sys/dir.h>
#include <sys/param.h>
#include <a.out.h>
#include <sys/core.h>
#include <netinet/in.h>

#include "debug.h"
#include "except.h"
#include "hpux_exec.h"

static char *_FileName_ = __FILE__;     /* Used by EXCEPT (see except.h)     */

extern int	DebugFlags;

#define IN_CORE -1

typedef struct header FILE_HDR;


typedef struct {
	int		addr;
	int		len;
} DATA_BLOCK;

#define HEAD_SIZE sizeof(struct corehead)
typedef struct corehead COREHEAD;

FILE_HDR		F_Header;

char	*do_malloc();
unsigned int checksum();

#if !defined(TRUE)
#define TRUE 1
#define FALSE 0
#endif

#if 0
#define MIN(a,b) ((a)<(b)?(a):(b))
#endif


_updateckpt( ckpt, obj, core )
char *ckpt, *obj, *core;
{
	int			exec_fd, core_fd, ckpt_fd;

	if( core == NULL ) {
		_mkckpt( ckpt, obj );
		return;
	}

	if( (exec_fd=open(obj,O_RDONLY,0)) < 0 ) {
		EXCEPT( "open(%s,O_RDONLY,0)", obj );
	}

	if( (core_fd=open(core,O_RDONLY,0)) < 0 ) {
		EXCEPT( "open(%s,O_RDONLY,0)", core );
	}

	if( (ckpt_fd=open(ckpt,O_WRONLY|O_CREAT|O_TRUNC,0775)) < 0 ) {
		EXCEPT( "open(%s,O_WRONLY|O_CREAT|O_TRUNC,0775)", ckpt );
	}

	dprintf( D_CKPT, "exec_fd = %d, core_fd = %d, ckpt_fd = %d\n",
		exec_fd, core_fd, ckpt_fd );

	do_it( exec_fd, core_fd, ckpt_fd );

	(void)close( exec_fd );
	(void)close( core_fd );
	(void)close( ckpt_fd );
}


STACK_AUX_HDR	*
find_stack_header( headers, aux_bytes )
struct aux_id	*headers;
int				aux_bytes;
{
	int		bytes_used;
	int		key;
	struct aux_id	*a;
	STACK_AUX_HDR   *stk_hdr;
	struct product_specific_aux_hdr *ps_hdr;

	bytes_used = 0;
	a = headers;
	while( bytes_used < aux_bytes ) {
		if( a->type == PRODUCT_SPECIFIC_AUX_ID ) {
			ps_hdr = (struct product_specific_aux_hdr *)a;
			key = ps_hdr->key[0];
			if( key == CONDOR_KEY ) {
				return (STACK_AUX_HDR *)a;
			}
		}
		bytes_used += sizeof(struct aux_id) + a->length;
		a = (char *)headers + bytes_used;
	}
	return (STACK_AUX_HDR *)0;
}

/*
** Create a new a.out (checkpoint) file given an old a.out and
** a core.
*/
do_it( old_fd, core_fd, new_fd )
int		old_fd;
int		core_fd;
int		new_fd;
{
	struct som_exec_auxhdr  *som;
	struct aux_id	*aux_headers;
	STACK_AUX_HDR	*stack_header, *find_stack_header();
	int				new_loc;
	DATA_BLOCK		data_loc;
	DATA_BLOCK		stack_loc;
	int				stk_size_net;	/* sizeof saved stack in net order */
	int				debug_info_delta;
	int				sb_dict_size;
	int				sp_dict_size;
	struct space_dictionary_record		*sp_dict;
	struct subspace_dictionary_record	*sb_dict;
	int				i;

		/* Find location and size of data and stack in core file */
	analyze_core( core_fd, &data_loc, &stack_loc );

		/* Read in file header from original a.out */
	do_read( old_fd, &F_Header, sizeof(F_Header) );

		/* Read in aux headers from original a.out */
	aux_headers = (struct aux_id *)do_malloc( F_Header.aux_header_size );
	do_seek( old_fd, F_Header.aux_header_location, 0 );
	do_read( old_fd, aux_headers, F_Header.aux_header_size );
	som = (struct som_exec_auxhdr *) aux_headers;

	stack_header = find_stack_header( aux_headers, F_Header.aux_header_size );
	if( stack_header == NULL  ) {
		dprintf( D_CKPT, "Generating new Aux Header for saved stack\n" );
		aux_headers = (STACK_AUX_HDR *)realloc( aux_headers,
					F_Header.aux_header_size + sizeof(STACK_AUX_HDR));
		stack_header = (char *)aux_headers + F_Header.aux_header_size;
		F_Header.aux_header_size += sizeof( STACK_AUX_HDR );
		stack_header->header_id.type = PRODUCT_SPECIFIC_AUX_ID;
		stack_header->header_id.length = sizeof( STACK_AUX_HDR );
		stack_header->key[0] = CONDOR_KEY;
	} else {
		dprintf( D_CKPT, "Using existing Aux Header for saved stack\n" );
		F_Header.unloadable_sp_size -= stack_header->exec_ssize +
													sizeof( stk_size_net );
	}

		/* Read in the space dictionary*/
	sp_dict_size = 
		F_Header.space_total * sizeof(struct space_dictionary_record);
	sp_dict = (struct space_dictionary_record *)do_malloc( sp_dict_size );
	do_seek( old_fd, F_Header.space_location, 0 );
	do_read( old_fd, sp_dict, sp_dict_size );

		/* Read in the subspace dictionary*/
	sb_dict_size = 
		F_Header.subspace_total * sizeof(struct subspace_dictionary_record);
	sb_dict = (struct subspace_dictionary_record *)do_malloc( sb_dict_size );
	do_seek( old_fd, F_Header.subspace_location, 0 );
	do_read( old_fd, sb_dict, sb_dict_size );

		/* Save room for aux headers */
	do_seek( new_fd, F_Header.aux_header_location, 0 );	/* Start of aux hdrs */
	do_seek( new_fd, F_Header.aux_header_size, 1 );		/* Orig aux hdrs */

		/* Copy space dictionary */
dprintf( D_CKPT, "Writing Space Dictionary\n" );
	new_loc = do_seek( new_fd, 0, 1 );
	copy_data( old_fd, F_Header.space_location, new_fd,
				F_Header.space_total * sizeof(struct space_dictionary_record) );
	F_Header.space_location = new_loc;
dprintf( D_CKPT, "\n" );

		/* Save room for the subspace dictionary */
	F_Header.subspace_location = do_seek( new_fd, 0, 1 );
	do_seek( new_fd, sb_dict_size, 1 );		

		/* Copy initialization array */
dprintf( D_CKPT, "Writing Initialization Array\n" );
	new_loc = do_seek( new_fd, 0, 1 );
	copy_data( old_fd, F_Header.init_array_location, new_fd,
		F_Header.init_array_total * sizeof(struct init_pointer_record) );
	F_Header.init_array_location = new_loc;
dprintf( D_CKPT, "\n" );

		/* Copy loader fixups */
dprintf( D_CKPT, "Writing Loader Fixups\n" );
	new_loc = do_seek( new_fd, 0, 1 );
	copy_data( old_fd, F_Header.loader_fixup_location, new_fd,
		F_Header.loader_fixup_total * sizeof(struct loader_fixup) );
	F_Header.loader_fixup_location = new_loc;
dprintf( D_CKPT, "\n" );

		/* Copy space strings */
dprintf( D_CKPT, "Writing Space Strings\n" );
	new_loc = do_seek( new_fd, 0, 1 );
	copy_data( old_fd, F_Header.space_strings_location, new_fd,
		F_Header.space_strings_size );
	F_Header.space_strings_location = new_loc;
dprintf( D_CKPT, "\n" );

		/* Copy symbol dictionary */
dprintf( D_CKPT, "Writing Symbol Dictionary\n" );
	new_loc = do_seek( new_fd, 0, 1 );
	copy_data( old_fd, F_Header.symbol_location, new_fd,
		F_Header.symbol_total * sizeof(struct symbol_dictionary_record) );
	F_Header.symbol_location = new_loc;
dprintf( D_CKPT, "\n" );

		/* Copy fixup requests */
dprintf( D_CKPT, "Writing Fixup Requests\n" );
	new_loc = do_seek( new_fd, 0, 1 );
	copy_data( old_fd, F_Header.fixup_request_location, new_fd,
		F_Header.fixup_request_total * sizeof(struct fixup_request_record) );
	F_Header.fixup_request_location = new_loc;
dprintf( D_CKPT, "\n" );

		/* Copy symbol strings */
dprintf( D_CKPT, "Writing Symbol Strings\n" );
	new_loc = do_seek( new_fd, 0, 1 );
	copy_data( old_fd, F_Header.symbol_strings_location, new_fd,
		F_Header.symbol_strings_size );
	F_Header.symbol_strings_location = new_loc;
dprintf( D_CKPT, "\n" );

		/* Copy compiler records */
dprintf( D_CKPT, "Writing Compiler Records\n" );
	new_loc = do_seek( new_fd, 0, 1 );
	copy_data( old_fd, F_Header.compiler_location, new_fd,
		F_Header.compiler_total * sizeof(struct compilation_unit) );
	F_Header.compiler_location = new_loc;
dprintf( D_CKPT, "\n" );

		/* Copy text */
dprintf( D_CKPT, "Writing Text\n" );
	align( new_fd, 0x1000 );
	new_loc = do_seek( new_fd, 0, 1 );
	copy_data( old_fd, som->exec_tfile, new_fd, som->exec_tsize );
	som->exec_tfile = new_loc;
dprintf( D_CKPT, "\n" );

		/* Copy data */
dprintf( D_CKPT, "Writing Data\n" );
	som->exec_dsize = data_loc.len;
	som->exec_bsize = 0;
	align( new_fd, 0x1000 );
	new_loc = do_seek( new_fd, 0, 1 );
	copy_data( core_fd, data_loc.addr, new_fd, data_loc.len );
	som->exec_dfile = new_loc;
dprintf( D_CKPT, "\n" );

		/* Copy unloadable space */
dprintf( D_CKPT, "Writing Unloadable Space\n" );
	align( new_fd, 0x1000 );
	new_loc = do_seek( new_fd, 0, 1 );
	copy_data( old_fd, F_Header.unloadable_sp_location, new_fd,
			F_Header.unloadable_sp_size );
	
	debug_info_delta = new_loc - F_Header.unloadable_sp_location;
	F_Header.unloadable_sp_location = new_loc;
dprintf( D_CKPT, "\n" );

		/* Copy saved stack */
dprintf( D_CKPT, "Writing Saved Stack\n" );
	new_loc = do_seek( new_fd, 0, 1 );
	copy_data( core_fd, stack_loc.addr, new_fd, stack_loc.len );
	F_Header.unloadable_sp_size += stack_loc.len;
dprintf( D_CKPT, "\n" );

		/* Copy size of saved stack area to end of file */
dprintf( D_CKPT, "Writing Sizeof Saved Stack\n" );
	stk_size_net = htonl( stack_loc.len );
	copy_data( IN_CORE, &stk_size_net, new_fd, sizeof(stk_size_net) );
	F_Header.unloadable_sp_size += sizeof(stk_size_net);
	F_Header.som_length = do_seek( new_fd, 0, 1 );
dprintf( D_CKPT, "\n" );

		/* Set up new AUX_HEADER to describe saved stack area */
dprintf( D_CKPT, "Setting up new aux header for saved stack area\n" );
	stack_header->exec_sfile = new_loc;
	stack_header->exec_ssize = stack_loc.len;
dprintf( D_CKPT, "\n" );

		/* Write out aux headers */
dprintf( D_CKPT, "Writing Aux Headers\n" );
	do_seek( new_fd, F_Header.aux_header_location, 0 );
	copy_data( IN_CORE, aux_headers, new_fd, F_Header.aux_header_size );
dprintf( D_CKPT, "\n" );


		/* Write out modified file header */
dprintf( D_CKPT, "Writing File Header\n" );
	F_Header.checksum = checksum( &F_Header, sizeof(F_Header) - sizeof(int) );
	do_seek( new_fd, 0, 0 );
	copy_data( IN_CORE, &F_Header, new_fd, sizeof(F_Header) );
dprintf( D_CKPT, "\n" );

		/* Fix up subspace dictionary file offsets */
dprintf( D_CKPT, "Debug_info_delta = 0x%x (%d)\n",
									debug_info_delta, debug_info_delta );
	i = sp_dict[2].subspace_index;
	while(  i<F_Header.subspace_total ) {
		sb_dict[i].file_loc_init_value += debug_info_delta;
		i += 1;
	}

		/* Write out subspace dictionary */
dprintf( D_CKPT, "Writing Subspace Dictionary\n" );
	do_seek( new_fd, F_Header.subspace_location, 0 );
	copy_data( IN_CORE, sb_dict, new_fd, sb_dict_size );
dprintf( D_CKPT, "\n" );


}


unsigned int
checksum( p, n )
unsigned int	*p;
int				n;
{
	int		i;
	unsigned int	answer;

	n /= sizeof( int );
	answer = 0;
	for( i=0; i<n; i++ ) {
		answer ^= p[i];
	}
	return answer;
}


/*
** Locate data and stack segments in the core file.
*/
analyze_core( fd, data, stack )
int		fd;
DATA_BLOCK	*data;
DATA_BLOCK	*stack;
{
	int		nbytes, next_hdr, this_obj;
	COREHEAD head, *h = &head;

	for(;;) {

		nbytes = do_read( fd, h, HEAD_SIZE );

		if( nbytes == 0 ) {
			break;
		}

		this_obj = do_seek( fd, 0, 1 );
		next_hdr = this_obj + h->len;

		switch( h->type ) {
			case CORE_DATA:
				data->addr = this_obj;
				data->len = h->len;
				break;
			case CORE_STACK:
				stack->addr = this_obj;
				stack->len = h->len;
				break;
			default:
				break;
		}

		do_seek( fd, next_hdr, 0 );
	}
}



/*
** Do a malloc(), but check the return value.
*/
char *
do_malloc( size )
int		size;
{
	char	*answer;

	answer = malloc( size );
	if( answer == NULL ) {
		perror( "malloc" );
		exit( 1 );
	}
	return answer;
}

/*
** Do a seek(), but check the return value.
*/
do_seek( fd, offset, whence )
int		fd;
int		offset;
int		whence;
{
	int		answer;

	if( (answer=lseek(fd,offset,whence))  < 0 ) {
		perror( "lseek" );
		exit( 1 );
	}
	dprintf( D_CKPT, "Seek'd fd %d to 0x%x\n", fd, answer );
	return answer;
}

/*
** Do a write(), but check the return value.
*/
do_write( fd, buf, nbytes )
int		fd;
char	*buf;
int		nbytes;
{
	int		answer;
	int		start;
	int		i;


	start = lseek( fd, 0, 1 );
	if( (answer=write(fd,buf,nbytes)) < 0 ) {
		perror( "write" );
		exit( 1 );
	}
	/*
	dprintf( D_CKPT, "do_write() wrote %d bytes beginning at %d\n",
											answer, start );
	*/
	return answer;
}

/*
** Do a read(), but check the return value.
*/
do_read( fd, buf, nbytes )
int		fd;
char	*buf;
int		nbytes;
{
	int		answer;

	if( (answer=read(fd,buf,nbytes)) < 0 ) {
		perror( "read" );
		exit( 1 );
	}
	return answer;
}

/*
** Copy a block of data into the file whose open file descriptor is
** out_fd.  If in_fd is non-negative, it is an open descriptor into
** a file where the data should be gotten.  In this case in_loc is
** an offset into the input file.  If in_fd is negative, the block
** of data should be copied from memory into the output file.  In 
** this case in_loc is the memory address of the block.
*/
copy_data( in_fd, in_loc, out_fd, len )
int		in_fd;
int		in_loc;
int		out_fd;
int		len;
{
	char	buf[1024];
	int		nbytes;
	int		out_loc;

	out_loc = do_seek( out_fd, 0, 1 );
	dprintf( D_CKPT,
		"Copying 0x%x bytes from fd %d 0x%x - 0x%x to fd %d 0x%x - 0x%x %d\n",
			len, in_fd, in_loc, in_loc + len-1,
			out_fd, out_loc, out_loc + len - 1 );
	if( in_fd < 0 ) {
		do_write( out_fd, in_loc, len );
	} else {
		do_seek( in_fd, in_loc, 0 );
		while( len ) {
			nbytes = MIN( len, sizeof(buf) );
			do_read( in_fd, buf, nbytes );
			do_write( out_fd, buf, nbytes );
			len -= nbytes;
		}
	}
}

/*
** We want to write data to a file which must be aligned on a "size" boundary.
** If needed, seek the file to the next such boundary.
*/ 
align( fd, size )
int		fd;
int		size;
{
	int		loc;
	int		new;

	loc = do_seek( fd, 0, 1 );
	new = ((loc + size - 1) / size) * size;
	if( new != loc ) {
		do_seek( fd, new, 0 );
		dprintf( D_CKPT, "Align 0x%x -> 0x%x\n", loc, new );
	} else {
		dprintf( D_CKPT, "Align 0x%x\n", loc );
	}
}
