/* 
** Copyright 1986, 1987, 1988, 1989, 1990, 1991 by the Condor Design Team
** 
** Permission to use, copy, modify, and distribute this software and its
** documentation for any purpose and without fee is hereby granted,
** provided that the above copyright notice appear in all copies and that
** both that copyright notice and this permission notice appear in
** supporting documentation, and that the names of the University of
** Wisconsin and the Condor Design Team not be used in advertising or
** publicity pertaining to distribution of the software without specific,
** written prior permission.  The University of Wisconsin and the Condor
** Design Team make no representations about the suitability of this
** software for any purpose.  It is provided "as is" without express
** or implied warranty.
** 
** THE UNIVERSITY OF WISCONSIN AND THE CONDOR DESIGN TEAM DISCLAIM ALL
** WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES
** OF MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE UNIVERSITY OF
** WISCONSIN OR THE CONDOR DESIGN TEAM BE LIABLE FOR ANY SPECIAL, INDIRECT
** OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS
** OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE
** OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE
** OR PERFORMANCE OF THIS SOFTWARE.
** 
** Authors:  Allan Bricker and Michael J. Litzkow,
** 	         University of Wisconsin, Computer Sciences Dept.
** 
*/ 


#include <stdio.h>
#include "errno.h"
#include <sys/types.h>
#include <sys/param.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <sys/stat.h>
#include <sys/file.h>

#if defined(AIX31) || defined(AIX32)
#include  <sys/statfs.h>
#include  <sys/utsname.h>
#endif

#if !defined(ultrix)
#ifdef IRIX331
#include <sys/statfs.h>
#else
#include <sys/vfs.h>
#endif
#endif !defined(ultrix)

#if defined(HPUX8)
#include <sys/ustat.h>
#include <sys/utsname.h>
#endif

#include <sys/wait.h>
#include <sys/ioctl.h>
#include <rpc/types.h>
#include <rpc/xdr.h>

#include "condor_types.h"
#include "condor_sys.h"
#include "condor_rsc.h"
#include "trace.h"
#include "except.h"
#include "debug.h"
#include "fileno.h"
#include "proc.h"
#include "xdr_lib.h"
#include "clib.h"
#include "shadow.h"

FILE	*fdopen();

static char *_FileName_ = __FILE__;		/* Used by EXCEPT (see except.h)     */

static int RSCSock;
static int LogSock;
extern int	PipeToParent;
extern int	LogPipe;
extern int	LockFd;
extern int	ImageSize;
extern int	ClientUid;
extern int	ClientGid;
extern struct rusage AccumRusage;

extern int free_fs_blocks();

static FILE *LogFP;

extern PROC Proc;

#define XDR_BUFSIZE	(4 * 1024)

static XDR xdr_RSC, *xdr_syscall = &xdr_RSC;

XDR *
RSC_ShadowInit( rscsock, errsock )
{
	int XDR_ShadowRead(), XDR_ShadowWrite();

	RSCSock = rscsock;
	xdrrec_create( xdr_syscall, XDR_BUFSIZE, XDR_BUFSIZE, &RSCSock,
			XDR_ShadowRead, XDR_ShadowWrite );
	
	xdr_syscall->x_op = XDR_ENCODE;

	LogSock = errsock;
	LogFP = fdopen(LogSock, "r");

	if( LogFP == NULL ) {
		EXCEPT("Cannot fdopen LogSock");
	}

	return( xdr_syscall );
}

static
XDR_ShadowRead( iohandle, buf, len )
register int *iohandle;
char 	*buf;
int 	len;
{
	register int cnt;
	fd_set readfds;
	int nfds;

	FD_ZERO(&readfds);

	nfds = MAX(*iohandle, LogSock) + 1;

	for(;;) {
		FD_SET(*iohandle, &readfds);
		FD_SET(LogSock, &readfds);


		dprintf(D_XDR,
		"Shadow: XDR_ShadowRead: about to select on <%d, %d, and %d>\n",
				*iohandle, LogSock, UMBILICAL);

		cnt = select(nfds, (int *)&readfds, (int *)0, (int *)0,
														(struct timeval *)0);
		if(cnt < 0 ) {
			EXCEPT("XDR_ShadowRead: select");
		}

		if( FD_ISSET(LogSock, &readfds) ) {
			HandleLog();
			continue;
		}

		if( FD_ISSET(*iohandle, &readfds) ) {
			break;
		}

		if( FD_ISSET(UMBILICAL, &readfds) ) {
			dprintf(D_ALWAYS,
				"Shadow: Local scheduler apparently died, so I die too\n");
			exit(1);
		}
	}

	dprintf(D_XDR, "Shadow: XDR_ShadowRead: about to read(%d, 0x%x, %d)\n",
			*iohandle, buf, len );
	
	cnt = read( *iohandle, buf, len );

	dprintf(D_XDR, "Shadow: XDR_ShadowRead: cnt = %d\n", cnt );

	return( cnt );
}

static
XDR_ShadowWrite( iohandle, buf, len )
register int *iohandle;
char *buf;
int	 len;
{
	register int cnt;
	fd_set readfds;
	fd_set writefds;
	int nfds;

	FD_ZERO(&readfds);
	FD_ZERO(&writefds);

	nfds = MAX(*iohandle, LogSock) + 1;

	for(;;) {
		FD_SET(*iohandle, &writefds);
		FD_SET(LogSock, &readfds);


		dprintf(D_XDR,
		"Shadow: XDR_ShadowWrite: about to select on <%d, %d, and %d>\n",
				*iohandle, LogSock, UMBILICAL);

		cnt = select(nfds, (int *)&readfds, (int *)&writefds, (int *)0, 
														(struct timeval *)0 );
		if( cnt < 0 ) {
			EXCEPT("XDR_ShadowWrite: select");
		}

		if( FD_ISSET(LogSock, &readfds) ) {
			HandleLog();
			continue;
		}

		if( FD_ISSET(*iohandle, &writefds) ) {
			break;
		}

		if( FD_ISSET(UMBILICAL, &readfds) ) {
			dprintf(D_ALWAYS,
				"Shadow: Local scheduler apparently died, so I'm dying too\n");
			exit(1);
		}
	}

	dprintf(D_XDR, "Shadow: XDR_ShadowWrite: about to write(%d, 0x%x, %d)\n",
			*iohandle, buf, len );
	
	cnt = write( *iohandle, buf, len );

	dprintf(D_XDR, "Shadow: XDR_ShadowWrite: cnt = %d\n", cnt );

	ASSERT(cnt == len);

	return( cnt );
}


static char *MemToFree = NULL;

#define RSC_SIMPLE(rval) { \
	return( RSC_simple(rval, condor_sysnum) ); \
}

#define RSC_NO_ARGS(func) { \
	int func(); \
	return( RSC_no_args(func, condor_sysnum) ); \
}

#define RSC_1INT(func) { \
	int func(); \
	return( RSC_1int(func, condor_sysnum) ); \
}

#if defined(SUNOS41) || defined(ULTRIX42) || defined(AIX31) || defined(AIX32) || defined(IRIX331) || defined(CMOS) || defined(HPUX8)
#define MODE_T_RSC_1INT(func) { \
	mode_t func(); \
	return( RSC_1int(func, condor_sysnum) ); \
}
#endif

#define RSC_2INT(func) { \
	int func(); \
	return( RSC_2int(func, condor_sysnum) ); \
}

#define RSC_3INT(func) { \
	int func(); \
	return( RSC_3int(func, condor_sysnum) ); \
}

#define RSC_1STR(func) { \
	int func(); \
	return( RSC_1str(func, condor_sysnum) ); \
}

#define RSC_1STR_1INT(func) { \
	int func(); \
	return( RSC_1str_1int(func, condor_sysnum) ); \
}

#define RSC_1STR_2INT(func) { \
	int func(); \
	return( RSC_1str_2int(func, condor_sysnum) ); \
}

#define RSC_2STR(func) { \
	int func(); \
	return( RSC_2str(func, condor_sysnum) ); \
}

#define RSC_1INT_1RBUF(func, s_name, xdr_func) { \
	int func(), xdr_func(); \
	return( RSC_1int_1Rbuf(func, condor_sysnum, \
					sizeof(struct s_name), xdr_func) ); \
}

#define RSC_1STR_1RBUF(func, s_name, xdr_func) { \
	int func(), xdr_func(); \
	return( RSC_1str_1Rbuf(func, condor_sysnum, \
				sizeof(struct s_name), xdr_func) ); \
}

int condor_sysnum;
do_REMOTE_syscall()
{
	int rval;
	/*
	extern errno;
	*/

	xdr_syscall->x_op = XDR_DECODE;


	ASSERT( xdrrec_skiprecord(xdr_syscall) );
	ASSERT( xdr_int(xdr_syscall, &condor_sysnum) );

	dprintf(D_SYSCALLS, "Shadow: got request for syscall %d <%s>\n",
				condor_sysnum, CONDOR_SYSCALLNAME(condor_sysnum));

	switch( condor_sysnum ) {
	/*
	**	System calls with no arguments
	*/
	case CONDOR_async_daemon:	/* async_daemon()                            */
		RSC_NO_ARGS(CONDOR_NotSupported);

	case CONDOR_fork:			/* fork()                                    */
		RSC_NO_ARGS(CONDOR_NotSupported);

	case CONDOR_getdtablesize:	/* getdtablesize()                           */
		RSC_NO_ARGS(getdtablesize);

#ifdef DAYAO
	case PSEUDO_processlogging: /* log file transfer times					 */
		RSC_2INT(ProcessLogging);
#endif DAYAO

	case PSEUDO_getegid:	/* getegid()                                     */
		RSC_SIMPLE( ClientGid );

	case PSEUDO_geteuid:	/* geteuid()                                     */
		RSC_SIMPLE( ClientUid );

	case CONDOR_getgid:			/* getgid()                                  */
		RSC_SIMPLE( ClientGid );

	case CONDOR_gethostid:		/* gethostid()                               */
		RSC_NO_ARGS(gethostid);

	case CONDOR_getpagesize:	/* getpagesize()                             */
		RSC_NO_ARGS(getpagesize);

	case CONDOR_getpid:			/* getpid()                                  */
		RSC_NO_ARGS(condor_getpid);

#if defined(AIX31) || defined(AIX32)
	case CONDOR_getpgrp:		/* getpgrp() 	                             */
		RSC_NO_ARGS(condor_getpgrp);
#else
	case CONDOR_getpgrp:		/* getpgrp( int pid )						 */
		RSC_1INT(condor_getpgrp);
#endif

	case PSEUDO_getppid:	/* getppid()                                     */
		RSC_NO_ARGS(condor_getppid);

	case CONDOR_getuid:			/* getuid()                                  */
		RSC_SIMPLE( ClientUid );

	case CONDOR_sync:			/* sync()                                    */
		RSC_NO_ARGS(sync);

	case CONDOR_vhangup:		/* vhangup()                                 */
#ifdef SYS_vhangup
		RSC_NO_ARGS(vhangup);
#else SYS_vhangup
		RSC_NO_ARGS(CONDOR_NotSupported);
#endif SYS_vhangup

	/*
	**	System calls which take one integer argument
	*/
	case PSEUDO_image_size:		/* image_size( ind d) virt size in megs     */
		RSC_1INT(condor_image_size);

#if defined(AIX31) || defined(AIX32)
	case CONDOR_getuidx:		/* getuidx( int which );					*/
		RSC_1INT(condor_getuidx);

	case CONDOR_getgidx:		/* getgidx( int which );					*/
		RSC_1INT(condor_getgidx);
#endif

	case CONDOR_close:			/* close( int d )                            */
		RSC_1INT(condor_close);

	case CONDOR_dup:			/* dup( int oldd )                           */
		RSC_1INT(dup);

	case CONDOR__exit:			/* _exit( int status )                       */
		RSC_1INT(condor_exit);

	case CONDOR_fsync:			/* fsync( int fd )                           */
		RSC_1INT(fsync);

#if defined(AIX31) || defined(AIX32)
	case CONDOR_kgetpgrp:		/* kgetpgrp( pid_t pid )					*/
		RSC_1INT(condor_kgetpgrp);
#endif

	case CONDOR_nfssvc:			/* nfssvc( int sock )                        */
		RSC_1INT(CONDOR_NotSupported);

#ifndef IRIX331
	case CONDOR_reboot:			/* reboot( int howto )                       */
		RSC_1INT(reboot);
#endif

	case CONDOR_umask:			/* umask( int numask )                       */
#if defined(SUNOS41) || defined(ULTRIX42) || defined(AIX31) || defined(AIX32) || defined(IRIX331) || defined(CMOS) || defined(HPUX8)
		MODE_T_RSC_1INT(umask);
#else
		RSC_1INT(umask);
#endif

	/*
	**	System calls which take one long argument
	*/
	case CONDOR_sethostid:		/* sethostid( int hostid )                   */
	{
		int hostid;

		ASSERT(xdr_int(xdr_syscall, &hostid));

#ifdef SYS_sethostid
		rval = sethostid( hostid );
#else SYS_sethostid
		rval = CONDOR_NotSupported();
#endif SYS_sethostid

		dprintf(D_SYSCALLS, "Shadow: sethostid(%d) = %d\n", hostid, rval);

		PUT_RVAL(xdr_syscall, rval);
		END_RVAL(xdr_syscall, 0);
	}

	/*
	**	System calls which take two integer arguments
	*/
	case CONDOR_dup2:			/* dup2( int oldd, int newd )                */
		RSC_2INT(condor_dup2);

	case CONDOR_fchmod:			/* fchmod( int fd, int mode )                */
		RSC_2INT(fchmod);

	case CONDOR_flock:			/* flock( int fd, int operation )            */
		RSC_2INT(flock);

#if !defined(HPUX8)
	case CONDOR_getpriority:	/* getpriority( int which, int who )         */
		RSC_2INT(getpriority);
#endif

	case CONDOR_kill:			/* kill( int pid, int sig )                  */
		RSC_2INT(CONDOR_NotSupported);

	case CONDOR_killpg:			/* killpg( int pgrp, int sig )               */
		RSC_2INT(CONDOR_NotSupported);

	case CONDOR_listen:			/* listen( int s, int backlog )              */
		RSC_2INT(listen);

	case CONDOR_setpgrp:		/* setpgrp( int pid, int pgrp )              */
		RSC_2INT(setpgrp);

	case CONDOR_setregid:		/* setregid( int rgid, int egid )            */
		RSC_2INT(setregid);

	case CONDOR_setreuid:		/* setreuid( int ruid, int euid )            */
		RSC_2INT(setreuid);

	case CONDOR_shutdown:		/* shutdown( int s, int how )                */
		RSC_2INT(shutdown);

	/*
	**	System calls which take an integer argument and an off_t argument
	*/
	case CONDOR_ftruncate:		/* ftruncate( int fd, off_t length )         */
	{
		int fd, length;

		ASSERT(xdr_int(xdr_syscall, &fd));
		ASSERT(xdr_int(xdr_syscall, &length));

		rval = ftruncate( fd, length );

		dprintf(D_SYSCALLS, "Shadow: ftruncate(%d, %d) = %d\n",
				fd, length, rval);

		PUT_RVAL(xdr_syscall, rval);
		END_RVAL(xdr_syscall, 0);
	}

	/*
	**	System calls which take three integer arguments
	*/
	case CONDOR_fchown:			/* fchown( int fd, int owner, int group )    */
		RSC_3INT(fchown);

	case CONDOR_fcntl:			/* fcntl( int fd, int cmd, int arg )         */
#if defined(AIX31) || defined(AIX32)
		return( RSC_3int(fcntl, condor_sysnum) );
#else
		RSC_3INT(fcntl);
#endif

#if defined(AIX31) || defined(AIX32)
	case CONDOR_kfcntl:			/* kfcntl( int fd, int cmd, int arg )         */
		RSC_3INT(kfcntl);
#endif


#if !defined(HPUX8)
	case CONDOR_setpriority:	/* setpriority( int which, int who,
								**					int prio )
								*/
		RSC_3INT(setpriority);
#endif

	case CONDOR_socket:			/* socket( int domain, int type,
								**					int protocol ) 
								*/
		RSC_3INT(CONDOR_NotSupported);

	/*
	**  System calls which take an integer argument followed by
	**  an off_t argument followed by another integer argument.
	*/
	case CONDOR_lseek:			/* lseek( int d, off_t offset, int whence )  */
	{
		int d, whence;
		off_t pos, offset;

		ASSERT(xdr_int(xdr_syscall, &d));
		ASSERT(xdr_int(xdr_syscall, &offset));
		ASSERT(xdr_int(xdr_syscall, &whence));

		pos = lseek( d, offset, whence );

		dprintf(D_SYSCALLS, "Shadow: lseek(%d, %d, %d) = %d\n",
				d, offset, whence, pos);

		PUT_RVAL(xdr_syscall, pos);
		END_RVAL(xdr_syscall, 0);
	}

	/*
	**	System calls which take one string argument
	*/

	case PSEUDO_perm_error:		/* perm_error( char *msg )					*/
		RSC_1STR( PERM_ERR );

	case CONDOR_acct:			/* acct( char *file  )                       */
		RSC_1STR(acct);

	case CONDOR_chdir:			/* chdir( char *path )                       */
		RSC_1STR(chdir);

	case CONDOR_chroot:			/* chroot( char *dirname )                   */
		RSC_1STR(chroot);

	case CONDOR_rmdir:			/* rmdir( char *path )                       */
		RSC_1STR(rmdir);

#ifndef IRIX331
	case CONDOR_swapon:			/* swapon( char *special )                   */
		RSC_1STR(swapon);
#endif

	case CONDOR_unlink:			/* unlink( char *path )                      */
		RSC_1STR(unlink);

	case CONDOR_umount:			/* umount( dev_t dev )                 		 */
		RSC_1INT(CONDOR_NotSupported);

	case CONDOR_unmount:		/* unmount( char *name )                     */
		RSC_1STR(CONDOR_NotSupported);

	case PSEUDO_free_fs_blocks:	/* free_fs_blocks( char *filename )          */
		RSC_1STR(free_fs_blocks);


	/*
	**	System calls which take one string and one integer as arguments
	*/
	case CONDOR_access:			/* access( char *path, int mode )            */
		RSC_1STR_1INT(access);

	case CONDOR_chmod:			/* chmod( char *path, int mode )             */
		RSC_1STR_1INT(chmod);

	case CONDOR_creat:			/* creat( char *path, int mode )             */
		RSC_1STR_1INT(creat);

	case CONDOR_mkdir:			/* mkdir( char *path, int mode )             */
		RSC_1STR_1INT(mkdir);

	case CONDOR_setdomainname:	/* setdomainname( char *name, int namelen )  */
#ifdef SYS_setdomainname
		RSC_1STR_1INT(setdomainname);
#else SYS_setdomainname
		RSC_1STR_1INT(CONDOR_NotSupported);
#endif SYS_setdomainname

	case CONDOR_sethostname:	/* sethostname( char *name, int namelen )    */
		RSC_1STR_1INT(sethostname);

	/*
	**	System calls which take one string and one off_t as arguments
	*/
	case CONDOR_truncate:		/* truncate( char *path, off_t length )      */
	{
		char path[ MAXPATHLEN ], *pp = path;
		off_t length;

		ASSERT(xdr_string(xdr_syscall, &pp, (u_int) MAXPATHLEN));
		ASSERT(xdr_int(xdr_syscall, &length));

		rval = truncate(path, length);

		dprintf(D_SYSCALLS, "Shadow: truncate(%s, %d) = %d\n",
				path, length, rval);

		PUT_RVAL(xdr_syscall, rval);
		END_RVAL(xdr_syscall, 0);
	}

	/*
	**	System calls which take one string and two integers as arguments
	*/
	case CONDOR_chown:			/* chown( char *path, int owner, int group ) */
		RSC_1STR_2INT(chown);

	case CONDOR_mknod:			/* mknod( char *path, int mode, int dev )    */
		RSC_1STR_2INT(mknod);

	case CONDOR_open:			/* open( char *path, int flags, int mode )   */
#if defined(AIX31) || defined(AIX32)
		return( RSC_1str_2int(open, condor_sysnum) );
#else
		RSC_1STR_2INT(open);
#endif


	/*
	**	System calls which take two strings as arguments
	*/
	case CONDOR_link:			/* link( char *name1, char *name2 )          */
		RSC_2STR(link);

	case CONDOR_rename:			/* rename( char *from, char *to )            */
		RSC_2STR(rename);

	case CONDOR_symlink:		/* symlink( char *name1, char *name2 )       */
		RSC_2STR(symlink);

	/*
	**	Others...
	*/

#if defined(AIX31) || defined(AIX32)
	case CONDOR_kioctl:			/* kioctl(int fd,int cmd,int arg,char *ext) */
	{
	int		fd;
	int		cmd;
	int		arg;
	char	ext[ MAXPATHLEN ], *ptr = ext;

	ASSERT(xdr_int(xdr_syscall, &fd) );
	ASSERT(xdr_int(xdr_syscall, &cmd) );
	ASSERT(xdr_int(xdr_syscall, &arg) );
	ASSERT(xdr_string(xdr_syscall, &ptr, (u_int) MAXPATHLEN));

	rval = kioctl( fd, cmd, arg, ext );

		/* Send back the final status */
	PUT_RVAL( xdr_syscall, rval );
	END_RVAL(xdr_syscall, 0);
	}
#endif /* AIX31 or AIX32 */

	case PSEUDO_send_file:		/* send_file( char *name, int mode )		 */
	{
	char name[ MAXPATHLEN ], *ptr = name;
	int	mode;
	char	buf[ XDR_BLOCKSIZ ];
	int rval = 0;
	int	bytes_to_go;
	int	file_len;
	int	checksum;
	int	fd;
	int	len, nbytes;

		/* Get the name of the file */
	ASSERT(xdr_string(xdr_syscall, &ptr, (u_int) MAXPATHLEN));
	ASSERT(xdr_int(xdr_syscall, &mode) );
	dprintf(D_SYSCALLS, "Shadow: %s(%s,0%o)\n",
		CONDOR_SYSCALLNAME(condor_sysnum), name, mode );

		/* Open the file for writing, send back status from the open */
	if( (fd=open(name,O_WRONLY|O_CREAT|O_TRUNC,mode)) < 0 ) {
		dprintf(D_SYSCALLS, "Shadow: open(%s,O_WRONLY|O_CREAD|O_TRUNC", name );
		dprintf(D_SYSCALLS, ",0664) = %d, errno = %d\n", fd, errno );
	}
	PUT_RVAL( xdr_syscall, fd );
	ASSERT( xdrrec_endofrecord(xdr_syscall,TRUE) );
	dprintf(D_SYSCALLS, "Shadow: open(%s,O_WRONLY|O_CREAD|O_TRUNC,0)",name);
	dprintf(D_SYSCALLS, " = %d\n", errno );

		/* Get the file length */
	xdr_syscall->x_op = XDR_DECODE;
	ASSERT( xdrrec_skiprecord(xdr_syscall) );
	ASSERT( xdr_int(xdr_syscall, &file_len) );
	dprintf(D_SYSCALLS, "Shadow: File size is %d\n", file_len );
		
		/* Transfer all the data */
	for( bytes_to_go = file_len; bytes_to_go; bytes_to_go -= len ) {
		len = bytes_to_go < sizeof(buf) ? bytes_to_go : sizeof(buf);
		nbytes = len;
		errno = 0;
		if( !xdr_opaque(xdr_syscall,buf,len) ) {
			dprintf( D_ALWAYS, "Shadow: xdr_opaque() failed, errno = %d\n",
													errno );
			rval = -1;
			nbytes = 0;
		}
		if( write(fd,buf,nbytes) < nbytes ) {
			dprintf( D_ALWAYS, "Shadow: Can't write to \"%s\", errno = %d\n",
													name, errno );
			rval = -1;
			nbytes = 0;
		}
	}
	(void)close( fd );

		/* Get a repeat of the file length as a check */
	ASSERT( xdr_int(xdr_syscall,&checksum) );
	dprintf(D_SYSCALLS, "Shadow: checksum is %d\n", checksum );
	ASSERT( checksum == file_len );

		/* Send back the final status */
	PUT_RVAL( xdr_syscall, rval );
	END_RVAL(xdr_syscall, 0);
	}

	case PSEUDO_get_file:		/* get_file( char *name )					 */
	{
	char name[ MAXPATHLEN ], *ptr = name;
	char	buf[ XDR_BLOCKSIZ ];
	int	len;
	int	read_status;
	int	file_size, bytes_to_go;
	int	fd;

		/* Get the remote file name */
	ASSERT(xdr_string(xdr_syscall, &ptr, (u_int) MAXPATHLEN));
	dprintf(D_SYSCALLS, "Shadow: %s(%s)\n",
		CONDOR_SYSCALLNAME(condor_sysnum), name );

		/* Open the remote file and return status from the open */
	errno = 0;
	fd = open( name, O_RDONLY, 0 );
	dprintf(D_SYSCALLS, "Shadow: open(%s,O_RDONLY,0) = %d, errno = %d\n",
														name, fd, errno );
	PUT_RVAL( xdr_syscall, fd );

		/* Send the size of the file */
	file_size = lseek( fd, 0, 2 );
	(void)lseek( fd, 0, 0 );
	dprintf(D_SYSCALLS, "Shadow: File size is %d bytes\n", file_size );
	PUT_RVAL(xdr_syscall,file_size);

		/* Transfer the data */
	for( bytes_to_go = file_size; bytes_to_go; bytes_to_go -= len ) {
		len = bytes_to_go < sizeof(buf) ? bytes_to_go : sizeof(buf);
		if( read(fd,buf,len) < len ) {
			dprintf( D_ALWAYS, "Can't read from \"%s\"\n", name );
			read_status = -1;
		}
		ASSERT( xdr_opaque(xdr_syscall,buf,len) );
	}
	(void)close( fd );

		/* Send the size of the file again as a check */
	ASSERT( xdr_int(xdr_syscall,&file_size) );
	dprintf(D_SYSCALLS, "Shadow: Sent file size %d\n", file_size );

		/* Send final status */
	PUT_RVAL( xdr_syscall, read_status );
	END_RVAL(xdr_syscall, 0);
	}

	case CONDOR_wait:			/* wait( R union wait *status )              */
	{
		union wait status;

		/*
		** rval = wait( &status );
		*/
		rval = CONDOR_NotSupported();

		dprintf(D_SYSCALLS, "Shadow: wait(0x%x) = %d\n", &status, rval );

		PUT_RVAL(xdr_syscall, rval);

		ASSERT(xdr_wait(xdr_syscall, &status));

		END_RVAL(xdr_syscall, 0);
	}

	case PSEUDO_send_rusage: /* send_rusage( M struct rusage *rusage ) */
	{

		ASSERT(xdr_rusage(xdr_syscall, &AccumRusage));

		rval = 0;

		dprintf(D_SYSCALLS, "Shadow: send_rusage(0x%x) = %d\n",
			&AccumRusage, rval );
		D_RUSAGE( D_SYSCALLS, &AccumRusage );

		PUT_RVAL(xdr_syscall, rval);
		END_RVAL(xdr_syscall, 0);
	}

	case PSEUDO_reallyexit:	/* reallyexit( M union wait *status )            */
	{
		extern union wait JobStatus;
		extern struct rusage JobRusage;

		ASSERT(xdr_wait(xdr_syscall, &JobStatus));
		ASSERT(xdr_rusage(xdr_syscall, &JobRusage));

		rval = 0;
		dprintf(D_SYSCALLS, "Shadow: reallyexit(0x%x, 0x%x) = %d\n",
								&JobStatus, &JobRusage, rval );

		PUT_RVAL(xdr_syscall, rval);
		END_RVAL(xdr_syscall, -1);
	}

	case PSEUDO_wait3:	/* wait3( R union wait *status, int options,
						**						struct rusage *rusage )
						*/
	{
		int xdr_rusage();
		union wait status;
		int options;
		struct rusage *rusage;

		ASSERT(xdr_int(xdr_syscall, &options));
		ASSERT(xdr_ptr(xdr_syscall, (caddr_t *) &rusage, xdr_rusage));

		/*
		**	rval = wait3( &status, options, rusage );
		*/
		rval = CONDOR_NotSupported();

		dprintf(D_SYSCALLS, "Shadow: wait3(0x%x, 0%o, 0x%x) = %d\n",
			&status, options, rusage, rval );

		PUT_RVAL(xdr_syscall, rval);

		ASSERT(xdr_wait(xdr_syscall, &status));
		ASSERT(xdr_ptr(xdr_syscall, (caddr_t *) &rusage, xdr_rusage));

		END_RVAL(xdr_syscall, 0);
	}

	case PSEUDO_getwd: /* getwd( R char *pathname )                          */
	{
		char pathname[ MAXPATHLEN ], *pnp = pathname;

		if( getwd(pathname) ) {
			rval = 1;
		} else {
			rval = 0;
		}

		dprintf(D_SYSCALLS, "Shadow: getwd(%s) = 0x%x\n", pathname, rval);

		PUT_RVAL(xdr_syscall, rval);
		ASSERT(xdr_string(xdr_syscall, &pnp, (u_int) MAXPATHLEN));
		END_RVAL(xdr_syscall, 0);
	}

	case CONDOR_lstat:			/* lstat( char *path, R struct stat *buf )   */
		RSC_1STR_1RBUF(lstat,stat,xdr_stat);

	case CONDOR_stat:			/* stat( char *path, R struct stat *buf )    */
		RSC_1STR_1RBUF(stat,stat,xdr_stat);

#if defined(AIX31) || defined(AIX32)
/*
** This should NOT return the data with xdr_opaque, but the variable
** lenght buffer makes it hard to do otherwise.  Probably will only be
** implemented for IBM machines which will all share the same data
** format anyway...
*/
	case CONDOR_statx:			/* statx( char *path, R struct stat *buf,
										int len, int cmd )					*/
	{
		char path[ MAXPATHLEN ], *s1 = path;
		caddr_t	buf;
		int	len;
		int	cmd;
		int rval;

		ASSERT(xdr_string(xdr_syscall, (char **)&s1, (u_int) MAXPATHLEN));
		ASSERT( xdr_int(xdr_syscall, &len) );
		ASSERT( xdr_int(xdr_syscall, &cmd) );

		MemToFree = buf = MALLOC( (unsigned) len );
		rval = statx( path, (struct stat *)buf, len, cmd );

		dprintf(D_SYSCALLS, "Shadow: %s(%s, 0x%x, %d, %d) = %d\n",
				CONDOR_SYSCALLNAME(condor_sysnum), s1, buf, len, cmd, rval);

		PUT_RVAL(xdr_syscall, rval);
		ASSERT( xdr_opaque(xdr_syscall,buf,(unsigned)len) );
		END_RVAL(xdr_syscall, 0);
	}
#endif /* AIX31 or AIX32 */

#if defined(AIX31) || defined(AIX32)
	case CONDOR_getdirent:	/* getdirent( int fd, char *buf, unsigned len ) */
	{
		caddr_t	buf;
		int	len;
		int	fd;
		int rval;

		ASSERT( xdr_int(xdr_syscall, &fd) );
		ASSERT( xdr_int(xdr_syscall, &len) );
		MemToFree = buf = MALLOC( (unsigned) len );
		rval = getdirent( fd, buf, len );

		dprintf(D_SYSCALLS, "Shadow: %s(%d, 0x%x, %d) = %d\n",
				CONDOR_SYSCALLNAME(condor_sysnum), fd, buf, len, rval);

		PUT_RVAL(xdr_syscall, rval);
		ASSERT( xdr_bytes(xdr_syscall,&buf,&len,4096) );
		END_RVAL(xdr_syscall, 0);
	}
#endif /* AIX31 or AIX32 */

#if defined(AIX31) || defined(AIX32)
	case CONDOR_uname:	/* uname( struct utsname * ) */
	{
		struct utsname	buf, *ptr = &buf;
		int	len = sizeof(struct utsname);
		int rval;

		rval = uname( ptr );

		dprintf(D_SYSCALLS, "Shadow: %s( 0x%x ) = %d\n",
				CONDOR_SYSCALLNAME(condor_sysnum), ptr, rval);

		PUT_RVAL(xdr_syscall, rval);
		ASSERT( xdr_bytes(xdr_syscall,&ptr,&len,len) );
		END_RVAL(xdr_syscall, 0);
	}
#endif

#if defined(AIX31) || defined(AIX32)
	case CONDOR_unamex:	/* uname( struct xutsname * ) */
	{
		struct xutsname	buf, *ptr = &buf;
		int	len = sizeof(struct xutsname);
		int rval;

		rval = unamex( ptr );

		dprintf(D_SYSCALLS, "Shadow: %s( 0x%x ) = %d\n",
				CONDOR_SYSCALLNAME(condor_sysnum), ptr, rval);

		PUT_RVAL(xdr_syscall, rval);
		ASSERT( xdr_long(xdr_syscall,&buf.nid) );
		END_RVAL(xdr_syscall, 0);
	}
#endif

#if defined(AIX31) || defined(AIX32)
	case CONDOR_usrinfo:	/* usrinfo( int cmd, char *buf, int count ) */
	{
		caddr_t	buf;
		int		cmd;
		int		count;
		int		rval;

		ASSERT(xdr_int(xdr_syscall, &cmd) );
		ASSERT( xdr_int(xdr_syscall, &count) );

		MemToFree = buf = MALLOC( (unsigned) count );

		rval = usrinfo( cmd, (char *)buf, count );

		dprintf(D_SYSCALLS, "Shadow: %s(%d, 0x%x, %d) = %d\n",
				CONDOR_SYSCALLNAME(condor_sysnum), cmd, buf, count, rval);

		PUT_RVAL(xdr_syscall, rval);
		ASSERT( xdr_bytes(xdr_syscall,(char **)&buf,&count,count) );
		END_RVAL(xdr_syscall, 0);
	}
#endif

#if defined(AIX31) || defined(AIX32)
	case CONDOR_statacl:	/* statacl( char *path, int cmd,
												struct acl *acl, int size ) */
	{
		char path[ MAXPATHLEN ], *s1 = path;
		caddr_t	buf;
		int		cmd;
		int		len;
		int		rval;

		ASSERT(xdr_string(xdr_syscall, (char **)&s1, (u_int) MAXPATHLEN));
		ASSERT(xdr_int(xdr_syscall, &cmd) );
		ASSERT( xdr_int(xdr_syscall, &len) );

		MemToFree = buf = MALLOC( (unsigned) len );

		rval = statacl( path, cmd, (struct acl *)buf, len );

		dprintf(D_SYSCALLS, "Shadow: %s(%s, %d, 0x%x, %d) = %d\n",
				CONDOR_SYSCALLNAME(condor_sysnum), path, cmd, buf, len, rval);

		PUT_RVAL(xdr_syscall, rval);
		ASSERT( xdr_bytes(xdr_syscall,(char **)&buf,&len,len) );
		END_RVAL(xdr_syscall, 0);
	}
#endif

#if defined(AIX31) || defined(AIX32)
	case CONDOR_fstatacl:	/* fstatacl( int fd, int cmd,
												struct acl *acl, int size ) */
	{
		int		fd;
		caddr_t	buf;
		int		cmd;
		int		len;
		int		rval;

		ASSERT(xdr_int(xdr_syscall, &fd) );
		ASSERT(xdr_int(xdr_syscall, &cmd) );
		ASSERT( xdr_int(xdr_syscall, &len) );

		MemToFree = buf = MALLOC( (unsigned) len );

		rval = fstatacl( fd, cmd, (struct acl *)buf, len );

		dprintf(D_SYSCALLS, "Shadow: %s(%d, %d, 0x%x, %d) = %d\n",
				CONDOR_SYSCALLNAME(condor_sysnum), fd, cmd, buf, len, rval);

		PUT_RVAL(xdr_syscall, rval);
		ASSERT( xdr_bytes(xdr_syscall,(char **)&buf,&len,len) );
		END_RVAL(xdr_syscall, 0);
	} 
#endif

#if defined(AIX31) || defined(AIX32)
	case CONDOR_fchacl:		/* fchacl( int fd, struct acl *acl, int len ) */
	{
		int		fd;
		caddr_t	buf;
		int		len;
		int		rval;

		ASSERT( xdr_int(xdr_syscall, &fd) );
		ASSERT( xdr_int(xdr_syscall, &len) );

		MemToFree = buf = MALLOC( (unsigned) len );
		ASSERT( xdr_bytes(xdr_syscall, (char **)&buf, &len, 4096) );
		rval = fchacl( fd, (struct acl *)buf, len );

		dprintf(D_SYSCALLS, "Shadow: %s(%d, 0x%x, %d) = %d\n",
				CONDOR_SYSCALLNAME(condor_sysnum), fd, buf, len, rval);

		PUT_RVAL(xdr_syscall, rval);
		END_RVAL(xdr_syscall, 0);
	}
#endif

#if defined(AIX31) || defined(AIX32)
	case CONDOR_chacl:		/* chacl( char *path, struct acl *acl, int len ) */
	{
		char path[ MAXPATHLEN ], *s1 = path;
		caddr_t	buf;
		int	len;
		int rval;

		ASSERT(xdr_string(xdr_syscall, (char **)&s1, (u_int) MAXPATHLEN));
		ASSERT( xdr_int(xdr_syscall, &len) );

		MemToFree = buf = MALLOC( (unsigned) len );
		ASSERT( xdr_bytes(xdr_syscall, (char **)&buf, &len, 4096) );
		rval = chacl( path, (struct acl *)buf, len );

		dprintf(D_SYSCALLS, "Shadow: %s(%s, 0x%x, %d) = %d\n",
				CONDOR_SYSCALLNAME(condor_sysnum), s1, buf, len, rval);

		PUT_RVAL(xdr_syscall, rval);
		END_RVAL(xdr_syscall, 0);
	}
#endif

	case CONDOR_pipe:			/* pipe( int fildes[2] )                     */
	{
		int fildes[2];

		/*
		**	rval = pipe(fildes);
		*/
		rval = CONDOR_NotSupported();

		dprintf(D_SYSCALLS, "Shadow: pipe(0x%x) = %d\n", fildes, rval);

		PUT_RVAL(xdr_syscall, rval);

		ASSERT(xdr_int(xdr_syscall, &fildes[0]));
		ASSERT(xdr_int(xdr_syscall, &fildes[1]));

		END_RVAL(xdr_syscall, 0);
	}

	case CONDOR_settimeofday:	/* settimeofday( struct timeval *tp,
								**				 struct timezone *tzp )
								*/
	{
		struct timeval tv, *tp = &tv;
		struct timezone tz, *tzp = &tz;
		int xdr_timezone();
		
		ASSERT(timeval_xdr(xdr_syscall, tp));
		ASSERT(xdr_ptr(xdr_syscall, (caddr_t *) &tzp, xdr_timezone));

		rval = settimeofday( tp, tzp );

		dprintf(D_SYSCALLS, "Shadow: settimeofday(0x%x, 0x%x) = %d\n",
					tp, tzp, rval);

		PUT_RVAL(xdr_syscall, rval);

		END_RVAL(xdr_syscall, 0);
	}

	case CONDOR_gettimeofday:	/* gettimeofday( R struct timeval *tp,
							**				 R struct timezone *tzp )
							*/
	{
		struct timeval tv, *tp = &tv;
		struct timezone tz, *tzp = &tz;
		int xdr_timezone();
		
		ASSERT(xdr_ptr(xdr_syscall, (caddr_t *) &tzp, xdr_timezone));

		rval = gettimeofday( tp, tzp );

		dprintf(D_SYSCALLS,
			"Shadow: gettimeofday(0x%x <%d, %d>, 0x%x <%d, %d>) = %d\n",
				tp, tp->tv_sec, tp->tv_usec,
				tzp, (tzp?tzp->tz_minuteswest:0), (tzp?tzp->tz_dsttime:0),
				rval);

		PUT_RVAL(xdr_syscall, rval);

		ASSERT(timeval_xdr(xdr_syscall, tp));
		ASSERT(xdr_ptr(xdr_syscall, (caddr_t *) &tzp, xdr_timezone));

		END_RVAL(xdr_syscall, 0);
	}

	case CONDOR_getdomainname:	/* getdomainname( R char *name, int namelen )*/
	{
		char *name;
		int namelen;
		
		ASSERT(xdr_int(xdr_syscall, &namelen));

		MemToFree = name = MALLOC( (unsigned)namelen );
		if( name == NULL ) {
			rval = -1;
		} else {
			rval = getdomainname( name, namelen );
		}

		dprintf(D_SYSCALLS, "Shadow: getdomainname(%s, %d) = %d\n",
					name, namelen, rval);

		PUT_RVAL(xdr_syscall, rval);
		ASSERT(xdr_string(xdr_syscall, &name, (u_int) namelen));
		END_RVAL(xdr_syscall, 0);
	}

	case CONDOR_gethostname:	/* gethostname( R char *name, int namelen )  */
	{
		char *name;
		int namelen;
		
		ASSERT(xdr_int(xdr_syscall, &namelen));

		MemToFree = name = MALLOC( (unsigned)namelen );
		if( name == NULL ) {
			rval = -1;
		} else {
			rval = gethostname( name, namelen );
		}

		dprintf(D_SYSCALLS, "Shadow: gethostname(%s, %d) = %d\n",
					name, namelen, rval);

		PUT_RVAL(xdr_syscall, rval);
		ASSERT(xdr_string(xdr_syscall, &name, (u_int) namelen));
		END_RVAL(xdr_syscall, 0);
	}

	case CONDOR_select:		/* select( int nfds,
							**		0/VR fd_set *readfds,
							**		0/VR fd_set *writefds,
							**		0/VR fd_set *exceptfds,
							**		0/V struct timeval *timeout )
							*/
	{
		int xdr_fdset(), timeval_xdr();
		int nfds;
		fd_set Rfds, *readfds = &Rfds;
		fd_set Wfds, *writefds = &Wfds;
		fd_set Efds, *exceptfds = &Efds;
		struct timeval TO, *timeout = &TO;

		ASSERT(xdr_int(xdr_syscall, &nfds));
		ASSERT(xdr_ptr(xdr_syscall, (caddr_t *) &readfds, xdr_fdset));
		ASSERT(xdr_ptr(xdr_syscall, (caddr_t *) &writefds, xdr_fdset));
		ASSERT(xdr_ptr(xdr_syscall, (caddr_t *) &exceptfds, xdr_fdset));
		ASSERT(xdr_ptr(xdr_syscall, (caddr_t *) &timeout, timeval_xdr));

		rval = select( nfds, (int *)readfds, (int *)writefds,
												(int *)exceptfds, timeout );

		PUT_RVAL(xdr_syscall, rval);

		ASSERT(xdr_ptr(xdr_syscall, (caddr_t *) &readfds, xdr_fdset));
		ASSERT(xdr_ptr(xdr_syscall, (caddr_t *) &writefds, xdr_fdset));
		ASSERT(xdr_ptr(xdr_syscall, (caddr_t *) &exceptfds, xdr_fdset));
		ASSERT(xdr_ptr(xdr_syscall, (caddr_t *) &timeout, timeval_xdr));

		END_RVAL(xdr_syscall, 0);
	}

	case CONDOR_setgroups:		/* setgroups( int ngroups, M int *gidset )   */
	{
		int ngroups, xdr_int();
		int GIDSet[ NGROUPS ], *gidset = GIDSet;

		ASSERT(xdr_array(xdr_syscall, &gidset, &ngroups, NGROUPS, sizeof(int), xdr_int));

		rval = setgroups( ngroups, gidset );

		PUT_RVAL(xdr_syscall, rval);
		END_RVAL(xdr_syscall, 0);
	}

	case CONDOR_setitimer:		/* setitimer( int which,
								**			M struct itimerval *value,
								**		  R/0 struct itimerval *ovalue )
								*/
	{
		int which;
		struct itimerval Val, *value = &Val;
		struct itimerval OVal, *ovalue = &OVal;
		
		ASSERT(xdr_int(xdr_syscall, &which));
		ASSERT(xdr_itimerval(xdr_syscall, value));
		ASSERT(xdr_ptr(xdr_syscall, (caddr_t *) &ovalue, timeval_xdr));

		/*
		** rval = setitimer( which, value, ovalue );
		*/
		rval = CONDOR_NotSupported();

		PUT_RVAL(xdr_syscall, rval);

		ASSERT(xdr_ptr(xdr_syscall, (caddr_t *) &ovalue, timeval_xdr));
		END_RVAL(xdr_syscall, 0);
	}

	case CONDOR_setrlimit:		/* setrlimit( int resource,
								**			M struct rlimit *rlp )
								*/
	{
		int resource;
		struct rlimit RLim, *rlp = &RLim;
		
		ASSERT(xdr_int(xdr_syscall, &resource));
		ASSERT(xdr_rlimit(xdr_syscall, rlp));

		rval = setrlimit( resource, rlp );

		PUT_RVAL(xdr_syscall, rval);
		END_RVAL(xdr_syscall, 0);
	}

	case CONDOR_getdirentries:	/* getdirentries( int fd,
								**				R char *buf,
								**				  int nbytes,
								**				R off_t *basep )
								*/
	{
		int fd;
		char *buf;
		int nbytes;
		long Base, *basep = &Base;

		ASSERT(xdr_int(xdr_syscall, &fd));
		ASSERT(xdr_int(xdr_syscall, &nbytes));

		MemToFree = buf = MALLOC( (unsigned)nbytes );
#if defined(SYS_getdirentries) && !defined(SUNOS40) && !defined(SUNOS41) && !defined(CMOS)
		rval = getdirentries( fd, buf, (unsigned)nbytes, basep );
#else defined(SYS_getdirentries) && !defined(SUNOS40) && !defined(SUNOS41)
		rval = CONDOR_NotSupported();
#endif defined(SYS_getdirentries) && !defined(SUNOS40) && !defined(SUNOS41)

		PUT_RVAL(xdr_syscall, rval);

		ASSERT(xdr_direntries(xdr_syscall, buf, rval));
		ASSERT(xdr_off_t(xdr_syscall, basep));

		END_RVAL(xdr_syscall, 0);
	}

	case CONDOR_getgroups:		/* getgroups( int gidsetlen, R int *gidset ) */
	{
		int gidsetlen;
		int GidSet[ NGROUPS ], *gidset = GidSet;
		register int i;

		ASSERT(xdr_int(xdr_syscall, &gidsetlen));

		rval = getgroups( gidsetlen, gidset );

		PUT_RVAL(xdr_syscall, rval);

		for( i = 0; i < rval; i++ ) {
			ASSERT(xdr_int(xdr_syscall, &gidset[i]));
		}
		END_RVAL(xdr_syscall, 0);
	}

	case CONDOR_getitimer:		/* getitimer( int which,
								**			R struct itimerval *value )
								*/
		RSC_1INT_1RBUF(getitimer,itimerval,xdr_itimerval);

	case CONDOR_getrlimit:		/* getrlimit( int resource,
								**			R struct rlimit *rlp )
								*/
		RSC_1INT_1RBUF(getrlimit,rlimit,xdr_rlimit);

	case CONDOR_getrusage:		/* getrusage( int who,
								**			R struct rusage *rusage )
								*/
		/*RSC_1INT_1RBUF(getrusage,rusage,xdr_rusage);*/
	{
		struct rusage *total_rusage;
		int				who;

		ASSERT(xdr_int(xdr_syscall, &who));

		/* 
		** Actual rusage to this point in time is rusage job started 
		** with(JobRusage) plus rusage accumulated since then(AccumRusage).
		*/
		MemToFree = MALLOC( (unsigned) sizeof(struct rusage) );
		total_rusage = (struct rusage *) MemToFree;
		bzero(total_rusage, sizeof(struct rusage));
		D_RUSAGE( D_SYSCALLS, &Proc.remote_usage );
		D_RUSAGE( D_SYSCALLS,  &AccumRusage );

		update_rusage(total_rusage, &Proc.remote_usage);
		update_rusage(total_rusage, &AccumRusage);
		D_RUSAGE( D_SYSCALLS, total_rusage );

		rval = 0;

		dprintf(D_SYSCALLS, "Shadow: getrusage(%d, 0x%x ) = %d\n",
				who, total_rusage, rval);

		PUT_RVAL(xdr_syscall, rval);
		ASSERT(xdr_rusage(xdr_syscall, total_rusage));
		END_RVAL(xdr_syscall, 0);
	}

	case CONDOR_fstat:			/* fstat( int fd, R struct stat *buf )       */
		RSC_1INT_1RBUF(fstat,stat,xdr_stat);

#if !defined(ultrix)
	case CONDOR_fstatfs:		/* fstatfs( int fd, struct statfs *buf )     */
		RSC_1INT_1RBUF(fstatfs,statfs,xdr_statfs);
#endif !defined(ultrix)

#if !defined(HPUX8)
	case CONDOR_utimes:			/* utimes( char *file,
								**			M struct timeval *tvp )
								*/
	{
		char File[ MAXPATHLEN ], *file = File;
		struct timeval tvp[2];

		ASSERT(xdr_string(xdr_syscall, &file, (u_int) MAXPATHLEN));
		ASSERT(timeval_xdr(xdr_syscall, &tvp[0]));
		ASSERT(timeval_xdr(xdr_syscall, &tvp[1]));

		rval = utimes( file, tvp );

		PUT_RVAL(xdr_syscall, rval);
		END_RVAL(xdr_syscall, 0);
	}
#endif

	case CONDOR_readlink:		/* readlink( char *path, char *buf,
								**					int bufsiz )
								*/
	{
		char Path[ MAXPATHLEN ], *path = Path;
		char *buf;
		int bufsiz;

		ASSERT(xdr_string(xdr_syscall, &path, (u_int) MAXPATHLEN));
		ASSERT(xdr_int(xdr_syscall, &bufsiz));

		MemToFree = buf = MALLOC( (unsigned)bufsiz );
		rval = readlink( path, buf, bufsiz );

		PUT_RVAL(xdr_syscall, rval);
		ASSERT(xdr_string(xdr_syscall, &buf, (u_int) bufsiz));
		END_RVAL(xdr_syscall, 0);
	}

	case PSEUDO_extern_name:		/* extern_name( char *path, char *buf,
								**					int bufsiz )
								*/
	{
		char Path[ MAXPATHLEN ], *path = Path;
		char *buf;
		int bufsiz;

		ASSERT(xdr_string(xdr_syscall, &path, (u_int) MAXPATHLEN));
		ASSERT(xdr_int(xdr_syscall, &bufsiz));

		MemToFree = buf = MALLOC( (unsigned)bufsiz );
		rval = external_name( path, buf, bufsiz );

		dprintf(D_SYSCALLS, "Shadow: extern_name(%s) = \"%s\"\n",
															Path, buf );


		PUT_RVAL(xdr_syscall, rval);
		ASSERT(xdr_string(xdr_syscall, &buf, (u_int) bufsiz));
		END_RVAL(xdr_syscall, 0);
	}

	case CONDOR_adjtime:		/* adjtime( M struct timeval *delta,
								**		  R/0 struct timeval *olddelta )
								*/
	{
		struct timeval Delta, *delta = &Delta;
		struct timeval OldDelta, *olddelta = &OldDelta;

		ASSERT(timeval_xdr(xdr_syscall, delta));
		ASSERT(xdr_ptr(xdr_syscall, (caddr_t *) &olddelta, timeval_xdr));

#if defined(DYNIX) || defined(HPUX8)
		rval = -1;
		errno = ESYSNOTSUPP;
#else
		rval = adjtime( delta, olddelta );
#endif !sequent

		PUT_RVAL(xdr_syscall, rval);

		ASSERT(xdr_ptr(xdr_syscall, (caddr_t *) &olddelta, timeval_xdr));

		END_RVAL(xdr_syscall, 0);
	}

#if !defined(ultrix)
	case CONDOR_statfs:			/* statfs( char *path, R struct statfs *buf )*/
		RSC_1STR_1RBUF(statfs,statfs,xdr_statfs);
#endif !defined(ultrix)

	case CONDOR_write:			/* write( int, M caddr_t, int )              */
	{
		int d, nbytes;
		char *buf = 0;

		ASSERT(xdr_int(xdr_syscall, &d));
		ASSERT(xdr_int(xdr_syscall, &nbytes));
		ASSERT(xdr_bytes(xdr_syscall, &buf, &nbytes, nbytes));

		rval = write( d, buf, nbytes );

		dprintf(D_SYSCALLS, "Shadow: write(%d, 0x%x, %d) = %d\n",
				d, buf, nbytes, rval);

		PUT_RVAL(xdr_syscall, rval);

		xdr_syscall->x_op = XDR_FREE;

		ASSERT(xdr_bytes(xdr_syscall, &buf, &nbytes, nbytes));

		END_RVAL(xdr_syscall, 0);
	}

	case CONDOR_read:			/* read( int, R caddr_t, int )               */
	{
		int d, nbytes;
		char *buf;

		ASSERT(xdr_int(xdr_syscall, &d));
		ASSERT(xdr_int(xdr_syscall, &nbytes));

		MemToFree = buf = MALLOC( (unsigned)nbytes );
		if( buf == NULL ) {
			rval = -1;
		} else {
			rval = read(d, buf, nbytes);
		}

		dprintf(D_SYSCALLS, "Shadow: read(%d, buf, %d) = %d\n",
				d, nbytes, rval);

		PUT_RVAL(xdr_syscall, rval);

		ASSERT(xdr_bytes(xdr_syscall, &buf, &nbytes, nbytes));

		END_RVAL(xdr_syscall, 0);
	}
#if defined(HPUX8)
	case CONDOR_utssys:
	{
		char	*buf;
		int		arg_2;
		int		cmd;
		int		len;

		ASSERT(xdr_int(xdr_syscall,&cmd));
		dprintf( D_SYSCALLS, "Shadow: _utssys( ... ), cmd = %d\n", cmd );

		switch( cmd ) {
			case 0: /* uname(struct utsname *name) - return name */
				len = sizeof( struct utsname );
				buf = MALLOC( sizeof(struct utsname) );
				rval = _utssys( buf, 0, cmd );
				dprintf( D_SYSCALLS, "Shadow: _utssys( 0x%x, %d, %d)= %d\n",
					buf, arg_2, cmd );
				PUT_RVAL(xdr_syscall,rval);
				ASSERT(xdr_bytes(xdr_syscall,&buf,&len,len));
				FREE( buf );
				break;
			case 2: /* ustat(int dev, struct ustat *buf) - get dev,return buf */
				ASSERT(xdr_int(xdr_syscall,&arg_2));
				buf = MALLOC( sizeof(struct ustat) );
				rval = _utssys( buf, arg_2, cmd );
				dprintf( D_SYSCALLS, "Shadow: _utssys( 0x%x, %d, %d)= %d\n",
					buf, arg_2, cmd );
				PUT_RVAL(xdr_syscall,rval);
				len = sizeof(struct ustat);
				ASSERT(xdr_bytes(xdr_syscall,&buf,&len,len));
				FREE( buf );
				break;
			case 3: /* setuname( struct utsname *name ) - get name */
				len = sizeof( struct utsname );
				buf = MALLOC( len );
				ASSERT(xdr_bytes(xdr_syscall,&buf,&len,len));
				rval = _utssys( buf, arg_2, cmd );
				dprintf( D_SYSCALLS, "Shadow: _utssys( 0x%x, %d, %d)= %d\n",
					buf, arg_2, cmd );
				PUT_RVAL(xdr_syscall,rval);
				FREE( buf );
				break;
			case 4: /* sethostname( char *name, int len ) /* get len, name */
				ASSERT(xdr_int(xdr_syscall,&arg_2));
				buf = MALLOC( arg_2 );		
				ASSERT(xdr_bytes(xdr_syscall,&buf,&arg_2,arg_2));
				rval = _utssys( buf, arg_2, cmd );
				dprintf( D_SYSCALLS, "Shadow: _utssys( 0x%x, %d, %d)= %d\n",
					buf, arg_2, cmd );
				PUT_RVAL(xdr_syscall,rval);
				FREE( buf );
				break;
			case 5: /* gethostname( char *name, int len ); /* return name,len */
				ASSERT(xdr_int(xdr_syscall,&arg_2));
				buf = MALLOC( arg_2 );
				rval = _utssys( buf, arg_2, cmd );
				dprintf( D_SYSCALLS, "Shadow: _utssys( 0x%x, %d, %d)= %d\n",
					buf, arg_2, cmd );
				PUT_RVAL(xdr_syscall,rval);
				ASSERT(xdr_bytes(xdr_syscall,&buf,&arg_2,arg_2));
				FREE( buf );
				break;
		}
		END_RVAL( xdr_syscall, 0 );
	}
#endif

	case CONDOR_ioctl:			/* ioctl( int d, u_long request, char *argp )*/
	{
		int d;
		char *argp = 0;
		u_long request;

		ASSERT(xdr_int(xdr_syscall, &d));
		ASSERT(xdr_u_long(xdr_syscall, &request));
#if defined(SUNOS41) || defined(ULTRIX42) || defined(CMOS)
		if( request & _IOC_IN ) {
			MemToFree = argp = MALLOC( (unsigned)(request & _IOCPARM_MASK) );
			if( argp == NULL ) {
				EXCEPT("ioctl: tried to allocate %d bytes\n",
							request & _IOCPARM_MASK);
			}
			ASSERT(xdr_opaque(xdr_syscall,argp,(u_int)(request&_IOCPARM_MASK)));
		}
#else defined(SUNOS41) || defined(ULTRIX42) || defined(CMOS)
		if( request & IOC_IN ) {
			MemToFree = argp = MALLOC( (unsigned)(request & IOCPARM_MASK) );
			if( argp == NULL ) {
				EXCEPT("ioctl: tried to allocate %d bytes\n",
							request & IOCPARM_MASK);
			}
			ASSERT(xdr_opaque(xdr_syscall,argp,(u_int)(request&IOCPARM_MASK)));
		}
#endif defined(SUNOS41) || defined(ULTRIX42) || defined(CMOS)

		/*
		**	rval = ioctl( d, request, argp );
		*/
		rval = CONDOR_NotSupported();

		dprintf(D_SYSCALLS, "Shadow: ioctl(%d, 0x%x, 0x%x) = %d\n",
				d, request, argp, rval);

		PUT_RVAL(xdr_syscall, rval);

#if defined(SUNOS41) || defined(ULTRIX42) || defined(CMOS)
		if( request & _IOC_OUT ) {
			ASSERT(xdr_opaque(xdr_syscall,argp,(u_int)(request&_IOCPARM_MASK)));
		}
#else defined(SUNOS41) || defined(ULTRIX42) || defined(CMOS)
		if( request & IOC_OUT ) {
			ASSERT(xdr_opaque(xdr_syscall,argp,(u_int)(request&IOCPARM_MASK)));
		}
#endif defined(SUNOS41) || defined(ULTRIX42) || defined(CMOS)

		END_RVAL(xdr_syscall, 0);
	}


#ifdef IMPLEMENTED
	/*
	**	System calls with (as yet) unknown arguments
	*/
	case CONDOR_exportfs:
	case CONDOR_getdopt:
	case CONDOR_getfh:
	case CONDOR_madvise:
	case CONDOR_mincore:
	case CONDOR_mmap:
	case CONDOR_mprotect:
	case CONDOR_mremap:
	case CONDOR_munmap:
	case CONDOR_setdopt:
	case CONDOR_sstk:
		break;


	/*
	**	Other system calls
	*/

		break;


	case CONDOR_writev:			/* writev( int d, struct iovec *iov,
								**					int iovcnt )
								*/
	case CONDOR_readv:			/* readv( int d, struct iovec *iov,
								**					int iovcnt )
								*/

	case CONDOR_ptrace:			/* ptrace( int, int, VR int *, int           */
	case CONDOR_quotactl:		/* quotaactl( int, str, int, VRU caddr_t )   */
	case CONDOR_execv:			/* execv( str, M str[] )                     */
	case CONDOR_execve:			/* execve( str, M str[], M str[] )           */

	case CONDOR_send:			/* send( int, M caddr_t, int, int )          */
	case CONDOR_sendto:			/* sendto( int, M caddr_t, int, int,
										M sockaddr *, int                    */
	case CONDOR_sendmsg:		/* sendmsg( int, V msghdr[], int )           */

	case CONDOR_recv:			/* recv( int, R caddr_t, int, int )          */
	case CONDOR_recvfrom:		/* recvfrom( int, R caddr_t, int, int,
											0/R sockaddr *, VR int * )       */
	case CONDOR_recvmsg:		/* recvmsg( int, VR msghdr[], int )          */

	case CONDOR_setsockopt:		/* setsockopt(int, int, int, M caddr_t, int) */
	case CONDOR_getsockopt:		/* getsockopt( int, int, int,
								**		R caddr_t, VR int * )
								*/
	case CONDOR_socketpair:		/* socketpair( int, int, int, R int[2] )     */
	case CONDOR_bind:			/* bind( int, sockaddr *, int )              */
	case CONDOR_connect:		/* connect( int, sockaddr *, int )           */
	case CONDOR_mount:			/* mount( int, str, int,
								**		MU caddr_t )
								*/

	case CONDOR_accept:			/* accept( int, R sockaddr *, VR int * )     */
	case CONDOR_getsockname:	/* getsockname( int, R sockaddr *, VR int * )*/
	case CONDOR_getpeername:	/* getpeername( int, R sockaddr *, VR int * )*/
		break;
#endif IMPLEMENTED

	default:
		dprintf(D_ALWAYS, "Shadow: ERROR: %d is an unknown system call\n",
			condor_sysnum);

		errno = EINVAL;

		PUT_RVAL(xdr_syscall, -1);
		END_RVAL(xdr_syscall, 0);
	}
}

#ifdef notdef
debug( str )
char	*str;
{
	dprintf( D_ALWAYS, str );
}

/*
** Convert a timeval struct to a float.
*/
float
tv_to_f( tv )
struct timeval	tv;
{
	return (float)tv.tv_sec + ((float)tv.tv_usec)/(float)1000000;
}
#endif notdef


char MsgBuf[ BUFSIZ ];
HandleChildLog( log_fp )
FILE	*log_fp;
{
	register char *nlp;
	char *index();
	char buf[ BUFSIZ ];

	for(;;) {
		bzero(buf, sizeof(buf));

		if( fgets(buf, sizeof(buf), log_fp) == NULL ) {
			return -1;
		}

		nlp = index(buf, '\n');
		if( nlp != NULL ) {
			*nlp = '\0';
		}

		dprintf(D_ALWAYS|D_NOHEADER, "%s\n", buf );
		if( strncmp("ERROR",buf,5) == 0 ) {
			(void)strcpy( MsgBuf, buf );
		}

		/*
		**	If there is more to read, read it, otherwise just return...
		*/
		if( log_fp->_cnt == 0 ) {
			return 0;
		}
	}
}

HandleLog()
{
	register char *nlp;
	char buf[ BUFSIZ ];
	char *index();

	/*
	** WARNING!  We assume here that the log messages will be sent
	** in chunks that consist of complete lines of text, terminated
	** by a newline.  We may become deadlocked if some program sends
	** us a chunk like "This is a line of text\n ".
	*/

	for(;;) {
		bzero(buf, sizeof(buf));

		while( fgets(buf, sizeof(buf), LogFP) == NULL ) {
			if( errno != EINTR ) {
				EXCEPT("Peer went away.  buf = '%s'", buf);
			}
		}


		nlp = index(buf, '\n');
		if( nlp != NULL ) {
			*nlp = '\0';
		}

		dprintf(D_ALWAYS, "Read: %s\n", buf );

		/*
		**	If there is more to read, read it, otherwise just return...
		*/
		if( LogFP->_cnt == 0 ) {
			return;
		}
	}
}

RSC_simple( rval, condor_sysnum )
int		rval;
int		condor_sysnum;
{
	dprintf(D_SYSCALLS, "Shadow: %s() = %d\n",
			CONDOR_SYSCALLNAME(condor_sysnum), rval);

	PUT_RVAL(xdr_syscall, rval);
	END_RVAL(xdr_syscall, 0);
}
	

RSC_no_args( func, condor_sysnum )
int (*func)();
int condor_sysnum;
{
	int rval = (*func)();

	dprintf(D_SYSCALLS, "Shadow: %s() = %d\n",
			CONDOR_SYSCALLNAME(condor_sysnum), rval);


	PUT_RVAL(xdr_syscall, rval);
	END_RVAL(xdr_syscall, 0);
}

RSC_1int(func, condor_sysnum)
int (*func)();
int condor_sysnum;
{
	int rval;
	int arg1;

	ASSERT(xdr_int(xdr_syscall, &arg1));

	rval = (*func)( arg1 );

	dprintf(D_SYSCALLS, "Shadow: %s(%d) = %d\n",
		CONDOR_SYSCALLNAME(condor_sysnum), arg1, rval);

	PUT_RVAL(xdr_syscall, rval);
	END_RVAL(xdr_syscall, 0);
}

RSC_2int(func, condor_sysnum)
int (*func)();
int condor_sysnum;
{
	int rval;
	int arg1, arg2;

	ASSERT(xdr_int(xdr_syscall, &arg1));
	ASSERT(xdr_int(xdr_syscall, &arg2));

	rval = (*func)( arg1, arg2 );

	dprintf(D_SYSCALLS, "Shadow: %s(%d, %d) = %d\n",
		CONDOR_SYSCALLNAME(condor_sysnum), arg1, arg2, rval);

	PUT_RVAL(xdr_syscall, rval);
	END_RVAL(xdr_syscall, 0);
}

extern int DebugFlags;
RSC_3int(func, condor_sysnum)
int (*func)();
int condor_sysnum;
{
	int rval;
	int arg1, arg2, arg3;
	int	saved_flags;

	ASSERT(xdr_int(xdr_syscall, &arg1));
	ASSERT(xdr_int(xdr_syscall, &arg2));
	ASSERT(xdr_int(xdr_syscall, &arg3));

	rval = (*func)( arg1, arg2, arg3 );

#if defined(AIX31) || defined(AIX32)
	if( condor_sysnum == CONDOR_kfcntl ) {
		dprintf(D_SYSCALLS, "Shadow: %s(%d, ",
			CONDOR_SYSCALLNAME(condor_sysnum), arg1 );
		saved_flags = DebugFlags;
		DebugFlags |= D_NOHEADER;
		switch( arg2 ) {
			case F_DUPFD:
				dprintf(D_SYSCALLS, "F_DUPFD, " );
				break;
			case F_GETFD:
				dprintf(D_SYSCALLS, "F_GETFD, " );
				break;
			case F_SETFD:
				dprintf(D_SYSCALLS, "F_SETFD, " );
				break;
			case F_GETFL:
				dprintf(D_SYSCALLS, "F_GETFL, " );
				break;
			case F_SETFL:
				dprintf(D_SYSCALLS, "F_SETFL, " );
				break;
			case F_GETLK:
				dprintf(D_SYSCALLS, "F_GETLK, " );
				break;
			case F_SETLK:
				dprintf(D_SYSCALLS, "F_SETLK, " );
				break;
			case F_SETLKW:
				dprintf(D_SYSCALLS, "F_SETLKW, " );
				break;
			case F_GETOWN:
				dprintf(D_SYSCALLS, "F_GETOWN, " );
				break;
			case F_SETOWN:
				dprintf(D_SYSCALLS, "F_SETOWN, " );
				break;
			case F_CLOSEM:
				dprintf(D_SYSCALLS, "F_CLOSEM, " );
				break;
			default:
				dprintf(D_SYSCALLS, "(UNKNOWN), " );
		}

		dprintf(D_SYSCALLS, "%d) = %d\n", arg3, rval );
		DebugFlags = saved_flags;
	} else {
		dprintf(D_SYSCALLS, "Shadow: %s(%d, %d, %d) = %d\n",
			CONDOR_SYSCALLNAME(condor_sysnum), arg1, arg2, arg3, rval);
	}
#else /* AIX31 or AIX32 */
	dprintf(D_SYSCALLS, "Shadow: %s(%d, %d, %d) = %d\n",
		CONDOR_SYSCALLNAME(condor_sysnum), arg1, arg2, arg3, rval);
#endif /* AIX31 or AIX32 */

	PUT_RVAL(xdr_syscall, rval);
	END_RVAL(xdr_syscall, 0);
}

RSC_1str(func, condor_sysnum)
int (*func)();
int condor_sysnum;
{
	char str1[ MAXPATHLEN ], *s1 = str1;
	int rval;

	ASSERT(xdr_string(xdr_syscall, &s1, (u_int) MAXPATHLEN));

	rval = (*func)(s1);

	dprintf(D_SYSCALLS, "Shadow: %s(%s) = %d\n",
		CONDOR_SYSCALLNAME(condor_sysnum), s1, rval );

	PUT_RVAL(xdr_syscall, rval);
	END_RVAL(xdr_syscall, 0);
}

RSC_1str_1int(func, condor_sysnum)
int (*func)();
int condor_sysnum;
{
	char str1[ MAXPATHLEN ], *s1 = str1;
	int int2;
	int rval;

	ASSERT(xdr_string(xdr_syscall, &s1, (u_int) MAXPATHLEN));
	ASSERT(xdr_int(xdr_syscall, &int2));

	rval = (*func)(s1, int2);

	dprintf(D_SYSCALLS, "Shadow: %s(%s, 0%o) = %d\n",
			CONDOR_SYSCALLNAME(condor_sysnum), s1, int2, rval );

	PUT_RVAL(xdr_syscall, rval);
	END_RVAL(xdr_syscall, 0);
}

RSC_1str_2int(func, condor_sysnum)
int (*func)();
int condor_sysnum;
{
	char str1[ MAXPATHLEN ], *s1 = str1;
	int int2, int3;
	int rval;
	int	save_errno;

	ASSERT(xdr_string(xdr_syscall, (char **)&s1, (u_int) MAXPATHLEN));
	ASSERT(xdr_int(xdr_syscall, &int2));
	ASSERT(xdr_int(xdr_syscall, &int3));

	rval = (*func)( s1, int2, int3 );
	save_errno = errno;

	dprintf(D_SYSCALLS, "Shadow: %s(%s, 0%o, 0%o) = %d, errno %d\n",
		CONDOR_SYSCALLNAME(condor_sysnum), s1, int2, int3, rval, save_errno );

	errno = save_errno;
	PUT_RVAL(xdr_syscall, rval);
	END_RVAL(xdr_syscall, 0);
}

RSC_2str(func, condor_sysnum)
int (*func)();
int condor_sysnum;
{
	char str1[ MAXPATHLEN ], *s1 = str1;
	char str2[ MAXPATHLEN ], *s2 = str2;
	int rval;

	ASSERT(xdr_string(xdr_syscall, (char **)&s1, (u_int) MAXPATHLEN));
	ASSERT(xdr_string(xdr_syscall, (char **)&s2, (u_int) MAXPATHLEN));

	rval = (*func)( s1, s2 );

	dprintf(D_SYSCALLS, "Shadow: %s(%s, %s) = %d\n",
			CONDOR_SYSCALLNAME(condor_sysnum), s1, s2, rval );

	PUT_RVAL(xdr_syscall, rval);
	END_RVAL(xdr_syscall, 0);
}

/*
**	System call handler for syscalls which take 1 integer
**	and return a buffer of size bufsiz.
*/
RSC_1int_1Rbuf(func, condor_sysnum, bufsiz, xdrf)
int (*func)();
int condor_sysnum, bufsiz;
int (*xdrf)();
{
	int arg1;
	caddr_t buf;
	int rval;

	ASSERT(xdr_int(xdr_syscall, &arg1));

	MemToFree = buf = MALLOC( (unsigned)bufsiz );

	rval = (*func)( arg1, buf );

	dprintf(D_SYSCALLS, "Shadow: %s(%d, 0x%x) = %d\n",
			CONDOR_SYSCALLNAME(condor_sysnum), arg1, buf, rval);

	PUT_RVAL(xdr_syscall, rval);
	ASSERT((*xdrf)(xdr_syscall, buf));
	END_RVAL(xdr_syscall, 0);
}

/*
**	System call handler for syscalls which take 1 string
**	and return a buffer of size bufsiz.
*/
RSC_1str_1Rbuf(func, condor_sysnum, bufsiz, xdrf)
int (*func)();
int condor_sysnum, bufsiz;
int (*xdrf)();
{
	char str1[ MAXPATHLEN ], *s1 = str1;
	caddr_t buf;
	int rval;

	ASSERT(xdr_string(xdr_syscall, (char **)&s1, (u_int) MAXPATHLEN));

	MemToFree = buf = MALLOC( (unsigned) bufsiz );

	rval = (*func)( s1, buf );

	dprintf(D_SYSCALLS, "Shadow: %s(%s, 0x%x) = %d\n",
			CONDOR_SYSCALLNAME(condor_sysnum), s1, buf, rval);

	PUT_RVAL(xdr_syscall, rval);
	ASSERT((*xdrf)(xdr_syscall, buf));
	END_RVAL(xdr_syscall, 0);
}

/*
Pseudo system call reporting virtual image size, to be used in calculating
future disk requirements.
*/
condor_image_size(d)
int d;
{
	extern PROC Proc;

	if( d > ImageSize ) {
		ImageSize = d;
		write_to_parent( SIZE, &ImageSize, sizeof(ImageSize) );
	}
	return 0;
}

/*
**	Make sure that they are not trying to
**	close a reserved descriptor.
*/
condor_close(d)
int d;
{
	if( d == RSC_SOCK || d == CLIENT_LOG ||
					d == PipeToParent || d == LogPipe ) {
		errno = EINVAL;
		return( -1 );
	}

	return( close(d) );
}

/*
**	Make sure that they are not trying to
**	dup to a reserved descriptor.
*/
condor_dup2(oldd, newd)
int oldd, newd;
{
	if( newd == RSC_SOCK || newd == CLIENT_LOG ||
					newd == PipeToParent || newd == LogPipe ) {
		errno = EINVAL;
		return( -1 );
	}

	return( dup2(oldd, newd) );
}

/*
**	We don't really want the shadow to exit here because the 
**	starter may want to make a new checkpoint file.  So, just
**	respond with an error-free return code and hang out; the
**	shadow will (should) go away when (if) the starter closes
**	the LogSock.
*/
/* ARGSUSED */
condor_exit(status)
int status;
{
	return( 0 );
}

/*
**	getppid normally returns the parents id, right?  Well in
**	the CONDOR case, this job may be started and checkpointed many
**	times.  The parent will change each time.  Instead we will
**	treat it as though the parent process died and this process
**	was inherited by init.
*/
condor_getppid()
{
	return( 1 );
}

/*
**	getpid should return the same number each time.  As indicated
**	above, this CONDOR job may be started and checkpointed several times
**	during its lifetime, we return the condor "process id".
*/
condor_getpid()
{
	extern PROC Proc;

	return( Proc.id.proc );
}

/*
**	getpgrp should return the same number each time.  As indicated
**	above, this CONDOR job may be started and checkpointed several times
**	during its lifetime, we return the condor "cluster id".
*/
#if defined(AIX31) || defined(AIX32)
condor_getpgrp()
#else
condor_getpgrp( pid )		/* Ignore it anyway */
int		pid;
#endif
{
	extern PROC Proc;

	return( Proc.id.cluster );
}

#if defined(AIX31) || defined(AIX32)
/*
**	kgetpgrp should return the same number each time.  As indicated
**	above, this CONDOR job may be started and checkpointed several times
**	during its lifetime, we return the condor "cluster id".  Since we
**	don't handle multi processing, should only be asking about own pgrp.
*/
condor_kgetpgrp( pid )
int		pid;
{
	extern PROC Proc;

	if( (pid != 0) && (pid != Proc.id.proc) ) {
		errno = EINVAL;
		return -1;
	}
	return( Proc.id.cluster );
}
#endif


#if defined(AIX31) || defined(AIX32)
/*
** Ignore which uid is wanted, just give out the condor client's uid
** in every case.
*/
condor_getuidx( which )
int		which;
{
	return ClientUid;
}
#endif

#if defined(AIX31) || defined(AIX32)
/*
** Ignore which gid is wanted, just give out the condor client's gid
** in every case.
*/
condor_getgidx( which )
int		which;
{
	return ClientGid;
}
#endif

/*VARARGS*/
CONDOR_NotSupported()
{
#ifdef NOTDEF
	errno = ESYSNOTSUPP;
	return( -1 );
#endif NOTDEF

	errno = 0;
	PERM_ERR( "%s - system call not supported by Condor",
									CONDOR_SYSCALLNAME(condor_sysnum) );
	EXCEPT( "Unsupported system call %s (%d)",
						CONDOR_SYSCALLNAME(condor_sysnum), condor_sysnum );
#ifdef LINT
	return -1;
#endif LINT
}
