/* 
** Copyright 1986, 1987, 1988, 1989, 1990, 1991 by the Condor Design Team
** 
** Permission to use, copy, modify, and distribute this software and its
** documentation for any purpose and without fee is hereby granted,
** provided that the above copyright notice appear in all copies and that
** both that copyright notice and this permission notice appear in
** supporting documentation, and that the names of the University of
** Wisconsin and the Condor Design Team not be used in advertising or
** publicity pertaining to distribution of the software without specific,
** written prior permission.  The University of Wisconsin and the Condor
** Design Team make no representations about the suitability of this
** software for any purpose.  It is provided "as is" without express
** or implied warranty.
** 
** THE UNIVERSITY OF WISCONSIN AND THE CONDOR DESIGN TEAM DISCLAIM ALL
** WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES
** OF MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE UNIVERSITY OF
** WISCONSIN OR THE CONDOR DESIGN TEAM BE LIABLE FOR ANY SPECIAL, INDIRECT
** OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS
** OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE
** OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE
** OR PERFORMANCE OF THIS SOFTWARE.
** 
** Authors:  Allan Bricker and Michael J. Litzkow,
** 	         University of Wisconsin, Computer Sciences Dept.
** 
*/ 


#include <stdio.h>
#include <signal.h>
#include <netdb.h>
#include <pwd.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/param.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <netinet/in.h>
#include <rpc/types.h>
#include <rpc/xdr.h>
#include "condor_types.h"
#include "sched.h"
#include "debug.h"
#include "trace.h"
#include "except.h"
#include "expr.h"
#include "manager.h"
#include "clib.h"

#ifndef LINT
static char *_FileName_ = __FILE__;		/* Used by EXCEPT (see except.h)     */
#endif LINT

CONTEXT		*create_context();
EXPR		*build_expr();
char		*strdup();
MACH_REC	*find_mach_rec(), *create_mach_rec();
bool_t 		xdr_int();


MACH_REC	*MachineList;

STATUS_LINE	*StatusLines[1024];
int			N_StatusLines;
int			MachineUpdateInterval;

extern int StablePriosInited;
extern char *gethostnamebyaddr();

int StatusLineErrors;

negotiator_info( xdrs )
XDR		*xdrs;
{
	PRIO_REC	record, *rec = &record;

	dprintf( D_ALWAYS, "Getting new priority info\n" );
	StatusLineErrors = 0;


	dprintf( D_FULLDEBUG, "******** Begin Read loop  ********\n" );
	for(;; ) {
		bzero( (char *)rec, sizeof(PRIO_REC) );
		if( !xdr_prio_rec(xdrs,rec) ) {
			dprintf( D_ALWAYS, "Can't read prio_rec from negotiator\n" );
			break;
		}
		if( rec->name == NULL ) {
			dprintf( D_FULLDEBUG, "Got END MARKER\n" );
			break;
		}
		if(  rec->name[0] == '\0' ) {
			dprintf( D_FULLDEBUG, "Got END MARKER\n" );
			FREE( rec->name );
			break;
		}
		update_prio( rec );
		dprintf( D_FULLDEBUG, "Got %s = %d\n", rec->name, rec->prio );
		FREE( rec->name );
	}
	dprintf( D_FULLDEBUG, "******** End Read loop  ********\n" );
	store_prios();
	event_mgr();
	/*
	dprintf( D_ALWAYS, "Errors  = %d, Lines = %d\n",
		StatusLineErrors, N_StatusLines );
	if( StatusLineErrors ) {
		dump_status_lines();
	}
	*/
}

update_status_prio( rec )
PRIO_REC *rec;
{
	int	i;

	/*
	** Update status lines.
	*/
	for( i=0; i<N_StatusLines; i++ ) {
		if( StatusLines[i] == (STATUS_LINE *)0 ) {
			continue;
		}
		if( strcmp(StatusLines[i]->name,rec->name) == MATCH ) {
			StatusLines[i]->prio = rec->prio;
			return;
		}
	}

	dprintf( D_ALWAYS, "Can't find status line for \"%s\"\n", rec->name );
	StatusLineErrors += 1;
}

update_mach_list_prio( rec )
PRIO_REC    *rec;
{
	MACH_REC *ptr;

	/*
	** Update machine records.
	*/
	for( ptr = MachineList->next; ptr->name; ptr = ptr->next ) {
		if( strcmp(ptr->name,rec->name) == MATCH ) {
			ptr->prio = rec->prio;
			return;
		}
	}
	dprintf( D_ALWAYS, "Can't find machine record for \"%s\"\n", rec->name );
}

update_prio( rec )
PRIO_REC	*rec;
{
	update_status_prio( rec );

	update_mach_list_prio( rec );
}

STATUS_LINE	Endmarker;
give_status_lines( xdrs )
XDR	*xdrs;
{
	int		i;

	if( !xdrrec_skiprecord(xdrs) ) {
		dprintf( D_ALWAYS, "Can't send mach_rec to client\n" );
		return;
	}

	xdrs->x_op = XDR_ENCODE;

	for( i=0; i<N_StatusLines; i++ ) {
		if( StatusLines[i] == (STATUS_LINE *)0 ) {
			continue;
		}
		if( !xdr_status_line(xdrs,StatusLines[i]) ) {
			dprintf( D_ALWAYS, "Can't send StatusLine to client\n" );
			return;
		}
	}
	if( !xdr_status_line(xdrs,&Endmarker) ) {
		dprintf( D_ALWAYS, "Can't send Endmarker to client\n" );
		return;
	}

	(void)xdrrec_endofrecord( xdrs, TRUE );
}

machine_info( xdrs, from, kind )
XDR		*xdrs;
struct sockaddr_in	*from;
int		kind;
{
	MACH_REC	*rec;
	PRIO_REC prio_rec;
	int stored_prio = 0;

	prio_rec.name = NULL;

	if( (rec = find_mach_rec(from)) == NULL ) {
		rec = create_mach_rec( from );
		insque( (struct qelem *)rec, (struct qelem *)MachineList );

		if( !StablePriosInited ) {
			init_stable_prios();
		}
		/*
		** Only update priorities for machines with a non-zero stored priority.
		*/
		if( (stored_prio = retrieve_prio(&from->sin_addr)) != 0 ) {
			prio_rec.name = gethostnamebyaddr(&from->sin_addr);
			prio_rec.prio = stored_prio;
			if(prio_rec.name) {
				update_mach_list_prio(&prio_rec);
			}
		}
		dprintf( D_ALWAYS, "Creating new record for \"%s\"\n", rec->name );
	} else {
		/*
		*/
		if( kind == STARTD_INFO ) {
			dprintf( D_ALWAYS, "STARTD \"%s\"\n", rec->name );
		} else {
			dprintf( D_ALWAYS, "SCHEDD \"%s\"\n", rec->name );
		}
	}

	if( rec->machine_context == NULL ) {
		rec->machine_context = create_context();
	}
	if( !xdr_context(xdrs,rec->machine_context) ) {
		dprintf( D_ALWAYS, "Error reading machine context from \"%s\"\n",
																rec->name );
		delete_mach_rec( rec );
		if(prio_rec.name)
			free(prio_rec.name);
		dprintf( D_ALWAYS, "Deleted machine record for \"%s\"\n", rec->name );
		return;
	}
	rec->time_stamp = time( (time_t *)0 );

	if( kind == STARTD_INFO ) {
		rec->busy = FALSE;
	}

	update_status_line( rec );
	if(prio_rec.name) {
		update_status_prio(&prio_rec);
		free(prio_rec.name);
	}
}

update_status_line( rec )
MACH_REC	*rec;
{
	int			idle;
	STATUS_LINE	*line = rec->line;

	if( line->name == NULL ) {
		line->name = strdup( rec->name );
	}

	(void) evaluate_int( "Running", &line->run, rec->machine_context,
															(CONTEXT *)0 );
	(void) evaluate_int( "Idle", &idle, rec->machine_context, (CONTEXT *)0 );
	line->tot = line->run + idle;
	if( line->state ) {
		FREE( line->state );
	}
	(void) evaluate_string( "State", &line->state, rec->machine_context,
																(CONTEXT *)0 );
	(void) evaluate_float( "LoadAvg", &line->load_avg, rec->machine_context,
																(CONTEXT *)0 );
	(void) evaluate_int( "KeyboardIdle", &line->kbd_idle, rec->machine_context,
																(CONTEXT *)0);
	if( line->arch == NULL ) {
		(void) evaluate_string( "Arch", &line->arch, rec->machine_context,
																(CONTEXT *)0 );
	}
	if( line->op_sys == NULL ) {
		(void) evaluate_string( "OpSys", &line->op_sys, rec->machine_context,
																(CONTEXT *)0 );
	}
}

MACH_REC	*
find_mach_rec( from )
struct sockaddr_in	*from;
{
	MACH_REC	*ptr;

	for( ptr = MachineList->next; ptr->name; ptr = ptr->next ) {
		if( ptr->net_addr.s_addr == from->sin_addr.s_addr ) {
			return ptr;
		}
	}
	return NULL;
}

STATUS_LINE	*
create_status_line()
{
	STATUS_LINE	*line;
	int		i;

	line = (STATUS_LINE *)CALLOC( 1, sizeof(STATUS_LINE) );

	for( i=0; i<N_StatusLines; i++ ) {
		if( StatusLines[i] == (STATUS_LINE *)0 ) {
			StatusLines[i] = line;
			return line;
		}
	}

	StatusLines[ N_StatusLines++ ] = line;
	return line;
}

delete_status_line( line )
STATUS_LINE	*line;
{
	int		i;

	for( i=0; i<N_StatusLines; i++ ) {
		if( StatusLines[i] == line ) {
			StatusLines[i] = (STATUS_LINE *)0;
			FREE( (char *)line );
			return;
		}
	}
	EXCEPT( "Cannot find status line pointed to by 0x%x\n", line );
}

MACH_REC	*
create_mach_rec( from )
struct sockaddr_in	*from;
{
	MACH_REC	*answer;
	ELEM		tmp;
	struct hostent	*hp, *gethostbyaddr();

	answer = (MACH_REC *)CALLOC( 1, sizeof(MACH_REC) );

	answer->next = answer;
	answer->prev = answer;
	answer->machine_context = create_context();
	if( !from ) {
		/*
		dprintf( D_ALWAYS, "Created NULL machine record\n" );
		*/
		return answer;
	}

	answer->net_addr = from->sin_addr;
	answer->net_addr_type = from->sin_family;
	answer->time_stamp = time( (time_t *)0 );

	if( (hp=gethostbyaddr((char *)&from->sin_addr,sizeof(struct in_addr),
												from->sin_family)) == NULL ) {
		dprintf( D_ALWAYS, "Can't find host name for \"%s\"\n",
											inet_ntoa(from->sin_addr) );



		/*
		free_mach_rec( answer );
		return NULL;
		*/
	}
	if( hp ) {
		answer->name = strdup( hp->h_name );
	} else {
		answer->name = strdup( inet_ntoa(from->sin_addr) );
	}

	tmp.type = INT;
	tmp.i_val = 0;
	store_stmt( build_expr("Users",&tmp), answer->machine_context );
	store_stmt( build_expr("Running",&tmp), answer->machine_context );
	store_stmt( build_expr("Idle",&tmp), answer->machine_context );
	store_stmt( build_expr("Prio",&tmp), answer->machine_context );

	answer->line = create_status_line();
	return answer;
}

delete_mach_rec( rec )
MACH_REC	*rec;
{
	rec->next->prev = rec->prev;
	rec->prev->next = rec->next;
	free_mach_rec( rec );
}

free_mach_rec( rec )
MACH_REC	*rec;
{
	if( rec->name ) {
		FREE( rec->name );
	}

	if( rec->machine_context ) {
		free_context( rec->machine_context );
	}

	if( rec->line ) {
		delete_status_line( rec->line );
	}

	FREE( (char *)rec );
}

give_status( xdrs )
XDR		*xdrs;
{
	MACH_REC	*ptr;


	if( !xdrrec_skiprecord(xdrs) ) {
		dprintf( D_ALWAYS, "Can't send mach_rec to client\n" );
		return;
	}

	dprintf( D_FULLDEBUG, "++++++++  Begin Send loop  ++++++++\n" );
	xdrs->x_op = XDR_ENCODE;
	for( ptr=MachineList->next; ptr->name; ptr = ptr->next ) {
		if( (int)time( (time_t *)0 ) - ptr->time_stamp >
												MachineUpdateInterval ) {
				/* Doesn't belong here, but it's the only place we
				   traverse the entire machine list. */
			FREE( ptr->line->state );
			ptr->line->state = strdup( "(DOWN)" );
		}

		if( !xdr_mach_rec(xdrs,ptr) ) {
			dprintf( D_ALWAYS, "Can't send mach_rec to client\n" );
			return;
		}
		dprintf( D_FULLDEBUG, "Sent mach_rec for \"%s\"\n", ptr->name );
	}

	if( !xdr_mach_rec(xdrs,ptr) ) {
		dprintf( D_ALWAYS, "Can't send mach_rec to client\n" );
		return;
	}
	dprintf( D_FULLDEBUG, "Sent END MARKER\n" );
	dprintf( D_FULLDEBUG, "++++++++  End Send Loop  +++++++++\n" );

	xdrrec_endofrecord( xdrs, TRUE );
}

dump_status_lines()
{
	int		i;
	STATUS_LINE	*line;

	for( i=0; i<N_StatusLines; i++ ) {
		if( StatusLines[i] == (STATUS_LINE *)0 ) {
			continue;
		}
		line = StatusLines[i];
		dprintf( D_ALWAYS, "\"%s\": %d\n", line->name, line->prio );
	}
}
