/* 
** Copyright 1986, 1987, 1988, 1989, 1990, 1991 by the Condor Design Team
** 
** Permission to use, copy, modify, and distribute this software and its
** documentation for any purpose and without fee is hereby granted,
** provided that the above copyright notice appear in all copies and that
** both that copyright notice and this permission notice appear in
** supporting documentation, and that the names of the University of
** Wisconsin and the Condor Design Team not be used in advertising or
** publicity pertaining to distribution of the software without specific,
** written prior permission.  The University of Wisconsin and the Condor
** Design Team make no representations about the suitability of this
** software for any purpose.  It is provided "as is" without express
** or implied warranty.
** 
** THE UNIVERSITY OF WISCONSIN AND THE CONDOR DESIGN TEAM DISCLAIM ALL
** WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES
** OF MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE UNIVERSITY OF
** WISCONSIN OR THE CONDOR DESIGN TEAM BE LIABLE FOR ANY SPECIAL, INDIRECT
** OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS
** OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE
** OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE
** OR PERFORMANCE OF THIS SOFTWARE.
** 
** Authors:  Allan Bricker and Michael J. Litzkow,
** 	         University of Wisconsin, Computer Sciences Dept.
** 
*/ 

#include <stdio.h>
#include <sys/errno.h>

#include <sys/types.h>
#include <sys/param.h>
#include <sys/file.h>
#include <sys/time.h>
#include <sys/resource.h>

#include "ckpt_file.h"
#include "condor_sys.h"
#include "debug.h"
#include "fileno.h"

RESTREC	 RestartInfo;
extern char	Condor_CWD[];

PreOpen( fdno )
int fdno;
{
	register FINFO *fi = &RestartInfo.rr_file[ fdno ];

	fi->fi_flags = (FI_OPEN | FI_PREOPEN);
	fi->fi_priv  = 0;
	fi->fi_pos   = 0;
	fi->fi_fdno  = fdno;
	fi->fi_path  = "Pre-Opened";
}

MarkFileOpen( fd, path, flags, method )
int fd;
char *path;
int flags;
int	method;
{
	register FINFO *fi;
	char fullpath[ MAXPATHLEN ];
	char *strdup();
	int		i;

	dprintf( D_FULLDEBUG,
					"MarkFileOpen(%d,%s,%d,%d)\n", fd, path, flags, method );

	i = find_avail_fd( 0 );
	if( i < 0 ) {
		errno = EMFILE;
		return -1;
	}
	fi = &RestartInfo.rr_file[i];

	fi->fi_flags = FI_OPEN | method;
	fi->fi_priv  = flags & (O_RDONLY|O_WRONLY|O_RDWR);
	fi->fi_fdno  = fd;

	if( *path == '/' ) {
		fi->fi_path = strdup( path );
	} else {
		sprintf(fullpath, "%s/%s", Condor_CWD, path);
		fi->fi_path = strdup( fullpath );
	}
	return i;
}

/*
** Find the lowest numbered unused file descriptor >= start.  Return -1
** if there is none.
*/
find_avail_fd( start )
int		start;
{
	int		i;

	for( i=start; i<NOFILE ; i++ ) {
		if( (RestartInfo.rr_file[i].fi_flags & FI_OPEN) == 0 )  {
			/*
			dprintf( D_ALWAYS, "find_avail_fd returns %d\n", i );
			*/
			return i;
		}
	}
	/*
	dprintf( D_ALWAYS, "find_avail_fd returns -1\n" );
	*/
	return -1;
}

MarkFileClosed( fd )
int fd;
{
	register FINFO *fi = &RestartInfo.rr_file[fd];
	register FINFO *dupfi;
	register i, isduped;
	int		duped_to_fd;

	if( (fi->fi_flags & FI_DUP) == FI_DUP ) {
		fi->fi_flags = 0;
		fi->fi_priv = 0;
		fi->fi_pos = 0;
		fi->fi_fdno = -1;
		fi->fi_dupof = 0;
		fi->fi_path  = (char *) 0;
		return;
	}

		/* May be that the file we are closing has some dup's.  If so,
		convert of of the to a base (not dup'd) fd. */
	isduped = 0;
	for( i=0; i<NOFILE; i++ ) {
		dupfi = &RestartInfo.rr_file[i];
		if( ((dupfi->fi_flags & FI_DUP) == FI_DUP) &&
									(dupfi->fi_fdno == fi->fi_fdno) ) {
			dupfi->fi_flags &= ~FI_DUP;
			dupfi->fi_dupof = 0;
			duped_to_fd = i;

			isduped = 1;
			break;
		}
	}

		/* If we had some dup's, change them to point to the new base fd. */
	if( isduped ) {
		for( i=0; i<NOFILE; i++ ) {
			dupfi = &RestartInfo.rr_file[i];
			if( ((dupfi->fi_flags & FI_DUP) == FI_DUP) &&
										(dupfi->fi_fdno == fi->fi_fdno) ) {
				dupfi->fi_dupof = duped_to_fd;
			}
		}
	}

	if( !isduped && (fi->fi_flags & FI_PREOPEN) != FI_PREOPEN ) {
		FREE( fi->fi_path );
	}
	fi->fi_flags = 0;
	fi->fi_fdno = -1;
}

/*
** Emulate the dup() system call by updating the file mapping table.
*/
DoDup( fd )
int		fd;
{
	register FINFO *fi = &RestartInfo.rr_file[fd];
	int		dupfd;

	if( (fi->fi_flags & FI_OPEN) != FI_OPEN ) {
		errno = EINVAL;
		return -1;
	}

	dupfd = find_avail_fd( 0 );
	if( dupfd < 0 ) {
		errno = EMFILE;
		return -1;
	}
	DoDup2( fd, dupfd );

	return dupfd;
}

/*
** Emulate the dup2() system call by updating the file mapping table.
*/
DoDup2( fd, dupfd )
int fd, dupfd;
{
	register FINFO *fi = &RestartInfo.rr_file[fd];
	register FINFO *dupfi = &RestartInfo.rr_file[dupfd];

	/*
	dprintf( D_ALWAYS, "Before DoDup2\n" );
	DumpOpenFds();
	*/

	if( (fi->fi_flags & FI_OPEN) != FI_OPEN ) {
		errno = EINVAL;
		return -1;
	}

	if( dupfi->fi_flags & FI_OPEN ) {
		close( dupfd );
	}

	dupfi->fi_flags = fi->fi_flags | FI_DUP;
	dupfi->fi_priv = fi->fi_priv;
	dupfi->fi_fdno  = fi->fi_fdno;
	dupfi->fi_dupof = fd;
	dupfi->fi_path = fi->fi_path;

	/*
	dprintf( D_ALWAYS, "After DoDup2(%d,%d) returns %d\n", fd, dupfd, dupfd );
	DumpOpenFds();
	*/

	return dupfd;
}

DumpOpenFds()
{
	register FINFO *fi;
	int				i;

	dprintf( D_ALWAYS, "-------------  Condor File Table --------------\n" );
	for( i=0; i<NOFILE; i++ ) {
		fi = &RestartInfo.rr_file[i];
		if( (fi->fi_flags & FI_OPEN) == FI_OPEN ) {
			if( (fi->fi_flags & FI_DUP) == FI_DUP ) {
				dprintf( D_ALWAYS,
					"FD %d->%d: '%s', dup of %d, priv 0%o, flags 0%o, pos %d\n",
					i, fi->fi_fdno, fi->fi_path, fi->fi_dupof,
					fi->fi_priv, fi->fi_flags, fi->fi_pos );
			} else if( (fi->fi_flags & FI_PREOPEN) == FI_PREOPEN ) {
				dprintf( D_ALWAYS,
					"FD %d->%d was pre-opened, flags 0%o, pos %d\n",
									i, fi->fi_fdno, fi->fi_flags, fi->fi_pos );
			} else if( (fi->fi_flags & FI_WELL_KNOWN) == FI_WELL_KNOWN ) {
				dprintf( D_ALWAYS, "FD %d->%d is well known sock to shadow\n",
															i, fi->fi_fdno);
			} else {
				dprintf( D_ALWAYS,
				"FD %d->%d: '%s', priv 0%o, flags 0%o, pos %d\n",
				i, fi->fi_fdno, fi->fi_path, fi->fi_priv,
				fi->fi_flags, fi->fi_pos );
			}
		}  /* else {
			dprintf( D_ALWAYS, "FD %d is not open\n", i );
		} */
	}
	dprintf( D_ALWAYS, "-----------------------------------------------\n" );
}


/*
** Return true iff this fd has a dup, or is a dup of some other fd.
*/
file_is_dupd( fd )
int		fd;
{
	register FINFO *fi;
	int		i;

	fi = &RestartInfo.rr_file[fd];
	if( (fi->fi_flags & FI_DUP) == FI_DUP ) {
		return 1;
	}

	for( i=0; i<NOFILE; i++ ) {
		fi = &RestartInfo.rr_file[i];
		if( (fi->fi_flags & FI_DUP) == FI_DUP && fi->fi_dupof == fd ) {
			return 1;
		}
	}
	return 0;
}

extern int	Syscalls;
map_fd( user_fd )
int		user_fd;
{
		/* UNRECORDED - not in open files table - don't map */
	if( (Syscalls & SYS_RECORDED) == 0 ) {
		return user_fd;
	}

		/* Trying to map a file which is not open -- error */
	if( (RestartInfo.rr_file[user_fd].fi_flags & FI_OPEN) != FI_OPEN ) {
		errno = EBADF;
		return -1;
	}

		/* Ok, return fd which user_fd is mapped to */
	return RestartInfo.rr_file[user_fd].fi_fdno;
}
