/* 
** Copyright 1986, 1987, 1988, 1989, 1990, 1991 by the Condor Design Team
** 
** Permission to use, copy, modify, and distribute this software and its
** documentation for any purpose and without fee is hereby granted,
** provided that the above copyright notice appear in all copies and that
** both that copyright notice and this permission notice appear in
** supporting documentation, and that the names of the University of
** Wisconsin and the Condor Design Team not be used in advertising or
** publicity pertaining to distribution of the software without specific,
** written prior permission.  The University of Wisconsin and the Condor
** Design Team make no representations about the suitability of this
** software for any purpose.  It is provided "as is" without express
** or implied warranty.
** 
** THE UNIVERSITY OF WISCONSIN AND THE CONDOR DESIGN TEAM DISCLAIM ALL
** WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES
** OF MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE UNIVERSITY OF
** WISCONSIN OR THE CONDOR DESIGN TEAM BE LIABLE FOR ANY SPECIAL, INDIRECT
** OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS
** OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE
** OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE
** OR PERFORMANCE OF THIS SOFTWARE.
** 
** Author:  Michael J. Litzkow
** 	         University of Wisconsin, Computer Sciences Dept.
** 
*/ 

#include <stdio.h>
#include <sys/types.h>
#include <sys/user.h>
#include <core.h>
#include <filehdr.h>
#include <aouthdr.h>
#include <scnhdr.h>
#include <loader.h>
#include <reloc.h>
#include <linenum.h>
#include <sys/ldr.h>
#include <sys/param.h>
#include <sys/pseg.h>
#include "debug.h"
#include "except.h"
#include "scn.h"

static char *_FileName_ = __FILE__;     /* Used by EXCEPT (see except.h)     */

char		*shmat(), *map_file(), *create_scn_hdr();
SCN_DESC	*locate_sections();

	/* Generate a pointer into memory given another pointer and an offset */
#define MSEEK(ptr,off) ((char *)(ptr)+(int)(off))

	/* Well known sizes */
#define AOUT_HDR_SIZE	sizeof(struct aouthdr)
#define COFF_HDR_SIZE	sizeof(struct filehdr)
#define SCN_HDR_SIZE	sizeof(struct scnhdr)
#define LD_HDR_SIZE		sizeof(struct ldhdr)
#define UBLOCK_SIZE		sizeof(struct user)
#define LD_INFO_SIZE	sizeof(struct ld_info)
#define CORE_HDR_SIZE	sizeof(struct core_dump)

	/* Round up to given increment */
#define ROUND_UP(q,incr) ((((q)+(incr)-1)/(incr))*(incr))

	/* Our section indicies are  zero based, loader uses 1 based numbers */
#define TEXT_SECT	(TEXT + 1)
#define BSS_SECT	(BSS + 1)
#define DATA_SECT	(DATA + 1)

	/* Loader relocation entry types */
#define TEXT_SYMBOL 0
#define DATA_SYMBOL 1
#define BSS_SYMBOL 2

	/* For use with strcmp, etc. */
#define MATCH 0

#define COPY_BLOCK(src,len) \
	bcopy((src),End,(len)); \
	dprintf( D_CKPT, "Copy 0x%x bytes from 0x%x - 0x%x to 0x%x - 0x%x\n", \
						(len), (src), (src)+(len), End, End+(len) ); \
	End += (len);

#define ZERO_BLOCK(len) \
	bzero(End,(len)); \
	dprintf( D_CKPT, "Zeroed 0x%x bytes from 0x%x - 0x%x\n", \
								(len), End, End+(len) ); \
	End += (len);


int		PAD1, TEXT, PAD2, DATA, BSS, PAD3, LOADER, DEBUG;
int		SHDATA, STACK;

char	*End;

extern int	DebugFlags;

_updateckpt( ckpt_file, obj_file, core_file )
char *ckpt_file, *obj_file, *core_file;
{
	char	*old_coff, *new_coff, *core;

	if( core_file == NULL ) {
		_mkckpt( ckpt_file, obj_file );
		return;
	}

		/* Open and map the files */
	old_coff = map_file( obj_file, O_RDONLY,0 );
	core = map_file( core_file, O_RDONLY,0 );
	new_coff = map_file( ckpt_file, O_RDWR|O_TRUNC|O_CREAT, 0775 );

		/* Build the checkpoint file */
	build_ckpt( old_coff, new_coff, core );

	unmap_file( old_coff );
	unmap_file( core );
	unmap_file( new_coff );
}

build_ckpt( old_coff, new_coff, core )
char	*old_coff;
char	*new_coff;
char	*core;
{
	int					n_sect, last_sect, i, lnno_delta;
	int					sym_tab_len, str_tab_len;
	char				*sym_tab, *str_tab;
	char				*new_data, *old_data;
	struct core_dump	*core_hdr;
	struct filehdr		*old_coff_hdr, *new_coff_hdr;
	struct aouthdr		*old_aout_hdr, *new_aout_hdr;
	struct scnhdr		*old_scn_hdrs, *new_scn_hdrs;
	struct ld_info		*aout_info, *libc_info, *ptr;
	struct ldhdr		*ld_hdr;
	struct ldrel		*ld_reloc;
	SCN_DESC			*scn_desc;
	BLOCK_DESC			sym_tab_desc, str_tab_desc;
	char				*core_stack, *all_core_data, *core_data;
	int					core_stack_len, all_core_data_len, core_data_len;
	char				*libc_data;
	int					libc_data_len, libc_data_vaddr, stack_vaddr;


	dprintf( D_CKPT, "old_coff = 0x%x, core = 0x%x, new_coff = 0x%x\n",
											old_coff, core, new_coff );

		/* Generate pointers into the original coff file */
	old_coff_hdr = (struct filehdr *)old_coff;
	old_aout_hdr = (struct aouthdr *)MSEEK(old_coff_hdr,COFF_HDR_SIZE);
	old_scn_hdrs = (struct scnhdr *)MSEEK(old_aout_hdr,AOUT_HDR_SIZE);
	n_sect = old_coff_hdr->f_nscns;

		/* Generate pointers into core file */
	core_hdr = (struct core_dump *)core;
	aout_info = (struct ld_info *)MSEEK( core_hdr, core_hdr->c_tab );
	libc_info = (struct ld_info *)MSEEK( aout_info, aout_info->ldinfo_next );
	if( DebugFlags & D_CKPT ) {
		display_ld_info( aout_info, "Ld_info from core" );
	}
	
	if( DebugFlags & D_CKPT ) {
		ptr = aout_info;;
		while( ptr->ldinfo_next ) {
			ptr = (struct ld_info *)MSEEK( ptr, ptr->ldinfo_next );
			display_ld_info( ptr, "Loader Info From Core" );
		}
	}

	core_stack_len = core_hdr->c_size;
	core_stack = MSEEK( core, core_hdr->c_stack );
	stack_vaddr = USRSTACK - core_stack_len;
	all_core_data = MSEEK( core_stack, core_stack_len );
	all_core_data_len = core_hdr->c_u.u_dsize;
	core_data = MSEEK( all_core_data,
						(int)aout_info->ldinfo_dataorg & ~PRIVORG );
	core_data_len = all_core_data_len -
						( (int)aout_info->ldinfo_dataorg & ~PRIVORG );
	dprintf( D_CKPT, "all_core_data = 0x%x, all_core_data_len = 0x%x\n",
		all_core_data, all_core_data_len );
	dprintf( D_CKPT, "core_data = 0x%x, core_data_len = 0x%x\n",
		core_data, core_data_len );
	libc_data = MSEEK( all_core_data,
						(int)libc_info->ldinfo_dataorg & ~PRIVORG );
	libc_data_len = libc_info->ldinfo_datasize;
	libc_data_vaddr = (int)libc_info->ldinfo_dataorg & ~PRIVORG;
		

		/* Figure out 0 based indicies of well known sections (numbers
		   in a.out header begin with 1) */
	PAD1 = old_aout_hdr->o_sntext - 2;	/* preceeds text section */
	TEXT = old_aout_hdr->o_sntext - 1;
	PAD2 = old_aout_hdr->o_sntext;		/* follows text section */
	DATA = old_aout_hdr->o_sndata - 1;
	BSS = old_aout_hdr->o_snbss - 1;
	PAD3 = old_aout_hdr->o_snbss;			/* follows bss section */
	LOADER = old_aout_hdr->o_snloader - 1;
	DEBUG = old_aout_hdr->o_snloader;		/* follows loader section */
	last_sect = DEBUG;
	STACK = ++last_sect;
	SHDATA = ++last_sect;
	old_data = MSEEK( old_coff, old_scn_hdrs[DATA].s_scnptr );

		/* Display original file parts */
	if( DebugFlags & D_CKPT ) {
		display_core_hdr( core_hdr, "Old Core Header" );
		display_coff_hdr( old_coff_hdr, "Old Coff Header" );
		display_aout_hdr( old_aout_hdr, "Old Aout Header" );
		display_scn_hdrs( old_scn_hdrs, "Old Section Headers", n_sect );
	}

		/* Sanity checks */
	check_scn_headers( old_scn_hdrs );
	check_core_hdr( core_hdr );

		/* Figure out where new headers will go */
	new_coff_hdr = (struct filehdr *)new_coff;
	new_aout_hdr = (struct aouthdr *)MSEEK(new_coff_hdr,COFF_HDR_SIZE);
	new_scn_hdrs = (struct scnhdr *)MSEEK(new_aout_hdr,AOUT_HDR_SIZE);

		/* Figure out where symbol table table will come from */
	sym_tab = MSEEK( old_coff, old_coff_hdr->f_symptr );
	sym_tab_len = old_coff_hdr->f_nsyms * SYMESZ;
	sym_tab_desc.src = sym_tab;
	sym_tab_desc.len =  sym_tab_len;

		/* Figure out where the string table will come from */
	str_tab = MSEEK( old_coff, old_coff_hdr->f_symptr + sym_tab_len );
	str_tab_len = *(int *)str_tab;
	str_tab_desc.src = str_tab;
	str_tab_desc.len = str_tab_len;

		/* Write out the original coff, a.out, and section headers */
	End = new_coff;
	COPY_BLOCK( old_coff_hdr, COFF_HDR_SIZE );
	COPY_BLOCK( old_aout_hdr, AOUT_HDR_SIZE );
	COPY_BLOCK( old_scn_hdrs, n_sect * SCN_HDR_SIZE );

		/* Add the new section headers */
	if( n_sect <= STACK ) {
		COPY_BLOCK( create_scn_hdr(".stack",stack_vaddr), SCN_HDR_SIZE );
		n_sect += 1;
		new_coff_hdr->f_nscns += 1;
		ptr = aout_info;;
		while( ptr->ldinfo_next ) {
			ptr = (struct ld_info *)MSEEK( ptr, ptr->ldinfo_next );
			COPY_BLOCK(
				create_scn_hdr(".shdata",(int)ptr->ldinfo_dataorg & ~PRIVORG),
				SCN_HDR_SIZE
			);
			n_sect += 1;
			new_coff_hdr->f_nscns += 1;
			dprintf( D_CKPT, "Created \".shdata\" header for \"%s\"\n",
				ptr->ldinfo_filename );
		}
	} else {
		new_scn_hdrs[STACK].s_paddr = stack_vaddr;
		new_scn_hdrs[STACK].s_vaddr = stack_vaddr;
	}

		/* Add all the section data, relocation info, and line numbers */
	scn_desc = locate_sections(  n_sect, old_coff, old_scn_hdrs, core_data,
		core_data_len, core_stack, core_stack_len, all_core_data, aout_info );
	copy_sections( scn_desc, n_sect, new_coff );

		/* Add the symbol and string tables */
	new_coff_hdr->f_symptr = End - new_coff;
	COPY_BLOCK( sym_tab_desc.src, sym_tab_desc.len );
	COPY_BLOCK( str_tab_desc.src, str_tab_desc.len );

		/* Adjust the a.out header */
	new_aout_hdr->o_dsize = new_scn_hdrs[DATA].s_size;
	new_aout_hdr->o_bsize = 0;

		/* No bss now */
	new_scn_hdrs[BSS].s_scnptr = 0;
	new_scn_hdrs[BSS].s_size = 0;
	new_scn_hdrs[PAD3].s_size = 0;

		/* Lets see it... */
	if( DebugFlags & D_CKPT ) {
		display_coff_hdr( new_coff_hdr, "New Coff Header" );
		display_aout_hdr( new_aout_hdr, "New Aout Header" );
		display_scn_hdrs( new_scn_hdrs, "New Section Headers", n_sect );
	}

		/* Figure out where things are in the new coff file */
	ld_hdr = (struct ldhdr *)MSEEK( new_coff, new_scn_hdrs[LOADER].s_scnptr );
	ld_reloc = (struct ldrel *)MSEEK( ld_hdr, LD_HDR_SIZE +
											LDSYMSZ * ld_hdr->l_nsyms );
	new_data = MSEEK( new_coff, new_scn_hdrs[DATA].s_scnptr );
	sym_tab = MSEEK( new_coff, new_coff_hdr->f_symptr );
	lnno_delta = new_scn_hdrs[TEXT].s_lnnoptr - old_scn_hdrs[TEXT].s_lnnoptr;

		/* Make the necessary adjustments */
	change_bss_to_data( ld_hdr, ld_reloc );
	unrelocate( aout_info, new_aout_hdr, ld_hdr, ld_reloc, new_data, old_data );
	adjust_sym_tab( sym_tab, sym_tab_len, lnno_delta );
}

/*
** The symbol table contains pointers to line number information in the
** coff file.  Since this information is now stored later in the file,
** we have to adjust all those pointers.
**
** Note: We should refer to the symbol table entries as tab[n], but
** there is some problem in the array index calculation (compiler bug?),
** so we cast tab to a char * and do the calculation ourself.
*/
adjust_sym_tab( tab, len, delta )
char			*tab;
int				len;
{
	int		i;
	struct syment	*sym;
	union auxent	*aux;
	int				n_aux;

	for( i=0; i<len; i++ ) {
		sym = (struct syment *)&tab[i*SYMESZ];
		n_aux = sym->n_numaux;
		if( ISFCN(sym->n_type) && (n_aux >= 2) ) {
			aux = (union auxent *)&tab[ (i+1) * SYMESZ ];
			aux->x_sym.x_fcnary.x_fcn.x_lnnoptr += delta;
		}
		i += n_aux;
	}
}



/*
** Undo the work the loader did in relocating things so next time it can
** do it over.
*/
unrelocate( ld_info, aouthdr, ld_hdr, ld_reloc, new_data, old_data )
struct ld_info	*ld_info;
struct aouthdr	*aouthdr;
struct ldhdr	*ld_hdr;
struct ldrel	*ld_reloc;
char			*new_data;
char			*old_data;
{
	int		text_relocation;
	int		data_relocation;
	struct ldrel	*p;
	int		datum;
	int		i;


		/* How much did things get relocated ? */
	text_relocation = (int)(ld_info->ldinfo_textorg) + 0x400 -
											aouthdr->text_start;
	data_relocation = (int)ld_info->ldinfo_dataorg;		/* rwong */


	for( i=0; i<ld_hdr->l_nreloc; i++ ) {
		p = &ld_reloc[i];
		if( p->l_rsecnm != DATA_SECT ) {
			continue;
		}

			/* Unrelocate all the text, data, and bss items */
		switch(  p->l_symndx ) {
			case TEXT_SYMBOL:
				datum = get_datum( p->l_vaddr, new_data );
				datum -= text_relocation;
				put_datum( datum, p->l_vaddr, new_data );
				break;
			case DATA_SYMBOL:
				datum = get_datum( p->l_vaddr, new_data );
				datum -= data_relocation;
				put_datum( datum, p->l_vaddr, new_data );
				break;
			case BSS_SYMBOL:
				datum = get_datum( p->l_vaddr, new_data );
				datum -= data_relocation;
				put_datum( datum, p->l_vaddr, new_data );
				break;
			default:
				datum = get_datum( p->l_vaddr, old_data );
				put_datum( datum, p->l_vaddr, new_data );
				break;
		}
	}
}

/*
** There is no bss in the new coff file, all bss must be converted to data.
** Change the symbol types here.
*/
change_bss_to_data( ld_hdr, ld_reloc )
struct ldhdr		*ld_hdr;
struct ldrel		*ld_reloc;
{
	int				i;

	if( DebugFlags & D_CKPT ) {
		display_ldsym_tab( ld_hdr, ld_hdr->l_nsyms );
	}

	dprintf( D_CKPT, "Changing BSS symbols to DATA symbols:\n" );
	if( DebugFlags & D_CKPT ) {
		display_reloc_labels();
	}
	for( i=0; i<ld_hdr->l_nreloc; i++ ) {
		if( ld_reloc[i].l_rsecnm == BSS_SECT ) {
			if( DebugFlags & D_CKPT ) {
				display_reloc( i, ld_hdr, TEXT_SECT, BSS_SECT, DATA_SECT );
			}
			ld_reloc[i].l_rsecnm = DATA_SECT;
			if( DebugFlags & D_CKPT ) {
				display_reloc( i, ld_hdr, TEXT_SECT, BSS_SECT, DATA_SECT );
			}
		}
		if( ld_reloc[i].l_symndx == BSS_SYMBOL ) {
			if( DebugFlags & D_CKPT ) {
				display_reloc( i, ld_hdr, TEXT_SECT, BSS_SECT, DATA_SECT );
			}
			ld_reloc[i].l_symndx = DATA_SYMBOL;
			if( DebugFlags & D_CKPT ) {
				display_reloc( i, ld_hdr, TEXT_SECT, BSS_SECT, DATA_SECT );
			}
		}
	}
}

/*
** Create a header for a new section.
*/
char *
create_scn_hdr( name, addr )
char			*name;
int				addr;
{
	struct scnhdr	*ptr;

	ptr = (struct scnhdr *)malloc( SCN_HDR_SIZE );
	strcpy( ptr->s_name, name );
	ptr->s_paddr = addr;
	ptr->s_vaddr = addr;
	ptr->s_size = 0;
	ptr->s_relptr = 0;
	ptr->s_lnnoptr = 0;
	ptr->s_nreloc = 0;
	ptr->s_nlnno = 0;
	ptr->s_flags = STYP_PAD;
	return (char *)ptr;
}

/*
** Append all the sections to a coff file.  We are given an array of
** descriptors which specify where each block of data is to come from.
*/
copy_sections( scn, n_sect, coff )
SCN_DESC	scn[];
int			n_sect;
char		*coff;
{
	int				i;
	struct scnhdr	*hdr;
	struct filehdr	*coff_hdr;

	coff_hdr = (struct filehdr *)coff;
	hdr = (struct scnhdr *)MSEEK(coff, COFF_HDR_SIZE + AOUT_HDR_SIZE);

	for( i=0; i<n_sect; i++ ) {
		if( scn[i].raw.len ) {
			hdr[i].s_scnptr = End - coff;
			hdr[i].s_size = scn[i].raw.len;
			if( scn[i].raw.src ) {
				COPY_BLOCK( scn[i].raw.src, scn[i].raw.len );
			} else {
				ZERO_BLOCK( scn[i].raw.len );
			}
		} else {
			hdr[i].s_scnptr = 0;
		}
	}

	for( i=0; i<n_sect; i++ ) {
		if( scn[i].reloc.len ) {
			hdr[i].s_relptr = End - coff;
			hdr[i].s_nreloc = scn[i].reloc.len / RELSZ;
			COPY_BLOCK( scn[i].reloc.src, scn[i].reloc.len );
		} else {
			hdr[i].s_relptr = 0;
			hdr[i].s_nreloc = 0;
		}
	}

	for( i=0; i<n_sect; i++ ) {
		if( scn[i].l_num.len ) {
			hdr[i].s_lnnoptr = End - coff;
			hdr[i].s_nlnno = scn[i].l_num.len / RELSZ;
			COPY_BLOCK( scn[i].l_num.src, scn[i].l_num.len );
		} else {
			hdr[i].s_lnnoptr = 0;
			hdr[i].s_nlnno = 0;
		}
	}
}

/*
** Display a list of descriptors specifying where each section's data
** is to come from.
*/
display_sections( scn, n_sect )
SCN_DESC	scn[];
int			n_sect;
{
	int		i;

	for( i=0; i<n_sect; i++ ) {
		display_block( i, &scn[i].raw, "raw" );
		display_block( i, &scn[i].reloc, "reloc" );
		display_block( i, &scn[i].l_num, "l_num" );
	}
}

/*
** Display the descriptor for one block of data.
*/
display_block( scn_num, block, label )
int			scn_num;
BLOCK_DESC	*block;
char		*label;
{
	if( block->len ) {
		if( block->src ) {
			dprintf( D_ALWAYS, "Sect %2d %s.src 0x%x, %s.len 0x%x\n",
					scn_num, label, block->src, label, block->len );
		} else {
			dprintf( D_ALWAYS, "Sect %2d %s.src (zeroes), %s.len 0x%x\n",
					scn_num, label, label, block->len );
		}
	}
}
		

/*
** Figure out where the various data blocks for all the sections should
** come from.
*/
SCN_DESC *
locate_sections(  n_sect, old_coff, old_scn_hdrs, core_data,
		core_data_len, core_stack, core_stack_len, all_core_data, ldi )
int				n_sect;
char			*old_coff;
struct scnhdr	*old_scn_hdrs;
char			*core_data;
int				core_data_len;
char			*core_stack;
int				core_stack_len;
char			*all_core_data;
struct ld_info	*ldi;
{
	int					hdr_end, text_end, i;
	int					text_off, data_off;
	SCN_DESC			*scn;

		/* Allocate an array to store our results in */
	scn = (SCN_DESC *)calloc( n_sect, sizeof(SCN_DESC) );

		/* Figure out where the text starts, and appropriate padding */
	hdr_end = COFF_HDR_SIZE + AOUT_HDR_SIZE + n_sect * SCN_HDR_SIZE;
	text_off = ROUND_UP( hdr_end, 1024 );
	scn[PAD1].raw.len = text_off - hdr_end;

		/* Place the text, text relocation, and text line numbers */
	scn[TEXT].raw.src = MSEEK( old_coff, old_scn_hdrs[TEXT].s_scnptr );
	scn[TEXT].raw.len = old_scn_hdrs[TEXT].s_size;
	scn[TEXT].reloc.src = MSEEK( old_coff, old_scn_hdrs[TEXT].s_relptr );
	scn[TEXT].reloc.len = old_scn_hdrs[TEXT].s_nreloc * RELSZ;
	scn[TEXT].l_num.src = MSEEK( old_coff, old_scn_hdrs[TEXT].s_lnnoptr );
	scn[TEXT].l_num.len = old_scn_hdrs[TEXT].s_nlnno * LINESZ;

		/* Figure out where the data starts, and appropriate padding */
	text_end = text_off + scn[TEXT].raw.len;
	data_off = ROUND_UP( text_end, 512 );
	scn[PAD2].raw.len = data_off - text_end;

	scn[DATA].raw.src = core_data;
	scn[DATA].raw.len = core_data_len;
	scn[DATA].reloc.src = MSEEK( old_coff, old_scn_hdrs[DATA].s_relptr );
	scn[DATA].reloc.len = old_scn_hdrs[DATA].s_nreloc * RELSZ;

	scn[PAD3].raw.src = NULL;
	scn[PAD3].raw.len = 0;

	scn[LOADER].raw.src = MSEEK( old_coff, old_scn_hdrs[LOADER].s_scnptr );
	scn[LOADER].raw.len = old_scn_hdrs[LOADER].s_size;

	scn[DEBUG].raw.src = MSEEK( old_coff, old_scn_hdrs[DEBUG].s_scnptr );
	scn[DEBUG].raw.len = old_scn_hdrs[DEBUG].s_size;

	scn[STACK].raw.src = core_stack;
	scn[STACK].raw.len = core_stack_len;

	for( i = SHDATA; ldi->ldinfo_next; i++ ) {
		ldi = (struct ld_info *)MSEEK( ldi, ldi->ldinfo_next );
		scn[i].raw.src = MSEEK( all_core_data,
						(int)ldi->ldinfo_dataorg & ~PRIVORG );
		scn[i].raw.len = ldi->ldinfo_datasize;
	}

	if( DebugFlags & D_CKPT ) {
		display_sections( scn, n_sect );
	}

	return scn;
}

/*
** Fetch an integer given a virtual address, and a base location.
*/
int
get_datum( v_addr, data_loc )
ulong	v_addr;
char	*data_loc;
{
	int	*p;
	p = (int *)MSEEK(data_loc, v_addr);
	return  *p;
}


/*
** Store an integer given a virtual address and a base location.
*/
put_datum( datum, v_addr, data_loc )
int		datum;
ulong	v_addr;
char	*data_loc;
{
	int	*p;
	p = (int *)MSEEK(data_loc, v_addr);
	*p = datum;
}

/*
** We're expecting some sections in a specific order in the coff file.  Check
** here to see if it's that way.
*/
check_scn_headers( ptr )
struct scnhdr	*ptr;
{
	if( strcmp(".pad",ptr[PAD3].s_name) != MATCH ) {
		EXCEPT( "Unexpected coff format, section %d not \".pad\"", PAD3 + 1 );
	}

	if( strcmp(".debug",ptr[DEBUG].s_name) != MATCH ) {
		EXCEPT( "Unexpected coff format, section %d not \".debug\"", DEBUG + 1);
	}
	dprintf( D_CKPT, "Section Headers - OK\n" );
}

/*
** Make sure we have a "complete" and reasonable looking core file.
*/
check_core_hdr( ptr )
struct core_dump	*ptr;
{
	if( ptr->c_signo != SIGQUIT ) {
		EXCEPT( "Core not created by checkpoint - caused by signal %d",
														ptr->c_signo );
	}

	if( !(ptr->c_flag&FULL_CORE) ) {
		EXCEPT( "Core not complete - no data" );
	}

	if( !(ptr->c_flag&UBLOCK_VALID) ) {
		EXCEPT( "Core not vaild - UBLOCK not valid" );
	}

	if( !(ptr->c_flag&USTACK_VALID) ) {
		EXCEPT( "Core not vaild - USTACK not valid" );
	}

	if( !(ptr->c_flag&LE_VALID) ) {
		EXCEPT( "Core not vaild - LE not valid" );
	}

	if( (ptr->c_flag&CORE_TRUNC) ) {
		EXCEPT( "Core not complete - truncated" );
	}

	dprintf( D_CKPT, "Core file - OK\n" );

}
