/* 
** Copyright 1986, 1987, 1988, 1989, 1990, 1991 by the Condor Design Team
** 
** Permission to use, copy, modify, and distribute this software and its
** documentation for any purpose and without fee is hereby granted,
** provided that the above copyright notice appear in all copies and that
** both that copyright notice and this permission notice appear in
** supporting documentation, and that the names of the University of
** Wisconsin and the Condor Design Team not be used in advertising or
** publicity pertaining to distribution of the software without specific,
** written prior permission.  The University of Wisconsin and the Condor
** Design Team make no representations about the suitability of this
** software for any purpose.  It is provided "as is" without express
** or implied warranty.
** 
** THE UNIVERSITY OF WISCONSIN AND THE CONDOR DESIGN TEAM DISCLAIM ALL
** WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES
** OF MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE UNIVERSITY OF
** WISCONSIN OR THE CONDOR DESIGN TEAM BE LIABLE FOR ANY SPECIAL, INDIRECT
** OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS
** OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE
** OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE
** OR PERFORMANCE OF THIS SOFTWARE.
** 
** Authors:  Allan Bricker and Michael J. Litzkow,
** 	         University of Wisconsin, Computer Sciences Dept.
** 
*/ 



#include <netdb.h>
#include <stdio.h>
#include <sys/types.h>
#include <utmp.h>
#include <nlist.h>
#include <signal.h>
#include <setjmp.h>

#include <pwd.h>
#include <errno.h>
#include <ctype.h>
#include <sys/stat.h>
#include <sys/param.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <sys/resource.h>

#if defined(AIX31)
#include <sys/statfs.h>
#endif

#if defined(ULTRIX42)
#include <sys/mount.h>
#endif

#ifdef IRIX331
#include <sys/statfs.h>
#endif

#if !defined(ULTRIX) && !defined(ULTRIX42) && !defined(IRIX331) && !defined(AIX31)
#include <sys/vfs.h>
#endif


#include <sys/file.h>
#include <netinet/in.h>
#include <rpc/types.h>
#include <rpc/xdr.h>

#ifdef DYNIX
#include <sys/tmp_ctl.h>
#include <sys/vm.h>
#endif DYNIX


#include <strings.h>
#include <arpa/inet.h>
#ifndef MAXHOSTNAMELEN
#define MAXHOSTNAMELEN 100
#endif MAXHOSTNAMELEN

#include "sched.h"
#include "debug.h"
#include "trace.h"
#include "except.h"
#include "expr.h"
#include "clib.h"

static char *_FileName_ = __FILE__;		/* Used by EXCEPT (see except.h)     */

CONTEXT		*create_context();
bool_t 		xdr_int();
XDR			*xdr_Udp_Init(), *xdr_Init();
ELEM		*create_elem();
EXPR		*create_expr(), *build_expr();;
float		calc_load_avg();
char		*strdup();

extern CONTEXT	*MachineContext;
extern CONTEXT	*GenericJob;
extern CONTEXT	*JobContext;
extern int		StarterPid;
extern int		LastTimeout;
extern char		*MgrHost;
extern char		*CondorAdministrator;
extern int		UdpSock;
extern int		Memory;
extern char		*Execute;
extern char		*Starter;
extern char		*KbdDev;
extern char		*MouseDev;
extern int		State;
extern int		PollsPerUpdate;
static int		Polls = 0;

char			*ClientMachine;

int		Last_X_Event;

job_dies()
{
	StarterPid = 0;
	if( JobContext ) {
		free_context( JobContext );
		JobContext = NULL;
	}
	cleanup_execute_dir();
	FREE( ClientMachine );
	ClientMachine = NULL;
	change_states( NO_JOB );
	Polls = 0;
	timeout();
}

timeout()
{



	evaluate_situation();

	if( Polls == 0 ) {
		(void)update_central_mgr();
	}
	if( ++Polls >= PollsPerUpdate ) {
		Polls = 0;
	}
	LastTimeout = (int)time( (time_t *)0 );

}


#define ABORT \
	if( job_context ) {				\
		free_context( job_context );\
	}								\
	if( ClientMachine ) {			\
		FREE( ClientMachine );		\
		ClientMachine = NULL;		\
	}								\
	return;

start_job_request( xdrs, from )
XDR		*xdrs;
struct sockaddr_in	*from;
{
	int		start, job_reqs;
	CONTEXT	*job_context = NULL;
	struct hostent  *hp, *gethostbyaddr();
    ELEM    tmp;

    if( (hp=gethostbyaddr((char *)&from->sin_addr,
					sizeof(struct in_addr), from->sin_family)) == NULL ) {
        EXCEPT(  "Can't find host name" );
    }
	ClientMachine = strdup( hp->h_name );


	dprintf( D_ALWAYS, "Got start_job_request from %s\n", ClientMachine );

		/* Make sure we have up to date swap space information */
	tmp.type = INT;
	tmp.i_val = calc_virt_memory();
	store_stmt( build_expr("VirtualMemory",&tmp), MachineContext );

		/* Read in job facts and booleans */
	job_context = create_context();
	if( !xdr_context(xdrs,job_context) ) {
		dprintf( D_ALWAYS, "Can't receive context from shadow\n" );
		return;
	}
	if( !xdrrec_skiprecord(xdrs) ) {
		dprintf( D_ALWAYS, "Can't receive context from shadow\n" );
		return;
	}

	dprintf( D_JOB, "JOB_CONTEXT:\n" );
	if( DebugFlags & D_JOB ) {
		display_context( job_context );
	}
	dprintf( D_JOB, "\n" );

	dprintf( D_MACHINE, "MACHINE_CONTEXT:\n" );
	if( DebugFlags & D_MACHINE ) {
		display_context( MachineContext );
	}
	dprintf( D_MACHINE, "\n" );

	if( State != NO_JOB ) {
		dprintf( D_ALWAYS, "State != NO_JOB\n" );
		(void)reply( xdrs, NOT_OK );
		ABORT;
	}

		/* See if machine and job meet each other's requirements, if so
		   start the job and tell shadow, otherwise refuse and clean up */
	if( evaluate_bool("START",&start,MachineContext,job_context) < 0 ) {
		start = 0;
	}
	if( evaluate_bool("JOB_REQUIREMENTS",&job_reqs,MachineContext,job_context)
																		< 0 ) {
		job_reqs = 0;
	}

	if( !start || !job_reqs ) {
		dprintf( D_ALWAYS, "start = %d, job_reqs = %d\n", start, job_reqs );
		(void)reply( xdrs, NOT_OK );
		ABORT;
	}

	if( !reply(xdrs,OK) ) {
		ABORT;
	}
	JobContext = job_context;
	if( start_job(xdrs) < 0 ) {
		ABORT;
	}
	change_states( JOB_RUNNING );
	Polls = 0;
	timeout();
}
#undef ABORT

x_event( xdrs, from )
XDR		*xdrs;
struct sockaddr_in	*from;
{
	Last_X_Event = time( (time_t *)0 );
	timeout();
}


evaluate_situation()
{
	int		tmp;
	int		start = FALSE;

	evaluate_load();

	/*
	dprintf( D_ALWAYS, "----------------\n" );
	display_context( MachineContext );
	dprintf( D_ALWAYS, "----------------\n" );
	display_context( GenericJob );
	dprintf( D_ALWAYS, "----------------\n" );
	*/

	if( State == NO_JOB ) {
		(void)evaluate_bool( "START", &start, MachineContext, GenericJob );
		if( start ) {
			set_machine_status( M_IDLE );
			dprintf( D_FULLDEBUG, "Set machine status to M_IDLE\n" );
		} else {
			set_machine_status( USER_BUSY );
			dprintf( D_FULLDEBUG, "Set machine status to USER_BUSY\n" );
		}
	}

	if( StarterPid && !JobContext ) {
		EXCEPT( "evaluate_situation, StarterPid = %d, but JobContext is NULL\n",
					StarterPid );
	}

	if( !JobContext ) {
		return;
	}

	if( State == CHECKPOINTING ) {
		if( evaluate_bool("KILL",&tmp,MachineContext,JobContext) < 0 ) {
			EXCEPT( "Can't evaluate machine boolean \"KILL\"\n" );
		}
		if( tmp ) {
			kill_job( NOT_OK );
			return;
		}
	}

	if( State == SUSPENDED ) {
		if( evaluate_bool("VACATE",&tmp,MachineContext,JobContext) < 0 ) {
			EXCEPT( "Can't evaluate machine boolean \"VACATE\"\n" );
		}
		if( tmp ) {
			vacate_order();
			return;
		}
	}

	if( State == JOB_RUNNING ) {
		if( evaluate_bool("SUSPEND",&tmp,MachineContext,JobContext) < 0 ) {
			EXCEPT( "Can't evaluate machine boolean \"SUSPEND\"\n" );
		}
		if( tmp ) {
			suspend_order();
			return;
		}
	}

	if( State == SUSPENDED ) {
		if( evaluate_bool("CONTINUE",&tmp,MachineContext,JobContext) < 0 ) {
			EXCEPT( "Can't evaluate machine boolean \"CONTINUE\"\n" );
		}
		if( tmp ) {
			resume_order();
			return;
		}
	}

}

update_central_mgr()
{
	int		cmd;
	XDR		xdr, *xdrs;
    ELEM    tmp;
	static int	first_time = 1;
	char	*calc_subnet_name();
	char	*max_jobs;


		/* Calculating virt memory is expensive, only do it if we are idle */
	if( machine_idle() || first_time ) {
		tmp.type = INT;
		tmp.i_val = calc_virt_memory();
		store_stmt( build_expr("VirtualMemory",&tmp), MachineContext );
		first_time = 0;

	}


	tmp.type = STRING;
	tmp.s_val = calc_subnet_name();
	if( strlen(tmp.s_val) != 0 )
		store_stmt( build_expr("Subnet", &tmp), MachineContext );
	free( tmp.s_val );

	tmp.type = STRING;
	tmp.s_val = (char *)param("FLAVOR");		/* rwong */
	if( tmp.s_val )
		store_stmt( build_expr("Flavor", &tmp), MachineContext );


	tmp.type = INT;

	max_jobs = (char *)param( "MAX_JOBS_RUNNING" );		/* rwong */
	if( max_jobs ) {
		tmp.i_val = atoi( max_jobs );
		free( max_jobs );
	} else {
		tmp.i_val = 15;
	}

	store_stmt( build_expr("MAX_JOBS_RUNNING", &tmp), MachineContext);


	xdrs = xdr_Udp_Init( &UdpSock, &xdr );


	xdrs->x_op = XDR_ENCODE;


		/* Send the command */
	dprintf( D_XDR, "About to send update to collector\n" );
	cmd = STARTD_INFO;


	if( !xdr_int(xdrs, &cmd) ) {
		xdr_destroy( xdrs );
		return -1;
	}


	if( !xdr_context(xdrs,MachineContext) ) {
		xdr_destroy( xdrs );
		return -1;
	}

	if( !xdrrec_endofrecord(xdrs,TRUE) ) {
		xdr_destroy( xdrs );
		return -1;
	}

	xdr_destroy( xdrs );
	dprintf( D_XDR, "Done sending update to collector\n" );


	return 0;
}

change_states( new_state )
int	new_state;
{
	ELEM	tmp;
	char	*name;
	int		start = FALSE;

	switch( new_state ) {
		case NO_JOB:
			name = "NoJob";
			(void)evaluate_bool( "START", &start, MachineContext, GenericJob );
			if( start ) {
				set_machine_status( M_IDLE );
			} else {
				set_machine_status( USER_BUSY );
			}
			break;
		case JOB_RUNNING:
			name = "Running";
			set_machine_status( JOB_RUNNING );
			break;
		case KILLED:
			name = "Killed";
			set_machine_status( KILLED );
			break;
		case CHECKPOINTING:
			name = "Checkpointing";
			set_machine_status( CHECKPOINTING );
			break;
		case SUSPENDED:
			name = "Suspended";
			set_machine_status( SUSPENDED );
			break;
		default:
			EXCEPT( "Change states, unknown state (%d)", new_state );
	}

	State = new_state;

	tmp.type = STRING;
	tmp.s_val = name;
	store_stmt( build_expr("State",&tmp), MachineContext );

	tmp.type = INT;
	tmp.i_val = (int)time( (time_t *)0 );
	store_stmt( build_expr("EnteredCurrentState",&tmp), MachineContext );
}

kill_job( status )
int		status;
{
	if( status == OK ) {
		dprintf( D_ALWAYS, "Called kill_job( OK )\n" );
	} else {
		dprintf( D_ALWAYS, "Called kill_job( NOT_OK )\n" );
	}
	if( !StarterPid ) {
		dprintf( D_ALWAYS, "Called kill_job, but no starter process active\n" );
		return;
	}

#ifdef MAIL_STUCK_JOB_MESSAGE
#define MAIL_STUCK_JOB_MESSAGE
	/*
	** This was taken out because the messages did not prove to be all that
	** useful.
	*/
	if( status != OK ) {
		mail_stuck_job_message();
	}
#endif MAIL_STUCK_JOB_MESSAGE

		/* Could be the starter will die naturally just before we try to
		   send the sig, so have to ignore errors here... */
	if( status == OK ) {
		(void) kill_starter( StarterPid, SIGINT );
		change_states( CHECKPOINTING );
	} else {
		(void) kill_starter( StarterPid, SIGKILL );
		change_states( KILLED );
	}
}

mail_stuck_job_message()
{
	char	cmd[512];
	char	hostname[512];
	FILE	*mailer, *popen();

	dprintf( D_ALWAYS, "Mailing administrator (%s)\n", CondorAdministrator );

	if( gethostname(hostname,sizeof(hostname)) < 0 ) {
		EXCEPT( "gethostname(0x%x,%d)", hostname, sizeof(hostname) );
	}

	(void)sprintf( cmd, "%s %s", BIN_MAIL, CondorAdministrator );
	if( (mailer=popen(cmd,"w")) == NULL ) {
		EXCEPT( "popen(\"%s\",\"w\")", cmd );
	}

	fprintf( mailer, "To: %s\n", CondorAdministrator );
	fprintf( mailer, "Subject: Condor Problem\n" );
	fprintf( mailer, "\n" );

	fprintf( mailer, "The startd on \"%s\" is killing a Condor job ",
				hostname );
	fprintf( mailer, "from \"%s\"\n", ClientMachine );
	fprintf( mailer, "because it didn't clean up after being sent a TSTP.\n"  );
	fprintf( mailer, "\n" );
	mail_context( "JobContext:", JobContext, mailer );
	fprintf( mailer, "\n" );
	mail_context( "MachineContext:", MachineContext, mailer );
	fprintf( mailer, "\n" );

	fprintf( mailer, "CurrentTime = %d\n", (int)time((time_t *)0) );

		/* don't use 'pclose()' here, it does its own wait, and messes
	       with our handling of SIGCHLD! */
	(void)fclose( mailer );
}

ckpt_order()
{
	dprintf( D_ALWAYS, "Called ckpt_order()\n" );

	if( !StarterPid ) {
		dprintf( D_ALWAYS,
					"Called ckpt_order, but no starter process active\n" );
	} else {
			/* Could be the starter will die naturally just before we try to
			   send the sig, so have to ignore errors here... */
		(void) kill_starter( StarterPid, SIGUSR2 );
		change_states( CHECKPOINTING );
	}
}

vacate_order()
{
	dprintf( D_ALWAYS, "Called vacate_order()\n" );

	dprintf( D_MACHINE, "MACHINE_CONTEXT:\n" );
	if( DebugFlags & D_MACHINE ) {
		display_context( MachineContext );
	}
	dprintf( D_MACHINE, "\n" );

	if( !StarterPid ) {
		dprintf( D_ALWAYS,
					"Called vacate_order, but no starter process active\n" );
	} else {
			/* Could be the starter will die naturally just before we try to
			   send the sig, so have to ignore errors here... */
		(void) kill_starter( StarterPid, SIGTSTP );
		change_states( CHECKPOINTING );
	}
}

suspend_order()
{
	dprintf( D_ALWAYS, "Called suspend_order()\n" );

	dprintf( D_MACHINE, "MACHINE_CONTEXT:\n" );
	if( DebugFlags & D_MACHINE ) {
		display_context( MachineContext );
	}
	dprintf( D_MACHINE, "\n" );

	if( !StarterPid ) {
		dprintf( D_ALWAYS,
					"Called suspend_order, but no starter process active\n" );
	} else {
			/* Could be the starter will die naturally just before we try to
			   send the sig, so have to ignore errors here... */
		(void) kill_starter( StarterPid, SIGUSR1 );
		change_states( SUSPENDED );
	}
}

resume_order()
{
	dprintf( D_ALWAYS, "Called resume_order()\n" );

	dprintf( D_MACHINE, "MACHINE_CONTEXT:\n" );
	if( DebugFlags & D_MACHINE ) {
		display_context( MachineContext );
	}
	dprintf( D_MACHINE, "\n" );

	if( !StarterPid ) {
		dprintf( D_ALWAYS,
					"Called resume_order, but no starter process active\n" );
	} else {
			/* Could be the starter will die naturally just before we try to
			   send the sig, so have to ignore errors here... */
		(void) kill_starter( StarterPid, SIGCONT );
		change_states( JOB_RUNNING );
	}
}


reply( xdrs, answer )
XDR	*xdrs;
int	answer;
{
	xdrs->x_op = XDR_ENCODE;
	if( !xdr_int(xdrs,&answer) ) {
		return FALSE;
	}
	if( !xdrrec_endofrecord(xdrs,TRUE) ) {
		return FALSE;
	}
	dprintf( D_ALWAYS, "Replied %s\n", answer ? "ACCEPTED" : "REFUSED" );
	return TRUE;
}

evaluate_load()
{
    ELEM    tmp;



    tmp.type = INT;
    tmp.i_val = calc_memory();

    store_stmt( build_expr("Memory",&tmp), MachineContext );

    tmp.i_val = calc_disk();
    store_stmt( build_expr("Disk",&tmp), MachineContext );

    tmp.i_val = user_idle_time();
    store_stmt( build_expr("KeyboardIdle",&tmp), MachineContext );

    tmp.i_val = calc_Ncpus();
    store_stmt( build_expr("Cpus",&tmp), MachineContext );

	tmp.type = FLOAT;
    tmp.f_val = calc_load_avg();
    store_stmt( build_expr("LoadAvg",&tmp), MachineContext );

	/*
	** display_context( MachineContext );
	*/
}

/*
** Calculate the main memory on our machine in megabytes.  For now we'll
** just look it up in the config file, someday we should probe the kernel
** for this information.
*/
calc_memory()
{
    return Memory;
}

/*
** Calculate the free disk in the filesystem which we use for hosting
** foreign executables in kilobytes.  We don't give the real answer,
** instead we allow a little "slop".
*/
#define FS_SLOP 512     /* Kilobytes */
calc_disk()
{
	return free_fs_blocks(Execute) - FS_SLOP;
}

/*
** I think this is all obsolete -- mike
*/
#if 0
#if defined(ULTRIX) || defined(ULTRIX42)

calc_disk()
{
    struct fs_data buf;

    if( statfs(Execute,&buf) < 0 ) {
        EXCEPT( "Can't do statfs on \"%s\"\n", Execute );
    }

    return( buf.fd_bfreen - FS_SLOP );
}

#else defined(ULTRIX) || defined(ULTRIX42)

#define FS_SLOP 512     /* Kilobytes */
calc_disk()
{
    struct statfs buf;
    int     kbytes;

    if( statfs(Execute,&buf) < 0 ) {
        EXCEPT( "Can't do statfs on \"%s\"\n", Execute );
    }

    kbytes = (buf.f_bavail * buf.f_bsize) / 1024;
    return kbytes - FS_SLOP;
}

#endif defined(ULTRIX) || defined(ULTRIX42)
#endif

#define MAXINT ((1<<31)-1)

#ifndef MIN
#define MIN(a,b) ((a)<(b)?(a):(b))
#endif

user_idle_time()
{
    FILE    *fp;
    int     tty_idle;
    int     user_idle    = MAXINT;
	int		now;
    struct utmp utmp;

    if( (fp=fopen("/etc/utmp","r")) == NULL ) {
        EXCEPT( "fopen of utmp" );
    }

    while( fread( (char *)&utmp, sizeof utmp, 1, fp ) ) {
#if defined(AIX31) || defined(IRIX331)
		if( utmp.ut_type != USER_PROCESS )
#else
		if( utmp.ut_name[0] == '\0' )
#endif
            continue;

        tty_idle = tty_idle_time( utmp.ut_line );
        user_idle = MIN( tty_idle, user_idle );
    }
    (void)fclose( fp );

	if( KbdDev ) {
        tty_idle = tty_idle_time( KbdDev );
        user_idle = MIN( tty_idle, user_idle );
		dprintf( D_FULLDEBUG, "/dev/%s idle %d seconds\n", KbdDev, tty_idle );
	}

	if( MouseDev ) {
        tty_idle = tty_idle_time( MouseDev );
        user_idle = MIN( tty_idle, user_idle );
		dprintf( D_FULLDEBUG, "/dev/%s idle %d seconds\n", MouseDev, tty_idle );
	}

    now = (int)time( (time_t *)0 );
	user_idle = MIN( now - Last_X_Event, user_idle );

	dprintf( D_FULLDEBUG, "User Idle Time is %d seconds\n", user_idle );
    return user_idle;
}

calc_Ncpus()
{
#ifdef sequent
    int     cpus = 0;
    int     eng;

    if( (eng=tmp_ctl(TMP_NENG,0)) < 0 ) {
        perror( "tmp_ctl(TMP_NENG,0)" );
        exit( 1 );
    }

    while( eng-- ) {
        if( tmp_ctl(TMP_QUERY,eng) == TMP_ENG_ONLINE ) {
            cpus++;
        }
    }
    return cpus;
#else sequent
    return 1;
#endif sequent
}


tty_idle_time( file )
char    *file;
{
    struct stat buf;
    long        now;
    long        answer;
    static char     pathname[100] = "/dev/";

    (void)strcpy( &pathname[5], file );
    if( stat(pathname,&buf) < 0 ) {
		dprintf( D_FULLDEBUG, "Error on stat(%s,0x%x), errno = %d\n",
			pathname, &buf, errno );
		buf.st_atime = 0;
    }

    now = (int)time( (time_t *)0 );
    answer = now - buf.st_atime;

		/* This only happens if somebody screws up the date on the
		   machine :-<  */
	if( answer < 0 ) {
		answer = 0;
	}

    /*
    printf( "%s: %d seconds idle\n", file, answer );
    */
    return answer;
}

mail_context( name, context, fp )
char	*name;
CONTEXT	*context;
FILE	*fp;
{
    int     i, j;
	EXPR	*expr;

	fprintf( fp, "%s\n", name );
    for( i=0; i<context->len; i++ ) {
        fprintf( fp, "Stmt %2d:", i );
		expr = context->data[i];

        for( j=0; j<expr->len; j++ ) {
            display_elem( expr->data[j], fp );
            if( j+1 < expr->len ) {
                (void)fputc( ' ', fp );
            }
        }
        (void)fputc( '\n', fp );
    }
}

extern int	errno;
kill_starter( pid, signo )
int pid, signo;
{
	struct stat buf;

	for( errno=0; stat(Starter, &buf) < 0; errno=0 ) {
		if( errno == ETIMEDOUT ) {
			continue;
		}
		EXCEPT( "kill_starter( %d, %d ): cannot stat <%s> -- errno = %d\n",
					pid, signo, Starter, errno );
	}

#ifdef NFSFIX
	/* Must be root to kill starter.  Starter will have client uid and gid. */
	set_root_euid(__FILE__,__LINE__);
#endif NFSFIX
	(void) kill( pid, signo );
#ifdef NFSFIX
	/* Back to being condor. */
	set_condor_euid(__FILE__,__LINE__);
#endif NFSFIX
}

abort()
{
	(void)signal( SIGQUIT, SIG_DFL );
	(void)kill( getpid(), SIGQUIT );
	(void)sigpause( 0 );
}

char *
calc_subnet_name()
{
	struct hostent	*hostp;
	char			hostname[MAXHOSTNAMELEN];
	char			subnetname[MAXHOSTNAMELEN];
	char			*subnet_ptr;
	char			*host_addr_string;
	int				subnet_length;
	struct in_addr	in;

	if( gethostname(hostname, sizeof(hostname)) == -1 ) {
		dprintf( D_ALWAYS, "Gethostname failed");
		return strdup("");
	}
	if( (hostp = gethostbyname(hostname)) == NULL ) {
		dprintf( D_ALWAYS, "Gethostbyname failed");
		return strdup("");
	}
	bcopy((char *)hostp->h_addr, (char *) &in, hostp->h_length);
	host_addr_string = inet_ntoa( in );
	if( host_addr_string ) {
		subnet_ptr = (char *) rindex(host_addr_string, '.');
		if(subnet_ptr == NULL) {
			return strdup("");
		}
		subnet_length = subnet_ptr - host_addr_string;
		strncpy(subnetname, host_addr_string, subnet_length);
		subnetname[subnet_length] = '\0';
		return (strdup(subnetname));
	}
	return strdup("");
}

machine_idle()
{
	int		start;
	char	*state;

	if( evaluate_bool("START",&start,MachineContext,GenericJob) < 0 ) {
		EXCEPT( "Can't evaluate START expression against GenericJob" );
	}
	if( !start ) {
		return 0;
	}

	if( evaluate_string("State",&state,MachineContext,GenericJob) < 0 ) {
		EXCEPT( "Can't evaluate State variable" );
	}
		
	return( strcmp("NoJob",state) == 0 );
}
