/* 
** Copyright 1986, 1987, 1988, 1989, 1990, 1991 University of Wisconsin
** 
** Permission to use, copy, modify, and distribute this software and its
** documentation for any purpose and without fee is hereby granted,
** provided that the above copyright notice appear in all copies and that
** both that copyright notice and this permission notice appear in
** supporting documentation, and that the name of the University of
** Wisconsin not be used in advertising or publicity pertaining to
** distribution of the software without specific, written prior
** permission.  The University of Wisconsin makes no representations about
** the suitability of this software for any purpose.  It is provided "as
** is" without express or implied warranty.
** 
** THE UNIVERSITY OF WISCONSIN DISCLAIMS ALL WARRANTIES WITH REGARD TO
** THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND
** FITNESS. IN NO EVENT SHALL THE UNIVERSITY OF WISCONSIN  BE LIABLE FOR
** ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
** WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
** ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
** OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
** 
** Authors:  Allan Bricker and Michael J. Litzkow,
** 	         University of Wisconsin, Computer Sciences Dept.
** 
*/ 


#include <stdio.h>
#include <sys/types.h>
#include <sys/param.h>
#include <sys/file.h>
#include "errno.h"
#include <sys/wait.h>
#include <sys/time.h>
#include <sys/ioctl.h>
#include <sys/acl.h>
#include <sys/utsname.h>
#include <sys/select.h>
#include <rpc/types.h>
#include <rpc/xdr.h>
#include <varargs.h>

#include "condor_types.h"
#include "ckpt_file.h"
#include "condor_sys.h"
#include "condor_rsc.h"
#include "xdr_lib.h"
#include "trace.h"
#include "except.h"
#include "debug.h"

#ifndef LINT
static char *_FileName_ = __FILE__;		/* Used by EXCEPT (see except.h)     */
#endif LINT

#ifdef CONDOR
static XDR xdr_data;	/* Only the RSCSock will use XDR */

static XDR *xdr_syscall;

static int RSCSock;
static int ErrSock;

#define XDR_BUFSIZ	(4 * 1024)

double get_time();

XDR *
RSC_Init( rscsock, errsock )
int rscsock, errsock;
{
	int XDR_Read(), XDR_Write();

	xdr_syscall = &xdr_data;
	RSCSock     = rscsock;
	xdrrec_create( xdr_syscall, XDR_BUFSIZ, XDR_BUFSIZ, &RSCSock,
			XDR_Read, XDR_Write );

	ErrSock = errsock;

	return( xdr_syscall );
}
#endif CONDOR

int CurrentSysCall;

/*
**	In the REMOTE_syscall routine, a1 through a6 are the arguments
**	to the system call.  They will be cast as needed.
*/


/*VARARGS1*/
/* REMOTE_syscall( condor_sysnum, a1, a2, a3, a4, a5, a6 ) */
REMOTE_syscall(va_alist)
va_dcl
{
	va_list pvar;
	int condor_sysnum;
#ifdef CONDOR
	char ErrBuf[ BUFSIZ ];
#endif CONDOR
	int rval;
	extern errno;

	va_start(pvar);

	condor_sysnum = va_arg(pvar, int);

	CurrentSysCall = condor_sysnum;

#ifndef CONDOR
	fprintf(stderr, "REMOTE syscall: CONDOR system call is %s\n",
		CONDOR_SYSCALLNAME(condor_sysnum));

	rval = -1;
	goto RETURN;
#else CONDOR

	/*
	**	It doesn't make sense to do these remotely.
	*/
	switch( condor_sysnum ) {
	case PSEUDO_brk:			/* char *brk( char * )                    */
	case CONDOR_brk:			/* char *brk( char *)					  */
	case CONDOR_sbrk:			/* char *sbrk( int )                      */
		sprintf(ErrBuf, "ERROR: %s call should be performed locally",
				CONDOR_SYSCALLNAME(condor_sysnum));
		dprintf(D_ALWAYS, "%s\n", ErrBuf);
		fprintf(stderr, "%s\n", ErrBuf);
		errno = ESYSNOTSUPP;
		rval = -1;
		goto RETURN;

	/*
	** Basically intended for system adminitration or otherwise not
	** very useful to condor programs.
	*/
	case CONDOR_profil:		/* profil( char *, int, int, int )        */
	case CONDOR_getevars:	/* Get environ of another process */
	case CONDOR_trcgen:		/* Kernel tracing stuff */
	case CONDOR_trcgent:	/* Kernel tracing stuff */
	case CONDOR_audit:
	case CONDOR_auditbin:
	case CONDOR_auditevents:
	case CONDOR_auditlog:
	case CONDOR_auditobj:
	case CONDOR_auditproc:
	case CONDOR_fscntl:
	case CONDOR_knlist:
	case CONDOR_mntctl:
	case CONDOR_sysconfig:
	case CONDOR_uvmount:
	case CONDOR_vmount:
	case CONDOR_swapoff:
	case CONDOR_swapqry:
	case CONDOR_ustat:
	case CONDOR_getargs:	/* Used by "ps" */
	case CONDOR_getproc:	/* Used by "ps" */
	case CONDOR_getuser:	/* Used by "ps" */
		sprintf(ErrBuf,
		"ERROR: %s: Administrative calls not implemented for condor programs",
				CONDOR_SYSCALLNAME(condor_sysnum));
		dprintf(D_ALWAYS, "%s\n", ErrBuf);
		fprintf(stderr, "%s\n", ErrBuf);
		errno = ESYSNOTSUPP;
		rval = -1;
		goto RETURN;

	/*
	** These are unimplemented because they are too difficult...
	*/
		/* Don't handle signals */
	case CONDOR_sigblock:
	case CONDOR_sigpause:
	case CONDOR_sigreturn:
	case CONDOR_sigsetmask:
	case CONDOR_sigstack:
	case CONDOR_sigvec:
	case CONDOR_sigaction:
	case CONDOR_sigcleanup:
	case CONDOR_sigprocmask:
	case CONDOR_sigsuspend:
	case CONDOR_sigpending:
		/* No fork or exec */
	case CONDOR_fork:
	case CONDOR_kwaitpid:
		/* No dynamic loading */
	case CONDOR_loadbind:
	case CONDOR_loadquery:
	case CONDOR_unload:
	case CONDOR_load:
		/* No shared memory */
	case CONDOR_shmctl:
	case CONDOR_shmget:
	case CONDOR_shmat:
	case CONDOR_shmdt:
		/* No semaphores */
	case CONDOR_semctl:
	case CONDOR_semget:
	case CONDOR_semop:
		/* No messages */
	case CONDOR_msgctl:
	case CONDOR_msgget:
	case CONDOR_msgsnd:
	case CONDOR_msgrcv:
	case CONDOR_poll:
		/* No interval timers */
	case CONDOR_absinterval:
	case CONDOR_gettimer:
	case CONDOR_gettimerid:
	case CONDOR_incinterval:
	case CONDOR_reltimerid:
	case CONDOR_resabs:
	case CONDOR_resinc:
	case CONDOR_restimer:
	case CONDOR_settimer:
	case CONDOR_nsleep:
		sprintf(ErrBuf,
		"ERROR: %s: too difficult to implement for condor programs",
				CONDOR_SYSCALLNAME(condor_sysnum));
		dprintf(D_ALWAYS, "%s\n", ErrBuf);
		fprintf(stderr, "%s\n", ErrBuf);
		errno = ESYSNOTSUPP;
		rval = -1;
		goto RETURN;

	/*
	** We just don't want condor programs doing these thing, mostly for
	** security reasons...
	*/
	case CONDOR_seteuid:
	case CONDOR_setgid:
	case CONDOR_setgidx:
	case CONDOR_setuid:
	case CONDOR_setuidx:
	case CONDOR_setpgid:
	case CONDOR_setgroups:
		sprintf(ErrBuf, "ERROR: %s: not allowed for condor programs",
				CONDOR_SYSCALLNAME(condor_sysnum));
		dprintf(D_ALWAYS, "%s\n", ErrBuf);
		fprintf(stderr, "%s\n", ErrBuf);
		errno = ESYSNOTSUPP;
		rval = -1;
		goto RETURN;

	/*
	** Undocumented system calls, even Marc Auslander doesn't know
	** about these...
	*/
	case CONDOR_Trconflag:
	case CONDOR_trchook:
	case CONDOR_trchk:
	case CONDOR_trchkt:
	case CONDOR_trchkl:
	case CONDOR_trchklt:
	case CONDOR_trchkg:
	case CONDOR_trchkgt:
		sprintf(ErrBuf, "ERROR: %s call is undocumented, can't execute\n",
				CONDOR_SYSCALLNAME(condor_sysnum));
		dprintf(D_ALWAYS, "%s\n", ErrBuf);
		fprintf(stderr, "%s\n", ErrBuf);
		errno = ESYSNOTSUPP;
		rval = -1;
		goto RETURN;

	/*
	** Unimplemented system calls, we might know how to do these, but haven't
	** done then yet...
	*/
	case CONDOR_chpriv:		/* Fine grain priveledge stuff */
	case CONDOR_fchpriv:	/* Fine grain priveledge stuff */
	case CONDOR_fstatpriv:	/* Fine grain priveledge stuff */
	case CONDOR_getpriv:	/* Fine grain priveledge stuff */
	case CONDOR_privcheck:	/* Fine grain priveledge stuff */
	case CONDOR_setpriv:	/* Fine grain priveledge stuff */
	case CONDOR_statpriv:	/* Fine grain priveledge stuff */
		sprintf(ErrBuf, "ERROR: %s: may be implemented, but not yet\n",
				CONDOR_SYSCALLNAME(condor_sysnum));
		dprintf(D_ALWAYS, "%s\n", ErrBuf);
		fprintf(stderr, "%s\n", ErrBuf);
		errno = ESYSNOTSUPP;
		rval = -1;
		_exit( 97 );
		goto RETURN;
	/*
	** These are implemented at the system call stub level.
	** Programs which are linked correctly with stubs should never
	** get here...
	*/
	case CONDOR_kwritev:	/* Implemented by calls to write() */
	case CONDOR_kreadv:		/* Implemented by calls to read() */
	case CONDOR_times:		/* Implemented by calls to getrusage and time */
		sprintf(ErrBuf, "ERROR: %s call is unimplemented by condor\n",
				CONDOR_SYSCALLNAME(condor_sysnum));
		dprintf(D_ALWAYS, "%s\n", ErrBuf);
		fprintf(stderr, "%s\n", ErrBuf);
		errno = ESYSNOTSUPP;
		rval = -1;
		_exit( 98 );
		goto RETURN;
	}

	xdr_syscall->x_op = XDR_ENCODE;
	ASSERT( xdr_int(xdr_syscall, &condor_sysnum) );

	dprintf(D_FULLDEBUG, "REMOTE_syscall: calling %s\n",
			CONDOR_SYSCALLNAME(condor_sysnum));

	switch( condor_sysnum ) {
	/*
	**	System calls with no arguments
	*/
	case CONDOR_async_daemon:	/* async_daemon()                          */
	case CONDOR_getdtablesize:	/* getdtablesize()                         */
	case PSEUDO_getegid:	/* getegid()                                   */
	case PSEUDO_geteuid:	/* geteuid()                                   */
	case CONDOR_getgid:			/* getgid()                                */
	case CONDOR_gethostid:		/* gethostid()                             */
	case CONDOR_getpagesize:	/* getpagesize()                           */
	case CONDOR_getpid:			/* getpid()                                */
	case CONDOR_getpgrp:	/* getpgrp */
	case PSEUDO_getppid:	/* getppid()                                   */
	case CONDOR_getuid:			/* getuid()                                */
	case CONDOR_sync:			/* sync()                                  */
	case CONDOR_vhangup:		/* vhangup()                               */
	case CONDOR_pause:
	case CONDOR_setsid:
		GET_RVAL(xdr_syscall);
		goto RETURN;

	/*
	**	System calls which take one integer argument
	*/
	case CONDOR__exit:			/* _exit( int status )                     */
	case CONDOR_close:			/* close( int d )                          */
	case CONDOR_dup:			/* dup( int oldd )                         */
	case CONDOR_fsync:			/* fsync( int fd )                         */
	case CONDOR_kgetpgrp:		/* kgetpgrp( int pid )                     */
	case CONDOR_nfssvc:			/* nfssvs( int ??? )                       */
	case CONDOR_reboot:			/* reboot( int howto )                     */
	case CONDOR_umask:			/* umask( int numask )                     */
	case PSEUDO_image_size:		/* image_size( int size ) (in kilobytes)   */
	case CONDOR_frevoke:
	case CONDOR_getgidx:
	case CONDOR_getpri:
	case CONDOR_getuidx:
	case CONDOR_plock:
	case CONDOR_psdanger:
	{
		int int_arg1 = va_arg(pvar, int);

		ASSERT(xdr_int(xdr_syscall, &int_arg1));
		GET_RVAL(xdr_syscall);
		goto RETURN;
	}

	/*
	**	System calls which take one long argument
	*/
	case CONDOR_sethostid:		/* sethostid( long hostid )                  */
	{
		long hostid = va_arg(pvar, long);

		ASSERT(xdr_long(xdr_syscall, &hostid));
		GET_RVAL(xdr_syscall);
		goto RETURN;
	}

	/*
	case CONDOR_dup2:			/* dup2( int oldd, int newd )                */
	case CONDOR_fchmod:			/* fchmod( int fd, int mode )                */
	case CONDOR_flock:			/* flock( int fd, int operation )            */
	case CONDOR_getpriority:	/* getpriority( int which, int who )         */
	case CONDOR_kill:			/* kill( int pid, int sig )                  */
	case CONDOR_killpg:			/* killpg( int pgrp, int sig )               */
	case CONDOR_listen:			/* listen( int s, int backlog )              */
	case CONDOR_setpgrp:		/* setpgrp( int pid, int pgrp )              */
	case CONDOR_setregid:		/* setregid( int rgid, int egid )            */
	case CONDOR_setreuid:		/* setreuid( int ruid, int euid )            */
	case CONDOR_shutdown:		/* shutdown( int s, int how )                */
	case CONDOR_fclear:
	case CONDOR_ulimit:
	{
		int int_arg1 = va_arg(pvar, int);
		int int_arg2 = va_arg(pvar, int);

		ASSERT(xdr_int(xdr_syscall, &int_arg1));
		ASSERT(xdr_int(xdr_syscall, &int_arg2));
		GET_RVAL(xdr_syscall);
		goto RETURN;
	}

	/*
	**	System calls which take an integer argument and an off_t argument
	*/
	case CONDOR_ftruncate:		/* ftruncate( int fd, off_t length )         */
	{
		int fd = va_arg(pvar, int);
		off_t length = va_arg(pvar, off_t);

		ASSERT(xdr_int(xdr_syscall, &fd));
		ASSERT(xdr_off_t(xdr_syscall, &length));
		GET_RVAL(xdr_syscall);
		goto RETURN;
	}

	/*
	**	System calls which take three integer arguments
	*/
	case CONDOR_fchown:			/* fchown( int fd, int owner, int group )   */
	case CONDOR_fcntl:			/* fcntl( int fd, int cmd, int arg )        */
	case CONDOR_setpriority:	/* setpriority( int which, int who,
																int prio )  */
	case CONDOR_socket:			/* socket( int domain, int type,			*/
	case CONDOR_lockf:
	case CONDOR_kfcntl:			/* kfcntl( int fd, int cmd, int arg ) 		*/
	case CONDOR_faccessx:		/* faccessx( int fd, int mode, int who )	*/
	{
		int int_arg1 = va_arg(pvar, int);
		int int_arg2 = va_arg(pvar, int);
		int int_arg3 = va_arg(pvar, int);

		ASSERT(xdr_int(xdr_syscall, &int_arg1));
		ASSERT(xdr_int(xdr_syscall, &int_arg2));
		ASSERT(xdr_int(xdr_syscall, &int_arg3));
		GET_RVAL(xdr_syscall);
		goto RETURN;
	}


	/*
	**	System calls which take an integer argument followed by
	**	an off_t argument followed by another integer argument.
	*/
	case CONDOR_lseek:			/* lseek( int d, off_t offset, int whence )  */
	{
		int d = va_arg(pvar, int);
		off_t offset = va_arg(pvar, off_t);
		int whence = va_arg(pvar, int);

		ASSERT(xdr_int(xdr_syscall, &d));
		ASSERT(xdr_off_t(xdr_syscall, &offset));
		ASSERT(xdr_int(xdr_syscall, &whence));
		GET_RVAL(xdr_syscall);
		goto RETURN;
	}

	/*
	**	System calls which take one string argument
	*/
	case CONDOR_acct:			/* acct( char *file )                        */
	case CONDOR_chdir:			/* chdir( char *path )                       */
	case CONDOR_chroot:			/* chroot( char *dirname )                   */
	case CONDOR_rmdir:			/* rmdir( char *path )                       */
	case CONDOR_swapon:			/* swapon( char *special )                   */
	case CONDOR_unlink:			/* unlink( char *path )                      */
	case CONDOR_umount:			/* umount( char *name )                     */
	case PSEUDO_free_fs_blocks:	/* free_fs_blocks()						   */
	case CONDOR_revoke:
	{
		char *str_arg1 = va_arg(pvar, char *);

		ASSERT(xdr_string(xdr_syscall, &str_arg1, MAXPATHLEN));
		GET_RVAL(xdr_syscall);
		goto RETURN;
	}

	/*
	**	System calls which take one string and one integer as arguments
	*/
	case CONDOR_access:			/* access( char *path, int mode )            */
	case CONDOR_chmod:			/* chmod( char *path, int mode )             */
	case CONDOR_creat:			/* creat( char *path, int mode )             */
	case CONDOR_mkdir:			/* mkdir( char *path, int mode )             */
	case CONDOR_setdomainname:	/* setdomainname( char *name, int namelen )  */
	case CONDOR_sethostname:	/* sethostname( char *name, int namelen )    */
	{
		char *str_arg1 = va_arg(pvar, char *);
		int int_arg2 = va_arg(pvar, int);


		ASSERT(xdr_string(xdr_syscall, &str_arg1, MAXPATHLEN));
		ASSERT(xdr_int(xdr_syscall, &int_arg2));
		GET_RVAL(xdr_syscall);
		goto RETURN;
	}

	/*
	**	System calls which take one string and one off_t as arguments
	*/
	case CONDOR_truncate:		/* truncate( char *path, off_t length )      */
	{
		char *path = va_arg(pvar, char *);
		off_t length = va_arg(pvar, off_t);

		ASSERT(xdr_string(xdr_syscall, &path, MAXPATHLEN));
		ASSERT(xdr_off_t(xdr_syscall, &length));
		GET_RVAL(xdr_syscall);
		goto RETURN;
	}

	case CONDOR_kioctl:
	{
		int int_arg1 = va_arg(pvar, int);
		int int_arg2 = va_arg(pvar, int);
		int int_arg3 = va_arg(pvar, int);
		char *str_arg4 = va_arg(pvar, char *);

		ASSERT(xdr_int(xdr_syscall, &int_arg1));
		ASSERT(xdr_int(xdr_syscall, &int_arg2));
		ASSERT(xdr_int(xdr_syscall, &int_arg3));
		ASSERT(xdr_string(xdr_syscall, &str_arg4, MAXPATHLEN));
		GET_RVAL(xdr_syscall);
		goto RETURN;
	}

	/*
	**	System calls which take one string and three integers as arguments
	*/
	case CONDOR_openx:
	{
		char *str_arg1 = va_arg(pvar, char *);
		int int_arg2 = va_arg(pvar, int);
		int int_arg3 = va_arg(pvar, int);
		int int_arg4 = va_arg(pvar, int);

		ASSERT(xdr_string(xdr_syscall, &str_arg1, MAXPATHLEN));
		ASSERT(xdr_int(xdr_syscall, &int_arg2));
		ASSERT(xdr_int(xdr_syscall, &int_arg3));
		ASSERT(xdr_int(xdr_syscall, &int_arg4));
		GET_RVAL(xdr_syscall);
		goto RETURN;
	}

	/*
	**	System calls which take one string and two integers as arguments
	*/
	case CONDOR_chown:			/* chown( char *path, int owner, int group ) */
	case CONDOR_mknod:			/* mknod( char *path, int mode, int dev )    */
	case CONDOR_open:			/* open( char *path, int flags, int mode )   */
	case CONDOR_accessx:		/* accessx( char *path, int mode, int who )  */
	{
		char *str_arg1 = va_arg(pvar, char *);
		int int_arg2 = va_arg(pvar, int);
		int int_arg3 = va_arg(pvar, int);

		ASSERT(xdr_string(xdr_syscall, &str_arg1, MAXPATHLEN));
		ASSERT(xdr_int(xdr_syscall, &int_arg2));
		ASSERT(xdr_int(xdr_syscall, &int_arg3));
		GET_RVAL(xdr_syscall);
		goto RETURN;
	}

	/*
	**	System calls which take two strings as arguments
	*/
	case CONDOR_link:			/* link( char *name1, char *name2 )          */
	case CONDOR_rename:			/* rename( char *from, char *to )            */
	case CONDOR_symlink:		/* symlink( char *name1, char *name2 )       */
	{
		char *str_arg1 = va_arg(pvar, char *);
		char *str_arg2 = va_arg(pvar, char *);

		ASSERT(xdr_string(xdr_syscall, &str_arg1, MAXPATHLEN));
		ASSERT(xdr_string(xdr_syscall, &str_arg2, MAXPATHLEN));
		GET_RVAL(xdr_syscall);
		goto RETURN;
	}

	/*
	** System calls with four integer arguments
	*/
	case CONDOR_fchownx:		/* fchown( int, int, int, int )    */
	{
		int int_arg1 = va_arg(pvar, int);
		int int_arg2 = va_arg(pvar, int);
		int int_arg3 = va_arg(pvar, int);
		int	int_arg4 = va_arg(pvar, int);

		ASSERT(xdr_int(xdr_syscall, &int_arg1));
		ASSERT(xdr_int(xdr_syscall, &int_arg2));
		ASSERT(xdr_int(xdr_syscall, &int_arg3));
		ASSERT(xdr_int(xdr_syscall, &int_arg4));
		GET_RVAL(xdr_syscall);
		goto RETURN;
	}

	/*
	**	Other system calls
	*/
	case CONDOR_chownx:
	{
		char *str_arg1 = va_arg(pvar, char *);
		int int_arg2 = va_arg(pvar, int);
		int int_arg3 = va_arg(pvar, int);
		int	int_arg4 = va_arg(pvar, int);

		ASSERT(xdr_string(xdr_syscall, &str_arg1, MAXPATHLEN));
		ASSERT(xdr_int(xdr_syscall, &int_arg2));
		ASSERT(xdr_int(xdr_syscall, &int_arg3));
		ASSERT(xdr_int(xdr_syscall, &int_arg4));
		GET_RVAL(xdr_syscall);
		goto RETURN;
	}


	case PSEUDO_get_file:  		/* get_file( char *remote, char *local,
													int mode )	 		*/
	{
		char	*remote = va_arg(pvar, char *);
		char	*local = va_arg(pvar, char *);
		int		mode = va_arg(pvar, int);
		int		loc_fd;
		int		open_status, write_status = 0, read_status;
		char	buf[ XDR_BLOCKSIZ ];
		int		checksum;
		int		len;
		int		file_size;
		int		bytes_to_go;
		double	start, finish;
		int		scm;

		scm = SetSyscalls( SYS_LOCAL | SYS_RECORDED );

		start = get_time();

			/* Open the local file for writing */
		loc_fd =  open( local, O_WRONLY|O_CREAT|O_TRUNC, mode );

			/* Send the remote file name */
		ASSERT(xdr_string(xdr_syscall, &remote, MAXPATHLEN));
		ASSERT( xdrrec_endofrecord(xdr_syscall,TRUE) );

			/* Get status from remote open */
		xdr_syscall->x_op = XDR_DECODE;
		ASSERT( xdrrec_skiprecord(xdr_syscall) );
		ASSERT( xdr_int(xdr_syscall,&open_status) );
		if( open_status < 0 )  {
			dprintf( D_FULLDEBUG, "Failed to open \"%s\"\n", remote );
			ASSERT( xdr_int(xdr_syscall,&errno) );
			dprintf( D_FULLDEBUG, "Errno = %d\n", errno );
			rval = -1;
			goto RETURN;
		}

			/* Get the size of the file */
		ASSERT( xdr_int(xdr_syscall,&file_size) );

			/* Transfer the data */
		for( bytes_to_go = file_size; bytes_to_go; bytes_to_go -= len ) {
			len = bytes_to_go < sizeof(buf) ? bytes_to_go : sizeof(buf);
			ASSERT(xdr_opaque(xdr_syscall,buf,len));
			write_status = write( loc_fd, buf, len );
		}
		(void)close( loc_fd );

			/* Should get file size again as a check */
		ASSERT( xdr_int(xdr_syscall,&checksum) );
		ASSERT( checksum == file_size );

			/* Get status from remote reads */
		ASSERT( xdr_int(xdr_syscall,&read_status) );

		finish = get_time();

		if( write_status >= 0 && read_status >= 0 ) {
			dprintf( D_ALWAYS,
				"PSEUDO_get_file: transferred %d bytes, %f bytes/sec\n",
				file_size, file_size / (finish - start )
			);
			rval = 0;
		} else {
			rval = -1;
		}

		(void)SetSyscalls( scm );
		goto RETURN;
	}

	case PSEUDO_send_file: 		/* send_file( char *local, char *remote,
															int mode )	 */
	{
		char	*local = va_arg(pvar, char *);
		char	*remote = va_arg(pvar, char *);
		int		mode = va_arg(pvar, int);
		int		loc_fd;
		int		read_status = 0, write_status;
		char	buf[ XDR_BLOCKSIZ ];
		int		len;
		int		bytes_to_go;
		int		file_size;
		double	start, finish;
		int		scm;

		scm = SetSyscalls( SYS_LOCAL | SYS_RECORDED );

		/*
		dprintf( D_ALWAYS, "Entered PSEUDO_send_file()\n" );
		dprintf( D_ALWAYS, "Local = \"%s\", Remote = \"%s\"\n", local, remote );
		*/
		start = get_time();

			/* Open the local file for reading */
		loc_fd = open( local, O_RDONLY, 0 );

			/* Send remote file name and mode, get back status from the open */
		ASSERT(xdr_string(xdr_syscall, &remote, MAXPATHLEN));
		ASSERT(xdr_int(xdr_syscall, &mode));
		GET_RVAL(xdr_syscall);
		if( rval < 0 ) {
			dprintf( D_FULLDEBUG, "Failed to open \"%s\", errno = %d\n",
																remote, errno );
			goto RETURN;
		}

			/* Send file length */
		if( loc_fd < 0 ) {
			file_size = 0;
		} else {
			file_size = lseek( loc_fd, 0, 2 );
			(void)lseek( loc_fd, 0, 0 );
		}
		xdr_syscall->x_op = XDR_ENCODE;
		ASSERT( xdr_int(xdr_syscall,&file_size) );

			/* Transfer the data */
		for( bytes_to_go = file_size; bytes_to_go; bytes_to_go -= len ) {
			len = bytes_to_go < sizeof(buf) ? bytes_to_go : sizeof(buf);
			read_status = read( loc_fd, buf, len );
			ASSERT( xdr_opaque(xdr_syscall,buf,len) );
		}
		(void)close( loc_fd );

			/* As a check, re-send the length */
		ASSERT( xdr_int(xdr_syscall,&file_size) );

			/* Get status from remote writes */
		GET_RVAL(xdr_syscall);
		write_status = rval;

		finish = get_time();

		if( loc_fd < 0 || read_status < 0 ) {
			rval = -1;
		} else {
			dprintf( D_ALWAYS,
				"PSEUDO_send_file: transferred %d bytes, %f bytes/sec\n",
				file_size, file_size / (finish - start )
			);

			rval = 0;
		}
		(void)SetSyscalls( scm );
		goto RETURN;
	}

	case CONDOR_wait:			/* wait( R union wait *status )              */
	{
		union wait *status = va_arg(pvar, union wait *);

		GET_RVAL(xdr_syscall);
		if( status != 0 ) {
			ASSERT(xdr_wait(xdr_syscall, status));
		} else {
			union wait dummy;
			ASSERT(xdr_wait(xdr_syscall, &dummy));
		}
		goto RETURN;
	}

	case PSEUDO_wait3: /* wait3( R union wait *status, int options,
					   **						struct rusage *rusage )
					   */
	{
		int xdr_rusage();
		union wait *status = va_arg(pvar, union wait *);
		int options = va_arg(pvar, int);
		struct rusage *rusage = va_arg(pvar, struct rusage *);
		
		ASSERT(xdr_int(xdr_syscall, &options));
		ASSERT(xdr_ptr(xdr_syscall, (caddr_t *) &rusage, xdr_rusage));
		GET_RVAL(xdr_syscall);
		ASSERT(xdr_wait(xdr_syscall, status));
		ASSERT(xdr_ptr(xdr_syscall, (caddr_t *) &rusage, xdr_rusage));
		goto RETURN;
	}

	case PSEUDO_send_rusage: /* send_rusage( M struct rusage *rusage ) */
	{
		struct rusage *rusage = va_arg(pvar, struct rusage *);

		ASSERT(xdr_rusage(xdr_syscall, rusage));	

		GET_RVAL(xdr_syscall);

		goto RETURN;
	}

	case PSEUDO_reallyexit: /* reallyexit( M union wait *status,
							**			   M struct rusage *rusage )         */
	{
		union wait *status = va_arg(pvar, union wait *);
		struct rusage *rusage = va_arg(pvar, struct rusage *);

		ASSERT(xdr_wait(xdr_syscall, status));
		ASSERT(xdr_rusage(xdr_syscall, rusage));

		GET_RVAL(xdr_syscall);

		goto RETURN;
	}

	case PSEUDO_getwd: /* getwd( R char *pathname )                          */
	{
		char *pathname;

		pathname = va_arg(pvar, char *);
		GET_RVAL(xdr_syscall);
		ASSERT(xdr_string(xdr_syscall, &pathname, MAXPATHLEN));

		if( rval != 0 ) {
			rval = (int) pathname;
		}

		goto RETURN;
	}

	case CONDOR_lstat:			/* lstat( char *path, R struct stat *buf )   */
	case CONDOR_stat:			/* stat( char *path, R struct stat *buf )    */
	{
		char *path = va_arg(pvar, char *);
		struct stat *buf = va_arg(pvar, struct stat *);
		
		ASSERT(xdr_string(xdr_syscall, &path, MAXPATHLEN));
		GET_RVAL(xdr_syscall);
		ASSERT(xdr_stat(xdr_syscall, buf));
		goto RETURN;
	}

	case CONDOR_uname:		/* uname( struct utsname * ) */
	{
		struct utsname *utsname = va_arg( pvar, struct utsname * );
		int				size = sizeof( struct utsname );

		GET_RVAL(xdr_syscall);
		ASSERT(xdr_bytes(xdr_syscall, (char **)&utsname, &size, size));
		goto RETURN;
	}

	case CONDOR_unamex:		/* uname( struct xutsname * ) */
	{
		struct xutsname *xutsname = va_arg( pvar, struct xutsname * );

		GET_RVAL(xdr_syscall);
		bzero( xutsname, sizeof(struct xutsname) );
		ASSERT(xdr_long(xdr_syscall, &(xutsname->nid)) );
		goto RETURN;
	}

	case CONDOR_pipe:			/* pipe( int fildes[2] )                     */
	{
		int *fildes = va_arg(pvar, int *);

		GET_RVAL(xdr_syscall);
		ASSERT(xdr_int(xdr_syscall, &fildes[0]));
		ASSERT(xdr_int(xdr_syscall, &fildes[1]));
		goto RETURN;
	}

	case CONDOR_settimeofday:	/* settimeofday( struct timeval *tp,
							**				 struct timezone *tzp )
							*/
	{
		struct timeval *tp = va_arg(pvar, struct timeval *);
		struct timezone *tzp = va_arg(pvar, struct timezone *);
		int xdr_timezone();
		
		ASSERT(timeval_xdr(xdr_syscall, tp));
		ASSERT(xdr_ptr(xdr_syscall, (caddr_t *) &tzp, xdr_timezone));
		GET_RVAL(xdr_syscall);
		goto RETURN;
	}

	case CONDOR_gettimeofday:	/* gettimeofday( R struct timeval *tp,
							**				 R struct timezone *tzp )
							*/
	{
		struct timeval *tp = va_arg(pvar, struct timeval *);
		struct timezone *tzp = va_arg(pvar, struct timezone *);
		int xdr_timezone();
		
		ASSERT(xdr_ptr(xdr_syscall, (caddr_t *) &tzp, xdr_timezone));

		GET_RVAL(xdr_syscall);
		ASSERT(timeval_xdr(xdr_syscall, tp));
		ASSERT(xdr_ptr(xdr_syscall, (caddr_t *) &tzp, xdr_timezone));
		goto RETURN;
	}

	case CONDOR_getdomainname:	/* getdomainname( R char *name, int namelen )*/
	case CONDOR_gethostname:	/* gethostname( R char *name, int namelen )  */
	{
		char *name = va_arg(pvar, char *);
		int namelen = va_arg(pvar, int);
		
		ASSERT(xdr_int(xdr_syscall, &namelen));
		GET_RVAL(xdr_syscall);
		ASSERT(xdr_string(xdr_syscall, &name, namelen));
		goto RETURN;
	}
		
	case CONDOR_select:			/* select( int nfds,
							**		0/VR fd_set *readfds,
							**		0/VR fd_set *writefds,
							**		0/VR fd_set *exceptfds,
							**		0/V struct timeval *timeout )
							*/
	{
		int xdr_fdset(), timeval_xdr();
		int nfds = va_arg(pvar, int);
		fd_set *readfds = va_arg(pvar, fd_set *);
		fd_set *writefds = va_arg(pvar, fd_set *);
		fd_set *exceptfds = va_arg(pvar, fd_set *);
		struct timeval *timeout = va_arg(pvar, struct timeval *);

		ASSERT(xdr_int(xdr_syscall, &nfds));
		ASSERT(xdr_ptr(xdr_syscall, (caddr_t *) &readfds, xdr_fdset));
		ASSERT(xdr_ptr(xdr_syscall, (caddr_t *) &writefds, xdr_fdset));
		ASSERT(xdr_ptr(xdr_syscall, (caddr_t *) &exceptfds, xdr_fdset));
		ASSERT(xdr_ptr(xdr_syscall, (caddr_t *) &timeout, timeval_xdr));

		GET_RVAL(xdr_syscall);

		ASSERT(xdr_ptr(xdr_syscall, (caddr_t *) &readfds, xdr_fdset));
		ASSERT(xdr_ptr(xdr_syscall, (caddr_t *) &writefds, xdr_fdset));
		ASSERT(xdr_ptr(xdr_syscall, (caddr_t *) &exceptfds, xdr_fdset));
		ASSERT(xdr_ptr(xdr_syscall, (caddr_t *) &timeout, timeval_xdr));

		goto RETURN;
	}

	case CONDOR_setgroups:		/* setgroups( int ngroups, M int *gidset )   */
	{
		int ngroups = va_arg(pvar, int);
		int *gidset = va_arg(pvar, int *);

		ASSERT(xdr_array(xdr_syscall, &gidset, &ngroups, NGROUPS, sizeof(int), xdr_int));
		GET_RVAL(xdr_syscall);
		goto RETURN;
	}

	case CONDOR_setitimer:		/* setitimer( int which,
							**			M struct itimerval *value,
							**		  R/0 struct itimerval *ovalue )
							*/
	{
		int which = va_arg(pvar, int);
		struct itimerval *value = va_arg(pvar, struct itimerval *);
		struct itimerval *ovalue = va_arg(pvar, struct itimerval *);
		
		ASSERT(xdr_int(xdr_syscall, &which));
		ASSERT(xdr_itimerval(xdr_syscall, value));
		ASSERT(xdr_ptr(xdr_syscall, (caddr_t *) &ovalue, timeval_xdr));

		GET_RVAL(xdr_syscall);

		ASSERT(xdr_ptr(xdr_syscall, (caddr_t *) &ovalue, timeval_xdr));
		goto RETURN;
	}

	case CONDOR_setrlimit:		/* setrlimit( int resource,
							**			M struct rlimit *rlp )
							*/
	{
		int resource = va_arg(pvar, int);
		struct rlimit *rlp = va_arg(pvar, struct rlimit *);
		
		ASSERT(xdr_int(xdr_syscall, &resource));
		ASSERT(xdr_rlimit(xdr_syscall, rlp));
		GET_RVAL(xdr_syscall);
		goto RETURN;
	}

	case CONDOR_getdirentries:	/* getdirentries( int fd,
							**				R char *buf,
							**				  int nbytes,
							**				R off_t *basep )
							*/
	{
		int fd = va_arg(pvar, int);
		char *buf = va_arg(pvar, char *);
		int nbytes = va_arg(pvar, int);
		long *basep = va_arg(pvar, long *);

		ASSERT(xdr_int(xdr_syscall, &fd));
		ASSERT(xdr_int(xdr_syscall, &nbytes));
		GET_RVAL(xdr_syscall);

		ASSERT(xdr_direntries(xdr_syscall, buf, rval));
		ASSERT(xdr_off_t(xdr_syscall, basep));

		goto RETURN;
	}

	case CONDOR_getgroups:		/* getgroups( int gidsetlen, R int *gidset ) */
	{
		int gidsetlen = va_arg(pvar, int);
		int *gidset = va_arg(pvar, int *);
		register int i;

		ASSERT(xdr_int(xdr_syscall, &gidsetlen));
		GET_RVAL(xdr_syscall);

		for( i = 0; i < rval; i++ ) {
			ASSERT(xdr_int(xdr_syscall, &gidset[i]));
		}
		goto RETURN;
	}

	case CONDOR_getitimer:		/* getitimer( int which,
							**			R struct itimerval *value )
							*/
	{
		int which = va_arg(pvar, int);
		struct itimerval *value = va_arg(pvar, struct itimerval *);
		
		ASSERT(xdr_int(xdr_syscall, &which));
		GET_RVAL(xdr_syscall);
		ASSERT(xdr_itimerval(xdr_syscall, value));
		goto RETURN;
	}

	case CONDOR_getrlimit:		/* getrlimit( int resource,
							**			R struct rlimit *rlp )
							*/
	{
		int resource = va_arg(pvar, int);
		struct rlimit *rlp = va_arg(pvar, struct rlimit *);
		
		ASSERT(xdr_int(xdr_syscall, &resource));
		GET_RVAL(xdr_syscall);
		ASSERT(xdr_rlimit(xdr_syscall, rlp));
		goto RETURN;
	}

	case CONDOR_getrusage:		/* getrusage( int who,
							**			R struct rusage *rusage )
							*/
	{
		int who = va_arg(pvar, int);
		struct rusage *rusage = va_arg(pvar, struct rusage *);

		
		ASSERT(xdr_int(xdr_syscall, &who));
		GET_RVAL(xdr_syscall);
		ASSERT(xdr_rusage(xdr_syscall, rusage));
		goto RETURN;
	}

	case CONDOR_fstat:			/* fstat( int fd, R struct stat *buf )       */
	{
		int fd = va_arg(pvar, int);
		struct stat *buf = va_arg(pvar, struct stat *);
		
		ASSERT(xdr_int(xdr_syscall, &fd));
		GET_RVAL(xdr_syscall);
		ASSERT(xdr_stat(xdr_syscall, buf));
		goto RETURN;
	}

	case CONDOR_usrinfo:	/* usrinfo( int cmd, char *buf, int count ) */
	{
		int		cmd = va_arg( pvar, int );
		char	*buf = va_arg( pvar, char *);
		int		count = va_arg( pvar, int );
		
		ASSERT(xdr_int(xdr_syscall, &cmd));
		ASSERT(xdr_int(xdr_syscall, &count));

		GET_RVAL(xdr_syscall);
		ASSERT(xdr_bytes(xdr_syscall, &buf, &count, count));
		goto RETURN;
	}

	case CONDOR_statacl:	/* statacl( char *path, int cmd,
												struct acl *acl, int size ) */
	{
		char *path = va_arg(pvar, char *);
		int cmd = va_arg(pvar, int);
		struct acl *acl = va_arg(pvar, struct acl *);
		int size = va_arg(pvar, int);
		
		ASSERT(xdr_string(xdr_syscall, &path, MAXPATHLEN));
		ASSERT(xdr_int(xdr_syscall, &cmd));
		ASSERT(xdr_int(xdr_syscall, &size));

		GET_RVAL(xdr_syscall);
		ASSERT(xdr_bytes(xdr_syscall, (char **)&acl, &size, size));
		goto RETURN;
	}

	case CONDOR_fstatacl:	/* fstatacl( int fd, int cmd,
											struct acl *acl, int size ) */
	{
		int fd = va_arg(pvar, int);
		int cmd = va_arg(pvar, int);
		struct acl *acl = va_arg(pvar, struct acl *);
		int size = va_arg(pvar, int);
		
		ASSERT(xdr_int(xdr_syscall, &fd ));
		ASSERT(xdr_int(xdr_syscall, &cmd));
		ASSERT(xdr_int(xdr_syscall, &size));

		GET_RVAL(xdr_syscall);
		ASSERT(xdr_bytes(xdr_syscall, (char **)&acl, &size, size));
		goto RETURN;
	}

	case CONDOR_fchacl:		/* fchacl( int fd, struct acl *acl, int size ) */
	{
		int fd = va_arg(pvar, int);
		struct acl *buf = va_arg(pvar, struct acl *);
		int	len = va_arg(pvar,int);
		
		ASSERT(xdr_int(xdr_syscall, &fd));
		ASSERT(xdr_int(xdr_syscall, &len));
		ASSERT(xdr_bytes(xdr_syscall, (char **)&buf, &len, 4096));
		GET_RVAL(xdr_syscall);
		goto RETURN;
	}

	case CONDOR_chacl:		/* chacl( char *path, struct acl *acl, int len ) */
	{
		char *path = va_arg(pvar, char *);
		struct acl *buf = va_arg(pvar, struct acl *);
		int	len = va_arg(pvar,int);
		
		ASSERT(xdr_string(xdr_syscall, &path, MAXPATHLEN));
		ASSERT(xdr_int(xdr_syscall, &len));
		ASSERT(xdr_bytes(xdr_syscall, (char **)&buf, &len, 4096));
		GET_RVAL(xdr_syscall);
		goto RETURN;
	}

	case CONDOR_statx:
	{
		char *path = va_arg(pvar, char *);
		struct stat *buf = va_arg(pvar, struct stat *);
		int	len = va_arg(pvar,int);
		int	cmd = va_arg(pvar,int);
		
		ASSERT(xdr_string(xdr_syscall, &path, MAXPATHLEN));
		ASSERT(xdr_int(xdr_syscall, &len));
		ASSERT(xdr_int(xdr_syscall, &cmd));
		GET_RVAL(xdr_syscall);
		ASSERT(xdr_opaque(xdr_syscall, buf,(unsigned)len));
		goto RETURN;
	}

	case CONDOR_getdirent:
	{
		int	fd = va_arg(pvar, int);
		char	*buf = va_arg(pvar, char *);
		int		count = va_arg(pvar,int);

		ASSERT( xdr_int(xdr_syscall,&fd));
		ASSERT( xdr_int(xdr_syscall,&count));
		GET_RVAL(xdr_syscall);
		ASSERT( xdr_bytes(xdr_syscall,&buf,&count,4096) );
		goto RETURN;
	}

	case CONDOR_fstatx:
	{
		int fd = va_arg(pvar, int);
		struct stat *buf = va_arg(pvar, struct stat *);
		int	len = va_arg(pvar,int);
		int	cmd = va_arg(pvar,int);
		
		ASSERT(xdr_int(xdr_syscall, &fd));
		ASSERT(xdr_int(xdr_syscall, &len));
		ASSERT(xdr_int(xdr_syscall, &cmd));
		GET_RVAL(xdr_syscall);
		ASSERT(xdr_stat(xdr_syscall, buf));
		goto RETURN;
	}

	case CONDOR_fstatfs:		/* fstatfs( int fd, struct statfs *buf )     */
	{
		int fd = va_arg(pvar, int);
		struct statfs *buf = va_arg(pvar, struct statfs *);
		
		ASSERT(xdr_int(xdr_syscall, &fd));
		GET_RVAL(xdr_syscall);
		ASSERT(xdr_statfs(xdr_syscall, buf));
		goto RETURN;
	}

	case CONDOR_utimes:			/* utimes( char *file, M struct timeval *tvp)*/
	{
		char *file = va_arg(pvar, char *);
		struct timeval *tvp = va_arg(pvar, struct timeval *);

		ASSERT(xdr_string(xdr_syscall, &file, MAXPATHLEN));
		ASSERT(timeval_xdr(xdr_syscall, &tvp[0]));
		ASSERT(timeval_xdr(xdr_syscall, &tvp[1]));

		GET_RVAL(xdr_syscall);
		goto RETURN;
	}

	case CONDOR_readlink:		/* readlink( char *path, char *buf,
																int bufsiz ) */
	{
		char *path = va_arg(pvar, char *);
		char *buf = va_arg(pvar, char *);
		int bufsiz = va_arg(pvar, int);

		ASSERT(xdr_string(xdr_syscall, &path, MAXPATHLEN));
		ASSERT(xdr_int(xdr_syscall, &bufsiz));
		GET_RVAL(xdr_syscall);
		ASSERT(xdr_string(xdr_syscall, &buf, bufsiz));
		goto RETURN;
	}

	case PSEUDO_extern_name:		/* extern_name( char *path, char *buf,
																int bufsiz ) */
	{
		char *path = va_arg(pvar, char *);
		char *buf = va_arg(pvar, char *);
		int bufsiz = va_arg(pvar, int);

		ASSERT(xdr_string(xdr_syscall, &path, MAXPATHLEN));
		ASSERT(xdr_int(xdr_syscall, &bufsiz));
		GET_RVAL(xdr_syscall);
		ASSERT(xdr_string(xdr_syscall, &buf, bufsiz));
		goto RETURN;
	}

	case CONDOR_adjtime:		/* adjtime( M struct timeval *delta,
							**		  R/0 struct timeval *olddelta )
							*/
	{
		struct timeval *delta = va_arg(pvar, struct timeval *);
		struct timeval *olddelta = va_arg(pvar, struct timeval *);

		ASSERT(timeval_xdr(xdr_syscall, delta));
		ASSERT(xdr_ptr(xdr_syscall, (caddr_t *) &olddelta, timeval_xdr));

		GET_RVAL(xdr_syscall);

		ASSERT(xdr_ptr(xdr_syscall, (caddr_t *) &olddelta, timeval_xdr));

		goto RETURN;
	}

	case CONDOR_statfs:			/* statfs( char *path, R struct statfs *buf )*/
	{
		char *path = va_arg(pvar, char *);
		struct statfs *buf = va_arg(pvar, struct statfs *);
		
		ASSERT(xdr_string(xdr_syscall, &path, MAXPATHLEN));
		GET_RVAL(xdr_syscall);
		ASSERT(xdr_statfs(xdr_syscall, buf));
		goto RETURN;
	}

	case CONDOR_write:			/* write( int d, M char *buf, int nbytes )   */
	{
		int d = va_arg(pvar, int);
		char *buf = va_arg(pvar, char *);
		int nbytes = va_arg(pvar, int);

		ASSERT(xdr_int(xdr_syscall, &d));
		ASSERT(xdr_int(xdr_syscall, &nbytes));
		ASSERT(xdr_bytes(xdr_syscall, (caddr_t *) &buf, &nbytes, nbytes));

		GET_RVAL(xdr_syscall);
		goto RETURN;
	}

	case CONDOR_read:			/* write( int d, R char *buf, int nbytes )   */
	{
		int d = va_arg(pvar, int);
		char *buf = va_arg(pvar, char *);
		int nbytes = va_arg(pvar, int);

		ASSERT(xdr_int(xdr_syscall, &d));
		ASSERT(xdr_int(xdr_syscall, &nbytes));

		GET_RVAL(xdr_syscall);

		ASSERT(xdr_bytes(xdr_syscall, (caddr_t *) &buf, &nbytes, nbytes));
		goto RETURN;
	}

	case CONDOR_ioctl:			/* ioctl( int d, u_long request, char *argp )*/
	{
		int d = va_arg(pvar, int);
		u_long request = va_arg(pvar, u_long);
		char *argp = va_arg(pvar, char *);

		ASSERT(xdr_int(xdr_syscall, &d));
		ASSERT(xdr_u_long(xdr_syscall, &request));

#if ( defined(SUNOS41) && defined(SPARC) ) || defined(ULTRIX42)
		if( request & _IOC_IN ) {
			ASSERT(xdr_opaque(xdr_syscall, argp, request & _IOCPARM_MASK));
		}
#else	/* SUNOS41 && SPARC */
		if( request & IOC_IN ) {
			ASSERT(xdr_opaque(xdr_syscall, argp, request & IOCPARM_MASK));
		}
#endif /* SUNOS41 && SPARC */

		GET_RVAL(xdr_syscall);

#if ( defined(SUNOS41) && defined(SPARC) ) || defined(ULTRIX42)
		if( request & _IOC_OUT ) {
			ASSERT(xdr_opaque(xdr_syscall, argp, request & _IOCPARM_MASK));
		}
#else	/* SUNOS41 && SPARC */
		if( request & IOC_OUT ) {
			ASSERT(xdr_opaque(xdr_syscall, argp, request & IOCPARM_MASK));
		}
#endif	/* SUNOS41 && SPARC */
	}

	case CONDOR_writev:			/* writev( int d, struct iovec *iov,
																int iovcnt ) */
	case CONDOR_readv:			/* readv( int, VR iov *, int )               */

	case CONDOR_ptrace:			/* ptrace( int, int, VR int *, int           */
	case CONDOR_quotactl:		/* quotaactl( int, str, int, VRU caddr_t )   */
	case CONDOR_execv:			/* execv( str, M str[] )                     */
	case CONDOR_execve:			/* execve( str, M str[], M str[] )           */

	case CONDOR_send:			/* send( int, M caddr_t, int, int )          */
	case CONDOR_sendto:			/* sendto( int, M caddr_t, int, int,
										M sockaddr *, int                    */
	case CONDOR_sendmsg:		/* sendmsg( int, V msghdr[], int )           */

	case CONDOR_recv:			/* recv( int, R caddr_t, int, int )          */
	case CONDOR_recvfrom:		/* recvfrom( int, R caddr_t, int, int,
											0/R sockaddr *, VR int * )       */
	case CONDOR_recvmsg:		/* recvmsg( int, VR msghdr[], int )          */

	case CONDOR_setsockopt:		/* setsockopt( int, int, int, M caddr_t,int) */
	case CONDOR_getsockopt:		/* getsockopt( int, int, int,
									R caddr_t, VR int * )                    */
	case CONDOR_socketpair:		/* socketpair( int, int, int, R int[2] )     */
	case CONDOR_bind:			/* bind( int, sockaddr *, int )              */
	case CONDOR_connect:		/* connect( int, sockaddr *, int )           */
	case CONDOR_mount:			/* mount( int, str, int,
										MU caddr_t )                         */
	case CONDOR_accept:			/* accept( int, R sockaddr *, VR int * )     */
	case CONDOR_getsockname:	/* getsockname( int, R sockaddr *, VR int * )*/

	case CONDOR_getpeername:	/* getpeername( int, R sockaddr *, VR int * )*/


	/*
	**	System calls with (as yet) unknown arguments
	*/
	case CONDOR_exportfs:
	case CONDOR_getdopt:
	case CONDOR_getfh:
	case CONDOR_madvise:
	case CONDOR_mincore:
	case CONDOR_mmap:
	case CONDOR_mprotect:
	case CONDOR_mremap:
	case CONDOR_munmap:
	case CONDOR_setdopt:
	case CONDOR_sstk:
		sprintf(ErrBuf, "ERROR: %s is a currently unsupported system call\n",
			CONDOR_SYSCALLNAME(condor_sysnum));
		dprintf(D_ALWAYS, "%s\n", ErrBuf);
		fprintf(stderr, "%s\n", ErrBuf);
		errno = ESYSNOTSUPP;
		rval = -1;
		goto RETURN;

	default:
		sprintf(ErrBuf, "REMOTE_syscall() ERROR: %d is an unknown system call number\n",
			condor_sysnum);
		dprintf(D_ALWAYS, "%s\n", ErrBuf);
		fprintf(stderr, "%s\n", ErrBuf);
		errno = ESYSNOTSUPP;
		rval = -1;
		goto RETURN;
	}
#endif ! CONDOR

RETURN:
	va_end(pvar);

	return( rval );
}

#ifdef CONDOR

static
XDR_Read( iohandle, buf, len )
int *iohandle;
char *buf;
u_int len;
{
	register int scm = SetSyscalls( SYS_LOCAL | SYS_RECORDED );
	int cnt;

	dprintf(D_XDR, "XDR_Read: about to read(%d, 0x%x, %d)\n",
			*iohandle, buf, len);

	cnt = read(*iohandle, buf, len);
	if( cnt == 0 ) {
		dprintf(D_XDR, "XDR_Read: cnt was zero, returning -1\n");
		cnt = -1;
	}

	dprintf(D_XDR, "XDR_Read: cnt = %d\n", cnt);

	(void) SetSyscalls( scm );

	return( cnt );
}

static
XDR_Write( iohandle, buf, len )
int *iohandle;
char *buf;
u_int len;
{
	register int scm = SetSyscalls( SYS_LOCAL | SYS_RECORDED );
	int cnt;

	dprintf(D_XDR, "XDR_Write: about to write(%d, 0x%x, %d)\n",
			*iohandle, buf, len);

	cnt = write( *iohandle, buf, len );

	dprintf(D_XDR, "XDR_Write: cnt = %d\n", cnt);

	if( cnt != len ) {
		EXCEPT("write(%d, 0x%x, %d) returned %d", *iohandle, buf, len, cnt);
	}

	(void) SetSyscalls( scm );

	return( cnt );
}

#endif CONDOR


#ifdef AIX31
wait_for_reply()
{
	int		count;
	fd_set	readfds;
	int		scm;

	scm = SetSyscalls( SYS_LOCAL | SYS_RECORDED );

	FD_ZERO( &readfds );
	FD_SET( RSCSock, &readfds );

	count = select( FD_SETSIZE, (int *)&readfds, (int *)0, (int *)0,
														(struct timeval *)0);
	
	if( count < 0 ) {
		EXCEPT( "select returns %d, errno = %d\n", count, errno );
	}
	if( !FD_ISSET(RSCSock,&readfds) ) {
		EXCEPT( "select returned, but fd %d NOT ready for reading\n", RSCSock );
	}
	(void)SetSyscalls( scm );
}
#endif AIX31
