/* 
** Copyright 1986, 1987, 1988, 1989, 1990, 1991 by the Condor Design Team
** 
** Permission to use, copy, modify, and distribute this software and its
** documentation for any purpose and without fee is hereby granted,
** provided that the above copyright notice appear in all copies and that
** both that copyright notice and this permission notice appear in
** supporting documentation, and that the names of the University of
** Wisconsin and the Condor Design Team not be used in advertising or
** publicity pertaining to distribution of the software without specific,
** written prior permission.  The University of Wisconsin and the Condor
** Design Team make no representations about the suitability of this
** software for any purpose.  It is provided "as is" without express
** or implied warranty.
** 
** THE UNIVERSITY OF WISCONSIN AND THE CONDOR DESIGN TEAM DISCLAIM ALL
** WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES
** OF MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE UNIVERSITY OF
** WISCONSIN OR THE CONDOR DESIGN TEAM BE LIABLE FOR ANY SPECIAL, INDIRECT
** OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS
** OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE
** OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE
** OR PERFORMANCE OF THIS SOFTWARE.
** 
** Authors:  Allan Bricker and Michael J. Litzkow,
** 	         University of Wisconsin, Computer Sciences Dept.
** 
*/ 

#ifdef VOID_SIGNAL_RETURN
#define sighandler_t void
#else
#define sighandler_t int
#endif

#define TRACE \
dprintf( D_ALWAYS, "Trace at %d in %s\n", __LINE__, __FILE__ )

#include <stdio.h>
#include <pwd.h>

#if defined(IRIX331)
#define __EXTENSIONS__
#include <signal.h>
#undef __EXTENSIONS__
#else
#include <signal.h>
#endif endif

#include <sys/types.h>
#include <sys/file.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <sys/param.h>

#if defined(AIX31)
#include <sys/statfs.h>
#endif

#if defined(ULTRIX42)
#include <sys/mount.h>
#endif

#ifdef IRIX331
#include <sys/statfs.h>
#endif

#if !defined(ULTRIX) && !defined(ULTRIX42) && !defined(IRIX331) && !defined(AIX31)
#include <sys/vfs.h>
#endif

#ifdef HPUX8
#include <a.out.h>
#undef n_name	/* black magic from HP-UX nlist() man page */
#include <sys/resource.h>
#define COFF
#endif


#include <sys/dir.h>
#include <sys/user.h>
#include <rpc/types.h>
#include <rpc/xdr.h>
#include <errno.h>
#include <fcntl.h>

#include <strings.h>

extern char *strdup();

char *CkptDir = NULL;	/* Directory where ckpt resides. */

#ifdef MIPS
#include <sys/exec.h>
#include <syms.h>
#define AOUT_MAGIC ZMAGIC
#define COFF_MAGIC MIPSELMAGIC
#define COFF
#else MIPS

/*
** The syms.h file includes a "#pragma once" directive, but the irix
** cpp doesn't accept it.  A.out.h eventually ends up including
** syms.h.  Here's a crude work around...
*/
#if defined(IRIX331)
#define __EXTENSIONS__
#define pragma define
#include <a.out.h>
#undef pragma
#define AOUT_MAGIC ZMAGIC
#define COFF_MAGIC XLMAGIC
#define COFF
#else
#include <a.out.h>
#endif


#ifdef AIX31
#undef n_name /* magic from AIX man page... */
#define COFF_MAGIC U802TOCMAGIC
#define AOUT_MAGIC 0413
#define COFF
#endif AIX31
#endif MIPS

#if defined(SUNOS32) || defined(SUNOS40) || defined(SUNOS41) || defined(CM5)
#include <sys/core.h>
#endif

#ifdef AIX31
#include <sys/statfs.h>
#include <sys/id.h>
#endif AIX31

#if defined(MIPS) || defined(AIX31)
/* This is usually included in <a.out.h> */
#include <nlist.h>
#endif

#include <ctype.h>

#include "debug.h"
#include "except.h"
#include "proc.h"
#include "trace.h"
#include "xdr_lib.h"
#include "condor_sys.h"
#include "fileno.h"
#include "expr.h"
#include "shadow.h"     /* Need to include for EXECFAILED. */
#include "ckpt_file.h"

#define MEG (1<<20)
#define SLOP 50



static char *_FileName_ = __FILE__;		/* Used by EXCEPT (see except.h)     */
char	*param();

#define MINUTE 60
#define HOUR (MINUTE * 60)

#if defined(IRIX331)
#define BADSIG SIG_ERR
#endif

extern int	errno;
extern FILE	*DebugFP;
extern char	*SigNames[];

char	*Execute;
int		MinCkptInterval;
int		MaxCkptInterval;
int		CkptInterval;

int		ChildPid;
int		ImageSize; /* in Kilobytes */
char	Shortname[ MAXPATHLEN ];
char	CkptTmpName[ MAXPATHLEN ];

#if defined(HPUX8)
char	Grot[ MAXPATHLEN ];
#endif

char	*CoreName = "core";
char	*ReturnedCore[ MAXPATHLEN ];

char	*OrigName = NULL;
char	*CkptName = NULL;
char	*Stdin = NULL;
char	*Stdout = NULL;
char	*Stderr = NULL;
char	*RootDir = NULL;
char	*CmdName = NULL;
char	*Args = NULL;
char	*Env = NULL;
char	*JobArgs[ MAXPATHLEN / 2 ];
char	Condor_CWD[1024];
int		Cluster, Proc;
int		Suspended;
int		CkptCreated;
int		JobExited;
int		Aborted;
struct rusage AccumRusage;
int		UserUid = -1;
int		UserGid = -1;

int		vacate_order(), checkpoint_order(), suspend_order(), resume_order();
int		Cleanup(), die(), abort();

double	StartCoredump, EndCoredump;
double	get_time();

main( argc, argv )
int		argc;
char	*argv[];
{
	int		i, nfds;
	union wait status;
	struct rusage rusage;
	XDR		*xdrs, *RSC_Init();
	int rval;
	char	**argp;
	int		scm;
	int		log_fd = -1;
	int		wait_for_debugger = 1;
	int		fd;
	char	*value, *GetEnvParam();

/*
** Wait up for one of those nice debuggers which attaches to a
** running process.
*/
#ifdef DEBUGGER
	while( wait_for_debugger )
		;
#endif

	(void)SetSyscalls( SYS_LOCAL | SYS_UNRECORDED, __LINE__, __FILE__ );

	fd = dup2( 1, RSC_SOCK );
	fd = dup2( 2, CLIENT_LOG );
	InitStaticFile( RSC_SOCK, FI_WELL_KNOWN );		/* remote system calls */
	InitStaticFile( CLIENT_LOG, FI_WELL_KNOWN );	/* all logging */

	(void) SetSyscalls( SYS_LOCAL | SYS_RECORDED, __LINE__, __FILE__ );

	setlinebuf( stderr );
	DebugFlags |= D_ALWAYS | D_NOHEADER;


#ifdef LOCAL_LOGGING
	log_fd = open( "/tmp/starter_log", O_WRONLY | O_CREAT, 0664 );
	lseek( log_fd, 0, L_XTND );
	close( fileno(stderr) );
	dup2( log_fd, fileno(stderr) );
#else /* LOCAL_LOGGING */
	/*
	stderr->_file = CLIENT_LOG;
	*/
	close( fileno(stderr) );
	dup2( CLIENT_LOG, fileno(stderr) );
#endif /* LOCAL_LOGGING */


	dprintf( D_ALWAYS, "********** STARTER starting up ***********\n" );

#if defined( HPUX8)	/* need to get this symbol defined for linking */
	getwd( Grot );
#endif

	config( argv[0], (CONTEXT *)0 );

#ifdef NFSFIX
	/* Must be condor to update ckpts. */
	set_condor_euid(__FILE__,__LINE__);
#endif NFSFIX

	_EXCEPT_Cleanup = Cleanup;


	if( argc != 2 ) {
		usage();
	}

	init_params();

	set_sigs();
	(void)sigsetmask( 0 );

	if( chdir(Execute) ) {
		EXCEPT( "chdir(%s)", Execute );
	}

		/* Initialize remote system calls */
	xdrs = RSC_Init( RSC_SOCK, CLIENT_LOG );

	xdrrec_skiprecord(xdrs);
	umask( 0 );

	GetFileNames( xdrs );

		/* find out if user wants checkpointing */
	value = GetEnvParam( "CHECKPOINT" );
	if( value && stricmp("false",value) ) {
		MinCkptInterval = 0;
	}

	(void)SetSyscalls( SYS_REMOTE | SYS_RECORDED, __LINE__, __FILE__ );
	(void)REMOTE_syscall( PSEUDO_getwd, &Condor_CWD[0] );
	UserUid = geteuid();
	UserGid = getegid();
	(void)SetSyscalls( SYS_LOCAL | SYS_RECORDED, __LINE__, __FILE__ );

	get_file( OrigName, Shortname, 0775 );

	if( magic_check(Shortname) < 0 ) {
		bzero( (char *)&AccumRusage, sizeof(AccumRusage) );
		status.w_termsig = 0;		/* no sig, but a coredump -- can only */
		status.w_coredump = 1;		/* be if we set it on purpose */
		status.w_retcode = ENOEXEC;	/* now say why we didn't run the job */
		(void)REMOTE_syscall( PSEUDO_reallyexit, &status, &AccumRusage);
		Cleanup( __LINE__, __FILE__ );
		exit( 0 );
	}

	if( symbol_main_check(Shortname) < 0 ) {
		dprintf( D_ALWAYS, "symbol_main_check() failed\n" );
		bzero( (char *)&AccumRusage, sizeof(AccumRusage) );
		status.w_termsig = 0;
		status.w_coredump = 1;
		status.w_retcode = 0;
		(void)REMOTE_syscall( PSEUDO_reallyexit, &status, &AccumRusage);
		Cleanup( __LINE__, __FILE__ );
		exit( 0 );
	}

	CkptInterval = MinCkptInterval;

#if !defined(HPUX8)	/* HPUX doesn't support any of these... */

		/* Set up reasonable resource limits */
	limit( RLIMIT_CPU, RLIM_INFINITY );
	limit( RLIMIT_FSIZE, RLIM_INFINITY );
	limit( RLIMIT_DATA, RLIM_INFINITY );
#ifndef AIX31
	limit( RLIMIT_STACK, RLIM_INFINITY );
#endif AIX31
	limit( RLIMIT_RSS, RLIM_INFINITY );
/*
** We'd like to limit the core size to avoid overflowing the local file
** system, but SUN doesn't do it right.  Their core files have "holes"
** in them, and are always at least 8 megabytes virtual size.  Generally
** the actual size (how much disk is required) is much smaller.
** Unfortunately, their version of RLIMIT_CORE limits the virtual
** rather than the actual size of the core file, sigh.
*/
#define FOO
#if defined(SUNOS40) || defined(SUNOS41) || defined(CM5) || defined(FOO)
	limit( RLIMIT_CORE, RLIM_INFINITY );
#else
	limit( RLIMIT_CORE, (calc_free_disk_blocks() - SLOP) * 1024 );
#endif

#endif /* !HPUX8 */

		/* The client should only have RSC_SOCK and CLIENT_LOG open
		   when it is born.  That's all we use, so close any others. */
	for( i=0; i<RSC_SOCK; i++ ) {

#if defined(IRIX331)
		if( i == fileno(stderr) ) {
			continue;
		}
#endif

		if( i==log_fd ) {
			continue;
		}
		(void)close( i );
	}

	for(;;) {
			/* Block every signal except TRAP, (so that we can run with a
			   debugger if desired). */
		(void)sigsetmask( ~sigmask(SIGTRAP) );
		if( (ChildPid = vfork()) < 0 ) {
			EXCEPT( "vfork" );
		}

		if( ChildPid ) {		/* The parent */
			(void)sigsetmask( 0 );

			if( CkptInterval ) {
				(void)alarm( (unsigned)CkptInterval );
				dprintf( D_ALWAYS, "Set alarm for %d seconds\n", CkptInterval );
			} else {
				dprintf( D_ALWAYS, "Periodic Checkpointing Disabled\n" );
			}

			WaitForChild(&status, &rusage);
			(void)alarm( (unsigned) 0 );

			update_rusage( &AccumRusage, &rusage );

			if( CkptCreated ) {
				/* Update shadow with accumulated usage for the job. */
				dprintf( D_FULLDEBUG, "Ckpt created.  Update rusage.\n" );
				rval = REMOTE_syscall( PSEUDO_send_rusage, &AccumRusage );
				dprintf( D_FULLDEBUG, "PSEUDO_send_rusage returned %d\n", rval);
			}

			dprintf( D_FULLDEBUG, "Accumulated Rusage\n" );
			dprintf( D_FULLDEBUG, "ru_utime = %d.%06d\n",
					AccumRusage.ru_utime.tv_sec, AccumRusage.ru_utime.tv_usec );
			dprintf( D_FULLDEBUG, "ru_stime = %d.%06d\n",
					AccumRusage.ru_stime.tv_sec, AccumRusage.ru_stime.tv_usec );

			if( Aborted ) {
				/* Send ckpt file only if enough space on submitting. */
				if( CkptCreated && FileWillFit(Shortname, CkptDir) ) {
					SendCkptFile( Shortname, CkptName );
					status.w_termsig = SIGQUIT;
				} else {
					status.w_termsig = SIGKILL;
				}
			}

			if( JobExited || Aborted ) {
				dprintf(D_FULLDEBUG, "Cleaning up now.\n");
				Cleanup( __LINE__, __FILE__ );
				dprintf( D_FULLDEBUG, "Accumulated Rusage After Cleanup\n" );
				dprintf( D_FULLDEBUG, "ru_utime = %d.%06d\n",
					AccumRusage.ru_utime.tv_sec, AccumRusage.ru_utime.tv_usec );
				dprintf( D_FULLDEBUG, "ru_stime = %d.%06d\n",
					AccumRusage.ru_stime.tv_sec, AccumRusage.ru_stime.tv_usec );
				dprintf( D_ALWAYS, "********** STARTER exiting ***********\n" );
				/*
				** Make sure that the log socket is read before the
				** syscall socket.
				sleep( 1 );
				*/
				(void)REMOTE_syscall( PSEUDO_reallyexit, &status, &AccumRusage);
				exit( 0 );
			}
			CkptInterval = MIN( 2 * CkptInterval, MaxCkptInterval );

		} else {				/* The child */
#ifdef NFSFIX
			/* Need to be root to set uid and gid. */
			set_root_euid(__FILE__,__LINE__);
#endif NFSFIX
#ifdef AIX31
			if( setgidx(ID_REAL|ID_EFFECTIVE,UserGid) < 0 ) {
				EXCEPT( "setgidx(ID_REAL|ID_EFFECTIVE,%d)", UserGid );
			}
			if( setuidx(ID_REAL|ID_EFFECTIVE,UserUid) < 0 ) {
				EXCEPT( "setuidx(ID_REAL|ID_EFFECTIVE,%d)", UserUid );
			}
#else AIX31
			if( setregid(UserGid,UserGid) < 0 ) {
				EXCEPT( "setregid(%d, %d)", UserGid, UserGid );
			}

			if( setreuid(UserUid,UserUid) < 0 ) {
				EXCEPT( "sereuid(%d, %d)", UserUid, UserUid );
			}
#endif AIX31


			StartCoredump = 0;
			dprintf( D_ALWAYS, "Starter: Doing execv( %s, %s",
												Shortname, JobArgs[0] );
			for( argp = &JobArgs[1]; *argp; argp++ ) {
				dprintf( D_ALWAYS, ", %s", *argp );
			}
			dprintf( D_ALWAYS, " )\n" );
			errno = 0;

			(void)execv( Shortname, JobArgs );

			/*
			** Just exit and do not log failed exec.  We have set our real
			** uid and real gid to that of the user.  Supply a significant
			** exit status.
			*/
			exit(EXECFAILED);

		}
	}
}


WaitForChild(statp, rusagep)
union wait *statp;
struct rusage *rusagep;
{
	int pid;
	int	image_size;
	int	scm;

	dprintf( D_FULLDEBUG, "Starter: waiting for \"%s\" (pid %d) to die\n",
												Shortname, ChildPid );

	set_root_euid(__FILE__,__LINE__);

	errno = 0;
	while( (pid=wait3(statp, 0, rusagep)) < 0 ) {
		if( errno != EINTR ) {
			dprintf( D_ALWAYS, "wait3 returns %d, errno = %d\n", pid, errno );
			EXCEPT( "wait3" );
		}
		errno = 0;
	}
	/* application may have called ckpt() explicitly, cancel pending alarm */
	(void)alarm( 0 );
	EndCoredump = get_time();

	set_condor_euid(__FILE__,__LINE__);

	if( pid != ChildPid ) {
		EXCEPT("wait3 returned pid %d (not %d)\n", pid, ChildPid );
	}
	ChildPid = 0;

#ifdef AIX31
		/* AIX incorrectly reports the time in nanoseconds... */
	rusagep->ru_utime.tv_usec /= 1000;
	rusagep->ru_stime.tv_usec /= 1000;
#endif AIX31

	dprintf(D_ALWAYS,
		"Pid %d exited, termsig = %d, coredump = %d, retcode = %d\n",
			pid, statp->w_termsig,
			statp->w_coredump, statp->w_retcode
	);

	if( statp->w_coredump ) {
		image_size = calc_image_size( "core" );
		if( image_size > ImageSize ) {
			ImageSize = image_size;
			(void)REMOTE_syscall( PSEUDO_image_size, ImageSize );
		}
	}

	dprintf( D_FULLDEBUG, "ru_utime = %d.%06d\n",
			rusagep->ru_utime.tv_sec, rusagep->ru_utime.tv_usec );
	dprintf( D_FULLDEBUG, "ru_stime = %d.%06d\n",
			rusagep->ru_stime.tv_sec, rusagep->ru_stime.tv_usec );

	switch( statp->w_termsig ) {
		case 0:			/* Normal Exit */
			JobExited = TRUE;
			break;
		case SIGUSR1:
			statp->w_termsig = SIGKILL;
			/* fall through */
		case SIGKILL: 	/* Kicked Off */
			bzero( (char *)rusagep, sizeof(struct rusage) );
			JobExited = TRUE;
			Aborted = TRUE;
			break;
		case SIGQUIT:	/* Periodic checkpoint */
			if( statp->w_coredump ) {
				if( StartCoredump ) {
					dprintf(D_ALWAYS,
						"Coredump took %0.1f seconds, %0.0f bytes/second\n",
						EndCoredump - StartCoredump,
						image_size*1024/(EndCoredump-StartCoredump) );
				}
				if( ImageSize < calc_free_disk_blocks() ) {
					do_ckpt();
				} else { /* got core, but not enough room to do ckpt */
					dprintf( D_ALWAYS, "No room for ckpt\n" );
					statp->w_termsig = SIGKILL;
					bzero( (char *)rusagep, sizeof(struct rusage) );
					JobExited = TRUE;
					Aborted = TRUE;
				}
			} else {	/* no room for the core - kick it off */
				(void)unlink( CoreName );
				sync();
				scm = SetSyscalls( SYS_LOCAL | SYS_RECORDED, __LINE__, __FILE__ );
				ImageSize = calc_free_disk_blocks();
				(void) SetSyscalls( scm, __LINE__, __FILE__ );
				dprintf( D_ALWAYS, "No room for core, image >= %d kilobytes\n",
															ImageSize );
				(void)REMOTE_syscall( PSEUDO_image_size, ImageSize );

				statp->w_termsig = SIGKILL;
				bzero( (char *)rusagep, sizeof(struct rusage) );
				JobExited = TRUE;
				Aborted = TRUE;
			}
			break;
		default:		/* Abnormal Exit */
			if( statp->w_coredump ) {
				if (!ValidCoresize()) {
					dprintf( D_ALWAYS, "Core file too large(%s)\n", Env);
#ifdef IBM032
					/* This is VERY kludgy.  A simple assignment to */
					/* statp->w_coredump does not work.  We need to */
					/* mask out the appropriate bit.  The bit format */
					/* for the wait union is as follows: */
					/*   bits 0-7: w_retcode */
					/*   bits 8-14: w_termsig */
					/*   bit 15: w_coredump */
					/*   bits 16-31: padding */
					/*dprintf(D_ALWAYS, "statp->w_status = %x\n", statp->w_status);*/
					statp->w_status = statp->w_status & 0x7f;
					/*dprintf(D_ALWAYS, "statp->w_status = %x\n", statp->w_status);*/
#else IBM032
					statp->w_coredump = 0;
#endif IBM032
					dprintf( D_ALWAYS, "Not sending back core file\n");
				}
				else {
					/* Check to see if enough space for core. */
					if( FileWillFit( "core", Condor_CWD ) ) {
						dprintf( D_ALWAYS, "Sending back core file\n" );
						dprintf(D_ALWAYS, "Sending back core file to \"%s\".\n",
																ReturnedCore);
						send_file( "core", ReturnedCore, 0664 );
					} else {
						dprintf( D_ALWAYS, "NOT sending back core file\n" );
						dprintf( D_ALWAYS, "Not enough space on submitting\n" );
#ifdef IBM032
						statp->w_status = statp->w_status & 0x7f;
#else IBM032
						statp->w_coredump = 0;
#endif IBM032
					}
				}
			}
			JobExited = TRUE;
			break;
	}
}


/*
** Look at the core file and try to guess how big a checkpoint file would
** be created from it.  Return the answer in 1024 byte blocks.
*/
calc_image_size( name )
char	*name;
{
	struct stat	stat_buf;
	int		scm;
	int		fd;
	int		text_blocks;
	int		ckpt_blocks;
	int		hdr_blocks;
	int		core_blocks;

	scm = SetSyscalls( SYS_LOCAL | SYS_RECORDED, __LINE__, __FILE__ );

	if( (fd = open(name,O_RDONLY,0)) < 0 ) {
		EXCEPT( "open(%s,O_RDONLY,0)", name );
	}

#ifdef IBM032
    if( lseek(fd, UPAGES*CLSIZE*NBPG - sizeof(struct user), L_SET) < 0 ) {
        EXCEPT("lseek(fd,  UPAGES*CLSIZE*NBPG - sizeof(struct user), L_SET)");
    }
#endif IBMO32
	text_blocks = calc_text_blocks( fd );

	if( fstat(fd,&stat_buf) < 0 ) {
		EXCEPT( "Can't fstat \"%s\"", name );
	}

	(void)close( fd );
	(void) SetSyscalls( scm, __LINE__, __FILE__ );


		/* What size are these blocks?  Probably 512, but is that documented? */
#if defined(IRIX331)
	core_blocks = stat_buf.st_size / 1024;
#else
	core_blocks = stat_buf.st_blocks / 2;
#endif

#if defined(AIX31) /* punt */
	hdr_blocks = 1024;
#endif

#if defined(HPUX8) /* punt */
	hdr_blocks = 1024;
#endif

#if defined(IRIX331)
	hdr_blocks = (sizeof(struct aouthdr) + 1023) / 1024;
#endif

#if !defined(AIX31) && !defined(IRIX331) && !defined(HPUX8)
	hdr_blocks = (sizeof(struct exec) + 1023) / 1024;
#endif

	ckpt_blocks = hdr_blocks + text_blocks + core_blocks + SLOP;

	dprintf( D_FULLDEBUG, "text: %d blocks\n", text_blocks );
	dprintf( D_FULLDEBUG, "core: %d blocks\n", core_blocks );
	dprintf( D_FULLDEBUG, "a.out hdr: %d blocks\n", hdr_blocks );
	dprintf( D_FULLDEBUG, "SLOP: %d blocks\n", SLOP );
	dprintf( D_FULLDEBUG, "Calculated image size: %dK\n", ckpt_blocks );

	return ckpt_blocks;
}

calc_text_blocks( fd )
int		fd;
{

#if defined(HPUX8)
	return 0;	/* figure out how to do this later... */
#endif

#if defined(IRIX331)
	return 0; /* really have to look in a.out file for this one */
#endif

#if defined (SUNOS32) || defined(SUNOS40) || defined(SUNOS41) || defined(CM5)
	struct core c;

    if( read(fd, (char *) &c, sizeof(c)) != sizeof(c) ) {
        EXCEPT("read struct core");
    }
    return c.c_tsize / 1024;
#endif

#if !defined(SUNOS32) && !defined(SUNOS40) && !defined(SUNOS41) && !defined(IRIX331) && !defined(CM5) && !defined(HPUX8)
	struct user	u;

    if( read(fd, (char *) &u, sizeof(u)) != sizeof(u) ) {
        EXCEPT("read");
    }

    return ctob(u.u_tsize) / 1024;
#endif
}

#if defined(NO_CKPT)
do_ckpt(){}
#else
do_ckpt()
{
	struct stat	buf;
	int rval;
	double	start, finish;
	int	scm;

	(void)sprintf(CkptTmpName, "%s.tmp", Shortname);
	dprintf(D_ALWAYS, "Starter: About to _updateckpt(%s, %s, %s)\n",
			CkptTmpName, Shortname, CoreName );

	scm = SetSyscalls( SYS_LOCAL | SYS_RECORDED, __LINE__, __FILE__ );
	start = get_time();
	_updateckpt( CkptTmpName, Shortname, CoreName );
	finish = get_time();

	(void)unlink( Shortname );
	(void)unlink( CoreName );
	(void)rename( CkptTmpName, Shortname );

	if( stat(Shortname,&buf) < 0 ) {
		EXCEPT( "Cannot stat \"%s\"", Shortname );
	}

	dprintf( D_ALWAYS, "Actual checkpoint size: %d blocks\n",
		(buf.st_size + 1024 - 1) / 1024 );

	dprintf( D_ALWAYS, "Ckpt took %0.1f seconds, %0.0f bytes/second\n",
		finish - start, buf.st_size/(finish-start) );

	CkptCreated = 1;
}
#endif



Cleanup( line, file )
int		line;
char	*file;
{
	dprintf( D_ALWAYS, "Cleanup called from %d in %s\n", line, file );
	(void) SetSyscalls( SYS_LOCAL | SYS_RECORDED, __LINE__, __FILE__ );

	(void)sigsetmask( ~0 );

#ifdef NFSFIX
	set_root_euid(__FILE__,__LINE__);
#endif NFSFIX

	if( ChildPid ) {
		(void)kill(ChildPid, SIGKILL);
	}

#ifdef NFSFIX
	set_condor_euid(__FILE__,__LINE__);
#endif NFSFIX

#if !defined( CM5 )
	(void)unlink( CoreName );
#endif

	if( Shortname[0] != 0 ) {
		(void)unlink( Shortname );
	}

	if( CkptTmpName && CkptTmpName[0] ) {
		(void)unlink( CkptTmpName );
	}
}

usage()
{
	dprintf( D_ALWAYS, "Usage: starter [-d] init_machine\n" );
	exit( 1 );
}

/*
** Get the Condor job off the machine asap.
*/
vacate_order(sig)
int	sig;
{
	union wait	fake_status;

	dprintf(D_ALWAYS, "vacate_order(): got %s\n", SigNames[sig]);
	Aborted = TRUE;

	if( ChildPid ) {
#ifdef NFSFIX
		set_root_euid(__FILE__,__LINE__);
#endif NFSFIX
		(void)kill(ChildPid, SIGCONT);
		(void)kill(ChildPid, SIGUSR1);
#ifdef NFSFIX
		set_condor_euid(__FILE__,__LINE__);
#endif NFSFIX
		dprintf( D_ALWAYS, "Sent SIGCONT & SIGUSR1 to user job\n" );
	} else {

		if( CkptCreated ) {
            if( FileWillFit(Shortname, CkptDir) ) {
                SendCkptFile( Shortname, CkptName );
                fake_status.w_termsig = SIGQUIT;    /* Kicked off with ckpt */
            } else {
                dprintf(D_ALWAYS, "ckpt not sent back -- not enough space.\n");
                fake_status.w_termsig = SIGKILL; /* Kicked off without ckpt */
            }
		} else {
			fake_status.w_termsig = SIGKILL;	/* Kicked off without ckpt */
		}
		(void)REMOTE_syscall( PSEUDO_reallyexit, &fake_status, &AccumRusage);
		dprintf( D_ALWAYS, "No user job running\n" );
		dprintf(D_FULLDEBUG, "Cleaning up now.\n");
		Cleanup( __LINE__, __FILE__ );
		exit( 1 );
	}
}

/*
** Get the Condor job off the machine judiciously.
*/
ckpt_and_vacate_order(sig)
int	sig;
{
	union wait	fake_status;

	dprintf(D_ALWAYS, "ckpt_and_vacate_order(): got %s\n", SigNames[sig]);
	Aborted = TRUE;

#ifdef NFSFIX
	set_root_euid(__FILE__,__LINE__);
#endif NFSFIX
	if( ChildPid ) {
		if( Suspended ) {	/* already suspended, just get rid of it */
			(void)kill(ChildPid, SIGCONT);
			(void)kill(ChildPid, SIGUSR1);
			dprintf( D_ALWAYS, "Sent SIGCONT & SIGUSR1 to user job\n" );
		} else {			/* force a checkpoint */
			(void)kill(ChildPid, SIGTSTP);
			StartCoredump = get_time();
			dprintf( D_ALWAYS, "Sent SIGTSTP to user job\n" );
				/* plan to evict if he doesn't checkpoint in reasonable time */
			(void)alarm( 5 * MINUTE );
			set_sig( SIGALRM, vacate_order );
		}
	} else {
		if( CkptCreated ) {
            if( FileWillFit(Shortname, CkptDir) ) {
                SendCkptFile( Shortname, CkptName );
                fake_status.w_termsig = SIGQUIT;    /* Kicked off with ckpt */
            } else {
                dprintf(D_ALWAYS, "ckpt not sent back -- not enough space.\n");
                fake_status.w_termsig = SIGKILL; /* Kicked off without ckpt */
            }
		} else {
			fake_status.w_termsig = SIGKILL;	/* Kicked off without ckpt */
		}
		(void)REMOTE_syscall( PSEUDO_reallyexit, &fake_status, &AccumRusage);
		dprintf( D_ALWAYS, "No user job running\n" );
		dprintf(D_FULLDEBUG, "Cleaning up now.\n");
		Cleanup( __LINE__, __FILE__ );
		exit( 1 );
	}
}

die(sig)
int	sig;
{
	dprintf( D_ALWAYS, "die(): line %d of %s: got signal %d\n",
		__LINE__, __FILE__, sig );
	Cleanup( __LINE__, __FILE__ );
	exit( 1 );
}

/*
** Send the Condor job an order to do a checkpoint.
*/
checkpoint_order(sig)
int	sig;
{
	if( ChildPid == 0 ) {
		dprintf(D_FULLDEBUG, "Cleaning up now.\n");
		Cleanup( __LINE__, __FILE__ );
	} else {
#ifdef NFSFIX
		set_root_euid(__FILE__,__LINE__);
#endif NFSFIX
		(void)kill(ChildPid, SIGCONT);
		(void)kill(ChildPid, SIGTSTP);
#ifdef NFSFIX
		set_condor_euid(__FILE__,__LINE__);
#endif NFSFIX
		dprintf(D_ALWAYS, "checkpoint_order(): got %s, %s\n",
			SigNames[sig], "sent SIGCONT & SIGTSTP to user job\n");
	}
}

int		SavedAlarm;

/*
** Suspend the Condor job.
*/
suspend_order(sig)
int	sig;
{
	if( ChildPid != 0 ) {
#ifdef NFSFIX
		set_root_euid(__FILE__,__LINE__);
#endif NFSFIX
		(void)kill(ChildPid, SIGSTOP);
#ifdef NFSFIX
		set_condor_euid(__FILE__,__LINE__);
#endif NFSFIX
		dprintf(D_ALWAYS, "suspend_order(): got %s, %s\n",
			SigNames[sig], "sent SIGSTOP to user job\n");
	}
	SavedAlarm = alarm( 0 );
	dprintf( D_ALWAYS, "Stopped alarm clock, %d seconds remaining\n",
														SavedAlarm );
	Suspended = TRUE;

		/* Try this for now, see how it works out */
	sigpause( sigmask(SIGUSR1) );
}

/*
** Resume running the Condor job.
*/
resume_order(sig)
int	sig;
{
	if( ChildPid != 0 ) {
#ifdef NFSFIX
		set_root_euid(__FILE__,__LINE__);
#endif NFSFIX
		(void)kill(ChildPid, SIGCONT);
#ifdef NFSFIX
		set_condor_euid(__FILE__,__LINE__);
#endif NFSFIX
		dprintf(D_ALWAYS, "resume_order(): got %s, %s\n",
			SigNames[sig], "sent SIGCONT to user job\n");
	}
	(void)alarm( (unsigned)SavedAlarm );
	dprintf( D_ALWAYS, "Reset alarm clock, %d seconds remaining\n",
														SavedAlarm );
	Suspended = FALSE;
}

#undef ASSERT
#define ASSERT(cond) \
	if( !(cond) ) { \
		dprintf( D_ALWAYS, "Assertion ERROR on (cond)\n"); \
		exit( 1 ); \
	} else {\
		dprintf( D_FULLDEBUG, "Assertion Ok on (cond)\n"); \
	}

char	ArgBuf[2048];
GetFileNames( xdrs )
XDR *xdrs;
{
	PROC	proc;
	int		argc;
	char *strip_pathname();

	bzero( (char *)&proc, sizeof(PROC) );

	xdrs->x_op = XDR_DECODE;
	ASSERT(xdr_proc(xdrs, &proc));
	ASSERT( xdr_string(xdrs,&CkptName,MAXPATHLEN) );
	ASSERT( xdr_string(xdrs,&OrigName,MAXPATHLEN) );


	Cluster = proc.id.cluster;
	Proc    = proc.id.proc;
	CmdName = proc.cmd;
	Stdin   = proc.in;
	Stdout  = proc.out;
	Stderr  = proc.err;
	Args    = proc.args;
	RootDir = proc.rootdir;
	Env		= proc.env;

	dprintf( D_FULLDEBUG, "Cluster = %d\n", Cluster );
	dprintf( D_FULLDEBUG, "Proc = %d\n", Proc );
	dprintf( D_FULLDEBUG, "CkptName = \"%s\"\n", CkptName );
	dprintf( D_FULLDEBUG, "OrigName = \"%s\"\n", OrigName );
	dprintf( D_FULLDEBUG, "Stdin = \"%s\"\n", Stdin );
	dprintf( D_FULLDEBUG, "Stdout = \"%s\"\n", Stdout );
	dprintf( D_FULLDEBUG, "Stderr = \"%s\"\n", Stderr );
	dprintf( D_FULLDEBUG, "RootDir = \"%s\"\n", RootDir );
	dprintf( D_FULLDEBUG, "CmdName = \"%s\"\n", CmdName );
	dprintf( D_FULLDEBUG, "Args = \"%s\"\n", Args );
	dprintf( D_FULLDEBUG, "Env = \"%s\"\n", Env );

	(void)sprintf( ReturnedCore, "core.%d.%d", Cluster, Proc );
	(void)sprintf( Shortname, "condor_exec%06d.%d", Cluster, Proc );
	sprintf( ArgBuf, "%s %s %s %s", Stdin, Stdout, Stderr, Args );
	mkargv( &argc, &JobArgs[1], ArgBuf );
	JobArgs[0] = Shortname;

	CkptDir = strip_pathname(CkptName);
	dprintf(D_FULLDEBUG, "CkptDir = <%s>\n", CkptDir);
}

GetJobFile( xdrs )
XDR		*xdrs;
{
	int		len;
	int		got, written;
	int		fd;
	char    buf[XDR_BLOCKSIZ];
	int		argc;

		/* Copy in the file */
	if( (fd=open(Shortname,O_WRONLY | O_CREAT,0775)) < 0 ) {
		EXCEPT( "open of \"%s\"", Shortname );
	}
	dprintf( D_FULLDEBUG, "Opened file \"%s\"\n", Shortname );
		
	ASSERT( xdr_int(xdrs,&len) );

	while( len ) {
		got = len < XDR_BLOCKSIZ ? len : XDR_BLOCKSIZ;

		errno = 0;
		ASSERT( xdr_opaque(xdrs, buf, got) );

		if( (written=write(fd,buf,got)) != got ) {
			EXCEPT( "write" );
		}

		len -= written;
	}

	(void)close( fd );
	dprintf( D_FULLDEBUG, "Closed \"%s\"\n", Shortname );

	sprintf( ArgBuf, "%s %s %s %s", Stdin, Stdout, Stderr, Args );
	mkargv( &argc, &JobArgs[1], ArgBuf );
	JobArgs[0] = Shortname;
}



SendCkptFile( local_name, remote_name )
char	*local_name;
char	*remote_name;
{
	char	tmp_name[ MAXPATHLEN ];
	int rval;

	(void)sprintf( tmp_name, "%s.tmp", remote_name );
	dprintf( D_FULLDEBUG, "SendCkptFile() called\n" );
	dprintf( D_FULLDEBUG, "local_name = \"%s\"\n", local_name );
	dprintf( D_FULLDEBUG, "remote_name = \"%s\"\n", remote_name );
	dprintf( D_FULLDEBUG, "tmp_name = \"%s\"\n", tmp_name );

	rval = send_file( local_name, tmp_name, 0664 );
	if( rval < 0 ) {
		dprintf( D_FULLDEBUG, "File NOT xferred.  Do NOT update rusage.\n" );
	} else {
		/* Update shadow with accumulated usage for the job. */
		dprintf( D_FULLDEBUG, "File xferred.  Update rusage.\n" );
		rval = REMOTE_syscall( PSEUDO_send_rusage, &AccumRusage );
		dprintf( D_FULLDEBUG, "PSEUDO_send_rusage returned %d\n", rval);
	}

	(void) SetSyscalls( SYS_REMOTE | SYS_RECORDED, __LINE__, __FILE__ );
	if( rename(tmp_name,remote_name) < 0 ) {
		EXCEPT( "remote_rename(%s,%s)", tmp_name, remote_name );
	}
	(void) SetSyscalls( SYS_LOCAL | SYS_RECORDED, __LINE__, __FILE__ );
	dprintf( D_ALWAYS, "Returned latest checkpoint\n" );
}


init_params()
{
	char	*tmp;

	if( (Execute=param("EXECUTE")) == NULL ) {
		EXCEPT( "Execute directory not specified in config file" );
	}

	if( (tmp=param("MIN_CKPT_INTERVAL")) == NULL ) {
		MinCkptInterval = 15 * MINUTE;
	} else {
		MinCkptInterval = atoi( tmp );
	}

	if( (tmp=param("MAX_CKPT_INTERVAL")) == NULL ) {
		MaxCkptInterval = 2 * HOUR;
	} else {
		MaxCkptInterval = atoi( tmp );
	}

	if( MaxCkptInterval > 2 * HOUR ) {
		MaxCkptInterval = 2 * HOUR;
	}

	if( MaxCkptInterval < 15 * MINUTE ) {
		MaxCkptInterval = 15 * MINUTE;
	}

	if( MinCkptInterval < 15 * MINUTE ) {
		MinCkptInterval = 15 * MINUTE;
	}

	if( MinCkptInterval > MaxCkptInterval ) {
		MinCkptInterval = MaxCkptInterval;
	}

}


static	int	catch_segv( sig )
int	sig;
{
	int	scm;
	int	mysig = SIGFPE;
	sighandler_t	(*myfunc)() = SIG_DFL;
	struct	rlimit	lim_core, lim_fsize;
	static	char	core_dir[] = "/local/condor/`hostname`";

	dprintf( D_ALWAYS, "catch_segv(): line %d of %s: got signal %d!\n",
		__LINE__, __FILE__, sig );
	dprintf( D_ALWAYS, "trying to dump core in \"%s\"\n", core_dir );
	scm = SetSyscalls( SYS_LOCAL | SYS_UNRECORDED, __LINE__, __FILE__ );
	chdir( ".." );
	signal( mysig, myfunc );

#if !defined(HPUX8)
	signal( SIGXFSZ, SIG_IGN );	/* file size limit exceeded signal */
#endif

	setgid( getgid() );
	setuid( getuid() );

#if !defined(HPUX8)
	getrlimit( RLIMIT_CORE, &lim_core );
	getrlimit( RLIMIT_FSIZE, &lim_fsize );
	lim_core.rlim_cur = lim_core.rlim_max;
	lim_fsize.rlim_cur = lim_fsize.rlim_max;
	setrlimit( RLIMIT_CORE, &lim_core );
	setrlimit( RLIMIT_FSIZE, &lim_fsize );
#endif

	kill( getpid(), mysig );
	scm = SetSyscalls( scm, __LINE__, __FILE__ );
	dprintf( D_ALWAYS, "Line %d of %s: WHAT THE HELL IS GOING ON?\n",
		__LINE__, __FILE__ );
	scm = SetSyscalls( scm, __LINE__, __FILE__ );
	exit( 99 );
}

set_sigs()
{
	int		i;

	for( i=1; i<NSIG; i++ ) {
		switch( i ) {
			case SIGILL:
			case SIGIOT:
				set_sig( i, abort );
				break;
			case SIGKILL:
			case SIGSTOP:
			case SIGQUIT:
				break;
			case SIGBUS:
			case SIGSEGV:
				set_sig( i, catch_segv );
				break;
			case SIGINT:
				set_sig( i, die );
				break;
			case SIGTSTP:
				set_sig( i, vacate_order );
				break;
			case SIGUSR1:
				set_sig( i, suspend_order );
				break;
			case SIGUSR2:
				set_sig( i, ckpt_and_vacate_order );
				break;
			case SIGCONT:
#if !defined(IRIX331)
				set_sig( i, resume_order );
#endif
				break;
			case SIGALRM:
				set_sig( i, checkpoint_order );
				break;
			case SIGCHLD:
#ifndef AIX31
				set_sig( i, SIG_IGN );
#endif AIX31
				break;
			default:
				set_sig( i, die );
		}
	}
}


/*
** Check the header of the file we are about to exec.  Make sure everything
** is reasonable.
*/
#ifdef COFF

typedef struct filehdr  FILE_HDR;
typedef struct aouthdr  AOUT_HDR;

#define FILE_HDR_SIZ    sizeof(FILE_HDR)
#define AOUT_HDR_SIZ    sizeof(AOUT_HDR)

#if defined(HPUX8)	/* figure this out later */
magic_check( a_out )
char	*a_out;
{
	return 0;
}
#else
magic_check( a_out )
char	*a_out;
{
	int		exec_fd;
	FILE_HDR	file_hdr;
	AOUT_HDR	aout_hdr;
#if !defined(AIX31) && !defined(HPUX8)
	HDRR		symbolic_hdr;
#endif


	if( (exec_fd=open(a_out,O_RDONLY,0)) < 0 ) {
		dprintf( D_ALWAYS, "open(%s,O_RDONLY,0)", a_out );
		return -1;
	}

		/* Deal with coff file header */
	if( read(exec_fd,&file_hdr,FILE_HDR_SIZ) != FILE_HDR_SIZ ) {
		dprintf(D_ALWAYS, "read(%d,0x%x,%d)", exec_fd, &file_hdr, FILE_HDR_SIZ);
		return -1;
	}

	if( file_hdr.f_magic != COFF_MAGIC ) {
		dprintf( D_ALWAYS, "BAD MAGIC NUMBER IN COFF HEADER\n" );
		return -1;
	}
	dprintf( D_ALWAYS, "COFF header - magic number OK\n" );


		/* Deal with optional header (a.out header) */
    if( file_hdr.f_opthdr != AOUT_HDR_SIZ ) {
		dprintf( D_ALWAYS, "BAD A.OUT HEADER SIZE IN COFF HEADER\n" );
		return -1;
    }


    if( read(exec_fd,&aout_hdr,AOUT_HDR_SIZ) != AOUT_HDR_SIZ ) {
    	dprintf( D_ALWAYS,"read(%d,0x%x,%d)", exec_fd, &aout_hdr, AOUT_HDR_SIZ);
		return -1;
	}

	if( aout_hdr.magic != AOUT_MAGIC ) {
		dprintf( D_ALWAYS, "BAD MAGIC NUMBER IN A.OUT HEADER\n" );
		return -1;
	}
	dprintf( D_ALWAYS, "AOUT header - magic number OK\n" );


#ifndef AIX31
		/* Bring symbol table header into core */
	if( lseek(exec_fd,file_hdr.f_symptr,0) != file_hdr.f_symptr ) {
		dprintf( D_ALWAYS, "lseek(%d,%d,0)", exec_fd, file_hdr.f_symptr );
		return -1;
	}
	if( read(exec_fd,&symbolic_hdr,cbHDRR) != cbHDRR ) {
		dprintf( D_ALWAYS,
			"read(%d,0x%x,%d)", exec_fd, &symbolic_hdr, cbHDRR );
		return -1;
	}

	if( symbolic_hdr.magic != magicSym ) {
		dprintf( D_ALWAYS, "Bad magic number in symbol table header\n" );
		dprintf( D_ALWAYS, " -- expected 0x%x, found 0x%x\n",
			magicSym, symbolic_hdr.magic );
		return -1;
	}
#endif

	return 0;
}
#endif

#else COFF

magic_check( name )
char	*name;
{
	int		fd;
	struct exec	exec;
	int		scm;

	scm = SetSyscalls( SYS_LOCAL | SYS_RECORDED, __LINE__, __FILE__ );

	if( (fd=open(name,O_RDONLY,0)) < 0 ) {
		EXCEPT( "open(%s,O_RDONLY,0)", name );
	}

	if( read(fd,(char *)&exec,sizeof(struct exec)) != sizeof(struct exec) ) {
		EXCEPT( "read(%d,0x%x,%d)", fd, &exec, sizeof(struct exec) );
	}
	(void)close( fd );

	(void)SetSyscalls( scm, __LINE__, __FILE__ );

#ifdef VAX
	if( exec.a_magic != ZMAGIC ) {
		dprintf( D_ALWAYS, "\"%s\": BAD MAGIC NUMBER\n", name );
		return -1;
	}
#endif

#ifdef DYNIX
	if( exec.a_magic != ZMAGIC ) {
		dprintf( D_ALWAYS, "\"%s\": BAD MAGIC NUMBER\n", name );
		return -1;
	}
#endif DYNIX

#ifdef SPARC
	if( exec.a_magic != ZMAGIC ) {
		dprintf( D_ALWAYS, "\"%s\": BAD MAGIC NUMBER\n", name );
		return -1;
	}
	if( exec.a_machtype != M_SPARC ) {
		dprintf( D_ALWAYS, "\"%s\": NOT COMPILED FOR SPARC ARCHITECTURE\n",
															name );
		return -1;
	}
	if( exec.a_dynamic ) {
		dprintf( D_ALWAYS, "\"%s\": LINKED FOR DYNAMIC LOADING\n", name );
		return -1;
	}
#endif SPARC
		
#if defined(MC68020) && (defined(SUNOS32) || defined(SUNOS41)) /* Sun 3 */
	if( exec.a_magic != ZMAGIC ) {
		dprintf( D_ALWAYS, "\"%s\": BAD MAGIC NUMBER\n", name );
		return -1;
	}
	if( exec.a_machtype != M_68020 ) {
		dprintf( D_ALWAYS, "\"%s\": NOT COMPILED FOR Sun 3s\n",
																name );
		return -1;
	}
#endif 

#if defined(MC68020) && defined(BSD43)	/* HP Bobcat */
	if( exec.a_magic != ZMAGIC ) {
		dprintf( D_ALWAYS, "\"%s\": BAD MAGIC NUMBER\n", name );
		return -1;
	}
	if( exec.a_machtype != MID_HP300 ) {
		dprintf( D_ALWAYS, "\"%s\": NOT COMPILED FOR Bobcats\n",
																name );
		return -1;
	}
#endif 

	return 0;
}
#endif COFF

limit( resource, new_limit )
int		resource;
int		new_limit;
{
	int		scm;
	struct	rlimit lim;

	scm = SetSyscalls( SYS_LOCAL | SYS_RECORDED, __LINE__, __FILE__ );


		/* Find out current limits */
	if( getrlimit(resource,&lim) < 0 ) {
		EXCEPT( "getrlimit(%d,0x%x)", resource, &lim );
	}

		/* Don't try to exceed the max */
	if( new_limit > lim.rlim_max ) {
		new_limit = lim.rlim_max;
	}

		/* Set the new limit */
	lim.rlim_cur = new_limit;
	if( setrlimit(resource,&lim) < 0 ) {
		EXCEPT( "setrlimit(%d,0x%x)", resource, &lim );
	}

	(void)SetSyscalls( scm, __LINE__, __FILE__ );
}

/*
** I think this is right, but haven't tested it out.
** -- mike
*/
calc_free_disk_blocks()
{
	return free_fs_blocks( "." );
}
#if 0
calc_free_disk_blocks()
{
#if defined(ULTRIX) || defined(ULTRIX42)
	struct fs_data buf;
#else defined(ULTRIX) || defined(ULTRIX42)
	struct statfs buf;
#endif defined(ULTRIX) || defined(ULTRIX42)
	int		answer;

	if( statfs(".",&buf) < 0 ) {
		EXCEPT( "Can't statfs \"%s\"", "." );
	}

#if defined(ULTRIX) || defined(ULTRIX42)
	answer = buf.fd_bfreen;
#else defined(ULTRIX) || defined(ULTRIX42)
	answer = (buf.f_bavail * buf.f_bsize) / 1024;
#endif defined(ULTRIX) || defined(ULTRIX42)

	dprintf( D_ALWAYS, "Found %d free disk blocks\n", answer );

	return answer;
}
#endif


ValidCoresize()
{
	char *value;
	int maxSize;
	char *GetEnvParam();
	struct stat statBuf;
	char *core = "core";
	int scm;

	scm = SetSyscalls( SYS_LOCAL | SYS_RECORDED, __LINE__, __FILE__ ); /* Use local syscalls */
	
	value = GetEnvParam( "CONDOR_CORESIZE" );	/* get parameter value */

	if( value == NULL ) {
		dprintf( D_ALWAYS, "Couldn't find environment variable CONDOR_CORESIZE\n");
		return( TRUE );
	}

	maxSize = atoi(value);
	dprintf( D_ALWAYS, "Maximum core file size is %dK bytes\n", maxSize);

	if( stat(core,&statBuf) == -1 ) {
		dprintf( D_ALWAYS, "ValidCoresize: stat returned error code %d",errno);

		return( TRUE );
	}
	dprintf( D_ALWAYS, "Core file size is %dK bytes\n", statBuf.st_size/1024);

	(void) SetSyscalls( scm, __LINE__, __FILE__ );

	/* stat returns size in bytes - compare in kilobytes */
	return( statBuf.st_size/1024 <= maxSize );
}

#define MATCH 0			/* result of strcmp */
#define EQUAL '='		/* chars looked for during parsing */
#define SPACE ' '
#define TAB '\t'
#define SEMI ';'

char *
GetEnvParam(param)
char *param;
{
	char *ptr = Env;
	char name[BUFSIZ];
	int i;
	char tmp[BUFSIZ];

	while( TRUE ) {
		/* skip until alphanumeric char */
		while( *ptr ) {
			if( isalnum( *ptr ) ) {
				break;
			}
			ptr++;
		}

		if( !*ptr ) {
			dprintf( D_ALWAYS, "No matching param found\n");
			return( NULL );
		}
		
		/* parse name */
		bzero( name, sizeof(name) );
		i = 0;
		while( *ptr ) {
			if( *ptr == SPACE || *ptr == TAB || *ptr == EQUAL 
							|| *ptr == SEMI ) {
				break;
			}
			name[i++] = *ptr;
			ptr++;
		}

		if( i > 0 ) {
			if( strcmp(name,param) != MATCH ) {
				/* No match - move to parameter delimiter ';' */
				while( *ptr ) {
					if( *ptr == ';' )
						break;
					ptr++;
				}
				if( !*ptr ) {
					dprintf( D_ALWAYS, "No matching param found\n");
					return( NULL );
				}
				continue;
			}
		} else {
			dprintf( D_ALWAYS, "No matching param found\n");
			return( NULL );
		}

		/* parse over white space and the equal character */
		while( *ptr ) {
			if( isalnum( *ptr ) ) {
				break;
			}
			ptr++;
		}

		/* if at end of env or at semi then return a value of "" */
		if( *ptr == '\0' || *ptr == SEMI ) {
			return( "" );
		}
	
		/* parse value */
		i = 0;
		while( *ptr ) {
			if( *ptr == SEMI ) {
				break;
			}
			tmp[i++] = *ptr;
			ptr++;
		}
		if( *tmp == '\0' )
			return( "" );	/* No value set, return a value of "" */
		else 
			return( tmp );
	}
}

symbol_main_check( name )
char *name;
{
	int scm;
	int	status;
	struct nlist nl[2];


#if defined(MIPS) || defined(IRIX331) || defined(HPUX8)
	nl[0].n_name = "MAIN";
	nl[1].n_name = "";
#endif

#ifdef AIX31
	nl[0]._n._n_name = ".MAIN";
	nl[1]._n._n_name = "";
#endif

#if !defined(MIPS) && !defined(AIX31) && !defined(IRIX331) && !defined(HPUX8)
/* everybody else */
	nl[0].n_un.n_name = "_MAIN";
	nl[1].n_un.n_name = "";
#endif not (mips or aix31)
	

	scm = SetSyscalls( SYS_LOCAL | SYS_RECORDED, __LINE__, __FILE__ );

	/* If the nlist call fails just return TRUE.  Executable may be stripped. */
	status = nlist( name, nl );
	if( status < 0 ) {
		(void)SetSyscalls( scm, __LINE__, __FILE__ );
		dprintf(D_ALWAYS, "Error: nlist(\"%s\",0x%x) returns %d, errno = %d\n",
													name, nl, status, errno);
		/*
		sleep( 5 );
		*/
		return(TRUE);
	}
	(void)SetSyscalls( scm, __LINE__, __FILE__ );

#ifdef AIX31
		/* N.B. must check n_value, not n_type for AIX machines.  If the
		   file was not compiled with "-g", n_value is always 0. */
	if( nl[0].n_value == 0 ) {
#else AIX31
	if( nl[0].n_type == 0 ) {
#endif AIX31
		dprintf( D_ALWAYS, "No symbol \"MAIN\" in executable(%s)\n", name );
		/*
		sleep( 5 );
		*/
		return(-1);
	}
	(void)SetSyscalls( scm, __LINE__, __FILE__ );
	return TRUE;
}

#if !defined(HPUX8)	/* HP-UX doesn't support any of these */
check_limits()
{
	dprintf( D_ALWAYS, "CPU_LIMIT is %d\n", check_limit(RLIMIT_CPU) );
	dprintf( D_ALWAYS, "DATA_LIMIT is %d\n", check_limit(RLIMIT_DATA) );
	dprintf( D_ALWAYS, "STACK_LIMIT is %d\n", check_limit(RLIMIT_STACK) );
	dprintf( D_ALWAYS, "RSS_LIMIT is %d\n", check_limit(RLIMIT_RSS) );
	dprintf( D_ALWAYS, "CORE_LIMIT is %d\n", check_limit(RLIMIT_CORE) );
}
#endif

check_limit( resource )
{
	int		scm;
	struct	rlimit lim;

	scm = SetSyscalls( SYS_LOCAL | SYS_RECORDED, __LINE__, __FILE__ );
	if( getrlimit(resource,&lim) < 0 ) {
		EXCEPT( "getrlimit(%d,0x%x)", resource, &lim );
	}
	(void)SetSyscalls( scm, __LINE__, __FILE__ );
	return lim.rlim_cur;
}

/*
** FileWillFit(file): check to see if the specified local file will fit onto
** the file system where the remote file resides.
*/
int
FileWillFit(local, remote)
char *local;
char *remote;
{
	int free_kbytes;
	int will_fit;
	int file_size;
	struct stat statbuf;
	int scm;

	dprintf(D_ALWAYS, "FileWillFit(%s, %s) called\n", local, remote);

	/* Find out how many kbytes are available on submitting machine. */
	free_kbytes = REMOTE_syscall( PSEUDO_free_fs_blocks, remote );

	/* Find file size. */
	scm = SetSyscalls( SYS_LOCAL | SYS_RECORDED, __LINE__, __FILE__ ); /* Use local syscalls */
	if( stat(local, &statbuf) == -1) {
		dprintf( D_ALWAYS, "FileWillFit: stat returned error code %d", errno);
		(void)SetSyscalls(scm, __LINE__, __FILE__);
		return( TRUE );
	}
	(void)SetSyscalls(scm, __LINE__, __FILE__);

	/* st_size is in bytes -- convert to kbytes. */
	file_size = statbuf.st_size / 1024;

	will_fit = (file_size < free_kbytes) ? TRUE : FALSE;

	return(will_fit);
}

/*
** Strip the last portion of a pathname.
*/
#define SLASH '/'
char *
strip_pathname(pathname)
char *pathname;
{
	char *last_slash;
	char newpath[MAXPATHLEN];

	strcpy(newpath,pathname);

	last_slash = rindex(newpath, SLASH);

	*last_slash = '\0';

	return(strdup(newpath));
}


#if 0
loop()
{
	for(;;);
}
#endif

set_sig( sig, func )
int		sig;
int		(*func)();
{
#ifdef AIX31
	struct sigaction	action;

	action.sa_handler = func;
	action.sa_flags = SA_RESTART | SA_FULLDUMP;
	SIGINITSET(action.sa_mask);
	if( sigaction(sig, &action, (struct sigaction *)0 ) == BADSIG ) {
		EXCEPT( "Can't set signal %d to 0x%x", sig, func );
	}
	dprintf( D_FULLDEBUG, "Set signal %d to 0x%x, flags = 0%o, mask = 0%o\n",
								sig, func, action.sa_flags, action.sa_mask );
#else
	if( signal(sig, func) == BADSIG ) {
		EXCEPT( "Can't set signal %d to 0x%x", sig, func );
	}
#endif
}


/*
** Arrange to die with a full core dump.
*/
abort()
{
#ifdef AIX31
	struct sigaction    action;
#endif
	dprintf( D_ALWAYS, "abort() Called\n" );

	/*
	Cleanup( __LINE__, __FILE__ );
	*/

	/* Set up SIGQUIT to produce a full core dump. */
#ifdef AIX31
	if( sigaction(SIGQUIT,NULL,&action) < 0 ) {
		EXCEPT( "can't examine sigaction" );
	}
	action.sa_flags |= SA_FULLDUMP;
	action.sa_handler = SIG_DFL;
	if( sigaction(SIGQUIT,&action,NULL) < 0 ) {
		EXCEPT( "can't set sigaction" );
	}
#else AIX31
	signal( SIGQUIT, SIG_DFL );
#endif AIX31

	display_syscall_mode( __LINE__, __FILE__ );
	dprintf( D_ALWAYS, "uid = %d, euid = %d\n", getuid(), geteuid() );
	dprintf( D_ALWAYS, "gid = %d, egid = %d\n", getgid(), getegid() );

	dprintf( D_ALWAYS, "Setting syscall mode to local/unrecorded\n" );
	SetSyscalls( SYS_LOCAL | SYS_UNRECORDED, __LINE__, __FILE__ );

#if !defined(HPUX8)
	limit( RLIMIT_CORE, RLIM_INFINITY );
#endif

	sigsetmask( 0 );
	dprintf( D_ALWAYS, "About to send SIGQUIT\n" );
	chdir( "/var/home/condor" );
	kill( getpid(), SIGQUIT );
	sigpause( 0 );
}
