/* 
** Copyright 1986, 1987, 1988, 1989 University of Wisconsin
** 
** Permission to use, copy, modify, and distribute this software and its
** documentation for any purpose and without fee is hereby granted,
** provided that the above copyright notice appear in all copies and that
** both that copyright notice and this permission notice appear in
** supporting documentation, and that the name of the University of
** Wisconsin not be used in advertising or publicity pertaining to
** distribution of the software without specific, written prior
** permission.  The University of Wisconsin makes no representations about
** the suitability of this software for any purpose.  It is provided "as
** is" without express or implied warranty.
** 
** THE UNIVERSITY OF WISCONSIN DISCLAIMS ALL WARRANTIES WITH REGARD TO
** THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND
** FITNESS. IN NO EVENT SHALL THE UNIVERSITY OF WISCONSIN  BE LIABLE FOR
** ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
** WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
** ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
** OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
** 
** Authors:  Allan Bricker and Michael J. Litzkow,
** 	         University of Wisconsin, Computer Sciences Dept.
** 
*/ 


#include <stdio.h>
#include <rpc/types.h>
#include <rpc/xdr.h>
#include <netinet/in.h>
#include <netdb.h>
#include <ctype.h>
#include "debug.h"
#include "except.h"
#include "trace.h"
#include "expr.h"
#include "sched.h"
#include "manager.h"
#include "clib.h"

static char *_FileName_ = __FILE__;		/* Used by EXCEPT (see except.h)     */

char	*param(), *strdup();
XDR		*xdr_Init();
CONTEXT	*create_context();

extern int	Terse;
extern int	Silent;

char	*CollectorHost;
char	*MyName;
int		Verbose;
int		SortByPrio;
int		RollCall;
char	*RosterFile;

STATUS_LINE		*StatusLines[1024];
int				N_StatusLines;

typedef struct {
	char	*name;
	char	*comment;
	int		present;
} ROSTER;

ROSTER	Roster[1024];
int		RosterLen = 0;
int		Absent;

char	*Unknowns[1024];
int		N_Unknowns = 0;

usage()
{
	fprintf( stderr, "Usage: %s [-l] [-p] name ...\n", MyName );
	exit( 1 );
}

main( argc, argv )
int		argc;
char	*argv[];
{
	int			sock = -1;
	XDR			xdr, *xdrs = NULL;

	MyName = argv[0];

	argv++;
	argc--;
	if( argc && argv[0][0] == '-' ) {
		if( argv[0][1] == 'l' ) {
			Verbose++;
		} else if( argv[0][1] == 'p' ) {
			SortByPrio = 1;
		} else {
			usage();
		}
		argv++;
		argc--;
	}

	config( MyName, (CONTEXT *)0 );
	init_params();
	Terse = TRUE;
	Silent = TRUE;

	if( RollCall ) {
		read_roster_file();
	}

		/* Connect to the collector */
	if( (sock = do_connect(CollectorHost, "condor_collector",
												COLLECTOR_PORT)) < 0 ) {
		dprintf( D_ALWAYS, "Can't connect to Condor Collector\n" );
		exit( 1 );
	}
	xdrs = xdr_Init( &sock, &xdr );
	xdrs->x_op = XDR_ENCODE;

	if( Verbose ) {
		get_status( argc, argv, xdrs );
	} else {
		get_status_lines( argc, argv, xdrs );
		if( RollCall ) {
			display_unknowns();
			display_absent();
		}
	}
}

get_status_lines( argc, argv, xdrs )
int		argc;
char	*argv[];
XDR		*xdrs;
{
	int			i;
	int			cmd;
	STATUS_LINE	*line;
	int			name_comp(), prio_comp();

	cmd = GIVE_STATUS_LINES;
	ASSERT( xdr_int(xdrs, &cmd) );
	ASSERT( xdrrec_endofrecord(xdrs,TRUE) );

	xdrs->x_op = XDR_DECODE;

	for(;;) {
		line = (STATUS_LINE *)calloc( 1, sizeof(STATUS_LINE) );
		if( !xdr_status_line( xdrs, line ) ) {
			EXCEPT( "Can't read status line from Condor Collector" );
		}
		if( line->name == NULL || line->name[0] == '\0' ) {
			break;
		}

		if( line->state == NULL ) {
			line->state = "";
		}

		if( strcmp(line->state,"(DOWN)") == MATCH ) {
			continue;
		}

		inc_summary( line );

		if( RollCall ) {
			mark_present( line->name );
		}

		if( !selected(strdup(line->name),argc,argv) ) {
			continue;
		}
		StatusLines[ N_StatusLines++ ] = line;
	}

	if( SortByPrio ) {
		qsort( (char *)StatusLines, N_StatusLines, sizeof(StatusLines[0]),
																prio_comp );
	} else {
		qsort( (char *)StatusLines, N_StatusLines, sizeof(StatusLines[0]),
																name_comp );
	}

	print_header();
	for( i=0; i<N_StatusLines; i++ ) {
		display_status_line( StatusLines[i] );
	}
	(void)putchar( '\n' );
	display_summaries();
}

get_status( argc, argv, xdrs )
int		argc;
char	*argv[];
XDR		*xdrs;
{
	int			cmd;
	MACH_REC	*rec;

	cmd = GIVE_STATUS;
	ASSERT( xdr_int(xdrs, &cmd) );
	ASSERT( xdrrec_endofrecord(xdrs,TRUE) );

	xdrs->x_op = XDR_DECODE;
	for(;;) {
		rec = (MACH_REC *) malloc( sizeof(MACH_REC) );
		if( rec == NULL ) {
			EXCEPT("Out of memory");
		}

		bzero( (char *)rec, sizeof(MACH_REC) );
		rec->machine_context = create_context();
		ASSERT( xdr_mach_rec(xdrs,rec) );
		if( !rec->name || !rec->name[0] ) {
			break;
		}
		if( !selected(strdup(rec->name),argc,argv) ) {
			continue;
		}
		display_verbose( rec );
	}
	
}


selected( name, argc, argv )
char	*name;
int		argc;
char	*argv[];
{
	int		i;
	char	*ptr, *index();

	if( argc == 0 ) {
		return 1;
	}

	for( i=0; i<argc; i++ ) {
		if( ptr=index(name,'.') ) {
			*ptr = '\0';
		}
		if( strcmp(argv[i],name) == MATCH ) {
			return 1;
		}
	}
	return 0;
}

display_verbose( ptr )
MACH_REC	*ptr;
{
	printf( "name: \"%s\"\n", ptr->name );
	printf( "machine_context:\n" );
	display_context( ptr->machine_context );
	printf( "time_stamp: %s", ctime( (time_t *)&ptr->time_stamp) );
	printf( "prio: %d\n", ptr->prio );
	printf( "\n" );
}


init_params()
{
	if( (CollectorHost = param("COLLECTOR_HOST")) == NULL ) {
		EXCEPT( "COLLECTOR_HOST not specified in config file\n" );
	}
	RosterFile = param( "ROSTER_FILE" );
	if( RosterFile ) {
		RollCall = TRUE;
	}
}


SetSyscalls(){}

name_comp( ptr1, ptr2 )
STATUS_LINE	**ptr1, **ptr2;
{
	int		status;

	if( status = strcmp( (*ptr1)->arch, (*ptr2)->arch ) ) {
		return status;
	}

	if( status = strcmp( (*ptr1)->op_sys, (*ptr2)->op_sys ) ) {
		return status;
	}

	return strcmp( (*ptr1)->name, (*ptr2)->name );
}

prio_comp( ptr1, ptr2 )
STATUS_LINE	**ptr1, **ptr2;
{
	int		status;

	if( status = strcmp( (*ptr1)->arch, (*ptr2)->arch ) ) {
		return status;
	}

	if( status = strcmp( (*ptr1)->op_sys, (*ptr2)->op_sys ) ) {
		return status;
	}

	if( status = (*ptr1)->prio - (*ptr2)->prio ) {
		return status;
	}

	return strcmp( (*ptr1)->name, (*ptr2)->name );
}

typedef struct {
	char	*arch;
	char	*op_sys;
	int		machines;
	int		jobs;
	int		running;
} SUMMARY;

SUMMARY		*Summary[50];
int			N_Summaries;
SUMMARY		Total;

inc_summary( line )
STATUS_LINE	*line;
{
	SUMMARY	*s, *get_summary();

	s = get_summary( line );
	s->machines += 1;
	s->jobs += line->tot;

	Total.machines += 1;
	Total.jobs += line->tot;

	if( strncmp(line->state, "Run", 3) == 0 ) {
		s->running += 1;
		Total.running += 1;
	}
}

SUMMARY *
get_summary( line )
STATUS_LINE	*line;
{
	int		i;
	SUMMARY	*new;

	for( i=0; i<N_Summaries; i++ ) {
		if( strcmp(Summary[i]->arch,line->arch) == MATCH &&
						strcmp(Summary[i]->op_sys,line->op_sys) == MATCH ) {
			return Summary[i];
		}
	}

	new = (SUMMARY *)calloc( 1, sizeof(SUMMARY) );
	new->arch = line->arch;
	new->op_sys = line->op_sys;
	Summary[ N_Summaries++ ] = new;
	return new;
}

display_summaries()
{
	int		i;

	for( i=0; i<N_Summaries; i++ ) {
		display_summary( Summary[i] );
	}
	display_summary( &Total );

}

display_summary( s )
SUMMARY		*s;
{
	char	tmp[256];

	if( s->arch ) {
		(void)sprintf( tmp, "%s/%s", s->arch, s->op_sys );
	} else {
		tmp[0] = '\0';
	}

	printf( "%-20s %3d machines %3d jobs %3d running\n",
					tmp, s->machines, s->jobs, s->running );
}


read_roster_file()
{
	FILE	*fp;
	char	*ltrunc();
	char	buf[1024];

	if( !RosterFile ) {
		fprintf( stderr, "\"ROSTER_FILE\" not specified in config file\n" );
		exit( 1 );
	}

	if( (fp=fopen(RosterFile,"r")) == NULL ) {
		RollCall = 0;
		return;
	}

	while( fgets(buf,sizeof buf,fp) ) {
		add_roster_elem( ltrunc(buf) );
	}
	Absent = RosterLen;
}

add_roster_elem( line )
char	*line;
{
	char	*ptr;
	char	*comment;

	for( ptr=line; *ptr && !isspace(*ptr); ptr++ )
		;

	if( *ptr ) {
		comment = ptr + 1;
		*ptr = '\0';
	} else {
		comment = ptr;
	}
	Roster[RosterLen].name = strdup( line );

	for( ptr=comment; *ptr && isspace(*ptr); ptr++ )
		;

	Roster[RosterLen].comment = strdup( ptr );
	Roster[RosterLen].present = FALSE;

	RosterLen += 1;
}

display_unknowns()
{
	int		i;

	if( N_Unknowns == 0 ) {
		return;
	}

	printf( "\n" );
	if( N_Unknowns == 1 ) {
		printf(
		"The following  machine is present, but not on the roster...\n");
	} else {
		printf(
		"The following %d machines are present, but not on the roster...\n",
		N_Unknowns );
	}

	for( i=0; i<N_Unknowns; i++ ) {
		printf( "%s\n", Unknowns[i] );
	}
}

display_absent()
{
	int		i;

	printf( "\n" );
	if( Absent == 0 ) {
		printf( "All machines on the roster are present\n" );
		return;
	}
		
	if( Absent == 1 ) {
		printf( "The following machine is absent...\n" );
	} else {
		printf( "The following %d machines are absent...\n", Absent );
	}

	for( i=0; i<RosterLen; i++ ) {
		if( !Roster[i].present ) {
			printf( "%-15s %s\n", Roster[i].name, Roster[i].comment );
		}

	}
}

mark_present( name )
char	*name;
{
	int		i;
	char	*ptr;

	if( ptr=index(name,'.') ) {
		*ptr = '\0';
	}

	for( i=0; i<RosterLen; i++ ) {
		if( strcmp(name,Roster[i].name) == 0 ) {
			Roster[i].present = TRUE;
			Absent -= 1;
			return;
		}
	}
	Unknowns[N_Unknowns] = strdup( name );
	N_Unknowns += 1;
}
