/* 
** Copyright 1986, 1987, 1988, 1989 University of Wisconsin
** 
** Permission to use, copy, modify, and distribute this software and its
** documentation for any purpose and without fee is hereby granted,
** provided that the above copyright notice appear in all copies and that
** both that copyright notice and this permission notice appear in
** supporting documentation, and that the name of the University of
** Wisconsin not be used in advertising or publicity pertaining to
** distribution of the software without specific, written prior
** permission.  The University of Wisconsin makes no representations about
** the suitability of this software for any purpose.  It is provided "as
** is" without express or implied warranty.
** 
** THE UNIVERSITY OF WISCONSIN DISCLAIMS ALL WARRANTIES WITH REGARD TO
** THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND
** FITNESS. IN NO EVENT SHALL THE UNIVERSITY OF WISCONSIN  BE LIABLE FOR
** ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
** WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
** ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
** OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
** 
** Authors:  Allan Bricker and Michael J. Litzkow,
** 	         University of Wisconsin, Computer Sciences Dept.
** 
*/ 


#include <stdio.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/param.h>
#include <sys/file.h>
#include <sys/dir.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <sys/stat.h>
#include "except.h"
#include "expr.h"
#include "proc.h"

#ifdef NDBM
#include <ndbm.h>
#else NDBM
#include "ndbm_fake.h"
#endif NDBM


#define OK		1
#define MATCH	0
#define BAD		0
#define MINUTE 60
#define HOUR (MINUTE * 60)

static char *_FileName_ = __FILE__;		/* Used by EXCEPT (see except.h)     */
extern int	errno;

int		check_spool_file();
int		check_log_file();
int		check_execute_file();
DBM     *Q, *OpenJobQueue();
FILE	*fdopen();

char	*Spool;
char	*Execute;
char	*Log;
char	*CondorAdministrator;
int     MaxCkptInterval;
char	template[] = "/tmp/condorXXXXXX";
char	MailFile[ sizeof(template) ];
FILE	*MailFp;
int		BadFiles;
char	*MyName;
int		MailFlag;

usage()
{
	fprintf( stderr, "Usage: %s [-m]\n", MyName );
	exit( 1 );
}

main( argc, argv )
int		argc;
char	*argv[];
{
	char	queue_name[MAXPATHLEN];

	MyName = argv[0];
	config( MyName, (CONTEXT *)0 );
	init_params();

	for( argv++; *argv; argv++ ) {
		if( (*argv)[0] == '-' && (*argv)[1] == 'm' ) {
			MailFlag++;
			continue;
		}
		usage();
	}

	if( MailFlag ) {
		open_mail_file();
	}


        /* Open job queue */
    (void)sprintf( queue_name, "%s/job_queue", Spool );
    if( (Q=OpenJobQueue(queue_name,O_RDONLY,0)) == NULL ) {
        EXCEPT( "OpenJobQueue(%s)", queue_name );
    }

    if( LockJobQueue(Q,READER) < 0 ) {
        EXCEPT( "LockJobQueue()" );
    }

	scan_dir( Spool, check_spool_file );
	scan_dir( Execute, check_execute_file );
	scan_dir( Log, check_log_file );

	if( MailFlag ) {
		fclose( MailFp );
		if( BadFiles ) {
			send_mail_file();
		}
		(void)unlink( MailFile );
	}

	exit( 0 );
}

scan_dir( name, func )
char	*name;
int		(*func)();
{
	DIR	*dirp, *opendir();
	struct direct	*dp;
	char	path[MAXPATHLEN];


	if( (dirp=opendir(name)) == NULL ) {
		EXCEPT( "opendir(%s)", name );
	}

	for( dp=readdir(dirp); dp != NULL; dp = readdir(dirp) ) {

		if( dp->d_namlen == 1 && strcmp(dp->d_name,".") == MATCH ) {
			continue;
		}

		if( dp->d_namlen == 2 && strcmp(dp->d_name,"..") == MATCH ) {
			continue;
		}

		if( func(name,dp->d_name) != OK ) {
			BadFiles += 1;
			sprintf( path, "%s/%s", name, dp->d_name );
			if( MailFlag ) {
				fprintf( MailFp,"%s\n", path );
			} else {
				printf( "%s BAD\n", path );
			}
			/* (void)unlink( path ); */
		}
	}
}

check_log_file( dir, name )
char	*dir;
char	*name;
{

	if( strcmp(name,"MasterLog") == MATCH ) {
		return OK;
	}

	if( strcmp(name,"MasterLog.old") == MATCH ) {
		return OK;
	}

	if( strcmp(name,"NegotiatorLog") == MATCH ) {
		return OK;
	}

	if( strcmp(name,"NegotiatorLog.old") == MATCH ) {
		return OK;
	}

	if( strcmp(name,"CollectorLog") == MATCH ) {
		return OK;
	}

	if( strcmp(name,"CollectorLog.old") == MATCH ) {
		return OK;
	}

	if( strcmp(name,"SchedLog") == MATCH ) {
		return OK;
	}

	if( strcmp(name,"SchedLog.old") == MATCH ) {
		return OK;
	}

	if( strcmp(name,"ShadowLog") == MATCH ) {
		return OK;
	}

	if( strcmp(name,"ShadowLog.old") == MATCH ) {
		return OK;
	}

	if( strcmp(name,"StartLog") == MATCH ) {
		return OK;
	}

	if( strcmp(name,"StartLog.old") == MATCH ) {
		return OK;
	}

	if( strcmp(name,"KbdLog") == MATCH ) {
		return OK;
	}

	if( strcmp(name,"KbdLog.old") == MATCH ) {
		return OK;
	}

	if( strcmp(name,"ShadowLock") == MATCH ) {
		return OK;
	}

	return BAD;
}

check_execute_file( dir, file )
char	*dir;
char	*file;
{
	char	pathname[MAXPATHLEN];
	int		age;
	struct stat	buf;

	sprintf( pathname, "%s/%s", dir, file );
	if( stat(pathname,&buf) < 0 ) {
		if( errno == ENOENT ) {
			return OK; /* got rm'd while we were doing this */
		} else {
			EXCEPT( "stat(%s,0x%x)", pathname, &buf );
		}
	}
	age = time(0) - buf.st_ctime;

	if( age > MaxCkptInterval + 1 * HOUR ) {
		return BAD;
	} else {
		return OK;
	}
}

check_spool_file( dir, name )
char	*dir;
char	*name;
{
	char	*ptr, *index();
	int		cluster;
	int		proc;

	if( strcmp(name,"job_queue.dir") == MATCH ) {
		return OK;
	}

	if( strcmp(name,"job_queue.pag") == MATCH ) {
		return OK;
	}

	if( strcmp(name,"history") == MATCH ) {
		return OK;
	}

	if( ptr=index(name,'.') ) {
		ptr += 1;
	} else {
		return BAD;
	}

	if( strcmp(ptr,"ickpt") == MATCH ) {
		cluster = atoi( name + 3 );
		if( check_cluster(cluster) == OK ) {
			return OK;
		} else {
			return BAD;
		}
	}

	if( strncmp(ptr,"ckpt.",5) == MATCH ) {
		cluster = atoi( name + 3 );
		ptr += 5;
		proc = atoi( ptr );
		if( check_job(cluster,proc) == OK ) {
			return OK;
		} else {
			return BAD;
		}
	}

	return BAD;
}

check_job( cluster, job )
int		cluster;
int		job;
{
	PROC	proc;
	int		answer;

	proc.id.cluster = cluster;
	proc.id.proc = job;

	if( FetchProc(Q,&proc) < 0 ) {
		return BAD;
	}
	switch( proc.status ) {
		case UNEXPANDED:
		case REMOVED:
		case COMPLETED:
		case SUBMISSION_ERR:
			answer = BAD;
			break;
		case IDLE:
		case RUNNING:
			answer = OK;
			break;
		default:
			EXCEPT( "Job %d.%d has unknown status (%d)",
										cluster, job, proc.status );
	}
	xdr_free_proc( &proc );
	return answer;
}
	

/*
** Check to see if the specified cluster exists in the job queue.  Return
** OK if so, and BAD otherwise.
*/
check_cluster( cluster )
int		cluster;
{
	PROC	proc;

	proc.id.cluster = cluster;
	proc.id.proc = 0;

	if( FetchProc(Q,&proc) < 0 ) {
		return BAD;
	}
	xdr_free_proc( &proc );
	return OK;
}

init_params()
{
    char    *param();
	char	*tmp;

	Spool = param("SPOOL");
    if( Spool == NULL ) {
        EXCEPT( "SPOOL not specified in config file\n" );
    }

	Log = param("LOG");
    if( Log == NULL ) {
        EXCEPT( "LOG not specified in config file\n" );
    }

	Execute = param("EXECUTE");
    if( Execute == NULL ) {
        EXCEPT( "EXECUTE not specified in config file\n" );
    }

    if( (tmp=param("MAX_CKPT_INTERVAL")) == NULL ) {
        MaxCkptInterval = 2 * HOUR;
    } else {
        MaxCkptInterval = atoi( tmp );
    }

    if( MaxCkptInterval > 2 * HOUR ) {
        MaxCkptInterval = 2 * HOUR;
    }

    if( MaxCkptInterval < 15 * MINUTE ) {
        MaxCkptInterval = 15 * MINUTE;
    }

    if( (CondorAdministrator = param("CONDOR_ADMIN")) == NULL ) {
        EXCEPT( "CONDOR_ADMIN not specified in config file" );
    }

}

open_mail_file()
{
	int		fd;
	char	host[1024];

	for(;;) {
		strcpy( MailFile, template );
		(void)mktemp( MailFile );
		if( (fd=open(MailFile,O_WRONLY|O_CREAT|O_EXCL,0664)) >= 0 ) {
			break;
		}
		if( errno != EEXIST ) {
			EXCEPT( "open(%s,O_WRONLY|O_CREAT|O_EXCL,0664)", MailFile );
		}
	}

	if( (MailFp=fdopen(fd,"w")) == NULL ) {
		EXCEPT( "fdopen(%d,\"w\")", fd );
	} 

	(void)gethostname( host, sizeof(host) );
	fprintf( MailFp, "To: %s\n", CondorAdministrator );
	fprintf( MailFp, "Subject: Junk Condor Files\n" );
	fprintf( MailFp, "\n" );
	fprintf( MailFp, "These files should be removed from <%s>:\n\n", host );
}

send_mail_file()
{
	char	cmd[1024];

	sprintf( cmd, "/bin/mail %s < %s", CondorAdministrator, MailFile );
	(void)system( cmd );
}


SetSyscalls(){}
