/* 
** Copyright 1986, 1987, 1988, 1989 University of Wisconsin
** 
** Permission to use, copy, modify, and distribute this software and its
** documentation for any purpose and without fee is hereby granted,
** provided that the above copyright notice appear in all copies and that
** both that copyright notice and this permission notice appear in
** supporting documentation, and that the name of the University of
** Wisconsin not be used in advertising or publicity pertaining to
** distribution of the software without specific, written prior
** permission.  The University of Wisconsin makes no representations about
** the suitability of this software for any purpose.  It is provided "as
** is" without express or implied warranty.
** 
** THE UNIVERSITY OF WISCONSIN DISCLAIMS ALL WARRANTIES WITH REGARD TO
** THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND
** FITNESS. IN NO EVENT SHALL THE UNIVERSITY OF WISCONSIN  BE LIABLE FOR
** ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
** WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
** ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
** OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
** 
** Authors:  Allan Bricker and Michael J. Litzkow,
** 	         University of Wisconsin, Computer Sciences Dept.
** 
*/ 


#include <stdio.h>
#include "errno.h"
#include <sys/types.h>
#include <sys/param.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <sys/stat.h>

#if !defined(ultrix)
#include <sys/vfs.h>
#endif !defined(ultrix)

#include <sys/wait.h>
#include <sys/ioctl.h>
#include <rpc/types.h>
#include <rpc/xdr.h>

#include "condor_types.h"
#include "condor_sys.h"
#include "condor_rsc.h"
#include "trace.h"
#include "except.h"
#include "debug.h"
#include "fileno.h"
#include "proc.h"
#include "xdr_lib.h"
#include "clib.h"

static char *_FileName_ = __FILE__;		/* Used by EXCEPT (see except.h)     */

static int RSCSock;
static int LogSock;
extern int	MyPipe;
extern int	LogPipe;
extern int	LockFd;

static FILE *LogFP;


#define XDR_BUFSIZE	(4 * 1024)

static XDR xdr_RSC, *xdr_syscall = &xdr_RSC;

XDR *
RSC_ShadowInit( rscsock, errsock )
{
	int XDR_ShadowRead(), XDR_ShadowWrite();

	RSCSock = rscsock;
	xdrrec_create( xdr_syscall, XDR_BUFSIZE, XDR_BUFSIZE, &RSCSock,
			XDR_ShadowRead, XDR_ShadowWrite );
	
	xdr_syscall->x_op = XDR_ENCODE;

	LogSock = errsock;
	LogFP = fdopen(LogSock, "r");

	if( LogFP == NULL ) {
		EXCEPT("Cannot fdopen LogSock");
	}

	return( xdr_syscall );
}

static
XDR_ShadowRead( iohandle, buf, len )
register int *iohandle;
char 	*buf;
int 	len;
{
	register int cnt;
	fd_set readfds;
	int nfds;

	FD_ZERO(&readfds);

	nfds = MAX(*iohandle, LogSock) + 1;

	for(;;) {
		FD_SET(*iohandle, &readfds);
		FD_SET(LogSock, &readfds);


		dprintf(D_XDR,
		"Shadow: XDR_ShadowRead: about to select on <%d, %d, and %d>\n",
				*iohandle, LogSock, UMBILICAL);

		cnt = select(nfds, (int *)&readfds, (int *)0, (int *)0,
														(struct timeval *)0);
		if( cnt < 0 ) {
			EXCEPT("XDR_ShadowRead: select");
		}

		if( FD_ISSET(LogSock, &readfds) ) {
			HandleLog();
			continue;
		}

		if( FD_ISSET(*iohandle, &readfds) ) {
			break;
		}

		if( FD_ISSET(UMBILICAL, &readfds) ) {
			dprintf(D_ALWAYS,
				"Shadow: Local scheduler apparently died, so I die too\n");
			exit(1);
		}
	}

	dprintf(D_XDR, "Shadow: XDR_ShadowRead: about to read(%d, 0x%x, %d)\n",
			*iohandle, buf, len );
	
	cnt = read( *iohandle, buf, len );

	dprintf(D_XDR, "Shadow: XDR_ShadowRead: cnt = %d\n", cnt );

	return( cnt );
}

static
XDR_ShadowWrite( iohandle, buf, len )
register int *iohandle;
char *buf;
int	 len;
{
	register int cnt;
	fd_set readfds;
	fd_set writefds;
	int nfds;

	FD_ZERO(&readfds);
	FD_ZERO(&writefds);

	nfds = MAX(*iohandle, LogSock) + 1;

	for(;;) {
		FD_SET(*iohandle, &writefds);
		FD_SET(LogSock, &readfds);


		dprintf(D_XDR,
		"Shadow: XDR_ShadowWrite: about to select on <%d, %d, and %d>\n",
				*iohandle, LogSock, UMBILICAL);

		cnt = select(nfds, (int *)&readfds, (int *)&writefds, (int *)0, 
														(struct timeval *)0 );
		if( cnt < 0 ) {
			EXCEPT("XDR_ShadowWrite: select");
		}

		if( FD_ISSET(LogSock, &readfds) ) {
			HandleLog();
			continue;
		}

		if( FD_ISSET(*iohandle, &writefds) ) {
			break;
		}

		if( FD_ISSET(UMBILICAL, &readfds) ) {
			dprintf(D_ALWAYS,
				"Shadow: Local scheduler apparently died, so I'm dying too\n");
			exit(1);
		}
	}

	dprintf(D_XDR, "Shadow: XDR_ShadowWrite: about to write(%d, 0x%x, %d)\n",
			*iohandle, buf, len );
	
	cnt = write( *iohandle, buf, len );

	dprintf(D_XDR, "Shadow: XDR_ShadowWrite: cnt = %d\n", cnt );

	ASSERT(cnt == len);

	return( cnt );
}


static char *MemToFree = NULL;

#define RSC_NO_ARGS(func) { \
	int func(); \
	return( RSC_no_args(func, condor_sysnum) ); \
}

#define RSC_1INT(func) { \
	int func(); \
	return( RSC_1int(func, condor_sysnum) ); \
}

#define RSC_2INT(func) { \
	int func(); \
	return( RSC_2int(func, condor_sysnum) ); \
}

#define RSC_3INT(func) { \
	int func(); \
	return( RSC_3int(func, condor_sysnum) ); \
}

#define RSC_1STR(func) { \
	int func(); \
	return( RSC_1str(func, condor_sysnum) ); \
}

#define RSC_1STR_1INT(func) { \
	int func(); \
	return( RSC_1str_1int(func, condor_sysnum) ); \
}

#define RSC_1STR_2INT(func) { \
	int func(); \
	return( RSC_1str_2int(func, condor_sysnum) ); \
}

#define RSC_2STR(func) { \
	int func(); \
	return( RSC_2str(func, condor_sysnum) ); \
}

#define RSC_1INT_1RBUF(func, s_name, xdr_func) { \
	int func(), xdr_func(); \
	return( RSC_1int_1Rbuf(func, condor_sysnum, \
					sizeof(struct s_name), xdr_func) ); \
}

#define RSC_1STR_1RBUF(func, s_name, xdr_func) { \
	int func(), xdr_func(); \
	return( RSC_1str_1Rbuf(func, condor_sysnum, \
				sizeof(struct s_name), xdr_func) ); \
}

int condor_sysnum;
do_REMOTE_syscall()
{
	int rval;
	extern errno;

	xdr_syscall->x_op = XDR_DECODE;

	ASSERT( xdrrec_skiprecord(xdr_syscall) );
	ASSERT( xdr_int(xdr_syscall, &condor_sysnum) );

	dprintf(D_SYSCALLS, "Shadow: got request for syscall %d <%s>\n",
				condor_sysnum, CONDOR_SYSCALLNAME(condor_sysnum));

	switch( condor_sysnum ) {
	/*
	**	System calls with no arguments
	*/
	case CONDOR_async_daemon:	/* async_daemon()                            */
		RSC_NO_ARGS(CONDOR_NotSupported);

	case CONDOR_fork:			/* fork()                                    */
		RSC_NO_ARGS(CONDOR_NotSupported);

	case CONDOR_getdtablesize:	/* getdtablesize()                           */
		RSC_NO_ARGS(getdtablesize);

#ifdef DAYAO
	case PSEUDO_processlogging: /* log file transfer times					 */
		RSC_2INT(ProcessLogging);
#endif DAYAO

	case PSEUDO_getegid:	/* getegid()                                     */
		RSC_NO_ARGS(getegid);

	case PSEUDO_geteuid:	/* geteuid()                                     */
		RSC_NO_ARGS(geteuid);

	case CONDOR_getgid:			/* getgid()                                  */
		RSC_NO_ARGS(getgid);

	case CONDOR_gethostid:		/* gethostid()                               */
		RSC_NO_ARGS(gethostid);

	case CONDOR_getpagesize:	/* getpagesize()                             */
		RSC_NO_ARGS(getpagesize);

	case CONDOR_getpid:			/* getpid()                                  */
		RSC_NO_ARGS(condor_getpid);

	case PSEUDO_getppid:	/* getppid()                                     */
		RSC_NO_ARGS(condor_getppid);

	case CONDOR_getuid:			/* getuid()                                  */
		RSC_NO_ARGS(getuid);

	case CONDOR_sync:			/* sync()                                    */
		RSC_NO_ARGS(sync);

	case CONDOR_vhangup:		/* vhangup()                                 */
		RSC_NO_ARGS(vhangup);

	/*
	**	System calls which take one integer argument
	*/
	case CONDOR_close:			/* close( int d )                            */
		RSC_1INT(condor_close);

	case CONDOR_dup:			/* dup( int oldd )                           */
		RSC_1INT(dup);

	case CONDOR__exit:			/* _exit( int status )                       */
		RSC_1INT(condor_exit);

	case CONDOR_fsync:			/* fsync( int fd )                           */
		RSC_1INT(fsync);


	case CONDOR_getpgrp:		/* getpgrp( int pid )                        */
		RSC_1INT(getpgrp);

	case CONDOR_nfssvc:			/* nfssvc( int sock )                        */
		RSC_1INT(CONDOR_NotSupported);

	case CONDOR_reboot:			/* reboot( int howto )                       */
		RSC_1INT(reboot);

	case CONDOR_umask:			/* umask( int numask )                       */
		RSC_1INT(umask);

	/*
	**	System calls which take one long argument
	*/
	case CONDOR_sethostid:		/* sethostid( int hostid )                   */
	{
		int hostid;

		ASSERT(xdr_int(xdr_syscall, &hostid));

#ifdef SYS_sethostid
		rval = sethostid( hostid );
#else SYS_sethostid
		rval = CONDOR_NotSupported();
#endif SYS_sethostid

		dprintf(D_SYSCALLS, "Shadow: sethostid(%d) = %d\n", hostid, rval);

		PUT_RVAL(xdr_syscall, rval);
		END_RVAL(xdr_syscall, 0);
	}

	/*
	**	System calls which take two integer arguments
	*/
	case CONDOR_dup2:			/* dup2( int oldd, int newd )                */
		RSC_2INT(condor_dup2);

	case CONDOR_fchmod:			/* fchmod( int fd, int mode )                */
		RSC_2INT(fchmod);

	case CONDOR_flock:			/* flock( int fd, int operation )            */
		RSC_2INT(flock);

	case CONDOR_getpriority:	/* getpriority( int which, int who )         */
		RSC_2INT(getpriority);

	case CONDOR_kill:			/* kill( int pid, int sig )                  */
		RSC_2INT(CONDOR_NotSupported);

	case CONDOR_killpg:			/* killpg( int pgrp, int sig )               */
		RSC_2INT(CONDOR_NotSupported);

	case CONDOR_listen:			/* listen( int s, int backlog )              */
		RSC_2INT(listen);

	case CONDOR_setpgrp:		/* setpgrp( int pid, int pgrp )              */
		RSC_2INT(setpgrp);

	case CONDOR_setregid:		/* setregid( int rgid, int egid )            */
		RSC_2INT(setregid);

	case CONDOR_setreuid:		/* setreuid( int ruid, int euid )            */
		RSC_2INT(setreuid);

	case CONDOR_shutdown:		/* shutdown( int s, int how )                */
		RSC_2INT(shutdown);

	/*
	**	System calls which take an integer argument and an off_t argument
	*/
	case CONDOR_ftruncate:		/* ftruncate( int fd, off_t length )         */
	{
		int fd, length;

		ASSERT(xdr_int(xdr_syscall, &fd));
		ASSERT(xdr_int(xdr_syscall, &length));

		rval = ftruncate( fd, length );

		dprintf(D_SYSCALLS, "Shadow: ftruncate(%d, %d) = %d\n",
				fd, length, rval);

		PUT_RVAL(xdr_syscall, rval);
		END_RVAL(xdr_syscall, 0);
	}

	/*
	**	System calls which take three integer arguments
	*/
	case CONDOR_fchown:			/* fchown( int fd, int owner, int group )    */
		RSC_3INT(fchown);

	case CONDOR_fcntl:			/* fcntl( int fd, int cmd, int arg )         */
		RSC_3INT(fcntl);

	case CONDOR_setpriority:	/* setpriority( int which, int who,
								**					int prio )
								*/
		RSC_3INT(setpriority);

	case CONDOR_socket:			/* socket( int domain, int type,
								**					int protocol ) 
								*/
		RSC_3INT(CONDOR_NotSupported);

	/*
	**  System calls which take an integer argument followed by
	**  an off_t argument followed by another integer argument.
	*/
	case CONDOR_lseek:			/* lseek( int d, off_t offset, int whence )  */
	{
		int d, whence;
		off_t pos, offset;

		ASSERT(xdr_int(xdr_syscall, &d));
		ASSERT(xdr_int(xdr_syscall, &offset));
		ASSERT(xdr_int(xdr_syscall, &whence));

		pos = lseek( d, offset, whence );

		dprintf(D_SYSCALLS, "Shadow: lseek(%d, %d, %d) = %d\n",
				d, offset, whence, pos);

		PUT_RVAL(xdr_syscall, pos);
		END_RVAL(xdr_syscall, 0);
	}

	/*
	**	System calls which take one string argument
	*/
	case CONDOR_acct:			/* acct( char *file  )                       */
		RSC_1STR(acct);

	case CONDOR_chdir:			/* chdir( char *path )                       */
		RSC_1STR(chdir);

	case CONDOR_chroot:			/* chroot( char *dirname )                   */
		RSC_1STR(chroot);

	case CONDOR_rmdir:			/* rmdir( char *path )                       */
		RSC_1STR(rmdir);

	case CONDOR_swapon:			/* swapon( char *special )                   */
		RSC_1STR(swapon);

	case CONDOR_unlink:			/* unlink( char *path )                      */
		RSC_1STR(unlink);

	case CONDOR_unmount:		/* unmount( char *name )                     */
		RSC_1STR(CONDOR_NotSupported);

	/*
	**	System calls which take one string and one integer as arguments
	*/
	case CONDOR_access:			/* access( char *path, int mode )            */
		RSC_1STR_1INT(access);

	case CONDOR_chmod:			/* chmod( char *path, int mode )             */
		RSC_1STR_1INT(chmod);

	case CONDOR_creat:			/* creat( char *path, int mode )             */
		RSC_1STR_1INT(creat);

	case CONDOR_mkdir:			/* mkdir( char *path, int mode )             */
		RSC_1STR_1INT(mkdir);

	case CONDOR_setdomainname:	/* setdomainname( char *name, int namelen )  */
#ifdef SYS_setdomainname
		RSC_1STR_1INT(setdomainname);
#else SYS_setdomainname
		RSC_1STR_1INT(CONDOR_NotSupported);
#endif SYS_setdomainname

	case CONDOR_sethostname:	/* sethostname( char *name, int namelen )    */
		RSC_1STR_1INT(sethostname);

	/*
	**	System calls which take one string and one off_t as arguments
	*/
	case CONDOR_truncate:		/* truncate( char *path, off_t length )      */
	{
		char path[ MAXPATHLEN ], *pp = path;
		off_t length;

		ASSERT(xdr_string(xdr_syscall, &pp, (u_int) MAXPATHLEN));
		ASSERT(xdr_int(xdr_syscall, &length));

		rval = truncate(path, length);

		dprintf(D_SYSCALLS, "Shadow: truncate(%s, %d) = %d\n",
				path, length, rval);

		PUT_RVAL(xdr_syscall, rval);
		END_RVAL(xdr_syscall, 0);
	}

	/*
	**	System calls which take one string and two integers as arguments
	*/
	case CONDOR_chown:			/* chown( char *path, int owner, int group ) */
		RSC_1STR_2INT(chown);

	case CONDOR_mknod:			/* mknod( char *path, int mode, int dev )    */
		RSC_1STR_2INT(mknod);

	case CONDOR_open:			/* open( char *path, int flags, int mode )   */
		RSC_1STR_2INT(open);

	/*
	**	System calls which take two strings as arguments
	*/
	case CONDOR_link:			/* link( char *name1, char *name2 )          */
		RSC_2STR(link);

	case CONDOR_rename:			/* rename( char *from, char *to )            */
		RSC_2STR(rename);

	case CONDOR_symlink:		/* symlink( char *name1, char *name2 )       */
		RSC_2STR(symlink);

	/*
	**	Others...
	*/
	case CONDOR_wait:			/* wait( R union wait *status )              */
	{
		union wait status;

		/*
		** rval = wait( &status );
		*/
		rval = CONDOR_NotSupported();

		dprintf(D_SYSCALLS, "Shadow: wait(0x%x) = %d\n", &status, rval );

		PUT_RVAL(xdr_syscall, rval);

		ASSERT(xdr_wait(xdr_syscall, &status));

		END_RVAL(xdr_syscall, 0);
	}

	case PSEUDO_reallyexit:	/* reallyexit( M union wait *status )            */
	{
		extern union wait JobStatus;
		extern struct rusage JobRusage;

		ASSERT(xdr_wait(xdr_syscall, &JobStatus));
		ASSERT(xdr_rusage(xdr_syscall, &JobRusage));

		rval = 0;
		dprintf(D_SYSCALLS, "Shadow: reallyexit(0x%x, 0x%x) = %d\n",
								&JobStatus, &JobRusage, rval );

		PUT_RVAL(xdr_syscall, rval);
		END_RVAL(xdr_syscall, -1);
	}

	case PSEUDO_wait3:	/* wait3( R union wait *status, int options,
						**						struct rusage *rusage )
						*/
	{
		int xdr_rusage();
		union wait status;
		int options;
		struct rusage *rusage;

		ASSERT(xdr_int(xdr_syscall, &options));
		ASSERT(xdr_ptr(xdr_syscall, (caddr_t *) &rusage, xdr_rusage));

		/*
		**	rval = wait3( &status, options, rusage );
		*/
		rval = CONDOR_NotSupported();

		dprintf(D_SYSCALLS, "Shadow: wait3(0x%x, 0%o, 0x%x) = %d\n",
			&status, options, rusage, rval );

		PUT_RVAL(xdr_syscall, rval);

		ASSERT(xdr_wait(xdr_syscall, &status));
		ASSERT(xdr_ptr(xdr_syscall, (caddr_t *) &rusage, xdr_rusage));

		END_RVAL(xdr_syscall, 0);
	}

	case PSEUDO_getwd: /* getwd( R char *pathname )                          */
	{
		char pathname[ MAXPATHLEN ], *pnp = pathname;

		if( getwd(pathname) ) {
			rval = 1;
		} else {
			rval = 0;
		}

		dprintf(D_SYSCALLS, "Shadow: getwd(%s) = 0x%x\n", pathname, rval);

		PUT_RVAL(xdr_syscall, rval);
		ASSERT(xdr_string(xdr_syscall, &pnp, (u_int) MAXPATHLEN));
		END_RVAL(xdr_syscall, 0);
	}

	case CONDOR_lstat:			/* lstat( char *path, R struct stat *buf )   */
		RSC_1STR_1RBUF(lstat,stat,xdr_stat);

	case CONDOR_stat:			/* stat( char *path, R struct stat *buf )    */
		RSC_1STR_1RBUF(stat,stat,xdr_stat);

	case CONDOR_pipe:			/* pipe( int fildes[2] )                     */
	{
		int fildes[2];

		/*
		**	rval = pipe(fildes);
		*/
		rval = CONDOR_NotSupported();

		dprintf(D_SYSCALLS, "Shadow: pipe(0x%x) = %d\n", fildes, rval);

		PUT_RVAL(xdr_syscall, rval);

		ASSERT(xdr_int(xdr_syscall, &fildes[0]));
		ASSERT(xdr_int(xdr_syscall, &fildes[1]));

		END_RVAL(xdr_syscall, 0);
	}

	case CONDOR_settimeofday:	/* settimeofday( struct timeval *tp,
								**				 struct timezone *tzp )
								*/
	{
		struct timeval tv, *tp = &tv;
		struct timezone tz, *tzp = &tz;
		int xdr_timezone();
		
		ASSERT(timeval_xdr(xdr_syscall, tp));
		ASSERT(xdr_ptr(xdr_syscall, (caddr_t *) &tzp, xdr_timezone));

		rval = settimeofday( tp, tzp );

		dprintf(D_SYSCALLS, "Shadow: settimeofday(0x%x, 0x%x) = %d\n",
					tp, tzp, rval);

		PUT_RVAL(xdr_syscall, rval);

		END_RVAL(xdr_syscall, 0);
	}

	case CONDOR_gettimeofday:	/* gettimeofday( R struct timeval *tp,
							**				 R struct timezone *tzp )
							*/
	{
		struct timeval tv, *tp = &tv;
		struct timezone tz, *tzp = &tz;
		int xdr_timezone();
		
		ASSERT(xdr_ptr(xdr_syscall, (caddr_t *) &tzp, xdr_timezone));

		rval = gettimeofday( tp, tzp );

		dprintf(D_SYSCALLS,
			"Shadow: gettimeofday(0x%x <%d, %d>, 0x%x <%d, %d>) = %d\n",
				tp, tp->tv_sec, tp->tv_usec,
				tzp, (tzp?tzp->tz_minuteswest:0), (tzp?tzp->tz_dsttime:0),
				rval);

		PUT_RVAL(xdr_syscall, rval);

		ASSERT(timeval_xdr(xdr_syscall, tp));
		ASSERT(xdr_ptr(xdr_syscall, (caddr_t *) &tzp, xdr_timezone));

		END_RVAL(xdr_syscall, 0);
	}

	case CONDOR_getdomainname:	/* getdomainname( R char *name, int namelen )*/
	{
		char *name, *malloc();
		int namelen;
		
		ASSERT(xdr_int(xdr_syscall, &namelen));

		MemToFree = name = malloc( (unsigned)namelen );
		if( name == NULL ) {
			rval = -1;
		} else {
			rval = getdomainname( name, namelen );
		}

		dprintf(D_SYSCALLS, "Shadow: getdomainname(%s, %d) = %d\n",
					name, namelen, rval);

		PUT_RVAL(xdr_syscall, rval);
		ASSERT(xdr_string(xdr_syscall, &name, (u_int) namelen));
		END_RVAL(xdr_syscall, 0);
	}

	case CONDOR_gethostname:	/* gethostname( R char *name, int namelen )  */
	{
		char *name, *malloc();
		int namelen;
		
		ASSERT(xdr_int(xdr_syscall, &namelen));

		MemToFree = name = malloc( (unsigned)namelen );
		if( name == NULL ) {
			rval = -1;
		} else {
			rval = gethostname( name, namelen );
		}

		dprintf(D_SYSCALLS, "Shadow: gethostname(%s, %d) = %d\n",
					name, namelen, rval);

		PUT_RVAL(xdr_syscall, rval);
		ASSERT(xdr_string(xdr_syscall, &name, (u_int) namelen));
		END_RVAL(xdr_syscall, 0);
	}

	case CONDOR_select:		/* select( int nfds,
							**		0/VR fd_set *readfds,
							**		0/VR fd_set *writefds,
							**		0/VR fd_set *exceptfds,
							**		0/V struct timeval *timeout )
							*/
	{
		int xdr_fdset(), timeval_xdr();
		int nfds;
		fd_set Rfds, *readfds = &Rfds;
		fd_set Wfds, *writefds = &Wfds;
		fd_set Efds, *exceptfds = &Efds;
		struct timeval TO, *timeout = &TO;

		ASSERT(xdr_int(xdr_syscall, &nfds));
		ASSERT(xdr_ptr(xdr_syscall, (caddr_t *) &readfds, xdr_fdset));
		ASSERT(xdr_ptr(xdr_syscall, (caddr_t *) &writefds, xdr_fdset));
		ASSERT(xdr_ptr(xdr_syscall, (caddr_t *) &exceptfds, xdr_fdset));
		ASSERT(xdr_ptr(xdr_syscall, (caddr_t *) &timeout, timeval_xdr));

		rval = select( nfds, (int *)readfds, (int *)writefds,
												(int *)exceptfds, timeout );

		PUT_RVAL(xdr_syscall, rval);

		ASSERT(xdr_ptr(xdr_syscall, (caddr_t *) &readfds, xdr_fdset));
		ASSERT(xdr_ptr(xdr_syscall, (caddr_t *) &writefds, xdr_fdset));
		ASSERT(xdr_ptr(xdr_syscall, (caddr_t *) &exceptfds, xdr_fdset));
		ASSERT(xdr_ptr(xdr_syscall, (caddr_t *) &timeout, timeval_xdr));

		END_RVAL(xdr_syscall, 0);
	}

	case CONDOR_setgroups:		/* setgroups( int ngroups, M int *gidset )   */
	{
		int ngroups, xdr_int();
		int GIDSet[ NGROUPS ], *gidset = GIDSet;

		ASSERT(xdr_array(xdr_syscall, &gidset, &ngroups, NGROUPS, sizeof(int), xdr_int));

		rval = setgroups( ngroups, gidset );

		PUT_RVAL(xdr_syscall, rval);
		END_RVAL(xdr_syscall, 0);
	}

	case CONDOR_setitimer:		/* setitimer( int which,
								**			M struct itimerval *value,
								**		  R/0 struct itimerval *ovalue )
								*/
	{
		int which;
		struct itimerval Val, *value = &Val;
		struct itimerval OVal, *ovalue = &OVal;
		
		ASSERT(xdr_int(xdr_syscall, &which));
		ASSERT(xdr_itimerval(xdr_syscall, value));
		ASSERT(xdr_ptr(xdr_syscall, (caddr_t *) &ovalue, timeval_xdr));

		/*
		** rval = setitimer( which, value, ovalue );
		*/
		rval = CONDOR_NotSupported();

		PUT_RVAL(xdr_syscall, rval);

		ASSERT(xdr_ptr(xdr_syscall, (caddr_t *) &ovalue, timeval_xdr));
		END_RVAL(xdr_syscall, 0);
	}

	case CONDOR_setrlimit:		/* setrlimit( int resource,
								**			M struct rlimit *rlp )
								*/
	{
		int resource;
		struct rlimit RLim, *rlp = &RLim;
		
		ASSERT(xdr_int(xdr_syscall, &resource));
		ASSERT(xdr_rlimit(xdr_syscall, rlp));

		rval = setrlimit( resource, rlp );

		PUT_RVAL(xdr_syscall, rval);
		END_RVAL(xdr_syscall, 0);
	}

	case CONDOR_getdirentries:	/* getdirentries( int fd,
								**				R char *buf,
								**				  int nbytes,
								**				R off_t *basep )
								*/
	{
		int fd;
		char *buf;
		int nbytes;
		long Base, *basep = &Base;

		ASSERT(xdr_int(xdr_syscall, &fd));
		ASSERT(xdr_int(xdr_syscall, &nbytes));

		MemToFree = buf = malloc( (unsigned)nbytes );
#if defined(SYS_getdirentries) && !defined(SUNOS40)
		rval = getdirentries( fd, buf, (unsigned)nbytes, basep );
#else defined(SYS_getdirentries) && !defined(SUNOS40)
		rval = CONDOR_NotSupported();
#endif defined(SYS_getdirentries) && !defined(SUNOS40)

		PUT_RVAL(xdr_syscall, rval);

		ASSERT(xdr_direntries(xdr_syscall, buf, rval));
		ASSERT(xdr_off_t(xdr_syscall, basep));

		END_RVAL(xdr_syscall, 0);
	}

	case CONDOR_getgroups:		/* getgroups( int gidsetlen, R int *gidset ) */
	{
		int gidsetlen;
		int GidSet[ NGROUPS ], *gidset = GidSet;
		register int i;

		ASSERT(xdr_int(xdr_syscall, &gidsetlen));

		rval = getgroups( gidsetlen, gidset );

		PUT_RVAL(xdr_syscall, rval);

		for( i = 0; i < rval; i++ ) {
			ASSERT(xdr_int(xdr_syscall, &gidset[i]));
		}
		END_RVAL(xdr_syscall, 0);
	}

	case CONDOR_getitimer:		/* getitimer( int which,
								**			R struct itimerval *value )
								*/
		RSC_1INT_1RBUF(getitimer,itimerval,xdr_itimerval);

	case CONDOR_getrlimit:		/* getrlimit( int resource,
								**			R struct rlimit *rlp )
								*/
		RSC_1INT_1RBUF(getrlimit,rlimit,xdr_rlimit);

	case CONDOR_getrusage:		/* getrusage( int who,
								**			R struct rusage *rusage )
								*/
		RSC_1INT_1RBUF(getrusage,rusage,xdr_rusage);

	case CONDOR_fstat:			/* fstat( int fd, R struct stat *buf )       */
		RSC_1INT_1RBUF(fstat,stat,xdr_stat);

#if !defined(ultrix)
	case CONDOR_fstatfs:		/* fstatfs( int fd, struct statfs *buf )     */
		RSC_1INT_1RBUF(fstatfs,statfs,xdr_statfs);
#endif !defined(ultrix)

	case CONDOR_utimes:			/* utimes( char *file,
								**			M struct timeval *tvp )
								*/
	{
		char File[ MAXPATHLEN ], *file = File;
		struct timeval tvp[2];

		ASSERT(xdr_string(xdr_syscall, &file, (u_int) MAXPATHLEN));
		ASSERT(timeval_xdr(xdr_syscall, &tvp[0]));
		ASSERT(timeval_xdr(xdr_syscall, &tvp[1]));

		rval = utimes( file, tvp );

		PUT_RVAL(xdr_syscall, rval);
		END_RVAL(xdr_syscall, 0);
	}

	case CONDOR_readlink:		/* readlink( char *path, char *buf,
								**					int bufsiz )
								*/
	{
		char Path[ MAXPATHLEN ], *path = Path;
		char *buf;
		int bufsiz;

		ASSERT(xdr_string(xdr_syscall, &path, (u_int) MAXPATHLEN));
		ASSERT(xdr_int(xdr_syscall, &bufsiz));

		MemToFree = buf = malloc( (unsigned)bufsiz );
		rval = readlink( path, buf, bufsiz );

		PUT_RVAL(xdr_syscall, rval);
		ASSERT(xdr_string(xdr_syscall, &buf, (u_int) bufsiz));
		END_RVAL(xdr_syscall, 0);
	}

	case CONDOR_adjtime:		/* adjtime( M struct timeval *delta,
								**		  R/0 struct timeval *olddelta )
								*/
	{
		struct timeval Delta, *delta = &Delta;
		struct timeval OldDelta, *olddelta = &OldDelta;

		ASSERT(timeval_xdr(xdr_syscall, delta));
		ASSERT(xdr_ptr(xdr_syscall, (caddr_t *) &olddelta, timeval_xdr));

#if !sequent
		rval = adjtime( delta, olddelta );
#else
		rval = -1;
		errno = ESYSNOTSUPP;
#endif !sequent

		PUT_RVAL(xdr_syscall, rval);

		ASSERT(xdr_ptr(xdr_syscall, (caddr_t *) &olddelta, timeval_xdr));

		END_RVAL(xdr_syscall, 0);
	}

#if !defined(ultrix)
	case CONDOR_statfs:			/* statfs( char *path, R struct statfs *buf )*/
		RSC_1STR_1RBUF(statfs,statfs,xdr_statfs);
#endif !defined(ultrix)

	case CONDOR_write:			/* write( int, M caddr_t, int )              */
	{
		int d, nbytes;
		char *buf = 0;

		ASSERT(xdr_int(xdr_syscall, &d));
		ASSERT(xdr_int(xdr_syscall, &nbytes));
		ASSERT(xdr_bytes(xdr_syscall, &buf, &nbytes, nbytes));

		rval = write( d, buf, nbytes );

		dprintf(D_SYSCALLS, "Shadow: write(%d, 0x%x, %d) = %d\n",
				d, buf, nbytes, rval);

		PUT_RVAL(xdr_syscall, rval);

		xdr_syscall->x_op = XDR_FREE;

		ASSERT(xdr_bytes(xdr_syscall, &buf, &nbytes, nbytes));

		END_RVAL(xdr_syscall, 0);
	}

	case CONDOR_read:			/* read( int, R caddr_t, int )               */
	{
		int d, nbytes;
		char *buf, *malloc();

		ASSERT(xdr_int(xdr_syscall, &d));
		ASSERT(xdr_int(xdr_syscall, &nbytes));

		MemToFree = buf = malloc( (unsigned)nbytes );
		if( buf == NULL ) {
			rval = -1;
		} else {
			rval = read(d, buf, nbytes);
		}

		dprintf(D_SYSCALLS, "Shadow: read(%d, buf, %d) = %d\n",
				d, nbytes, rval);

		PUT_RVAL(xdr_syscall, rval);

		ASSERT(xdr_bytes(xdr_syscall, &buf, &nbytes, nbytes));

		END_RVAL(xdr_syscall, 0);
	}

	case CONDOR_ioctl:			/* ioctl( int d, u_long request, char *argp )*/
	{
		int d;
		char *argp = 0;
		u_long request;

		ASSERT(xdr_int(xdr_syscall, &d));
		ASSERT(xdr_u_long(xdr_syscall, &request));

		if( request & IOC_IN ) {
			MemToFree = argp = malloc( (unsigned)(request & IOCPARM_MASK) );
			if( argp == NULL ) {
				EXCEPT("ioctl: tried to allocate %d bytes\n",
							request & IOCPARM_MASK);
			}

			ASSERT(xdr_opaque(xdr_syscall,argp,(u_int)(request&IOCPARM_MASK)));
		}

		/*
		**	rval = ioctl( d, request, argp );
		*/
		rval = CONDOR_NotSupported();

		dprintf(D_SYSCALLS, "Shadow: ioctl(%d, 0x%x, 0x%x) = %d\n",
				d, request, argp, rval);

		PUT_RVAL(xdr_syscall, rval);

		if( request & IOC_OUT ) {
			ASSERT(xdr_opaque(xdr_syscall,argp,(u_int)(request&IOCPARM_MASK)));
		}

		END_RVAL(xdr_syscall, 0);
	}


#ifdef IMPLEMENTED
	/*
	**	System calls with (as yet) unknown arguments
	*/
	case CONDOR_exportfs:
	case CONDOR_getdopt:
	case CONDOR_getfh:
	case CONDOR_madvise:
	case CONDOR_mincore:
	case CONDOR_mmap:
	case CONDOR_mprotect:
	case CONDOR_mremap:
	case CONDOR_munmap:
	case CONDOR_setdopt:
	case CONDOR_sstk:
		break;


	/*
	**	Other system calls
	*/

		break;


	case CONDOR_writev:			/* writev( int d, struct iovec *iov,
								**					int iovcnt )
								*/
	case CONDOR_readv:			/* readv( int d, struct iovec *iov,
								**					int iovcnt )
								*/

	case CONDOR_ptrace:			/* ptrace( int, int, VR int *, int           */
	case CONDOR_quotactl:		/* quotaactl( int, str, int, VRU caddr_t )   */
	case CONDOR_execv:			/* execv( str, M str[] )                     */
	case CONDOR_execve:			/* execve( str, M str[], M str[] )           */

	case CONDOR_send:			/* send( int, M caddr_t, int, int )          */
	case CONDOR_sendto:			/* sendto( int, M caddr_t, int, int,
										M sockaddr *, int                    */
	case CONDOR_sendmsg:		/* sendmsg( int, V msghdr[], int )           */

	case CONDOR_recv:			/* recv( int, R caddr_t, int, int )          */
	case CONDOR_recvfrom:		/* recvfrom( int, R caddr_t, int, int,
											0/R sockaddr *, VR int * )       */
	case CONDOR_recvmsg:		/* recvmsg( int, VR msghdr[], int )          */

	case CONDOR_setsockopt:		/* setsockopt(int, int, int, M caddr_t, int) */
	case CONDOR_getsockopt:		/* getsockopt( int, int, int,
								**		R caddr_t, VR int * )
								*/
	case CONDOR_socketpair:		/* socketpair( int, int, int, R int[2] )     */
	case CONDOR_bind:			/* bind( int, sockaddr *, int )              */
	case CONDOR_connect:		/* connect( int, sockaddr *, int )           */
	case CONDOR_mount:			/* mount( int, str, int,
								**		MU caddr_t )
								*/

	case CONDOR_accept:			/* accept( int, R sockaddr *, VR int * )     */
	case CONDOR_getsockname:	/* getsockname( int, R sockaddr *, VR int * )*/
	case CONDOR_getpeername:	/* getpeername( int, R sockaddr *, VR int * )*/
		break;
#endif IMPLEMENTED

	default:
		dprintf(D_ALWAYS, "Shadow: ERROR: %d is an unknown system call\n",
			condor_sysnum);

		errno = EINVAL;

		PUT_RVAL(xdr_syscall, -1);
		END_RVAL(xdr_syscall, 0);
	}
}

#ifdef notdef
debug( str )
char	*str;
{
	dprintf( D_ALWAYS, str );
}

/*
** Convert a timeval struct to a float.
*/
float
tv_to_f( tv )
struct timeval	tv;
{
	return (float)tv.tv_sec + ((float)tv.tv_usec)/(float)1000000;
}
#endif notdef


char MsgBuf[ BUFSIZ ];
HandleChildLog( log_fp )
FILE	*log_fp;
{
	register char *nlp;
	char *index();
	char buf[ BUFSIZ ];

	for(;;) {
		bzero(buf, sizeof(buf));

		if( fgets(buf, sizeof(buf), log_fp) == NULL ) {
			return -1;
		}

		nlp = index(buf, '\n');
		if( nlp != NULL ) {
			*nlp = '\0';
		}

		dprintf(D_ALWAYS|D_NOHEADER, "%s\n", buf );
		if( strncmp("ERROR",buf,5) == 0 ) {
			(void)strcpy( MsgBuf, buf );
		}

		/*
		**	If there is more to read, read it, otherwise just return...
		*/
		if( log_fp->_cnt == 0 ) {
			return 0;
		}
	}
}

HandleLog()
{
	register char *nlp;
	char buf[ BUFSIZ ];
	char *index();

	for(;;) {
		bzero(buf, sizeof(buf));

		if( fgets(buf, sizeof(buf), LogFP) == NULL ) {
			EXCEPT("Peer went away.  buf = '%s'", buf);
		}


		nlp = index(buf, '\n');
		if( nlp != NULL ) {
			*nlp = '\0';
		}

		dprintf(D_ALWAYS, "Read: %s\n", buf );

		/*
		**	If there is more to read, read it, otherwise just return...
		*/
		if( LogFP->_cnt == 0 ) {
			return;
		}
	}
}

RSC_no_args( func, condor_sysnum )
int (*func)();
int condor_sysnum;
{
	int rval = (*func)();

	dprintf(D_SYSCALLS, "Shadow: %s() = %d\n",
			CONDOR_SYSCALLNAME(condor_sysnum), rval);

	PUT_RVAL(xdr_syscall, rval);
	END_RVAL(xdr_syscall, 0);
}

RSC_1int(func, condor_sysnum)
int (*func)();
int condor_sysnum;
{
	int rval;
	int arg1;

	ASSERT(xdr_int(xdr_syscall, &arg1));

	rval = (*func)( arg1 );

	dprintf(D_SYSCALLS, "Shadow: %s(%d) = %d\n",
		CONDOR_SYSCALLNAME(condor_sysnum), arg1, rval);

	PUT_RVAL(xdr_syscall, rval);
	END_RVAL(xdr_syscall, 0);
}

RSC_2int(func, condor_sysnum)
int (*func)();
int condor_sysnum;
{
	int rval;
	int arg1, arg2;

	ASSERT(xdr_int(xdr_syscall, &arg1));
	ASSERT(xdr_int(xdr_syscall, &arg2));

	rval = (*func)( arg1, arg2 );

	dprintf(D_SYSCALLS, "Shadow: %s(%d, %d) = %d\n",
		CONDOR_SYSCALLNAME(condor_sysnum), arg1, arg2, rval);

	PUT_RVAL(xdr_syscall, rval);
	END_RVAL(xdr_syscall, 0);
}

RSC_3int(func, condor_sysnum)
int (*func)();
int condor_sysnum;
{
	int rval;
	int arg1, arg2, arg3;

	ASSERT(xdr_int(xdr_syscall, &arg1));
	ASSERT(xdr_int(xdr_syscall, &arg2));
	ASSERT(xdr_int(xdr_syscall, &arg3));

	rval = (*func)( arg1, arg2, arg3 );

	dprintf(D_SYSCALLS, "Shadow: %s(%d, %d, %d) = %d\n",
		CONDOR_SYSCALLNAME(condor_sysnum), arg1, arg2, arg3, rval);

	PUT_RVAL(xdr_syscall, rval);
	END_RVAL(xdr_syscall, 0);
}

RSC_1str(func, condor_sysnum)
int (*func)();
int condor_sysnum;
{
	char str1[ MAXPATHLEN ], *s1 = str1;
	int rval;

	ASSERT(xdr_string(xdr_syscall, &s1, (u_int) MAXPATHLEN));

	rval = (*func)(s1);

	dprintf(D_SYSCALLS, "Shadow: %s(%s) = %d\n",
		CONDOR_SYSCALLNAME(condor_sysnum), s1, rval );

	PUT_RVAL(xdr_syscall, rval);
	END_RVAL(xdr_syscall, 0);
}

RSC_1str_1int(func, condor_sysnum)
int (*func)();
int condor_sysnum;
{
	char str1[ MAXPATHLEN ], *s1 = str1;
	int int2;
	int rval;

	ASSERT(xdr_string(xdr_syscall, &s1, (u_int) MAXPATHLEN));
	ASSERT(xdr_int(xdr_syscall, &int2));

	rval = (*func)(s1, int2);

	dprintf(D_SYSCALLS, "Shadow: %s(%s, 0%o) = %d\n",
			CONDOR_SYSCALLNAME(condor_sysnum), s1, int2, rval );

	PUT_RVAL(xdr_syscall, rval);
	END_RVAL(xdr_syscall, 0);
}

RSC_1str_2int(func, condor_sysnum)
int (*func)();
int condor_sysnum;
{
	char str1[ MAXPATHLEN ], *s1 = str1;
	int int2, int3;
	int rval;

	ASSERT(xdr_string(xdr_syscall, (char **)&s1, (u_int) MAXPATHLEN));
	ASSERT(xdr_int(xdr_syscall, &int2));
	ASSERT(xdr_int(xdr_syscall, &int3));

	rval = (*func)( s1, int2, int3 );

	dprintf(D_SYSCALLS, "Shadow: %s(%s, 0%o, 0%o) = %d\n",
		CONDOR_SYSCALLNAME(condor_sysnum), s1, int2, int3, rval);

	PUT_RVAL(xdr_syscall, rval);
	END_RVAL(xdr_syscall, 0);
}

RSC_2str(func, condor_sysnum)
int (*func)();
int condor_sysnum;
{
	char str1[ MAXPATHLEN ], *s1 = str1;
	char str2[ MAXPATHLEN ], *s2 = str2;
	int rval;

	ASSERT(xdr_string(xdr_syscall, (char **)&s1, (u_int) MAXPATHLEN));
	ASSERT(xdr_string(xdr_syscall, (char **)&s2, (u_int) MAXPATHLEN));

	rval = (*func)( s1, s2 );

	dprintf(D_SYSCALLS, "Shadow: %s(%s, %s) = %d\n",
			CONDOR_SYSCALLNAME(condor_sysnum), s1, s2, rval );

	PUT_RVAL(xdr_syscall, rval);
	END_RVAL(xdr_syscall, 0);
}

/*
**	System call handler for syscalls which take 1 integer
**	and return a buffer of size bufsiz.
*/
RSC_1int_1Rbuf(func, condor_sysnum, bufsiz, xdrf)
int (*func)();
int condor_sysnum, bufsiz;
int (*xdrf)();
{
	int arg1;
	caddr_t buf;
	int rval;

	ASSERT(xdr_int(xdr_syscall, &arg1));

	MemToFree = buf = malloc( (unsigned)bufsiz );

	rval = (*func)( arg1, buf );

	dprintf(D_SYSCALLS, "Shadow: %s(%d, 0x%x) = %d\n",
			CONDOR_SYSCALLNAME(condor_sysnum), arg1, buf, rval);

	PUT_RVAL(xdr_syscall, rval);
	ASSERT((*xdrf)(xdr_syscall, buf));
	END_RVAL(xdr_syscall, 0);
}

/*
**	System call handler for syscalls which take 1 string
**	and return a buffer of size bufsiz.
*/
RSC_1str_1Rbuf(func, condor_sysnum, bufsiz, xdrf)
int (*func)();
int condor_sysnum, bufsiz;
int (*xdrf)();
{
	char str1[ MAXPATHLEN ], *s1 = str1;
	caddr_t buf;
	int rval;

	ASSERT(xdr_string(xdr_syscall, (char **)&s1, (u_int) MAXPATHLEN));

	MemToFree = buf = malloc( (unsigned) bufsiz );

	rval = (*func)( s1, buf );

	dprintf(D_SYSCALLS, "Shadow: %s(%s, 0x%x) = %d\n",
			CONDOR_SYSCALLNAME(condor_sysnum), s1, buf, rval);

	PUT_RVAL(xdr_syscall, rval);
	ASSERT((*xdrf)(xdr_syscall, buf));
	END_RVAL(xdr_syscall, 0);
}

/*
**	Make sure that they are not trying to
**	close a reserved descriptor.
*/
condor_close(d)
int d;
{
	if( (d >= MIN_RSRV_FD) && (d <= MAX_RSRV_FD) ||
					d == MyPipe || d == LockFd ) {
		errno = EINVAL;
		return( -1 );
	}

	return( close(d) );
}

/*
**	Make sure that they are not trying to
**	dup to a reserved descriptor.
*/
condor_dup2(oldd, newd)
int oldd, newd;
{
	if( (newd >= MIN_RSRV_FD) && (newd <= MAX_RSRV_FD) ||
						newd == MyPipe || newd == LockFd || newd == LogPipe ) {
		errno = EINVAL;
		return( -1 );
	}

	return( dup2(oldd, newd) );
}

/*
**	We don't really want the shadow to exit here because the 
**	starter may want to make a new checkpoint file.  So, just
**	respond with an error-free return code and hang out; the
**	shadow will (should) go away when (if) the starter closes
**	the LogSock.
*/
/* ARGSUSED */
condor_exit(status)
int status;
{
	return( 0 );
}

/*
**	getppid normally returns the parents id, right?  Well in
**	the CONDOR case, this job may be started and checkpointed many
**	times.  The parent will change each time.  Instead we will
**	treat it as though the parent process died and this process
**	was inherited by init.
*/
condor_getppid()
{
	return( 1 );
}

/*
**	getpid should return the same number each time.  As indicated
**	above, this CONDOR job may be started and checkpointed several times
**	during its lifetime, so we will make it a function of the CONDOR
**	process id...
*/
condor_getpid()
{
	extern PROC Proc;

	return( (Proc.id.cluster * 1000) + Proc.id.proc );
}

/*VARARGS*/
CONDOR_NotSupported()
{
#ifdef NOTDEF
	errno = ESYSNOTSUPP;
	return( -1 );
#endif NOTDEF

	errno = 0;
	PERM_ERR( "%s - system call not supported by Condor",
									CONDOR_SYSCALLNAME(condor_sysnum) );
	EXCEPT( "Unsupported system call %s (%d)",
						CONDOR_SYSCALLNAME(condor_sysnum), condor_sysnum );
#ifdef LINT
	return -1;
#endif LINT
}
