/* 
** Copyright 1986, 1987, 1988, 1989 University of Wisconsin
** 
** Permission to use, copy, modify, and distribute this software and its
** documentation for any purpose and without fee is hereby granted,
** provided that the above copyright notice appear in all copies and that
** both that copyright notice and this permission notice appear in
** supporting documentation, and that the name of the University of
** Wisconsin not be used in advertising or publicity pertaining to
** distribution of the software without specific, written prior
** permission.  The University of Wisconsin makes no representations about
** the suitability of this software for any purpose.  It is provided "as
** is" without express or implied warranty.
** 
** THE UNIVERSITY OF WISCONSIN DISCLAIMS ALL WARRANTIES WITH REGARD TO
** THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND
** FITNESS. IN NO EVENT SHALL THE UNIVERSITY OF WISCONSIN  BE LIABLE FOR
** ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
** WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
** ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
** OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
** 
** Authors:  Allan Bricker and Michael J. Litzkow,
** 	         University of Wisconsin, Computer Sciences Dept.
** 
*/ 


#include <stdio.h>
#include <sys/types.h>
#include <sys/param.h>
#include <sys/file.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <netinet/in.h>

#include "condor_sys.h"
#include "trace.h"
#include "ckpt.h"
#include "machdep.h"
#include "except.h"
#include "debug.h"

static char *_FileName_ = __FILE__;		/* Used by EXCEPT (see except.h)     */

#define DEBUG

#define STACKSIZ	(32*1024)

extern char	 TmpStack[ STACKSIZ ];
extern char	*TmpSP;
extern int		SavedSP;
extern int		SavedFP;

#ifdef MIPS
extern int		SavedRA;
extern int		SavedGP;
#endif MIPS

extern char	 CWD[ MAXPATHLEN ];
extern char	 CkptName[ MAXPATHLEN ];

extern RESTREC	 RestartInfo;

extern int CkptInProgress;
extern int SyscallInProgress;
extern int CkptWanted;

#ifdef mc68020
#define NREGS	16
extern long SavedRegs[ NREGS ];
#endif mc68020

#ifdef sparc
#define NREGS	8
extern long SavedRegs[ NREGS ];
#endif sparc

#if defined(MIPS) || defined(I386)
#include <setjmp.h>

jmp_buf LongJumpBuf;

#endif MIPS

ckpt()
{
	int scm = SetSyscalls( SYS_LOCAL | SYS_UNRECORDED );
	int pid = getpid();

	dprintf(D_CKPT,
		"CKPT: CkptWanted %d, CkptInProgress %d, SyscallInProgress %d\n",
			CkptWanted, CkptInProgress, SyscallInProgress);

	dprintf(D_CKPT, "ckpt: about to kill(%d, SIGTSTP)\n", pid);

	kill( pid, SIGTSTP );

	(void) SetSyscalls( scm );
}

#ifdef MIPS
void
#endif MIPS

CKPT()
{
	extern int CurrentSysCall;

	dprintf(D_CKPT,
		"CKPT: CkptWanted %d, CkptInProgress %d, SyscallInProgress %d\n",
			CkptWanted, CkptInProgress, SyscallInProgress);

	if( CkptInProgress || SyscallInProgress ) {
		if( SyscallInProgress ) {
			dprintf(D_ALWAYS, "CKPT: [%s] in progress, checkpoint wanted\n",
					CONDOR_SYSCALLNAME(CurrentSysCall));
		}
		CkptWanted = 1;
		return;
	}

	do_CKPT();

	Break();
}

do_CKPT()
{
	/*
	**	Should not put local variable here.  They will disappear
	**	when the stack pointer is moved.
	*/

	CkptInProgress = 1;
	CkptWanted = 0;

	/*
	**	Save the current offset of all open files.
	*/
	SetFileInfo();

#if defined(MIPS) || defined(I386)
{
	int rval = setjmp(LongJumpBuf);

	Break();

	if( rval != 0 ) {
		dprintf(D_ALWAYS, "do_CKPT: setjmp returned %d\n", rval);
		Break();
		return;
	} else {
		dprintf(D_ALWAYS, "do_CKPT: setjmp returned zero\n");
		Break();
	}
}
#else MIPS

	/*
	**	Save registers on the real stack
	**	and set up a temporary stack.
	*/
	SAVE_STATE();

	dprintf(D_ALWAYS,
		"do_CKPT: after SAVE_STATE: GP 0x%x (%d), SP 0x%x (%d), RA 0x%x (%d)\n",
			SavedGP, SavedGP, SavedSP, SavedSP, SavedRA, SavedRA);
#endif MIPS

#ifdef NEED_ASMS_IN_CKPT
#ifdef vax
	/*
	;asm("pushr	$0xffff");
	*/
	;asm("pushl fp");
	;asm("movl	sp,_SavedSP");
#endif vax

#ifdef mc68020
	/*
	;asm("moveml #0xffff,_SavedRegs");
	*/
	;asm("movl	a6,sp@-")
	;asm("movl	sp,_SavedSP");
#endif mc68020

#ifdef i386
	/*
	;asm("pushal");
	*/
	;asm("pushl     %ebp");
	;asm("movl %esp,_SavedSP");
#endif i386

#ifdef sparc
#ifdef NOTDEF
#define SAVE	;asm("save	%sp,-(16*4),%sp");
#define SAVE8	SAVE; SAVE; SAVE; SAVE; SAVE; SAVE; SAVE; SAVE;
#define SAVE_GREG(name,pos) \
	;asm("sethi	%hi(_SavedRegs+(4*pos)),%o0"); \
	;asm("st	%name,[%o0+%lo(_SavedRegs+(4*pos))]");

	SAVE_GREG(g0,0); SAVE_GREG(g1,1); SAVE_GREG(g2,2); SAVE_GREG(g3,3);
	SAVE_GREG(g4,4); SAVE_GREG(g5,5); SAVE_GREG(g6,6); SAVE_GREG(g7,7);

	/*
	**	Perform 32 (the max allowed by any sparc implementation)
	**	register window saves
	*/
	SAVE8; SAVE8; SAVE8; SAVE8;

	;asm("sethi	%hi(_SavedSP),%o0");
	;asm("st	%sp,[%o0+%lo(_SavedSP)]");

	;asm("sethi	%hi(_SavedFP),%o0");
	;asm("st	%fp,[%o0+%lo(_SavedFP)]");

#undef SAVE
#undef SAVE8
#undef SAVE_GREG
#else NOTDEF
	;asm("sethi	%hi(_SavedSP),%o0");
	;asm("st	%sp,[%o0+%lo(_SavedSP)]");

	;asm("sethi	%hi(_SavedFP),%o0");
	;asm("st	%fp,[%o0+%lo(_SavedFP)]");
#endif NOTDEF
#endif sparc
#endif NEED_ASMS_IN_CKPT

	dump_core();
}

static int	CkptSCM;
dump_core()
{
	CkptSCM = SetSyscalls( SYS_LOCAL | SYS_UNRECORDED );

#ifdef notdef
	write_stack("stack", 0x7fffd000, 8192);
#endif notdef

	kill( getpid(), SIGQUIT );
}

#ifdef notdef
write_stack( file, addr, len )
char *file;
char *addr;
int len;
{
	int fd = creat(file, 0666);

	if(fd < 0 ) {
		perror("stack");
		dprintf(D_ALWAYS, "write_stack(0x%x, %d): cannot creat stack\n",
			addr, len);
		return;
	}

	if( write(fd, addr, len) != len ) {
		perror("write_stack");
	}

	close( fd );
}
#endif notdef

struct sigvec	SigVec;
int				SigMask;

restart()
{
	/*
	**	Set up a temporary stack
	*/
	TmpSP = &TmpStack[ STACKSIZ ];

#ifdef notdef
#ifdef MIPS
	GrowStack();
#endif MIPS
#endif notdef

	TMP_STACK();

#ifdef NEED_ASMS_IN_CKPT
#ifdef vax
	;asm("movl	_TmpSP,sp");
#endif vax

#ifdef mc68020
	;asm("movl	_TmpSP,sp");
#endif mc68020

#ifdef i386
	;asm("movl	_TmpSP,%esp");
#endif i386

#ifdef sparc
	;asm("sethi	%hi(_TmpSP),%o0");
	;asm("ld	[%o0+%lo(_TmpSP)],%sp");
#endif sparc
#endif NEED_ASMS_IN_CKPT

	restart2();
}

restart2()
{
	dprintf(D_CKPT,
		"restart2: CkptWanted %d,CkptInProgress %d, SyscallInProgress %d\n",
			CkptWanted, CkptInProgress, SyscallInProgress);

	/*
	**	Open the checkpoint file to extract any information
	**	that we need (eg. the saved stack).
	*/
	ReadRestartInfo();

	Break();

	/*
	**	Reopen and reposition all active files.  The open file table
	**	does not need to be read from the checkpoint file because it
	**	resides in memory.  The checkpoint version is there for
	**	initialization and for other programs to examine.
	*/
#ifdef CONDOR
	(void) SetSyscalls( SYS_REMOTE | SYS_UNRECORDED );
#else CONDOR
	(void) SetSyscalls( SYS_LOCAL | SYS_UNRECORDED );
#endif CONDOR

	dprintf( D_CKPT, "CkptSCM = 0x%x\n", CkptSCM );

	ReopenFiles();

	/*
	**	Change to the previous current working directory
	*/
	if( chdir(CWD) < 0 ) {
		EXCEPT(CWD);
	}

	SigVec.sv_handler = CKPT;
	SigVec.sv_mask    = 0;
	SigVec.sv_onstack = 0;

	(void) SetSyscalls( SYS_LOCAL | SYS_UNRECORDED );

	if( sigvec(SIGTSTP, &SigVec, (struct sigvec *) 0) < 0 ) {
		EXCEPT("sigvec(SIGTSTP, CKPT), errno = %d", errno );
	} else {
		/*
		EXCEPT( "Set SIGTSTP to CKPT()" );
		*/
	}

#ifdef NOTDEF
	/*
	**	Unblock SIGTSTP while leaving everything else alone.
	*/
	SigMask = sigblock( 0 );
	if( SigMask < 0 ) {
		EXCEPT( "sigblock(0), errno = %d", errno );
	}

	SigMask &= ~sigmask( SIGTSTP );
	if( sigsetmask(SigMask) < 0 ) {
		EXCEPT( "sigsetmask(0x%x), errno = %d",
				SigMask, errno );
	} else {
		/*
		EXCEPT( "Unblocked SIGTSTP" );
		*/
	}
	
#else
	sigsetmask( 0 );
	(void) SetSyscalls( CkptSCM );
#endif

#if defined(MIPS) || defined(I386)
	dprintf(D_ALWAYS, "restart2: about to longjmp!\n");
	CkptInProgress = 0;
	longjmp( LongJumpBuf, 1 );
	dprintf(D_ALWAYS, "restart2: what happened to longjmp????\n");
#else MIPS

	dprintf(D_ALWAYS,
	"do_CKPT: before RESTORE_STATE: GP 0x%x (%d), SP 0x%x (%d), RA 0x%x (%d)\n",
			SavedGP, SavedGP, SavedSP, SavedSP, SavedRA, SavedRA);

	/*
	**	Restore the stack and saved registers
	*/
	RESTORE_STATE();

	Break();

	dprintf(D_ALWAYS,
	"do_CKPT: after RESTORE_STATE: GP 0x%x (%d), SP 0x%x (%d), RA 0x%x (%d)\n",
			SavedGP, SavedGP, SavedSP, SavedSP, SavedRA, SavedRA);
#ifdef NEED_ASMS_IN_CKPT
#ifdef vax
	;asm("movl	_SavedSP,sp");
	;asm("movl	(sp)+,fp");
	/*
	;asm("popr	$0xffff");
	*/
#endif vax

#ifdef mc68020
	;asm("movl	_SavedSP,sp");
	;asm("movl	sp@+,a6");
	/*
	;asm("moveml _SavedRegs,#0xffff");
	*/
#endif mc68020

#ifdef i386
	;asm("movl	_SavedSP,%esp");
	;asm("popl      %ebp");
	/*
	;asm("popal");
	*/
#endif i386

#ifdef sparc
#ifdef NOTDEF
#define RESTORE	;asm("restore	%sp,(16*4),%sp");
#define RESTORE8 \
	RESTORE; RESTORE; RESTORE; RESTORE; \
	RESTORE; RESTORE; RESTORE; RESTORE;

#define REST_GREG(name,pos) \
	;asm("sethi	%hi(_SavedRegs+(4*pos)),%o0"); \
	;asm("ld	[%o0+%lo(_SavedRegs+(4*pos))],%name");

	REST_GREG(g0,0); REST_GREG(g1,1); REST_GREG(g2,2); REST_GREG(g3,3);
	REST_GREG(g4,4); REST_GREG(g5,5); REST_GREG(g6,6); REST_GREG(g7,7);

	;asm("sethi	%hi(_SavedSP),%o0");
	;asm("ld	[%o0+%lo(_SavedSP)],%sp");

	;asm("sethi	%hi(_SavedFP),%o0");
	;asm("ld	[%o0+%lo(_SavedFP)],%fp");

	/*
	**	Perform 32 (the max allowed by any sparc implementation)
	**	register window restores
	*/
	RESTORE8; RESTORE8; RESTORE8; RESTORE8;

#undef RESTORE
#undef RESTORE8
#undef REST_GREG
#else NOTDEF
	;asm("sethi	%hi(_SavedSP),%o0");
	;asm("ld	[%o0+%lo(_SavedSP)],%sp");

	;asm("sethi	%hi(_SavedFP),%o0");
	;asm("ld	[%o0+%lo(_SavedFP)],%fp");
#endif NOTDEF
#endif sparc
#endif NEED_ASMS_IN_CKPT

	CkptInProgress = 0;
#endif MIPS
}

Break()
{
}

SetFileInfo()
{
	register FINFO *fi;
	register int i;
	register int flags;
	register int scm;

#ifdef CONDOR
	scm = SetSyscalls( SYS_REMOTE | SYS_UNRECORDED );
#else CONDOR
	scm = SetSyscalls( SYS_LOCAL | SYS_UNRECORDED );
#endif CONDOR

	for( i = 0; i < NOFILE; i++ ) {
		fi = &RestartInfo.rr_file[i];
		flags = fi->fi_flags;

		if( ((flags & FI_OPEN) == FI_OPEN) && ((flags & FI_DUP) == 0) ) {
			fi->fi_pos = lseek(fi->fi_fdno,0,L_INCR);
		}
	}

	(void) SetSyscalls( scm );
}

ReadRestartInfo()
{
	register int fd;
	int scm;	/* System call mode (local or remote */
	int ssize;
#ifdef CM_JOB
	CKPTMAP ckmap;
#endif CM_JOB

	scm = SetSyscalls( SYS_LOCAL | SYS_UNRECORDED );

	fd = open(CkptName, O_RDONLY, 0);
	if( fd < 0 ) {
		EXCEPT( "Could not open '%s', errno = %d",
				CkptName, errno );
	}

#ifdef CM_JOB
	ReadCkptMap( fd, &ckmap );
	ReadStack( fd, &ckmap.cm_segs[CM_STACK] );
#else CM_JOB
	ssize = ReadStackSize( fd );
	ReadStack( fd, ssize );
#endif CM_JOB

	close( fd );

	(void) SetSyscalls( scm );
}

#ifdef CM_JOB
ReadCkptMap( fd, ckmap )
register int fd;
register CKPTMAP *ckmap;
{
	register int i;

	if( lseek(fd, -sizeof(CKPTMAP), L_XTND) < 0 ) {
		EXCEPT("Could not lseek to CKPTMAP structure");
	}

	if( read(fd, (char *) ckmap, sizeof(CKPTMAP)) != sizeof(CKPTMAP) ) {
		EXCEPT("Could not read Checkpoint Map");
	}

	/*
	**	For each segment, convert network order to host order.
	*/
	for( i = 0; i < NCKPTSEGS; i++ ) {
		ckmap->cm_segs[i].cs_pos = ntohl(ckmap->cm_segs[i].cs_pos);
		ckmap->cm_segs[i].cs_len = ntohl(ckmap->cm_segs[i].cs_len);

#ifdef notdef
		dprintf(D_CKPT, "ReadCkptMap: segment %d: pos %d, len %d\n",
				i, ckmap->cm_segs[i].cs_pos, ckmap->cm_segs[i].cs_len );
#endif notdef
	}
}

ReadStack(fd, cseg)
register int fd;
register CKPTSEG *cseg;
{
	register char *TStack = (char *) USRSTACK;

	if( lseek(fd, cseg->cs_pos, L_SET) < 0 ) {
		EXCEPT( "lseek(%d,%d,L_SET)", fd, cseg->cs_pos );
		goto error;
	}


	TStack -= cseg->cs_len;	/* Set TStack pointing at the base of the stack  */

	dprintf(D_CKPT, "ReadStack: USRSTACK = 0x%x, TStack = 0x%x\n",
			USRSTACK, TStack );

	/*
	**	The kernel (probably) won't grow the user stack in the middle
	**	of a system call, so lets grow it now...
	**	Make sure to account for any extra stack the system may give us...
	*/

	*(int *) (TStack+STACKGROW) = 0;

	dprintf(D_CKPT, "ReadStack: Just set *(TStack 0x%x + STACKGROW 0x%x)\n",
					TStack, STACKGROW);

	dprintf(D_CKPT, "ReadStack: About to read(%d, 0x%x, %d)\n",
					fd, TStack, cseg->cs_len);

	if( read(fd, TStack, cseg->cs_len) != cseg->cs_len ) {
		goto error;
	}

	dprintf(D_CKPT, "ReadStack: Just read %d bytes of stack\n",
					cseg->cs_len);

	return;

error:
	EXCEPT("Could not read Checkpoint Stack, errno = %d",
				errno );
}
#else CM_JOB
ReadStackSize( fd )
register int fd;
{
	long ssize;
	int	 where;

	if( (where=lseek(fd, -sizeof(ssize), L_XTND)) < 0 ) {
		EXCEPT("Could not lseek to stack size value");
	}
	fprintf( stderr, "Seeked to %d\n", where );

	if( read(fd, (char *) &ssize, sizeof(ssize)) != sizeof(ssize) ) {
		EXCEPT("Could not read stack size");
	}
	fprintf( stderr, "Read %d bytes\n", sizeof(ssize) );

	fprintf(stderr, "ReadStackSize: ssize = %d ntohl(ssize) = %d\n",
		ssize, ntohl(ssize) );

	return( ntohl(ssize) );
}

ReadStack(fd, ssize)
register int fd;
register long ssize;
{
	register char *TStack = (char *) USRSTACK;

	if( lseek(fd, -(ssize+sizeof(ssize)), L_XTND) < 0 ) {
		EXCEPT( "lseek(%d,%d,L_XTND)", fd, -(ssize+sizeof(ssize)) );
		goto error;
	}

	TStack -= ssize;	/* Set TStack pointing at the base of the stack  */

	dprintf(D_CKPT, "ReadStack: USRSTACK = 0x%x, TStack = 0x%x\n",
			USRSTACK, TStack );

	/*
	**	The kernel (probably) won't grow the user stack in the middle
	**	of a system call, so lets grow it now...
	**	Make sure to account for any extra stack the system may give us...
	*/

	*(int *) (TStack+STACKGROW) = 0;

	dprintf(D_CKPT, "ReadStack: Just set *(TStack 0x%x + STACKGROW 0x%x)\n",
					TStack, STACKGROW);

	dprintf(D_CKPT, "ReadStack: About to read(%d, 0x%x, %d)\n",
					fd, TStack, ssize);

	if( read(fd, TStack, ssize) != ssize ) {
		goto error;
	}

	dprintf(D_CKPT, "ReadStack: Just read %d bytes of stack\n", ssize);

	return;

error:
	EXCEPT("Could not read Checkpoint Stack, errno = %d",
				errno );
}

#endif CM_JOB

ReopenFiles()
{
	register FINFO *fi;
	register int i, fd;
	register int flags;

	for( i = 0; i < NOFILE; i++ ) {
		fi = &RestartInfo.rr_file[i];
		flags = fi->fi_flags;

		if( (flags & FI_PREOPEN) == FI_PREOPEN ) {
			if( lseek(i, fi->fi_pos, L_SET) < 0 ) {
				EXCEPT("re-lseek(%d, %d, L_SET)", i, fi->fi_pos);
			}
			continue;
		}

		if( (flags & FI_OPEN) == FI_OPEN ) {
			if( (flags & FI_DUP) == FI_DUP ) {
				if( dup2(fi->fi_fdno, i) < 0 ) {
					EXCEPT("dup2");
				}
			} else {
				fd = open(fi->fi_path, fi->fi_priv, 0);
				if( fd < 0 ) {
					EXCEPT("open(%s, 0x%x, 0)", fi->fi_path, fi->fi_priv);
				}

				if( fd != i ) {
					if( dup2(fd, i) < 0 ) {
						EXCEPT("dup2 (again)");
					}
					if( close(fd) < 0 ) {
						EXCEPT("close");
					}
				}

				if( lseek(i, fi->fi_pos, L_SET) < 0 ) {
					EXCEPT("re-lseek");
				}
			}
		}
	}
}
