/* 
** Copyright 1986, 1987, 1988, 1989 University of Wisconsin
** 
** Permission to use, copy, modify, and distribute this software and its
** documentation for any purpose and without fee is hereby granted,
** provided that the above copyright notice appear in all copies and that
** both that copyright notice and this permission notice appear in
** supporting documentation, and that the name of the University of
** Wisconsin not be used in advertising or publicity pertaining to
** distribution of the software without specific, written prior
** permission.  The University of Wisconsin makes no representations about
** the suitability of this software for any purpose.  It is provided "as
** is" without express or implied warranty.
** 
** THE UNIVERSITY OF WISCONSIN DISCLAIMS ALL WARRANTIES WITH REGARD TO
** THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND
** FITNESS. IN NO EVENT SHALL THE UNIVERSITY OF WISCONSIN  BE LIABLE FOR
** ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
** WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
** ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
** OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
** 
** Authors:  Allan Bricker and Michael J. Litzkow,
** 	         University of Wisconsin, Computer Sciences Dept.
** 
*/ 


#include <stdio.h>
#include <utmp.h>
#include <nlist.h>
#include <signal.h>
#include <setjmp.h>
#include <netdb.h>
#include <pwd.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/param.h>
#include <sys/socket.h>

#ifdef ultrix
#include <sys/mount.h>
#else ultrix
#include <sys/vfs.h>
#endif ultrix

#include <sys/file.h>
#include <netinet/in.h>
#include <rpc/types.h>
#include <rpc/xdr.h>

#ifdef sequent
#include <sys/tmp_ctl.h>
#include <sys/vm.h>
#endif sequent

#ifdef sparc
#include <sys/param.h>
#endif sparc

#ifdef mc68020
#include <sys/param.h>
#endif mc68020

#ifdef mips
#include <sys/fixpoint.h>
#endif mips

#include "sched.h"
#include "debug.h"
#include "trace.h"
#include "except.h"
#include "expr.h"
#include "clib.h"

static char *_FileName_ = __FILE__;		/* Used by EXCEPT (see except.h)     */

CONTEXT		*create_context();
bool_t 		xdr_int();
XDR			*xdr_Udp_Init(), *xdr_Init();
ELEM		*create_elem();
EXPR		*create_expr(), *build_expr();;
float		calc_load_avg();
char		*strdup();

extern CONTEXT	*MachineContext;
extern CONTEXT	*JobContext;
extern int		StarterPid;
extern int		LastTimeout;
extern char		*MgrHost;
extern char		*CondorAdministrator;
extern int		UdpSock;
extern int		Memory;
extern char		*Spool;
extern char		*Starter;
extern int		State;
extern int		PollsPerUpdate;

char			*ClientMachine;

int		Kmem;
int		Last_X_Event;

struct nlist nl[] = {
    { "_avenrun" },
#define X_AVENRUN 0
    { "_avenrun_scale" },		/* Really only ibm032, but it doesn't matter */
#define X_AVENRUN_SCALE 1
    { "" },
};

job_dies()
{
	StarterPid = 0;
	if( JobContext ) {
		free_context( JobContext );
		JobContext = NULL;
	}
	cleanup_execute_dir();
	free( ClientMachine );
	ClientMachine = NULL;
	change_states( NO_JOB );
	timeout();
}

timeout()
{
	static int		Polls = 0;

	evaluate_situation();
	if( Polls == 0 ) {
		(void)update_central_mgr();
	}
	if( ++Polls >= PollsPerUpdate ) {
		Polls = 0;
	}
	LastTimeout = (int)time( (time_t *)0 );
}


#define ABORT \
	if( job_context ) {				\
		free_context( job_context );\
	}								\
	if( ClientMachine ) {			\
		free( ClientMachine );		\
		ClientMachine = NULL;		\
	}								\
	return;

start_job_request( xdrs, from )
XDR		*xdrs;
struct sockaddr_in	*from;
{
	int		start, job_reqs;
	CONTEXT	*job_context = NULL;
	struct hostent  *hp, *gethostbyaddr();

    if( (hp=gethostbyaddr((char *)&from->sin_addr,
					sizeof(struct in_addr), from->sin_family)) == NULL ) {
        EXCEPT(  "Can't find host name" );
    }
	ClientMachine = strdup( hp->h_name );


	dprintf( D_ALWAYS, "Got start_job_request from %s\n", ClientMachine );

		/* Read in job facts and booleans */
	job_context = create_context();
	if( !xdr_context(xdrs,job_context) ) {
		dprintf( D_ALWAYS, "Can't receive context from shadow\n" );
		return;
	}
	if( !xdrrec_skiprecord(xdrs) ) {
		dprintf( D_ALWAYS, "Can't receive context from shadow\n" );
		return;
	}

	dprintf( D_JOB, "JOB_CONTEXT:\n" );
	if( DebugFlags & D_JOB ) {
		display_context( job_context );
	}
	dprintf( D_JOB, "\n" );

	dprintf( D_MACHINE, "MACHINE_CONTEXT:\n" );
	if( DebugFlags & D_MACHINE ) {
		display_context( MachineContext );
	}
	dprintf( D_MACHINE, "\n" );

	if( State != NO_JOB ) {
		dprintf( D_ALWAYS, "State != NO_JOB\n" );
		(void)reply( xdrs, NOT_OK );
		ABORT;
	}

		/* See if machine and job meet each other's requirements, if so
		   start the job and tell shadow, otherwise refuse and clean up */
	if( evaluate_bool("START",&start,MachineContext,job_context) < 0 ) {
		start = 0;
	}
	if( evaluate_bool("JOB_REQUIREMENTS",&job_reqs,MachineContext,job_context)
																		< 0 ) {
		job_reqs = 0;
	}

	if( !start || !job_reqs ) {
		dprintf( D_ALWAYS, "start = %d, job_reqs = %d\n", start, job_reqs );
		(void)reply( xdrs, NOT_OK );
		ABORT;
	}

	if( !reply(xdrs,OK) ) {
		ABORT;
	}
	JobContext = job_context;
	if( start_job(xdrs) < 0 ) {
		ABORT;
	}
	change_states( JOB_RUNNING );
	timeout();
}
#undef ABORT

x_event( xdrs, from )
XDR		*xdrs;
struct sockaddr_in	*from;
{
	Last_X_Event = time( (time_t *)0 );
	timeout();
}


evaluate_situation()
{
	int		tmp;

	evaluate_load();

	if( StarterPid && !JobContext ) {
		EXCEPT( "evaluate_situation, StarterPid = %d, but JobContext is NULL\n",
					StarterPid );
	}

	if( !JobContext ) {
		return;
	}

	if( State == CHECKPOINTING ) {
		if( evaluate_bool("KILL",&tmp,MachineContext,JobContext) < 0 ) {
			EXCEPT( "Can't evaluate machine boolean \"KILL\"\n" );
		}
		if( tmp ) {
			kill_job( NOT_OK );
			return;
		}
	}

	if( State == SUSPENDED ) {
		if( evaluate_bool("VACATE",&tmp,MachineContext,JobContext) < 0 ) {
			EXCEPT( "Can't evaluate machine boolean \"VACATE\"\n" );
		}
		if( tmp ) {
			vacate_order();
			return;
		}
	}

	if( State == JOB_RUNNING ) {
		if( evaluate_bool("SUSPEND",&tmp,MachineContext,JobContext) < 0 ) {
			EXCEPT( "Can't evaluate machine boolean \"SUSPEND\"\n" );
		}
		if( tmp ) {
			suspend_order();
			return;
		}
	}

	if( State == SUSPENDED ) {
		if( evaluate_bool("CONTINUE",&tmp,MachineContext,JobContext) < 0 ) {
			EXCEPT( "Can't evaluate machine boolean \"CONTINUE\"\n" );
		}
		if( tmp ) {
			resume_order();
			return;
		}
	}
}

update_central_mgr()
{
	int		cmd;
	XDR		xdr, *xdrs;

	xdrs = xdr_Udp_Init( &UdpSock, &xdr );
	xdrs->x_op = XDR_ENCODE;

		/* Send the command */
	cmd = STARTD_INFO;
	if( !xdr_int(xdrs, &cmd) ) {
		xdr_destroy( xdrs );
		return -1;
	}

	if( !xdr_context(xdrs,MachineContext) ) {
		xdr_destroy( xdrs );
		return -1;
	}

	if( !xdrrec_endofrecord(xdrs,TRUE) ) {
		xdr_destroy( xdrs );
		return -1;
	}

	xdr_destroy( xdrs );
	return 0;
}

change_states( new_state )
int	new_state;
{
	ELEM	tmp;
	char	*name;

	switch( new_state ) {
		case NO_JOB:
			name = "NoJob";
			break;
		case JOB_RUNNING:
			name = "Running";
			break;
		case KILLED:
			name = "Killed";
			break;
		case CHECKPOINTING:
			name = "Checkpointing";
			break;
		case SUSPENDED:
			name = "Suspended";
			break;
		default:
			EXCEPT( "Change states, unknown state (%d)", new_state );
	}

	State = new_state;

	tmp.type = STRING;
	tmp.s_val = name;
	store_stmt( build_expr("State",&tmp), MachineContext );

	tmp.type = INT;
	tmp.i_val = (int)time( (time_t *)0 );
	store_stmt( build_expr("EnteredCurrentState",&tmp), MachineContext );
}

kill_job( status )
int		status;
{
	if( status == OK ) {
		dprintf( D_ALWAYS, "Called kill_job( OK )\n" );
	} else {
		dprintf( D_ALWAYS, "Called kill_job( NOT_OK )\n" );
	}
	if( !StarterPid ) {
		dprintf( D_ALWAYS, "Called kill_job, but no starter process active\n" );
		return;
	}

	if( status != OK ) {
		mail_stuck_job_message();
	}

		/* Could be the starter will die naturally just before we try to
		   send the sig, so have to ignore errors here... */
	if( status == OK ) {
		(void) kill_starter( StarterPid, SIGINT );
	} else {
		(void) kill_starter( StarterPid, SIGQUIT );
	}
	change_states( KILLED );
}

mail_stuck_job_message()
{
	char	cmd[512];
	char	hostname[512];
	FILE	*mailer, *popen();

	dprintf( D_ALWAYS, "Mailing administrator (%s)\n", CondorAdministrator );

	if( gethostname(hostname,sizeof(hostname)) < 0 ) {
		EXCEPT( "gethostname(0x%x,%d)", hostname, sizeof(hostname) );
	}

	(void)sprintf( cmd, "/bin/mail %s", CondorAdministrator );
	if( (mailer=popen(cmd,"w")) == NULL ) {
		EXCEPT( "popen(\"%s\",\"w\")", cmd );
	}

	fprintf( mailer, "To: %s\n", CondorAdministrator );
	fprintf( mailer, "Subject: Condor Problem\n" );
	fprintf( mailer, "\n" );

	fprintf( mailer, "The startd on \"%s\" is killing a Condor job ",
				hostname );
	fprintf( mailer, "from \"%s\"\n", ClientMachine );
	fprintf( mailer, "because it didn't clean up after being sent a TSTP.\n"  );
	fprintf( mailer, "\n" );
	mail_context( "JobContext:", JobContext, mailer );
	fprintf( mailer, "\n" );
	mail_context( "MachineContext:", MachineContext, mailer );
	fprintf( mailer, "\n" );

	fprintf( mailer, "CurrentTime = %d\n", (int)time((time_t *)0) );

		/* don't use 'pclose()' here, it does its own wait, and messes
	       with our handling of SIGCHLD! */
	(void)fclose( mailer );
}

vacate_order()
{
	dprintf( D_ALWAYS, "Called vacate_order()\n" );

	dprintf( D_MACHINE, "MACHINE_CONTEXT:\n" );
	if( DebugFlags & D_MACHINE ) {
		display_context( MachineContext );
	}
	dprintf( D_MACHINE, "\n" );

	if( !StarterPid ) {
		dprintf( D_ALWAYS,
					"Called vacate_order, but no starter process active\n" );
	} else {
			/* Could be the starter will die naturally just before we try to
			   send the sig, so have to ignore errors here... */
		(void) kill_starter( StarterPid, SIGTSTP );
		change_states( CHECKPOINTING );
	}
}

suspend_order()
{
	dprintf( D_ALWAYS, "Called suspend_order()\n" );

	dprintf( D_MACHINE, "MACHINE_CONTEXT:\n" );
	if( DebugFlags & D_MACHINE ) {
		display_context( MachineContext );
	}
	dprintf( D_MACHINE, "\n" );

	if( !StarterPid ) {
		dprintf( D_ALWAYS,
					"Called suspend_order, but no starter process active\n" );
	} else {
			/* Could be the starter will die naturally just before we try to
			   send the sig, so have to ignore errors here... */
		(void) kill_starter( StarterPid, SIGUSR1 );
		change_states( SUSPENDED );
	}
}

resume_order()
{
	dprintf( D_ALWAYS, "Called resume_order()\n" );

	dprintf( D_MACHINE, "MACHINE_CONTEXT:\n" );
	if( DebugFlags & D_MACHINE ) {
		display_context( MachineContext );
	}
	dprintf( D_MACHINE, "\n" );

	if( !StarterPid ) {
		dprintf( D_ALWAYS,
					"Called resume_order, but no starter process active\n" );
	} else {
			/* Could be the starter will die naturally just before we try to
			   send the sig, so have to ignore errors here... */
		(void) kill_starter( StarterPid, SIGCONT );
		change_states( JOB_RUNNING );
	}
}


reply( xdrs, answer )
XDR	*xdrs;
int	answer;
{
	xdrs->x_op = XDR_ENCODE;
	if( !xdr_int(xdrs,&answer) ) {
		return FALSE;
	}
	if( !xdrrec_endofrecord(xdrs,TRUE) ) {
		return FALSE;
	}
	dprintf( D_ALWAYS, "Replied %s\n", answer ? "ACCEPTED" : "REFUSED" );
	return TRUE;
}

evaluate_load()
{
    ELEM    tmp;

    tmp.type = INT;
    tmp.i_val = calc_memory();
    store_stmt( build_expr("Memory",&tmp), MachineContext );

    tmp.i_val = calc_disk();
    store_stmt( build_expr("Disk",&tmp), MachineContext );

    tmp.i_val = user_idle_time();
    store_stmt( build_expr("KeyboardIdle",&tmp), MachineContext );

    tmp.i_val = calc_Ncpus();
    store_stmt( build_expr("Cpus",&tmp), MachineContext );

	tmp.type = FLOAT;
    tmp.f_val = calc_load_avg();
    store_stmt( build_expr("LoadAvg",&tmp), MachineContext );

	/*
	** display_context( MachineContext );
	*/
}

/*
** Calculate the main memory on our machine in megabytes.  For now we'll
** just look it up in the config file, someday we should probe the kernel
** for this information.
*/
calc_memory()
{
    return Memory;
}

/*
** Calculate the free disk in the filesystem which we use for hosting
** foreign executables in kilobytes.  We don't give the real answer,
** instead we allow a little "slop".
*/
#ifdef ultrix

#define FS_SLOP 512     /* Kilobytes */
calc_disk()
{
    struct fs_data buf;

    if( statfs(Spool,&buf) < 0 ) {
        EXCEPT( "Can't do statfs on \"%s\"\n", Spool );
    }

    return( buf.fd_bfreen - FS_SLOP );
}

#else ultrix

#define FS_SLOP 512     /* Kilobytes */
calc_disk()
{
    struct statfs buf;
    int     kbytes;

    if( statfs(Spool,&buf) < 0 ) {
        EXCEPT( "Can't do statfs on \"%s\"\n", Spool );
    }

    kbytes = (buf.f_bavail * buf.f_bsize) / 1024;
    return kbytes - FS_SLOP;
}

#endif ultrix

#define MAXINT ((1<<31)-1)

#ifndef MIN
#define MIN(a,b) ((a)<(b)?(a):(b))
#endif

user_idle_time()
{
    FILE    *fp;
    int     tty_idle;
    int     user_idle    = MAXINT;
	int		now;
    struct utmp utmp;

    if( (fp=fopen("/etc/utmp","r")) == NULL ) {
        EXCEPT( "fopen of utmp" );
    }

    while( fread( (char *)&utmp, sizeof utmp, 1, fp ) ) {
        if( utmp.ut_name[0] == '\0' )
            continue;

        tty_idle = tty_idle_time( utmp.ut_line );
        user_idle = MIN( tty_idle, user_idle );
    }
    (void)fclose( fp );

    now = (int)time( (time_t *)0 );
	user_idle = MIN( now - Last_X_Event, user_idle );

	dprintf( D_FULLDEBUG, "User Idle Time is %d seconds\n", user_idle );
    return user_idle;
}

calc_Ncpus()
{
#ifdef sequent
    int     cpus = 0;
    int     eng;

    if( (eng=tmp_ctl(TMP_NENG,0)) < 0 ) {
        perror( "tmp_ctl(TMP_NENG,0)" );
        exit( 1 );
    }

    while( eng-- ) {
        if( tmp_ctl(TMP_QUERY,eng) == TMP_ENG_ONLINE ) {
            cpus++;
        }
    }
    return cpus;
#else sequent
    return 1;
#endif sequent
}

float
calc_load_avg()
{
    off_t addr = nl[X_AVENRUN].n_value;

#ifdef vax          /* Really any Vax, 4.3BSD */
    double  avenrun[3];
#endif vax

#ifdef mips			/* MIPS/ULTRIX DECstation 3100 */
	fix		avenrun[3];
#endif mips

#ifdef sequent      /* Really SYMMETRY, Dynix */
    int     avenrun[3];
#endif sequent

#ifdef mc68020      /* Really SUN3, SunOS 3.2 */
    long avenrun[3];
#endif mc68020

#ifdef sparc        /* Really SUN4, SunOS 4.0 */
    long avenrun[3];
#endif sparc

#if defined(IBM032) && defined(BSD43)
    double  avenrun[3];
#endif defined(IBM032) && defined(BSD43)

	avenrun[0] = 0;
	avenrun[1] = 0;
	avenrun[2] = 0;

    if( lseek(Kmem,addr,0) != addr ) {
        dprintf( D_ALWAYS, "Can't seek to addr 0x%x in /dev/kmem\n", addr );
        EXCEPT( "lseek" );
    }

    if( read(Kmem,(char *)avenrun,sizeof avenrun) != sizeof avenrun ) {
        EXCEPT( "read" );
    }

        /* just give back the 1 minute average for now */
#ifdef vax
    return (float)avenrun[0];
#endif vax

#ifdef mips
	return (float)avenrun[0]/(1<<FBITS);
#endif mips

#ifdef sequent
    return (float)avenrun[0]/FSCALE;
#endif sequent

#ifdef mc68020
    return (float)avenrun[0]/FSCALE;
#endif mc68020

#ifdef sparc
    return (float)avenrun[0]/FSCALE;
#endif sparc

#if defined(IBM032) && defined(BSD43)
    return (float)avenrun[0];
#endif defined(IBM032) && defined(BSD43)
}

get_k_vars()
{
    int     old_uid;


    old_uid = geteuid();
    (void)setuid( 0 );

        /* Open kmem for reading */
    if( (Kmem=open("/dev/kmem",O_RDONLY,0)) < 0 ) {
        EXCEPT( "open(/dev/kmem,O_RDONLY,0)" );
    }

    (void)seteuid( old_uid );

	/* Get the kernel's name list */
#if sequent
	(void)nlist( "/dynix", nl );
#else sequent
	(void)nlist( "/vmunix", nl );
#endif sequent

}

tty_idle_time( file )
char    *file;
{
    struct stat buf;
    long        now;
    long        answer;
    static char     pathname[100] = "/dev/";

    (void)strcpy( &pathname[5], file );
    if( stat(pathname,&buf) < 0 ) {
        EXCEPT( "stat(%s,0x%x)", pathname, &buf );
    }

    now = (int)time( (time_t *)0 );
    answer = now - buf.st_atime;

		/* This only happens if somebody screws up the date on the
		   machine :-<  */
	if( answer < 0 ) {
		answer = 0;
	}

    /*
    printf( "%s: %d seconds idle\n", file, answer );
    */
    return answer;
}

mail_context( name, context, fp )
char	*name;
CONTEXT	*context;
FILE	*fp;
{
    int     i, j;
	EXPR	*expr;

	fprintf( fp, "%s\n", name );
    for( i=0; i<context->len; i++ ) {
        fprintf( fp, "Stmt %2d:", i );
		expr = context->data[i];

        for( j=0; j<expr->len; j++ ) {
            display_elem( expr->data[j], fp );
            if( j+1 < expr->len ) {
                (void)fputc( ' ', fp );
            }
        }
        (void)fputc( '\n', fp );
    }
}

extern int	errno;
kill_starter( pid, signo )
int pid, signo;
{
	struct stat buf;

	for( errno=0; stat(Starter, &buf) < 0; errno=0 ) {
		if( errno == ETIMEDOUT ) {
			continue;
		}
		EXCEPT( "kill_starter( %d, %d ): cannot stat <%s> -- errno = %d\n",
					pid, signo, Starter, errno );
	}

	(void) kill( pid, signo );
}

abort()
{
	(void)signal( SIGQUIT, SIG_DFL );
	(void)kill( getpid(), SIGQUIT );
	(void)sigpause( 0 );
}
