/*
**  Copyright (c) 1991 Bolt Beranek and Newman, Inc.
**  All rights reserved.
**
**  Redistribution and use in source and binary forms are permitted
**  provided that: (1) source distributions retain this entire copyright
**  notice and comment, and (2) distributions including binaries display
**  the following acknowledgement:  ``This product includes software
**  developed by Bolt Beranek and Newman, Inc. and CREN/CSNET'' in the
**  documentation or other materials provided with the distribution and in
**  all advertising materials mentioning features or use of this software.
**  Neither the name of Bolt Beranek and Newman nor CREN/CSNET may be used
**  to endorse or promote products derived from this software without
**  specific prior written permission.
**
**  THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
**  WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
**  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
*/
#include <stdio.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <net/if.h>
#include <sys/param.h>
#include <netinet/in_systm.h>
#include <netinet/in.h>
#include <netinet/ip.h>
#include <net/if_du.h>
#include "dialupip.h"


#define EQ(a, b)		(strcmp((a), (b)) == 0)


typedef struct _FLAG {
    char	*Name;
    int		Value;
} FLAG;

static FLAG	Flags[] = {
    { "ESCAPED   ",	DS_ESCAPED },   /* Saw a FRAME_ESCAPE */
    { "OACTIVE   ",	DS_OACTIVE },	/* Actively sending a packet */
    { "LWAITING  ",	DS_LWAITING },	/* Waiting for a tty */
    { "LACTIVE   ",	DS_LACTIVE},	/* Have active line */
    { "LDOWN     ",	DS_LDOWN }, 	/* Line gone or no perm to call */
    { "FAILCALL  ",	DS_FAILCALL },	/* Recently failed to call */
    { "MONITORON ",	DS_MONITORON },	/* Monitoring activity */
    { "ENABLECALL",	DS_ENABLECALL},	/* Line can initiating calls */
    { NULL }
};

static char	*WHERE = "duioctl";

extern int	optind;
extern char	*strcpy();



static void
fatal(p)
    char	*p;
{
    perror(p);
    exit(1);
}


static void
printflags(i)
    int		i;
{
    FLAG	*fp;

    for (fp = Flags; fp->Name; fp++)
	(void)printf("\t%c%s\t 0x%04x\n",
	    (i & fp->Value) ? '*' : ' ', fp->Name, fp->Value);
}


static void
usage()
{
    (void)printf("Usage:  %s du# %s\n\t%s\n\t%s\n",
	progname,
	"[CLEARQ] [GWTIMEO] [SWTIMEO time] [GATIMEO] [SATIMEO time]",
	"[GSOFTFLAGS] [SSOFTFLAGS val] [SSOFTTIMER time]",
	"[BRINGUP] [ENABLECALL] [DISABLECALL]");
    (void)printf("time is the number of seconds in decimal;\n");
    (void)printf("SOFTFLAGS is a hex sum of the following:\n");
    printflags(0);
    exit(1);
}


static int
getnumber(param, value)
    char	*param;
    char	*value;
{
    char	*p;

    if (value) {
	while (*value && isspace(*value))
	    value++;
	if (*value) {
	    for (p = value; *p && isdigit(*p); p++)
		;
	    if (*p == '\0')
		return atoi(value);
	}
    }
    (void)fprintf(stderr, "Bad decimal number \"%s\" for %s.\n", value, param);
    exit(1);
    /* NOTREACHED */
}

main(argc, argv)
    int			argc;
    char		*argv[];
{
    struct ifreq	ifr;
    int			s;
    int			i;
    char		*device;

    /* Set up. */
    setprogname(argv[0]);
    WHERE = progname;
    (void)printf("%s %s\n", WHERE, dip_release());

    /* Do we have command line arguments to use? */
    while ((i = getopt(argc, argv, "")) != EOF)
	switch (i) {
	default:
	    usage();
	    /* NOTREACHED */
	}
    argc -= optind;
    argv += optind;
    if (*argv == NULL)
	usage();
    device = *argv;

    /* Create a socket. */
    if ((s = socket(AF_INET, SOCK_STREAM, 0)) < 0)
	fatal("Can't create socket");
    (void)strcpy(ifr.ifr_name, device);

    while (*++argv) {
#ifdef	SIOCGIFMETRIC
	if (EQ(*argv, "GIFMETRIC")) {
	    if (ioctl(s, SIOCGIFMETRIC, (caddr_t)&ifr) < 0)
		fatal("Can't SIOCGIFMETRIC");
	    (void)printf("Metric value is %d\n", GETDATAVAL(ifr));
	}
	else 
#endif
	if (EQ(*argv,"GATIMEO")) {
	    if (ioctl(s, SIOCGATIMEO, (caddr_t)&ifr) < 0)
		fatal("Can't SIOCGATIMEO");
	    (void)printf("ATIMEO value is %d seconds\n", GETDATAVAL(ifr));
	}
	else if (EQ(*argv, "GWTIMEO")) {
	    if (ioctl(s, SIOCGWTIMEO, (caddr_t)&ifr) < 0)
		fatal("Can't SIOGWTIMEO");
	    (void)printf("WTIMEO value is %d seconds\n", GETDATAVAL(ifr));
	}
	else if (EQ(*argv, "SATIMEO")) {
	    i = getnumber(argv[0], argv[1]);
	    argv++;
	    SETDATAVAL(ifr, i);
	    if (ioctl(s, SIOCSATIMEO, (caddr_t)&ifr) < 0)
		fatal("Can't SIOCSATIMEO");
	    d_log(DLOG_GENERAL, WHERE, "Set ATIMEO on \"%s\" to %d",
		device, i);
	    (void)printf("Set ATIMEO to %d\n", s);
	}
	else if (EQ(*argv, "SWTIMEO")) {
	    i = getnumber(argv[0], argv[1]);
	    argv++;
	    SETDATAVAL(ifr, i);
	    if (ioctl(s, SIOCSWTIMEO, (caddr_t)&ifr) < 0)
		fatal("Can't SIOCSWTIMEO");
	    d_log(DLOG_GENERAL, WHERE, "Set WTIMEO on \"%s\" to to %d",
		device, i);
	    (void)printf("Set WTIMEO to %d\n", i);
	}
	else if (EQ(*argv,"SSOFTTIMER")) {
	    i = getnumber(argv[0], argv[1]);
	    argv++;
	    SETDATAVAL(ifr, i);
	    if (ioctl(s, SIOCSSOFTTIMER, (caddr_t)&ifr) < 0)
		fatal("Can't SIOCSSOFTTIMER");
	    d_log(DLOG_GENERAL, WHERE, "Set SOFTTIMER on \"%s\" to %d",
		i, device);
	    (void)printf("Set SOFTTIMER to %d\n", i);
	}
	else if (EQ(*argv, "ENABLECALL")) {
	    if (ioctl(s, SIOCGSOFTFLAGS, (caddr_t)&ifr) < 0)
		fatal("Can't SIOCGSOFTFLAGS to enable call");
	    i = GETDATAVAL(ifr) | DS_ENABLECALL;
	    SETDATAVAL(ifr, i);
	    if (ioctl(s, SIOCSSOFTFLAGS, (caddr_t)&ifr) < 0)
		fatal("Can't SIOCSSOFTFLAGS to enable call");
	    d_log(DLOG_GENERAL, WHERE, "Enabled calling on \"%s\"", device);
	    (void)printf("Enabled calling\n");
	}
	else if (EQ(*argv, "DISABLECALL")) {
	    if (ioctl(s, SIOCGSOFTFLAGS, (caddr_t)&ifr) < 0)
		fatal("Can't SIOCGSOFTFLAGS to disable call");
	    i = GETDATAVAL(ifr) & ~DS_ENABLECALL;
	    SETDATAVAL(ifr, i);
	    if (ioctl(s, SIOCSSOFTFLAGS, (caddr_t)&ifr) < 0)
		fatal("Can't SIOCSSOFTFLAGS to disable call");
	    d_log(DLOG_GENERAL, WHERE, "Disabled calling on \"%s\"", device);
	    (void)printf("Disabled calling\n");
	}
	else if (EQ(*argv, "GSOFTFLAGS")) {
	    if (ioctl(s, SIOCGSOFTFLAGS, (caddr_t)&ifr) < 0)
		fatal("Can't SIOCGSOFTFLAGS");
	    i = GETDATAVAL(ifr);
	    (void)printf("SOFTFLAGS are set to 0x%x where values are:\n", i);
	    printflags(i);
	}
	else if (EQ(*argv, "SSOFTFLAGS")) {
	    if (*++argv == NULL) {
		(void)fprintf(stderr, "Missing flag value for SSOFTFLAGS\n");
		exit(1);
	    }
	    if (sscanf(*argv, "%x", &i) != 1) {
		(void)fprintf(stderr, "Bad hex value for SSOFTFLAGS\n");
		exit(1);
	    }
	    SETDATAVAL(ifr, i);
	    if (ioctl(s, SIOCSSOFTFLAGS, (caddr_t)&ifr) < 0)
		fatal("Can't SIOCSSOFTFLAGS");
	    d_log(DLOG_GENERAL, WHERE, "Set SOFTFLAGS to 0x%x on %s",
		i, device);
	    (void)printf("Set SOFTFLAGS to 0x%x\n", i);
	}
	else if (EQ(*argv, "CLEARQ")) {
	    if (ioctl(s, SIOCCLEARQ, (caddr_t)&ifr) < 0)
		fatal("Can't SIOCLEARQ");
	    d_log(DLOG_GENERAL, WHERE, "Cleared output queue on \"%s\"",
		device);
	    (void)printf("Cleared output queue\n");
	}
	else if (EQ(*argv, "FAILCALL")) {
	    if (ioctl(s, SIOCFAILCALL, (caddr_t)&ifr) < 0)
		fatal("Can't SIOCFAILCALL");
	    d_log(DLOG_GENERAL, WHERE, "Marked failed call on \"%s\"", device);
	    (void)printf("Marked failed call\n");
	}
	else if (EQ(*argv, "BRINGUP")) {
	    if (ioctl(s, SIOCBRINGUP, (caddr_t)&ifr) < 0)
		fatal("Can't SIOCBRINGUP");
	    d_log(DLOG_GENERAL, WHERE, "BRINGUP on \"%s\"", device);
	    (void)printf("Sent BRINGUP request\n");
	}
	else {
	    (void)printf("Unknown command \"%s\"\n", *argv);
	    usage();
	}
    }

    exit(0);
    /* NOTREACHED */
}
