/*
 * Copyright (c) 1989, 1990 by the University of Washington
 * Copyright (c) 1992, 1993 by the University of Southern California
 *
 * For copying and distribution information, please see the files
 * <uw-copyright.h> and <usc-copyr.h>.
 */

#include <uw-copyright.h>
#include <usc-copyr.h>

#include <stdio.h>
#include <strings.h>

#include <pfs.h>
#include <perrno.h>

int	perrno;
int	pfs_debug = 0;

static void usage();

main(argc,argv)
    int		argc;
    char	*argv[];
{
    int		tmp;
    int         modreq = 0;     /* modification request. */

    VLINK       vl;
    PATTRIB     at = atalloc();
    FILTER      fl = flalloc(); /* in case a filter argument is specified. */

    char        *linkname = NULL;
    char        *nativehost = NULL;
    char        *nativehandle = NULL;

    /* Initialize attribute; this may be modified by later flags. */
    at->precedence = ATR_PREC_OBJECT;
    at->nature = ATR_NATURE_APPLICATION;
    at->avtype = ATR_UNKNOWN;
    at->value.sequence = NULL;
    at->previous = NULL;
    at->next = NULL;

    fl->type = FIL_DIRECTORY;   /* only type supported right now, I believe. */
    fl->execution_location = FIL_CLIENT;
    fl->pre_or_post = FIL_POST;
    for (;*++argv;) {                  /* process flags */
        /* We are now positioned at an unread argument. */

        if (strnequal(argv[0],"-D",2)) {
            pfs_debug = 1; /* Default debug level */
            sscanf(argv[0],"-D%d",&pfs_debug);
            continue;
        }

        /* Modification requests. */
        if (strequal(argv[0], "-add")) {
            modreq = EOI_ADD;
            continue;
        }
        if (strequal(argv[0], "-delete")) {
            modreq = EOI_DELETE;
            continue;
        }
        if (strequal(argv[0], "-delete-all")) {
            modreq = EOI_DELETE_ALL;
            continue;
        }
        if (strequal(argv[0], "-replace")) {
            modreq = EOI_REPLACE;
            continue;
        }

        /* Attribute values. */
        if (strequal(*argv, "-object")) {
            at->precedence = ATR_PREC_OBJECT;
            continue;
        }
        if (strequal(*argv, "-linkprec")) {
            at->precedence = ATR_PREC_LINK;
            continue;
        }
        if (strequal(*argv, "-cached")) {
            at->precedence = ATR_PREC_CACHED;
            continue;
        }
        if (strequal(*argv, "-replacement")) {
            at->precedence = ATR_PREC_REPLACE;
            continue;
        }
        if (strequal(*argv, "-additional")) {
            at->precedence = ATR_PREC_ADD;
            continue;
        }

        /* nature */
        if (strequal(*argv, "-field")) {
            at->nature = ATR_NATURE_FIELD;
            continue;
        }
        if (strequal(*argv, "-application")) {
            at->nature = ATR_NATURE_APPLICATION;
            continue;
        }
        if (strequal(*argv, "-intrinsic")) {
            at->nature = ATR_NATURE_INTRINSIC;
            continue;
        }

        if (strequal(argv[0], "-native")) {
            if (!argv[1] || !argv[2]) usage();
            if (linkname) {
                fprintf(stderr, "Can't use -native after you've already \n
specified a target link.\n");
                usage();
            }
            nativehost = *++argv;
            nativehandle = *++argv;
            /* leave us on top of the last used argument. */
            continue;
        } 

        /* Filter arguments. */
        if (strequal(*argv, "-filter-predefined")) {
            assert(at->value.filter = fl); /* make sure flalloc() didn't return
                                              NULL.  */
            at->avtype = ATR_FILTER;
            if (!argv[1]) usage();
            fl->name = *++argv;
            continue;
        }
        if (strequal(*argv, "-filter-loadable")) {
            assert(at->value.filter = fl); /* make sure flalloc() didn't return
                                              NULL.  */
            at->avtype = ATR_FILTER;
            if (!argv[1]) usage();
            perrno = 0, *p_err_string = '\0';
            if((fl->link = rd_vlink(*++argv)) == NULL) {
                fprintf(stderr, "%s not found: ", *argv);
                perrmesg(NULL, 0, NULL);
                exit(1);
            }
            continue;
        }
        if (strequal(*argv, "-client")) {
            fl->execution_location = FIL_CLIENT;
            continue;
        }
        if (strequal(*argv, "-server")) {
            fl->execution_location = FIL_SERVER;
            continue;
        }
        if (strequal(*argv, "-pre")) {
            fl->pre_or_post = FIL_PRE;
            continue;
        }
        if (strequal(*argv, "-post")) {
            fl->pre_or_post = FIL_POST;
            continue;
        }
        if (strequal(*argv, "-args")) {
            while (argv[1]) 
                fl->args = tkappend(*++argv, fl->args);
            continue;
        }

        /* Other types of attributes -- link & sequence. */
        if (strequal(*argv, "-link")) {
            at->avtype = ATR_LINK;
            perrno = 0, *p_err_string = '\0';
            if((at->value.link = rd_vlink(linkname = *++argv)) == NULL) {
                fprintf(stderr, "%s not found: ", linkname);
                perrmesg(NULL, 0, NULL);
                exit(1);
            }
            continue;
        }

        if (strequal(*argv, "-sequence")) {
            if (at->avtype != ATR_UNKNOWN) {
                fprintf(stderr, "set_atr: May not specify more than one of \
-sequence, -link, -filter-loadable and -filter-predefined\n");
                exit(1);
            }
            ++argv;             /* put us on top of an unused argument */
            break;              /* bail */
        }

        /* Unrecognized argument.  If no host specified yet, it's the link to
           munge.  Otherwise, it's the attribute name.   */
        if (!nativehost && !linkname) {
            linkname = *argv;
            continue;
        } else if (!at->aname) {
            at->aname = stcopy(*argv);
            continue;
        } else {
            break;              /* must be a sequence argument. */
        }
    }
    /*  We're done reading all the arguments (except, perhaps, for sequence
        values. */
    if (!nativehost && !linkname) {
        fprintf(stderr, "No target specified!\n");
        usage();
    }
    if (!at->aname) { 
        fprintf(stderr, "No attribute name specified!\n");
        usage();
    }
    if(!modreq) {
        fprintf(stderr, "One of -add, -delete, -replace, or -delete-all must be specified.\n");
        usage();
    }	    

    /* Read the sequence values. */
    if (at->avtype != ATR_UNKNOWN) {
        if (*argv) {  /* not a sequence type. */
            fprintf(stderr, "Too many arguments.\n");
            usage();
        }
    } else {
        at->avtype = ATR_SEQUENCE;
    }
    while (*argv)
        at->value.sequence = tkappend(*argv++, at->value.sequence);
    

    if (at->precedence == ATR_PREC_OBJECT) {
        /* Set the link VL to point to the object whose attributes we're
           modifying. */
        if (nativehost) {
            if(vl = vlalloc()) {
                vl->host = stcopy(nativehost);
                vl->hsoname = stcopy(nativehandle);
            } else {
                fprintf(stderr, "Out of memory\n");
                exit(1);
            }
        } else {
            perrno = 0, *p_err_string = '\0';
            if((vl = rd_vlink(linkname)) == NULL) {
                fprintf(stderr, "%s not found: ", linkname);
                perrmesg(NULL, 0, NULL);
                exit(1);
            }
        }
        perrno = 0, *p_err_string = '\0';
        if(tmp = pset_at(vl, modreq, at)) {
            perrmesg("set_atr: pset_at() failed: ", 0, NULL);
            exit(1);
        }
    } else { 
        /* Modifying other than an object attribute. */
        char *compname; 
        VDIR_ST	dir_st;
        VDIR	dir = &dir_st;

        vdir_init(dir);
        if (!linkname)  {
            fprintf(stderr, "-native only works to set OBJECT attributes.\n");
            usage();
        }
        compname = p_uln_rindex(linkname, '/');
        if(compname) *compname++ = '\0';
        else {
            compname = linkname;
            linkname = "";
        }
        /* linkname is now the name of the directory in question. */
        perrno = 0, *p_err_string = '\0';
        tmp = rd_vdir(linkname, 0, dir, RVD_DFILE_ONLY);
        if (tmp || (dir->links == NULL)) return(DIRSRV_NOT_DIRECTORY);
        /* Dir->links is a link to the directory */
        if(tmp = pset_linkat(dir->links, compname, modreq, at)) {
            perrmesg("set_atr: pset_linkat() failed: ", 0, NULL);
            exit(1);
        }
    }        
    if(pwarn) pwarnmesg("WARNING: ",0,NULL);

    exit(0);
}


static void
usage()
{
    fprintf(stderr, 
            "Usage: set_atr [-D[#]] { -native host handle | linkname } atrname\n\
\t{ -add | -delete | -replace | -delete-all }\n\
\t{ -object | -linkprec | -cached | -replacement | -additional }\n\
\t{ -field | -application | -intrinsic }\n\
\t{ { -filter-predefined name | -filter-loadable linkname }\
 { -client | -server } { -pre | -post } [ -args arg1 arg2... ] }\n\
\t | -link linkname  | [ [-sequence] seqelem1 seqelem2 ... ]\n");
    exit(1);
}
