/*
 * Copyright (c) 1989, 1990, 1991 by the University of Washington
 * Copyright (c) 1992, 1993 by the University of Southern California
 *
 * For copying and distribution information, please see the files
 * <uw-copyright.h> and <usc-copyr.h>.
 */

#include <uw-copyright.h>
#include <usc-copyr.h>
#include <sys/param.h>
#include <sys/stat.h>
#include <sys/dir.h>
#include <stdio.h>
#include <sgtty.h>
#include <strings.h>

#include <ardp.h>
#include <pfs.h>
#include <plog.h>
#include <pprot.h>
#include <perrno.h>
#include <pmachine.h>
#include <pparse.h>

#define DIR_VNO		     5  /* version # of the directory format. */

extern char	root[];
extern char	shadow[];
extern char	dirshadow[];
extern char	dircont[];
extern char	pfsdat[];
extern char	aftpdir[];

extern char	hostname[];
extern char	hostwport[];

extern char	*acltypes[];


static int indsrdir_v5(INPUT in, char *name, char vfs_dirname[], VDIR dir, VLINK ul);

/*
 * dsrdir - Read the contents of a directory from disk
 *
 *   DSRDIR is used by the directory server to read the contents of
 *   a directory off disk.  It takes the host specific part of the system
 *   level name of the directory as an argument, and a directory structure
 *   which it fills in.
 *   If the directory is forwarded, it returns DSRFINFO_FORWARDED. 
 *   On error, it returns an error code and logs the circumstances to plog().
 */
dsrdir(name,magic,dir,ul,flags)
    char	*name;
    int		magic;
    VDIR	dir;
    VLINK	ul;      /* Pointer to the vlink currently being expanded */
	                 /* used for location to insert new union links   */
    int		flags;   /* DSRD_ATTRIBUTES = get attributes for files    */
{
    FILE 		*vfs_dir;
    char		vfs_dirname[MAX_DIR_LINESIZE];
    char		native_dirname[MAX_DIR_LINESIZE];
    char		rel_filename[MAX_DIR_LINESIZE];
    char		*slash;
    VLINK		cur_link = NULL;
    VLINK		cur_fil;
    ACL		cur_acl = NULL;
    ACL		prev_acl = NULL;
    int		tmp;
    char *command, *next_word;
    char include_native[30];

    int		return_value = DSRDIR_NOT_A_DIRECTORY;

    dir->version = -1;
    dir->magic_no = 0;
    dir->inc_native = VDIN_INCLREAL;    /* Include native if can't read */
    dir->f_info = NULL;
    dir->dacl = NULL;

    if(ul == NULL) {               /* Don't clear links if expanding */
        dir->links = NULL;
        dir->ulinks = NULL;
    }

    /* Determine name of directory contents */
    if((*name != '/') && *aftpdir && (strncmp(name,"AFTP",4) == 0)) {
	/* Special file name */
	strcpy(native_dirname,aftpdir);
	strcat(native_dirname,name+4);
    }
    else strcpy(native_dirname,name);
    
    strcpy(vfs_dirname,shadow);  
    strcat(vfs_dirname,native_dirname);
    strcat(vfs_dirname,"/");

    strcat(vfs_dirname,dircont);


    /* NOTE: A temporary inefficient directory format is  */
    /* in use.  For this reason, the code supporting it   */
    /* is also interim code, and does not do any checking */
    /* to make sure that the directory actually follows   */
    /* the format.  This, a munged directory will result  */
    /* in unpredictable results.			      */

    /* Read the contents of the VFS directory */
    if((vfs_dir = fopen(vfs_dirname,"r")) != NULL) {
        INPUT_ST in_st;
        INPUT in = &in_st;
        if(tmp = wholefiletoin(vfs_dir, in))
            return tmp;
        return_value = indsrdir_v5(in, name, vfs_dirname, dir, ul);
        fclose(vfs_dir);
    } else     /* Note that this directory is entirely native */
        (dir->inc_native = VDIN_ONLYREAL);

    /* Read the contents of the native directory */
    if(dir->inc_native != VDIN_NONATIVE) {
        DIR			*dirp;
        struct direct	*dp;
        if(dirp = opendir(native_dirname)) {
            return_value = PSUCCESS;

            for (dp = readdir(dirp); dp != NULL; dp = readdir(dirp)) {
		strcpy(rel_filename,name);	

                /* We must special case unix concept of . and .. */
                /* if we want to eliminate useless parts of path */
                /* If the filename is "..", then                 */
                if(!strcmp(dp->d_name,"..")) {
#ifdef NODOTDOT
                    continue;
#else
                    if(dir->inc_native == VDIN_INCLREAL ||
                       dir->inc_native == VDIN_ONLYREAL) continue;
                    slash = rindex(rel_filename,'/');
                    if(slash) *slash = '\0';
                    else strcpy(rel_filename,"/");
                    if(!(*rel_filename)) strcpy(rel_filename,"/");
#endif
                }
                /* Else if ".", do nothing.  If not "." add to   */
                /* rel_filename					 */
                else if(strcmp(dp->d_name,".")) {
                    if(*(rel_filename + strlen(rel_filename) - 1)!='/')
                        strcat(rel_filename,"/");
                    strcat(rel_filename,dp->d_name);
                }
                /* If ".", must still decide if we include it */
                else if(dir->inc_native == VDIN_INCLREAL ||
                        dir->inc_native == VDIN_ONLYREAL) {
                    continue;
                }

                cur_link = vlalloc();

                cur_link->name = stcopy(dp->d_name);
                cur_link->hsoname = stcopy(rel_filename);
                cur_link->host = stcopy(hostwport);
#ifdef INVISIBLE_DOT_FILES
                /* Native . files are invisible. */
                if (*cur_link->name == '.') 
                    cur_link->linktype = 'I';
                else 
#else
                    cur_link->linktype = 'N';
#endif
                if(flags == DSRD_ATTRIBUTES) {
                   PFILE_ST	fi_st;
                   PFILE		fi = &fi_st;

                   tmp = dsrfinfo(cur_link->hsoname,
                                  cur_link->f_magic_no,fi);
                   if(tmp == -1) 
                      cur_link->target = stcopyr("DIRECTORY",cur_link->target);

                   if(tmp <= 0) cur_link->lattrib = fi->attributes;
                } 
                vl_insert(cur_link,dir,VLI_ALLOW_CONF);
            }
            closedir(dirp);
        }
    }
    if((return_value != PSUCCESS) || (dir->magic_no != magic)) {
        PFILE		dfi = pfalloc();

        tmp = dsrfinfo(name,magic,dfi);
        if(tmp == DSRFINFO_FORWARDED) {
            dir->f_info = dfi;
            return(DSRFINFO_FORWARDED);
        }
        /* This bit of code may not be necessary.  Anyway, this stuff is
           getting replaced soon. --swa@isi.edu */
        else if((dir->magic_no < 0) && dfi->forward) {
            dir->f_info = dfi;
            return(DSRFINFO_FORWARDED);
        }
        pffree(dfi);
    }
    return(return_value);
}


static
int
indsrdir_v5(INPUT in, char *name, char vfs_dirname[], VDIR dir, VLINK ul)
{
    char *command, *next_word;
    char include_native[30];
    char *cp;                   /* dummy pointer */
    VLINK cur_link = NULL;      /* set this when we see a link.  */
    int     seen_version = 0;   /* seen directory format version #?  */

    while(in_nextline(in)) {
        if(in_line(in, &command, &next_word)) {
            plog(L_DATA_FRM_ERR, NOREQ, "Couldn't read line from %s: %s",
                         vfs_dirname, p_err_string);
            return PFAILURE;
        }

        if (qsscanf(command, "VERSION %d %r", &(dir->version), &cp) == 1) {
            if (dir->version != DIR_VNO) {
                plog(L_DATA_FRM_ERR, NOREQ, 
                     "Bad directory info format: %s: Encountered VERSION %d; \
dirsrv can only interpret version %d", vfs_dirname, dir->version, DIR_VNO);
                return PFAILURE;
            }
            seen_version++;
            continue;
        }
        if (!seen_version) {
            plog(L_DATA_FRM_ERR, NOREQ, 
                 "Bad directory info format: %s: VERSION line not found; must \
appear at the start of the file.", vfs_dirname);
            return PFAILURE;
        }
        if (qsscanf(command, "INCLUDE-NATIVE %!!s %r", 
                    include_native, sizeof include_native, &cp) == 1) {
            if(strequal(include_native, "INCLNATIVE"))
                dir->inc_native = VDIN_INCLNATIVE;
            else if (strequal(include_native, "INCLREAL"))
                dir->inc_native = VDIN_INCLREAL;
            else if (strequal(include_native, "NONATIVE"))
                dir->inc_native = VDIN_NONATIVE;
            else {
                plog(L_DATA_FRM_ERR, NOREQ, 
                     "Bad directory info format %s: %s", vfs_dirname, command);
                return PFAILURE;
            }
            continue;
        }
        if (qsscanf(command, "MAGIC-NUMBER %d %r", &(dir->magic_no), &cp) == 1)
            continue;
        if (qsscanf(command, "ACL %r", &cp) == 1) {
            if (cur_link) {
                if(in_ge1_acl(in, command, next_word, &cur_link->acl)) {
                    plog(L_DATA_FRM_ERR, NOREQ, 
                         "Bad directory info format %s: %s",
                         vfs_dirname, p_err_string);
                    return PFAILURE;
                }
            } else {
                if(in_ge1_acl(in, command, next_word, &dir->dacl)) {
                    plog(L_DATA_FRM_ERR, NOREQ, 
                         "Bad directory info format %s: %s",
                         vfs_dirname, p_err_string);
                    return PFAILURE;
                }
            }
            continue;
        }
        /* in_link() will automatically call in_atrs. */
        if (qsscanf(command, "LINK %r", &next_word) == 1) {
            if(in_link(in, command, next_word, 0, &cur_link, 
                    (TOKEN *) NULL)) {
                plog(L_DATA_FRM_ERR, NOREQ, 
                     "Bad directory info format %s: %s",
                     vfs_dirname, p_err_string);
                return PFAILURE;
            }
            /* if union link without UL variable set, then vl_insert will
               insert it at the end of the list of union links. */
                
            /* This call to ul_insert() makes sure that, *if*
               list() is in the process of expanding union links, any new
               union links are inserted in the right position (i.e., right
               after the link we're currently expanding).   This makes sure
               that nested union links are expanded in the proper order,
               whereas calling vl_insert() would instead mean that the
               new union link CUR_LINK was stuck at the end of the
               directory's list of union links; this would be wrong. 
               Also, ul_insert() checks for conflicts if given an insertion
               positon. */
            if(ul && cur_link->linktype == 'U') {
                int tmp = ul_insert(cur_link,dir,ul);
                if(!tmp) ul = cur_link; /* A subsequent identical link in this
                                           directory won't supersede this one.
                                           */ 
            } else {
                vl_insert(cur_link, dir, VLI_ALLOW_CONF);
            }
            continue;
        }
        /* Gee, nothing seems to match.  Guess it must be an error. */
        plog(L_DATA_FRM_ERR, NOREQ, "Bad directory info format %s: %s",
             vfs_dirname, command);
        return PFAILURE;
    }
    return PSUCCESS;
}

/*
 * dswdir - Write directory contents to disk
 *
 */
dswdir(name,dir)
    char	*name;
    VDIR	dir;
{

    char		vfs_dirname[MAX_DIR_LINESIZE];
    FILE 		*vfs_dir;
    VLINK		cur_link;
    ACL		cur_acl;
    char		qatype[MAX_DIR_LINESIZE];
    char		qrights[MAX_DIR_LINESIZE];
    char		qprin[MAX_DIR_LINESIZE];


    /* if a VFS directory, then create it if necessary */
    if(!strncmp(pfsdat,name,strlen(pfsdat)))
        mkdirs(name);

    /* Determine name of shadow */
    strcpy(vfs_dirname,shadow);
    if((*name != '/') && *aftpdir && (strncmp(name,"AFTP",4) == 0)) {
	/* Special file name */
	strcat(vfs_dirname,aftpdir);
	strcat(vfs_dirname,name+4);
    }
    else strcat(vfs_dirname,name);

    /* Create the shadow directory if necessary */
    mkdirs(vfs_dirname);

    /* Determine name of directory contents */
    strcat(vfs_dirname,"/");
    strcat(vfs_dirname,dircont);


    /* NOTE: A temporary inefficient directory format is  */
    /* in use.  For this reason, the code supporting it   */
    /* is also interim code, and does not do any checking */
    /* to make sure that the directory actually follows   */
    /* the format.  This, a munged directory will result  */
    /* in unpredictable results.			      */

    /* Write the contents of the VFS directory */
    if((vfs_dir = fopen(vfs_dirname,"w")) != NULL) {
        fdswdir_v5(vfs_dir, dir);
        fclose(vfs_dir);
        return(PSUCCESS);
    }
    else return(PFAILURE);
}

static int dump_links(OUTPUT out, VLINK cur_link, int write_native);

int fdswdir_v5(FILE *vfs_dir, VDIR dir)
{
    OUTPUT_ST out_st;
    OUTPUT out = &out_st;

    filetoout(vfs_dir, out);
    qoprintf(out, "VERSION %d\n", DIR_VNO);
    if (dir->magic_no)
        qoprintf(out, "MAGIC-NUMBER %d\n", dir->magic_no);
    qoprintf(out, "INCLUDE-NATIVE ");
    if(dir->inc_native == VDIN_INCLREAL || dir->inc_native == VDIN_ONLYREAL) 
        qoprintf(out,"INCLREAL\n");
    else if(dir->inc_native == VDIN_INCLNATIVE)
        qoprintf(out,"INCLNATIVE\n");
    else if (dir->inc_native == VDIN_NONATIVE)
        qoprintf(out,"NONATIVE\n");
    else
        internal_error("unknown value of dir->inc_native");
    /* print out the directory ACL. */
    out_acl(out, dir->dacl);
    dump_links(out, dir->links, dir->inc_native == VDIN_NONATIVE);
    dump_links(out, dir->ulinks, dir->inc_native == VDIN_NONATIVE);
    return PSUCCESS;
}


static int
dump_links(OUTPUT out, VLINK cur_link, int dump_native)
{
    for (; cur_link; cur_link = cur_link->next) {
        PATTRIB ca;
        FILTER cur_fil;

           /* don't output native links to directory unless we just converted
              it to NONATIVE. */
        if (!dump_native && cur_link->linktype == 'N')
            continue;
        qoprintf(out, "LINK ");
        out_link(out, cur_link, 0, (TOKEN) NULL);
        for (ca = cur_link->lattrib; ca; ca = ca->next)
            if ((ca->nature != ATR_NATURE_INTRINSIC)
                && (ca->precedence != ATR_PREC_OBJECT))
            out_atr(out, ca, 0);
        for(cur_fil = cur_link->filters; cur_fil; cur_fil = cur_fil->next) {
            qoprintf(out, "ATTRIBUTE LINK FIELD FILTER FILTER ");
            out_filter(out, cur_fil, 0);
        }
        out_acl(out, cur_link->acl);
    }
}

