/*
 * Copyright (c) 1989, 1990, 1991 by the University of Washington
 * Copyright (c) 1991, 1992, 1993 by the University of Southern California
 *
 * For copying and distribution information, please see the files
 * <uw-copyright.h> and <usc-copyr.h>.
 */

#include <uw-copyright.h>
#include <usc-copyr.h>
#include <sys/param.h>
#include <sys/wait.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <strings.h>
#include <stdio.h>

#include <pfs.h>
#include <psite.h>
#include <perrno.h>
#include <pcompat.h>
#include <pmachine.h>

extern int	errno;
int		perrno;
int		pfs_debug = 0;

extern char *getenv(char *varname);

static int
retrieve_link(VLINK vl, char *lclfil, char *debugflag, char *verboseflag);

static int copy_file(char *inname, char *outname, int forcecopyflag);

main(argc, argv)
    char *argv[];
{
    char		path[MAX_VPATH];
    char		*lclfil;
    char		*method;
    VLINK		vl;

    int		message_option = 0;
    int		avsflag = 0;  /* If set - use active VS (no closure) */
    char		*debugflag = "-";
    char		*verboseflag = "-v";

    argc--;argv++;

    while (argc > 0 && *argv[0] == '-' && *(argv[0]+1) != '\0') {
        switch (*(argv[0]+1)) {

        case 'D':
            pfs_debug = 1; /* Default debug level */
            sscanf(argv[0],"-D%d",&pfs_debug);
            debugflag = argv[0];
            break;

        case 'a':
            avsflag++;
            break;

        case 'm':
            message_option++; 
            break;

        case 'q':
            verboseflag = "-";
            break;

        case 'v':
            verboseflag = "-v";
            break;

        default:
            fprintf(stderr,
                    "Usage: vget [-a,-m,-q,-v] virtual-file [local-file]\n");
            exit(1);
        }
        argc--; argv++;
    }

    /* The next argument must be the name of the file to retrieve */
    /* within the virtual file system unless it is to be extracted   */
    /* from a mail message                                        */

    if(argc < 1 && !message_option) {
        fprintf(stderr,"Usage: vget [-a,-m,-q,-v] virtual-file [local-file]\n");
        exit(1);
    }

    if (argc >= 1) strcpy(path,argv[0]);

    /* if stdin is not a tty and OK to use closure */
    /* then we have to extract closure info        */
    if(!avsflag && !isatty(0)) {
        char	*s;
        s = readheader(stdin,"virtual-system-name:");
        if(!s)  {
            fprintf(stderr,"vget: Can't find Virtual-System-Name.\n");
            exit(1);
        }
        sprintf(path,"%s:",s);

        if(message_option) {
            s = readheader(stdin,"virtual-file-name:");
            if(!s) {
                fprintf(stderr,"vget: Can't find Virtual-file-name.\n");
                exit(1);
            }
            strcat(path,s);
        }
        else strcat(path,argv[0]);

    }
    else if(message_option) {
        fprintf(stderr,"vget: Can't find Virtual-file-name.\n");
        exit(1);
    }

    /* If second name was not specified, derive it from first */
    if(argc == (message_option ? 0 : 1)) {
        char *p;
        p = p_uln_rindex(path,'/');
        lclfil = (p != NULL) ? p + 1 : path;
    }
    /* Otherwise local file is the second name */
    else lclfil = (message_option ? argv[0] : argv[1]);

    vl = rd_vlink(path);

    if(!vl) {
        if(perrno == PSUCCESS) fprintf(stderr,"vget: File not found\n");
        else perrmesg("vget: ", 0, NULL);
        exit(1);
    }

    if(retrieve_link(vl, lclfil, debugflag, verboseflag) != PSUCCESS)
        exit(1);
    exit(0);
}

/* This function looks a heck of a lot like lib/pcompat/mapname().  If there's
   a bug here, there's one there too. */
static int
retrieve_link(VLINK vl, char *lclfil, char *debugflag, char *verboseflag)
{
    TOKEN           am_args;
    int             am;         /* access method number */
    union wait 	status;
    int 		pid;
    int                 methods = P_AM_LOCAL | P_AM_AFTP | P_AM_FTP |
        P_AM_GOPHER | P_AM_RCP;
    static char	*myvhostname = NULL;
    char    *vcargv[12];        /* vcache's ARGV.  Enough to hold any known
                                   access method arguments. */
    char    **vcargvp = vcargv; /* pointer to vcache's argv. */
    char    npath[MAXPATHLEN];  /* local pathname, if needed. */

    /* First time called, find out hostname and remember it */
    if(!myvhostname) myvhostname = myhostname();

#ifdef P_NFS
    methods |= P_AM_NFS;
#endif P_NFS

#ifdef P_AFS
    methods |= P_AM_AFS;
#endif P_AFS

    am = pget_am(vl,&am_args,methods);
    if (!am) {
        if (perrno) perrmesg("vget: ", 0, NULL);
        else fprintf(stderr,"vget: Can't access file using any available \
access method.\n");
        return PFAILURE;
    }
    if(pwarn) pwarnmesg("WARNING: ",0,NULL);

    switch(am) {
    case P_AM_AFTP:
    case P_AM_FTP:
    case P_AM_GOPHER:
    case P_AM_RCP:
        pid = fork();
        *vcargvp++ = "vcache";
        *vcargvp++ = debugflag;
        *vcargvp++ = verboseflag;
        *vcargvp++ = lclfil;
        for (;am_args; am_args = am_args->next)
            *vcargvp++ = am_args->token;
        *vcargvp = NULL;

        if (pid == 0) {
            char		vcachebin[MAXPATHLEN];
            char *p_binaries = getenv("P_BINARIES");
#ifdef P_BINARIES
            if (!p_binaries)
                p_binaries = P_BINARIES;
#endif

            if (!p_binaries || !*p_binaries) {
                DISABLE_PFS(execvp("vcache", vcargv));
                strcpy(vcachebin, "vcache"); /* for error message */
            } else {
                qsprintf(vcachebin, sizeof vcachebin, "%s/vcache", p_binaries);
                DISABLE_PFS(execv(vcachebin,vcargv));
            }
            fprintf(stderr, 
                    "vget: exec failed for %s (errno=%d): ",vcachebin,errno);
            perror(NULL);
            return PFAILURE;
        }
        else wait(&status);

        if(status.w_T.w_Retcode) {
            fprintf(stderr,"vget: Retrieve failed\n");
            return PFAILURE;
        }
        return PSUCCESS;

    case P_AM_LOCAL:
        return copy_file(elt(am_args, 4), lclfil, 1);

#ifdef P_NFS
    case P_AM_NFS:
        /* XXX You must change pmap_nfs() to meet the needs of your site. */
        tmp = pmap_nfs(link->host,link->hsoname,npath, sizeof npath, am_args);
        if (!tmp) tmp = copy_file(npath, lclfil, 1);
        return(tmp);
#endif

#ifdef P_AFS
    case P_AM_AFS:
        strcpy(npath,P_AFS);
        strcat(npath,elt(am_args,4)); /* 4th element is the hsoname. */
        return copy_file(npath, lclfil, 1);
        return(PSUCCESS);
#endif
        
    default:
        fprintf(stderr,"vget: Can't access file using any available \
access method.\n");
        return PFAILURE;
    }
}


static
int
copy_file(char *inname, char *outname, int forcecopyflag)
{
    char buf[1024];
    FILE *in, *out;
    int numread, numwritten;
    struct stat st_buf;

    /* Check if the local file is a directory.  If so, don't copy it. */
    if(stat(inname, &st_buf)) {
        perror("vget: Can't access file");
        return PFAILURE;
    }
#ifdef S_ISDIR
    if (S_ISDIR(st_buf.st_mode)) {
#else
    if ((st_buf.st_mode & S_IFMT) == ST_IFDIR) {
#endif
        fprintf(stderr, "vget: Can't retrieve file %s: it is actually a \
directory.", inname);
        return PFAILURE;
    }
    if (!forcecopyflag) {
        strcpy(buf, "y");
        fputs("This file is available in your native filesystem as %s.\
 Still do the copy? [RETURN will default to yes]", stdout);
        fflush(stdout);
        gets(buf);
        if (*buf != 'y' && *buf == '\0')
            return PSUCCESS;
    }
    if ((in = fopen(inname, "r")) == NULL) {
        fprintf(stderr, "vget: Couldn't open the file %s for reading.  Copy \
failed.\n", inname);
        return PFAILURE;
    } 
    if ((out = fopen(outname, "w")) == NULL) {
        fprintf(stderr, "vget: Couldn't open the file %s for writing.  Copy \
failed.\n", outname);
        fclose(in);
        return PFAILURE;
    }
 again:
    numread = fread(buf, 1, sizeof buf, in);
    if (numread <= 0) {
        if (ferror(in)) {
            fprintf(stderr, "Error while reading from %s; copying to %s \
aborted\n", inname, outname);
            fclose(in); 
            fclose(out);
            return PFAILURE;
        } else {
            /* done! */
            fclose(in);
            fclose(out);
            return PSUCCESS;
        }
    }
    numwritten = fwrite(buf, 1, numread, out);
    if (numwritten != numread) {
        fprintf(stderr, "vget: tried to write %d items to %s; only wrote %d. \
Copying aborted.\n", numread, outname, numwritten);
        fclose(in); 
        fclose(out);
        return PFAILURE;
    }
    goto again;
}
